# (c)2004-2005 Dagmar d'Surreal and the Dropline Gnome Project
# Date: December 7th, 2005.
# Version: 0.97

# The same convention of grep "^##" will generate documentation for the things
# done by this file.

# Oh, and yes, I know there's a lot of stuff in here that's just not
# symmetrical with other things

## DLG_ROOT/SCRIPTS/dropline-functions.sh Documentation
##

## Bash function call documentation
##

## spew() level message
##   This will emit the message if the level is less than or equal to the
##   DLG_VERBOSITY setting.  Very handy for avoiding console spam.  Try to use
##   this instead of flagrantly echoing things to the terminal.
##
spew() {
  local level=${1:?'Internal error, spew() requires a verbosity threshold'}
  local message=${2:?'Internal error, spew() requires a message'}
  local -a color
  color[0]="$cred"
  color[1]="$cwht$bold"
  color[2]="$cwht"
  color[3]="$cyel"
  color[4]="$cgrn"
  color[5]="$cblu"

  if [ $level -le $DLG_VERBOSITY ]; then
    echo "${color[$level]}$message$boff"
  fi
}

## warn() message
##   This will emit a message that is considered a non-fatal error.  If the
##   verbosity level is 0 or 1, it will be suppressed.
##
warn() {
  local message=${*:?'Internal error: warn() requires a message'}
  if [  $DLG_VERBOSITY -ge 2 ]; then
    echo "$cyel$message$boff"
  fi
}

## spewdo() message
##   This came about after repeatedly running into issues with minor
##   differences between an entered command and what is supposed to be echoed.
##   It'll just add the proper level and pass the message long to spew(),
##   and then eval it.
##
spewdo() {
  local command=${1:?'Internal error, spewdo() requires a command.'}
  spew 5 "$command"
  eval $command
}

## removearg() varname argument
##   This routine allows you to remove a (whitespace-separated) argument from
##   a variable without having to go through a lot of fuss and bother.
##
removearg() {
  local var_ra=${1:?'Internal error, removearg() requires a varname.'}
  local argument=${2:?'Internal error, removearg() requires an argument.'}
  local retval
  local item

  for item in ${!var_ra}; do
    if [ "$item" != "$argument" ]; then
      retval=${retval:+$retval }$item
    fi
  done
  export $var_ra="$retval" 
}

## addarg() varname argument
##   This routine allows you to add a (whitespace-separated) argument to a
##   variable without the risk of duplicating it if it's already there.
##   Note that this will only work for variables that are expected to contain
##   a simple list of white-space separated arguments!
##
addarg() {
  local var_aa=${1:?'Internal error, addarg() requires a varname.'}
  local argument=${2:?'Internal error, addarg() requires an argument.'}
  local retval
  local item

  removearg $var_aa $argument
  retval="${!var_aa} $argument"
  export $var_aa="$retval" 
}

## sysvinitsetup() filename runlevels startpoint stoppoint
##   This routine is pretty straightforward if you're used to looking at the
##   chkconfig lines in RedHat's init scripts.  Example: "sysvinitsetup
##   $TMPDIR/rc.messagebus 2345 97 03".  Note that the actual symlink will
##   be composed of only the "messagebus" part of the filename.  If you only
##   supply the filename, no symlinks will be created.
##
sysvinitsetup() {
  local filename=${1:?'Internal error: sysvinitsetup() requires a filename!'}
  local runlevels=$2
  local startpoint=$3
  local stoppoint=$4
  
  spew 4 "Setting up SysVinit hooks..."

  if [ ! -r $filename ]; then
    die "Fatal error from sysvinitsetup(): '$filename' is unreadable."
  fi

  # If your script should default OFF, chmod it after calling this function
  spewdo "createdir $PKG_STAGEROOT/etc/rc.d/"
  install -m 755 $filename $PKG_STAGEROOT/etc/rc.d

  if [ -z "$runlevels" ]; then
    return
  fi

  # Your init script should be named "rc.somethingorother".
  local stubname=`basename $filename | cut -d. -f2`

  if [ -z "$stubname" ]; then
    spew 3 "...we can't actually make the symlinks tho.  UTSL."
    return
  fi

  if [ -z "$startpoint" ]; then
    spew 3 "...we can't actually make the symlinks tho.  UTSL."
    return
  fi

  # We can probably make some assumptions about the stoppoint.
  if [ -z "$stoppoint" ]; then
    stoppoint=$(( 100 - $startpoint ))
    stoppoint=`printf "%02d" $stoppoint`
  fi

  for ((a=1 ; a <= ${#runlevels} ; a++)); do
    spewdo "createdir $PKG_STAGEROOT/etc/rc.d/rc${runlevels:$((a - 1)):1}.d"
    cd $PKG_STAGEROOT/etc/rc.d/rc${runlevels:$((a - 1)):1}.d
    ln -sf ../`basename $filename` S$startpoint$stubname
    ln -sf ../`basename $filename` K$stoppoint$stubname
  done
}

## set_title() message
##   This is one of Todd's routines from the original build system.  It does
##   the simple task of setting the xterm titlebar and it does it well.
##
set_title() {
  if [ ${TERM} = "xterm" ]; then
    echo -e -n "\033]0;${1}\007"
  fi
}

## die() message
##   This is a somewhat obvious function.  It emits a message if you send it
##   one, but otherwise exits the program.
##
die() {
  local message=$*
  set_title "$NAME-$VERSION-$REVISION: build failure!"
  if [ -n "$message" ]; then
    echo "$bell$cred$bold$message$boff"
  fi
  exit 1
}

## preparescratchspace()
##   This just cleans out the stale sources and such from the last attempt
##   and then re-makes the directories.
##
preparescratchspace() {
  spew 4 "Clearing out the work directory: $PKG_WORKROOT"
  removedir $PKG_WORKROOT
  createdir $PKG_WORKROOT

  spew 4 "Clearing out the staging directory: $PKG_STAGEROOT"
  removedir $PKG_STAGEROOT
  createdir $PKG_STAGEROOT

  spew 4 "Clearing out the log directory: $PKG_LOGROOT"
  removedir $PKG_LOGROOT
  createdir $PKG_LOGROOT
}

## prepare()
##   This just nukes the old source directories, leftover log files, and opens
##   up the new tarball in the proper place.
##
prepare() {
  local filename

  # It's banner time
  spew 1 " "
  spew 1 "*** Building $NAME-$VERSION-$PKG_ARCH-$REVISION$PKG_PKGER."

  set_title "$NAME-$VERSION-$REVISION: preparation phase"
  spew 2 "Preparing the sources and work areas..."

  preparescratchspace

  findtarball filename
  if [ -z $filename ]; then
    echo "Unable to find an appropriate tarball for this package.  Sorry."
    exit 1
  fi

  opentarball $filename
  sanifysource $PKG_SRCROOT
}

## sanifysource() dirname
## This routine merely chowns, chmods, and touches the source files.
##
sanifysource() {
  local dirname=${1:?'Internal error: sanifysource() requires a dirname!'}
  if [ "$dirname" = "/" ]; then
    die "No, I will not trash your filesystem."
  fi
	# This syntax DOES NOT WORK!  It winds up ignoring directories.
  #spewdo "find $dirname -type d -o -type f -exec touch {} \;"
  spewdo "find $dirname \( -type d -or -type f \) -exec touch {} \;"
  #spewdo "find $dirname -type d -o -type f -exec chown root.root {} \;"
  spewdo "find $dirname \( -type d -or -type f \) -exec chown root.root {} \;"
  spewdo "find $dirname -type d -exec chmod go-w+x {} \;"
  # This _should_ be safe.
  spewdo "find $dirname -type f -exec chmod go-wx {} \;"
}

## analyzesource()
##   This is a big chunk of programming voodoo that checks the source package
##   for various things and sets control variables accordingly.
##
analyzesource() {
  set_title "$NAME-$VERSION-$REVISION: Analyzing source structure"

  spew 2 "Analyzing source structure..."

  spew 3 "Searching for the presence of Scrollkeeper files..."
  if [ `find $PKG_SRCROOT . -name *.omf -o -name *.omf.in | wc -l 2>&1 ` -ne 0 ]; then
    spew 4 "  ...scrollkeeper files detected."
    PKG_SCROLLKEEPER=1
  fi

  spew 3 "Searching for the presence of GConf schemas..."
  if [ `find $PKG_SRCROOT . -name *.schemas -o -name *.schemas.in -o -name *.schemas.in.in | wc -l 2>&1 ` -ne 0 ]; then
    spew 4 "  ...gconf schemas detected."
    PKG_GCONFTWO=1
	fi
  # GNU info and *. files are better checked at the end of this.
}

## configfile() filename
##   This routine just starts building a list of which files are configuration
##   files so that they can be handled properly in the doinst.sh script.  Be
##   certain you specify fhem as absolute paths as if the staging directory
##   really were /.
##
configfile() {
  local filename=${1:?'Internal error.  configfile() needs a filename.'}

  spew 4 "Added $filename as a configuration file"

  appendarg PKG_CONFIGFILELIST $filename
}

## logfile() filename
##   This routine startsbuilding a list of the files which are merely
##   placeholders for log files which have no actual content when we need
##   to be sure they are created with the correct permissions.
##
logfile() {
  local filename=${1:?'Internal error.  logfile() needs a filename.'}

  spew 4 "Added $filename as a log file"

  appendarg PKG_LOGFILELIST $filename
}

## cleanup()
##   This routine goes ahead and cleans up the mess left by the package
##   creation process.  It deletes the source files, and the binary files,
##   and then compresses any log files that are left over.
##
cleanup() {
  local pref
  local compressor
  set_title "$NAME-$VERSION-$REVISION: Cleaning up..."

  spew 4 "Deleting the leftover source code and binary files..."
  removedir $PKG_WORKROOT
  removedir $PKG_STAGEROOT
  
  pref=`echo $PKG_ZPREF | awk '{ print $1 }'`
  case pref in
    bz2)
      compressor=bzip2
      ;;
    *)
      compressor=gzip 
  esac 

  spew 4 "Compressing the logfiles using $compressor compression."
  for filename in $PKG_LOGROOT/*.txt; do
    spew 5 "$compressor -9f $filename"
    $compressor -9f $filename
  done
  
  set_title "$NAME-$VERSION-$REVISION: all done!"
}


## releasescripts()
##   This function simply recreates the build definition files (patches,
##   build script, changelog, package description, etc) under /usr/src
##   so that someone with the current package installed stands a fighting
##   chance of building a replacement package without having to do a lot
##   of research into how their current package was built.  It is VERY
##   IMPORTANT that you reference all your build files with the PKG_PATCH
##   variables (excluding the main PKG_SOURCE URLs, since those are
##   supposed to be downloadable if at all possible) so that they can be
##   picked up and properly distributed by this routine.
releasescripts() {
	local idx
	local filename

	spew 4 "Copying the build scripts over into the package for release."
	# PKG_PREFIX is not something you'd use here, since this is pointing
	# at the location of the build system, which for our purposes doesn't
	# ever move to somewhere else.
	createdir "$PKG_STAGEROOT/usr/src/dropline-build-system/SOURCES/$NAME"
	createdir "$PKG_STAGEROOT/usr/src/dropline-build-system/SCRIPTS/$NAME"

	for (( idx=0 ; idx < ${#PKG_PATCH[*]} ; idx++ ))
	do
		spew 4 "  Copying patch #$idx..."
		if [ -z "${PKG_PATCH[$idx]}" ]; then
   			spew 4 "  Undefined patch #$idx requested."
  		fi

  		findpatch filename ${PKG_PATCH[$idx]}
  		if [ $? != 0 ]; then
    			die "Unable to locate or download patch #$idx!"  
		fi
		cp $filename "$PKG_STAGEROOT/usr/src/dropline-build-system/SOURCES/$NAME"
	done

	for idx in doinst.sh README.Dropline
	do
		if [ -r $DLG_ROOT/SOURCES/$NAME/$idx ]; then
			cp $DLG_ROOT/SOURCES/$NAME/$idx "$PKG_STAGEROOT/usr/src/dropline-build-system/SOURCES/$NAME" 
		fi
	done

	for idx in ChangeLog build desc doinst.sh README.Dropline
	do
		if [ -r $DLG_ROOT/SCRIPTS/$NAME/$idx ]; then
			cp $DLG_ROOT/SCRIPTS/$NAME/$idx "$PKG_STAGEROOT/usr/src/dropline-build-system/SCRIPTS/$NAME" 
		fi
	done
}

## packagize() [leavelinks] [chown] [dontprepend]
##   This routine actually calls makepkg in the PKG_STAGEROOT.  The
##   default method is to remove symlinks, to NOT chown files, and
##   to prepend symlink recreation to the doinst.sh you may have already
##   created for the package.  The optional arguments are for reversing
##   those behaviours.
##
packagize() {
  local argument
  local command
  local status
  local rmlinks='y'
  local chown='n'
  local prepend=' --prepend'

  # This is all we really need to do about x86-64, since the dash
  # will break pkgtools filename parsing.
  local tmp_arch=`echo "$PKG_ARCH" | sed 's/-/_/g;'`
  # And if this was unset, it was probably for a reason.
  if [ -z "$tmp_arch" ]; then
    tmp_arch="noarch"
  fi
  local filename="$NAME-$VERSION-$tmp_arch-$REVISION$DLG_PKGER.tgz"

  local oldpwd=`pwd`
  set_title "$NAME-$VERSION-$REVISION: Creating binary package!"

  for argument in $*; do
    case $argument in 
      leavelinks)
        rmlinks='n'
        ;;
      chown)
        chown='y'
        ;;
      dontprepend)
        unset prepend
        ;;
      prepend)
	# This left here for compatibility's sake.  Remove for 2.0.0.
	;;
      *)
        die "Unrecognized argument \"$argument\" to packagize() seen."
    esac
  done

  releasescripts

  command="makepkg$prepend --linkadd $rmlinks --chown $chown $filename"
  cd $PKG_STAGEROOT
  loggedcommand packagize "$command"
  status=$?
  if [ $status != 0 ]; then
    die "The makepkg command exited with status $status!"
  fi

  stowpackage $filename

  spew 1 "$filename completed!"
}

## includescripts()
##   Just a hook to alert people who are still using includescripts, since
##   it's poorly named and not something package builders should have to
##   type a command for.  It should be part of the packagize function.
##
includescripts() {
	die "The includescripts() function is now deprecated and has been
	replaced by the releasescripts() function, which is called from within
	packagize() so you no longer need to worry about it at all."
}

## applypatch patchnum [stripnum]
##   This function will apply the numbered patch to the source tree after
##   it has been opened.  The patchnum is the index of the entry in the
##   PKG_PATCHFILE[n] array.  The stripnum is what is passed to -p when
##   patch is invoked.  Patches will be applied from the PKG_SRCROOT
##   directory.
## 
applypatch() {
  local patchnum=${1:?'Internal error.  applypatch() needs an index number.'}
  local stripnum=$2
  local filename
  local oldpwd=`pwd`

  if [ -z "${PKG_PATCH[$patchnum]}" ]; then
    die "Undefined patch #$patchnum requested."
  fi

  findpatch filename ${PKG_PATCH[$patchnum]}
  if [ $? != 0 ]; then
    die "Unable to locate or download patch #$patchnum!"
  fi

  spew 2 "Applying patch $patchnum..."
  spewdo "cd $PKG_SRCROOT"
  loggedcommand patch-$patchnum "patch -Np$stripnum -i $filename"
  status=$?
  if [ $status != 0 ]; then
    die "Patch process exited with status $status!"
  fi
  cd $oldpwd
}

## findpatch varname URL
##   A cousin to the findtarball function, this routine will search the
##   DLG_SOURCEREPOS directories for the file the URL would download.
##   Note that unlike findtarball() this routine actually stores patches
##   downloaded within the dropline-build-scripts tree.  The URL parser
##   is pretty crude, so I wouldn't bother with file:// designations.
##   The function will return the location of the tarball in varname.
findpatch() {
  local varname=${1:?'Internal error.  findpatch requires a varname'}
  local url=${2:?'Internal error.  findpatch() requires a URL'} 
  local dirname
  local comptype 
  local found
  local tmp
  local basename=`echo $url | awk -F/ '{ print $NF }'`

  # This block is messy.
  for dirname in $PKG_COREREPOS $DLG_SOURCEREPOS; do
    spew 4 "Looking for $basename in $dirname"
    if [ -r $dirname/$basename ]; then
      spew 3 "Found $dirname/$basename!"
      found="$dirname/$basename"
      break
    fi
    for comptype in $DLG_ZPREF; do
      spew 4 "Looking for $basename.$comptype in $dirname"
      if [ -r $dirname/$basename.$comptype ]; then
        spew 3 "Found $dirname/$basename.$comptype!"
        found="$dirname/$basename.$comptype"
        break
      fi 
    done
    if [ ! -z "$found" ]; then
      break
    fi
  done

  if [ -z "$found" ]; then
    if [ ! -z "$url" ]; then
      createdir $PKG_COREREPOS
      downloadfile $PKG_COREREPOS $url
      if [ $? = 0 ]; then
        tmp=`echo $url | awk -F/ '{ print $NF }'`
        eval $varname="$PKG_COREREPOS/$tmp"
        return
      fi
      return 1
    fi
    return 1
  else
    eval $varname="$found"
  fi
}

## checkmd5sum() filename md5sum
##   This function simplifies the task of checking the md5um on a file.
##   It merely returns the appropriate exit value, instead of calling die()
##   on error, because there are cases where this would be non-fatal.
##
checkmd5sum() {
  local filename=${1:?'Internal error.  checkmd5sum() requires a filename.'}
  local md5sum=${2:?'Internal error.  checkmd5sum() requires an md5sum.'}

  local tmp

  if [ ! -r $filename ]; then
    die "Internal error: checkmd5sum() needs a $filename to be readable."
  fi

  spew 4 "Verifying that the md5sum of $filename is '$md5sum'."

  tmp=`md5sum $filename 2>/dev/null | head -n 1 | cut -d" " -f1`
  spew 4 "  ...and it's '$tmp'."
  if [ "$md5sum" = "$tmp" ]; then
    return
  fi
  warn "md5sum of $filename does not appear to be what we were looking for."
  return 1
}

## stowpackage() [filename] 
##   This function will stash the completed binary package in its proper
##   place and make the necessary signing files. 
##
stowpackage() {
  local package=${1:-"$NAME-$VERSION-$PKG_ARCH-$REVISION$DLG_PKGER.tgz"}
  local oldpwd=`pwd`

  set_title "$NAME-$VERSION-$REVISION: Storing release candidate package..."
 
  spew 3 "Relocating package to $PKG_PACKAGEROOT..."
  createdir $PKG_PACKAGEROOT
  cd $PKG_STAGEROOT
  spewdo "mv -f $package $PKG_PACKAGEROOT"
  status=$?
  if [ $status != 0 ]; then
    die "Moving the package to $PKG_PACKAGEROOT exited with status $status!"
  fi

  spew 3 "Generating md5sum for $package"
  cd $PKG_PACKAGEROOT
  spewdo "md5sum $package > $package.md5"
  status=$?
  if [ $status != 0 ]; then
    die "Generating the md5 hash of $package exited with status $status!"
  fi
  cd $oldpwd
}

## finalizepackage() 
##   This is the place where everything is supposed to happen that turns a
##   bunch of files sitting in a relocated root into an actual distributable
##   software package.  The very next command in the build recipe should be
##   the packagizing command.
##
finalizepackage() {
  set_title "$NAME-$VERSION-$REVISION: Finalizing package contents"

  createdir $PKG_STAGEROOT/install
  if [ -r $DLG_ROOT/SOURCES/$NAME/slack-desc ]; then
    spew 4 "Deploying the description file..."
    spewdo "cat $DLG_ROOT/SOURCES/$NAME/slack-desc > $PKG_STAGEROOT/install/slack-desc"
    spewdo "chmod 644 $PKG_STAGEROOT/install/slack-desc"
  elif [ -r $DLG_ROOT/SCRIPTS/$NAME/desc ]; then
    spew 4 "Deploying the description file..."
    spewdo "cat $DLG_ROOT/SCRIPTS/$NAME/desc > $PKG_STAGEROOT/install/slack-desc"
    spewdo "chmod 644 $PKG_STAGEROOT/install/slack-desc"
  else
    spew 4 "Deploying a generic description file..."
cat >> $PKG_STAGEROOT/install/slack-desc <<EOF
$NAME: $NAME
$NAME:
$NAME: No descripion is currently available for this package.
EOF
  fi

  # This has to come before any binaries which may depend upon the existance
  # of a configuration file's presence.
  if [ ! -z "$PKG_CONFIGFILELIST" ]; then
    doinstconfigfiles
  fi

  if [ ! -z "$PKG_LOGFILELIST" ]; then
    doinstlogfiles
  fi

  if [ -r $DLG_ROOT/SOURCES/$NAME/doinst.sh ]; then
    spew 4 "Deploying the installation script..."
    cat $DLG_ROOT/SOURCES/$NAME/doinst.sh >> $PKG_STAGEROOT/install/doinst.sh
    chmod 644 $PKG_STAGEROOT/install/doinst.sh
  fi
 
  compressmanpages
  compressinfofiles

  if [ "$PKG_GCONFTWO" = "1" ]; then
    doinstgconfschemas
  fi

  if [ "$PKG_SCROLLKEEPER" = "1" ]; then
    doinstscrollkeeper
  fi

  if [ "$PKG_DESKTOPFILES" = "1" ]; then
    doinstdesktopfiles
  fi

  if [ "$PKG_MIMEFILES" = "1" ]; then
    doinstmimedatabase
  fi

  if [ "$PKG_GNUINFO" = "1" ]; then
    doinstgnuinfo
  fi

  if [ "$PKG_ICONTHEMES" = "1" ]; then
    doinsticonthemes
  fi

  if [ "$PKG_GTKTWOPIXBUFLOADERS" = "1" ]; then
    doinstgtktwopixbufloaders
  fi

  if [ -r $DLG_ROOT/SOURCES/$NAME/doinst.sh-latter ]; then
    spew 4 "Deploying the latter part of the installation script..."
    cat $DLG_ROOT/SOURCES/$NAME/doinst.sh-latter >> $PKG_STAGEROOT/install/doinst.sh
    chmod 644 $PKG_STAGEROOT/install/doinst.sh
  fi

  polishfiles
}

## compressinfofiles()
##   This simply compresses any GNU info files it sees using the approriate
##   compression utility.
##
compressinfofiles() {
  local compressor
  local item

  set_title "$NAME-$VERSION-$REVISION: Compressing info files..."
  spew 2 "Compressing info files..."

  firstarg fileext $PKG_ZPREF
  if [ "$fileext" = "bz2" ]; then
    spew 4 "  ...with bzip2."
    compressor=bzip2
  elif [ "$fileext" = "gz" ]; then
    spew 4 "  ...with gzip."
    compressor=gzip
  else
    spew 4 "  ...with gzip (no supported methods are defined for \"$fileext\")."
    compressor=gzip
  fi

  if [ -d $PKG_STAGEROOT$PKG_PREFIX/info ]; then
    for item in `find $PKG_STAGEROOT$PKG_PREFIX/info -type f -name *.info*`; do
      spewdo "$compressor -9 $item"
    done
  fi
}

## compressmanpages()
##   This does exactly what it's name implies.  It compresses the man pages.
##   This may not properly account for "linking" man pages.  Those need to be
##   edited, but at least they're rare.
##
compressmanpages() {
  local compressor
  local fileext
  local item
  local location
  local linkdest
  local directory
  local filename
  local oldpwd=`pwd`

  set_title "$NAME-$VERSION-$REVISION: Compressing man pages..."

  if [ -d $PKG_STAGEROOT$PKG_DATADIR/man ]; then
    spew 2 "Relocating man pages to their canonical location..."
    mkdir -p $PKG_STAGEROOT$PKG_PREFIX/man
    mv $PKG_STAGEROOT$PKG_DATADIR/man/* $PKG_STAGEROOT$PKG_PREFIX/man 
    removedir $PKG_STAGEROOT$PKG_DATADIR/man
  fi

  spew 2 "Compressing man pages..."
  
  firstarg fileext $PKG_ZPREF
  if [ "$fileext" = "bz2" ]; then
    compressor=bzip2
  elif [ "$fileext" = "gz" ]; then
    compressor=gzip
  else
    spew 3 "No recognizable compression method found.  Assuming gzip."
    compressor=gzip
    fileext=gz
  fi
 
  if [ -d $PKG_STAGEROOT$PKG_PREFIX/man ]; then
    # Compressing pages.
    spew 3 "Compressing man page files..."
    for item in `find $PKG_STAGEROOT$PKG_PREFIX/man -type f`; do
      spewdo "$compressor -9 $item"
    done
  
    spew 3 "Readjusting symlinks..."
    for item in `find $PKG_STAGEROOT$PKG_PREFIX/man -type l`; do
      filename=`basename $item`
      directory=`dirname $item` 
      ( spewdo "cd $directory"
        linkdest=`ls -l $filename | awk '{ print $NF }'`
       spewdo "rm $item"
        spewdo "ln -s $linkdest.$fileext $filename.$fileext"
      )
    done 

    # TG at LinuxPackages feels this is important, and I can't see how this
    # could ever hurt anything so we might as well go ahead with it.
    spew 3 "Enforcing modes appropriate for text files on man pages..."
    spewdo "find $PKG_STAGEROOT$PKG_PREFIX/man -type f -exec chmod 644 {} \;"
  fi
}

## doinstconfigfiles()
##   This is a bit of boilerplate to ensure that configuration files don't
##   blindly stomp all over an existing configuration file.  TODO: Make this
##   little fiasco honor $ROOT properly.
##
doinstconfigfiles() {
  local oldpwd=`pwd`
  local filename

  spew 4 "Creating doinst.sh hooks to properly handle new configuration files."
  cd $PKG_STAGEROOT
cat >>install/doinst.sh <<XdoinstconfigfilesX
config() {
  NEW="\$1"
  OLD="\`dirname \$NEW\`/\`basename \$NEW .new\`"
  # If there's no config file by that name, mv it over:
  if [ ! -r /\$ROOT/\$OLD ]; then
    mv /\$ROOT/\$NEW /\$ROOT/\$OLD
  elif [ "\`cat /\$ROOT/\$OLD 2>/dev/null | md5sum\`" = "\`cat /\$ROOT/\$NEW 2>/dev/null | md5sum\`" ]; then # toss the redundant copy
    rm /\$ROOT/\$NEW
  fi
  # Otherwise, we leave the .new copy for the admin to consider...
}

XdoinstconfigfilesX

  for filename in $PKG_CONFIGFILELIST; do
    spewdo "mv $PKG_STAGEROOT$filename $PKG_STAGEROOT$filename.new"
    echo "config $filename.new" >>install/doinst.sh
  done
  echo "" >>install/doinst.sh

  cd $oldpwd
}

## doinstlogfiles()
##   This routine ensures that log files are actually created with the
##   proper permissions and ownerships at installation time.
##
doinstlogfiles() {
	local oldpwd=`pwd`
	local filename string perms
	local ownership permuser permgroup permother

	spew 4 "Creating doinst.sh hooks to properly handle the creation of logs."
	cd $PKG_STAGEROOT

  for filename in $PKG_LOGFILELIST; do
	  if [ ! -f $PKG_STAGEROOT$filename ]; then
			die "logfile() argument $filename does not exist!"
		fi
		string=`ls -al $PKG_STAGEROOT$filename`
		perms=`echo $string | awk '{ print $1 }'`
		owner=`echo $string | awk '{ print $3 ":" $4 }'`
		permuser=`echo ${perms:1:3} | sed 's/-//g'`
		permgroup=`echo ${perms:4:3} | sed 's/-//g'`
		permother=`echo ${perms:7:3} | sed 's/-//g'`

		spew 4 "...adding creation of log file $filename to doinst.sh."
		echo "touch \$ROOT$filename" >> install/doinst.sh
		echo "chown $owner \$ROOT$filename" >> install/doinst.sh
		echo "chmod u=$permuser,g=$permgroup,o=$permother \$ROOT$filename" >> install/doinst.sh

		spewdo "rm $PKG_STAGEROOT$filename"
	done

	cd $oldpwd
}

## doinstgconfschemas()
##   Creates the hooks that ensure gconf schemas are handled properly.
##
doinstgconfschemas() {
  local filename
  local oldpwd=`pwd`

  spew 4 "Creating doinst.sh hooks to get gconf schemas properly installed"
  cd $PKG_STAGEROOT
  echo "if [ -x usr/bin/gconftool-2 -a -x usr/bin/chroot ]; then" >>install/doinst.sh
  echo "  GCONF_CONFIG_SOURCE=\`usr/bin/chroot /\$ROOT /usr/bin/gconftool-2 --get-default-source\`" >>install/doinst.sh

  for filename in `find ${PKG_SYSCONFDIR/\//}/gconf/schemas -name "*.schemas"`; do
    echo "  usr/bin/chroot /\$ROOT /usr/bin/env GCONF_CONFIG_SOURCE=\$GCONF_CONFIG_SOURCE /usr/bin/gconftool-2 --makefile-install-rule /$filename 2>&1 >/dev/null" >>install/doinst.sh
  done

	for filename in `find ${PKG_SYSCONFDIR/\//}/gconf/schemas -name "*.entries"`; do
    echo "  usr/bin/chroot /\$ROOT /usr/bin/env GCONF_CONFIG_SOURCE=\$GCONF_CONFIG_SOURCE /usr/bin/gconftool-2 --makefile-install-rule --load /$filename 2>&1 >/dev/null" >>install/doinst.sh
	done

  echo "fi" >>install/doinst.sh
  cd $oldpwd
}

## doinstscrollkeeper()
##   Creates the hooks that ensure scrollkeeper updates it's database.
##
doinstscrollkeeper() {
  local filename

	spew 4 "Checking for erroneous scrollkeeper indices..."
	if [ -d $PKG_STAGEROOT$PKG_LOCALSTATEDIR/scrollkeeper/TOC ]; then
		spewdo "removedir $PKG_STAGEROOT$PKG_LOCALSTATEDIR/scrollkeeper/TOC"
		spew 4 "$PKG_LOCALSTATEDIR/scrollkeeper/TOC found and eliminated."
	fi
	if [ -d $PKG_STAGEROOT$PKG_LOCALSTATEDIR/scrollkeeper/index ]; then
		spewdo "removedir $PKG_STAGEROOT$PKG_LOCALSTATEDIR/scrollkeeper/index"
		spew 4 "$PKG_LOCALSTATEDIR/scrollkeeper/index found and eliminated."
	fi

	# There's a _lot_ of packages that seem to think it's okay to go ahead
	# and ignore whether or not DESTDIR is defined when they attempt to
	# install their scrollkeeper data.  The result is that the packages
	# basically trash scrollkeeper unless we clear those files out.
	if [ ! -z "`ls $PKG_STAGEROOT/var/lib/scrollkeeper/`" ]; then
		spew 4 "Deleting extraneous processed scrollkeeper files..."
		spewdo "rm -rf $PKG_STAGEROOT/var/lib/scrollkeeper/*"
	fi

  spew 4 "Creating doinst.sh hooks to get scrollkeeper's database updated"
  cd $PKG_STAGEROOT
  echo "if [ -x usr/bin/chroot -a -x usr/bin/scrollkeeper-update ]; then" >>install/doinst.sh
  for filename in `ls -d ${PKG_DATADIR/\//}/omf/*`; do
    spew 5 "Handling /$filename"
    echo "  usr/bin/chroot /\$ROOT /usr/bin/scrollkeeper-update -n -p $PKG_LOCALSTATEDIR/scrollkeeper -o /$filename 2>/dev/null" >>install/doinst.sh
  done
  echo "fi" >> install/doinst.sh
  cd $oldpwd
}

## doinstdesktopfiles()
##   Creates the hooks to ensure the desktop database is updated.
##
doinstdesktopfiles() {
  local oldpwd=`pwd`
  spew 4 "Creating doinst.sh hooks to update the desktop database..."
  cd $PKG_STAGEROOT
  echo "if [ -x usr/bin/update-desktop-database -a -x usr/bin/chroot ]; then" >> install/doinst.sh

  echo "  usr/bin/chroot /\$ROOT /usr/bin/update-desktop-database &> /dev/null" >> install/doinst.sh
  echo "fi" >> install/doinst.sh
  cd $OLDPWD
}

## doinstmimedatabase()
##   Creates the hooks to ensure the mime database is updated.
##
doinstmimedatabase() {
  local oldpwd=`pwd`

	spew 4 "Purging any premature mime database files..."
  if [ -e $PKG_STAGEROOT$PKG_DATADIR/mime/XMLnamespaces ]; then
		spewdo "rm -f $PKG_STAGEROOT$PKG_DATADIR/mime/XMLnamespaces"
	fi
  if [ -e $PKG_STAGEROOT$PKG_DATADIR/mime/globs ]; then
		spewdo "rm -f $PKG_STAGEROOT$PKG_DATADIR/mime/globs"
	fi
  if [ -e $PKG_STAGEROOT$PKG_DATADIR/mime/magic ]; then
		spewdo "rm -f $PKG_STAGEROOT$PKG_DATADIR/mime/magic"
	fi

  spew 4 "Creating doinst.sh hooks to update the mime database..."
  cd $PKG_STAGEROOT
  echo "if [ -x usr/bin/update-mime-database -a -x usr/bin/chroot ]; then" >> install/doinst.sh

  echo "  usr/bin/chroot /\$ROOT /usr/bin/update-mime-database $PKG_DATADIR/mime &> /dev/null" >> install/doinst.sh
  echo "fi" >> install/doinst.sh
  cd $OLDPWD
}


## doinstgnuinfo()
##  Creates the hooks to ensure GNU info pages are indexed properly.
##
doinstgnuinfo() {
  local flag
  local filename
  local oldpwd=`pwd`

	if [ -e $PKG_STAGEROOT$PKG_PREFIX/info/dir ]; then
		spew 4 "Removing spurious GNU info directory index file..."
		spewdo "rm $PKG_STAGEROOT$PKG_PREFIX/info/dir"
	fi
	if [ -e $PKG_STAGEROOT$PKG_PREFIX/info/dir.gz -o \
		-e $PKG_STAGEROOT$PKG_PREFIX/info/dir.bz2 ]; then
		spew 4 "Removing spurious GNU info directory index file..."
		spewdo "rm $PKG_STAGEROOT$PKG_PREFIX/info/dir.{bz2,gz}"
	fi

  spew 4 "Creating doinst.sh hooks to properly install GNU info files..."
	for filename in `ls $PKG_STAGEROOT$PKG_PREFIX/info/*.info.{bz2,gz} 2>/dev/null`; do
		spew 5 "Handling /$filename"
		if [ -z "$flag" ]; then
			echo "if [ -x usr/bin/install-info ]; then" >>$PKG_STAGEROOT/install/doinst.sh
			flag=x
		fi
		echo "  usr/bin/install-info --info-dir=${PKG_PREFIX/\//}/info ${PKG_PREFIX/\//}/info/`basename $filename` 2>/dev/null" >>$PKG_STAGEROOT/install/doinst.sh
	done

	if [ ! -z "$flag" ]; then
		echo "fi" >>$PKG_STAGEROOT/install/doinst.sh
  fi
}

## doinsticonthemes()
##   Create the bits in the doinst.sh that ensure gtk icon caches get made.
##
doinsticonthemes() {
  spew 4 "Creating doinst.sh hooks to properly create icon theme caches..."

	mkdir -p $PKG_STAGEROOT/install
  echo "if [ -x usr/bin/gtk-update-icon-cache -a -x usr/bin/chroot ]; then" >> $PKG_STAGEROOT/install/doinst.sh

 	for dirname in `find $PKG_STAGEROOT$PKG_PREFIX/share/icons -type d -mindepth 1 -maxdepth 1`; do
  	echo "  usr/bin/chroot /\$ROOT /usr/bin/gtk-update-icon-cache -f -q /usr/share/icons/`basename $dirname` &> /dev/null" >> $PKG_STAGEROOT/install/doinst.sh

	done
	echo "fi" >>$PKG_STAGEROOT/install/doinst.sh
}

## doinstgtktwopixbufloaders()
##   Create the bits in the doinst.sh that ensure the pixbuf loaders index
##   is repopulated at install time.
##
doinstgtktwopixbufloaders() {
	spew 4 "Creating doinst.sh hooks to update the gdk-pixbuf loaders list..."
  mkdir -p $PKG_STAGEROOT/install
  echo "if [ -x usr/bin/chroot -a -x usr/bin/gdk-pixbuf-query-loaders ]; then" >>$PKG_STAGEROOT/install/doinst.sh 
  echo "  usr/bin/chroot /\$ROOT /usr/bin/gdk-pixbuf-query-loaders > /etc/gtk-2.0/gdk-pixbuf.loaders 2>/dev/null" >>$PKG_STAGEROOT/install/doinst.sh
  echo "fi" >>$PKG_STAGEROOT/install/doinst.sh
}

## analyzebinaries()
##   This routine is for some last-minute analysis of the new binary package
##   to spot Gnome *.desktop files, GNU info pages and so on.
##
analyzebinaries() {
  set_title "$NAME-$VERSION-$REVISION: Analyzing binary structure"

  spew 2 "Analyzing binary structure..."

  spew 3 "Searching $PKG_DATADIR for the presence of Gnome/KDE/FDO *.desktop files..."
  if [ `find $PKG_STAGEROOT/$PKG_DATADIR -name *.desktop 2>/dev/null | wc -l` -ne 0 ]; then
    spew 4 "  ...found some!"
    PKG_DESKTOPFILES=1
  fi

  spew 3 "Searching $PKG_DATADIR for the presence of mime keys..."

  spew 3 "Searching for the presence of GNU info pages..."
  if [ `find $PKG_STAGEROOT/$PKG_PREFIX/info -name *.info 2>/dev/null | wc -l` -ne 0 ]; then
    spew 4 "  ...found some!"
    PKG_GNUINFO=1
  fi

  spew 3 "Searching for the presence of new MIME data..."
  if [ -d $PKG_STAGEROOT/$PKG_PREFIX/share/mime -o \
       -d $PKG_STAGEROOT/$PKG_PREFIX/share/mime-info ]; then
    spew 4 "  ...found a good indication!"
    PKG_MIMEFILES=1
  fi

  spew 3 "Searching again for the presence of GConf schemas..."
  if [ `find $PKG_SRCROOT . -name *.schemas -o -name *.entries | wc -l 2>&1 ` -ne 0 ]; then
    spew 4 "  ...gconf schemas detected."
    PKG_GCONFTWO=1
  fi

  spew 3 "Checking to see if any icon themes have been stashed in here..."
  if [ `find $PKG_STAGEROOT$PKG_PREFIX/share/icons -type d -maxdepth 1 2>/dev/null | wc -l` -gt 1 ]; then
		spew 4 "  ...found some!"
		PKG_ICONTHEMES=1
	fi

	spew 3 "Checking for any gdk-pixbuf loaders for Gtk+2..."
  if [ `ls $PKG_STAGEROOT$PKG_PREFIX/lib/gtk-2.0/2.4.0/loaders/*.so 2>/dev/null | wc -l` -gt 0 ]; then
		spew 4 "  ...found some!"
		PKG_GTKTWOPIXBUFLOADERS=1
	fi
}

## polishfiles() 
##   This function is called once all the binaries have been placed where they
##   should in the PKG_STAGEROOT.  It only does a few things right now, but will
##   wind up doing many more very soon.
polishfiles() {
  local dirname
  set_title "$NAME-$VERSION-$REVISION: Polishing the files up a bit..."
 
  spew 4 "Making all the files have the same mtime"
  # Touching symlinks _creates_ their targets, which may or may not exist
  # yet or in their proper place so we don't want that to happen.
  find $PKG_STAGEROOT -not -type l -exec touch -t $PKG_TIMESTAMP {} \;

  for dirname in /bin /sbin /usr/bin /usr/sbin /usr/X11R6/bin; do
    if [ -d $PKG_STAGEROOT$dirname ]; then
      spew 4 "Fixing ownerships in $dirname"
      spewdo "chown root.bin $PKG_STAGEROOT$dirname"
      spewdo "find $PKG_STAGEROOT$dirname -type f ! -perm +7000 -exec chown root.bin {} \;"
    fi
  done
}

## tidydocumentation()
##   This function moves any stray documentation from /usr/share/doc to the
##   PKG_DOCDIR directory, and copies the documentation specified by
##   PKG_DOCFILES into the PKG_DOCDIR directory as well.  If it's not a safe
##   assumption that _all_ of our packages have at least one file to put in
##   the docdir, then someone speak up about it.
##
tidydocumentation() {
  local oldpwd=`pwd`

  createdir $PKG_STAGEROOT$PKG_DOCDIR
  set_title "$NAME-$VERSION-$REVISION: Tidying up documentation"
  spew 4 "Tidying up the package documentation..."

  if ( ls $PKG_STAGEROOT$PKG_DATADIR/doc/* &> /dev/null ); then
    # This assumes there's only one directory under doc/
    spew 4 "Moving misplaced files from /usr/share/doc to /usr/doc..."
    spewdo "mv $PKG_STAGEROOT$PKG_DATADIR/doc/*/* $PKG_STAGEROOT$PKG_DOCDIR/"
    spewdo "removedir $PKG_STAGEROOT$PKG_DATADIR/doc"
  fi

  spew 4 "Copying the packages licences, changelogs, etc over..."
  for filename in $PKG_DOCFILES; do
    # We know some of the files will likely not exist, and we don't care.
    spew 5 "cp -adR $PKG_SRCROOT/$filename $PKG_STAGEROOT$PKG_DOCDIR 2>/dev/null"
    cp -adR $PKG_SRCROOT/$filename $PKG_STAGEROOT$PKG_DOCDIR 2>/dev/null
  done

  if [ -r $DLG_ROOT/SCRIPTS/$NAME/ChangeLog ]; then
    spewdo "cp $DLG_ROOT/SCRIPTS/$NAME/ChangeLog $PKG_STAGEROOT$PKG_DOCDIR/ChangeLog.Dropline 2>/dev/null"
  fi
}

## removedir() dirname
##   This acts as a wrapper for rm, to reduce the chances of deadly errors,
##   and to generally make things more readable.
##
removedir() {
  local dirname=${1:?'Internal error.  removedir requires a dirname.'}
  if [ -d $dirname ]; then
    if [ "$dirname" = "/" ]; then
      set_title "Error: suicidal directive encountered"
      die "No, I will not delete the entire filesystem.  Check your code."
    fi
    spew 5 "rm -rf $dirname"
    rm -rf $dirname
  fi
}

## createdir() dirname
##   This acts as an elegant wrapper for mkdir.  New directories will be mode
##   0755, will automatically have parent directories created, and so on.
##
createdir() {
  local dirname=${1:?'Internal error.  removedir requires a dirname.'}
  spew 5 "mkdir -m 755 -p $dirname"
  mkdir -m 755 -p $dirname
  if [ $? != 0 ]; then
    die "mkdir exited with status $? when attempting to create $dirname!"
  fi
}

## opentarball() filename destination
##   This function will change to the destination directory and then open
##   the tarball named.by filename.  All errors are considered fatal.
##   The destination will be assumed to be PKG_WORKROOT.  This will set
##   PKG_SRCROOT on exit if the tarball contains only one directory, and
##   the package maintainer hasn't already defined it using PKG_SRCDIR.
##
opentarball() {
  local filename=${1:?'Internal error, opentarball() requires a filename'}
  local location=${2:-$PKG_WORKROOT}
  local compressor
  local dirname

  # Sanity checks.
  if [ ! -r $filename ]; then
    echo "Unable to read \"$filename\" for opening!"
    exit 1
  fi

  if [ ! -d $location ]; then
    echo "The directory $location does not exist!"
    exit 1
  fi

  # checkmd5sum $filename  ***NYI***

  case $filename in
    *.tar.bz2)
      compressor=bzip2
      ;;
    *.tar.gz)
      compressor=gzip
      ;;
    *.tgz)
      compressor=gzip
      ;;
    *.tar)
      compressor=""
      ;;
    *)
      echo "Unrecognizable compression extention: $filename"
      exit 1
  esac

  spew 3 "Uncompressing $filename into $PKG_WORKROOT."
  set_title "Uncompressing $filename..."
  spewdo "cd $PKG_WORKROOT"
  spew 5 "tar ${compressor:+"--use=$compressor "}-xvf $filename"
  # TODO: Recode this so we can see when untarring something fails!
  dirname=`/bin/tar ${compressor:+"--use=$compressor"} -xvf $filename \
    | awk -F/ '{ print $1 }' | uniq | egrep -v '^.$'`
  if [ $? != 0 ]; then
    die "The untarring of $filename failed!"
  fi

  if [ `echo $dirname | wc -w` -gt 1 ]; then
    spew 4 "No obvious root directory found in tarball."
    $dirname=""
  else
    spew 4 "Tarball root directory appears to be $dirname"
  fi

  # If PKG_SRCDIR is unset, then we will go with the clue from the tarball
  # Note: I'm not 100% convinced this is the "best" approach.  --Dagmar
  if [ -z "$dirname" -a -z "$PKG_SRCDIR" ]; then
   die "Unable to determine PKG_SRCDIR from the tarball contents and nothing ws specified by the package maintainer!"
  else
    spewdo "PKG_SRCROOT=$PKG_WORKROOT/${PKG_SRCDIR:-$dirname}"
    if [ ! -z "$PKG_BUILDDIR" ]; then
      spewdo "PKG_BUILDROOT=$PKG_SRCROOT/$PKG_BUILDDIR"
      createdir $PKG_BUILDROOT
    else 
      spewdo "PKG_BUILDROOT=$PKG_SRCROOT"
    fi
  fi
}

## findtarball() varname
##   This function will go and find the absolute location of the tarball
##   considered appropriate for the package you are building, based on what
##   the contents of NAME and VERSION are.  It will return the answer in the
##   variable you name as varname (don't put a $ in front of it!) and return
##   null ("") if it doesn't find one suitable.  Most of the time this will
##   be invoked for you by prepare().
##
  # Let's be smart about this for a change.  Instead of dancing around trying
  # to add a variable to change what tarball is searched for when the distro
  # package name varies from it in some minor case, let's go ahead and see if
  # we can extract something useable from the PKG_SOURCEURL variable, and if
  # we can then we'll prefer that over anything else.

findtarball() {
  local varname=${1:?'Internal error, findtarball() requires a return var'}
  local _filename
  local filestub
  local _location
  local tmp
  local dirlist
  local filelist

  # First we check the private source repository in DLG_COREREPOS.
  if [ ! -z "$PKG_TARNAME" ]; then
    filestub=$PKG_TARNAME
  elif [ ! -z "$PKG_SOURCEURL" ]; then
    _filename=`echo $PKG_SOURCEURL | awk -F/ '{ print $NF }'`
    filestub=${_filename%.tgz}
    for tmp in $PKG_ZPREF; do 
      filestub=${filestub%.$tmp}
    done
    filestub=${filestub%.tar}
  else
    filestub=$NAME-$VERSION
  fi
  spew 4 "Searching for $filestub source code..."

  stubsearchlist filelist $filestub
  for _filename in $filelist; do
    spew 5 "  ...looking for $DLG_COREREPOS/$NAME/$_filename"
    if [ -r $DLG_COREREPOS/$NAME/$_filename ]; then
      spew 5 "  ...found!"
      _location='$DLG_SOURCEREPOS/$NAME/$_filename'
      break
    fi
  done
  if [ ! -z "$_location" ]; then
    eval $varname=$_location
    return
  fi

  filelist=''
  _filename=''
  # If that fails then we can look in the rest of the local archives
  if [ ! -z "$PKG_TARNAME" ]; then
    filestub=$PKG_TARNAME
  elif [ ! -z "$PKG_SOURCEURL" ]; then
    _filename=`echo $PKG_SOURCEURL | awk -F/ '{ print $NF }'`
    filelist=$_filename
    filestub=${_filename%.tgz}
    for tmp in $PKG_ZPREF; do 
      filestub=${filestub%.$tmp}
    done
    filestub=${filestub%.tar}
  else 
    filestub=$NAME-$VERSION
  fi

  stubsearchlist tmp $filestub
  appenduniqargs filelist $tmp

  for tmp in $DLG_SOURCEREPOS; do
    for _filename in $filelist; do
      spew 5 "  ...looking for $tmp/$_filename"
      if [ -r "$tmp/$_filename" ]; then
        spew 5 "  ...found!"
        _location="$tmp/$_filename"
        break
      fi
    done
    if [ ! -z "$_location" ]; then
      break
    fi
  done

  # Kind of hackish here...  --Dagmar
  if [ -z $_location ]; then
    if [ ! -z "$PKG_SOURCEURL" ]; then
			if [ "$DLG_ORDERLYSOURCES" = "" ]; then
      	lastarg dirname $DLG_SOURCEREPOS
			else
				dirname=$PKG_COREREPOS
			fi
      downloadfile $dirname $PKG_SOURCEURL
      if [ $? = 0 ]; then
        stubname=`echo $PKG_SOURCEURL | awk -F/ '{ print $NF }'`
        _location=$dirname/$stubname
      else
        return 1
      fi
    else 
      return 1
    fi 
  fi
  eval $varname=$_location
}

## stubsearchlist() varname [stubname]
##   This routine dictates the order in which various permutations of a tar
##   archive's name are searched for.  If stubname is not given, it is assumed
##   to be $PKG_TARNAME, or failing that, $NAME-$VERSION.  It returns the list
##   in the variable indicated by varname.
##
stubsearchlist() {
  local _varname=${1:?'Internal error: stubsearchlist() requires a varname.'}
  local _stubname=${2:-${PKG_TARNAME:-"$NAME-$VERSION"}}
  local list
  local comptype

  for comptype in $PKG_ZPREF; do
    case $comptype in
      bz2)
        appenduniqargs list "$_stubname.tar.bz2"
        ;;
      gz)
        appenduniqargs list "$_stubname.tar.gz"
        appenduniqargs list "$_stubname.tgz"
        ;;
      *)
        warn "Unrecognized compression type found in \$PKG_ZPREF!"
        ;;
    esac
  done

  eval $_varname='$list'
}

## lastarg() varname itemlist
##   This routine is just for having a visually tidy way of getting the last
##   item from a space-separated list.
##
lastarg() {
  local varname=${1:?'Internal error.  lastarg() requires a varname'}
  shift
  local list=${*:?'Internal error.  lastarg() requires some list items'}
  local item=`echo $list | awk '{ print $NF }'`
  eval $varname='$item'
}

## firstarg() varname itemlist
##   This routine is just for having a visually tidy way of getting the first
##   item from a space-separated list.
##
firstarg() {
  local varname=${1:?'Internal error.  firstarg() requires a varname'}
  shift
  local list=${*:?'Internal error.  firstarg() requires some list items'}
  local item=`echo $list | awk '{ print $1 }'`
  eval $varname='$item'
}

## appendarg() varname [item]
##   This routine will append $item to the end of the variable indicated by
##   varname (which is assumed to be a space-separated list).  
##
appendarg() {
  local _varname=${1:?'Internal error.  appendarg() requires a varname'}
  local _item=$2
  local _arglist

  if [ -z "$_item" ]; then
    return # ...because we have nothing to add.
  fi

  eval _arglist=\$$_varname
  eval $_varname=\'${_arglist:+"$_arglist "}$_item\'
}

## appenduniqargs() varname items
##   This routine will add items to the end of the variable indicated by
##   varname (with it's contents treated as a space-separated list) provided
##   that the items _do not already exist in varname_.
##
appenduniqargs() {
  local _varname=${1:?'Internal error.  appenduniqargs() requires a varname'}
  shift
  local items=$*
  local matchcount
  local item
  local dupe
  local tmp
  local newlist
  local oldlist

  if [ -z "$items" ]; then
    return # ...because we have nothing to add.
  fi

  eval oldlist=\$$_varname
  tmp="$oldlist${oldlist:+" "}$items"

  for item in $tmp; do
    matchcount=0
    for dupe in $newlist; do
      if [ "$item" = "$dupe" ]; then
        (( matchcount++ ))
      fi
    done
    if [ $matchcount -eq 0 ]; then
      appendarg newlist $item
    fi
  done

  eval $_varname='$newlist'
}

## findfile() varname filename [md5sum]
##   This is a pretty simple routine.  It will look in the repository
##   directories for a given file.  If you supply an md5sum, it will check
##   more carefully and only return a hit if the file has that checksum.
##   
findfile() {
  local ff_varname=${1:?'Internal error: findfile() requires a varname.'}
  local ff_filename=${2:?'Internal error: findfile() requires a filename.'}
  local ff_md5sum=$3
  local ff_location
  local ff_match

  for ff_location in $PKG_COREREPOS $DLG_SOURCEREPOS; do
    if [ -r $ff_location/$ff_filename ]; then
      if [ -z "$ff_md5sum" ]; then
        ff_match="$ff_location/$ff_filename"
        break 
      fi
      checkmd5sum $ff_location/$ff_filename $ff_md5sum
      if [ $? = 0 ]; then
        ff_match="$ff_location/$ff_filename"
        break
      fi
    fi
  done
  
  if [ -z "$ff_match" ]; then
    return 1
  fi
  eval $ff_varname='$ff_match'
}

## obtainfile() varname URL [md5sum]
##   This is a more heavy-handed routine for obtaining source files.  It will
##   first search to see if the file is already present (checking for a match
##   on the md5sum as it goes) and if not, it will download the file and in 
##   either case return false, or put the new location of the file into
##   the variable indicated by varname.
##
obtainfile() {
  local of_varname=${1:?'Internal error: obtainfile() requires a varname.'}
  local of_url=${2:?'Internal error: downloadfile() requires a URL.'}
  local of_md5sum=$3

  local of_filename=`echo $of_url | awk -F/ '{ print $NF }'`
  local of_location
  
  # Do we already have the file?
  findfile of_location $of_filename $of_md5sum
  if [ $? = 0 ]; then
    eval $of_varname=$of_location
    return
  fi
  spew 4 "Did not find $of_filename locally stored..."

  # no?  ...then we'll try and get it.
  lastarg of_location "$PKG_COREREPOS $DLG_SOURCEREPOS"
  echo "$of_location $of_url filename=$of_filename md5sum=$of_md5sum"
  downloadfile $of_location $of_url filename=$of_filename md5sum=$of_md5sum
  if [ $? != 0 ]; then
    die "Unable to download $url!"
  fi

  eval $of_varname='$of_location'
}

## downloadfile() directory URL [filename=] [md5sum=]
##   This routine will merely try to download a URL from the internet and
##   store it in the directory you specify.  If the URL doesn't end in the
##   filename you prefer, you may optionally specify a new filename for the
##   file to be stored as.  
##
downloadfile() {
  local dirname=${1:?'Internal error: downloadfile() requires a directory.'}
  local url=${2:?'Internal error: downloadfile() requires a URL.'}
  shift
  shift
  local filename=`echo $url | awk -F/ '{ print $NF }'`
  local md5sum
  local command
  local tmp
  local oldpwd=`pwd`

  spew 4 "Downloading $url to $dirname"

  while [ ! -z "$*" ]; do
    case $1 in
      filename=*)
        eval $1
        spew 4 "  ...forced filename to $filename"        
      ;;
      md5sum=*)
        eval $1
        spew 4 "  ...checking md5um matches $md5sum"
      ;;
      *)
        die "Unrecognized optional argument passed to downloadfile()!"
    esac
    shift
  done	

  command="$PKG_WGET${PKG_WGETARGS:+" "}$PKG_WGETARGS"
  command="$command --output-document=$dirname/$filename $url"

  set_title "$NAME-$VERSION-$REVISION: Downloading $filename"

  spew 2 "Attempting to download $filename from the Internet..."
  spew 4 "Source URL: $url"
  spew 4 "Destination: $dirname"

  loggedcommand download-$filename "$command"
  status=$?

  if [ $status != 0 ]; then
    spew 4 "Download failed.  Destroying temporary file."
    spewdo "rm -f $dirname/$filename"
    die "Download of $url failed!"
  else 
    spew 2 "Download successful."
  fi

  if [ ! -z "$md5sum" ]; then
    spew 4 "Validation of downloaded $filename by md5sum requested."
    checkmd5sum $dirname/$filename $md5sum
    if [ $? != 0 ]; then
      die "$url was downloaded, but the md5sum check failed!"
    fi
    spew 3 "Downloaded file passes md5sum check."
  fi

  return $status
}

## gconfoneneeded() - This just sets a few things so that gconf packages with
##   gconf-1 schemas in /etc/gconf are handled properly.
##
gconfoneneeded() {
  PKG_GCONFONE=1
}

## gconftwoneeded() - This sets a few things so that packages with gconf-2
##   schemas will be handled properly.
##
gconftwoneeded() {
  PKG_GCONFTWO=1
}

## nocflags() - Just a more attactive way of deactivating customized CFLAGS.
##
nocflags() {
  PKG_NOCFLAGS=1
  unset PKG_CFLAGS
  spew 3 "Custom CFLAGS for this package have disabled by package definition."
}

## nocxxflags() - C++ analogue of nocflags
##
nocxxflags() {
  PKG_NOCXXFLAGS=1
  unset PKG_CXXFLAGS
  spew 3 "Custom CXXFLAGS for this package have been disabled by package definition."
}

## cxxflags() [setting]
##   A fancy way of setting CXXFLAGS.  If no argument is given, it will use
##   the same routine as is used for building CFLAGS.  If an argument is
##   given, it will use it's value.
##
cxxflags() {
  local args=$*
  local tmp
  if [ ! -z "$args" ]; then
    PKG_CXXFLAGS="$args"
    spew 4 "Custom CXXFLAGS set to \"$PKG_CXXFLAGS\" by package definition."
  else 
    if [ "$PKG_NOCXXFLAGS" -o "$DLG_NOCXXFLAGS" ]; then
      spew 4 "Custom CXXFLAGS settings are disabled for this package."
    else
      # These allow you granular replacement of the arguments
      PKG_OPTLEVEL=${PKG_OPTLEVEL:-$DLG_OPTLEVEL}
      PKG_ARCH=${PKG_ARCH:-$DLG_ARCH}
      PKG_CPU=${PKG_CPU:-$DLG_CPU}

      # ...and this builds the argument _cleanly_
      tmp=$PKG_OPTLEVEL
      tmp=$tmp${PKG_CPU:+"${tmp:+" "}-mcpu=$PKG_CPU"}
      PKG_CXXFLAGS=$tmp${PKG_ARCH:+"${tmp:+" "}-march=$PKG_ARCH"}
      spewdo "PKG_CXXFLAGS=\"$PKG_CXXFLAGS\""
    fi
  fi
}

## cflags() [setting]
##   A fancy way of setting CFLAGS.  If no argument is given, it will attempt
##   to create a CFLAGS setting based on the global or local defaults, unless
##   PKG_NOCFLAGS or DLG_NOCFLAGS are set.  Any argument given will override
##   PKG_NOCFLAGS and DLG_NOCFLAGS and set CFLAGS for the build explicitly to
##   the argument list.
##
cflags() {
  local args=$*
  local tmp
  if [ ! -z "$args" ]; then
    PKG_CFLAGS="$args"
    spew 4 "Custom CFLAGS set to \"$PKG_CFLAGS\" by package definition."
  else 
    if [ "$PKG_NOCFLAGS" -o "$DLG_NOCFLAGS" ]; then
      spew 4 "Custom CFLAGS settings are disabled for this package."
    else
      # These allow you granular replacement of the arguments
      PKG_OPTLEVEL=${PKG_OPTLEVEL:-$DLG_OPTLEVEL}
      PKG_ARCH=${PKG_ARCH:-$DLG_ARCH}
      PKG_CPU=${PKG_CPU:-$DLG_CPU}


      # This builds the argument _cleanly_, and let's not optimize it any
      # further.  It's already unreadable enough.
      tmp=$PKG_OPTLEVEL
      tmp=$tmp${PKG_CPU:+"${tmp:+" "}-mcpu=$PKG_CPU"}
      PKG_CFLAGS=$tmp${PKG_ARCH:+"${tmp:+" "}-march=$PKG_ARCH"}
      spewdo "PKG_CFLAGS=\"$PKG_CFLAGS\""
    fi
  fi
}

## loggedcommand() logname command
##   This is the core of the logging system.  The first argument is the file
##   into which the command's output will be logged.  The remaining arguments
##   will be executed in a subshell.  It is suggested that you pass the command
##   as one long double-quoted string to avoid confusion.  Try not to think
##   about the way it logs commands--you might need have a lie down.
##
loggedcommand() {
  local target=${1:?'Internal error.  loggedcommand() requires a target.'}
  shift
  local command=${*:?'Internal error.  loggedcommand() requires a command.'}
  local outfile="$PKG_LOGROOT/$target.txt"
  local cmdfile="$PKG_LOGROOT/$target.sh"
  local kludge="$PKG_LOGROOT/$target.exited"
  local status

  spew 4 "Executing \"$target\" in $cmdfile, output in $outfile"
  echo "$command" > $cmdfile
  if [ $? != 0 ]; then
    die "Error!  Unable to create temporary command script."
  fi

  spew 5 "$command"
  if [ "$DLG_VERBOSITY" -ge 3 ]; then
    { sh $cmdfile ; echo $? > $kludge ; } 2>&1 | tee -a $outfile
  else
    { sh $cmdfile ; } 2>&1 >> $outfile
    echo $? > $kludge
  fi
  status=`cat $kludge`
  rm $kludge
  return $status
}

## make() [target]
##   This runs a make in the PKG_SRCROOT.  It will take an optional target
##   for when you have to do something other than just "make".  Don't use
##   this for a "make install" except as a last resort.
##
make() {
  local target=$*
  local oldpwd=`pwd`
  local command='make'
  local status

  set_title "$NAME-$VERSION-$REVISION: make phase"
  spew 4 "Make arguments supplied are \"$target\"."
  spew 2 "Running the make..."

  spewdo "cd $PKG_BUILDROOT"
  command="$command${target:+" "}$target"
  loggedcommand make "$command"
  status=$?
  if [ $status != 0 ]; then
    die "Error!  make exited with status: $status"
  fi
  cd $oldpwd
}

## makecheck() [check|test|whatever]
##   This routine is meant to run the makefile's self-test, usually called
##   via the check (default) or test targets.  The idea is that if something
##   is really, _really_ wrong with the build then this target will fail and
##   exit with non-zero status.  It is not always an indicator of sanity,
##   since seme package's self-tests are broken, and some require the package
##   to already be installed to function properly, but if a previous build
##   completed this step successfully and a new one doesn't, you generally
##   have cause for alarm.  The default target is 'check' but will be
##   overridden if you supply an argument.
##
makecheck() {
  local target=${*:-"check"}
  local oldpwd=`pwd`
  local command='make'
  local status

  set_title "$NAME-$VERSION-$REVISION: make phase"
  spew 4 "Make self-test invoked using target \"$target\"."
  spew 2 "Running the makefile's self-test..."

  spewdo "cd $PKG_BUILDROOT"
  command="$command${target:+" "}$target"
  loggedcommand makecheck "$command"
  status=$?
  if [ $status != 0 ]; then
    die "Error!  Build sanity check make $target exited with status: $status"
  fi
  cd $oldpwd
}

## makeinstall() [target] 
##   This runs a packagizing `make install` in the PKG_SRCROOT which will
##   assume the proper make target is "install" unless you specify another
##   target.  It will use the PKG_WORKROOT
##   a new one.  This function will probably change if we need more than a few
##   install targets that are not named 'install'.
##
makeinstall() {
  local target=${*:-"install"}
  local oldpwd=`pwd`
  local command="make $PKG_DESTDIRVAR=$PKG_STAGEROOT $target"

  set_title "$NAME-$VERSION-$REVISION: make install phase"
  spew 2 "Running the make install, with $PKG_DESTDIRVAR=$PKG_STAGEROOT..."

  spewdo "cd $PKG_BUILDROOT"
  loggedcommand makeinstall "$command"
  if [ $status != 0 ]; then
    die "Error!  Configure exited with status: $status"
  fi
  cd $oldpwd
}

## configure() [arguments]
##   This will cd to PKG_SRCROOT and execute a somewhat automagically generated
##   ./configure command.  It will append any arguments you specify to the end
##   of the automagically generated list, and prepend any CFLAGS or CXXFLAGS
##   settings (based on PKG_C*FLAGS contents) to the command.
##
configure() {
  local arglist=$*
  local oldpwd=`pwd`
  local envvars 
  local switches
  local command

  # What goes before ./configure
  configenvs envvars

  # What follows ./configure
  configargs switches

  command="$envvars"

  if [ -z "$PKG_USEAUTOGENSH" ]; then
    appendarg command "${PKG_SRCREL}/configure"
  else
    appendarg command "${PKG_SRCREL}/autogen.sh"
  fi
  appendarg command "$switches"
  appendarg command "$arglist"
  spew 4 "Generated configure line is '$command'."

  spewdo "cd $PKG_BUILDROOT"
  spew 2 "Running the configure script..."
  set_title "$NAME-$VERSION-$REVISION: configure phase"
  loggedcommand configure "$command"
  status=$?
  if [ $status != 0 ]; then
    die "Error!  Configure exited with status: $status"
  fi
  cd $oldpwd
}

## configenvs() varname
##   This routine will generate the environment variables that are typically
##   fed to GNU-style configure scripts (like CFLAGS, for instance).  See
##   configargs() for why this function exists.
##
configenvs() {
  local varname=${1:?'Internal error: configargs() requires a variable name.'}
  local tmp

  appendarg tmp ${PKG_CXXFLAGS:+"CXXFLAGS=\"$PKG_CXXFLAGS\""}
  appendarg tmp ${PKG_CFLAGS:+"CFLAGS=\"$PKG_CFLAGS\""}
  spew 4 "Generated environment variables for configure are '$tmp'."

  eval $varname='$tmp'
}

## configargs() varname
##   This routine will generate the arguments typically fed to GNU-style
##   configure scripts.  This was broken out into it's own routine so that if
##   you wish to have the usual run of arguments, but would like to exclude
##   one or two specific ones from the list, you can use this to get it, do
##   whatever you like to it, and then 
##
configargs() {
  local varname=${1:?'Internal error: configargs() requires a variable name.'}
  local tmp

  appendarg tmp ${PKG_PREFIX:+"--prefix=$PKG_PREFIX"}
  appendarg tmp ${PKG_SYSCONFDIR:+"--sysconfdir=$PKG_SYSCONFDIR"} 
  appendarg tmp ${PKG_LOCALSTATEDIR:+"--localstatedir=$PKG_LOCALSTATEDIR"} 
  appendarg tmp ${PKG_GCONFTWO:+'--disable-schemas-install'}
  if [ -z "$PKG_NEEDSTATIC" ]; then
    appendarg tmp "--disable-static"
  fi
  spew 4 "Generated configure arguments are '$tmp'."

  eval $varname='$tmp'
}


# Set the defaults on the global variables
DLG_SCRATCHROOT=${DLG_SCRATCHROOT:-"/tmp/DLG"}
DLG_COREREPOS=${DLG_COREREPOS:-"$DLG_ROOT/SOURCES"}
DLG_LOGROOT=${DLG_LOGROOT:-"$DLG_SCRATCHROOT/LOGS"}
DLG_WORKROOT=${DLG_WORKROOT:-"$DLG_SCRATCHROOT/WORK"} 
DLG_STAGEROOT=${DLG_STAGEROOT:-"$DLG_SCRATCHROOT/STAGE"}
DLG_KEEPLOGS=${DLG_KEEPLOGS:-"1"}
DLG_VERBOSITY=${DLG_VERBOSITY:-"0"}
DLG_ZPREF=${DLG_ZPREF:-"bz2 gz"}
DLG_WGET=${DLG_WGET:-"`which wget`"}
DLG_WGETARGS='-c --tries=3 --random-wait'
DLG_PREFIX=${DLG_PREFIX:-"/usr"}
DLG_SYSCONFDIR=${DLG_SYSCONFDIR:-"/etc"}
DLG_LOCALSTATEDIR=${DLG_LOCALSTATEDIR:-"/var/lib"}
DLG_DATADIR=${DLG_DATADIR:-"$DLG_PREFIX/share"}
DLG_OPTLEVEL=${DLG_OPTLEVEL:-"-O2"}
DLG_ARCH=${DLG_ARCH:-"i686"}
DLG_CPU=${DLG_CPU:-""} # ...in case we decide to use it later --Dagmar
DLG_DESTDIRVAR=${DLG_DESTDIRVAR:-"DESTDIR"}
DLG_DOCDIR=${DLG_DOCDIR:-"$DLG_PREFIX/doc"}
DLG_DOCFILES=${DLG_DOCFILES:-"ABOUT* ANNOUNCEMENT API AUTHORS BUGS ChangeLog* CHANGES Copyright COPYRIGHT COPYING* FAQ HACKING INSTALL* LICEN?E MAINTAINERS NEWS* README* STYLEGUIDE TODO* THANKS USAGE VERSION"}
DLG_PACKAGEROOT=${DLG_PACKAGEROOT:-"$DLG_SCRATCHROOT/PACKAGES"}
DLG_PKGER=${DLG_PKGER:-"dl"}
DLG_SRCREL=${DLG_SRCREL:-"."}
DLG_SFMIRROR=${DLG_SFMIRROR:-"easynews"}
DLG_SOURCEREPOS=${DLG_SOURCEREPOS:-"/usr/src/pc/SOURCES /space/tarballs"}

# Now to make those defaults "flow down" into the package-specific variables
PKG_KEEPLOGS=${PKG_KEEPLOGS:-$DLG_KEEPLOGS}
PKG_LOGROOT=${PKG_LOGROOT:-$DLG_LOGROOT}
PKG_VERBOSITY=${PKG_VERBOSITY:-$DLG_VERBOSITY}
PKG_COREREPOS=${PKG_COREREPOS:-"$DLG_COREREPOS/$NAME"}
PKG_ZPREF=${PKG_ZPREF:-$DLG_ZPREF}
PKG_WGET=${PKG_WGET:-$DLG_WGET}
PKG_WGETARGS=${PKG_WGETARGS:-$DLG_WGETARGS}
PKG_PREFIX=${PKG_PREFIX:-$DLG_PREFIX}
PKG_SYSCONFDIR=${PKG_SYSCONFDIR:-$DLG_SYSCONFDIR}
PKG_LOCALSTATEDIR=${PKG_LOCALSTATEDIR:-$DLG_LOCALSTATEDIR}
PKG_DATADIR=${PKG_DATADIR:-$DLG_DATADIR}
PKG_OPTLEVEL=${PKG_OPTLEVEL:-$DLG_OPTLEVEL}
PKG_ARCH=${PKG_ARCH:-$DLG_ARCH}
PKG_CPU=${PKG_CPU:-$DLG_CPU}
PKG_DOCDIR=${PKG_DOCDIR:-"$DLG_DOCDIR/$NAME-$VERSION"}
PKG_DOCFILES=${PKG_DOCFILES:-$DLG_DOCFILES}
PKG_PKGER=${PKG_PKGER:-$DLG_PKGER}
PKG_SRCREL=${PKG_SRCREL:-$DLG_SRCREL}
PKG_SFMIRROR=${PKG_SFMIRROR:-$DLG_SFMIRROR}

## Processor/Architecture-specific fixups for Athlon64
if [ "$PKG_ARCH" = "x86-64" -o "$PKG_CPU" = "x86-64" ]; then
  addarg PKG_OPTLEVEL "-fPIC"
fi

# This shouldn't be able to affect anything except compilation speed.
if [ -z "$PKG_GCCNOPIPE" ]; then
  spew 4 "Adding -pipe to PKG_OPTLEVEL"
  addarg PKG_OPTLEVEL "-pipe"
fi

## Package-specific variable documentation
##
## PKG_WORKROOT - $DLG_WORKROOT/$NAME-$VERSION-$REVISION
PKG_WORKROOT="$DLG_WORKROOT/$NAME-$VERSION-$REVISION"
##   This is a subdirectory of the work root so that we can build more than
##   one package at a time without risking them getting tangled up.  This is
##   scoped to the specific package being built.
##
## PKG_DESTDIRVAR - "DESTDIR" (all you normally need)
PKG_DESTDIRVAR=${PKG_DESTDIRVAR:-$DLG_DESTDIRVAR}
##   This variable is passed to makefiles to change the root directory that
##   files are installed to.  If your package needs something like "prefix"
##   instead of DESTDIR, this is the appropriate place to set it.
##
## PKG_STAGEROOT - $DLG_STAGEROOT/$NAME-$VERSION-$REVISION
PKG_STAGEROOT="$DLG_STAGEROOT/$NAME-$VERSION-$REVISION"
##   In package scope, this variable controls where the compiled binary
##   files will be put to prepare for packagizing.  You should never need
##   to change this directly.
##
## PKG_SRCROOT - set by the package maintainer
##   This directory should be the absolute path of the top of the source
##   tree, and only needs to be set if the directory name does not match
##   the pattern $NAME-$VERSION, or if the tarball contains more than one
##   top-level directory or file.
##
## PKG_SRCDIR - normally set by the opentarball() function
##   Shoud the autoselection of PKG_SRCDIR fail or be unuseable (generally
##   when a tarball doesn't have an exact match of $NAME-$VERSION as it's
##   top directory you may specify a relative path from $PKG_SRCROOT here
##   and it will override the auto-selection mechanism of opentarball().
##   (Basically if foo-1.1.tgz contains "foodir/" instead of "foo-1.1/" set
##   this variable to "foodir")
##
## PKG_LOGROOT - $DLG_LOGROOT/$NAME-$VERSION-$REVISION
PKG_LOGROOT="$DLG_LOGROOT/$NAME-$VERSION-$REVISION"
##   This is a subdirectory of the log root so that we can keep track of what
##   we've done before so that we can look for problems later or compare
##   build outputs without having to redo everything.
##
## PKG_TIMESTAMP - DLG_TIMESTAMP or `date +%Y%m%d%H%M.%S`
PKG_TIMESTAMP=${DLG_TIMESTAMP:-"`date +%Y%m%d%H%M.%S`"}
##   This timestamp is used by the polishfiles() function to reset the mtimes
##   of each file to a given time.  This adds yet another shortcut for hinting
##   at which files came from which build of what.  If you don't set this, it
##   will be set on an per-package basis.
## 
## PKG_DISKSET - optionally set by the package maintainer
#PKG_DISKSET='dl'
##   This variable is here because eventually we may wind up breaking these
##   packages down into separate disksets.  This will let us do that.
##
## PKG_PACKAGEROOT - modified by PKG_DISKSET
PKG_PACKAGEROOT="$DLG_PACKAGEROOT${PKG_DISKSET:+"/$PKG_DISKSET"}"
##   This variable will be the same as DLG_PACKAGEROOT if PKG_DISKSET is
##   null or unset.  It is the directory where all completed binary packages
##   and their associated signature files are stored upon completion.
## 
## PKG_COMPILE_DIR - Normally unset.
##   This is the working directory from within which the build commands (such
##   as configure, make, etc) should be executed, relative to PKG_SRCROOT.
##
## PKG_SOURCE_REL - Normally "./"
##   This is a relative path to the configure script from PKG_COMPILE_DIR.
##   Normally you only need to change this when the directory you're running
##   commands in isn't the one that contains the packager's build scripts
##   (like configure and the makefiles) or if you're building outside the
##   source root (as is suggested for packages like binutils).
##
## PKG_NEEDSTATIC - Null if --disable-static is useable
##   Normally we pass --disable-static to configure by default, but this
##   breaks some builds rather horribly.  Define this to something if you
##   actually need the package to build static libraries.
##
## PKG_USEAUTOGENSH - Normally null.
##   Set this to a non-null value if the source package needs autogen.sh run
##   instead of the normal configure script during the configure stage.
##
## The following booleans may be set if you know the analysis functions won't
## catch on to the presence of certain facilities belonging to a package.
## (or if you don't trust the automagic package analyzer)
##
## PKG_SCROLLKEEPER - Scrollkeeper
## PKG_GNUINFO - GNU info files
## PKG_GCONFONE - GConf 1.x, from older Gnome releases mainly ***NYI***
## PKG_GCONFTWO - Gconf 2.x, the current GConf release
## PKG_DESKTOPFILES - Gnome/KDE/FDO *.desktop files


# Make spews easier to see
bold=`tput bold`
boff=`tput sgr0`
bell=`tput bel`
cblu=`tput setf 1`
cgrn=`tput setf 2`
ccyn=`tput setf 3`
cred=`tput setf 4`
cmag=`tput setf 5`
cyel=`tput setf 6`
cwht=`tput setf 7`

if [ ! -z $DLG_CFLAGS ]; then
  spew 4 "DLG_CFLAGS setting detected.  Don't say we didn't warn you!"
fi

if [ ! -z $DLG_CXXFLAGS ]; then
  spew 4 "DLG_CXXFLAGS setting detected.  Don't say we didn't warn you!"
fi

# Set some other variables for utility purposes
VER_MAJ=`echo $VERSION | awk -F. '{ print $1 }'`
VER_MIN=`echo $VERSION | awk -F. '{ print $2 }'`
VER_MIC=`echo $VERSION | awk -F. '{ print $3 }'`
VER_TEN=`echo $VERSION | awk -F. '{ print $4 }'`

# Process the PKG_SOURCEURL list
if [ ! -z $PKG_SOURCEURL ]; then
  spewdo "PKG_SOURCEURL=$PKG_SOURCEURL"
fi

# vim: ts=2
