#!/bin/sh
#
# Copyright 2002-2003 Todd Kulesza
#
# The Dropline GNOME Installation System is free software.
# You can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
#

# file information
NAME=$1
VERSION=$2
REVISION=$3
CONFIG_FLAGS=$4
MAKE_FLAGS=$5
MAKEINSTALL_FLAGS=$6
MAKEINSTALL_METHOD=$7
SOURCE=$8
USR=$9
ETC=${10}
LIB=${11}
VAR=${12}
INC=${13}

# build directories
BASEDIR=/usr/src/gnome
SRCDIR=$BASEDIR/sources
SCRIPTDIR=$BASEDIR/scripts
PACKAGEDIR=$BASEDIR/packages
BUILDDIR=$BASEDIR/builds
PKGBUILDDIR=$BUILDDIR/$NAME-$VERSION
INSTALLDIR=$PKGBUILDDIR/install

# fix some problems with libtool
export SED=sed

# set the install prefixes
if [ "$USR" == "" ]; then
	USR=/usr
fi

if [ "$ETC" == "" ]; then
	ETC=/etc
fi

if [ "$LIB" == "" ]; then
	LIB=/usr/lib
fi

if [ "$VAR" == "" ]; then
	VAR=/var/lib
fi

if [ "$SOURCE" == "" ]; then
	SOURCE=$NAME-$VERSION
fi

# we use exported variables here because I got tired of adding to the parameter list ;)
if [ "$DROPLINE_ARCH" == "" ]; then
	ARCH=i686
else
	ARCH=$DROPLINE_ARCH
fi
echo "Building for architecture $ARCH."

# doc files
DOC_FILES=
DOCDIR=$PKGBUILDDIR/usr/doc/$NAME-$VERSION

# clean up the build directories
if [ -d $PKGBUILDDIR ]; then
	echo "Deleting $PKGBUILDDIR..."
	rm -r $PKGBUILDDIR
fi
echo "Creating $PKGBUILDDIR..."
mkdir -p $PKGBUILDDIR

# prep the source directory and untar the package
cd $SRCDIR
if [ -d $SRCDIR/$SOURCE ]; then
	echo "Deleting $SRCDIR/$SOURCE..."
	rm -r $SRCDIR/$SOURCE
fi

echo "attempting to un-archive $SOURCE..."
if [ -e $SRCDIR/$SOURCE.tar.gz ]; then
	tar zxpfv $SRCDIR/$SOURCE.tar.gz
elif [ -e $SRCDIR/$SOURCE.tar.bz2 ]; then
	tar jxpfv $SRCDIR/$SOURCE.tar.bz2
elif [ -e $SRCDIR/$SOURCE.tgz ]; then
	tar zxpfv $SRCDIR/$SOURCE.tgz
else
	echo "$SOURCE could not be found in a known archive format."
	exit 3
fi

cd $SOURCE

# if a patch exists, apply it
PATCH=1
while [ -e $SCRIPTDIR/$NAME.patch$PATCH ]
do
	echo "Applying $NAME.patch$PATCH..."
	patch -p0 < $SCRIPTDIR/$NAME.patch$PATCH
	
	if [ "$?" != "0" ]; then
		exit $?
	fi

	((PATCH += 1))
done

# oh, silly abiword...
if [ "$NAME" == "abiword" ]; then
	cd abi
# and silly db...
elif [ "$NAME" == "db4" ]; then
	cd dist
fi

# run any pre-build commands
if [ -e $SCRIPTDIR/$NAME.pre ]; then
	echo "Running $SCRIPTDIR/$NAME.pre..."
	/bin/sh $SCRIPTDIR/$NAME.pre $PKGBUILDDIR $SCRIPTDIR $SRCDIR $NAME $VERSION
	
	if [ "$?" != "0" ]; then
		exit $?
	fi
fi

# run the configure script

# avoid any errors
if [ "$ARCH" == "noarch" ]; then
	export CFLAGS="-O2 -march=i686 $MAKE_FLAGS"
else
	export CFLAGS="-O2 -march=$ARCH $MAKE_FLAGS"
fi

if [ -e ./configure ]; then
	./configure \
		--prefix=$USR \
		--sysconfdir=$ETC \
		--localstatedir=$VAR \
		$CONFIG_FLAGS

	if [ "$?" != "0" ]; then
		exit $?
	fi
fi

# run any post-configure commands
if [ -e $SCRIPTDIR/$NAME.postconfig ]; then
	echo "Running $SCRIPTDIR/$NAME.pre..."
	/bin/sh $SCRIPTDIR/$NAME.postconfig $PKGBUILDDIR $SCRIPTDIR $SRCDIR $NAME $VERSION
	
	if [ "$?" != "0" ]; then
		exit $?
	fi
fi

if [ -e Makefile ]; then
	make
	
	if [ "$?" != "0" ]; then
		echo "Exiting with error code $?"
		exit $?
	fi
fi

if [ "$MAKEINSTALL_METHOD" == "NORMAL" ]; then
	echo ""
	echo "Installing via PREFIX..."
	echo ""
	make install \
		prefix=$PKGBUILDDIR$USR \
		sysconfdir=$PKGBUILDDIR$LIB \
		localstatedir=$PKGBUILDDIR$VAR \
		LIBRARY_PATH=$PKGBUILDDIR$LIB \
		$MAKEINSTALL_FLAGS
elif [ "$MAKEINSTALL_METHOD" == "AUTOMAKE" ]; then
	echo ""
	echo "Installing via AUTOMAKE..."
	echo ""
	make install \
		AM_MAKEFLAGS="DESTDIR=$PKGBUILDDIR LIBRARY_PATH=$PKGBUILDDIR$LIB" \
		$MAKEINSTALL_FLAGS
elif [ "$MAKEINSTALL_METHOD" == "DESTDIR" ]; then
	echo ""
	echo "Installing via DESTDIR..."
	echo ""
	make install DESTDIR=$PKGBUILDDIR
elif [ "$MAKEINSTALL_METHOD" == "BUILDROOT" ]; then
	echo ""
	echo "Installing via BUILDROOT..."
	echo ""
	make BUILDROOT=$PKGBUILDDIR install
elif [ "$MAKEINSTALL_METHOD" == "INSTALLROOT" ]; then
	echo ""
	echo "Installing via INSTALLROOT..."
	echo ""
	make install INSTALLROOT="$PKGBUILDDIR""/usr"
elif [ "$MAKEINSTALL_METHOD" == "NONE" ]; then
	echo ""
fi

if [ "$?" != "0" ]; then
	exit $?
fi

# install the documentation
echo "Copying documentation to $DOCDIR..."
mkdir -p $DOCDIR
if [[ -e ABOUT-NLS && -s ABOUT-NLS ]]; then
	DOC_FILES="$DOC_FILES ABOUT-NLS"
fi
if [[ -e AUTHORS && -s AUTHORS ]]; then
	DOC_FILES="$DOC_FILES AUTHORS"
fi
if [[ -e COPYING && -s COPYING ]]; then
	DOC_FILES="$DOC_FILES COPYING"
fi
if [[ -e COPYING.LIB && -s COPYING.LIB ]]; then
	DOC_FILES="$DOC_FILES COPYING.LIB"
fi
if [[ -e COPYRIGHT && -s COPYRIGHT ]]; then
	DOC_FILES="$DOC_FILES COPYRIGHT"
fi
if [[ -e ChangeLog && -s ChangeLog ]]; then
	DOC_FILES="$DOC_FILES ChangeLog"
fi
if [[ -e HACKING && -s HACKING ]]; then
	DOC_FILES="$DOC_FILES HACKING"
fi
if [[ -e NEWS && -s NEWS ]]; then
	DOC_FILES="$DOC_FILES NEWS"
fi
if [[ -e README && -s README ]]; then
	DOC_FILES="$DOC_FILES README"
fi
if [[ -e TODO && -s TODO ]]; then
	DOC_FILES="$DOC_FILES TODO"
fi
cp -a $DOC_FILES $DOCDIR/
if [ -d $PKGBUILDDIR/usr/doc/$NAME ]; then
	cp -r $PKGBUILDDIR/usr/doc/$NAME/* $DOCDIR/
	rm -rf $PKGBUILDDIR/usr/doc/$NAME &> /dev/null
fi
chmod 644 $DOCDIR/*
chown root.root $DOCDIR/*

# remove gconf database defaults, we install these through gconftool
# with the installation script
if [ -d $PKGBUILDDIR/etc/gconf/gconf.xml.defaults ]; then
	echo "Removing application-specific GConf database..."
	rm -r $PKGBUILDDIR/etc/gconf/gconf.xml.defaults
fi

# remove scrollkeeper database, we install these through scrollkeeper-
# update with the installation script

if [ -d $PKGBUILDDIR/var/lib/scrollkeeper ]; then
	echo "Removing application-specific scrollkeeper database..."
	rm -r $PKGBUILDDIR/var/lib/scrollkeeper
fi

if [ -d $PKGBUILDDIR/var/scrollkeeper ]; then
	echo "Removing application-specific scrollkeeper database..."
	rm -r $PKGBUILDDIR/var/scrollkeeper
fi

# fill /install
echo "Copying package description..."
mkdir -p $INSTALLDIR
cat $SCRIPTDIR/$NAME.desc > $INSTALLDIR/slack-desc
if [ -e $SCRIPTDIR/$NAME.inst ]; then
	echo "Copying package install script..."
	cat $SCRIPTDIR/$NAME.inst > $INSTALLDIR/doinst.sh
fi
if [ -e $SCRIPTDIR/$NAME.fix ]; then
	echo "Running $SCRIPTDIR/$NAME.fix..."
	/bin/sh $SCRIPTDIR/$NAME.fix $PKGBUILDDIR $SCRIPTDIR $SRCDIR $NAME $VERSION
fi

# check for gconf schemas
(
if [[ -d $PKGBUILDDIR/etc/gconf/schemas && "$DROPLINE_NOGCONF" != "1" ]]; then
	cd $PKGBUILDDIR/etc/gconf/schemas
	
	# create the install script if it doesn't exist, append a new line
	# if it does
	if [ -e "$INSTALLDIR/doinst.sh" ]; then
		echo "" >> $INSTALLDIR/doinst.sh
	else
		echo "#!/bin/sh" > $INSTALLDIR/doinst.sh
	fi
	
	echo 'export GCONF_CONFIG_SOURCE=`gconftool-2 --get-default-source`' >> $INSTALLDIR/doinst.sh
	for schema in *.schemas
	do
		if [ "$schema" != "*.schemas" ]; then
			echo "gconftool-2 --makefile-install-rule /etc/gconf/schemas/$schema &> /dev/null" >> $INSTALLDIR/doinst.sh
		fi
	done
	
	for entry in *.entries
	do
		if [ "$entry" != "*.entries" ]; then
			echo "gconftool-2 --direct --config-source=\$GCONF_CONFIG_SOURCE --load /etc/gconf/schemas/$entry" >> $INSTALLDIR/doinst.sh
		fi
	done
fi
)

# update the scrollkeeper database
if [[ -d $PKGBUILDDIR/usr/share/omf ]]; then
	# create the install script if it doesn't exist, append a new line
	# if it does
	if [ -e "$INSTALLDIR/doinst.sh" ]; then
		echo "" >> $INSTALLDIR/doinst.sh
	else
		echo "#!/bin/sh" > $INSTALLDIR/doinst.sh
	fi
	
	echo "/usr/bin/scrollkeeper-update -p /var/lib/scrollkeeper &> /dev/null" >> $INSTALLDIR/doinst.sh
fi

# strip out debugging info
find $PKGBUILDDIR | xargs file | grep ELF | cut -f 1 -d : | xargs strip --strip-unneeded

# compress man pages
find $PKGBUILDDIR | xargs file | grep "troff" | cut -f 1 -d : | xargs gzip -9

REVISION="$REVISION""dl"

echo "Building package $NAME-$VERSION-$ARCH-$REVISION..."
cd $PKGBUILDDIR
/sbin/makepkg -l y -c n $NAME-$VERSION-$ARCH-$REVISION.tgz
if [ "$?" != "0" ]; then
	exit $?
fi
mv $NAME-$VERSION-$ARCH-$REVISION.tgz $PACKAGEDIR
echo "Finished building $NAME-$VERSION-$ARCH-$REVISION.tgz."
exit 0

#
# ChangeLog
#
# 01.11.03 - Todd Kulesza <todd@dropline.net>
#	- add support for postconfig commands
#
# 11.30.03 - Todd Kulesza <todd@dropline.net>
#	- strip out debugging info and compress the man pages
# 
# 10.09.03 - Todd Kulesza <todd@dropline.net>
#	- Move all documentation to /usr/doc/$NAME-$VERSION
#
# 09.14.03 - Todd Kulesza <todd@dropline.net>
# 	- Began ChangeLog
#	- Added support for loading GConf entry files
