/*
 *  The Dropline Update Applet
 *
 *  Copyright 2003-2004 Todd Kulesza <todd@dropline.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnome/libgnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <curl/curl.h>
#include <libintl.h>
#include <locale.h>

#define LOG_FILE "/tmp/dropline-applet-log"

typedef struct _DroplineData DroplineData;
typedef struct _HTTPMem HTTPMem;

struct _DroplineData {
	GtkWidget *applet;
	GtkWidget *image;
	GtkWidget *prefs;
	GtkTooltips *tooltips;
	
	GConfClient *gconf;
	
	GAsyncQueue *net_queue;
	GMutex *net_mutex;
	GMutex *queue_mutex;
	
	PanelAppletOrient orientation;
	
	gboolean proxy_enable;
	gboolean proxy_auth;
	gchar *proxy_user;
	gchar *proxy_pass;
	gchar *proxy_host;
	gint proxy_port;
	gint update_interval;
	gboolean check_os;
	gboolean check_deps;
	
	guint timeout_id;
	const gchar *icon_path;
};

struct _HTTPMem
{
	gchar *data;
	guint len;
};

enum STATUS
{
	STATUS_IDLE,
	STATUS_CHECKING,
	STATUS_ERROR,
	STATUS_AVAILABLE
};

#define IS_PANEL_HORIZONTAL(o) (o == PANEL_APPLET_ORIENT_UP || o == PANEL_APPLET_ORIENT_DOWN)
#define MAX_FILES				512
#define MILISECONDS_IN_SECOND	1000
#define SECONDS_IN_MINUTE		60
#define MINUTES_IN_HOUR			60

static void
update_alert (DroplineData *data, gint status);

gboolean
check_for_updates_cb (gpointer gdata);

void
update_log (const gchar *message)
{
	FILE *log;
	gchar time_string [64];
	gchar *formatted_mesg;
	struct tm *t;
	time_t seconds;
	
	if (!message)
		return;
	
	log = fopen (LOG_FILE, "a+");
	if (!log)
		return;
	
	seconds = time (NULL);
	t = gmtime (&seconds);
	strftime (time_string, 64, "%c", t);
	
	if (message [strlen (message) - 1] != '\n')
		formatted_mesg = g_strdup_printf ("%s: %s\n", time_string, message);
	else
		formatted_mesg = g_strdup_printf ("%s: %s", time_string, message);
	
	fwrite (formatted_mesg, sizeof (gchar), strlen (formatted_mesg), log);
	
	fclose (log);
	
	g_free (formatted_mesg);
	
	return;
}

void
load_gconf_data (DroplineData *data)
{
	// not the best way to deal with gconf, but it doesn't work well when
	// called from the network thread
	data->proxy_enable = gconf_client_get_bool (data->gconf, 
			"/system/http_proxy/use_http_proxy", NULL);
	data->proxy_auth = gconf_client_get_bool (data->gconf, 
			"/system/http_proxy/use_authentication", NULL);
	data->proxy_user = gconf_client_get_string (data->gconf, 
			"/system/http_proxy/authentication_user" , NULL);
	data->proxy_pass = gconf_client_get_string (data->gconf, 
			"/system/http_proxy/authentication_password", NULL);
	data->proxy_host = gconf_client_get_string (data->gconf, 
			"/system/http_proxy/host", NULL);
	data->proxy_port = gconf_client_get_int (data->gconf, 
			"/system/http_proxy/port", NULL);
	data->update_interval = panel_applet_gconf_get_int (
			PANEL_APPLET (data->applet), "update_interval", NULL);
	data->check_os = panel_applet_gconf_get_bool (
			PANEL_APPLET (data->applet), "enable_os_check", NULL);
	data->check_deps = panel_applet_gconf_get_bool (
			PANEL_APPLET (data->applet), "enable_deps_check", NULL);
	
	return;
}

gboolean
package_disliked (const gchar *package)
{
	gboolean retval;
	gchar *cache_dir, *disliked_file, *disliked_packs, *newline;
	gchar **lines;
	gint i;
	
	if (g_file_get_contents ("/etc/dropline/cache_dir", &cache_dir, NULL, NULL))
	{
		newline = strstr (cache_dir, "\n");
		if (newline)
			cache_dir [newline - cache_dir] = '\0';
		
	}
	else
		cache_dir = g_strdup ("/var/cache/dropline-installer");
	
	disliked_file = g_strdup_printf ("%s/DroplineFilesDisliked", cache_dir);
	
	if (g_file_get_contents (disliked_file, &disliked_packs, NULL, NULL))
	{
		retval = FALSE;
		lines = g_strsplit (disliked_packs, "\n", MAX_FILES);
		for (i = 0; lines [i] && lines [i][0] != '\0' && !retval; i++)
		{
			if (!strncmp (lines [i], package, strlen (lines [i])))
				retval = TRUE;
		}
		
		g_strfreev (lines);
	}
	else
		retval = FALSE;
	
	return retval;
}

gboolean
package_blacklisted (const gchar *package)
{
	gboolean retval;
	gchar *blacklist_file, *blacklist_packs;
	gchar **lines;
	gint i;
	
	blacklist_file = g_strdup ("/etc/dropline/blacklist");
	
	if (g_file_get_contents (blacklist_file, &blacklist_packs, NULL, NULL))
	{
		retval = FALSE;
		lines = g_strsplit (blacklist_packs, "\n", MAX_FILES);
		
		for (i = 0; lines [i] && lines [i][0] != '\0' && !retval; i++)
		{
			gchar *msg;
			
			msg = g_strdup_printf ("blacklist line[%d]: %s\n", i, lines[i]);
			update_log (msg);
			
			if (!strncmp (lines [i], package, strlen (lines [i])))
				retval = TRUE;

			g_free (msg);
		}
		
		g_strfreev (lines);
	}
	else
		retval = FALSE;
	
	return retval;
}

void
update_parse (DroplineData *data, const gchar *mesg)
{
	GDir *package_dir;
	const gchar *package_name_installed;
	gchar *colon, *log;
	gchar **lines;
	gint i, j, status, dl_length;
	gboolean found;
	
	update_log (_("Parsing package list...\n"));
	
	dl_length = strlen ("dropline-installer");
	
	if (	!mesg || 
		(strlen (mesg) < dl_length) || 
		(strncmp (mesg, "dropline-installer", dl_length)))
	{
		status = STATUS_ERROR;
	}
	else
	{
		log = g_strdup_printf (_("Package list size is %d\n"), strlen (mesg));
		update_log (log);
		g_free (log);
		
		lines = g_strsplit (mesg, "\n", MAX_FILES);
		for (i = 0; lines [i]; i++)
		{
			if (strstr (lines [i], ":obsolete_lib:") ||
				strstr (lines [i], ":obsolete_app:") ||
				strstr (lines [i], ":obsolete:") ||
				strstr (lines [i], ":replaced:"))
				lines [i] = '\0';
			else
			{
				colon = strstr (lines [i], ":");
				if (colon)
					lines [i][colon - lines [i]] = '\0';
			}
		}
	
		j = 0;
		for (i = 0; lines [i] && lines [i][0] != '\0'; i++)
		{
			found = FALSE;
			package_dir = g_dir_open ("/var/log/packages", 0, NULL);
			while ((package_name_installed = g_dir_read_name (package_dir)) && !found)
			{
				if (!strncmp (package_name_installed, lines [i], 
						strlen (package_name_installed)))
					found = TRUE;
			}
			g_dir_close (package_dir);
			if (!found)
			{
				colon = g_strrstr (lines [i], ".");
				update_log (colon);
				lines [i][colon - lines[i] - 2] = '\0';
				update_log (lines[i]);
				if (!package_disliked (lines [i]) &&
					!package_blacklisted (lines [i]))
				{
					j++;
				}
			}
		}
		
		if (!j)
			status = STATUS_IDLE;
		else
			status = STATUS_AVAILABLE;
			
		g_strfreev (lines);
	}

	update_alert (data, status);
	
	update_log (_("Finished parsing package list.\n"));
	
	return;
}

size_t
net_write_cb (void *ptr, size_t size, size_t nmemb, void *data)
{
	gint realsize = size * nmemb;
	HTTPMem *memory = data;

	memory->data = (gchar *) g_realloc(memory->data, memory->len + realsize + 1);

	memcpy (&(memory->data [memory->len]), ptr, realsize);
	memory->len += realsize;
	memory->data [memory->len] = '\0';

	return realsize;
}

gpointer
net_send_request (gpointer gdata)
{
	CURL *session;
	HTTPMem *memory;
	gchar *user, *pass, *userpwd;
	DroplineData *data = (DroplineData *) gdata;
	
	g_mutex_lock (data->net_mutex);
	
	update_log (_("Checking for updates...\n"));
	
	memory = g_new (HTTPMem, 1);
	memory->len = 0;
	memory->data = NULL;
	
	user = pass = userpwd = NULL;
	
	session = curl_easy_init ();
	
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, net_write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, "http://droplinegnome.org/DroplineFiles2.14");
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, TRUE);
	curl_easy_setopt (session, CURLOPT_USERAGENT, "Dropline Alert " VERSION);

	// Proxy stuff
	if (data->proxy_enable)
	{
		curl_easy_setopt (session, CURLOPT_PROXY, data->proxy_host);
		curl_easy_setopt (session, CURLOPT_PROXYPORT, data->proxy_port);
		if (data->proxy_auth)
		{
			userpwd = g_strdup_printf ("%s:%s", user, pass);
			curl_easy_setopt (session, CURLOPT_PROXYUSERPWD, userpwd);
		}
	}
	
	curl_easy_perform (session);
	
	g_mutex_lock (data->queue_mutex);
	if (memory->data)
		g_async_queue_push (data->net_queue, g_strdup (memory->data));
	else
		g_async_queue_push (data->net_queue, g_strdup ("error"));
	g_mutex_unlock (data->queue_mutex);
	
	curl_easy_cleanup (session);
	
	g_free (user);
	g_free (pass);
	g_free (userpwd);
	g_free (memory->data);
	g_free (memory);
	
	update_log (_("Update check complete.\n"));
	
	g_mutex_unlock (data->net_mutex);
	
	return NULL;
}

static void
load_image (GtkImage *image, const gchar *path, gint size)
{
	GdkPixbuf *pixbuf;
	
	pixbuf = gdk_pixbuf_new_from_file_at_size (path, size, size, NULL);
	gtk_image_set_from_pixbuf (image, pixbuf);
	
	g_object_unref (G_OBJECT (pixbuf));
	
	return;
}

static void
update_alert (DroplineData *data, gint status)
{
	const gchar *tip;
	gint size;
	
	size = panel_applet_get_size (PANEL_APPLET (data->applet));
	
	switch (status)
	{
	case STATUS_IDLE:
	{
		data->icon_path = GNOME_ICONDIR"/dropline/applet-idle.svg";
		tip = _("No Dropline GNOME updates are available");
		break;
	}
	case STATUS_CHECKING:
	{
		data->icon_path = GNOME_ICONDIR"/dropline/applet-checking.svg";
		tip = _("Looking for new Dropline GNOME software...");
		break;
	}
	case STATUS_AVAILABLE:
	{
		data->icon_path = GNOME_ICONDIR"/dropline/applet-available.svg";
		tip = _("New Dropline GNOME software is available");
		break;
	}
	default:
	{
		data->icon_path = GNOME_ICONDIR"/dropline/applet-error.svg";
		tip = _("Error while looking for updates");
		break;
	}
	}
	
	gdk_threads_enter ();
	load_image (GTK_IMAGE (data->image), data->icon_path, size);
	gtk_tooltips_set_tip (data->tooltips, data->applet, tip, NULL);
	gdk_threads_leave ();
	
	return;
}

gboolean
net_loop (gpointer gdata)
{
	gchar *mesg;
	DroplineData *data = (DroplineData *) gdata;
	
	g_mutex_lock (data->queue_mutex);
	
	if ((mesg = g_async_queue_try_pop (data->net_queue)))
	{
		if (!strncmp (mesg, "error", strlen (mesg)))
		{
			update_alert (data, STATUS_ERROR);
		}
		else
		{
			update_parse (data, mesg);
		}
		
		g_free (mesg);
	}
	
	g_mutex_unlock (data->queue_mutex);
	
	return TRUE;
}

gboolean
check_for_updates_cb (gpointer gdata)
{
	DroplineData *data = (DroplineData *) gdata;
	
	update_alert (data, STATUS_CHECKING);

	load_gconf_data (data);
	
	g_thread_create (net_send_request, data, FALSE, NULL);
	
	return TRUE;
}

static void
prefs_response (GtkWidget *dialog, gint response, DroplineData *data)
{
	if (response == GTK_RESPONSE_ACCEPT)
		gtk_widget_destroy (dialog);
	
	data->prefs = NULL;
	
	return;
}

static void
run_installer (DroplineData *data)
{
	gchar *command, *check_os, *check_deps;
	
	update_alert (data, STATUS_IDLE);
	
	load_gconf_data (data);
	
	if (!data->check_os)
		check_os = g_strdup ("--force");
	else
		check_os = g_strdup ("");
	
	if (!data->check_deps)
		check_deps = g_strdup ("--nodeps");
	else
		check_deps = g_strdup ("");
	
	command = g_strdup_printf ("dropline-installer %s %s", check_os, check_deps);
	
	gnome_execute_terminal_shell (NULL, command);
	
	g_free (check_os);
	g_free (check_deps);
	g_free (command);
	
	return;
}

static void
applet_destroy_cb (GtkWidget *widget, DroplineData *data)
{	
	g_free (data);
	
	curl_global_cleanup ();
	
	return;
}

static void
applet_change_background_cb (PanelApplet *applet, 
		PanelAppletBackgroundType type, GdkColor *color, const gchar *pixmap, 
		DroplineData *data)
{
	gdk_threads_enter ();
	
	if (type == PANEL_NO_BACKGROUND) {
		GtkRcStyle *rc_style = gtk_rc_style_new ();

		gtk_widget_modify_style (data->applet, rc_style);
		gtk_rc_style_unref (rc_style);
	}
	else if (type == PANEL_COLOR_BACKGROUND) {
		gtk_widget_modify_bg (data->applet,
				      GTK_STATE_NORMAL,
				      color);
	} else { /* pixmap */
		/* FIXME: Handle this when the panel support works again */
	}
	
	gdk_threads_leave ();
	
	return;
}

static void
applet_change_orient_cb (GtkWidget *w, PanelAppletOrient o, DroplineData *data)
{	
	data->orientation = o;
	
	return;
}

static void
applet_change_size_cb (GtkWidget *w, int size, DroplineData *data)
{
	load_image (GTK_IMAGE (data->image), data->icon_path, size);
	
	return;
}

static gboolean
button_press_cb (GtkWidget *w, GdkEventButton *event, DroplineData *data)
{
	gboolean handled;
	
	/* determine whether to handle the event or not */
	if ((event->type == GDK_BUTTON_PRESS) && (event->button == 1))
	{
		run_installer (data);
		handled = TRUE;
	}
	else
		handled = FALSE;
		
	return handled;
}

static void
dropline_update_cb (BonoboUIComponent *uic, DroplineData *data, const gchar *verbname)
{
	run_installer (data);
		
	return;
}

void
update_interval_cb (GtkSpinButton *spin, DroplineData *data)
{
	gint value, interval;
	
	value = gtk_spin_button_get_value_as_int (spin);
	panel_applet_gconf_set_int (PANEL_APPLET (data->applet), 
			"update_interval", value, NULL);
	data->update_interval = value;
	
	// reset the update check loop
	g_source_remove (data->timeout_id);
	if (data->update_interval < 1)
		data->update_interval = 1;
	interval = data->update_interval * MILISECONDS_IN_SECOND * SECONDS_IN_MINUTE * MINUTES_IN_HOUR;
	data->timeout_id = g_timeout_add (interval, check_for_updates_cb, data);
	
	return;
}

void
check_os_cb (GtkToggleButton *button, DroplineData *data)
{
	gboolean value;
	
	value = gtk_toggle_button_get_active (button);
	panel_applet_gconf_set_bool (PANEL_APPLET (data->applet),
			"enable_os_check", value, NULL);
	data->check_os = value;
	
	return;
}

void
check_deps_cb (GtkToggleButton *button, DroplineData *data)
{
	gboolean value;
	
	value = gtk_toggle_button_get_active (button);
	panel_applet_gconf_set_bool (PANEL_APPLET (data->applet),
			"enable_deps_check", value, NULL);
	data->check_deps = value;
	
	return;
}

static void
dropline_prefs_cb (BonoboUIComponent *uic, DroplineData *data, const gchar *verbname)
{
	GtkWidget *dialog;
	GtkWidget *hbox1, *hbox2, *vbox1, *vbox2;
	GtkWidget *check;
	GtkWidget *spin;
	GtkWidget *label;
	gint spin_value;
	gboolean check_value;
	
	if (data->prefs)
	{
		gtk_window_present (GTK_WINDOW (data->prefs));
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (_("Dropline Update Alert Preferences"),
			NULL, GTK_DIALOG_NO_SEPARATOR, 
			GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT,
			NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);
	data->prefs = dialog;
	
	vbox1 = gtk_vbox_new (FALSE, 12);
	gtk_container_set_border_width (GTK_CONTAINER (vbox1), 12);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox1, TRUE, TRUE, 0);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>General</b>"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (vbox1), label, FALSE, FALSE, 0);
	
	hbox1 = gtk_hbox_new (FALSE, 12);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, TRUE, TRUE, 0);
	
	vbox2 = gtk_vbox_new (FALSE, 12);
	gtk_box_pack_start (GTK_BOX (hbox1), vbox2, FALSE, FALSE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	
	vbox2 = gtk_vbox_new (FALSE, 12);
	gtk_box_pack_start (GTK_BOX (hbox1), vbox2, TRUE, TRUE, 0);
	
	hbox2 = gtk_hbox_new (FALSE, 12);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);
	
	label = gtk_label_new_with_mnemonic (_("Hours between _update checks:"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);
	
	spin = gtk_spin_button_new_with_range (1.0, 999.0, 1.0);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), spin);
	gtk_entry_set_activates_default (GTK_ENTRY (spin), TRUE);
	spin_value = panel_applet_gconf_get_int (PANEL_APPLET (data->applet),
			"update_interval", NULL);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), (gfloat) spin_value);
	g_signal_connect (G_OBJECT (spin), "value_changed",
			G_CALLBACK (update_interval_cb), data);
	gtk_box_pack_start (GTK_BOX (hbox2), spin, FALSE, FALSE, 0);
	
	hbox2 = gtk_hbox_new (FALSE, 12);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);
	
	check = gtk_check_button_new_with_mnemonic (_("Check for a supported _OS"));
	check_value = panel_applet_gconf_get_bool (PANEL_APPLET (data->applet),
			"enable_os_check", NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), check_value);
	g_signal_connect (G_OBJECT (check), "toggled",
			G_CALLBACK (check_os_cb), data);
	gtk_box_pack_start (GTK_BOX (hbox2), check, FALSE, FALSE, 0);
	
	hbox2 = gtk_hbox_new (FALSE, 12);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);
	
	check = gtk_check_button_new_with_mnemonic (_("Check for required _files"));
	check_value = panel_applet_gconf_get_bool (PANEL_APPLET (data->applet),
			"enable_deps_check", NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), check_value);
	g_signal_connect (G_OBJECT (check), "toggled",
			G_CALLBACK (check_deps_cb), data);
	gtk_box_pack_start (GTK_BOX (hbox2), check, FALSE, FALSE, 0);
	
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (prefs_response), data);
	
	gtk_widget_show_all (dialog);
	
	return;
}

static void
dropline_about_cb (BonoboUIComponent *uic, DroplineData *data, const gchar *verbname)
{
	static GtkWidget *about = NULL;
	GdkPixbuf *pixbuf = NULL;
	gchar *file;
	static const gchar *authors [] = 
	{
		"Todd Kulesza <todd@dropline.net>",
		NULL
	};
	const gchar *documenters [] = 
	{
		NULL
	};
	const gchar *translator_credits = _("translator_credits");
	
	if (about)
	{
		gtk_window_set_screen (GTK_WINDOW (about), 
				gtk_widget_get_screen (data->applet));
		gtk_window_present (GTK_WINDOW (about));
		
		return;
	}
        
	file = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP, 
			"dropline.svg", TRUE, NULL);
	pixbuf = gdk_pixbuf_new_from_file (file, NULL);
	g_free (file);

	about = gnome_about_new (_("Dropline Update Alert"), VERSION, 
			_("Copyright 2005-2006 The Dropline GNOME Team"),
			_("Receive alerts when new Dropline GNOME software is available."),
			authors, documenters,
			strcmp (translator_credits, "translator_credits") != 0 ? 
					translator_credits : NULL,
			pixbuf);

	gtk_window_set_screen (GTK_WINDOW (about), 
			gtk_widget_get_screen (data->applet));
	gtk_window_set_wmclass (GTK_WINDOW(about), "dropline update alert", 
			"Dropline Update Alert");
	gnome_window_icon_set_from_file (GTK_WINDOW (about), 
			GNOME_ICONDIR"/dropline.svg");
	g_signal_connect (G_OBJECT (about), "destroy",
		G_CALLBACK (gtk_widget_destroyed), &about);
	
	gtk_widget_show (about);
	
	return;
}

const BonoboUIVerb dropline_applet_menu_verbs [] = 
{
	BONOBO_UI_UNSAFE_VERB ("Update", dropline_update_cb),
	BONOBO_UI_UNSAFE_VERB ("Preferences", dropline_prefs_cb),
	BONOBO_UI_UNSAFE_VERB ("About",    dropline_about_cb),
	
	BONOBO_UI_VERB_END
};

void
dropline_applet_create (PanelApplet *applet)
{
	DroplineData *data;
	guint interval;
	
	curl_global_init (CURL_GLOBAL_ALL);
	
	data = g_new0 (DroplineData, 1);

	gdk_threads_enter ();
	
	data->applet = GTK_WIDGET (applet);
	data->image = gtk_image_new ();
	data->tooltips = gtk_tooltips_new ();
	data->prefs = NULL;
	data->gconf = gconf_client_get_default ();
	data->net_mutex = g_mutex_new ();
	data->queue_mutex = g_mutex_new ();
	data->net_queue = g_async_queue_new ();
	
	panel_applet_add_preferences (applet, "/schemas/apps/dropline_applet/prefs", NULL);

	load_gconf_data (data);
	
	gtk_container_add (GTK_CONTAINER (data->applet), data->image);
	
	gtk_widget_show_all (GTK_WIDGET (data->applet));
	
	g_signal_connect (applet, "destroy", G_CALLBACK (applet_destroy_cb), data);
	g_signal_connect (applet, "change_orient", G_CALLBACK (applet_change_orient_cb), data);
	g_signal_connect (applet, "change_background", G_CALLBACK (applet_change_background_cb), data);
	g_signal_connect (applet, "change_size", G_CALLBACK (applet_change_size_cb), data);
	g_signal_connect (G_OBJECT (applet), "button_press_event", G_CALLBACK (button_press_cb), data);
	
	panel_applet_setup_menu_from_file (applet, 
					   NULL, // opt_datadir
					   "Dropline_UpdateApplet.xml",
					   NULL,
					   dropline_applet_menu_verbs,
					   data);
	
	applet_change_orient_cb (GTK_WIDGET (applet), panel_applet_get_orient (applet), data);
	
	gdk_threads_leave ();
	
	update_alert (data, STATUS_IDLE);

	if (data->update_interval < 1)
		data->update_interval = 1;
	interval = data->update_interval * MILISECONDS_IN_SECOND * SECONDS_IN_MINUTE * MINUTES_IN_HOUR;
	data->timeout_id = g_timeout_add (interval, check_for_updates_cb, data);
	g_timeout_add (5000, net_loop, data);
	
	check_for_updates_cb (data);
	
	return;
}

static gboolean
dropline_applet_factory (PanelApplet *applet, const gchar *iid, gpointer data)
{
	update_log (_("Creating applet...\n"));
	
	dropline_applet_create (applet);

	update_log (_("Applet created.\n"));
	
	return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:Dropline_UpdateApplet_Factory",
			     PANEL_TYPE_APPLET,
			     "dropline_applet",
			     "0",
			     dropline_applet_factory,
			     NULL)
