/*  XMMS - ALSA output plugin
 *  Copyright (C) 2001 Matthieu Sozeau <mattam@netcourrier.com>
 *
 *  Contains code
 *      Copyright (C) 1998-2001  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *      Copyright (C) 1999-2001  Hvard Kvlen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "alsa.h"

#include <signal.h>

#define error(str) do { \
xmms_show_message("ALSA-XMMS Error", str, "OK", 0, NULL, NULL); \
g_warning(str); alsa_initialized = 0; return; } while(0);

//  g_warning(str, snd_strerror(-err)); alsa_initialized = 0; return; }while(0);

#define alsa_error(str, err) do { \
char *msg = g_strdup_printf(str, snd_strerror(-err)); \
xmms_show_message("ALSA error", msg, "OK", 0, NULL, NULL); \
g_free(msg); alsa_initialized = 0; return; } while(0);



#define debug(str) do { if(alsa_cfg.debug) { printf(str); printf("\n"); } } while(0);

#define test(str) { if(str != 0) { printf("failed: %s:%i (%s)", __FILE__, __LINE__, __FUNCTION__); exit(-1); } }

static snd_pcm_t		*alsa_pcm 	    = NULL;
static snd_pcm_t		*alsa_pcm_slave     = NULL;
static snd_pcm_hw_params_t	*hwparams 	    = NULL;
static snd_pcm_sw_params_t	*swparams 	    = NULL;
static snd_pcm_format_t		 alsa_format;
static snd_pcm_status_t		*alsa_status 	    = NULL;
static snd_pcm_channel_area_t   *areas              = NULL;

static snd_output_t		*logs 		    = NULL;

static gboolean                  alsa_initialized;
static unsigned int 		 alsa_rate;
static unsigned int 		 alsa_channels 	    = 2;
static int 			 alsa_buffer_size;
static int 			 alsa_period_size;
static int 			 alsa_buffer_time;
static int 			 alsa_bits_per_sample;
static int 			 alsa_frame_size;
static int 			 written_frames;
static int 			 alsa_bps 	    = 0;
static gint64 		         alsa_total_written = 0;

//Set/Get volume
static long int 		 alsa_min_vol, alsa_max_vol;
static snd_pcm_info_t 		*info 		    = NULL;
static snd_mixer_selem_id_t 	*pcm_elt 	    = NULL;
static snd_mixer_elem_t 	*pcm_element 	    = NULL;
static snd_mixer_t 		*mixer 		    = NULL;
static char 			 card_id[] 	    = "hw:0";

static int 			 alsa_card 	    = 0, alsa_device = 0, alsa_subdevice = 0;
static int                       mmap = 1;
//Loop
static int 			 going              = 0;
static int                       pause_time;         
static gboolean 		 paused 	    = FALSE;
static int 			 realtime;
static int                       first_write        = 1;

static gpointer 		 buffer;
static gint                      buffer_size        = 0;

static int alsa_can_pause;

typedef struct _snd_format {
	unsigned int rate;
	unsigned int channels;
	snd_pcm_format_t format;
	AFormat xmms_format;
} _snd_format_t;

typedef _snd_format_t * snd_format_t;

static snd_format_t xmms_input = 0;
static snd_format_t effect_input = 0;


void alsa_setup_mixer(void);

//static int alsa_volume_callback(snd_mixer_elem_t *elem, unsigned int mask);
static void alsa_setup(snd_format_t f);
static void *alsa_loop(void *arg);
void alsa_write(gpointer data, gint length);
static void alsa_write_audio(gpointer data, gint length);

gint alsa_real_open(snd_format_t f);

static int err;


static snd_format_t snd_format_from_xmms(AFormat fmt, gint rate, gint channels);

gint alsa_playing(void)
{
	if(!going || paused)
		return FALSE;

	return(snd_pcm_state(alsa_pcm) == SND_PCM_STATE_RUNNING);
}

void xrun_recover()
{
	if(alsa_cfg.debug) {
		snd_pcm_status_alloca(&alsa_status);
		if ((err = snd_pcm_status(alsa_pcm, alsa_status))<0) {
			alsa_error("status error", err);
		}
		printf("Status:\n");
		snd_pcm_status_dump(alsa_status, logs);
	}

	if (snd_pcm_state(alsa_pcm) == SND_PCM_STATE_XRUN) {
		if ((err = snd_pcm_prepare(alsa_pcm)) < 0) {
			alsa_error("xrun: prepare error", err);
		}
		if (mmap)
			first_write = 1;
	}
}

gint alsa_free(void)
{
	int avail;
	if (paused)
		return 0;
	else
	{

	        if ((avail = snd_pcm_avail_update(alsa_pcm)) < 0)
			xrun_recover();

		return (avail * alsa_frame_size);
	}
}

void alsa_pause(short p)
{
	debug("alsa_pause");
	paused = p;

	if (alsa_can_pause && !mmap)
		snd_pcm_pause(alsa_pcm, p);
	else {
		if (p)
			snd_pcm_drop(alsa_pcm);
		snd_pcm_prepare(alsa_pcm);	
	}

	first_write = 1;
}


void alsa_close(void)
{
	debug("Closing device");

	if (!going)
		return;

	going = 0;
	first_write = 1;

	pcm_element = NULL;

	snd_mixer_close(mixer);
	mixer = NULL;

	snd_pcm_close(alsa_pcm);
	alsa_pcm = NULL;

	if (mmap)
		free(buffer);

	debug("Device closed")
		}

void alsa_flush(gint time)
{
	alsa_total_written = (gint64) time * alsa_bps / 1000;
}

static long int a, b;
void alsa_setup_mixer(void)
{
	gchar *dev;

	debug("alsa_setup_mixer");
	pcm_element = 0;
	    
	if(alsa_cfg.use_user_device)
	{
		dev = g_strdup(alsa_cfg.user_device);
	} else {
		dev = g_strdup_printf("hw:%i", alsa_cfg.audio_card);
	}

	if ((err = snd_mixer_open(&mixer, 0)) < 0) {
		printf("Could not setup mixer\n");
		return;
	}
		
	if ((err = snd_mixer_attach(mixer, dev)) < 0)
		alsa_error("snd_mixer_attach: %s", err);

	if ((err = snd_mixer_selem_register(mixer, NULL, NULL)) < 0)
		alsa_error("snd_mixer_selem_register: %s", err);
	if ((err = snd_mixer_load (mixer)) < 0)
		alsa_error("snd_mixer_load: %s", err);
	if ((err = snd_mixer_selem_id_malloc(&pcm_elt)) < 0)
		alsa_error("selem_id_malloc: %s", err);
			
	if(!alsa_cfg.mixer_device)
	{
		pcm_element = snd_mixer_first_elem(mixer);  
		if (!pcm_element) return;
		snd_mixer_selem_get_id(pcm_element, pcm_elt);
		while(strcmp(snd_mixer_selem_id_get_name(pcm_elt), "Master"))
		{
			pcm_element = snd_mixer_elem_next(pcm_element);
			if(!pcm_element) break;
			snd_mixer_selem_get_id(pcm_element, pcm_elt);
		}
	}
	else
	{
		pcm_element = snd_mixer_first_elem(mixer);  
		if (!pcm_element) return;
		snd_mixer_selem_get_id(pcm_element, pcm_elt);
		while(strcmp(snd_mixer_selem_id_get_name(pcm_elt), alsa_cfg.mixer_device))
		{
			pcm_element = snd_mixer_elem_next(pcm_element);
			if(!pcm_element) break;
			snd_mixer_selem_get_id(pcm_element, pcm_elt);
		}
	}
	
	if(!pcm_element) return;
	snd_mixer_selem_get_playback_volume_range(pcm_element, &alsa_min_vol, &alsa_max_vol);
	snd_mixer_selem_set_playback_volume_range(pcm_element, 0, 100);
  
	if (alsa_max_vol == 0)
	{
		pcm_element = NULL;
		return;
	}
	
	snd_mixer_selem_get_playback_volume(pcm_element, SND_MIXER_SCHN_FRONT_LEFT, &a);
	snd_mixer_selem_get_playback_volume(pcm_element, SND_MIXER_SCHN_FRONT_RIGHT, &b);

	alsa_set_volume(a * 100 / alsa_max_vol, b * 100 / alsa_max_vol);
	
//	snd_mixer_elem_set_callback(pcm_element, alsa_volume_callback);

	g_free(dev);
	debug("alsa_setup_mixer: end")
		}

static long int volleft = 0, volright = 0;
/*
  int alsa_volume_callback(snd_mixer_elem_t *elem, unsigned int mask)
  {
  printf("alsa_volume_callback : %i\n", mask);
  return 0;
  }
*/

void alsa_get_volume(int *l, int *r)
{
	static gboolean first = TRUE;
	if (first) {
		alsa_setup_mixer();
		first = !first;
	}

	if (!pcm_element)
		return;

	snd_mixer_selem_get_playback_volume(pcm_element, SND_MIXER_SCHN_FRONT_LEFT, &volleft);
	snd_mixer_selem_get_playback_volume(pcm_element, SND_MIXER_SCHN_FRONT_RIGHT, &volright);

	*l = volleft;
	*r = volright; 
}



void alsa_set_volume(int l, int r)
{
	if(!pcm_element) return;

	test(snd_mixer_selem_set_playback_volume(pcm_element, SND_MIXER_SCHN_FRONT_LEFT, l));
	test(snd_mixer_selem_set_playback_volume(pcm_element, SND_MIXER_SCHN_FRONT_RIGHT, r));
}


gint alsa_get_output_time(void)
{
	if (going) {
		snd_pcm_sframes_t time, delay;
		if (snd_pcm_delay(alsa_pcm, &delay) < 0)
			return (gint) ((gint64) alsa_total_written * 1000 / alsa_bps); 
		time = alsa_total_written - snd_pcm_frames_to_bytes(alsa_pcm, delay);
		if (time < 0)
			return 0;
		return (gint) ((gint64) time * 1000 / alsa_bps); 
	}
	return 0;
}

gint alsa_get_written_time(void)
{ 
	return (gint) ((gint64) alsa_total_written * 1000 / alsa_bps); 
}

void alsa_write(gpointer data, gint length)
{
	int cnt = 0;
	gint off = 0;
	snd_pcm_uframes_t offset, frames;
	snd_pcm_sframes_t avail, size, commitres;
	const snd_pcm_channel_area_t *chan_areas = areas;
	EffectPlugin *ep;

	if(paused) return;
	
	if (effects_enabled() && (ep = get_current_effect_plugin ()))
	{
		gint new_frequency = xmms_input->rate, new_channels = xmms_input->channels;
		AFormat f = xmms_input->xmms_format;	       
		
		if (ep->query_format)
		{
			ep->query_format (&f, &new_frequency, &new_channels);
			
			if ((!effect_input) || 
			    (f != effect_input->xmms_format ||
			     new_frequency != effect_input->rate ||
			     new_channels != effect_input->channels))
			{
				int tmp = alsa_get_written_time();
				debug("Changing audio format for effect plugin");
				if (effect_input)
					free(effect_input);

				effect_input = snd_format_from_xmms (f, new_frequency, new_channels);
				alsa_close ();

				alsa_real_open (effect_input);
				alsa_flush (tmp);
			}
			
		}
	
		length = ep->mod_samples(&data, length,
					 xmms_input->xmms_format,
					 xmms_input->rate,
					 xmms_input->channels);
		
	} else
		if (effect_input)
		{ 
			int tmp = alsa_get_written_time();
			free(effect_input);
			effect_input = 0;
			alsa_close();
			alsa_real_open (xmms_input);
			alsa_flush (tmp);
		}

	if (mmap) {
		while (length > 0)
		{			
			avail = snd_pcm_avail_update(alsa_pcm);
			
			if (avail < 0)
				xrun_recover();

			if (avail < alsa_period_size) {
				err = snd_pcm_wait (alsa_pcm, 1);
				if (err < 0)
					alsa_error("snd_pcm_wait: %s\n", err);
				continue;
			}
						
			frames = length / alsa_frame_size;

			if ((err = snd_pcm_mmap_begin(alsa_pcm, &chan_areas, &offset, &frames) < 0))
				alsa_error("mmap_begin %s", err);

			/*printf("in write %i\n", frames);

			printf("frames: %i, length: %i, error: %i\n", frames, length, err);
			printf("offset: %i, buffer offset: %i, buffer_size: %i, frame size: %i\n", 
			offset, offset * alsa_frame_size, buffer_size, alsa_frame_size);		
			printf("memcpy: cnt: %i\n", frames * alsa_frame_size);
			*/

			cnt = frames * alsa_frame_size;
			
			memcpy(chan_areas[0].addr + offset * alsa_frame_size, (char *) data + off, 
			       cnt);
			
			err = snd_pcm_mmap_commit(alsa_pcm, offset, frames);
			if (err < 0)
				xrun_recover();
			else if (err != frames)
				alsa_error("mmap_commit %s", err);
			
			alsa_total_written += cnt;

			length -= cnt;
			off += cnt;

			if (first_write) {
				first_write = 0;
				err = snd_pcm_start(alsa_pcm);					
				if (err < 0) {
					alsa_error("Start error: %s\n", err);
				} else {
					debug("Stream started\n");
				}
			}
		}
	} else
		alsa_write_audio(data, length);

}

void alsa_write_audio(gpointer data, gint length)
{
	int written;
	while(length > 0)
	{       	
		written_frames = snd_pcm_writei(alsa_pcm, data, length / alsa_frame_size);
		
		if(written_frames > 0)
		{
			written = written_frames * alsa_frame_size;
			alsa_total_written += written;
			length -= written;
			data += written;
		}
		else if (written_frames == -EAGAIN || (written_frames > 0 && written_frames < (length / alsa_frame_size)))
		{
			snd_pcm_wait(alsa_pcm, 100);
		}
		else if(written_frames == -EPIPE)
		{
			xrun_recover();
		}
		else
		{
			alsa_error("read/write error: %s", -written_frames);
			break;
		}
	}
}

gint alsa_open(AFormat fmt, gint rate, gint nch)
{
	if(xmms_input)
		free(xmms_input);

	xmms_input = snd_format_from_xmms(fmt, rate, nch);
	
	return alsa_real_open (xmms_input);
}

gint alsa_real_open(snd_format_t f)
{
	alsa_initialized = 1;

	if(alsa_pcm) alsa_close();
	
	if(alsa_cfg.debug) snd_output_stdio_attach(&logs, stdout, 0);

	mmap = alsa_cfg.mmap;
	
	alsa_setup (f);
	if (!alsa_initialized)
		return 0;

	alsa_setup_mixer ();
	if (!alsa_initialized)
		return 0;
	
	alsa_total_written = 0;
	going = first_write = 1;
	paused = FALSE;
			
	snd_pcm_prepare (alsa_pcm);  
	
	return alsa_initialized;
}

static snd_format_t snd_format_from_xmms(AFormat fmt, gint rate, gint channels)
{

	snd_format_t f = malloc(sizeof(_snd_format_t));

	f->xmms_format = fmt;

	/*
	  if (fmt == FMT_U16_LE || fmt == FMT_S16_LE)
	  printf("Little-endian data\n");
	  else if (fmt == FMT_U16_BE || fmt == FMT_S16_BE)
	  printf("Big-endian data\n");
	  else if (fmt == FMT_U16_NE || fmt == FMT_S16_NE)
	  printf("Native-endian data\n");
	*/
	
	switch(fmt)
	{
	case FMT_U8:
		f->format = SND_PCM_FORMAT_U8;
		break;
	case FMT_S8:
		f->format = SND_PCM_FORMAT_S8;
		break;
	case FMT_U16_LE:
		f->format = SND_PCM_FORMAT_U16_LE;
		break;
	case FMT_U16_BE:
		f->format = SND_PCM_FORMAT_U16_BE;
		break;
	case FMT_U16_NE:
#ifdef AFMT_U16_NE
		error("Unrecognized format: AFMT_U16_NE");
		break;
#else
#ifdef WORDS_BIGENDIAN
		f->format = SND_PCM_FORMAT_U16_BE;
#else
		f->format = SND_PCM_FORMAT_U16_LE;
#endif
#endif
		break;
	case FMT_S16_LE:
		f->format = SND_PCM_FORMAT_S16_LE;
		break;
	case FMT_S16_BE:
		f->format = SND_PCM_FORMAT_S16_BE;
		break;
	case FMT_S16_NE:
#ifdef AFMT_S16_NE
		error("Unrecognized format: AFMT_S16_NE");
		break;
#else
#ifdef WORDS_BIGENDIAN
		f->format = SND_PCM_FORMAT_S16_BE;
#else
		f->format = SND_PCM_FORMAT_S16_LE;
#endif
#endif
		break;
	default:
		printf("Unrecognized format %i\n", fmt);
		exit(-1);
		break;

	}

	f->rate = rate;
	f->channels = channels;

	return (f);

}


static void alsa_setup(snd_format_t f)
{
	int chosen_rate, chn;

	debug("alsa_setup");
	   

	alsa_format = f->format;
	alsa_rate = f->rate;
	alsa_channels = f->channels;

	alsa_pcm = NULL;

	if(alsa_cfg.use_user_device)
	{
		if(alsa_cfg.debug) printf("Opening user device: %s\n", alsa_cfg.user_device);
		if((err = snd_pcm_open(&alsa_pcm, alsa_cfg.user_device, SND_PCM_STREAM_PLAYBACK, 0)) < 0)
			alsa_error("error opening alsa device: %s", err);
	} else {
		int const MAX_DEVICE_NAME_LEN = 20;
		char device[MAX_DEVICE_NAME_LEN];
		
		snprintf(device, MAX_DEVICE_NAME_LEN,
			 "hw:%d,%d", alsa_cfg.audio_card, alsa_cfg.audio_device);
		device[MAX_DEVICE_NAME_LEN - 1] = '\0';	
		if(alsa_cfg.debug) printf("Opening device: %s\n", device);
		if((err = snd_pcm_open(&alsa_pcm, device, SND_PCM_STREAM_PLAYBACK, 0)) < 0)		       			
			alsa_error("error opening alsa device: %s", err);
	}

	if(!alsa_pcm)
		error("PCM Handle not opened");

	if(alsa_cfg.debug) 
	{
		if(!info) snd_pcm_info_malloc(&info);
		snd_pcm_info(alsa_pcm, info);
		alsa_card =  snd_pcm_info_get_card(info);
		alsa_device = snd_pcm_info_get_device(info);
		alsa_subdevice = snd_pcm_info_get_subdevice(info);
		printf("Card %i, Device %i, Subdevice %i\n",  alsa_card,   alsa_device,  alsa_subdevice);  
	}


	snd_pcm_hw_params_alloca(&hwparams);

	if(!hwparams) 
		error("No hw params allocated");

	if((err = snd_pcm_hw_params_any(alsa_pcm, hwparams)) < 0)		
		alsa_error("broken configuration for playback: no configuration available: %s", err);
	
	if (mmap) {
		if((err = snd_pcm_hw_params_set_access(alsa_pcm, hwparams, SND_PCM_ACCESS_MMAP_INTERLEAVED)) < 0) 
			alsa_error("cannot set mmap'ed mode, falling back to direct write: %s", err);
	} else {
		if((err = snd_pcm_hw_params_set_access(alsa_pcm, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0) 
			alsa_error("cannot set direct write mode, abandonning: %s", err);
	}
	
	if((err = snd_pcm_hw_params_set_format(alsa_pcm, hwparams, alsa_format)) < 0)
		alsa_error("sample format not available for playback: %s", err);
	
	snd_pcm_hw_params_set_channels(alsa_pcm, hwparams, alsa_channels);
	chosen_rate = snd_pcm_hw_params_set_rate_near(alsa_pcm, hwparams, alsa_rate, 0);
	if(chosen_rate < 0)
		error("Cannot set rate.");
	if( chosen_rate!=alsa_rate ) {
		if(alsa_cfg.debug) {
			printf("Rate doesn't match (requested %iHz, get %iHz)\n",
			       alsa_rate, chosen_rate);
			printf("Workaround, creating plugin...\n");
		}
		alsa_pcm_slave = alsa_pcm;
		snd_pcm_rate_open(&alsa_pcm,
				  "",
				  alsa_format,
				  chosen_rate,
				  alsa_pcm_slave,
				  1);
		snd_pcm_hw_params_any(alsa_pcm, hwparams);

		snd_pcm_hw_params_set_access(alsa_pcm, hwparams,(mmap ? SND_PCM_ACCESS_MMAP_INTERLEAVED
								 : SND_PCM_ACCESS_RW_INTERLEAVED));
		
		snd_pcm_hw_params_set_format(alsa_pcm, hwparams, alsa_format);
		snd_pcm_hw_params_set_channels(alsa_pcm, hwparams, alsa_channels);
		chosen_rate =
			snd_pcm_hw_params_set_rate_near(alsa_pcm, hwparams, alsa_rate, 0);
		if(chosen_rate!=alsa_rate)
			error("Cannot set rate.");
	}
  

	alsa_buffer_time = snd_pcm_hw_params_set_buffer_time_near(alsa_pcm, hwparams, 
								  alsa_cfg.buffer_time * 1000, 0);
	if (alsa_buffer_time < 0) 
		error("Buffer time < 0");
	
	if (snd_pcm_hw_params_set_period_time_near(alsa_pcm, hwparams,
						   alsa_cfg.period_time * 1000, 0) < 0) 
		error("Period time < 0");
	
	if (snd_pcm_hw_params(alsa_pcm, hwparams) < 0) {
		if(alsa_cfg.debug) 
			snd_pcm_hw_params_dump(hwparams, logs);
		error("Unable to install hw params");
	}

	alsa_buffer_size = snd_pcm_hw_params_get_buffer_size(hwparams);
	if ((alsa_period_size = snd_pcm_hw_params_get_period_size(hwparams, 0)) == alsa_buffer_size) 
		error("buffer size == period size");

	
	alsa_can_pause = snd_pcm_hw_params_can_pause(hwparams);

	snd_pcm_sw_params_alloca(&swparams);

	snd_pcm_sw_params_current(alsa_pcm, swparams);
	
	if(!swparams) 
		error("No sw params allocated");
	
	/*err = snd_pcm_sw_params_set_start_threshold(alsa_pcm, swparams, 1);
	  if (err < 0)
	  alsa_error("Unable to set start threshold mode for playback: %s\n", err);*/
	
	if (snd_pcm_sw_params(alsa_pcm, swparams) < 0)
		error("Unable to install sw params");	

	if(alsa_cfg.debug) {
		snd_pcm_sw_params_dump(swparams, logs);
		snd_pcm_dump(alsa_pcm, logs);
	}
	
	alsa_bits_per_sample = snd_pcm_format_physical_width(alsa_format); 
	alsa_frame_size = (alsa_bits_per_sample * alsa_channels) / 8;
	alsa_bps = alsa_rate * alsa_frame_size;

	if (mmap) {
		buffer_size = (alsa_buffer_size * alsa_frame_size);
		buffer = malloc(buffer_size);
		
		areas = calloc(alsa_channels, sizeof(snd_pcm_channel_area_t));
		
		for (chn = 0; chn < alsa_channels; chn++) 
		{
			areas[chn].addr = buffer;
			areas[chn].first = chn * alsa_bits_per_sample;
			areas[chn].step = alsa_channels * alsa_bits_per_sample;
		}
	}

	if(alsa_cfg.debug)
	{
		printf("Device setup: buffer time: %i, size: %i bytes, bits per sample: %i, frame size: %i\n", alsa_buffer_time, buffer_size, alsa_bits_per_sample, alsa_frame_size);
		printf("Bps: %i\n", alsa_bps);
	}
}
