/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <stdio.h>
#include <unistd.h>
#include <gnome.h>
#include <gdk-pixbuf/gdk-pixdata.h>
#include <string.h>
#include <pwd.h>
#include <dlfcn.h>
#include <curl/curl.h>

#include "pollcreator.h"
#include "tray.h"
#include "drivel.h"
#include "about.h"
#include "login.h"
#include "network.h"
#include "dialogs.h"
#include "journal.h"

#include <config.h>

static void *xmms_dl = NULL; /* for dlopen(). */
extern gchar * cuserid (gchar *);

gchar *
get_home_dir (void)
{
	gchar *dir, *name;
	struct passwd *pass;
	
	dir = NULL;
	
	if ( (name = cuserid (NULL)) )
		if ( (pass = getpwnam (name)) )
			dir = g_strdup (pass->pw_dir);
	
	return dir;
}

void
remove_autosave (DrivelClient *dc)
{
	gchar *dir, *autosave;
	
	dir = get_home_dir ();
	autosave = g_strdup_printf ("%s/.drivel_autosave", dir);
	
	unlink (autosave);
	
	dc->modified_autosave = FALSE;
	
	g_free (autosave);
	
	return;
}

gint
get_line_len (const gchar *string, gint len, gint offset)
{
	gint i;
	
	for (i = offset; i < len; i++)
	{
		if (string [i] == '\n')
			return i - offset;
	}

	return i - offset;
}

/* this is straight from logjam */

static void
music_xmms_grab(GtkWidget *w, gpointer d) {
	gint pos;
	gchar *text;

	static gint (*xrgpp)(gint) = NULL;
	static gchar* (*xrgpt)(gint, gint) = NULL;

	if (xrgpp == NULL || xrgpt == NULL) {
		xrgpp = dlsym (xmms_dl, "xmms_remote_get_playlist_pos");
		if (dlerror ())
			return;
		xrgpt = dlsym (xmms_dl, "xmms_remote_get_playlist_title");
		if (dlerror ())
			return;
	}
	
	pos = xrgpp (0);
	text = xrgpt (0, pos);
	if (text != NULL) {
		gtk_entry_set_text(GTK_ENTRY (d), text);
		g_free(text);
	} else {
		gtk_entry_set_text(GTK_ENTRY (d), "[unable to connect to XMMS]");
	}
}

gboolean
xmms_load_ok (void)
{
	if (xmms_dl == NULL)
	{
		xmms_dl = dlopen ("libxmms.so", RTLD_LAZY);

		if (!xmms_dl) 
			xmms_dl = dlopen ("libxmms.so.1", RTLD_LAZY);
		if (!xmms_dl)
			return FALSE; /* no library; don't use xmms. */
	}
	return TRUE;
}

void
open_file_cb (GtkWidget *button, gpointer data)
{
	const gchar *filename;
	gchar input_data [4096], value [4096], *autosave, *dir;
	gint offset, len, item;
	gboolean state;
	GnomeVFSHandle *handle;
	GnomeVFSFileSize bytes_read;
	GnomeVFSResult result;
	GtkTextBuffer *buffer;
	DrivelClient *dc = (DrivelClient *) data;

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (dc->journal_text));
	
	/* use button to figure out if this is being called from the save dialog or autosave */
	if (button)
	{
		filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dc->save_dialog));
		result = gnome_vfs_open (&handle, filename, GNOME_VFS_OPEN_READ);
	}
	else
	{
		if ( (dir = get_home_dir ()) )
		{
			autosave = g_strdup_printf ("%s/.drivel_autosave", dir);
			result = gnome_vfs_open (&handle, autosave, GNOME_VFS_OPEN_READ);
			if (result != GNOME_VFS_OK)
				return;
			else
				dc->modified = TRUE;
		}
		else
			return;
		g_free (autosave);
	}
	
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	result = gnome_vfs_seek (handle, GNOME_VFS_SEEK_START, 0);
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	result = gnome_vfs_read (handle, input_data, 4096, &bytes_read);
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	result = gnome_vfs_close (handle);
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	offset = 0;
	
	len = get_line_len (input_data, bytes_read, offset);
	memcpy (value, input_data + offset, len);
	value [len] = '\0';
	offset += len + 1;
	gtk_entry_set_text (GTK_ENTRY (dc->journal_subject), value);
	
	len = get_line_len (input_data, bytes_read, offset);
	memcpy (value, input_data + offset, len);
	value [len] = '\0';
	offset += len + 1;
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dc->journal_mood)->entry), value);
	
	len = get_line_len (input_data, bytes_read, offset);
	memcpy (value, input_data + offset, len);
	value [len] = '\0';
	offset += len + 1;
	gtk_entry_set_text (GTK_ENTRY (dc->journal_music), value);
	
	len = get_line_len (input_data, bytes_read, offset);
	memcpy (value, input_data + offset, len);
	value [len] = '\0';
	offset += len + 1;
	item = (gint) g_ascii_strtod (value, NULL);
	gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_security), item);
	
	len = get_line_len (input_data, bytes_read, offset);
	memcpy (value, input_data + offset, len);
	value [len] = '\0';
	offset += len + 1;
	item = (gint) g_ascii_strtod (value, NULL);
	gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_picture), item);
	
	len = get_line_len (input_data, bytes_read, offset);
	memcpy (value, input_data + offset, len);
	value [len] = '\0';
	offset += len + 1;
	state = (gboolean) g_ascii_strtod (value, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_comment), state);
	
	len = get_line_len (input_data, bytes_read, offset);
	memcpy (value, input_data + offset, len);
	value [len] = '\0';
	offset += len + 1;
	state = (gboolean) g_ascii_strtod (value, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_autoformat), state);
	
	memcpy (value, input_data + offset, bytes_read - offset);
	value [bytes_read - offset] = '\0';
	gtk_text_buffer_set_text (buffer, value, strlen (value));
	
	dc->modified = FALSE;
	remove_autosave (dc);
	
	return;
}

void
open_draft_cb (GtkWidget *widget, gpointer data)
{
	GtkWidget *dialog;
	DrivelClient *dc = (DrivelClient *) data;
	
	dialog = gtk_file_selection_new (_("Drivel - Open Draft"));

	g_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
			"clicked",	G_CALLBACK (open_file_cb), dc);
	
	g_signal_connect_swapped (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
			"clicked",	G_CALLBACK (gtk_widget_destroy), GTK_WIDGET (dialog));
	g_signal_connect_swapped (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button),
			"clicked",	G_CALLBACK (gtk_widget_destroy), dialog);
	
	dc->open_dialog = dialog;
	
	gtk_widget_show_all (dialog);
	
	return;
}

void
save_file_cb (GtkWidget *button, gpointer data)
{
	const gchar *filename;
	gchar *output_data, *text, *subject, *mood, *music;
	gchar *autosave, *dir;
	gint security, picture, comment, autoformat;
	GnomeVFSHandle *handle;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	GnomeVFSFileSize written;
	GnomeVFSResult result;
	DrivelClient *dc = (DrivelClient *) data;
	
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (dc->journal_text));
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	if (!text)
		text = g_strdup ("");
	
	subject = gtk_editable_get_chars (GTK_EDITABLE (dc->journal_subject), 0, -1);
	if (!subject)
		subject = g_strdup ("");
	mood = gtk_editable_get_chars (GTK_EDITABLE (GTK_COMBO (dc->journal_mood)->entry), 0, -1);
	if (!mood)
		mood = g_strdup ("");
	music = gtk_editable_get_chars (GTK_EDITABLE (dc->journal_music), 0, -1);
	if (!music)
		music = g_strdup ("");
	security = gtk_option_menu_get_history (GTK_OPTION_MENU (dc->journal_security));
	picture = gtk_option_menu_get_history (GTK_OPTION_MENU (dc->journal_picture));
	comment = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->journal_comment));
	autoformat = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->journal_autoformat));
	
	output_data = g_strdup_printf (
			"%s\n"
			"%s\n"
			"%s\n"
			"%d\n"
			"%d\n"
			"%d\n"
			"%d\n"
			"%s\n",
			subject, mood, music, security, picture,
			comment, autoformat, text);
	
	/* use button to figure out if this is being called from the save dialog or autosave */
	if (button)
	{
		filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dc->save_dialog));
		result = gnome_vfs_create (&handle, filename, GNOME_VFS_OPEN_WRITE, FALSE,
			GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_WRITE);
	}
	else
	{
		if ( (dir = get_home_dir ()) )
		{
			autosave = g_strdup_printf ("%s/.drivel_autosave", dir);
			result = gnome_vfs_create (&handle, autosave, GNOME_VFS_OPEN_WRITE, FALSE,
					GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_WRITE);
		}
		else
			return;
		g_free (autosave);
	}
	
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	result = gnome_vfs_seek (handle, GNOME_VFS_SEEK_START, 0);
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	result = gnome_vfs_write (handle, output_data, strlen (output_data), &written);
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	result = gnome_vfs_close (handle);
	if (result != GNOME_VFS_OK)
	{
		display_gnomevfs_error_dialog (dc, result);
		
		return;
	}
	
	if (button)
	{
		dc->modified = FALSE;
		remove_autosave (dc);
	}
	
	g_free (output_data);
	g_free (text);
	g_free (subject);
	g_free (mood);
	g_free (music);
	
	return;
}

void
save_draft_cb (GtkWidget *widget, gpointer data)
{
	GtkWidget *dialog;
	gint response;
	DrivelClient *dc = (DrivelClient *) data;
	
	dialog = gtk_file_selection_new (_("Drivel - Save Draft"));

	dc->save_dialog = dialog;
	
	/* this needs to be modal for the save confirmation dialog to work */
	response = gtk_dialog_run (GTK_DIALOG (dialog));
	switch (response)
	{
		case GTK_RESPONSE_OK:
			save_file_cb (widget, dc);
		default:
			gtk_widget_destroy (dialog);
	}
			
	return;
}

gboolean
autosave_cb (gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	if (!dc->modified)
		return TRUE;
	
	save_file_cb (NULL, dc);
	
	return TRUE;
}

void
post_entry (DrivelClient *dc, DrivelRequestType mode)
{
	gchar *post_data, *text, *mood;
	gchar *music, *subject, *pic, *security, *key;
	gchar *mode_specific, *mesg;
	GtkTextIter start, end;
	GtkTextBuffer *buffer;
	gint i, allowmask, mood_id;
	gboolean autoformat, no_comments;
	gchar *vars [11], *values [11];
	gchar *active_journal;

	journal_enable (dc, FALSE);
				
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (dc->journal_text));
	
	gtk_text_buffer_get_bounds (buffer, &start,& end);
	
	allowmask = 0;
	text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	mood = gtk_editable_get_chars (GTK_EDITABLE (GTK_COMBO (dc->journal_mood)->entry), 0, -1);
	music = gtk_editable_get_chars (GTK_EDITABLE (dc->journal_music), 0, -1);
	subject = gtk_editable_get_chars (GTK_EDITABLE (dc->journal_subject), 0, -1);
	i = gtk_option_menu_get_history (GTK_OPTION_MENU (dc->journal_security));
	switch (i)
	{
		case 2:
		{
			security = g_strdup ("private");
			break;
		}
		case 1:
		{
			security = g_strdup ("usemask");
			allowmask = 1;
			break;
		}
		default:
			security = g_strdup ("public");
	}
	i = gtk_option_menu_get_history (GTK_OPTION_MENU (dc->journal_picture));
	key = g_strdup_printf ("pickw_%d", i);
	pic = g_hash_table_lookup (dc->picture_keywords, key);
	autoformat = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->journal_autoformat));
	no_comments = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->journal_comment));
	
	if (mood && (mesg = g_hash_table_lookup (dc->mood_icons, mood)))
		mood_id = (gint) g_ascii_strtod (mesg, NULL);
	else
		mood_id = 0;
	
	switch (mode)
	{
		case REQUEST_POSTEVENT:
		{
			fill_time (dc);
			
			mode_specific = g_strdup_printf (
					"mode=postevent");
			break;
		}
		case REQUEST_EDITEVENT:
		{
			mode_specific = g_strdup_printf (
					"mode=editevent&"
					"itemid=%s",
					dc->itemid);
			break;
		}
		default:
		{
			mode_specific = NULL;
			g_assert_not_reached ();
			break;
		}
	}
	
	vars [0] = NULL;
	vars [1] = NULL;
	vars [2] = NULL;
	vars [3] = lj_format_string ("prop_current_music");
	vars [4] = lj_format_string ("prop_current_mood");
	vars [5] = lj_format_string ("prop_current_moodid");
	vars [6] = NULL;
	vars [7] = lj_format_string ("prop_picture_keyword");
	vars [8] = lj_format_string ("prop_opt_nocomments");
	vars [9] = lj_format_string ("prop_opt_preformatted");
	vars [10] = NULL;
	values [0] = lj_format_string (dc->username);
	values [1] = NULL;
	values [2] = lj_format_string (text);
	values [3] = lj_format_string (music);
	values [4] = lj_format_string (mood);
	values [5] = NULL;
	values [6] = lj_format_string (subject);
	values [7] = lj_format_string (pic);
	values [8] = NULL;
	values [9] = NULL;
	values [10] = lj_format_string (dc->active_journal);
	
	if (dc->active_journal)
		active_journal = g_strdup_printf ("&usejournal=%s", values [10]);
	else
		active_journal = g_strdup ("");
	
	post_data = g_strdup_printf (
			"%s&ver=1&user=%s&hpassword=%s&"
			"event=%s&lineendings=unix&"
			"%s=%s&%s=%s&"
			"%s=%d&subject=%s&"
			"security=%s&allowmask=%d&"
			"%s=%s&"
			"year=%04d&mon=%02d&day=%02d&"
			"hour=%02d&min=%02d&"
			"%s=%d&"
			"%s=%d"
			"%s",
			mode_specific, values [0], dc->password,
			values [2],
			vars [3], values [3], vars [4], values [4],
			vars [5], mood_id, values [6],
			security, allowmask, vars [7], values [7],
			dc->time.year, dc->time.month,
			dc->time.day, dc->time.hour, dc->time.minute,
			vars [8], no_comments,
			vars [9], autoformat,
			active_journal);

	lj_send_request (dc, post_data, REQUEST_POSTEVENT);

	g_free (mode_specific);	
	g_free (text);
	g_free (mood);
	g_free (music);
	g_free (subject);
	g_free (security);
	g_free (active_journal);
	
	for (i = 0; i < 11; i++)
	{
		if (vars [i])
			curl_free (vars [i]);
		if (values [i])
			curl_free (values [i]);
	}
			
	return;
}

void
journal_display_defaults (DrivelClient *dc)
{
	gchar *string;
	GtkTextIter start, end;
	
	gtk_text_buffer_get_bounds (dc->buffer, &start, &end);
	
	gtk_text_buffer_delete (dc->buffer, &start, &end);
	
	gtk_entry_set_text (GTK_ENTRY (dc->journal_subject), "");

	string = get_default_text (dc->client, dc->gconf->default_mood, "");
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dc->journal_mood)->entry), string);
	g_free (string);

	string = get_default_text (dc->client, dc->gconf->default_music, "");
	gtk_entry_set_text (GTK_ENTRY (dc->journal_music), string);
	g_free (string);
	
	gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_picture),
			gconf_client_get_int (dc->client, dc->gconf->default_picture, NULL));
	
	gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_security),
			gconf_client_get_int (dc->client, dc->gconf->default_security, NULL));
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_comment),
			gconf_client_get_bool (dc->client, dc->gconf->default_comment, NULL));
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_autoformat),
			gconf_client_get_bool (dc->client, dc->gconf->default_autoformat, NULL));
	
	gtk_widget_set_sensitive (dc->menu_cut, FALSE);
	gtk_widget_set_sensitive (dc->menu_copy, FALSE);
	
	gtk_widget_grab_focus (dc->journal_text);
	
	return;
}

void
journal_edit_entry (DrivelClient *dc, const gchar *itemid, const gchar *event, const gchar *security,
		const gchar *allowmask, const gchar *subject, const gchar *mood, const gchar *music, const gchar *picture,
		const gchar *eventtime, const gchar *comments, const gchar *autoformat)
{
	gboolean bool_comments, bool_autoformat;
	gchar time [5], *pickw, *text;
	gint item, i;
	
	time [4] = '\0';
	
	memcpy (time, eventtime, 4);
	dc->time.year = (gint) g_ascii_strtod (time, NULL);
	time [2] = '\0';
	memcpy (time, eventtime + 5, 2);
	dc->time.month = (gint) g_ascii_strtod (time, NULL);
	memcpy (time, eventtime + 8, 2);
	dc->time.day = (gint) g_ascii_strtod (time, NULL);
	memcpy (time, eventtime + 11, 2);
	dc->time.hour = (gint) g_ascii_strtod (time, NULL);
	memcpy (time, eventtime + 14, 2);
	dc->time.minute = (gint) g_ascii_strtod (time, NULL);
	
	if (!security)
		item = 0;
	else if (!strcmp ("private", security))
		item = 2;
	else
		item = 1;

	gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_security), item);
	
	if (comments)
		bool_comments = (gboolean) g_ascii_strtod (comments, NULL);
	else
		bool_comments = FALSE;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_comment),
			bool_comments);
	
	if (autoformat)
		bool_autoformat = (gboolean) g_ascii_strtod (autoformat, NULL);
	else
		bool_autoformat = FALSE;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_autoformat),
			bool_autoformat);

	if (picture)
	{
		for (i = 0; i < dc->pictures + 1; i++)
		{
			pickw = g_strdup_printf ("pickw_%d", i);
			text = g_hash_table_lookup (dc->picture_keywords, pickw);

			if (!strcmp (picture, text))
				gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_picture), i);

			g_free (pickw);
		}
	}
	else
		gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_picture), 0);
		
	if (!event)
		event = g_strdup ("");
	if (!subject)
		subject = g_strdup ("");
	if (!mood)
		mood = g_strdup ("");
	if (!music)
		music = g_strdup ("");
		
	gtk_text_buffer_set_text (dc->buffer, event, -1);
	
	gtk_entry_set_text (GTK_ENTRY (dc->journal_subject), subject);
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dc->journal_mood)->entry), mood);
	gtk_entry_set_text (GTK_ENTRY (dc->journal_music), music);
	
	dc->itemid = g_strdup (itemid);
	
	gtk_widget_hide (dc->journal_post);
	gtk_widget_show (dc->edit_delete);
	gtk_widget_show (dc->edit_save);
	gtk_widget_show (dc->edit_cancel);
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_options), TRUE);
	
	gtk_widget_grab_default (dc->edit_save);
	gtk_widget_grab_focus (dc->journal_text);
	
	dc->modified = FALSE;
	
	return;
}

void
journal_edit_entry_finished (DrivelClient *dc)
{
	g_free (dc->itemid);
	dc->itemid = NULL;
	
	gtk_widget_hide (dc->edit_delete);
	gtk_widget_hide (dc->edit_save);
	gtk_widget_hide (dc->edit_cancel);
	gtk_widget_show (dc->journal_post);
		
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_options), FALSE);
	
	journal_display_defaults (dc);
	
	return;
}

void
server_list_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gint item;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	item = gconf_value_get_int (value);
	
	gtk_option_menu_set_history (GTK_OPTION_MENU (dc->pref_server_list), item);
	
	return;
}

void
server_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	const gchar *string;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	string = gconf_value_get_string (value);
	
	g_free (dc->server);
	dc->server = g_strdup (string);

	lj_http_reslv_name (dc);
	
	return;
}

void
proxy_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gboolean state;
	const gchar *proxy;
	DrivelClient *dc = (DrivelClient *) data;

	value = gconf_entry_get_value (entry);
	state = gconf_value_get_bool (value);
	
	dc->proxy = state;
	
	/* make sure the preferences dialog is shown */
	if (dc->pref_proxy_url_label)
	{
		gtk_widget_set_sensitive (dc->pref_proxy_url_label, state);
		gtk_widget_set_sensitive (dc->pref_proxy_url, state);
		gtk_widget_set_sensitive (dc->pref_proxy_port_label, state);
		gtk_widget_set_sensitive (dc->pref_proxy_port, state);
		gtk_widget_set_sensitive (dc->pref_proxy_user_label, state);
		gtk_widget_set_sensitive (dc->pref_proxy_user, state);
		gtk_widget_set_sensitive (dc->pref_proxy_pass_label, state);
		gtk_widget_set_sensitive (dc->pref_proxy_pass, state);
	}
	
	if (state)
	{
		proxy = gtk_editable_get_chars (GTK_EDITABLE (dc->pref_proxy_url), 0, -1);
		
		if (proxy && strlen (proxy) > 1)
			lj_http_reslv_name (dc);
	}
	
	return;
}

void
proxy_url_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	const gchar *url;
	DrivelClient *dc = (DrivelClient *) data;

	value = gconf_entry_get_value (entry);
	url = gconf_value_get_string (value);
	
	g_free (dc->proxy_url);
	dc->proxy_url = g_strdup (url);
	
	lj_http_reslv_name (dc);
	
	return;
}

void
proxy_port_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gint port;
	DrivelClient *dc = (DrivelClient *) data;

	value = gconf_entry_get_value (entry);
	port = gconf_value_get_int (value);
	
	dc->proxy_port = port;
	
	return;
}

void
tray_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;

	value = gconf_entry_get_value (entry);
	state = gconf_value_get_bool (value);
	
	dc->tray = state;
	
	if (state && dc->journal_window)
		tray_create (dc);
	else
		tray_destroy (dc);
	
	return;
}

void
mood_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	const gchar *string;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	string = gconf_value_get_string (value);
	
	if (dc->journal_mood)
		gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dc->journal_mood)->entry),
				string);
	
	return;
}

void
music_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	const gchar *string;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	string = gconf_value_get_string (value);
	
	if (dc->journal_music)
		gtk_entry_set_text (GTK_ENTRY (dc->journal_music), string);
	
	return;
}

void
picture_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gint num;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	num = gconf_value_get_int (value);
	
	if (dc->journal_picture)
		gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_picture),
				num);
	
	return;
}

void
security_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gint num;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	num = gconf_value_get_int (value);
	
	if (dc->journal_security)
		gtk_option_menu_set_history (GTK_OPTION_MENU (dc->journal_security),
				num);
	
	return;
}

void
comment_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	state = gconf_value_get_bool (value);

	if (dc->journal_comment)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_comment),
				state);

	return;
}

void
autoformat_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	state = gconf_value_get_bool (value);

	if (dc->journal_autoformat)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->journal_autoformat),
				state);

	return;
}

gboolean
delete_event_cb (GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	gboolean retval;
/*	
	if (gconf_client_get_bool (dc->client, dc->gconf->tray, NULL) && 
		(dc->journal_window != NULL))
	{
		retval = TRUE;
		
		tray_hide_entry (dc);
	}
	else 
	{ */
		retval = FALSE;
		
		journal_window_state_save (dc);
		
		display_save_dialog (dc);
	
		curl_global_cleanup ();
	
		gtk_main_quit ();
//	}
	
	return retval;
}

void
exit_cb (GtkWidget *widget, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	if (dc->journal_window)
		display_save_dialog (dc);
	
	gtk_main_quit ();
	
	return;
}

void
logout_cb (GtkWidget *widget, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	display_save_dialog (dc);
	
	gtk_widget_destroy (dc->journal_window);
	gtk_widget_show_all (dc->login_window);
	drivel_pop_current_window (dc);
	dc->journal_window = NULL;
	
	if (dc->picture_keywords)
	{
		hash_table_clear (dc->picture_keywords);
		dc->picture_keywords = NULL;
	}
	
	if (dc->journal_list)
	{
		g_slist_foreach (dc->journal_list, list_free_item, NULL);
		g_slist_free (dc->journal_list);
		dc->journal_list = NULL;
	}
	
	if (dc->menu_list)
	{
		g_slist_foreach (dc->menu_list, menu_list_free_item, NULL);
		g_slist_free (dc->menu_list);
		dc->menu_list = NULL;
	}
	
	g_source_remove (dc->tag_autosave);
	
	drivel_fill_journal_null (dc);
	
	journal_enable (dc, TRUE);
	
	/* clear the password if the user doesn't want it saved */
	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->save_password)))
	{
		gtk_entry_set_text (GTK_ENTRY (dc->login_password), "");
		g_free (dc->password);
		dc->password = NULL;
	}
	
	tray_destroy (dc);
	
	gtk_editable_select_region (GTK_EDITABLE (GTK_COMBO (dc->login_name)->entry), 0, -1);
	gtk_widget_grab_focus (GTK_COMBO (dc->login_name)->entry);

	return;
}

void
text_changed_cb (GtkWidget *widget, gpointer data)
{
	gint text;
	DrivelClient *dc = (DrivelClient *) data;
	
	
	text = gtk_text_buffer_get_char_count (GTK_TEXT_BUFFER (widget));
	
	if (text)
	{
		dc->modified = TRUE;
		gtk_widget_set_sensitive (dc->journal_post, TRUE);
	}
	else
	{
		dc->modified = FALSE;
		gtk_widget_set_sensitive (dc->journal_post, FALSE);
	}
	
	return;
}

void
insert_html (DrivelClient *dc, const gchar *open, const gchar *close)
{
	GtkTextIter start, end;
	GtkTextMark *mark_insert, *mark_selection;
	gboolean reverse = FALSE;

	mark_selection = gtk_text_buffer_get_selection_bound (dc->buffer);
	mark_insert = gtk_text_buffer_get_insert (dc->buffer);
	
	gtk_text_buffer_get_iter_at_mark (dc->buffer, &start, mark_selection);
	gtk_text_buffer_get_iter_at_mark (dc->buffer, &end, mark_insert);
	if (gtk_text_iter_compare (&start, &end) > 0)
	{
		gtk_text_buffer_get_iter_at_mark (dc->buffer, &start, mark_insert);
		reverse = TRUE;
	}
	gtk_text_buffer_insert (dc->buffer, &start, open, -1);

	if (!close)
		return;
		
	if (reverse)
		gtk_text_buffer_get_iter_at_mark (dc->buffer, &end, mark_selection);
	else
		gtk_text_buffer_get_iter_at_mark (dc->buffer, &end, mark_insert);
	gtk_text_buffer_insert (dc->buffer, &end, close, -1);
	
	if (reverse)
		gtk_text_buffer_get_iter_at_mark (dc->buffer, &end, mark_selection);
	else
		gtk_text_buffer_get_iter_at_mark (dc->buffer, &end, mark_insert);
	gtk_text_iter_backward_chars (&end, strlen (close));
	gtk_text_buffer_place_cursor (dc->buffer, &end);
	
	return;	
}

void
skip_tag (DrivelClient *dc, gchar *tag)
{
	gchar *text, *search_text, *search_tag;
	GtkTextMark *mark_insert;
	GtkTextIter start, end;
	gint characters = 0, matched = 0, to_match;
	gunichar ch1, ch2;
	
	mark_insert = gtk_text_buffer_get_insert (dc->buffer);
	gtk_text_buffer_get_iter_at_mark (dc->buffer, &start, mark_insert);
	gtk_text_buffer_get_end_iter (dc->buffer, &end);
		
	text = gtk_text_buffer_get_text (dc->buffer, &start, &end, FALSE);
	search_text = text;
	search_tag = tag;
	ch2 = g_utf8_get_char (search_tag);
	to_match = g_utf8_strlen (tag, -1);
	
	while (*search_text)
	{
		characters++;
		ch1 = g_utf8_get_char (search_text);
		if (ch1 == ch2)
		{
			if (++matched == to_match)
				break;
			
			search_tag = g_utf8_next_char (search_tag);
			ch2 = g_utf8_get_char (search_tag);
		}
		else if (matched)
		{
			matched = 0;
			search_tag = tag;
			ch2 = g_utf8_get_char (search_tag);
		}
		
		search_text = g_utf8_next_char (search_text);
		ch1 = g_utf8_get_char (search_text);
	}
	
	gtk_text_iter_forward_chars (&start, characters);
	gtk_text_buffer_place_cursor (dc->buffer, &start);
	
	g_free (text);
	
	return;
}

gboolean
inside_tag (DrivelClient *dc, const gchar *open, const gchar *close)
{
	GtkTextIter start, end;
	GtkTextMark *mark_insert;
	gchar *text, *location_close, *location_open;
	gboolean retval = FALSE;
	
	if (!close)
		return FALSE;
	
	if (gtk_text_buffer_get_selection_bounds (dc->buffer, &start, &end))
	{
		text = gtk_text_buffer_get_text (dc->buffer, &start, &end, FALSE);
		if (!strncmp (text, open, strlen (open)) &&
				!strncmp (text + strlen (text) - strlen (close), close, strlen (close)))
			retval = TRUE;
		g_free (text);
	}
	else
	{
		mark_insert = gtk_text_buffer_get_insert (dc->buffer);
		gtk_text_buffer_get_iter_at_mark (dc->buffer, &start, mark_insert);
		gtk_text_buffer_get_end_iter (dc->buffer, &end);
		
		text = gtk_text_buffer_get_text (dc->buffer, &start, &end, FALSE);
		
		location_close = strstr (text, close);
		location_open = strstr (text, open);
		if ((location_close && !location_open) ||
				(location_close && location_close < location_open))
			retval = TRUE;
		
		g_free (text);
	}
	
	return retval;
}

void
handle_insert_html (DrivelClient *dc, gchar *open, gchar *close)
{
	if (inside_tag (dc, open, close))
		skip_tag (dc, close);
	else
		insert_html (dc, open, close);
	
	return;
}

void
bold_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<b>", "</b>");
	
	return;
}

void
italic_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<i>", "</i>");
	
	return;
}

void
underline_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<u>", "</u>");
	
	return;
}

void
strikethrough_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<strike>", "</strike>");
	
	return;
}

void
superscript_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<sup>", "</sup>");
	
	return;
}

void
subscript_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<sub>", "</sub>");
	
	return;
}

void
list_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<ul>", "</ul>");
	
	return;
}

void
listitem_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<li>", NULL);
	
	return;
}

void
blockquote_cb (GtkWidget *menu, DrivelClient *dc)
{
	handle_insert_html (dc, "<blockquote>", "</blockquote>");
	
	return;
}

gboolean
key_press_cb (GtkWidget *text, GdkEventKey *event, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	if (event->state & GDK_CONTROL_MASK)
	{
		if (event->keyval == GDK_B || event->keyval == GDK_b)
			bold_cb (NULL, dc);
		else if (event->keyval == GDK_I || event->keyval == GDK_i)
			italic_cb (NULL, dc);
		else if (event->keyval == GDK_U || event->keyval == GDK_u)
			underline_cb (NULL, dc);
		else if (event->keyval == GDK_S || event->keyval == GDK_s)
			strikethrough_cb (NULL, dc);
	}
	
	return FALSE;
}

void
grey_edit_menu (DrivelClient *dc)
{
	GtkClipboard *clipboard;
	
	if (gtk_text_buffer_get_selection_bounds (dc->buffer, NULL, NULL))
	{
		gtk_widget_set_sensitive (dc->menu_cut, TRUE);
		gtk_widget_set_sensitive (dc->menu_copy, TRUE);
	}
	else
	{
		gtk_widget_set_sensitive (dc->menu_cut, FALSE);
		gtk_widget_set_sensitive (dc->menu_copy, FALSE);
	}
	
	clipboard = gtk_clipboard_get (GDK_NONE);
	if (gtk_clipboard_wait_is_text_available (clipboard))
		gtk_widget_set_sensitive (dc->menu_paste, TRUE);
	else
		gtk_widget_set_sensitive (dc->menu_paste, FALSE);
	
	return;
}

gboolean
text_button_release_cb (GtkWidget *text, GdkEventButton *event, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	grey_edit_menu (dc);
	
	return FALSE;
}

gboolean
text_key_release_cb (GtkWidget *text, GdkEventKey *event, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	grey_edit_menu (dc);
	
	return TRUE;
}

void
add_gconf_notifies (DrivelClient *dc)
{
	dc->id.server_id = gconf_client_notify_add (dc->client, dc->gconf->default_server, server_changed_cb,
			dc, NULL, NULL);
	dc->id.mood_id = gconf_client_notify_add (dc->client, dc->gconf->default_mood, mood_changed_cb,
			dc, NULL, NULL);
	dc->id.music_id = gconf_client_notify_add (dc->client, dc->gconf->default_music, music_changed_cb,
			dc, NULL, NULL);
	dc->id.picture_id = gconf_client_notify_add (dc->client, dc->gconf->default_picture, picture_changed_cb,
			dc, NULL, NULL);
	dc->id.security_id = gconf_client_notify_add (dc->client, dc->gconf->default_security, security_changed_cb,
			dc, NULL, NULL);
	dc->id.comment_id = gconf_client_notify_add (dc->client, dc->gconf->default_comment, comment_changed_cb,
			dc, NULL, NULL);
	dc->id.autoformat_id = gconf_client_notify_add (dc->client, dc->gconf->default_autoformat, autoformat_changed_cb,
			dc, NULL, NULL);
	dc->id.proxy_id = gconf_client_notify_add (dc->client, dc->gconf->proxy, proxy_changed_cb,
			dc, NULL, NULL);
	dc->id.proxy_url_id = gconf_client_notify_add (dc->client, dc->gconf->proxy_url, proxy_url_changed_cb,
			dc, NULL, NULL);
	dc->id.proxy_port_id = gconf_client_notify_add (dc->client, dc->gconf->proxy_port, proxy_port_changed_cb,
			dc, NULL, NULL);
	dc->id.tray_id = gconf_client_notify_add (dc->client, dc->gconf->tray, tray_changed_cb,
			dc, NULL, NULL);
			
	return;
}

void
remove_gconf_notifies (GConfClient *client, DrivelIDs *id)
{
	gconf_client_notify_remove (client, id->server_id);
	gconf_client_notify_remove (client, id->mood_id);
	gconf_client_notify_remove (client, id->music_id);
	gconf_client_notify_remove (client, id->picture_id);
	gconf_client_notify_remove (client, id->security_id);
	gconf_client_notify_remove (client, id->comment_id);
	gconf_client_notify_remove (client, id->autoformat_id);
	gconf_client_notify_remove (client, id->proxy_id);
	gconf_client_notify_remove (client, id->proxy_url_id);
	gconf_client_notify_remove (client, id->proxy_port_id);
	
	return;
}

GtkWidget *
get_focused_widget (DrivelClient *dc)
{
	GtkWidget *focused;
	
	if (gtk_widget_is_focus (dc->journal_subject))
		focused = dc->journal_subject;
	else if (gtk_widget_is_focus (GTK_COMBO (dc->journal_mood)->entry))
		focused = GTK_COMBO (dc->journal_mood)->entry;
	else if (gtk_widget_is_focus (dc->journal_music))
		focused = dc->journal_music;
	else
		focused = dc->journal_text;
	
	return focused;
}

void
cut_cb (GtkWidget *menu, gpointer data)
{
	GtkWidget *editable;
	GtkClipboard *clipboard;
	DrivelClient *dc = (DrivelClient *) data;
	
	editable = get_focused_widget (dc);
	
	if (GTK_IS_EDITABLE (editable))
		gtk_editable_cut_clipboard (GTK_EDITABLE (editable));
	else
	{
		clipboard = gtk_clipboard_get (GDK_NONE);
		gtk_text_buffer_cut_clipboard (dc->buffer, clipboard, TRUE);
	}
	
	gtk_widget_set_sensitive (dc->menu_cut, FALSE);
	gtk_widget_set_sensitive (dc->menu_copy, FALSE);
		
	return;
}

void
copy_cb (GtkWidget *menu, gpointer data)
{
	GtkWidget *editable;
	GtkClipboard *clipboard;
	DrivelClient *dc = (DrivelClient *) data;
	
	editable = get_focused_widget (dc);
	
	if (GTK_IS_EDITABLE (editable))
		gtk_editable_copy_clipboard (GTK_EDITABLE (editable));
	else
	{
		clipboard = gtk_clipboard_get (GDK_NONE);
		gtk_text_buffer_copy_clipboard (dc->buffer, clipboard);
	}
	
	return;
}

void
paste_cb (GtkWidget *menu, gpointer data)
{
	GtkWidget *editable;
	GtkClipboard *clipboard;
	DrivelClient *dc = (DrivelClient *) data;
	
	editable = get_focused_widget (dc);
	
	if (GTK_IS_EDITABLE (editable))
		gtk_editable_paste_clipboard (GTK_EDITABLE (editable));
	else
	{
		clipboard = gtk_clipboard_get (GDK_NONE);
		gtk_text_buffer_paste_clipboard (dc->buffer, clipboard, NULL, TRUE);
	}
	
	return;
}

void
clear_cb (GtkWidget *menu, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	journal_display_defaults (dc);
	
	return;
}

void
select_all_cb (GtkWidget *menu, gpointer data)
{
	GtkWidget *editable;
	GtkTextIter start, end;
	DrivelClient *dc = (DrivelClient *) data;
	
	editable = get_focused_widget (dc);
	
	if (GTK_IS_EDITABLE (editable))
		gtk_editable_select_region (GTK_EDITABLE (editable), 0, -1);
	else
	{
		gtk_text_buffer_get_bounds (dc->buffer, &start, &end);
		gtk_text_buffer_move_mark_by_name (dc->buffer, "insert", &start);
		gtk_text_buffer_move_mark_by_name (dc->buffer, "selection_bound", &end);	
	}
	
	return;
}

void
insert_link_cb (GtkWidget *menu, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	display_insert_link_dialog (dc);
	
	return;
}

void
insert_image_cb (GtkWidget *menu, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	display_insert_image_dialog (dc);
	
	return;
}

void
insert_poll_cb (GtkWidget *menu, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
//	display_insert_poll_dialog (dc);
	
	run_poll_creator_dlg (dc);
	
	return;
}

void
edit_entry_cb (GtkWidget *widget, gpointer data)
{
	gchar *post_data, *active_journal;
	gchar *values [3];
	gint i;
	DrivelClient *dc = (DrivelClient *) data;
	
	dc->edit_entry = TRUE;
	
	display_save_dialog (dc);

	journal_enable (dc, FALSE);
	
	values [0] = lj_format_string (dc->username);
	values [1] = lj_format_string ("-1");
	values [2] = lj_format_string (dc->active_journal);
	
	if (dc->active_journal)
		active_journal = g_strdup_printf ("&usejournal=%s", values [2]);
	else
		active_journal = g_strdup ("");
	
	post_data = g_strdup_printf (
		"mode=getevents&ver=1&user=%s&hpassword=%s&"
		"selecttype=one&itemid=%s&"
		"lineendings=unix"
		"%s",
		values [0], dc->password, values [1], active_journal);
	
	lj_send_request (dc, post_data, REQUEST_GETEVENTS);
	
	for (i = 0; i < 3; i++)
		curl_free (values [i]);

	g_free (active_journal);

	return;
}

void
active_journal_cb (GtkWidget *widget, gpointer data)
{
	DrivelClient *dc;
	gchar *name;
	
	name = (gchar *) data;
	dc = g_object_get_data (G_OBJECT (widget), "dc");
	
	g_free (dc->active_journal);
	if (!strncmp (name, dc->username, strlen (name)))
		dc->active_journal = NULL;
	else
		dc->active_journal = g_strdup (name);
	
	return;
}

void
edit_preferences_cb (GtkWidget *widget, gpointer data)
{
	GtkWidget *dialog;
	GSList *user_list;
	DrivelClient *dc = (DrivelClient *) data;

	if (!dc->journal_window && dc->username && strcmp (dc->username, ""))
	{
		drivel_gconf_data_fill (dc->gconf, dc->username, dc->client, &dc->id, dc);

		gconf_client_set_string (dc->client, "/apps/drivel/global_settings/current_user", dc->username, NULL);

		user_list = gconf_client_get_list (dc->client, "/apps/drivel/global_settings/user_list",
				GCONF_VALUE_STRING, NULL);
		if (user_list != NULL && g_slist_find_custom (user_list, dc->username, string_compare))
		{
		}
		else
		{
			user_list = g_slist_prepend (user_list, g_strdup (dc->username));
			gconf_client_set_list (dc->client, "/apps/drivel/global_settings/user_list", GCONF_VALUE_STRING, user_list, NULL);
		}

		g_free (dc->server);
		dc->server = gconf_client_get_string (dc->client, dc->gconf->default_server, NULL);
		if (!dc->server)
			dc->server = g_strdup ("www.livejournal.com");			

		if (dc->id.server_id)
			gconf_client_notify_remove (dc->client, dc->id.server_id);
		dc->id.server_id = gconf_client_notify_add (dc->client, dc->gconf->default_server, server_changed_cb,
			dc, NULL, NULL);
	}
	else if (!dc->username || !strcmp (dc->username, ""))
	{
		dialog = gtk_message_dialog_new (
				GTK_WINDOW (dc->current_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_CLOSE,
				_("You must enter your username first."));
		
		gtk_dialog_run (GTK_DIALOG (dialog));
		
		gtk_widget_destroy (dialog);
		
		return;
	}
	
	display_edit_preferences_dialog (dc);
	
	return;
}

void
edit_friends_cb (GtkWidget *widget, gpointer data)
{
	gchar *post_data;
	DrivelClient *dc = (DrivelClient *) data;
	
	post_data = g_strdup_printf (
			"mode=getfriends&ver=1&user=%s&"
			"hpassword=%s&includefriendof=1",
			dc->username, dc->password);
	
	lj_send_request (dc, post_data, REQUEST_GETFRIENDS);
	
	return;
}

void
edit_history_cb (GtkWidget *widget, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	display_edit_history_dialog (dc);
	
	return;
}

void
options_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	if (state)
		gtk_widget_show_all (dc->options_frame);
	else
		gtk_widget_hide_all (dc->options_frame);
	
	return;
}

void
journal_post_cb (GtkWidget *button, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	post_entry (dc, REQUEST_POSTEVENT);
	
	return;
}

void
journal_save_cb (GtkWidget *button, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	post_entry (dc, REQUEST_EDITEVENT);
	
	return;
}

void
journal_cancel_cb (GtkWidget *button, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	journal_edit_entry_finished (dc);
	
	return;
}
	
void
journal_delete_cb (GtkWidget *button, gpointer data)
{
	GtkTextIter start, end;
	GtkTextBuffer *buffer;
	DrivelClient *dc = (DrivelClient *) data;
	
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (dc->journal_text));
	
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	
	gtk_text_buffer_delete (buffer, &start, &end);
	
	post_entry (dc, REQUEST_EDITEVENT);
	
	return;
}

void
journal_window_state_save (DrivelClient *dc)
{
	gboolean max;
	gint x, y, height, width;
	GdkWindowState window_state;
	
	/* save the window position and size */
	if (dc->journal_window)
	{
		gtk_window_get_position (GTK_WINDOW (dc->journal_window), &x, &y);
		gtk_window_get_size (GTK_WINDOW (dc->journal_window), &width, &height);
		window_state = gdk_window_get_state (dc->journal_window->window);
		max = (window_state & GDK_WINDOW_STATE_MAXIMIZED);
		if (!max)
		{
			gconf_client_set_int (dc->client, dc->gconf->entry_x, x, NULL);
			gconf_client_set_int (dc->client, dc->gconf->entry_y, y, NULL);
			gconf_client_set_int (dc->client, dc->gconf->entry_height, height, NULL);
			gconf_client_set_int (dc->client, dc->gconf->entry_width, width, NULL);
		}
		gconf_client_set_bool (dc->client, dc->gconf->entry_max, max, NULL);
	}
	
	return;
}

void
journal_window_state_restore (DrivelClient *dc)
{
	gboolean max;
	gint x, y, height, width;
	
	if (dc->journal_window)
	{
		x = gconf_client_get_int (dc->client, dc->gconf->entry_x, NULL);
		y = gconf_client_get_int (dc->client, dc->gconf->entry_y, NULL);
		height = gconf_client_get_int (dc->client, dc->gconf->entry_height, NULL);
		width = gconf_client_get_int (dc->client, dc->gconf->entry_width, NULL);
		max = gconf_client_get_bool (dc->client, dc->gconf->entry_max, NULL);
		
		if (height < 1)
			height = 400;
		if (width < 1)
			width = 600;
		
		gtk_window_move (GTK_WINDOW (dc->journal_window), x, y);
		gtk_window_resize (GTK_WINDOW (dc->journal_window), width, height);
		if (max)
			gtk_window_maximize (GTK_WINDOW (dc->journal_window));
	}
	
	return;
}

void
journal_enable (DrivelClient *dc, gboolean enable)
{
	gtk_widget_set_sensitive (dc->current_window, enable);
	
	return;
}

gchar *
fix_mnemonic_string (const gchar *string)
{
	gchar *new_string;
	gint i, j, len, count;
	
	len = strlen (string);
	
	for (i = count = 0; i < len; i++)
		if (string [i] == '_')
			count++;
	
	new_string = g_new0 (gchar, len + count + 1);
	new_string [len + count] = '\0';
	
	for (i = j = 0; i < len; i++)
	{
		if (string [i] == '_')
			new_string [j++] = '_';
		new_string [j++] = string [i];
	}
	
	return new_string;
}

void
build_active_journal_menu (DrivelClient *dc)
{
	GSList *list;
	gint i;
	gchar *string;
	GnomeUIInfo *active_journal_menu;
	GnomeUIInfo *radio_list;
	GnomeUIInfo *subtree;

	active_journal_menu = g_new0 (GnomeUIInfo, dc->journals + 2);
	radio_list = g_new0 (GnomeUIInfo, 2);
	subtree = g_new0 (GnomeUIInfo, 2);
	
	list = dc->journal_list;

	for (i = 0; i < dc->journals + 1; i++)
	{
		string = fix_mnemonic_string (list->data);
		active_journal_menu [i].type = GNOME_APP_UI_ITEM;
		active_journal_menu [i].label = string;
		active_journal_menu [i].hint = g_strdup (N_("Select a journal to post to"));
		active_journal_menu [i].moreinfo = active_journal_cb;
		active_journal_menu [i].user_data = g_strdup (list->data);
		active_journal_menu [i].unused_data = NULL;
		active_journal_menu [i].pixmap_type = GNOME_APP_PIXMAP_NONE;
		active_journal_menu [i].pixmap_info = NULL;
		active_journal_menu [i].accelerator_key = 0;
		active_journal_menu [i].ac_mods = 0;
		
		list = list->next;
	}

	active_journal_menu [i].type = GNOME_APP_UI_ENDOFINFO;
	
	radio_list [0].type = GNOME_APP_UI_RADIOITEMS;
	radio_list [0].label = NULL;
	radio_list [0].hint = NULL;
	radio_list [0].moreinfo = active_journal_menu;
	radio_list [0].user_data = NULL;
	radio_list [0].unused_data = NULL;
	radio_list [0].pixmap_type = GNOME_APP_PIXMAP_NONE;
	radio_list [0].pixmap_info = NULL;
	radio_list [0].accelerator_key = 0;
	radio_list [0].ac_mods = 0;
	
	radio_list [1].type = GNOME_APP_UI_ENDOFINFO;
	
	subtree [0].type = GNOME_APP_UI_SUBTREE;
	subtree [0].label = g_strdup (N_("Active _Journal"));
	subtree [0].hint = g_strdup (N_("Select a journal to post to"));
	subtree [0].moreinfo = radio_list;
	subtree [0].user_data = NULL;
	subtree [0].unused_data = NULL;
	subtree [0].pixmap_type = GNOME_APP_PIXMAP_NONE;
	subtree [0].pixmap_info = NULL;
	subtree [0].accelerator_key = 0;
	subtree [0].ac_mods = 0;
	
	subtree [1].type = GNOME_APP_UI_ENDOFINFO;

	gnome_app_insert_menus (GNOME_APP (dc->journal_window), _("Edit/History..."), subtree);
	
	/* add callback data */
	for (i = 0; i < dc->journals + 1; i++)
		g_object_set_data (G_OBJECT (active_journal_menu [i].widget), "dc", dc);

	return;
}

void
web_menu_cb (GtkWidget *menu, gpointer data)
{
	gchar *url = (gchar *) data;
	
	gnome_url_show (url, NULL);
	
	return;
}

gchar *
create_mnemonic_string (const gchar *string, gchar **mnemonics)
{
	gint i, j, len;
	gchar *new_string, *fixed_string, needle [2];
	gboolean added = FALSE;
	
	needle [1] = '\0';
	
	fixed_string = fix_mnemonic_string (string);
	len = strlen (fixed_string);
	
	new_string = g_new0 (gchar, len + 2);
	j = 0;

	if (mnemonics [0][0] == '\0')
	{
		new_string [j++] = '_';
		mnemonics [0][0] = fixed_string [0];
		added = TRUE;
	}
	
	for (i = 0; i < len; i++)
	{
		if (!added)
		{
			needle [0] = fixed_string [i];

			if (!strstr (mnemonics [0], needle))
			{
				new_string [j++] = '_';
				added = TRUE;
				mnemonics [0][strlen (mnemonics [0])] = fixed_string [i];
				mnemonics [0][strlen (mnemonics [0]) + 1] = '\0';
			}
		}
		new_string [j++] = fixed_string [i];
	}

	return new_string;
}
		
void
build_web_link_menu (DrivelClient *dc)
{
	GnomeUIInfo **items;
	GSList *list;
	LJMenuItem *menu;
	gint i, j, count, deepest_sub;
	gchar *mnemonics, *label, *text;
	
	deepest_sub = 0;
	
	for (list = dc->menu_list; list; list = list->next)
	{
		menu = list->data;
		if (menu->menu_index > deepest_sub)
			deepest_sub = menu->menu_index;
	}
	
	items = g_new (GnomeUIInfo *, deepest_sub + 1);
	
	for (i = deepest_sub; i >= 0; i--)
	{
		count = 0;
		j = 0;
		
		for (list = dc->menu_list; list; list = list->next)
		{
			menu = list->data;
			if (menu->menu_index == i)
				count++;
		}
		
		items [i] = g_new0 (GnomeUIInfo, count + 1);
		mnemonics = g_new0 (gchar, count + 1);
		mnemonics [0] = '\0';
		
		for (list = dc->menu_list; list; list = list->next)
		{
			menu = list->data;
			if (menu->menu_index == i)
			{
				if (!strncmp (menu->label, "-", strlen (menu->label)))
				{
					items [i][j].type = GNOME_APP_UI_SEPARATOR;
					j++;
					continue;
				}
				
				if (!menu->url)
				{
					items [i][j].type = GNOME_APP_UI_SUBTREE;
					items [i][j].label = create_mnemonic_string (menu->label, &mnemonics);
					items [i][j].moreinfo = items [menu->sub_menu];
					j++;
					continue;
				}
				
				text = create_mnemonic_string (menu->label, &mnemonics);
				label = g_strconcat (text, "...", NULL);
				g_free (text);
				
				items [i][j].type = GNOME_APP_UI_ITEM;
				items [i][j].label = label;
				items [i][j].moreinfo = web_menu_cb;
				items [i][j].user_data = g_strdup (menu->url);
				j++;
			}
		}
		
		items [i][j].type = GNOME_APP_UI_ENDOFINFO;
		
		g_free (mnemonics);
	}
	
	gnome_app_insert_menus (GNOME_APP (dc->journal_window), _("Web Links/"), items [0]);
	
	return;
}

void
journal_window_build (DrivelClient *dc)
{
	GtkWidget *window;
	GtkWidget *vbox, *vbox2, *hbox, *hbox2;
	GtkWidget *text_area;
	GtkWidget *scrolled_window;
	GtkWidget *post_button, *options_button, *xmms_button;
	GtkWidget *security, *picture;
	GtkWidget *mood, *music;
	GtkWidget *comment, *autoformat;
	GtkWidget *subject;
	GtkWidget *label;
	GtkWidget *menu, *menuitem;
	GtkWidget *options_frame;
	GtkWidget *edit_delete, *edit_save, *edit_cancel;
	GtkTextBuffer *buffer;
	GdkPixbuf *pixbuf, *pixbuf2;
	GtkWidget *image;
	gchar *title, *pickw, *text;
	gint i;

	GnomeUIInfo file_menu [] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("_Open Draft..."), N_("Open an unfinished journal entry"), open_draft_cb, GTK_STOCK_OPEN),
		GNOMEUIINFO_ITEM_STOCK (N_("_Save Draft..."), N_("Save the current text without adding it to your journal"), save_draft_cb, GTK_STOCK_SAVE),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK (N_("_Log Out"), N_("Switch usernames"), logout_cb, GTK_STOCK_UNDO),
		GNOMEUIINFO_MENU_EXIT_ITEM (exit_cb, dc),
		GNOMEUIINFO_END
	};
	
	GnomeUIInfo edit_menu [] =
	{
		GNOMEUIINFO_MENU_CUT_ITEM (cut_cb, dc),
		GNOMEUIINFO_MENU_COPY_ITEM (copy_cb, dc),
		GNOMEUIINFO_MENU_PASTE_ITEM (paste_cb, dc),
		GNOMEUIINFO_MENU_CLEAR_ITEM (clear_cb, dc),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_SELECT_ALL_ITEM (select_all_cb, dc),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK (N_("Last _Entry"), N_("Edit the last entry you posted"), edit_entry_cb, GTK_STOCK_FIND_AND_REPLACE),
		GNOMEUIINFO_ITEM_NONE (N_("_Friends..."), N_("Edit your friends page"), edit_friends_cb),
		GNOMEUIINFO_ITEM_NONE (N_("_History..."), N_("Edit or view a previous entry"), edit_history_cb),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK (N_("P_references..."), N_("Edit the program preferences"), edit_preferences_cb, GTK_STOCK_PREFERENCES),
		GNOMEUIINFO_END
	};
	
	GnomeUIInfo format_menu [] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("_Bold"), NULL, bold_cb, GTK_STOCK_BOLD),
		GNOMEUIINFO_ITEM_STOCK (N_("It_alic"), NULL, italic_cb, GTK_STOCK_ITALIC),
		GNOMEUIINFO_ITEM_STOCK (N_("_Underline"), NULL, underline_cb, GTK_STOCK_UNDERLINE),
		GNOMEUIINFO_ITEM_STOCK (N_("Stri_ke"), NULL, strikethrough_cb, GTK_STOCK_STRIKETHROUGH),
		GNOMEUIINFO_ITEM_NONE (N_("Supe_rscript"), NULL, superscript_cb),
		GNOMEUIINFO_ITEM_NONE (N_("_Subscript"), NULL, subscript_cb),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_NONE (N_("L_ist"), NULL, list_cb),
		GNOMEUIINFO_ITEM_NONE (N_("List Ite_m"), NULL, listitem_cb),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_NONE (N_("I_ndent"), NULL, blockquote_cb),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK (N_("Insert _Link..."), N_("Create a link to a web page, livejournal user, or a long journal entry"), insert_link_cb, "drivel-insert-link"),
		GNOMEUIINFO_ITEM_STOCK (N_("Insert Ima_ge..."), N_("Insert an image into your journal entry"), insert_image_cb, "drivel-insert-image"),
		GNOMEUIINFO_ITEM_NONE (N_("Insert _Poll..."), NULL, insert_poll_cb),
		
		GNOMEUIINFO_END
	};
	
	GnomeUIInfo help_menu [] =
	{
		GNOMEUIINFO_MENU_ABOUT_ITEM (about_cb, dc),
		GNOMEUIINFO_END
	};

	GnomeUIInfo web_menu [] =
	{
		GNOMEUIINFO_END
	};
	
	GnomeUIInfo main_menu [] =
	{
		GNOMEUIINFO_MENU_FILE_TREE (file_menu),
		GNOMEUIINFO_MENU_EDIT_TREE (edit_menu),
		GNOMEUIINFO_SUBTREE (N_("Fo_rmat"), format_menu),
		GNOMEUIINFO_SUBTREE (N_("_Web Links"), web_menu),
		GNOMEUIINFO_MENU_HELP_TREE (help_menu),
		GNOMEUIINFO_END
	};
	
	file_menu [0].ac_mods = GDK_CONTROL_MASK;
	file_menu [0].accelerator_key = GDK_o;
	file_menu [1].ac_mods = GDK_CONTROL_MASK;
	file_menu [1].accelerator_key = GDK_s;

//	edit_menu [5].ac_mods = GDK_CONTROL_MASK;
//	edit_menu [5].accelerator_key = GDK_a;
	edit_menu [7].ac_mods = GDK_CONTROL_MASK;
	edit_menu [7].accelerator_key = GDK_e;

	format_menu [0].ac_mods = GDK_CONTROL_MASK;
	format_menu [0].accelerator_key = GDK_b;
	format_menu [1].ac_mods = GDK_CONTROL_MASK;
	format_menu [1].accelerator_key = GDK_i;
	format_menu [2].ac_mods = GDK_CONTROL_MASK;
	format_menu [2].accelerator_key = GDK_u;
	format_menu [3].ac_mods = GDK_CONTROL_MASK;
	format_menu [3].accelerator_key = GDK_k;
	format_menu [4].ac_mods = GDK_CONTROL_MASK;
	format_menu [4].accelerator_key = GDK_equal;
	format_menu [5].ac_mods = GDK_CONTROL_MASK;
	format_menu [5].accelerator_key = GDK_minus;
	format_menu [7].ac_mods = GDK_CONTROL_MASK;
	format_menu [7].accelerator_key = GDK_l;
	format_menu [8].ac_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
	format_menu [8].accelerator_key = GDK_l;
	format_menu [12].ac_mods = GDK_CONTROL_MASK;
	format_menu [12].accelerator_key = GDK_w;
	format_menu [13].ac_mods = GDK_CONTROL_MASK;
	format_menu [13].accelerator_key = GDK_p;
	
	title = g_strdup_printf (_("Drivel - %s's Journal"), dc->name);
	window = gnome_app_new (PACKAGE, title);
	gtk_widget_set_size_request (window, 100, 50);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 2);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 2);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, TRUE, 5);
		
	label = gtk_label_new_with_mnemonic (_("_Subject:"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);
	
	subject = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (subject), TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), subject);
	gtk_box_pack_start (GTK_BOX (hbox2), subject, TRUE, TRUE, 0);
	
	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 5);
	
	label = gtk_label_new_with_mnemonic (_("Securit_y:"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);
	
	menu = gtk_menu_new ();
	
	/* FIXME: this can't be right */
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "public.png", NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	menuitem = gtk_image_menu_item_new_with_label (_("Public"));
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "protected.png", NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	menuitem = gtk_image_menu_item_new_with_label (_("Friends-Only"));
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "private.png", NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	menuitem = gtk_image_menu_item_new_with_label (_("Private"));
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	
	security = gtk_option_menu_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), security);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (security), menu);
	gtk_box_pack_start (GTK_BOX (hbox2), security, TRUE, TRUE, 0);
	
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
				GTK_SHADOW_IN);
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);
	
	text_area = gtk_text_view_new ();
	GTK_WIDGET_SET_FLAGS (text_area, GTK_CAN_FOCUS);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (text_area), GTK_WRAP_WORD);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_area));
	gtk_container_add (GTK_CONTAINER (scrolled_window), text_area);
	
	options_frame = gtk_frame_new (_("Options"));
	gtk_container_set_border_width (GTK_CONTAINER (options_frame), 3);
	gtk_box_pack_start (GTK_BOX (vbox), options_frame, FALSE, FALSE, 0);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 1);
	gtk_container_add (GTK_CONTAINER (options_frame), vbox2);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 3);
	
	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, TRUE, 5);
	
	pixbuf2 = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "mood.png", NULL);
	pixbuf = gdk_pixbuf_scale_simple (pixbuf2, 16, 16, GDK_INTERP_BILINEAR);
	image = gtk_image_new_from_pixbuf (pixbuf);
	gtk_box_pack_start (GTK_BOX (hbox2), image, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Mood:"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);
	
	mood = gtk_combo_new ();
	gtk_combo_disable_activate (GTK_COMBO (mood));
	gtk_entry_set_activates_default (GTK_ENTRY (GTK_COMBO (mood)->entry), TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), GTK_COMBO (mood)->entry);
	dc->mood_list = g_slist_sort (dc->mood_list, string_compare);
	gtk_combo_set_popdown_strings (GTK_COMBO (mood), (GList *) dc->mood_list);
	gtk_widget_set_size_request (mood, 125, -1);
	gtk_box_pack_start (GTK_BOX (hbox2), mood, TRUE, TRUE, 0);
	
	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, TRUE, 5);
	
	pixbuf2 = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "music.png", NULL);
	pixbuf = gdk_pixbuf_scale_simple (pixbuf2, 16, 16, GDK_INTERP_BILINEAR);
	image = gtk_image_new_from_pixbuf (pixbuf);
	gtk_box_pack_start (GTK_BOX (hbox2), image, FALSE, FALSE, 6);
	
	label = gtk_label_new_with_mnemonic (_("M_usic:"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);
	
	music = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (music), TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), music);
	gtk_widget_set_size_request (music, 125, -1);
	gtk_box_pack_start (GTK_BOX (hbox2), music, TRUE, TRUE, 3);

	xmms_button = gtk_button_new_with_mnemonic (_("<- _XMMS"));
	gtk_box_pack_start (GTK_BOX (hbox2), xmms_button, FALSE, FALSE, 0);
	
	if (!xmms_load_ok ()) 
		gtk_widget_set_sensitive (xmms_button, FALSE);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 3);
	
	comment = gtk_check_button_new_with_mnemonic (_("Disa_llow comments"));
	gtk_box_pack_start (GTK_BOX (hbox), comment, TRUE, FALSE, 3);
	
	autoformat = gtk_check_button_new_with_mnemonic (_("Don't auto-format _text"));
	gtk_box_pack_start (GTK_BOX (hbox), autoformat, TRUE, FALSE, 3);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, TRUE, 5);
	
	label = gtk_label_new_with_mnemonic (_("Pictu_re:"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);

	menu = gtk_menu_new ();
	
	for (i = 0; i < dc->pictures + 1; i++)
	{
		pickw = g_strdup_printf ("pickw_%d", i);
		text = g_hash_table_lookup (dc->picture_keywords, pickw);

		menuitem = gtk_menu_item_new_with_label (text);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
		
		g_free (pickw);
	}

	picture = gtk_option_menu_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), picture);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (picture), menu);
	gtk_box_pack_start (GTK_BOX (hbox2), picture, TRUE, TRUE, 0);
/*	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 3);
	
	line = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (hbox), line, TRUE, TRUE, 10);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 3);
	
	options_save = gtk_check_button_new_with_mnemonic (_("Re_member options"));
	gtk_box_pack_start (GTK_BOX (hbox), options_save, TRUE, FALSE, 0);
*/	
	hbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox), GTK_BUTTONBOX_END);
	gtk_box_set_spacing (GTK_BOX (hbox), 10);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	
	edit_delete = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_box_pack_start (GTK_BOX (hbox), edit_delete, FALSE, TRUE, 0);
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (hbox), edit_delete, TRUE);
	
	options_button = gtk_check_button_new_with_mnemonic (_("View _Options"));
	gtk_box_pack_start (GTK_BOX (hbox), options_button, FALSE, TRUE, 0);
	
	edit_cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_box_pack_start (GTK_BOX (hbox), edit_cancel, FALSE, TRUE, 0);
	
	edit_save = gtk_button_new_from_stock (GTK_STOCK_SAVE);
	GTK_WIDGET_SET_FLAGS (edit_save, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (hbox), edit_save, FALSE, TRUE, 0);
	
	post_button = gtk_button_new_from_stock ("drivel-post");
	GTK_WIDGET_SET_FLAGS (post_button, GTK_CAN_DEFAULT);
	gtk_widget_set_sensitive (post_button, FALSE);
	gtk_box_pack_start (GTK_BOX (hbox), post_button, FALSE, TRUE, 0);
	
	gnome_app_set_contents (GNOME_APP (window), vbox);

	dc->journal_text = text_area;
	dc->journal_window = window;
	dc->journal_subject = subject;
	dc->journal_mood = mood;
	dc->journal_music = music;
	dc->journal_security = security;
	dc->journal_picture = picture;
	dc->journal_comment = comment;
	dc->journal_autoformat = autoformat;
	dc->options_frame = options_frame;
	dc->journal_options = options_button;
	dc->journal_post = post_button;
	dc->edit_delete = edit_delete;
	dc->edit_save = edit_save;
	dc->edit_cancel = edit_cancel;
	dc->modified = FALSE;
	dc->buffer = buffer;
	
	drivel_push_current_window (dc, window);
	
	gnome_app_create_menus_with_data (GNOME_APP (window), main_menu, dc);
	build_active_journal_menu (dc);
	build_web_link_menu (dc);
//	gnome_app_install_menu_hints (GNOME_APP (window), main_menu);

	dc->menu_cut = edit_menu [0].widget;
	dc->menu_copy = edit_menu [1].widget;
	dc->menu_paste = edit_menu [2].widget;
	
	gtk_widget_show_all (window);
	
	gtk_widget_hide (edit_delete);
	gtk_widget_hide (edit_save);
	gtk_widget_hide (edit_cancel);
	
	journal_display_defaults (dc);
	
	g_signal_connect (G_OBJECT (window), "delete_event",
			G_CALLBACK (delete_event_cb), dc);
	g_signal_connect (G_OBJECT (buffer), "changed",
			G_CALLBACK (text_changed_cb), dc);
/*	g_signal_connect (G_OBJECT (text_area), "key_press_event",
			G_CALLBACK (key_press_cb), dc); */
	g_signal_connect (G_OBJECT (text_area), "button_release_event",
			G_CALLBACK (text_button_release_cb), dc);
	g_signal_connect (G_OBJECT (text_area), "key_release_event",
			G_CALLBACK (text_key_release_cb), dc);
	g_signal_connect (G_OBJECT (post_button), "clicked",
			G_CALLBACK (journal_post_cb), dc);
	g_signal_connect (G_OBJECT (edit_delete), "clicked",
			G_CALLBACK (journal_delete_cb), dc);
	g_signal_connect (G_OBJECT (edit_save), "clicked",
			G_CALLBACK (journal_save_cb), dc);
	g_signal_connect (G_OBJECT (edit_cancel), "clicked",
			G_CALLBACK (journal_cancel_cb), dc);
	g_signal_connect (G_OBJECT (xmms_button), "clicked",
			G_CALLBACK (music_xmms_grab), music);
	g_signal_connect (G_OBJECT (options_button), "toggled",
			G_CALLBACK (options_cb), dc);
	
	dc->tag_autosave = g_timeout_add (5000, autosave_cb, dc);
	
	/* hide the frame if un-toggled */
	options_cb (options_button, dc);
	
	text_key_release_cb (NULL, NULL, dc);
	
	open_file_cb (NULL, dc);
	
	gtk_widget_grab_default (post_button);
	
	g_free (title);
	
	journal_window_state_restore (dc);
	
	if (gconf_client_get_bool (dc->client, dc->gconf->tray, NULL))
		tray_create (dc);
	
	if (gconf_client_get_bool (dc->client, dc->gconf->min_start, NULL))
		gtk_window_iconify (GTK_WINDOW (dc->journal_window));
	
	return;
}
