/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002-2003 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <gnome.h>
#include <gdk-pixbuf/gdk-pixdata.h>
#include <string.h>
#include <curl/curl.h>

#include <config.h>

#include "drivel.h"
#include "journal.h"
#include "network.h"
#include "dialogs.h"

void
history_list_edit_cb (GtkWidget *button, gpointer data);

void
grey_button_cb (GtkWidget *entry, gpointer data)
{
	GtkWidget *w;
	gchar *text;
	GSList *current;
	gboolean state = TRUE;
	DrivelButtonVAList *bval = (DrivelButtonVAList *) data;
	
	for (current = bval->entries; current; current = current->next)
	{
		w = GTK_WIDGET (current->data);
		
		if (!GTK_WIDGET_IS_SENSITIVE (w))
			continue;
		
		text = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	
		if (!text || !strncmp (text, "", strlen (text)))
			state = FALSE;
	
		g_free (text);
	}
	
	gtk_widget_set_sensitive (bval->button, state);
	
	return;
}

DrivelButtonVAList *
dialog_grey_button_on_invalid (GtkWidget *button, GtkWidget *entry, ...)
{
	va_list *args;
	gint signal;
	GtkWidget *w;
	DrivelButtonVAList *bval;
	
	args = g_new (va_list, 1);
	
	va_start (*args, entry);

	bval = g_new0 (DrivelButtonVAList, 1);
	bval->button = button;
	bval->entries = NULL;
	bval->signals = NULL;
	
	for (w = entry; w; w = va_arg (*args, GtkWidget *))
	{
		bval->entries = g_slist_prepend (bval->entries, w);
		signal = g_signal_connect (G_OBJECT (w), "changed",
				G_CALLBACK (grey_button_cb), bval);
		bval->signals = g_slist_prepend (bval->signals,
				GINT_TO_POINTER (signal));
	}
	
	va_end (*args);
	
	return bval;
}

void
update_history_marks (DrivelClient *dc, gint8 days [])
{
	gint i;
	
	gtk_calendar_freeze (GTK_CALENDAR (dc->history_calendar));
	
	gtk_calendar_clear_marks (GTK_CALENDAR (dc->history_calendar));
	
	for (i = 0; i < 31; i++)
		if (days [i])
			gtk_calendar_mark_day (GTK_CALENDAR (dc->history_calendar), i + 1);
	
	gtk_calendar_thaw (GTK_CALENDAR (dc->history_calendar));
	
	return;
}

void
mark_history_list (DrivelClient *dc)
{
	gchar *post_data, *values [2], *active_journal;
	guint i;
	
	values [0] = lj_format_string (dc->username);
	values [1] = lj_format_string (dc->active_journal);
	
	if (dc->active_journal)
		active_journal = g_strdup_printf ("&usejournal=%s", values [1]);
	else
		active_journal = g_strdup ("");

	post_data = g_strdup_printf (
			"mode=getdaycounts&ver=1&user=%s&hpassword=%s&%s",
			values [0], dc->password, active_journal);

	lj_send_request (dc, post_data, REQUEST_GETDAYCOUNTS);
	
	for (i = 0; i < 2; i++)
		curl_free (values [i]);
	
	g_free (active_journal);
	
	return;
}

void
mark_history_list_cb (GtkWidget *calendar, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	mark_history_list (dc);
	
	return;
}

void
dialog_close_cb (GtkWidget *widget, gint arg, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	if (arg == GTK_RESPONSE_CLOSE || arg == GTK_RESPONSE_DELETE_EVENT)
	{
		drivel_pop_current_window (dc);
		
		journal_enable (dc, TRUE);
			
		gtk_widget_destroy (widget);
	}
	
	return;
}

void
history_response_cb (GtkWidget *widget, gint arg, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	if (arg == 2)
		history_list_edit_cb (NULL, data);

	drivel_pop_current_window (dc);
	
	journal_enable (dc, TRUE);
	
	gtk_widget_destroy (widget);
	dc->edit_history_window = NULL;
	
	return;
}
	
gboolean
preferences_delete_event_cb (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	return TRUE;
}

void
preferences_close_cb (GtkWidget *widget, gint arg, gpointer data)
{
	gchar *text;
	gboolean close = TRUE;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (arg == GTK_RESPONSE_CLOSE || arg == GTK_RESPONSE_DELETE_EVENT)
	{
		if (GTK_WIDGET_IS_SENSITIVE (dc->pref_proxy_url))
		{
			text = gtk_editable_get_chars (GTK_EDITABLE (dc->pref_proxy_url),
					0, -1);
			if (!text || !strncmp (text, "", strlen (text)))
			{
				display_error_dialog (dc,
						_("Please enter a proxy server address or disable proxy support."));
				close = FALSE;
			}
			
			g_free (text);
		}
		
		if (GTK_WIDGET_IS_SENSITIVE (dc->pref_server))
		{
			text = gtk_editable_get_chars (GTK_EDITABLE (dc->pref_server),
					0, -1);
			if (!text || !strncmp (text, "", strlen (text)))
			{
				display_error_dialog (dc,
						_("Please enter a journal server address."));
				close = FALSE;
			}
			
			g_free (text);
		}

		if (close)
		{
			gtk_widget_destroy (widget);
		}
		
		drivel_pop_current_window (dc);
		
		journal_enable (dc, TRUE);
	}
	
	return;
}

void
update_friends_list (DrivelClient *dc)
{
	GSList *list_item;
	LJFriend *friend;
	GdkPixbuf *user, *community, *link, *type;
	GdkPixbuf *leftarrow, *rightarrow, *doublearrow;
	GtkTreeIter iter;
	
	gtk_list_store_clear (dc->list_store);
	
	user = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps" 
		G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S "user.png",
		NULL);
	community = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps" 
		G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S "community.png",
		NULL);
	leftarrow = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps" 
		G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S "leftarrow.png",
		NULL);
	rightarrow = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps" 
		G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S "rightarrow.png",
		NULL);
	doublearrow = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps" 
		G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S "doublearrow.png",
		NULL);
	
	for (list_item = dc->friends_list; list_item; list_item = list_item->next)
	{
		friend = list_item->data;
		if (friend->type)
			type = community;
		else
			type = user;
		
		if (friend->friend && !friend->friend_of)
			link = rightarrow;
		else if (!friend->friend && friend->friend_of)
			link = leftarrow;
		else
			link = doublearrow;
		
		gtk_list_store_append (dc->list_store, &iter);
		gtk_list_store_set (dc->list_store, &iter,
				TYPE_COL, type,
				LINK_COL, link,
				USERNAME_COL, friend->username,
				NAME_COL, friend->name,
				FRIEND_COL, friend->friend,
				FRIENDOF_COL, friend->friend_of,
				FG_COL, friend->fg,
				BG_COL, friend->bg,
				TYPE_BOOL_COL, friend->type,
				-1);
	}
	
	gtk_widget_set_sensitive (dc->friend_edit, FALSE);
	gtk_widget_set_sensitive (dc->friend_remove, FALSE);
	
	return;
}

void
update_history_list_cb (GtkWidget *widget, gpointer data)
{
	gchar *post_data, *values [2], *active_journal;
	guint day, month, year, i;
	DrivelClient *dc = (DrivelClient *) data;
	
	dc->edit_entry = FALSE;
	
	gtk_calendar_get_date (GTK_CALENDAR (dc->history_calendar),
			&year, &month, &day);
	month++; /* the months are numbered starting with 0 */
	
	values [0] = lj_format_string (dc->username);
	values [1] = lj_format_string (dc->active_journal);
	
	if (dc->active_journal)
		active_journal = g_strdup_printf ("&usejournal=%s", values [1]);
	else
		active_journal = g_strdup ("");

	post_data = g_strdup_printf (
			"mode=getevents&ver=1&user=%s&hpassword=%s&"
			"truncate=50&prefersubject=1&noprops=1&"
			"selecttype=day&year=%d&month=%d&day=%d&"
			"lineendings=unix%s",
			values [0], dc->password, year, month, day,
			active_journal);

	lj_send_request (dc, post_data, REQUEST_GETEVENTS);
	
	for (i = 0; i < 2; i++)
		curl_free (values [i]);
	
	g_free (active_journal);
	
	return;
}

void
update_history_list (DrivelClient *dc, gchar **itemid, gchar **event, gchar **eventtime, gint count)
{
	GtkTreeIter iter;
	gint i, len;
	gchar *time;
	
	gtk_list_store_clear (dc->history_store);
	
	for (i = 0; i < count; i++)
	{
		len = strlen (eventtime [i]);
		time = g_new0 (gchar, len + 1);
		memcpy (time, eventtime [i] + 10, len - 13);
		
		gtk_list_store_append (dc->history_store, &iter);
		gtk_list_store_set (dc->history_store, &iter,
				HISTORY_TIME_COL, time,
				HISTORY_SUBJECT_COL, event [i],
				HISTORY_ITEMID_COL, itemid [i],
				-1);

		g_free (time);
	}
	
	if (i == 0)
	{
		gtk_list_store_append (dc->history_store, &iter);
		gtk_list_store_set (dc->history_store, &iter,
				HISTORY_TIME_COL, "",
				HISTORY_SUBJECT_COL, _("[No journal entries]"),
				-1);
	}
	
	return;
}

void
add_friend_dialog_cb (GtkWidget *widget, gint arg, gpointer data)
{
	gchar *post_data;
	gchar *friend, *fg, *bg;
	gchar *values [4];
	gint i;
	guint8 r, g, b;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (arg == GTK_RESPONSE_CANCEL)
	{
		gtk_widget_destroy (widget);
		
		journal_enable (dc, TRUE);
	}
	else if (arg == GTK_RESPONSE_OK)
	{
		friend = gtk_editable_get_chars (GTK_EDITABLE (dc->dialog_add_friend),
				0, -1);
		gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (dc->dialog_add_fg),
				&r, &g, &b, NULL);
		fg = g_strdup_printf ("#%02X%02X%02X", r, g, b);
		gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (dc->dialog_add_bg),
				&r, &g, &b, NULL);
		bg = g_strdup_printf ("#%02X%02X%02X", r, g, b);
		
		if (!friend || !strcmp (friend, ""))
		{
			display_error_dialog (dc, _("Please enter a username to add."));
			
			return;
		}
		
		values [0] = lj_format_string (dc->username);
		values [1] = lj_format_string (friend);
		values [2] = lj_format_string (fg);
		values [3] = lj_format_string (bg);
		
		post_data = g_strdup_printf (
				"mode=editfriends&ver=1&user=%s&"
				"hpassword=%s&editfriend%%5Fadd%%5F1%%5Fuser=%s&"
				"editfriend%%5Fadd%%5F1%%5Ffg=%s&"
				"editfriend%%5Fadd%%5F1%%5Fbg=%s",
				values [0], dc->password, values [1],
				values [2], values [3]);

		lj_send_request (dc, post_data, REQUEST_EDITFRIENDS);
		
		for (i = 0; i < 4; i++)
			curl_free (values [i]);

		g_free (friend);
		g_free (fg);
		g_free (bg);

		gtk_widget_destroy (widget);
		
		journal_enable (dc, TRUE);
	}
	
	return;
}

void
edit_friend_dialog_cb (GtkWidget *widget, gint arg, gpointer data)
{
	gchar *post_data;
	gchar *friend, *fg, *bg;
	gchar *values [4];
	gint i;
	guint8 r, g, b;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (arg == GTK_RESPONSE_CANCEL)
	{
		gtk_widget_destroy (widget);
		
		journal_enable (dc, TRUE);
	}
	else if (arg == GTK_RESPONSE_OK)
	{
		friend = gtk_editable_get_chars (GTK_EDITABLE (dc->dialog_edit_friend),
				0, -1);
		gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (dc->dialog_edit_fg),
				&r, &g, &b, NULL);
		fg = g_strdup_printf ("#%02X%02X%02X", r, g, b);
		gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (dc->dialog_edit_bg),
				&r, &g, &b, NULL);
		bg = g_strdup_printf ("#%02X%02X%02X", r, g, b);
		
		values [0] = lj_format_string (dc->username);
		values [1] = lj_format_string (friend);
		values [2] = lj_format_string (fg);
		values [3] = lj_format_string (bg);
		
		post_data = g_strdup_printf (
				"mode=editfriends&ver=1&user=%s&"
				"hpassword=%s&editfriend%%5Fdelete%%5F%s=1&"
				"editfriend%%5Fadd%%5F1%%5Fuser=%s&"
				"editfriend%%5Fadd%%5F1%%5Ffg=%s&"
				"editfriend%%5Fadd%%5F1%%5Fbg=%s",
				values [0], dc->password, values [1],
				values [1], values [2], values [3]);
		
		lj_send_request (dc, post_data, REQUEST_EDITFRIENDS);
				
		for (i = 0; i < 4; i++)
			curl_free (values [i]);

		g_free (friend);
		g_free (fg);
		g_free (bg);

		gtk_widget_destroy (widget);
		
		journal_enable (dc, TRUE);
	}
	
	return;
}

/* FIXME: figure out how this function is supposed to work */
gint
sort_type_func (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer data)
{
	gboolean a_type, b_type;
	gint retval;
	
	gtk_tree_model_get (model, a, TYPE_BOOL_COL, &a_type, -1);
	gtk_tree_model_get (model, b, TYPE_BOOL_COL, &b_type, -1);

	if (a_type < b_type)
		retval = 1;
	else
		retval = 0;
	
	return retval;
}

void
entry_set_string (GConfClient *client, gchar *key, GtkWidget *entry)
{
	gchar *text;
	
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

	gconf_client_set_string (client, key, text, NULL);
	
	g_free (text);

	return;
}

void
server_list_cb (GtkWidget *menu, gpointer data)
{
	gint item;
	gchar *server;
	DrivelClient *dc = (DrivelClient *) data;
	
	item = gtk_option_menu_get_history (GTK_OPTION_MENU (menu));
	
	switch (item)
	{
		case 0:
		{
			server = g_strdup ("www.livejournal.com");
			break;
		}
		case 1:
		{
			server = g_strdup ("www.deadjournal.com");
			break;
		}
		case 2:
		{
			server = g_strdup ("www.parajournal.com");
			break;
		};
		default:
		{
			gtk_widget_set_sensitive (dc->pref_server, TRUE);
			gtk_widget_set_sensitive (dc->pref_server_label, TRUE);
			server = g_strdup ("");
		}
	}
	if (item < 3)
	{
		gtk_widget_set_sensitive (dc->pref_server, FALSE);
//		gtk_widget_set_sensitive (dc->pref_server_label, FALSE);
		gconf_client_set_string (dc->client, dc->gconf->default_server, server, NULL);
	}
	
	gtk_entry_set_text (GTK_ENTRY (dc->pref_server), server);
	gconf_client_set_int (dc->client, dc->gconf->default_server_list, item, NULL);
	
	return;
}

void
server_cb (GtkWidget *entry, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	entry_set_string (dc->client, dc->gconf->default_server, entry);

	return;
}

void
server_focus_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	gchar *text;
	DrivelClient *dc = (DrivelClient *) data;

	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	if (!text || !strncmp (text, "", strlen (text)))
	{
//		display_error_dialog (dc, _("Please enter the address of a journal server."));
//		gtk_widget_grab_focus (entry);
	}
	else
		entry_set_string (dc->client, dc->gconf->default_server, entry);

	g_free (text);
	
	return;
}

void
proxy_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	gconf_client_set_bool (dc->client, dc->gconf->proxy, state, NULL);
	
	return;
}

void
proxy_url_cb (GtkWidget *entry, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	entry_set_string (dc->client, dc->gconf->proxy_url, entry);
		
	return;
}

void
proxy_url_focus_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	gchar *text;
	DrivelClient *dc = (DrivelClient *) data;
	
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	if (!text || !strncmp (text, "", strlen (text)))
	{
//		display_error_dialog (dc, _("Please enter the address of your proxy server."));
//		gtk_widget_grab_focus (entry);
	}
	else
		entry_set_string (dc->client, dc->gconf->proxy_url, entry);
	
	g_free (text);
	
	return;
}

void
proxy_port_cb (GtkWidget *spin_button, gpointer data)
{
	gint port;
	DrivelClient *dc = (DrivelClient *) data;
	
	port = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin_button));
	
	gconf_client_set_int (dc->client, dc->gconf->proxy_port, port, NULL);
	
	return;
}

void
proxy_user_cb (GtkWidget *entry, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	entry_set_string (dc->client, dc->gconf->proxy_user, entry);
		
	return;
}

void
proxy_user_focus_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	gchar *text;
	DrivelClient *dc = (DrivelClient *) data;
	
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	if (!text)
		text = g_strdup ("");
	
	entry_set_string (dc->client, dc->gconf->proxy_user, entry);
	
	g_free (text);
	
	return;
}

void
proxy_pass_cb (GtkWidget *entry, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	entry_set_string (dc->client, dc->gconf->proxy_pass, entry);
		
	return;
}

void
proxy_pass_focus_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	gchar *text;
	DrivelClient *dc = (DrivelClient *) data;
	
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	if (!text)
		text = g_strdup ("");

	entry_set_string (dc->client, dc->gconf->proxy_pass, entry);
	
	g_free (text);
	
	return;
}

void
tray_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	gconf_client_set_bool (dc->client, dc->gconf->tray, state, NULL);
	
	return;
}

void
min_start_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	gconf_client_set_bool (dc->client, dc->gconf->min_start, state, NULL);
	
	return;
}
void
min_post_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	gconf_client_set_bool (dc->client, dc->gconf->min_post, state, NULL);
	
	return;
}

void
mood_cb (GtkWidget *entry, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	entry_set_string (dc->client, dc->gconf->default_mood, 
			GTK_COMBO (dc->preferences_mood)->entry);
	
	return;
}

void
mood_focus_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;

	entry_set_string (dc->client, dc->gconf->default_mood, entry);
	
	return;
}

void
music_cb (GtkWidget *entry, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	entry_set_string (dc->client, dc->gconf->default_music, entry);
	
	return;
}

void
music_focus_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	entry_set_string (dc->client, dc->gconf->default_music, entry);
	
	return;
}

void
picture_cb (GtkWidget *menu, gpointer data)
{
	gint number;
	DrivelClient *dc = (DrivelClient *) data;

	number = gtk_option_menu_get_history (GTK_OPTION_MENU (menu));

	gconf_client_set_int (dc->client, dc->gconf->default_picture, number, NULL);
	
	return;
}

void
security_cb (GtkWidget *menu, gpointer data)
{
	gint number;
	DrivelClient *dc = (DrivelClient *) data;

	number = gtk_option_menu_get_history (GTK_OPTION_MENU (menu));

	gconf_client_set_int (dc->client, dc->gconf->default_security, number, NULL);
	
	return;
}

void
comments_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	gconf_client_set_bool (dc->client, dc->gconf->default_comment, state, NULL);
	
	return;
}

void
autoformat_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	gconf_client_set_bool (dc->client, dc->gconf->default_autoformat, state, NULL);
	
	return;
}

void
history_list_changed (GtkTreeSelection *selection, gpointer data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *subject;
	gboolean state = FALSE;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get (model, &iter, HISTORY_SUBJECT_COL, &subject, -1);
		
		if (strcmp (subject, _("[No journal entries]")))
			state = TRUE;
	}
	
	gtk_widget_set_sensitive (dc->history_edit, state);
	
	return;
}

void
history_list_edit_cb (GtkWidget *button, gpointer data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	gchar *itemid, *values [2], *post_data, *active_journal;
	gint i;
	DrivelClient *dc = (DrivelClient *) data;
	
	dc->edit_entry = TRUE;
	
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dc->history_list));
	gtk_tree_selection_get_selected (selection, &model, &iter);
	gtk_tree_model_get (model, &iter, HISTORY_ITEMID_COL, &itemid, -1);
	
	values [0] = lj_format_string (dc->username);
	values [1] = lj_format_string (dc->active_journal);
	
	if (dc->active_journal)
		active_journal = g_strdup_printf ("&usejournal=%s", values [1]);
	else
		active_journal = g_strdup ("");

	post_data = g_strdup_printf (
			"mode=getevents&ver=1&user=%s&hpassword=%s&"
			"prefersubject=0&noprops=0&"
			"selecttype=one&itemid=%s&"
			"lineendings=unix%s",
			values [0], dc->password, itemid,
			active_journal);

	lj_send_request (dc, post_data, REQUEST_GETEVENTS);
	
	for (i = 0; i < 2; i++)
		curl_free (values [i]);
	
	g_free (active_journal);
	
	return;
}

void
history_list_row_activated_cb (GtkTreeView *view, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	history_response_cb (dc->edit_history_window, 2, dc);
	
	return;
}

void
friends_list_changed (GtkTreeSelection *selection, gpointer data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean friend;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get (model, &iter, FRIEND_COL, &friend, -1);
		
		gtk_widget_set_sensitive (dc->friend_edit, friend);
		gtk_widget_set_sensitive (dc->friend_remove, friend);
	}
	
	return;
}

void
friends_list_add_cb (GtkWidget *button, gpointer data)
{
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *color;
	GtkWidget *hbox, *hbox2, *vbox;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean friend;
	gchar *friend_name;
	static GtkWidget *dialog = NULL;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel - Add Friend"),
			GTK_WINDOW (dc->friends_list_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			NULL);
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
	dc->dialog_add_ok = gtk_dialog_add_button (GTK_DIALOG (dialog),
			GTK_STOCK_ADD, GTK_RESPONSE_OK);
	gtk_widget_set_sensitive (dc->dialog_add_ok, FALSE);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
			GTK_RESPONSE_OK);
	
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dc->friend_list));
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get (model, &iter, FRIEND_COL, &friend,
				USERNAME_COL, &friend_name, -1);
	}
	else
		friend_name = g_strdup ("");
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 5);
	
	label = gtk_label_new_with_mnemonic (_("_Friend's username"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	entry = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
	dc->dialog_add_friend = entry;
	dc->link_bval = dialog_grey_button_on_invalid (dc->dialog_add_ok, entry, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
	
	if (!friend)
		gtk_entry_set_text (GTK_ENTRY (entry), friend_name);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 5);
	
	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, FALSE, 0);
	
	label = gtk_label_new_with_mnemonic (_("_Text color"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);
	
	color = gnome_color_picker_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), color);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (color), 0, 0, 0, 0);
	dc->dialog_add_fg = color;
	gtk_box_pack_start (GTK_BOX (hbox2), color, FALSE, FALSE, 3);
	
	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, FALSE, 0);
	
	label = gtk_label_new_with_mnemonic (_("_Background color"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);
	
	color = gnome_color_picker_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), color);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (color), 255, 255, 255, 0);
	dc->dialog_add_bg = color;
	gtk_box_pack_start (GTK_BOX (hbox2), color, FALSE, FALSE, 3);
	
	g_signal_connect (G_OBJECT (dialog), "destroy",
			G_CALLBACK (gtk_widget_destroyed), &dialog);
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (add_friend_dialog_cb), dc);
			
	gtk_widget_show_all (dialog);
	
	GTK_WIDGET_SET_FLAGS (dialog, GTK_CAN_FOCUS);
	gtk_widget_grab_focus (dialog);
	GTK_WIDGET_SET_FLAGS (entry, GTK_CAN_FOCUS);
	gtk_widget_grab_focus (entry);
	
	g_free (friend_name);
	
	return;
}

void
friends_list_edit_cb (GtkWidget *button, gpointer data)
{
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *color;
	GtkWidget *hbox, *hbox2, *vbox;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *friend, *fg, *bg;
	gchar hex [5];
	guint8 r, g, b;
	static GtkWidget *dialog = NULL;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		return;
	}
	
	hex [0] = '0';
	hex [1] = 'X';
	hex [4] = '\0';
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel - Edit Friend"),
			GTK_WINDOW (dc->friends_list_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_CANCEL,
			GTK_RESPONSE_CANCEL,
			GTK_STOCK_SAVE,
			GTK_RESPONSE_OK,
			NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
			GTK_RESPONSE_OK);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 5);
	
	label = gtk_label_new (_("Friend's username:"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	entry = gtk_entry_new ();
	gtk_widget_set_sensitive (entry, FALSE);
	dc->dialog_edit_friend = entry;
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
	
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dc->friend_list));
	gtk_tree_selection_get_selected (selection, &model, &iter);
	gtk_tree_model_get (model, &iter, USERNAME_COL, &friend, -1);
	gtk_entry_set_text (GTK_ENTRY (entry), friend);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 5);
	
	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, FALSE, 0);
	
	label = gtk_label_new_with_mnemonic (_("_Text color"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);
	
	color = gnome_color_picker_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), color);
	gtk_tree_model_get (model, &iter, FG_COL, &fg, -1);
	g_snprintf (hex + 2, 3, "%s", fg + 1);
	r = (guint8) g_ascii_strtod (hex, NULL);
	g_snprintf (hex + 2, 3, "%s", fg + 3);
	g = (guint8) g_ascii_strtod (hex, NULL);
	g_snprintf (hex + 2, 3, "%s", fg + 5);
	b = (guint8) g_ascii_strtod (hex, NULL);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (color),
			r, g, b, 0);
	dc->dialog_edit_fg = color;
	gtk_box_pack_start (GTK_BOX (hbox2), color, FALSE, FALSE, 3);
	
	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, FALSE, 0);
	
	label = gtk_label_new_with_mnemonic (_("_Background color"));
	gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 3);
	
	color = gnome_color_picker_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), color);
	gtk_tree_model_get (model, &iter, BG_COL, &bg, -1);
	g_snprintf (hex + 2, 3, "%s", bg + 1);
	r = (guint8) g_ascii_strtod (hex, NULL);
	g_snprintf (hex + 2, 3, "%s", bg + 3);
	g = (guint8) g_ascii_strtod (hex, NULL);
	g_snprintf (hex + 2, 3, "%s", bg + 5);
	b = (guint8) g_ascii_strtod (hex, NULL);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (color),
			r, g, b, 0);
	dc->dialog_edit_bg = color;
	gtk_box_pack_start (GTK_BOX (hbox2), color, FALSE, FALSE, 3);
	
	g_signal_connect (G_OBJECT (dialog), "destroy",
			G_CALLBACK (gtk_widget_destroyed), &dialog);
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (edit_friend_dialog_cb), dc);
			
	gtk_widget_show_all (dialog);
	
	g_free (friend);
	g_free (fg);
	g_free (bg);
	
	return;
}

void
friends_list_row_activated_cb (GtkTreeView *view, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean friend;
	
	selection = gtk_tree_view_get_selection (view);
	gtk_tree_selection_get_selected (selection, &model, &iter);
	gtk_tree_model_get (model, &iter, FRIEND_COL, &friend, -1);
	
	if (friend)
		friends_list_edit_cb (NULL, data);
	else
		friends_list_add_cb (NULL, data);
	
	return;
}

void
friends_list_remove_cb (GtkWidget *widget, gpointer data)
{
	gchar *post_data;
	gchar *friend;
	gchar *values [2];
	gint i;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	DrivelClient *dc = (DrivelClient *) data;
	
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dc->friend_list));
	gtk_tree_selection_get_selected (selection, &model, &iter);
	gtk_tree_model_get (model, &iter, USERNAME_COL, &friend, -1);
	
	values [0] = lj_format_string (dc->username);
	values [1] = lj_format_string (friend);
	
	post_data = g_strdup_printf (
			"mode=editfriends&ver=1&user=%s&"
			"hpassword=%s&editfriend%%5Fdelete%%5F%s=1",
			values [0], dc->password, values [1]);
	
	lj_send_request (dc, post_data, REQUEST_EDITFRIENDS);
				
	for (i = 0; i < 2; i++)
		curl_free (values [i]);

	g_free (friend);
	
	return;
}

void
set_link_text_cb (GtkWidget *entry, gpointer data)
{
	gchar *text;
	DrivelClient *dc = (DrivelClient *) data;
	
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	gtk_entry_set_text (GTK_ENTRY (dc->link_text), text);
	
	g_free (text);
	
	return;
}

void
link_response_cb (GtkWidget *widget, gint arg, gpointer data)
{
	gchar *text, *link, *insert;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (arg == GTK_RESPONSE_CANCEL)
		gtk_widget_destroy (widget);
	
	if (arg == GTK_RESPONSE_OK)
	{
		text = gtk_editable_get_chars (GTK_EDITABLE (dc->link_text), 0, -1);
		
		if (GTK_WIDGET_SENSITIVE (dc->link_url))
		{
			link = gtk_editable_get_chars (GTK_EDITABLE (dc->link_url), 0, -1);
			insert = g_strdup_printf (
					"<a href=\"http://%s\">%s</a>",
					link, text);
		}
		else if (GTK_WIDGET_SENSITIVE (dc->link_user))
		{
			link = gtk_editable_get_chars (GTK_EDITABLE (dc->link_user), 0, -1);
			insert = g_strdup_printf (
					"<lj user=\"%s\">", link);
		}
		else
		{
			link = NULL;
			if (text [0] == '\0') // text is empty
				insert = g_strdup ("<lj-cut>");
			else
				insert = g_strdup_printf ("<lj-cut text=\"%s\">", text);
		}
		
		gtk_widget_destroy (widget);
	
		gtk_text_buffer_insert_at_cursor (dc->buffer, insert, -1);
		
		g_free (text);
		g_free (link);
		g_free (insert);
	}
	
	journal_enable (dc, TRUE);
	
	return;
}

void
image_response_cb (GtkWidget *widget, gint arg, gpointer data)
{
	gchar *url, *alt, *height_text, *width_text, *image_text;
	gint height, width;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (arg == GTK_RESPONSE_CANCEL)
		gtk_widget_destroy (widget);
	
	if (arg == GTK_RESPONSE_OK)
	{
		url = gtk_editable_get_chars (GTK_EDITABLE (dc->image_url), 0, -1);
		alt = gtk_editable_get_chars (GTK_EDITABLE (dc->image_alt), 0, -1);
		
		height = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dc->image_height));
		width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dc->image_width));
		
		if (height)
			height_text = g_strdup_printf ("height=\"%d\"", height);
		else
			height_text = g_strdup ("");
		
		if (width)
			width_text = g_strdup_printf ("width=\"%d\"", width);
		else
			width_text = g_strdup ("");
		
		image_text = g_strdup_printf (
				"<img src=\"http://%s\" %s %s alt=\"%s\">",
				url, height_text, width_text, alt);
		
		gtk_text_buffer_insert_at_cursor (dc->buffer, image_text, -1);
		
		g_free (image_text);
		g_free (url);
		g_free (alt);
		g_free (height_text);
		g_free (width_text);
		
		gtk_widget_destroy (widget);
	}
	
	journal_enable (dc, TRUE);
	
	return;
}

void
link_url_cb (GtkWidget *button, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	gtk_widget_set_sensitive (dc->link_text, TRUE);
	gtk_widget_set_sensitive (dc->link_url, TRUE);
	gtk_widget_set_sensitive (dc->link_url_label, TRUE);
	gtk_widget_set_sensitive (dc->link_user, FALSE);
	gtk_widget_set_sensitive (dc->link_user_label, FALSE);
	
	drivel_button_list_clear (dc->link_bval);
	dc->link_bval = dialog_grey_button_on_invalid (
			dc->link_ok, dc->link_text, dc->link_url, NULL);
	
	g_signal_emit_by_name (dc->link_text, "changed");
	
	gtk_widget_grab_focus (dc->link_url);
	
	return;
}

void
link_user_cb (GtkWidget *button, gpointer data)
{
	gchar *text;
	DrivelClient *dc = (DrivelClient *) data;
	
	gtk_widget_set_sensitive (dc->link_text, FALSE);
	gtk_widget_set_sensitive (dc->link_url, FALSE);
	gtk_widget_set_sensitive (dc->link_url_label, FALSE);
	gtk_widget_set_sensitive (dc->link_user, TRUE);
	gtk_widget_set_sensitive (dc->link_user_label, TRUE);
	
	drivel_button_list_clear (dc->link_bval);
	dc->link_bval = dialog_grey_button_on_invalid (
			dc->link_ok, dc->link_user, NULL);
	
	text = gtk_editable_get_chars (GTK_EDITABLE (dc->link_user), 0, -1);
	gtk_entry_set_text (GTK_ENTRY (dc->link_text), text);
	g_free (text);

	g_signal_emit_by_name (dc->link_user, "changed");
	
	gtk_widget_grab_focus (dc->link_user);
	
	return;
}

void
link_cut_cb (GtkWidget *button, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	gtk_widget_set_sensitive (dc->link_text, TRUE);
	gtk_widget_set_sensitive (dc->link_url, FALSE);
	gtk_widget_set_sensitive (dc->link_url_label, FALSE);
	gtk_widget_set_sensitive (dc->link_user, FALSE);
	gtk_widget_set_sensitive (dc->link_user_label, FALSE);
	gtk_widget_set_sensitive (dc->link_ok, TRUE);
	
	drivel_button_list_clear (dc->link_bval);
	dc->link_bval = NULL;
	
	g_signal_emit_by_name (dc->link_text, "changed");
	
	gtk_widget_grab_focus (dc->link_text);
	
	return;
}

/****************************/
/*    GLOBAL FUNCTIONS      */
/****************************/

void
display_gnomevfs_error_dialog (DrivelClient *dc, GnomeVFSResult result)
{
	GtkWidget *dialog;
	
	dialog = gtk_message_dialog_new (
				GTK_WINDOW (dc->current_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_CLOSE,
				gnome_vfs_result_to_string (result));
	
	gtk_dialog_run (GTK_DIALOG (dialog));
		
	gtk_widget_destroy (dialog);
	
	return;
}

void
display_save_dialog (DrivelClient *dc)
{
	GtkWidget *dialog;
	gint response;
	
	if (!dc || !dc->journal_window)
		return;
	
	if (dc->modified)
	{
		dialog = gtk_message_dialog_new (
				GTK_WINDOW (dc->current_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_QUESTION,
				GTK_BUTTONS_NONE,
				_("Your current entry has not been saved."));

		gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				"drivel-dont-save", GTK_RESPONSE_NONE, "drivel-save-draft", 1, NULL);
		gtk_dialog_set_default_response (GTK_DIALOG (dialog), 1);
		
		response = gtk_dialog_run (GTK_DIALOG (dialog));
		
		if (response == 1)
			save_draft_cb (NULL, dc);

		remove_autosave (dc);
		
		gtk_widget_destroy (dialog);
		
		dc->modified = FALSE;
	}
	
	return;
}

void
display_edit_history_dialog (DrivelClient *dc)
{
	GtkWidget *calendar;
	GtkWidget *vbox;
	GtkWidget *scrolled_window;
	GtkWidget *list;
	GtkListStore *store;
	GtkTreeSelection *selection;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	static GtkWidget *dialog = NULL;
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel - Edit History"),
			GTK_WINDOW (dc->current_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			NULL);
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
	dc->history_edit = gtk_dialog_add_button (GTK_DIALOG (dialog),
			"drivel-edit", 2);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), 2);
	gtk_widget_set_size_request (dialog, 375, 400);
	dc->edit_history_window = dialog;
	
	journal_enable (dc, FALSE);
	drivel_push_current_window (dc, dialog);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	
	calendar = gtk_calendar_new ();
	dc->history_calendar = calendar;
	g_signal_connect (G_OBJECT (calendar), "day-selected",
			G_CALLBACK (update_history_list_cb), dc);
	g_signal_connect (G_OBJECT (calendar), "month-changed",
			G_CALLBACK (mark_history_list_cb), dc);
	gtk_box_pack_start (GTK_BOX (vbox), calendar, FALSE, FALSE, 3);
	
	store = gtk_list_store_new (
			HISTORY_N_COLUMNS,
			G_TYPE_STRING,
			G_TYPE_STRING,
			G_TYPE_STRING);
	dc->history_store = store;
	
	list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

	g_signal_connect (G_OBJECT (selection), "changed",
			G_CALLBACK (history_list_changed), dc);
	g_signal_connect (G_OBJECT (list), "row_activated",
			G_CALLBACK (history_list_row_activated_cb), dc);

	dc->history_list = list;
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (
			_("Time"),
			renderer,
			"text", HISTORY_TIME_COL,
			NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (
			_("Journal Entry"),
			renderer,
			"text", HISTORY_SUBJECT_COL,
			NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
	
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
			GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (scrolled_window), list);
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 3);
	
	g_signal_connect (G_OBJECT (dialog), "destroy",
			G_CALLBACK (gtk_widget_destroyed), &dialog);
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (history_response_cb), dc);
	
	gtk_widget_set_sensitive (dc->history_edit, FALSE);

	gtk_widget_show_all (dialog);
	
	mark_history_list (dc);
	update_history_list_cb (NULL, dc);
	
	gtk_widget_grab_focus (calendar);
	
	return;
}

void
display_edit_friends_dialog (DrivelClient *dc)
{
	GtkListStore *store;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column, *default_col;
	GtkTreeSelection *selection;
	GtkWidget *list;
	GtkWidget *scrolled_window;
	GtkWidget *vbox;
	GtkWidget *button_box;
	GtkWidget *button;
	static GtkWidget *dialog = NULL;
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		update_friends_list (dc);
		
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel - Edit Friends"),
			GTK_WINDOW (dc->current_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_CLOSE,
			GTK_RESPONSE_CLOSE,
			NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
	gtk_widget_set_size_request (dialog, 425, 350);
	dc->friends_list_window = dialog;
	
	drivel_push_current_window (dc, dialog);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	
	store = gtk_list_store_new (N_COLUMNS,
			GDK_TYPE_PIXBUF,
			GDK_TYPE_PIXBUF,
			G_TYPE_STRING,
			G_TYPE_STRING,
			G_TYPE_BOOLEAN,
			G_TYPE_BOOLEAN,
			G_TYPE_STRING,
			G_TYPE_STRING,
			G_TYPE_BOOLEAN);
	dc->list_store = store;
	
	list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (selection), "changed",
			G_CALLBACK (friends_list_changed), dc);
	g_signal_connect (G_OBJECT (list), "row_activated",
			G_CALLBACK (friends_list_row_activated_cb), dc);
	dc->friend_list = list;
	
	renderer = gtk_cell_renderer_pixbuf_new ();
	column = gtk_tree_view_column_new_with_attributes (
			_("Type"),
			renderer,
			"pixbuf", TYPE_COL,
			NULL);
/*	gtk_tree_view_column_set_sort_column_id (column, SORT_TYPE); */
	gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
	
	renderer = gtk_cell_renderer_pixbuf_new ();
	column = gtk_tree_view_column_new_with_attributes (
			_("Link"),
			renderer,
			"pixbuf", LINK_COL,
			NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (
			_("Username"),
			renderer,
			"text", USERNAME_COL,
			NULL);
	gtk_tree_view_column_set_sort_column_id (column, SORT_USERNAME);
	default_col = column;
	gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (
			_("Name"),
			renderer,
			"text", NAME_COL,
			NULL);
	gtk_tree_view_column_set_sort_column_id (column, SORT_REALNAME);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
	
	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (dc->list_store),
			SORT_TYPE, sort_type_func, NULL, NULL);
	
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
			GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (scrolled_window), list);
	
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);
	
	button_box = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (button_box), GTK_BUTTONBOX_START);
	gtk_box_set_spacing (GTK_BOX (button_box), 10);
	gtk_box_pack_start (GTK_BOX (vbox), button_box, FALSE, FALSE, 5);
	
	button = gtk_button_new_from_stock ("drivel-add-friend");
	g_signal_connect (G_OBJECT (button), "clicked",
			G_CALLBACK (friends_list_add_cb), dc);
	gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	
	button = gtk_button_new_from_stock ("drivel-edit");
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			G_CALLBACK (friends_list_edit_cb), dc);
	dc->friend_edit = button;
	gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	
	button = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			G_CALLBACK (friends_list_remove_cb), dc);
	dc->friend_remove = button;
	gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	
	g_signal_connect (G_OBJECT (dialog), "destroy",
			G_CALLBACK (gtk_widget_destroyed), &dialog);
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (dialog_close_cb), dc);
	
	update_friends_list (dc);
	
	gtk_widget_show_all (dialog);

	gtk_tree_view_column_clicked (default_col);

	return;
}

void
display_insert_link_dialog (DrivelClient *dc)
{
	GtkWidget *vbox, *vbox2, *hbox;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *radio_button;
	static GtkWidget *dialog = NULL;
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel - Insert Link"),
			GTK_WINDOW (dc->current_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			NULL);
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
	dc->link_ok = gtk_dialog_add_button (GTK_DIALOG (dialog),
			GTK_STOCK_OK, GTK_RESPONSE_OK);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	gtk_widget_set_sensitive (dc->link_ok, FALSE);
	
	vbox = gtk_vbox_new (FALSE, 18);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	
	/* Link text */
	
	vbox2 = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, FALSE, 0);
	
	label = gtk_label_new ("");
	gtk_label_set_markup_with_mnemonic (GTK_LABEL (label), _("<b>Link _text</b>"));
	dc->link_text_label = label;
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, FALSE, 0);
	
	entry = gtk_entry_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	dc->link_text = entry;
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
	
	vbox2 = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Create a link to...</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
		
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	radio_button = gtk_radio_button_new_with_mnemonic (NULL, _("a _web page"));
	g_signal_connect (G_OBJECT (radio_button), "clicked",
			G_CALLBACK (link_url_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), radio_button, FALSE, FALSE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_http://"));
	dc->link_url_label = label;
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	
	entry = gtk_entry_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	dc->link_url = entry;
	dc->link_bval = dialog_grey_button_on_invalid (dc->link_ok, entry, dc->link_text, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	radio_button = gtk_radio_button_new_with_mnemonic_from_widget (
			GTK_RADIO_BUTTON (radio_button), _("another _LiveJournal user"));
	g_signal_connect (G_OBJECT (radio_button), "clicked",
			G_CALLBACK (link_user_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), radio_button, FALSE, FALSE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Username:"));
	gtk_widget_set_sensitive (label, FALSE);
	dc->link_user_label = label;
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	
	entry = gtk_entry_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	dc->link_user = entry;
	gtk_widget_set_sensitive (entry, FALSE);
	g_signal_connect (G_OBJECT (entry), "changed",
			G_CALLBACK (set_link_text_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	radio_button = gtk_radio_button_new_with_mnemonic_from_widget (
			GTK_RADIO_BUTTON (radio_button), _("the _rest of this entry"));
	g_signal_connect (G_OBJECT (radio_button), "clicked",
			G_CALLBACK (link_cut_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), radio_button, FALSE, FALSE, 0);
	
	g_signal_connect (G_OBJECT (dialog), "destroy",
			G_CALLBACK (gtk_widget_destroyed), &dialog);
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (link_response_cb), dc);
	
	gtk_widget_show_all (dialog);
	
	GTK_WIDGET_SET_FLAGS (dc->link_text, GTK_CAN_FOCUS);
	gtk_widget_grab_focus (dc->link_text);
	
	return;
}

void
display_insert_image_dialog (DrivelClient *dc)
{
	GtkWidget *vbox, *vbox2, *hbox;
	GtkWidget *label, *entry;
	GtkWidget *spin_button;
	GtkSizeGroup *size_group;
	static GtkWidget *dialog = NULL;
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel - Insert Image"),
			GTK_WINDOW (dc->current_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			NULL);
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
	dc->image_ok = gtk_dialog_add_button (GTK_DIALOG (dialog),
			GTK_STOCK_OK, GTK_RESPONSE_OK);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	gtk_widget_set_sensitive (dc->image_ok, FALSE);
	
	vbox = gtk_vbox_new (FALSE, 18);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	
	/* Image location */

	vbox2 = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
	
	label = gtk_label_new ("");
	gtk_label_set_markup_with_mnemonic (GTK_LABEL (label), _("<b>Image _Location</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new ("http://");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	entry = gtk_entry_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	dc->image_url = entry;
	dc->image_bval = dialog_grey_button_on_invalid (dc->image_ok, entry, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
	
	/* Image attributes */

	size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	vbox2 = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, FALSE, 0);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Image Attributes (Optional)</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, FALSE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Height:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	spin_button = gtk_spin_button_new_with_range (0, 10000, 1);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), spin_button);
	dc->image_height = spin_button;
	gtk_box_pack_start (GTK_BOX (hbox), spin_button, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, FALSE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Width:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	spin_button = gtk_spin_button_new_with_range (0, 10000, 1);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), spin_button);
	dc->image_width = spin_button;
	gtk_box_pack_start (GTK_BOX (hbox), spin_button, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, FALSE, 0);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Description:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	entry = gtk_entry_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	dc->image_alt = entry;
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
	
	g_signal_connect (G_OBJECT (dialog), "destroy",
			G_CALLBACK (gtk_widget_destroyed), &dialog);
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (image_response_cb), dc);
	
	gtk_widget_show_all (dialog);
	
	return;
}

void
display_insert_poll_dialog (DrivelClient *dc)
{
	GtkWidget *vbox;
	static GtkWidget *dialog = NULL;
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel - Insert Poll"),
			GTK_WINDOW (dc->current_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	
	vbox = gtk_vbox_new (FALSE, 18);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	
	return;
}

void
display_edit_preferences_dialog (DrivelClient *dc)
{
	GtkWidget *vbox, *vbox2, *hbox;
	GtkWidget *label, *server_list;
	GtkWidget *server, *mood, *music, *security, *picture;
	GtkWidget *comments, *autoformat, *tray;
	GtkWidget *proxy, *proxy_url, *proxy_port, *proxy_user, *proxy_pass;
	GtkWidget *min_post, *min_start;
	GtkWidget *menu, *menuitem;
	GtkWidget *notebook, *page;
	GtkSizeGroup *size_group;
	gchar *string, *pickw;
	gboolean state;
	gint i;
	static GtkWidget *dialog = NULL;
	
	journal_enable (dc, FALSE);
	
	if (dialog)
	{
		gtk_window_present (GTK_WINDOW (dialog));
		
		return;
	}
	
	dialog = gtk_dialog_new_with_buttons (
			_("Drivel Preferences"),
			GTK_WINDOW (dc->current_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_CLOSE,
			GTK_RESPONSE_CLOSE,
			NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
	
	drivel_push_current_window (dc, dialog);
	
//	hbox2 = gtk_hbox_new (FALSE, 0);
//	gtk_container_set_border_width (GTK_CONTAINER (hbox2), 12);
//	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox2, TRUE, FALSE, 0);
	
	notebook = gtk_notebook_new ();
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), notebook, TRUE, TRUE, 0);
	
	/* Notification settings */
	
	page = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (page), 12);
	label = gtk_label_new (_("General"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (page), vbox, TRUE, TRUE, 0);
	
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 6);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Notification Area</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	tray = gtk_check_button_new_with_mnemonic (_("Enable tray icon"));
	state = gconf_client_get_bool (dc->client, dc->gconf->tray, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tray), state);			
	g_signal_connect (G_OBJECT (tray), "toggled",
			G_CALLBACK (tray_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), tray, FALSE, FALSE, 3);
	
	/* Iconify settings */
	
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 6);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Entry Behavior</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	min_start = gtk_check_button_new_with_mnemonic (_("Minimize after logging in"));
	state = gconf_client_get_bool (dc->client, dc->gconf->min_start, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (min_start), state);			
	g_signal_connect (G_OBJECT (min_start), "toggled",
			G_CALLBACK (min_start_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), min_start, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	min_post = gtk_check_button_new_with_mnemonic (_("Minimize after posting"));
	state = gconf_client_get_bool (dc->client, dc->gconf->min_post, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (min_post), state);			
	g_signal_connect (G_OBJECT (min_post), "toggled",
			G_CALLBACK (min_post_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), min_post, FALSE, FALSE, 3);
	
	/* Server settings */
	
	page = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (page), 12);
	label = gtk_label_new (_("Network"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (page), vbox, TRUE, TRUE, 0);
	
	size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 6);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Journal Server</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Server:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	menu = gtk_menu_new ();
	
	menuitem = gtk_menu_item_new_with_label (_("LiveJournal"));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label (_("DeadJournal"));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label (_("ParaJournal"));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label (_("Custom..."));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	
	server_list = gtk_option_menu_new ();
	GTK_WIDGET_SET_FLAGS (server_list, GTK_CAN_FOCUS);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), server_list);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (server_list), menu);
	i = gconf_client_get_int (dc->client, dc->gconf->default_server_list, NULL);
	gtk_option_menu_set_history (GTK_OPTION_MENU (server_list), i);
	g_signal_connect (G_OBJECT (server_list), "changed",
			G_CALLBACK (server_list_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), server_list, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
		
	label = gtk_label_new_with_mnemonic (_("_Address:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	server = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (server), TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), server);
	string = get_default_text (dc->client, dc->gconf->default_server, "www.livejournal.com");
	gtk_entry_set_text (GTK_ENTRY (server), string);
	g_free (string);
	g_signal_connect (G_OBJECT (server), "focus_out_event",
			G_CALLBACK (server_focus_cb), dc);
	g_signal_connect (G_OBJECT (server), "activate",
			G_CALLBACK (server_cb), dc);
	if (i < 3)
		gtk_widget_set_sensitive (server, FALSE);
	
	gtk_box_pack_start (GTK_BOX (hbox), server, TRUE, TRUE, 3);
	dc->pref_server = server;
	dc->pref_server_label = label;
	
	/* Proxy settings */
	
	size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 6);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Proxy Server</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	proxy = gtk_check_button_new_with_mnemonic (_("P_roxy Support"));
	state = gconf_client_get_bool (dc->client, dc->gconf->proxy, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (proxy), state);			
	g_signal_connect (G_OBJECT (proxy), "toggled",
			G_CALLBACK (proxy_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), proxy, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("        ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("A_ddress:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_widget_set_sensitive (label, state);
	dc->pref_proxy_url_label = label;
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	proxy_url = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (proxy_url), TRUE);
	gtk_widget_set_sensitive (proxy_url, state);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), proxy_url);
	string = get_default_text (dc->client, dc->gconf->proxy_url, "");
	gtk_entry_set_text (GTK_ENTRY (proxy_url), string);
	g_free (string);
	g_signal_connect (G_OBJECT (proxy_url), "focus_out_event",
			G_CALLBACK (proxy_url_focus_cb), dc);
	g_signal_connect (G_OBJECT (proxy_url), "activate",
			G_CALLBACK (proxy_url_cb), dc);
	dc->pref_proxy_url = proxy_url;
	gtk_box_pack_start (GTK_BOX (hbox), proxy_url, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("        ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("P_ort:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_widget_set_sensitive (label, state);
	dc->pref_proxy_port_label = label;
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	proxy_port = gtk_spin_button_new_with_range (1.0, 80000.0, 1.0);
	gtk_widget_set_sensitive (proxy_port, state);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), proxy_port);
	i = gconf_client_get_int (dc->client, dc->gconf->proxy_port, NULL);
	if (!i)
		i = 1080;
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (proxy_port), i);			
	g_signal_connect (G_OBJECT (proxy_port), "value-changed",
			G_CALLBACK (proxy_port_cb), dc);
	dc->pref_proxy_port = proxy_port;
	gtk_box_pack_start (GTK_BOX (hbox), proxy_port, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("        ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Username:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_widget_set_sensitive (label, state);
	dc->pref_proxy_user_label = label;
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	proxy_user = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (proxy_user), TRUE);
	gtk_widget_set_sensitive (proxy_user, state);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), proxy_user);
	string = get_default_text (dc->client, dc->gconf->proxy_user, "");
	gtk_entry_set_text (GTK_ENTRY (proxy_user), string);
	g_free (string);
	g_signal_connect (G_OBJECT (proxy_user), "focus_out_event",
			G_CALLBACK (proxy_user_focus_cb), dc);
	g_signal_connect (G_OBJECT (proxy_user), "activate",
			G_CALLBACK (proxy_user_cb), dc);
	dc->pref_proxy_user = proxy_user;
	gtk_box_pack_start (GTK_BOX (hbox), proxy_user, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("        ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Password:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_widget_set_sensitive (label, state);
	dc->pref_proxy_pass_label = label;
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	proxy_pass = gtk_entry_new ();
	gtk_entry_set_visibility (GTK_ENTRY (proxy_pass), FALSE);
	gtk_entry_set_activates_default (GTK_ENTRY (proxy_pass), TRUE);
	gtk_widget_set_sensitive (proxy_pass, state);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), proxy_pass);
	string = get_default_text (dc->client, dc->gconf->proxy_pass, "");
	gtk_entry_set_text (GTK_ENTRY (proxy_pass), string);
	g_free (string);
	g_signal_connect (G_OBJECT (proxy_pass), "focus_out_event",
			G_CALLBACK (proxy_pass_focus_cb), dc);
	g_signal_connect (G_OBJECT (proxy_pass), "activate",
			G_CALLBACK (proxy_pass_cb), dc);
	dc->pref_proxy_pass = proxy_pass;
	gtk_box_pack_start (GTK_BOX (hbox), proxy_pass, TRUE, TRUE, 3);
	
	/* Default settings */
	
	page = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (page), 12);
	label = gtk_label_new (_("Entry Defaults"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (page), vbox, TRUE, TRUE, 0);
	
	size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 6);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Default Post Options</b>"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 6);
	
	label = gtk_label_new_with_mnemonic (_("_Mood:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	mood = gtk_combo_new ();
	gtk_combo_disable_activate (GTK_COMBO (mood));
	gtk_entry_set_activates_default (GTK_ENTRY (GTK_COMBO (mood)->entry), TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), GTK_COMBO (mood)->entry);
	if (dc->mood_list)
		gtk_combo_set_popdown_strings (GTK_COMBO (mood),
				(GList *) dc->mood_list);
	string = get_default_text (dc->client, dc->gconf->default_mood, " ");
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (mood)->entry), string);
	g_free (string);
	g_signal_connect (G_OBJECT (GTK_COMBO (mood)->entry), "focus_out_event",
			G_CALLBACK (mood_focus_cb), dc);
	g_signal_connect (G_OBJECT (GTK_COMBO (mood)->list), "selection_changed",
			G_CALLBACK (mood_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), mood, TRUE, TRUE, 3);
	dc->preferences_mood = mood;
	
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 6);
	
	label = gtk_label_new_with_mnemonic (_("M_usic:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	music = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (music), TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), music);
	string = get_default_text (dc->client, dc->gconf->default_music, "");
	gtk_entry_set_text (GTK_ENTRY (music), string);
	g_free (string);
	g_signal_connect (G_OBJECT (music), "focus_out_event",
			G_CALLBACK (music_focus_cb), dc);
	g_signal_connect (G_OBJECT (music), "activate",
			G_CALLBACK (music_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), music, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 6);
	
	label = gtk_label_new_with_mnemonic (_("Pictu_re:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	menu = gtk_menu_new ();
	picture = gtk_option_menu_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), picture);
	if (dc->picture_keywords)
	{	
		for (i = 0; i < dc->pictures + 1; i++)
		{
			pickw = g_strdup_printf ("pickw_%d", i);
			string = g_hash_table_lookup (dc->picture_keywords, pickw);

			menuitem = gtk_menu_item_new_with_label (string);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
		
			g_free (pickw);
		}
	}
	else
	{
		gtk_widget_set_sensitive (picture, FALSE);
		gtk_widget_set_sensitive (label, FALSE);
	}
	
	gtk_option_menu_set_menu (GTK_OPTION_MENU (picture), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (picture),
			gconf_client_get_int (dc->client, dc->gconf->default_picture, NULL));
	g_signal_connect (G_OBJECT (picture), "changed",
			G_CALLBACK (picture_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), picture, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 6);
	
	label = gtk_label_new_with_mnemonic (_("Securit_y:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	menu = gtk_menu_new ();
	
	menuitem = gtk_menu_item_new_with_label (_("Public"));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label (_("Friends-Only"));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label (_("Private"));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	
	security = gtk_option_menu_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), security);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (security), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (security),
			gconf_client_get_int (dc->client, dc->gconf->default_security, NULL));
	g_signal_connect (G_OBJECT (security), "changed",
			G_CALLBACK (security_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), security, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 6);
	
	comments = gtk_check_button_new_with_mnemonic (
			_("Disa_llow comments"));
	state = gconf_client_get_bool (dc->client, dc->gconf->default_comment, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (comments), state);
	g_signal_connect (G_OBJECT (comments), "toggled",
			G_CALLBACK (comments_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), comments, TRUE, TRUE, 3);
	
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 3);
	
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 6);
	
	autoformat = gtk_check_button_new_with_mnemonic (
			_("Don't auto-format _text"));
	state = gconf_client_get_bool (dc->client, dc->gconf->default_autoformat, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (autoformat), state);
	g_signal_connect (G_OBJECT (autoformat), "toggled",
			G_CALLBACK (autoformat_cb), dc);
	gtk_box_pack_start (GTK_BOX (hbox), autoformat, TRUE, TRUE, 3);
	
	
//	g_signal_connect (G_OBJECT (dialog), "delete_event",
//			G_CALLBACK (preferences_delete_event_cb), NULL);
	g_signal_connect (G_OBJECT (dialog), "destroy",
			G_CALLBACK (gtk_widget_destroyed), &dialog);
	g_signal_connect (G_OBJECT (dialog), "response",
			G_CALLBACK (preferences_close_cb), dc);

	gtk_widget_show_all (dialog);
	
	gtk_widget_grab_focus (server_list);
	
	return;
}
