/*
 *  The Dropline Update Applet
 *
 *  Copyright 2003 Todd Kulesza <todd@dropline.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnome/libgnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <curl/curl.h>
#include <libintl.h>
#include <locale.h>

typedef struct _DroplineData DroplineData;
typedef struct _HTTPMem HTTPMem;
	
struct _DroplineData {
	GtkWidget *applet;
	GtkWidget *image;
	GtkTooltips *tooltips;
	
	GConfClient *gconf;
	
	GAsyncQueue *net_queue;
	GMutex *net_mutex;
	GMutex *queue_mutex;
	
	PanelAppletOrient orientation;
	
	gboolean proxy_enable;
	gboolean proxy_auth;
	gchar *proxy_user;
	gchar *proxy_pass;
	gchar *proxy_host;
	gint proxy_port;
};

struct _HTTPMem
{
	gchar *data;
	guint len;
};

enum STATUS
{
	STATUS_IDLE,
	STATUS_CHECKING,
	STATUS_ERROR,
	STATUS_AVAILABLE
};

#define IS_PANEL_HORIZONTAL(o) (o == PANEL_APPLET_ORIENT_UP || o == PANEL_APPLET_ORIENT_DOWN)
#define MAX_FILES	512

static void
update_alert (DroplineData *data, gint status);

gboolean
check_for_updates_cb (gpointer gdata);

gboolean
package_disliked (const gchar *package)
{
	gboolean retval;
	gchar *cache_dir, *disliked_file, *disliked_packs, *newline;
	gchar **lines;
	gint i;
	
	if (g_file_get_contents ("/root/.dropline/cache_dir", &cache_dir, NULL, NULL))
	{
		newline = strstr (cache_dir, "\n");
		if (newline)
			cache_dir [newline - cache_dir] = '\0';
		
	}
	else
		cache_dir = g_strdup ("/var/cache/dropline-installer");
	
	disliked_file = g_strdup_printf ("%s/DroplineFilesDisliked", cache_dir);
	
	if (g_file_get_contents (disliked_file, &disliked_packs, NULL, NULL))
	{
		retval = FALSE;
		lines = g_strsplit (disliked_packs, "\n", MAX_FILES);
		for (i = 0; lines [i] && lines [i][0] != '\0' && !retval; i++)
		{
			if (!strncmp (lines [i], package, strlen (lines [i])))
				retval = TRUE;
		}
		
		g_strfreev (lines);
	}
	else
		retval = FALSE;
	
	return retval;
}

void
update_parse (DroplineData *data, const gchar *mesg)
{
	GDir *package_dir;
	const gchar *package_name_installed;
	gchar *colon;
	gchar **lines;
	gint i, j, status;
	gboolean found;
	
	lines = g_strsplit (mesg, "\n", MAX_FILES);
	for (i = 0; lines [i]; i++)
	{
		if (strstr (lines [i], ":obsolete:"))
			lines [i] = '\0';
		else
		{
			colon = strstr (lines [i], ":");
			if (colon)
				lines [i][colon - lines [i]] = '\0';
		}
	}
	
	j = 0;
	for (i = 0; lines [i] && lines [i][0] != '\0'; i++)
	{
		found = FALSE;
		package_dir = g_dir_open ("/var/log/packages", 0, NULL);
		while ((package_name_installed = g_dir_read_name (package_dir)) && !found)
		{
			if (!strncmp (package_name_installed, lines [i], 
					strlen (package_name_installed)))
				found = TRUE;
		}
		g_dir_close (package_dir);
		if (!found)
		{
			colon = strstr (lines [i], ".");
			lines [i][colon - lines[i] - 2] = '\0';
			if (!package_disliked (lines [i]))
				j++;
		}
	}

	if (!j)
		status = STATUS_IDLE;
	else
		status = STATUS_AVAILABLE;

	update_alert (data, status);
	
	g_strfreev (lines);
	
	return;
}

size_t
net_write_cb (void *ptr, size_t size, size_t nmemb, void *data)
{
	gint realsize = size * nmemb;
	HTTPMem *memory = data;

	memory->data = (gchar *) g_realloc(memory->data, memory->len + realsize + 1);

	memcpy (&(memory->data [memory->len]), ptr, realsize);
	memory->len += realsize;
	memory->data [memory->len] = '\0';

	return realsize;
}

gpointer
net_send_request (gpointer gdata)
{
	CURL *session;
	HTTPMem *memory;
	gchar *user, *pass, *userpwd;
	DroplineData *data = (DroplineData *) gdata;
	
	g_mutex_lock (data->net_mutex);
		
	memory = g_new (HTTPMem, 1);
	memory->len = 0;
	memory->data = NULL;
	
	user = pass = userpwd = NULL;
	
	session = curl_easy_init ();
	
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, net_write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, "http://www.dropline.net/gnome/DroplineFiles2c");
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, TRUE);
	curl_easy_setopt (session, CURLOPT_USERAGENT, "Dropline Alert " VERSION);
//	curl_easy_setopt (session, CURLOPT_TIMEOUT, 45);
//	curl_easy_setopt (session, CURLOPT_NOSIGNAL, 1);

	// Proxy stuff
	if (data->proxy_enable)
	{
		curl_easy_setopt (session, CURLOPT_PROXY, data->proxy_host);
		curl_easy_setopt (session, CURLOPT_PROXYPORT, data->proxy_port);
		if (data->proxy_auth)
		{
			userpwd = g_strdup_printf ("%s:%s", user, pass);
			curl_easy_setopt (session, CURLOPT_PROXYUSERPWD, userpwd);
		}
	}
	
	curl_easy_perform (session);
	
	g_mutex_lock (data->queue_mutex);
	if (memory->data)
		g_async_queue_push (data->net_queue, g_strdup (memory->data));
	else
		g_async_queue_push (data->net_queue, g_strdup ("error"));
	g_mutex_unlock (data->queue_mutex);
	
	curl_easy_cleanup (session);
	
	g_free (user);
	g_free (pass);
	g_free (userpwd);
	g_free (memory->data);
	g_free (memory);
	
	g_mutex_unlock (data->net_mutex);
	
	return NULL;
}

static void
update_alert (DroplineData *data, gint status)
{
	const gchar *tip;
	GdkPixbuf *pixbuf;
	
	switch (status)
	{
	case STATUS_IDLE:
	{
		pixbuf = gdk_pixbuf_new_from_file (GNOME_ICONDIR"/dropline/applet-idle.png", NULL);
		tip = _("No Dropline GNOME updates are available");
		break;
	}
	case STATUS_CHECKING:
	{
		pixbuf = gdk_pixbuf_new_from_file (GNOME_ICONDIR"/dropline/applet-checking.png", NULL);
		tip = _("Looking for new Dropline GNOME software...");
		break;
	}
	case STATUS_AVAILABLE:
	{
		pixbuf = gdk_pixbuf_new_from_file (GNOME_ICONDIR"/dropline/applet-available.png", NULL);
		tip = _("New Dropline GNOME software is available");
		break;
	}
	default:
	{
		pixbuf = gdk_pixbuf_new_from_file (GNOME_ICONDIR"/dropline/applet-error.png", NULL);
		tip = _("Error while looking for updates");
		break;
	}
	}
	
	gdk_threads_enter ();
	gtk_image_set_from_pixbuf (GTK_IMAGE (data->image), pixbuf);
	gtk_tooltips_set_tip (data->tooltips, data->applet, tip, NULL);
	g_object_unref (G_OBJECT (pixbuf));
	gdk_threads_leave ();
	
	return;
}

gboolean
net_loop (gpointer gdata)
{
	gchar *mesg;
	DroplineData *data = (DroplineData *) gdata;
	
	g_mutex_lock (data->queue_mutex);
	
	if ((mesg = g_async_queue_try_pop (data->net_queue)))
	{
		if (!strncmp (mesg, "error", strlen (mesg)))
		{
			update_alert (data, STATUS_ERROR);
		}
		else
		{
			update_parse (data, mesg);
		}
		
		g_free (mesg);
	}
	
	g_mutex_unlock (data->queue_mutex);
	
	return TRUE;
}

gboolean
check_for_updates_cb (gpointer gdata)
{
	DroplineData *data = (DroplineData *) gdata;
	
	update_alert (data, STATUS_CHECKING);

	g_thread_create (net_send_request, data, FALSE, NULL);
	
	return TRUE;
}

static void
applet_destroy_cb (GtkWidget *widget, DroplineData *data)
{	
	g_free (data);
	
	curl_global_cleanup ();
	
	return;
}

static void
applet_change_background_cb (PanelApplet *applet, 
		PanelAppletBackgroundType type, GdkColor *color, const gchar *pixmap, 
		DroplineData *data)
{
	gdk_threads_enter ();
	
	if (type == PANEL_NO_BACKGROUND) {
		GtkRcStyle *rc_style = gtk_rc_style_new ();

		gtk_widget_modify_style (data->applet, rc_style);
		gtk_rc_style_unref (rc_style);
	}
	else if (type == PANEL_COLOR_BACKGROUND) {
		gtk_widget_modify_bg (data->applet,
				      GTK_STATE_NORMAL,
				      color);
	} else { /* pixmap */
		/* FIXME: Handle this when the panel support works again */
	}
	
	gdk_threads_leave ();
	
	return;
}

static void
applet_change_orient_cb (GtkWidget *w, PanelAppletOrient o, DroplineData *data)
{	
	data->orientation = o;
	
	return;
}
	
static void
dropline_update_cb (BonoboUIComponent *uic, DroplineData *data, const gchar *verbname)
{
	update_alert (data, STATUS_IDLE);
	
	gnome_execute_terminal_shell (NULL, "dropline-installer");
		
	return;
}

static void
dropline_about_cb (BonoboUIComponent *uic, DroplineData *data, const gchar *verbname)
{
	static GtkWidget *about = NULL;
	GdkPixbuf *pixbuf = NULL;
	gchar *file;
	static const gchar *authors [] = 
	{
		"Todd Kulesza <todd@dropline.net>",
		NULL
	};
	const gchar *documenters [] = 
	{
		NULL
	};
	const gchar *translator_credits = "translator_credits";
	
	if (about)
	{
		gtk_window_set_screen (GTK_WINDOW (about), 
				gtk_widget_get_screen (data->applet));
		gtk_window_present (GTK_WINDOW (about));
		
		return;
	}
        
	file = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP, 
			"dropline.png", TRUE, NULL);
	pixbuf = gdk_pixbuf_new_from_file (file, NULL);
	g_free (file);

	about = gnome_about_new (_("Dropline Update Alert"), VERSION, 
			_("Copyright 2003 Todd Kulesza"),
			_("Receive alerts when new Dropline GNOME software is available."),
			authors, documenters,
			strcmp (translator_credits, "translator_credits") != 0 ? 
					translator_credits : NULL,
			pixbuf);

	gtk_window_set_screen (GTK_WINDOW (about), 
			gtk_widget_get_screen (data->applet));
	gtk_window_set_wmclass (GTK_WINDOW(about), "dropline update alert", 
			"Dropline Update Alert");
	gnome_window_icon_set_from_file (GTK_WINDOW (about), 
			GNOME_ICONDIR"/dropline.png");
	g_signal_connect (G_OBJECT (about), "destroy",
		G_CALLBACK (gtk_widget_destroyed), &about);
	
	gtk_widget_show (about);
	
	return;
}

const BonoboUIVerb dropline_applet_menu_verbs [] = 
{
	BONOBO_UI_UNSAFE_VERB ("Update", dropline_update_cb),
	BONOBO_UI_UNSAFE_VERB ("About",    dropline_about_cb),
	
	BONOBO_UI_VERB_END
};

void
dropline_applet_create (PanelApplet *applet)
{
	DroplineData *data;
	
	curl_global_init (CURL_GLOBAL_ALL);
	
	data = g_new0 (DroplineData, 1);

	gdk_threads_enter ();
	
	data->applet = GTK_WIDGET (applet);
	data->image = gtk_image_new ();
	data->tooltips = gtk_tooltips_new ();
	data->gconf = gconf_client_get_default ();
	data->net_mutex = g_mutex_new ();
	data->queue_mutex = g_mutex_new ();
	data->net_queue = g_async_queue_new ();
	
	// not the best way to deal with gconf, but it doesn't work well when
	// called from the network thread
	data->proxy_enable = gconf_client_get_bool (data->gconf, 
			"/system/http_proxy/use_http_proxy", NULL);
	data->proxy_auth = gconf_client_get_bool (data->gconf, 
			"/system/http_proxy/use_authentication", NULL);
	data->proxy_user = gconf_client_get_string (data->gconf, 
			"/system/http_proxy/authentication_user" , NULL);
	data->proxy_pass = gconf_client_get_string (data->gconf, 
			"/system/http_proxy/authentication_password", NULL);
	data->proxy_host = gconf_client_get_string (data->gconf, 
			"/system/http_proxy/host", NULL);
	data->proxy_port = gconf_client_get_int (data->gconf, 
			"/system/http_proxy/port", NULL);
	
	panel_applet_add_preferences (applet, "/schemas/apps/dropline_applet/prefs", NULL);

	gtk_container_add (GTK_CONTAINER (data->applet), data->image);
	
	gtk_widget_show_all (GTK_WIDGET (data->applet));
	
	g_signal_connect (applet, "destroy", G_CALLBACK (applet_destroy_cb), data);
	g_signal_connect (applet, "change_orient", G_CALLBACK (applet_change_orient_cb), data);
	g_signal_connect (applet, "change_background", G_CALLBACK (applet_change_background_cb), data);

	panel_applet_setup_menu_from_file (applet, 
					   NULL, // opt_datadir
					   "Dropline_UpdateApplet.xml",
					   NULL,
					   dropline_applet_menu_verbs,
					   data);
	
	applet_change_orient_cb (GTK_WIDGET (applet), panel_applet_get_orient (applet), data);
	
	gdk_threads_leave ();
	
	update_alert (data, STATUS_IDLE);

	// use 3600000 for 1 hour timeout
	g_timeout_add (3600000, check_for_updates_cb, data);
	g_timeout_add (5000, net_loop, data);
	
	check_for_updates_cb (data);
	
	return;
}

static gboolean
dropline_applet_factory (PanelApplet *applet, const gchar *iid, gpointer data)
{
	dropline_applet_create (applet);

	return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:Dropline_UpdateApplet_Factory",
			     PANEL_TYPE_APPLET,
			     "dropline_applet",
			     "0",
			     dropline_applet_factory,
			     NULL)
