/* The GIMP -- an image manipulation program 
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis 
 * 
 * GIMP FreeType Plug-in
 * Copyright (C) 2000  Sven Neumann <sven@gimp.org>
 *                     Jens Lautenbacher <jtl@gimp.org>
 *
 * This plug-in contains code taken from the freetype2 library and 
 * its demos, which is licensed under The FreeType Project LICENSE.
 *  Copyright 1996-2000 by
 *   David Turner, Robert Wilhelm, and Werner Lemberg
 *
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 * 
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#include "config.h"

#include <glib.h>
#include <string.h>
#ifndef G_OS_WIN32
#include <dirent.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <freetype/freetype.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "freetype-intl.h"
#include "freetype-types.h"

#include "fontsel.h"
#include "interface.h"
#include "main.h"


#define FONTPATH_TOKEN   "freetype-fontpath"
#ifndef G_OS_WIN32
#define DEFAULT_FONTPATH "/usr/X11R6/lib/X11/fonts/Type1"
#else
#define DEFAULT_FONTPATH "C:\\Windows\\fonts"
#endif

typedef struct 
{
  GtkWidget *fontsel;
  GTree     *family;
  gchar     *style_name;
} StyleSelect;


static void     fontsel_insert_face             (const FT_Face  face,
						 const gchar   *file_name,
						 const gint     face_index);

static void     fontsel_family_select_callback  (GtkCList       *clist,
						 gint            row,
						 gint            column,
						 GdkEventButton *event,
						 gpointer        data);
static void     fontsel_style_select_callback   (GtkCList       *clist,
						 gint            row,
						 gint            column,
						 GdkEventButton *event,
						 gpointer        data);

static gint     fontsel_family_list_insert      (gpointer        key,
						 gpointer        value,
						 gpointer        data);
static gint     fontsel_style_list_insert       (gpointer        key,
						 gpointer        value,
						 gpointer        data);
static gint     fontsel_remove_family           (gpointer        key,
						 gpointer        value,
						 gpointer        data);
static gint     fontsel_remove_face             (gpointer        key,
						 gpointer        value,
						 gpointer        data);

static gboolean fontsel_is_font_file            (const gchar    *name);
static void     fontsel_scan_directory          (const gchar    *path);

static void     fontsel_directories_ok_callback (GtkWidget      *widget, 
						 gpointer        data);
static gboolean fontsel_directories_dialog      (gchar          *message,
					         gchar         **path);
static gint     fontsel_set_style_by_name       (StyleSelect    *select);


GTree    *families      = NULL;
FontFace *face_selected = NULL;


GtkWidget *
fontsel_new (ProgressData *pdata)
{
  GtkWidget *hbox;
  GtkWidget *clist;
  GtkWidget *clist2;
  GtkWidget *scrolled_window;

  hbox = gtk_hbox_new (TRUE, 2);

  gtk_object_class_user_signal_new (GTK_OBJECT (hbox)->klass,
				    "face_changed",
				    GTK_RUN_LAST,
				    gtk_signal_default_marshaller,
				    GTK_TYPE_NONE,
				    0,
				    NULL);

  gtk_object_set_data (GTK_OBJECT (hbox), "progress_data", pdata);

  /*  family  */
  clist = gtk_clist_new (1);
  gtk_clist_set_column_title (GTK_CLIST (clist), 0, _("Font Family"));
  gtk_clist_column_titles_show (GTK_CLIST (clist));
  gtk_clist_column_titles_passive (GTK_CLIST (clist));
  gtk_widget_show (clist);

  scrolled_window = 
    gtk_scrolled_window_new (gtk_clist_get_hadjustment (GTK_CLIST (clist)),
			     gtk_clist_get_vadjustment (GTK_CLIST (clist)));
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_ALWAYS);
  gtk_box_pack_start (GTK_BOX (hbox), scrolled_window, TRUE, TRUE, 2);
  gtk_container_add (GTK_CONTAINER (scrolled_window), clist);
  gtk_widget_show (scrolled_window);

  /*  style  */
  clist2 = gtk_clist_new (1);
  gtk_clist_set_column_title (GTK_CLIST (clist2), 0, _("Font Style"));
  gtk_clist_column_titles_show (GTK_CLIST (clist2));
  gtk_clist_column_titles_passive (GTK_CLIST (clist2));
  gtk_widget_show (clist2);

  scrolled_window = 
    gtk_scrolled_window_new (gtk_clist_get_hadjustment (GTK_CLIST (clist2)),
			     gtk_clist_get_vadjustment (GTK_CLIST (clist2)));
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_ALWAYS);
  gtk_box_pack_start (GTK_BOX (hbox), scrolled_window, TRUE, TRUE, 2);
  gtk_container_add (GTK_CONTAINER (scrolled_window), clist2);
  gtk_widget_show (scrolled_window);

  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
		      GTK_SIGNAL_FUNC (fontsel_family_select_callback),
		      clist2);
  gtk_signal_connect (GTK_OBJECT (clist2), "select_row",
		      GTK_SIGNAL_FUNC (fontsel_style_select_callback),
		      hbox);

  gtk_object_set_data (GTK_OBJECT (hbox), "family_list", clist);
  gtk_object_set_data (GTK_OBJECT (hbox), "style_list",  clist2);

  return hbox;
}

void
fontsel_configure (GtkWidget *fontsel,
		   gboolean   init)
{
  gchar *path;
  GList *list;
  GList *dirs;
  GtkWidget *clist;
  ProgressData *pdata;

  g_return_if_fail (fontsel != NULL);

  path = gimp_gimprc_query (FONTPATH_TOKEN);

  if (init)
    {
      if (path == NULL || !*path)
	{
	  path = g_strdup (DEFAULT_FONTPATH);
	  fontsel_directories_dialog
	    (_("You seem to be running the FreeType plug-in "
	       "for the first time. You need to specify a "
	       "list of directories where font files can be "
	       "found on your system."), 
	     &path);
	}
    }
  else
    {
      init = fontsel_directories_dialog (NULL, &path);
    }

  if (init)
    {
      pdata = gtk_object_get_data (GTK_OBJECT (fontsel), "progress_data");
      start_progress (pdata);

      if (families)
	{
	  g_tree_traverse (families,
			   (GTraverseFunc) fontsel_remove_family, 
			   G_IN_ORDER, 
			   NULL);
	  g_tree_destroy (families);
	}
      families = g_tree_new ((GCompareFunc)strcmp);

      dirs = gimp_path_parse (path, 128, TRUE, NULL);

      for (list = dirs; list; list = g_list_next (list))
	{
	  path = (gchar *)(list->data);
	  fontsel_scan_directory (path);
	}

      clist = gtk_object_get_data (GTK_OBJECT (fontsel), "family_list");
      gtk_clist_freeze (GTK_CLIST (clist));
      gtk_clist_clear (GTK_CLIST (clist));
      g_tree_traverse (families,
		       (GTraverseFunc)fontsel_family_list_insert, 
		       G_IN_ORDER, 
		       clist);
      gtk_clist_thaw (GTK_CLIST (clist));

      stop_progress (pdata);
    }
}

/* ugh, this is an ugly workaround until the fontselector 
   becomes a proper widget */
void
fontsel_get_selected_entry (FontFace **font_face)
{
  *font_face = face_selected;
}

void        
fontsel_set_font_by_name (GtkWidget *fontsel,
			  gchar     *family_name,
			  gchar     *style_name)
{
  GTree       *family = NULL;  
  GtkCList    *clist;
  gint         row;
  StyleSelect *style_select;

  g_return_if_fail (fontsel != NULL);

  if (family_name)
    {
      family = g_tree_lookup (families, family_name);
    }

  if (family)
    {
      clist = gtk_object_get_data (GTK_OBJECT (fontsel), "family_list");
      row = gtk_clist_find_row_from_data (GTK_CLIST (clist), family);
      gtk_clist_select_row (GTK_CLIST (clist), row, 0);

      if ( !(gtk_clist_row_is_visible (clist, row) & GTK_VISIBILITY_FULL))
	gtk_clist_moveto (clist, row, 0, 0.0, 0.0);

      if (style_name)
	{
	  style_select = g_new (StyleSelect, 1);
	  style_select->fontsel    = fontsel;
	  style_select->family     = family;
	  style_select->style_name = style_name;

	  gtk_idle_add ((GtkFunction)fontsel_set_style_by_name, style_select);
	}
    }
}

FontFace *
fontsel_get_font_by_name (gchar *family_name,
			  gchar *style_name)
{
  GTree *family = NULL;
  
  g_return_val_if_fail (families != NULL, NULL);
  g_return_val_if_fail (family_name != NULL && style_name != NULL, NULL);
  
  family = g_tree_lookup (families, family_name);
  if (!family)
    return NULL;

  return g_tree_lookup (family, style_name);
}


/* private functions */

static void
fontsel_family_select_callback (GtkCList       *clist,
				gint            row,
				gint            column,
				GdkEventButton *event,
				gpointer        data)
{
  FontFace  *font_face = NULL;
  GtkWidget *clist2    = (GtkWidget *) data;
  GTree     *family;
  gint       face_row  = 0;

  family = gtk_clist_get_row_data (clist, row);

  gtk_clist_freeze (GTK_CLIST (clist2));
  gtk_clist_clear (GTK_CLIST (clist2));
  g_tree_traverse (family, 
		   (GTraverseFunc) fontsel_style_list_insert, 
		   G_IN_ORDER, 
		   clist2);
  gtk_clist_thaw (GTK_CLIST (clist2));

  if (face_selected && face_selected->style_name)
    font_face = g_tree_lookup (family, face_selected->style_name);

  if (! font_face)
    font_face = g_tree_lookup (family, "Regular");

  if (! font_face)
    font_face = g_tree_lookup (family, "Roman");

  if (font_face)
    face_row = gtk_clist_find_row_from_data (GTK_CLIST (clist2), font_face);

  gtk_clist_select_row (GTK_CLIST (clist2), face_row, 0);

  if (! (gtk_clist_row_is_visible (GTK_CLIST (clist2), face_row) &
	 GTK_VISIBILITY_FULL))
    gtk_clist_moveto (clist, face_row, 0, 0.0, 0.0);
}

static void
fontsel_style_select_callback (GtkCList       *clist,
			       gint            row,
			       gint            column,
			       GdkEventButton *event,
			       gpointer        data)
{
  face_selected = gtk_clist_get_row_data (clist, row);

  gtk_signal_emit_by_name (GTK_OBJECT (data), "face_changed");
}

static gint
fontsel_family_list_insert (gpointer key,
			    gpointer value,
			    gpointer data)
{
  gint       row;
  gchar     *text[1];
  GtkWidget *clist = (GtkWidget *)data;

  text[0] = (gchar *)key;
  row = gtk_clist_append (GTK_CLIST (clist), text);
  gtk_clist_set_row_data (GTK_CLIST (clist), row, value);

  return FALSE;
}

static gint
fontsel_style_list_insert (gpointer key,
			   gpointer value,
			   gpointer data)
{
  gint       row;
  gchar     *text[1];
  GtkWidget *clist = (GtkWidget *)data;

  text[0] = (gchar *)key;
  row = gtk_clist_append (GTK_CLIST (clist), text);
  gtk_clist_set_row_data (GTK_CLIST (clist), row, value);

  return FALSE;  
}

static void
fontsel_insert_face (const FT_Face  face,
		     const gchar   *file_name,
		     const gint     face_index)
{
  GTree    *family;
  FontFace *font;
  gchar    *name;

  family = g_tree_lookup (families, face->family_name);
  if (!family)
    {
      family = g_tree_new ((GCompareFunc)strcmp);
      name   = g_strdup (face->family_name);
      g_tree_insert (families, name, family);
    }

  font = g_new (FontFace, 1);
  font->file_name   = g_strdup (file_name);
  font->face_index  = face_index;
  font->family_name = g_strdup (face->family_name);

  if (face->style_name)
    name = g_strdup (face->style_name);
  else
    name = g_strdup ("(unknown)");

  font->style_name  = name;

  g_tree_insert (family, name, font);
}

static gint
fontsel_remove_family (gpointer key,
		       gpointer value,
		       gpointer data)
{
  GTree *family = (GTree *)value;

  g_free (key);
  g_tree_traverse (family, 
		   (GTraverseFunc)fontsel_remove_face, 
		   G_IN_ORDER, 
		   NULL);
  g_tree_destroy (family);

  return FALSE;
}

static gint
fontsel_remove_face (gpointer key,
		     gpointer value,
		     gpointer data)
{
  FontFace *font = (FontFace *) value;

  g_free (key);
  g_free (font->file_name);
  g_free (font->family_name);
  g_free (font);

  return FALSE;
}

static gboolean
fontsel_is_font_file (const gchar *name)
{
  struct stat filestat;
#ifndef G_OS_WIN32
  uid_t euid;
  gid_t egid;
#endif
  gint  err;
  gint  len;

#ifndef G_OS_WIN32
  euid = geteuid ();
  egid = getegid ();
#endif

  err = stat (name, &filestat);

  if (!err && S_ISREG (filestat.st_mode)
#ifndef G_OS_WIN32
					 &&
      
      ((filestat.st_mode & S_IRUSR) ||
       
       ((filestat.st_mode & S_IRGRP) &&
	(euid != filestat.st_uid)) ||
       
       ((filestat.st_mode & S_IROTH) &&
	(euid != filestat.st_uid) &&
	(egid != filestat.st_gid)))
#endif
				   )
    {
      len = strlen (name);
      if (len > 4 &&
	  (g_strncasecmp (&name[len-4], ".pfa", 4) == 0 ||
	   g_strncasecmp (&name[len-4], ".pfb", 4) == 0 ||
	   g_strncasecmp (&name[len-4], ".ttf", 4) == 0))
	{
	  return TRUE;
	}
    }
  
  return FALSE;
}

static void
fontsel_scan_directory (const gchar *path)
{
  DIR *dir;
  struct dirent *entry;
  gchar *fullname;
  FT_Face  face;
  FT_Error error;
  gint i;

  dir = opendir (path);
  if (dir)
    {
      while ((entry = readdir (dir)) != NULL)
	{
	  fullname = g_strconcat (path, entry->d_name, NULL);
	  if (fontsel_is_font_file (fullname))
	    {
	      error = FT_New_Face (library, fullname, 0, &face);
	      if (!error)
		{
		  if (face->face_flags & FT_FACE_FLAG_SCALABLE)
		    fontsel_insert_face (face, fullname, 0);
		  
		  for (i = 1; i < face->num_faces; i++)
		    {
		      FT_Done_Face (face);
		      error = FT_New_Face (library, fullname, i, &face);
		      if (!error)
			{
			  if (face->face_flags & FT_FACE_FLAG_SCALABLE)
			    fontsel_insert_face (face, fullname, i);
			}
		    }

		  FT_Done_Face (face);
		}
	    }
	  g_free (fullname);

	  while (gtk_events_pending ())
	    gtk_main_iteration ();
	}
      closedir (dir);
    }
}

static void
fontsel_directories_ok_callback (GtkWidget *widget, 
				 gpointer   data)
{
  gchar     **path;
  GtkObject  *path_editor;
  
  path        = gtk_object_get_data (GTK_OBJECT (data), "path");
  path_editor = gtk_object_get_data (GTK_OBJECT (data), "path_editor");

  *path = gimp_path_editor_get_path (GIMP_PATH_EDITOR (path_editor));

  gtk_widget_destroy (GTK_WIDGET (data));
}

static gboolean
fontsel_directories_dialog (gchar  *message,
			    gchar **path)
			    
{
  GtkWidget *dialog;
  GtkWidget *path_editor;
  GtkWidget *label;    
  gchar     *new_path = NULL;
  
  dialog = gimp_dialog_new (_("Configure Font Paths"), 
			    "freetype_fontpath_select",
			    NULL, NULL,
			    GTK_WIN_POS_MOUSE,
			    FALSE, TRUE, FALSE, 
			    
			    _("OK"), fontsel_directories_ok_callback, 
			    NULL, NULL, NULL, TRUE, FALSE,
			    _("Cancel"), gtk_widget_destroy, 
			    NULL, 1, NULL, FALSE, TRUE,

			    NULL);
			    
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      GTK_SIGNAL_FUNC (gtk_main_quit),
		      NULL);

  if (message)
    {
      label = gtk_label_new (message);
      gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
      gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
      gtk_misc_set_padding (GTK_MISC (label), 6, 6);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), 
			  label, FALSE, FALSE, 0);
    }

  path_editor = gimp_path_editor_new (_("Choose a Font Directory"), *path);
  gtk_container_set_border_width (GTK_CONTAINER (path_editor), 6);
  gtk_widget_set_usize (GTK_WIDGET (path_editor), -1, 240);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), 
		      path_editor, TRUE, TRUE, 0);

  label = gtk_label_new (_("You may specify multiple font paths here."));
  gtk_misc_set_padding (GTK_MISC (label), 6, 6);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), 
		      label, FALSE, FALSE, 0);

  gtk_object_set_data (GTK_OBJECT (dialog), "path", &new_path);
  gtk_object_set_data (GTK_OBJECT (dialog), "path_editor", path_editor);

  gtk_widget_show_all (dialog);

  gtk_main ();

  if (new_path &&
      (strcmp (*path, new_path) != 0 ||
       strcmp (*path, DEFAULT_FONTPATH) == 0))
    {
      g_free (*path);
      *path = new_path;
      gimp_gimprc_set (FONTPATH_TOKEN, *path);

      return TRUE;
    }
  else
    return FALSE;
}


/*  this is called from an idle loop  */

static gint
fontsel_set_style_by_name (StyleSelect *select)
{
  FontFace *font_face = NULL;  
  GtkCList *clist;
  gint      row;

  g_return_val_if_fail (select != NULL, FALSE);

  font_face = g_tree_lookup (select->family, select->style_name);

  if (font_face)
    {
      clist = gtk_object_get_data (GTK_OBJECT (select->fontsel), "style_list");
      row = gtk_clist_find_row_from_data (GTK_CLIST (clist), font_face);
      gtk_clist_select_row (GTK_CLIST (clist), row, 0);

      if ( !(gtk_clist_row_is_visible (clist, row) & GTK_VISIBILITY_FULL))
	gtk_clist_moveto (clist, row, 0, 0.0, 0.0);
    }

  g_free (select);

  return FALSE;
}
