/* GKrellM
|  Copyright (C) 1999-2003 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"
#include "gkrellm-private.h"
#include "gkrellm-sysdeps.h"


/* ----------------------------------------------------------------------*/
/* Hostname label on the top frame event_box. */

static GkrellmMonitor		*mon_host;

static GkrellmPanel		*host;
static GkrellmDecal		*decal_host;
static GkrellmDecal		*decal_sysname;
static GkrellmAlert		*server_alert;

static GtkWidget	*host_vbox;

static gint			style_id;

static gboolean		hostname_visible,
					hostname_short,
					system_name_visible;

static gchar		*hostname,
					*system_name;

static gint
host_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	if (widget == host->drawing_area)
		{
		gdk_draw_drawable(widget->window, gkrellm_draw_GC(1), host->pixmap,
			ev->area.x, ev->area.y, ev->area.x, ev->area.y,
			ev->area.width, ev->area.height);
		}
	return FALSE;
	}


static void
draw_hostname(void)
	{
	GkrellmTextstyle	*ts;
	gchar		*s, buf[128];
	gint		w;

	if (decal_host)
		{
		ts = gkrellm_meter_textstyle(style_id);
		strncpy(buf, hostname, sizeof(buf) - 1);
		buf[sizeof(buf) - 1] = '\0';
		if (_GK.hostname_short)		/* XXX */
			{
			s = strchr(buf, (int) '.');
			if (s)
				*s = '\0';
			}
		w = gkrellm_gdk_string_width(ts->font, buf);
		if (w > decal_host->w)
			{
			ts = gkrellm_meter_alt_textstyle(style_id);
			w = gkrellm_gdk_string_width(ts->font, buf);
			}
		decal_host->x_off = (decal_host->w - w) / 2;
		if (decal_host->x_off < 0)
			decal_host->x_off = 0;
		decal_host->text_style.font = ts->font;
		gkrellm_draw_decal_text(host, decal_host, buf, -1);
		}
	if (decal_sysname)
		{
		ts = gkrellm_meter_textstyle(style_id);
		w = gkrellm_gdk_string_width(ts->font, system_name);
		if (w > decal_sysname->w)
			{
			ts = gkrellm_meter_alt_textstyle(style_id);
			w = gkrellm_gdk_string_width(ts->font, system_name);
			}
		decal_sysname->x_off = (decal_sysname->w - w) / 2;
		if (decal_sysname->x_off < 0)
			decal_sysname->x_off = 0;
		decal_sysname->text_style.font = ts->font;
		gkrellm_draw_decal_text(host, decal_sysname, system_name, -1);
		}
	if (decal_host || decal_sysname)
		gkrellm_draw_panel_layers(host);
	}

void
update_host(void)
	{
	gint		connect_state;
	static gint	reconnect_timeout;

	if (!_GK.client_mode || !GK.second_tick)
		return;

	/* If we loose the server connection, trigger a hardwired alarm.
	*/
	connect_state = gkrellm_client_server_connect_state();
	if (connect_state == 2)		/* thread is trying a reconnect */
		{
		if (_GK.client_server_reconnect_timeout <= 0)
			gkrellm_check_alert(server_alert, 1.0);		/* Warning	*/
		}
	else if (connect_state == 0)	/* Lost connection			*/
		{
		gkrellm_check_alert(server_alert, 2.0);		/* Alarm 	*/
		if (   _GK.client_server_reconnect_timeout > 0
			&& ++reconnect_timeout > _GK.client_server_reconnect_timeout
		   )
			{
			gkrellm_client_mode_connect_thread();
			reconnect_timeout = 0;
			}
		}
	else
		{
		gkrellm_check_alert(server_alert, 0.0);		/* Alert off	*/
		reconnect_timeout = 0;
		if (   _GK.client_server_io_timeout > 0
			&& _GK.time_now >
					_GK.client_server_read_time + _GK.client_server_io_timeout
		   )
			gkrellm_client_mode_disconnect();
		}
	}

static void
cb_command_process(GkrellmAlert *alert, gchar *src, gchar *buf, gint size,
			gpointer data)
	{
	gchar	*s, c;
	gint	len;

	if (!buf || size < 1)
		return;
	--size;
	*buf = '\0';
	if (!src)
		return;
	for (s = src; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			if ((c = *(s + 1)) == 'H')
				len = snprintf(buf, size, "%s", gkrellm_sys_get_host_name());
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';
	}

static void
cb_alert_trigger(GkrellmAlert *alert, gpointer data)
	{
	alert->panel = host;
	}

static void
create_server_alert(void)
	{
	if (!server_alert)
		{
		/* This is a hardwired alert, so zero config values.
		|  Use the high alert as a binary alert by setting the high warn
		|  and alarm limits to 0.5 and 1.5.  A 1.0 check above will
		|  trigger a warning and a 2.0 check will trigger an alarm.
		|  A 0.0 check will reset it.
		*/
		server_alert = gkrellm_alert_create(host, NULL,
					_("gkrellmd server disconnect"),
					TRUE,	/* check high */
					FALSE, 	/* no check low */
					TRUE, 0.0, 0.0, 0.0, 0.0, 0);
		}
	gkrellm_alert_set_triggers(server_alert, 1.5, 0.5, 0.0, 0.0);
	gkrellm_alert_commands_config(server_alert, TRUE, FALSE);
	gkrellm_alert_trigger_connect(server_alert, cb_alert_trigger, NULL);
	gkrellm_alert_command_process_connect(server_alert,
				cb_command_process, NULL);
	}

static void
create_hostname(GtkWidget *vbox, gint first_create)
	{
	GkrellmStyle			*style;
	gint			yy, y, l, r, w, a, d;

	if (first_create)
		host = gkrellm_panel_new0();
	host_vbox = vbox;

	if (_GK.client_mode)
		{
		hostname = _GK.server_hostname;
		system_name = _GK.server_sysname;
		create_server_alert();
		}
	else
		{
		hostname = gkrellm_sys_get_host_name();
		system_name = gkrellm_sys_get_system_name();
		}
	style = gkrellm_meter_style(style_id);

	decal_host = decal_sysname = NULL;
	y = yy = 0;
	if (_GK.enable_hostname)
		{
		decal_host = gkrellm_create_decal_text(host, "8Hgk",
					gkrellm_meter_textstyle(style_id), style, -1, -1, -1);
		y = decal_host->h + 1;
		}
	if (_GK.enable_system_name && system_name)
		{
		decal_sysname = gkrellm_create_decal_text(host, system_name,
					gkrellm_meter_textstyle(style_id), style, -1, -1, -1);
		decal_sysname->y += y;
		gkrellm_winop_text_extents(decal_sysname->text_style.font, system_name,
					strlen(system_name), &l, &r, &w, &a, &d);
		yy = (d > 0) ? 0 : 1;
		}

	gkrellm_panel_configure(host, NULL, style);
	if (yy > 0)
		gkrellm_panel_configure_add_height(host, yy);

	gkrellm_panel_create(vbox, mon_host, host);

	if (first_create)
		g_signal_connect(G_OBJECT(host->drawing_area), "expose_event",
				G_CALLBACK(host_expose_event), NULL);

	hostname_visible = _GK.enable_hostname;
	hostname_short = _GK.hostname_short;
	system_name_visible = _GK.enable_system_name;
	draw_hostname();
	if (!decal_host && !decal_sysname && !_GK.decorated)
		{
		gkrellm_panel_hide(host);
		gkrellm_spacers_hide(mon_host);
		}
	else
		gkrellm_spacers_show(mon_host);
	}

/* ================================================================== */
  /* Config is done in gui.c General
  */
#define	HOST_CONFIG_KEYWORD		"hostname"

void
gkrellm_apply_hostname_config(void)
	{
	if (   hostname_visible != _GK.enable_hostname
		|| system_name_visible != _GK.enable_system_name
	   )
		{
		hostname_visible = _GK.enable_hostname;
		system_name_visible = _GK.enable_system_name;
		gkrellm_panel_destroy(host);
		gkrellm_alert_destroy(&server_alert);
		create_hostname(host_vbox, TRUE);
		}
	else if (_GK.hostname_short != hostname_short)
		{
		hostname_short = _GK.hostname_short;
		draw_hostname();
		}
	}

void
cb_gkrellmd_disconnect_alert(GtkWidget *button, gpointer data)
	{
	gkrellm_alert_config_window(&server_alert);
	}

static void
save_host_config(FILE *f)
	{
	if (_GK.client_mode && server_alert)
		gkrellm_save_alertconfig(f, server_alert, HOST_CONFIG_KEYWORD, NULL);
	}

static void
load_host_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (!strcmp(config, GKRELLM_ALERTCONFIG_KEYWORD))
			{
			if (!server_alert)
				create_server_alert();
			gkrellm_load_alertconfig(&server_alert, item);
			}
		}
	}

static GkrellmMonitor	monitor_host =
	{
	NULL,		/* Name, for config tab. Done in General config */
	MON_HOST,	/* Id,  0 if a plugin		*/
	create_hostname,
	update_host, /* The update function		*/
	NULL,		/* The config tab create function	*/
	NULL,		/* Instant apply */

	save_host_config,	/* Save user conifg			*/
	load_host_config,	/* Load user config			*/
	HOST_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

GkrellmMonitor *
init_host_monitor(void)
	{
	style_id = gkrellm_add_meter_style(&monitor_host, HOST_STYLE_NAME);
	mon_host = &monitor_host;
	return &monitor_host;
	}

GkrellmMonitor *
gkrellm_mon_host(void)
	{
	return &monitor_host;
	}
