/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellmd.h"

GList	*gkrellmd_monitor_list;

static GList	*serveflag_done_list;

static struct tm	gkrellmd_current_tm;

gint
gkrellm_get_timer_ticks(void)
	{
	return GK.timer_ticks;
	}

gboolean
check_client_version(GkrellmdClient *client, gint major, gint minor, gint rev)
	{
	if (   client->major_version > major
		|| (client->major_version == major && client->minor_version > minor)
	    || (   client->major_version == major && client->minor_version == minor
			&& client->rev_version >= rev
		   )
	   )
		return TRUE;
	return FALSE;
	}


/* ======================================================= */
typedef struct
	{
	gint	instance;
	gulong	user,
			nice,
			sys,
			idle;
	}
	CpuData;

static gchar	*n_cpus_setup;
static gboolean	nice_time_unsupported;

static GList	*cpu_list;
static GList	*instance_list;

void
gkrellm_cpu_set_number_of_cpus(gint n)
	{
	CpuData	*cpu;
	GList	*list;
	gint	i;

	n_cpus_setup = g_strdup_printf("n_cpus %d\n", n);
	for (i = 0; i < n; ++i)
		{
		cpu = g_new0(CpuData, 1);
		cpu_list = g_list_append(cpu_list, cpu);

		if (instance_list && (list = g_list_nth(instance_list, i)) != NULL)
			cpu->instance = GPOINTER_TO_INT(list->data);
		else
			cpu->instance = i;
		}
	}

void
gkrellm_cpu_add_instance(gint instance)
	{
	instance_list = g_list_append(instance_list, GINT_TO_POINTER(instance));
	}

void
gkrellm_cpu_nice_time_unsupported(void)
	{
	nice_time_unsupported = TRUE;
	}

void
gkrellm_cpu_assign_composite_data(gulong user, gulong nice,
			gulong sys, gulong idle)
	{
	return;		/* let client gkrellm compute it */
	}

void
gkrellm_cpu_assign_data(gint n, gulong user, gulong nice,
			gulong sys, gulong idle)
	{
	CpuData	*cpu = NULL;
	GList	*list;

	for (list = cpu_list; list; list = list->next)
		{
		cpu = (CpuData *) list->data;
		if (cpu->instance == n)
			break;
		}
	if (list)
		{
		cpu->user = user;
		cpu->nice = nice;
		cpu->sys = sys;
		cpu->idle = idle;
		}
	}

static void
update_cpu(gboolean force)
	{
	gkrellm_sys_cpu_read_data();
	}

static gchar *
serve_cpu_data(GkrellmdClient *client)
	{
	CpuData	*cpu;
	GList	*list;
	gchar	*s, *d, buf[128];

	s = g_strdup("<cpu>\n");
	for (list = cpu_list; list; list = list->next)
		{
		cpu = (CpuData *) list->data;
		sprintf(buf, "%d %lu %lu %lu %lu\n", cpu->instance,
				cpu->user, cpu->nice, cpu->sys, cpu->idle);
		d = g_strconcat(s, buf, NULL);
		g_free(s);
		s = d;
		}
	return s;
	}

static void
serve_cpu_setup(GkrellmdClient *client)
	{
	GList	*list;
	gchar	buf[64];

	send_to_client(client->fd, "<cpu_setup>\n");
	for (list = instance_list; list; list = list->next)
		{
		snprintf(buf, sizeof(buf), "cpu_instance %d\n",
				GPOINTER_TO_INT(list->data));
		send_to_client(client->fd, buf);
		}
	send_to_client(client->fd, n_cpus_setup);
	if (nice_time_unsupported)
		send_to_client(client->fd, "nice_time_unsupported\n");
	}

static GkrellmdMonitor cpu_monitor =
	{
	"cpu",
	update_cpu,
	serve_cpu_data,
	serve_cpu_setup
	};


static GkrellmdMonitor *
init_cpu_monitor(void)
	{
	if (gkrellm_sys_cpu_init())
		return &cpu_monitor;
	return NULL;
	}

/* ======================================================= */
struct
	{
	gboolean	changed;
	gint		n_processes,
				n_running,
				n_users;
	gulong		n_forks;
	gfloat		fload;
	}
	proc;

void
gkrellm_proc_assign_data(gint n_processes, gint n_running,
			gulong n_forks, gfloat load)
	{
	if (   proc.n_processes != n_processes
		|| proc.n_running != n_running
		|| proc.n_forks != n_forks
		|| proc.fload != load
	   )
		{
		proc.n_processes = n_processes;
		proc.n_running = n_running;
		proc.n_forks = n_forks;
		proc.fload = load;
		proc.changed = TRUE;
		}
	}

void
gkrellm_proc_assign_users(gint n_users)
	{
	if (proc.n_users != n_users)
		{
		proc.n_users = n_users;
		proc.changed = TRUE;
		}
	}

static void
update_proc(gboolean force)
	{
	gkrellm_sys_proc_read_data();
	if (force || GK.five_second_tick)
		gkrellm_sys_proc_read_users();
	}

static gchar *
serve_proc_data(GkrellmdClient *client)
	{
	gchar	*s = NULL;

	if (proc.changed || !client->served)
		{
		s = g_strdup_printf("<proc>\n%d %d %lu %.2f %d\n",
				proc.n_processes, proc.n_running,
				proc.n_forks, proc.fload, proc.n_users);
		}
	return s;
	}

static GkrellmdMonitor proc_monitor =
	{
	"proc",
	update_proc,
	serve_proc_data,
	NULL
	};


static GkrellmdMonitor *
init_proc_monitor(void)
	{
	if (!gkrellm_sys_proc_init())
		return NULL;
	serveflag_done_list = g_list_append(serveflag_done_list, &proc.changed);
	return &proc_monitor;
	}

/* ======================================================= */
typedef struct
	{
	gchar		*name;
	gchar		*subdisk_parent;
	gint		order,
				subdisk,
				changed;
	gint		device_number,
				unit_number;
	guint64		rb,
				wb;
	}
	DiskData;

static GList	*disk_list;
static gint		n_disks;
static gboolean	units_are_blocks;


static DiskData *
add_disk(gchar *name, gint order, gint device_number, gint unit_number)
	{
	DiskData	*disk;
	GList		*list;
	gint		i;

	disk = g_new0(DiskData, 1);
	disk->name = g_strdup(name);
	disk->order = order;
	disk->subdisk = -1;
	disk->device_number = device_number;
	disk->unit_number = unit_number;
	if (order >= 0)
		{
		for (i = 0, list = disk_list; list; list = list->next, ++i)
			if (disk->order < ((DiskData *) list->data)->order)
				break;
		disk_list = g_list_insert(disk_list, disk, i);
		}
	else
		disk_list = g_list_append(disk_list, disk);
	++n_disks;
	return disk;
	}

static DiskData *
add_subdisk(gchar *subdisk_name, gchar *disk_name, gint subdisk)
	{
	DiskData	*sdisk = NULL;
#if GLIB_CHECK_VERSION(2,0,0)
	DiskData	*disk;
	GList		*list = NULL;

	for (list = disk_list; list; list = list->next)
		{
		disk = (DiskData * ) list->data;
		if (!strcmp(disk_name, disk->name))
			break;
		}
	if (!list)
		return NULL;
	sdisk = g_new0(DiskData, 1);
	sdisk->name = g_strdup(subdisk_name);
	sdisk->subdisk_parent = g_strdup(disk_name);
	sdisk->order = disk->order;
	sdisk->subdisk = subdisk;

	for (list = list->next; list; list = list->next)
		{
		disk = (DiskData * ) list->data;
		if (disk->subdisk == -1 || disk->subdisk > subdisk)
			break;
		}
	disk_list = g_list_insert_before(disk_list, list, sdisk);
	++n_disks;
#endif
	return sdisk;
	}

static void
disk_assign_data(DiskData *disk, guint64 rb, guint64 wb)
	{
	if (disk)
		{
		if (disk->rb != rb || disk->wb != wb)
			disk->changed = TRUE;
		else
			disk->changed = FALSE;
		disk->rb = rb;
		disk->wb = wb;
		}
	}

void
gkrellm_disk_reset_composite(void)
	{
	/* Don't handle this. */
	}

void
gkrellm_disk_units_are_blocks(void)
	{
	units_are_blocks = TRUE;
	}

void
gkrellm_disk_assign_data_by_device(gint device_number, gint unit_number,
			guint64 rb, guint64 wb)
	{
	GList		*list;
	DiskData	*disk = NULL;
	gchar		*name;
	gint		order = -1;

	for (list = disk_list; list; list = list->next)
		{
		disk = (DiskData * ) list->data;
		if (   disk->device_number == device_number
			&& disk->unit_number == unit_number
		   )
			break;
		disk = NULL;
		}
	if (!disk)
		{
		name = gkrellm_sys_disk_name_from_device(device_number,
					unit_number, &order);
		if (name)
			disk = add_disk(name, order, device_number, unit_number);
		}
	disk_assign_data(disk, rb, wb);
	}

void
gkrellm_disk_assign_data_nth(gint n, guint64 rb, guint64 wb)
	{
	DiskData	*disk;
	gchar		name[32];

	if (n < n_disks)
		disk = (DiskData *) g_list_nth_data(disk_list, n);
	else
		{
		sprintf(name, "%s%c", _("Disk"), 'A' + n);
		disk = add_disk(name, n, 0, 0);
		}
	disk_assign_data(disk, rb, wb);
	}

void
gkrellm_disk_assign_data_by_name(gchar *name, guint64 rb, guint64 wb)
	{
	GList		*list;
	DiskData	*disk = NULL;
	gint		order = -1;

	for (list = disk_list; list; list = list->next)
		{
		disk = (DiskData * ) list->data;
		if (!strcmp(name, disk->name))
			break;
		disk = NULL;
		}
	if (!disk)
		{
		order = gkrellm_sys_disk_order_from_name(name);
		disk = add_disk(name, order, 0, 0);
		}
	disk_assign_data(disk, rb, wb);
	}

void
gkrellm_disk_subdisk_assign_data_by_name(gchar *subdisk_name, gchar *disk_name,
				guint64 rb, guint64 wb)
	{
	GList		*list;
	DiskData	*disk = NULL;
	gchar		*s, *endptr;
	gint		subdisk;

	if (!subdisk_name || !disk_name)
		return;
	for (list = disk_list; list; list = list->next)
		{
		disk = (DiskData * ) list->data;
		if (!strcmp(subdisk_name, disk->name))
			break;
		disk = NULL;
		}
	if (!disk)
		{
		/* A subdisk name is expected to be the disk_name with a number string
		|  appended.  Eg. "hda1" is a subdisk_name of disk_name "hda"
		*/
		s = subdisk_name + strlen(disk_name);
		subdisk = strtol(s, &endptr, 0);
		if (!*s || *endptr)
			return;
		disk = add_subdisk(subdisk_name, disk_name, subdisk);
		}
	disk_assign_data(disk, rb, wb);
	}

static void
update_disk(gboolean force)
	{
	gkrellm_sys_disk_read_data();
	}


static gchar *
serve_disk_data(GkrellmdClient *client)
	{
	DiskData	*disk;
	GList		*list;
	gchar		buf[128], *d, *s, *fmt1, *fmt2;

#if defined(__solaris__)
    fmt1 = "%s %lld %lld\n";
    fmt2 = "%s %s %lld %lld\n";
#else
    fmt1 = "%s %Lu %Lu\n";
    fmt2 = "%s %s %Lu %Lu\n";
#endif
	s = g_strdup("");
	for (list = disk_list; list; list = list->next)
		{
		disk = (DiskData *) list->data;
		if (!disk->changed && client->served)
			continue;
		if (!disk->subdisk_parent)
			sprintf(buf, fmt1, disk->name, disk->rb, disk->wb);
		else if (client->feature_subdisk)
			sprintf(buf, fmt2, disk->name, disk->subdisk_parent,
						disk->rb, disk->wb);
		else
			continue;
		d = g_strconcat(s, buf, NULL);
		g_free(s);
		s = d;
		}
	if (*s)
		{
		d = g_strconcat("<disk>\n", s, NULL);
		g_free(s);
		s = d;
		}
	return s;
	}

static void
serve_disk_setup(GkrellmdClient *client)
	{
	if (units_are_blocks)
		send_to_client(client->fd, "<disk_setup>\nunits_are_blocks\n");
	if (check_client_version(client, 2,1,3))
		client->feature_subdisk = TRUE;
	}

static GkrellmdMonitor disk_monitor =
	{
	"disk",
	update_disk,
	serve_disk_data,
	serve_disk_setup
	};


static GkrellmdMonitor *
init_disk_monitor(void)
	{
	if (gkrellm_sys_disk_init())
		return &disk_monitor;
	return NULL;
	}

/* ======================================================= */
#include "../src/inet.h"

typedef struct
	{
	ActiveTCP	tcp;
	gboolean	alive,
				new_connection;
	}
	InetData;

static GList	*inet_list,
				*inet_dead_list;

static gboolean	inet_unsupported,
				inet_new;

void
gkrellm_inet_log_tcp_port_data(gpointer data)
	{
	GList		*list;
	InetData	*in;
	ActiveTCP	*tcp, *active_tcp = NULL;
	gchar		*ap, *aap;
	gint		slen;

	tcp = (ActiveTCP *) data;
	for (list = inet_list; list; list = list->next)
		{
		in = (InetData *) list->data;
		active_tcp = &in->tcp;
		if (tcp->family == AF_INET)
			{
			ap = (char *)&tcp->remote_addr;
			aap = (char *)&active_tcp->remote_addr;
			slen = sizeof(struct in_addr);
			}
#if defined(INET6)
		else if (tcp->family == AF_INET6)
			{
			ap = (char *)&tcp->remote_addr6;
			aap = (char *)&active_tcp->remote_addr6;
			slen = sizeof(struct in6_addr);
			}
#endif
		else
			return;
		if (   memcmp(aap, ap, slen) == 0
			&& active_tcp->remote_port == tcp->remote_port
			&& active_tcp->local_port == tcp->local_port
		   )
			{
			in->alive = TRUE;	/* Old alive connection still alive */
			return;
			}
		}
	inet_new = TRUE;
	in = g_new0(InetData, 1);
	in->tcp = *tcp;
	in->alive = TRUE;
	in->new_connection = TRUE;
	inet_list = g_list_append(inet_list, in);
	}

void
free_glist_and_data(GList **list_head)
	{
	GList	*list;

	if (*list_head == NULL)
		return;
	for (list = *list_head; list; list = list->next)
		if (list->data)
			g_free(list->data);
	g_list_free(*list_head);
	*list_head = NULL;
	}

static void
update_inet(gboolean force)
	{
	GList		*list;
	InetData	*in;

	if (!force && !GK.second_tick)
		return;

	free_glist_and_data(&inet_dead_list);
	inet_new = FALSE;

	for (list = inet_list; list; list = list->next)
		{
		in = (InetData *) list->data;
		in->alive = FALSE;
		in->new_connection = FALSE;
		}

	gkrellm_sys_inet_read_tcp_data();

	for (list = inet_list; list; )
		{
		in = (InetData *) list->data;
		if (!in->alive)
			{
			if (list == inet_list)
				inet_list = inet_list->next;
			list = g_list_remove(list, in);
			inet_dead_list = g_list_append(inet_dead_list, in);
			}
		else
			list = list->next;
		}
	}

static gchar *
serve_inet_data(GkrellmdClient *client)
	{
	InetData	*in;
	ActiveTCP	*tcp;
	GList		*list;
	gchar		buf[128], *d, *s, *cp;
#if defined(INET6) && defined(HAVE_GETADDRINFO)
	struct sockaddr_in6	sin6;
	char		addrbuf[NI_MAXHOST];
#endif

	if (client->served && (!GK.second_tick || (!inet_new && !inet_dead_list)))
		return NULL;
	s = g_strdup("");
	if (inet_new || !client->served)
		{
		for (list = inet_list; list; list = list->next)
			{
			in = (InetData *) list->data;
			tcp = &in->tcp;
			if (   tcp->family == AF_INET
				&& (in->new_connection || !client->served)
			   )
				{
				cp = inet_ntoa(tcp->remote_addr);
				sprintf(buf, "+0 %x %s:%x\n",
							tcp->local_port, cp, tcp->remote_port);
				d = g_strconcat(s, buf, NULL);
				g_free(s);
				s = d;
				}
#if defined(INET6) && defined(HAVE_GETADDRINFO)
			else if (tcp->family == AF_INET6
				 && (in->new_connection || !client->served))
				{
				memset(&sin6, 0, sizeof(sin6));
				memcpy(&sin6.sin6_addr, &tcp->remote_addr6,
				       sizeof(struct in6_addr));
				sin6.sin6_family = AF_INET6;
#ifdef SIN6_LEN
				sin6.sin6_len = sizeof(struct sockaddr_in6);
#endif
				if (getnameinfo((struct sockaddr *)&sin6,
						sizeof(struct sockaddr_in6),
						addrbuf, sizeof(addrbuf),
						NULL, 0,
						NI_NUMERICHOST|NI_WITHSCOPEID)
				    != 0)
					continue;
				snprintf(buf, sizeof(buf), "+6 %x [%s]:%x\n",
					 tcp->local_port, addrbuf,
					 tcp->remote_port);
				d = g_strconcat(s, buf, NULL);
				g_free(s);
				s = d;
				}
#endif
			}
		}
	if (client->served)
		{
		for (list = inet_dead_list; list; list = list->next)
			{
			in = (InetData *) list->data;
			tcp = &in->tcp;
			if (tcp->family == AF_INET)
				{
				cp = inet_ntoa(tcp->remote_addr);
				sprintf(buf, "-0 %x %s:%x\n",
							tcp->local_port, cp, tcp->remote_port);
				d = g_strconcat(s, buf, NULL);
				g_free(s);
				s = d;
				}
#if defined(INET6) && defined(HAVE_GETADDRINFO)
			else if (tcp->family == AF_INET6)
				{
				memset(&sin6, 0, sizeof(sin6));
				memcpy(&sin6.sin6_addr, &tcp->remote_addr6,
				       sizeof(struct in6_addr));
				sin6.sin6_family = AF_INET6;
#ifdef SIN6_LEN
				sin6.sin6_len = sizeof(struct sockaddr_in6);
#endif
				if (getnameinfo((struct sockaddr *)&sin6,
						sizeof(struct sockaddr_in6),
						addrbuf, sizeof(addrbuf),
						NULL, 0,
						NI_NUMERICHOST|NI_WITHSCOPEID)
				    != 0)
					continue;
				snprintf(buf, sizeof(buf), "-6 %x [%s]:%x\n",
					 tcp->local_port, addrbuf,
					 tcp->remote_port);
				d = g_strconcat(s, buf, NULL);
				g_free(s);
				s = d;
				}
#endif
			}
		}
	if (*s)
		{
		d = g_strconcat("<inet>\n", s, NULL);
		g_free(s);
		s = d;
		}
	return s;
	}

static void
serve_inet_setup(GkrellmdClient *client)
	{
	if (inet_unsupported)
		send_to_client(client->fd, "<inet_setup>\ninet_unsupported\n");
	}

static GkrellmdMonitor inet_monitor =
	{
	"inet",
	update_inet,
	serve_inet_data,
	serve_inet_setup
	};


static GkrellmdMonitor *
init_inet_monitor(void)
	{
	if (gkrellm_sys_inet_init())
		return &inet_monitor;
	inet_unsupported = TRUE;
	return NULL;
	}

/* ======================================================= */
typedef struct
	{
	gchar		*name;
	gboolean	changed,
				up,
				up_prev,
				up_event,
				down_event;
	gulong		rx,
				tx;
	}
	NetData;

GList	*net_list,
		*net_sys_list;

static gboolean	net_use_routed;

gchar *
gkrellm_net_mon_first(void)
	{
	gchar	*name = NULL;

	net_sys_list = net_list;
	if (net_sys_list)
		{
		name = ((NetData *) (net_sys_list->data))->name;
		net_sys_list = net_sys_list->next;
		}
	return name;
	}

gchar *
gkrellm_net_mon_next(void)
	{
	gchar	*name = NULL;

	if (net_sys_list)
		{
		name = ((NetData *) (net_sys_list->data))->name;
		net_sys_list = net_sys_list->next;
		}
	return name;
	}

void
gkrellm_net_use_routed(gboolean real_routed /* not applicable in server */)
	{
	net_use_routed = TRUE;
	}

void
gkrellm_net_assign_data(gchar *name, gulong rx, gulong tx)
	{
	GList	*list;
	NetData	*net;

	for (list = net_list; list; list = list->next)
		{
		net = (NetData *) list->data;
		if (!strcmp(net->name, name))
			{
			if (net->rx != rx || net->tx != tx)
				net->changed = TRUE;
			else
				net->changed = FALSE;
			break;
			}
		}
	if (!list)
		{
		net = g_new0(NetData, 1);
		net->name = g_strdup(name);
		net_list = g_list_append(net_list, net);
		}
	if (GK.second_tick && !net_use_routed)
		net->up = TRUE;
	net->rx = rx;
	net->tx = tx;
	}

void
gkrellm_net_routed_event(gchar *name, gboolean routed)
	{
	GList	*list;
	NetData	*net;

	for (list = net_list; list; list = list->next)
		{
		net = (NetData *) list->data;
		if (!strcmp(net->name, name))
			break;
		}
	if (!list)
		{
		net = g_new0(NetData, 1);
		net->name = g_strdup(name);
		net_list = g_list_append(net_list, net);
		}
	if (routed)
		net->up_event = TRUE;
	else
		net->down_event = TRUE;
	net->up = routed;
	}

void
gkrellm_net_add_timer_type_ppp(gchar *name)
	{
	/* Not supported remotely */
	}

void
gkrellm_net_add_timer_type_ippp(gchar *name)
	{
	/* Not supported remotely */
	}

void
gkrellm_net_set_lock_directory(gchar *dir)
	{
	/* Not supported remotely */
	}

static void
update_net(gboolean force)
	{
	GList	*list;
	NetData	*net;

	if (GK.second_tick)
		{
		if (!net_use_routed)
			{
			for (list = net_list; list; list = list->next)
				{
				net = (NetData *) list->data;
				net->up_prev = net->up;
				net->up = FALSE;
				}
			}
		else
			gkrellm_sys_net_check_routes();
		}
	gkrellm_sys_net_read_data();
	
	if (GK.second_tick && !net_use_routed)
		{
		for (list = net_list; list; list = list->next)
			{
			net = (NetData *) list->data;
			if (net->up && !net->up_prev)
				net->up_event = TRUE;
			else if (!net->up && net->up_prev)
				net->down_event = TRUE;
			}
		}
	}

static gchar *
serve_net_data(GkrellmdClient *client)
	{
	NetData		*net;
	GList		*list;
	gchar		buf[128], *d, *s, *ss;
	gboolean	fake_up_event;

	s = g_strdup("");
	for (list = net_list; list; list = list->next)
		{
		net = (NetData *) list->data;
		if (net->changed || !client->served)
			{
			sprintf(buf, "%s %lu %lu\n", net->name, net->rx, net->tx);
			d = g_strconcat(s, buf, NULL);
			g_free(s);
			s = d;
			}
		}
	if (*s)
		{
		d = g_strconcat("<net>\n", s, NULL);
		g_free(s);
		s = d;
		}

  /* Since the server transmits changes only, use the routed interface
  |  to the client regardless if the sysdep code uses routed.
  */
	if (GK.second_tick || !client->served)
		{
		ss = g_strdup("");
		for (list = net_list; list; list = list->next)
			{
			net = (NetData *) list->data;
			fake_up_event = (!client->served && net->up);
			if (net->up_event || net->down_event || fake_up_event)
				{
				sprintf(buf, "%s %d\n", net->name,
						fake_up_event ? TRUE : net->up_event);
				d = g_strconcat(ss, buf, NULL);
				g_free(ss);
				ss = d;
				}
			if (client->last_client)
				net->up_event = net->down_event = FALSE;
			}
		if (*ss)
			{
			d = g_strconcat(s, "<net_routed>\n", ss, NULL);
			g_free(s);
			s = d;
			}
		g_free(ss);
		}
	return s;
	}

static void
serve_net_setup(GkrellmdClient *client)
	{
	/* The client <-> server link always uses routed mode, but the client
	|  needs to know if server sysdep uses routed for config purposes.
	*/
	if (net_use_routed)
		send_to_client(client->fd, "<net_setup>\nnet_use_routed\n");
	}

static GkrellmdMonitor net_monitor =
	{
	"net",
	update_net,
	serve_net_data,
	serve_net_setup
	};


static GkrellmdMonitor *
init_net_monitor(void)
	{
	if (gkrellm_sys_net_init())
		return &net_monitor;
	return NULL;
	}


/* ======================================================= */
struct
	{
	gboolean	mem_changed;
	guint64		total,
				used,
				free,
				shared,
				buffers,
				cached;

	gboolean	swap_changed;
	guint64		swap_total,
				swap_used;
	gulong		swap_in,
				swap_out;
	}
	mem;

void
gkrellm_mem_assign_data(guint64 total, guint64 used, guint64 free,
			guint64 shared, guint64 buffers, guint64 cached)
	{
	if (   mem.total != total
		|| mem.used != used
		|| mem.free != free
		|| mem.shared != shared
		|| mem.buffers != buffers
		|| mem.cached != cached
	   )
		{
		mem.total = total;
		mem.used = used;
		mem.free = free;
		mem.shared = shared;
		mem.buffers = buffers;
		mem.cached = cached;
		mem.mem_changed = TRUE;
		}
	}

void
gkrellm_swap_assign_data(guint64 total, guint64 used,
			gulong swap_in, gulong swap_out)
	{
	if (   mem.swap_total != total
		|| mem.swap_used != used
		|| mem.swap_in != swap_in
		|| mem.swap_out != swap_out
	   )
		{
		mem.swap_total = total;
		mem.swap_used = used;
		mem.swap_in = swap_in;
		mem.swap_out = swap_out;
		mem.swap_changed = TRUE;
		}
	}

static void
update_mem(gboolean force)
	{
	gkrellm_sys_swap_read_data();
	if (force || GK.five_second_tick)
		gkrellm_sys_mem_read_data();
	}

static gchar *
serve_mem_data(GkrellmdClient *client)
	{
	gchar		buf[128], *s, *d, *fmt1, *fmt2;

#if defined(__solaris__)
    fmt1 = "<mem>\n%lld %lld %lld %lld %lld %lld\n";
    fmt2 = "%lld %lld %lu %lu\n";
#else
    fmt1 = "<mem>\n%Lu %Lu %Lu %Lu %Lu %Lu\n";
    fmt2 = "%Lu %Lu %lu %lu\n";
#endif
	if (mem.mem_changed || !client->served)
		{
		s = g_strdup_printf(fmt1,
				mem.total, mem.used, mem.free,
				mem.shared, mem.buffers, mem.cached);
		}
	else
		s = g_strdup("");		

	if (mem.swap_changed || !client->served)
		{
		sprintf(buf, fmt2,
				mem.swap_total, mem.swap_used,
				mem.swap_in, mem.swap_out);
		d = g_strconcat(s, "<swap>\n", buf, NULL);
		g_free(s);
		s = d;
		}
	return s;
	}

static GkrellmdMonitor mem_monitor =
	{
	"mem",
	update_mem,
	serve_mem_data,
	NULL
	};


static GkrellmdMonitor *
init_mem_monitor(void)
	{
	if (!gkrellm_sys_mem_init())
		return NULL;
	serveflag_done_list = g_list_append(serveflag_done_list, &mem.mem_changed);
	serveflag_done_list = g_list_append(serveflag_done_list,&mem.swap_changed);
	return &mem_monitor;
	}

/* ======================================================= */
typedef struct
	{
	gboolean	busy,
				deleted,
				is_mounted,
				is_nfs,
				changed;
	gchar		*directory,
				*device,
				*type,
				*options;
	gulong		blocks,
				bavail,
				bfree,
				bsize;
	}
	Mount;

static GList	*mounts_list,
				*fstab_list;

static gboolean	nfs_check,
				fs_check,
				fs_need_serve,
				fstab_list_modified,
				mounts_list_modified,
				mounting_unsupported;

static gchar *remote_fs_types[] =
	{
	"nfs",
	"smbfs"
	};

void
gkrellm_fs_setup_eject(gchar *eject_tray, gchar *close_tray,
			void (*eject_func)(), void (*close_func)())
	{
	/* Not supported remotely */
	}

void
gkrellm_fs_add_to_mounts_list(gchar *dir, gchar *dev, gchar *type)
	{
	GList	*list;
	Mount	*m;
	gint	i;

	for (list = mounts_list; list; list = list->next)
		{
		m = (Mount *) list->data;
		if (   !strcmp(m->directory, dir)
			&& !strcmp(m->device, dev)
			&& !strcmp(m->type, type)
		   )
			break;
		}
	if (!list)
		{
		m = g_new0(Mount, 1);
		m->directory = g_strdup(dir);
		m->device = g_strdup(dev);
		m->type = g_strdup(type);
		mounts_list = g_list_append(mounts_list, m);
		mounts_list_modified = TRUE;

		for (i = 0; i < (sizeof(remote_fs_types) / sizeof(gchar *)); ++i)
			{
			if (!strcmp(m->type, remote_fs_types[i]))
				{
				m->is_nfs = TRUE;
				break;
				}
			}
		}
	m->is_mounted = TRUE;
	}

void
gkrellm_fs_add_to_fstab_list(gchar *dir, gchar *dev, gchar *type, gchar *opt)
	{
	Mount	*m;

	m = g_new0(Mount, 1);
	m->directory = g_strdup(dir);
	m->device = g_strdup(dev);
	m->type = g_strdup(type);
	fstab_list = g_list_append(fstab_list, m);
	}

void
gkrellm_fs_assign_fsusage_data(gpointer pointer,
			gulong blocks, gulong bavail, gulong bfree, gulong bsize)
	{
	Mount	*m = (Mount *) pointer;

	if (   m->blocks != blocks
		|| m->bavail != bavail
		|| m->bfree  != bfree
		|| m->bsize  != bsize
	   )
		{
		m->blocks = blocks;
		m->bavail = bavail;
		m->bfree  = bfree;
		m->bsize  = bsize;

		m->changed = TRUE;
		}
	}

void
gkrellm_fs_mounting_unsupported(void)
	{
	mounting_unsupported = TRUE;
	}

static void
refresh_mounts_list(void)
	{
	GList	*list;
	Mount	*m;

	for (list = mounts_list; list; list = list->next)
		((Mount *) list->data)->is_mounted = FALSE;

	gkrellm_sys_fs_get_mounts_list();

	for (list = mounts_list; list; )
		{
		m = (Mount *) list->data;
		if (!m->is_mounted)
			{
			if (list == mounts_list)
				mounts_list = mounts_list->next;
			list = g_list_remove_link(list, list);
			g_free(m->directory);
			g_free(m->device);
			g_free(m->type);
			if (m->busy)
				m->deleted = TRUE;
			else
				g_free(m);
			mounts_list_modified = TRUE;
			}
		else
			list = list->next;
		}
	}

static void
refresh_fstab_list(void)
	{
	Mount	*m;

	while (fstab_list)
		{
		m = (Mount *) fstab_list->data;
		g_free(m->directory);
		g_free(m->device);
		g_free(m->type);
		g_free(m);
		fstab_list = g_list_remove(fstab_list, fstab_list->data);
		}
	gkrellm_sys_fs_get_fstab_list();
	fstab_list_modified = TRUE;
	}

#if GLIB_CHECK_VERSION(2,0,0)
static gpointer
get_fsusage_thread(void *data)
	{
	Mount	*m = (Mount *) data;

	gkrellm_sys_fs_get_fsusage(m, m->directory);
	if (m->deleted)
		g_free(m);
	else
		m->busy = FALSE;
	return NULL;
	}
#endif

static void
update_fs(gboolean force)
	{
	GList		*list;
	Mount		*m;
	static gint	check_tick;

	if (GK.second_tick)
		++check_tick;
	fs_check = !(check_tick % _GK.fs_interval);
	nfs_check = !(check_tick % _GK.nfs_interval);

	if (!force && (!GK.second_tick || (!fs_check && !nfs_check)))
		return;
	refresh_mounts_list();
	for (list = mounts_list; list; list = list->next)
		{
		m = (Mount *) list->data;
		if (fs_check && !m->is_nfs)
			gkrellm_sys_fs_get_fsusage(m, m->directory);
		else if (nfs_check && m->is_nfs && !m->busy)
			{
			m->busy = TRUE;
#if GLIB_CHECK_VERSION(2,0,0)
			g_thread_create(get_fsusage_thread, m, FALSE, NULL);
#else
			gkrellm_sys_fs_get_fsusage(m, m->directory);
#endif
			}
		}
	if (gkrellm_sys_fs_fstab_modified())
		refresh_fstab_list();
	fs_need_serve = TRUE;
	}

static gchar *
serve_fs_data(GkrellmdClient *client)
	{
	Mount	*m;
	GList	*list;
	gchar	buf[128], *d, *s;

	if (!fs_need_serve && client->served)
		return NULL;

	if (mounts_list_modified || !client->served)
		{
		s = g_strconcat("<fs_mounts>\n", ".clear\n", NULL);
		for (list = mounts_list; list; list = list->next)
			{
			m = (Mount *) list->data;
			snprintf(buf, sizeof(buf), "%s %s %s %lu %lu %lu %lu\n",
					m->directory, m->device, m->type,
					m->blocks, m->bavail, m->bfree, m->bsize);
			d = g_strconcat(s, buf, NULL);
			g_free(s);
			s = d;
			m->changed = FALSE;
			}
		}
	else
		{
		s = g_strdup("");
		for (list = mounts_list; list; list = list->next)
			{
			m = (Mount *) list->data;
			if (!m->changed)
				continue;
			snprintf(buf, sizeof(buf), "%s %s %lu %lu %lu %lu\n",
					m->directory, m->device,
					m->blocks, m->bavail, m->bfree, m->bsize);
			d = g_strconcat(s, buf, NULL);
			g_free(s);
			s = d;
			m->changed = FALSE;
			}
		if (*s)
			{
			d = g_strconcat("<fs>\n", s, NULL);
			g_free(s);
			s = d;
			}
		}
	if (fstab_list_modified || !client->served)
		{
		d = g_strconcat(s, "<fs_fstab>\n", ".clear\n", NULL);
		g_free(s);
		s = d;
		for (list = fstab_list; list; list = list->next)
			{
			m = (Mount *) list->data;
			snprintf(buf, sizeof(buf), "%s %s %s\n",
					m->directory, m->device, m->type);
			d = g_strconcat(s, buf, NULL);
			g_free(s);
			s = d;
			}
		}
	return s;
	}

static void
serve_fs_setup(GkrellmdClient *client)
	{
	if (mounting_unsupported)
		send_to_client(client->fd, "<fs_setup>\nmounting_unsupported\n");
	}

static GkrellmdMonitor fs_monitor =
	{
	"fs",
	update_fs,
	serve_fs_data,
	serve_fs_setup
	};


static GkrellmdMonitor *
init_fs_monitor(void)
	{
	if (!gkrellm_sys_fs_init())
		return NULL;
	serveflag_done_list = g_list_append(serveflag_done_list, &fs_need_serve);
	serveflag_done_list =
				g_list_append(serveflag_done_list, &fstab_list_modified);
	serveflag_done_list =
				g_list_append(serveflag_done_list, &mounts_list_modified);
	return &fs_monitor;
	}

/* ======================================================= */

void
gkrellm_mail_local_unsupported(void)
	{
	}

#if GLIB_CHECK_VERSION(2,0,0)
GThread *
#else
gpointer
#endif
gkrellm_mail_get_active_thread(void)
	{
	return NULL;
	}

/* ======================================================= */
struct
	{
	gboolean	changed;
	gboolean	battery_is_available,
				ac_is_on_line,
				battery_is_charging;
	gint		battery_percentage;
	gint		battery_time_left;
	}
	bat;

void
gkrellm_battery_assign_data(gint id, gboolean available, gboolean on_line,
			gboolean charging, gint percent, gint time_left)
	{
	if (id > 0)
		return;		/* Fix to handle more than one battery later */

	if (   available != bat.battery_is_available
		|| on_line   != bat.ac_is_on_line
		|| charging  != bat.battery_is_charging
		|| percent   != bat.battery_percentage
		|| time_left != bat.battery_time_left
	   )
		{
		bat.battery_is_available = available;
		bat.ac_is_on_line = on_line;
		bat.battery_is_charging = charging;
		bat.battery_percentage = percent;
		bat.battery_time_left = time_left;
		bat.changed = TRUE;
		}
	}

gint
gkrellm_battery_full_cap_fallback()
	{
	return 5000;	/* XXX Linux ACPI bug not handled by server */
	}

static void
update_battery(gboolean force)
	{
	if (force || GK.five_second_tick)
		gkrellm_sys_battery_read_data();
	}

static gchar *
serve_battery_data(GkrellmdClient *client)
	{
	gchar		*s;

	if (!bat.changed && client->served)
		return NULL;
	s = g_strdup_printf("<battery>\n%d %d %d %d %d\n",
				bat.battery_is_available,
				bat.ac_is_on_line, bat.battery_is_charging,
				bat.battery_percentage, bat.battery_time_left);
	return s;
	}

static void
serve_battery_setup(GkrellmdClient *client)
	{
	gkrellm_sys_battery_read_data();
	if (bat.battery_is_available)
		{
		send_to_client(client->fd, "<battery_setup>\nbattery_available\n");
		}
	}

static GkrellmdMonitor battery_monitor =
	{
	"battery",
	update_battery,
	serve_battery_data,
	serve_battery_setup
	};

static GkrellmdMonitor *
init_battery_monitor(void)
	{
	if (!gkrellm_sys_battery_init())
		return NULL;
	serveflag_done_list = g_list_append(serveflag_done_list, &bat.changed);
	return &battery_monitor;
	}

/* ======================================================= */
#define SENSOR_TEMPERATURE  0
#define SENSOR_FAN          1
#define SENSOR_VOLTAGE      2

typedef struct
	{
	gboolean	changed;
	gint		type;

	gchar		*dir;			/* Optional convenience prefix for basename */
	gchar		*path;			/* Constructed from dir and basename and    */
								/*  passed to get sensor functions */

	gchar		*basename;		/* These 4 are unique sensor identification */
	gint		id;				/*   of a particular sensor type */
	gint		iodev;			/* One or any combination may be used. */
	gint		inter;

	gchar		*vref;
	gchar		*default_label;

	gfloat		factor;
	gfloat		offset;
	gfloat		raw_value;
	}
	Sensor;

static GList	*sensors_list;

static gboolean thread_busy,
				sensors_need_serve;

static gpointer
read_sensors(void *data)
	{
	GList	*list;
	Sensor	*sensor;
	gfloat	tmp;

	for (list = sensors_list; list; list = list->next)
		{
		sensor = (Sensor *) list->data;
		tmp = sensor->raw_value;
		if (sensor->type == SENSOR_TEMPERATURE)
			gkrellm_sys_sensors_get_temperature(sensor->path, sensor->id,
				sensor->iodev, sensor->inter, &sensor->raw_value);
		else if (sensor->type == SENSOR_FAN)
			gkrellm_sys_sensors_get_fan(sensor->path, sensor->id,
				sensor->iodev, sensor->inter, &sensor->raw_value);
		else if (sensor->type == SENSOR_VOLTAGE)
			gkrellm_sys_sensors_get_voltage(sensor->path, sensor->id,
				sensor->iodev, sensor->inter, &sensor->raw_value);
		if (sensor->raw_value != tmp)
			{
			sensor->changed = TRUE;
			sensors_need_serve = TRUE;
			}
		else
			sensor->changed = FALSE;
		}
	thread_busy = FALSE;
	return NULL;
	}

static void
run_sensors_thread(void)
	{
#if GLIB_CHECK_VERSION(2,0,0)
	if (thread_busy)
		return;
	thread_busy = TRUE;
	g_thread_create(read_sensors, NULL, FALSE, NULL);
#else
	read_sensors(NULL);
#endif
	}


void
gkrellm_sensors_update_volt_order_base(void)
	{
	}

  /* A sensor within a type is uniquely identified by its basename.
  |  A sysdep interface may additionally use any of the triple integer
  |  set (id, iodev, inter) for internal identification.
  |  The "dir" arg is an optional common path prefix to basename which
  |  is used to construct the sensor->path value.  As a convenience to the
  |  sysdep code, the path value instead of basename is passed to the
  |  get_temperature, get_fan, and get_voltage functions.  So the monitor
  |  code here uses path to read the sensor values, but basename is only
  |  passed to the client since that is all that is needed for identification
  |  (the client is no longer interfacing to sysdep code).
  */
void
gkrellm_sensors_add_sensor(gint type, gchar *dir, gchar *basename,
		gint id, gint iodev, gint inter,
		gfloat factor, gfloat offset, gchar *vref, gchar *default_label)
	{
	Sensor	*sensor;

	if (!basename || !*basename || type < 0 || type > 2)
		return;

	sensor = g_new0(Sensor, 1);
	sensor->basename = g_strdup(basename);

	if (dir)
		{
		sensor->dir = g_strdup(dir);
		sensor->path = g_strdup_printf("%s/%s", dir, basename);
		}
	else
		{
		sensor->dir = g_strdup("NONE");
		sensor->path = g_strdup(basename);
		}
	sensor->vref = g_strdup(vref ? vref : "NONE");
	sensor->default_label = g_strdup(default_label ? default_label : "NONE");

	sensor->factor = factor;
	sensor->offset = offset;
	sensor->type = type;
	sensor->id = id;
	sensor->iodev = iodev;
	sensor->inter = inter;
	sensors_list = g_list_append(sensors_list, sensor);
	}

static void
update_sensors(gboolean force)
	{
	if (!GK.five_second_tick && !force)
		return;
	if (force)
		read_sensors(NULL);		/* No thread on first read */
	else
		run_sensors_thread();
	}

static gchar *
serve_sensors_data(GkrellmdClient *client)
	{
	Sensor	*sr;
	GList	*list;
	gchar	*s, *d, buf[128];

	if (!sensors_need_serve && client->served)
		return NULL;
	s = g_strdup("<sensors>\n");
	for (list = sensors_list; list; list = list->next)
		{
		sr = (Sensor *) list->data;
		if (sr->changed || !client->served)
			{
			sprintf(buf, "%d \"%s\" %d %d %d %.2f\n",
					sr->type, sr->basename,
					sr->id, sr->iodev, sr->inter, sr->raw_value);
			d = g_strconcat(s, buf, NULL);
			g_free(s);
			s = d;
			}
		}
	return s;
	}

static void
serve_sensors_setup(GkrellmdClient *client)
	{
	GList	*list;
	Sensor	*s;
	gchar	buf[256];

	send_to_client(client->fd, "<sensors_setup>\n");
	for (list = sensors_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		sprintf(buf, "%d \"%s\" %d %d %d %.4f %.4f \"%s\" \"%s\"\n",
				s->type, s->basename,
				s->id, s->iodev, s->inter,
				s->factor, s->offset, s->vref, s->default_label);
		send_to_client(client->fd, buf);
		}
	}

static GkrellmdMonitor sensors_monitor =
	{
	"sensors",
	update_sensors,
	serve_sensors_data,
	serve_sensors_setup
	};

static GkrellmdMonitor *
init_sensors_monitor(void)
	{
	if (!gkrellm_sys_sensors_init())
		return NULL;
	serveflag_done_list =
			g_list_append(serveflag_done_list, &sensors_need_serve);
	return &sensors_monitor;
	}

/* ======================================================= */
static time_t	base_uptime,
				up_seconds;
static gulong	up_minutes = -1;

static gboolean	need_uptime_serve;

void
gkrellm_uptime_set_base_uptime(time_t base)
	{
	base_uptime = base;
	}

static void
update_uptime(gboolean force)
	{
	gint	prev_up;

	if (GK.ten_second_tick || up_minutes < 0 || force)
		{
		prev_up = up_minutes;
		up_seconds = gkrellm_sys_uptime_read_uptime();
		if (up_seconds > 0)
			up_minutes = (gint) (up_seconds / 60);
		else
			up_minutes = (gint)(time(0) - _GK.start_time + base_uptime) / 60;
		if (up_minutes != prev_up)
			need_uptime_serve = TRUE;
		}
	}

static gchar *
serve_uptime_data(GkrellmdClient *client)
	{
	gchar	*s = NULL;

	if (need_uptime_serve || !client->served)
		{
		s = g_strdup_printf("<uptime>\n%lu\n", (gulong) up_minutes);
		need_uptime_serve = FALSE;
		}
	return s;
	}

static GkrellmdMonitor uptime_monitor =
	{
	"uptime",
	update_uptime,
	serve_uptime_data,
	NULL
	};

static GkrellmdMonitor *
init_uptime_monitor(void)
	{
	if (!gkrellm_sys_uptime_init())
		return NULL;
	return &uptime_monitor;
	}

/* ======================================================= */
static void
send_time(gint fd)
	{
	struct tm	*t;
	gchar		buf[128];

	t = &gkrellmd_current_tm;
	sprintf(buf, "<time>\n%d %d %d %d %d %d %d %d %d\n",
		t->tm_sec, t->tm_min, t->tm_hour,
		t->tm_mday, t->tm_mon, t->tm_year,
		t->tm_wday, t->tm_yday, t->tm_isdst);
	send_to_client(fd, buf);
	}

/* ======================================================= */

static void
add_monitor(GkrellmdMonitor *mon)
	{
	if (!mon)
		return;
	gkrellmd_monitor_list = g_list_append(gkrellmd_monitor_list, mon);
	}

void
load_monitors(void)
	{
	add_monitor(init_sensors_monitor());
	add_monitor(init_cpu_monitor());
	add_monitor(init_proc_monitor());
	add_monitor(init_disk_monitor());
	add_monitor(init_net_monitor());
	add_monitor(init_inet_monitor());
	add_monitor(init_mem_monitor());
	add_monitor(init_fs_monitor());
	add_monitor(init_battery_monitor());
	add_monitor(init_uptime_monitor());
	}

gint
update_monitors(void)
	{
	GList			*list, *c_list;
	GkrellmdMonitor	*mon;
	GkrellmdClient	*client;
	struct tm		*pCur;
	static time_t	time_prev, time_now;
	gchar			*d, *s, *send_string, buf[64];
	static gboolean	force = TRUE;

	time(&time_now);
	GK.second_tick = (time_now == time_prev) ? FALSE : TRUE;
	time_prev = time_now;

	if (GK.second_tick)
		{
		pCur = localtime(&time_now);
		GK.two_second_tick  = ((pCur->tm_sec % 2) == 0) ? TRUE : FALSE;
		GK.five_second_tick = ((pCur->tm_sec % 5) == 0) ? TRUE : FALSE;
		GK.ten_second_tick  = ((pCur->tm_sec % 10) == 0) ? TRUE : FALSE;
		GK.minute_tick = (pCur->tm_min  != gkrellmd_current_tm.tm_min);
		gkrellmd_current_tm = *pCur;
		}
	else
		{
		GK.two_second_tick = FALSE;
		GK.five_second_tick = FALSE;
		GK.ten_second_tick = FALSE;
		GK.minute_tick = FALSE;
		}

	for (list = gkrellmd_monitor_list; list; list = list->next)
		{
		mon = (GkrellmdMonitor *) list->data;
		if (mon->update_monitor)
			(*(mon->update_monitor))(force);
		}
	++GK.timer_ticks;
	for (c_list = gkrellmd_client_list; c_list; c_list = c_list->next)
		{
		client = (GkrellmdClient *) c_list->data;
		client->last_client = !c_list->next;
		if (!client->served)
			send_to_client(client->fd, "<initial_update>\n");
		send_string = g_strdup("");
		for (list = gkrellmd_monitor_list; list; list = list->next)
			{
			mon = (GkrellmdMonitor *) list->data;
			if (mon->serve_data)
				{
				d = (*(mon->serve_data))(client);
				if (d && *d)
					{
					s = g_strconcat(send_string, d, NULL);
					g_free(send_string);
					send_string = s;
					}
				g_free(d);
				}
			}
		send_to_client(client->fd, send_string);
		g_free(send_string);

		if (GK.minute_tick || !client->served)
			send_time(client->fd);
		else if (GK.second_tick)
			{
			sprintf(buf, "<.%d>\n", gkrellmd_current_tm.tm_sec);
			send_to_client(client->fd, buf);
			}

		if (!client->served)
			send_to_client(client->fd, "</initial_update>\n");
		client->served = TRUE;
		}

	for (list = serveflag_done_list; list; list = list->next)
		*((gboolean *) list->data) = FALSE;

	force = FALSE;
	return TRUE;
	}

void
gkrellmd_serve_setup(GkrellmdClient *client)
	{
	GList			*list;
	GkrellmdMonitor	*mon;
	gchar			buf[32], *s, *name;

	send_to_client(client->fd, "<gkrellmd_setup>\n");
	for (list = gkrellmd_monitor_list; list; list = list->next)
		{
		mon = (GkrellmdMonitor *) list->data;
		if (mon->serve_setup)
			(*(mon->serve_setup))(client);
		}
	name = gkrellm_sys_get_host_name();
	s = g_strconcat("<hostname>\n", name, "\n", NULL);
	send_to_client(client->fd, s);
	g_free(s);

	name = gkrellm_sys_get_system_name();
	s = g_strconcat("<sysname>\n", name, "\n", NULL);
	send_to_client(client->fd, s);
	g_free(s);

	send_time(client->fd);

	send_to_client(client->fd, "<monitors>\n");
	for (list = gkrellmd_monitor_list; list; list = list->next)
		{
		mon = (GkrellmdMonitor *) list->data;
		sprintf(buf, "%s\n", mon->name);
		send_to_client(client->fd, buf);
		}

	send_to_client(client->fd, "</gkrellmd_setup>\n");
	}

void
gkrellm_client_read(gint fd, gint nbytes)
	{
	GList			*list;
	GkrellmdClient	*client;
	gchar			buf[128];

	for (list = gkrellmd_client_list; list; list = list->next)
		{
		client = (GkrellmdClient *) list->data;
		if (client->fd != fd)
			continue;
		recv(fd, buf, nbytes, 0);
		printf("received %d bytes: %s\n", nbytes, buf);
		}
	}
