/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "gkrellm.h"
#include "gkrellm-private.h"


static GList	*gkrellm_alert_list;
static gint		voice_fifo = -1;

static gboolean
gvoice_fifo_open(void)
	{
	gchar	buf[32];

	if (voice_fifo < 0)
		{
		sprintf(buf, "/tmp/gvoice_%08x", getpid());
		if (!access(buf, W_OK))
			voice_fifo = open(buf, O_NONBLOCK|O_WRONLY);
		}
	if (voice_fifo >= 0)
		return TRUE;
	return FALSE;
	}
	
static void
send_gvoice(gchar *str)
	{
	if (!str || !*str || !gvoice_fifo_open())
		return;
	write(voice_fifo, str, strlen(str));
	}

static void
run_command(gchar *command)
	{
	if (!command || !*command)
		return;
	g_spawn_command_line_async(command, NULL);
	}

void
gkrellm_render_default_alert_decal(GkrellmAlert *alert)
	{
	GkrellmPiximage	*im;
	GkrellmAlertdecal		*ad;

	if (!alert)
		return;
	ad = &alert->ad;
	if (ad->w <= 0 || ad->h <= 0)
		return;
	if (alert->high.alarm_on || alert->low.alarm_on)
		gkrellm_get_decal_alarm_piximage(&im, &ad->nframes);
	else if (alert->high.warn_on || alert->low.warn_on)
		gkrellm_get_decal_warn_piximage(&im, &ad->nframes);
	else
		return;
	gkrellm_scale_piximage_to_pixmap(im, &ad->pixmap, &ad->mask, ad->w,
				ad->h * ad->nframes);
	}

static gboolean
create_alert_objects(GkrellmAlert *alert)
	{
	GkrellmAlertdecal		*ad;
//	GkrellmKrell			*k;

	/* Whichever monitor created the alert that is being triggered:
	|  1) has set a panel pointer in the create and is done, so I will make
	|     here a default panel sized alert decal for him, or
	|  2) has not set a panel ptr, but will do so in the callback and then
	|     let me make the default alert, or
	|  3) has done one of the 2 above panel possibilities, and will also
	|     render a custom size/position for the alert decal and set the pixmap.
	|     For this case, the alert decal may be a render of the default decal
	|     image or may be a custom image from the monitor.  The work needed is
	|     setting ad x,y,w,h and nframes and rendering to ad->pixmap.
	*/
	if (alert->cb_trigger)
		(*alert->cb_trigger)(alert, alert->cb_trigger_data);
	if (!alert->panel)
		return FALSE;		/* Monitor may need trigger deferred */
	ad = &alert->ad;
	if (!alert->ad.pixmap /* && style is decal alert */)
		{
		ad->x = 0;
		ad->y = 0;
		ad->w = alert->panel->w;
		ad->h = alert->panel->h;
		gkrellm_render_default_alert_decal(alert);
		}
	/* Don't let create_decal append the decal, I want to insert it first so
	|  it will appear under all other panel decals.
	*/
	ad->decal = gkrellm_create_decal_pixmap(NULL, ad->pixmap, ad->mask,
				ad->nframes, NULL, ad->x, ad->y);
	gkrellm_insert_decal(alert->panel, ad->decal, FALSE /* prepend */);

#if 0
	k = gkrellm_create_krell(NULL,
			gkrellm_krell_mini_piximage(), gkrellm_krell_mini_style());
	k->y0 = 5;
	k->full_scale = 100;

	alert->ak.krell = k;
	gkrellm_insert_krell(alert->panel, k, TRUE);
#endif
	return TRUE;
	}

static void
destroy_alert_objects(GkrellmAlert *alert)
	{
	gkrellm_destroy_decal(alert->ad.decal);
	if (alert->ad.pixmap)
		g_object_unref(G_OBJECT(alert->ad.pixmap));
	alert->ad.pixmap = NULL;
	alert->ad.mask = NULL;
	alert->ad.decal = NULL;
	gkrellm_destroy_krell(alert->ak.krell);
	alert->ak.krell = NULL;
	}

static void
trigger_warn(GkrellmAlert *alert, GkrellmTrigger *trigger)
	{
	if (!trigger->warn_on)
		{
		trigger->warn_on = TRUE;
		create_alert_objects(alert);
		gkrellm_alert_list = g_list_append(gkrellm_alert_list, alert);
		if (!alert->suppress_command)
			run_command(alert->warn_command);
		alert->suppress_command = FALSE;
		alert->warn_repeat = alert->warn_repeat_set;
		send_gvoice(alert->voice_string);
		alert->voice_repeat = alert->voice_repeat_set;
		}
	}

static void
stop_warn(GkrellmAlert *alert, GkrellmTrigger *trigger)
	{
	if (trigger->warn_on)
		{
		trigger->warn_on = FALSE;
		destroy_alert_objects(alert);
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		alert->warn_repeat = 0;
		alert->suppress_command = FALSE;
		}
	}

static void
trigger_alarm(GkrellmAlert *alert, GkrellmTrigger *trigger)
	{
	if (!trigger->alarm_on)
		{
		trigger->alarm_on = TRUE;
		create_alert_objects(alert);
		gkrellm_alert_list = g_list_append(gkrellm_alert_list, alert);
		if (!alert->suppress_command)
			run_command(alert->alarm_command);
		alert->suppress_command = FALSE;
		alert->alarm_repeat = alert->alarm_repeat_set;
		send_gvoice(alert->voice_string);
		alert->voice_repeat = alert->voice_repeat_set;
		}
	}

static void
stop_alarm(GkrellmAlert *alert, GkrellmTrigger *trigger)
	{
	if (trigger->alarm_on)
		{
		trigger->alarm_on = FALSE;
		destroy_alert_objects(alert);
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		alert->alarm_repeat = 0;;
		alert->suppress_command = FALSE;
		}
	}

void
gkrellm_freeze_alert(GkrellmAlert *alert)
	{
	if (alert)
		alert->freeze = TRUE;
	}

void
gkrellm_thaw_alert(GkrellmAlert *alert)
	{
	if (alert)
		alert->freeze = FALSE;
	}

void
gkrellm_check_alert(GkrellmAlert *alert, gfloat value)
	{
	if (!alert || !alert->activated || alert->freeze || !_GK.initialized)
		return;
	if (alert->check_low)
		{
		if (value <= alert->low.alarm_limit)
			{				
			stop_alarm(alert, &alert->high);
			stop_warn(alert, &alert->high);
			stop_warn(alert, &alert->low);
			trigger_alarm(alert, &alert->low);
			}
		else
			{
			stop_alarm(alert, &alert->low);
			if (value <= alert->low.warn_limit)
				{
				stop_alarm(alert, &alert->high);
				stop_warn(alert, &alert->high);
				trigger_warn(alert, &alert->low);
				}
			else
				stop_warn(alert, &alert->low);
			}
		}
	if (alert->check_high)
		{
		if (value >= alert->high.alarm_limit)
			{
			stop_warn(alert, &alert->high);
			stop_warn(alert, &alert->low);
			stop_alarm(alert, &alert->low);
			trigger_alarm(alert, &alert->high);
			}
		else
			{
			stop_alarm(alert, &alert->high);
			if (value >= alert->high.warn_limit)
				{
				stop_warn(alert, &alert->low);
				stop_alarm(alert, &alert->low);
				trigger_warn(alert, &alert->high);
				}
			else
				stop_warn(alert, &alert->high);
			}
		}
	}

void
gkrellm_alert_trigger_connect(GkrellmAlert *alert, void (*func)(),
			gpointer data)
	{
	if (!alert)
		return;
	alert->cb_trigger = func;
	alert->cb_trigger_data = data;
	}

void
gkrellm_alert_stop_connect(GkrellmAlert *alert, void (*func)(), gpointer data)
	{
	if (!alert)
		return;
	alert->cb_stop = func;
	alert->cb_stop_data = data;
	}

static void
destroy_alert(GkrellmAlert *alert)
	{
	if (g_list_find(gkrellm_alert_list, alert))
		{
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		destroy_alert_objects(alert);
		}
	g_free(alert->name);
	g_free(alert->unit_string);
	g_free(alert->alarm_command);
	g_free(alert->warn_command);
	g_free(alert);
	}

void
gkrellm_alert_destroy(GkrellmAlert **alert)
	{
	if (!alert || !*alert)
		return;
	gkrellm_alert_window_destroy(alert);
	destroy_alert(*alert);
	*alert = NULL;
	}

GkrellmAlert *
gkrellm_alert_create(GkrellmPanel *p, gchar *name, gchar *unit_string,
		gboolean check_high, gboolean check_low, gboolean do_updates,
		gfloat max_high, gfloat min_low,
		gfloat step0, gfloat step1, gint digits)
	{
	GkrellmAlert	*alert;

	alert = g_new0(GkrellmAlert, 1);
	alert->panel = p;
	alert->name = g_strdup(name);
	alert->unit_string = g_strdup(unit_string ? unit_string : "");
	alert->check_high = check_high;
	alert->check_low = check_low;
	alert->do_panel_updates = do_updates;
	alert->max_high = max_high;
	alert->min_low = min_low;
	alert->step0 = step0;
	alert->step1 = step1;
	alert->digits = digits;
	return alert;
	}

  /* Set alarm trigger values for hardwired alarms which have no config
  */
void
gkrellm_alert_set_triggers(GkrellmAlert *alert,
				gfloat high_alarm, gfloat high_warn,
				gfloat low_warn, gfloat low_alarm)
	{
	if (alert->check_high)
		{
		alert->high.alarm_limit = high_alarm;
		alert->high.warn_limit = high_warn;
		}
	if (alert->check_low)
		{
		alert->low.warn_limit = low_warn;
		alert->low.alarm_limit = low_alarm;
		}
	alert->activated = TRUE;
	}

static void
reset_alert(GkrellmAlert *alert)
	{
	GkrellmTrigger	*th = &alert->high,
					*tl = &alert->low;

	if (   ((th->alarm_on || tl->alarm_on) && alert->alarm_repeat_set == 0)
		|| ((th->warn_on || tl->warn_on) && alert->warn_repeat_set == 0)
	   )
		alert->suppress_command = TRUE;
	th->alarm_on = FALSE;
	th->warn_on = FALSE;
	tl->alarm_on = FALSE;
	tl->warn_on = FALSE;
	alert->alarm_repeat = 0;
	alert->warn_repeat = 0;
	destroy_alert_objects(alert);
	}

void
gkrellm_reset_alert(GkrellmAlert *alert)
	{
	GList	*list;

	if (!alert)
		return;
	list = g_list_find(gkrellm_alert_list, alert);
	if (list)
		{
		reset_alert(alert);
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		}
	}

void
gkrellm_reset_panel_alerts(GkrellmPanel *p)
	{
	GList			*list;
	GkrellmAlert	*alert;
	gboolean		done = FALSE;

	if (!p)
		return;
	while (!done)
		{
		done = TRUE;	/* Assume won't find any */
		for (list = gkrellm_alert_list; list; list = list->next)
			{
			alert = (GkrellmAlert *) list->data;
			if (alert->panel != p)
				continue;
			done = FALSE;
			reset_alert(alert);
			gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
			break;
			}
		}
	}

  /* At theme changes, turn all alerts off so there won't be any alert decals
  |  in monitor lists when they are destroyed.  The alerts should just get
  |  retriggered.  Surely an alert going off is a good time to change themes.
  */
void
reset_all_alerts(void)
	{
	GList	*list;

	for (list = gkrellm_alert_list; list; list = list->next)
		reset_alert((GkrellmAlert *) list->data);
	g_list_free(gkrellm_alert_list);
	gkrellm_alert_list = NULL;
	}

void
update_alerts(void)
	{
	GList			*list;
	GkrellmAlert	*alert;
	GkrellmAlertdecal *ad;
	GkrellmAlertkrell *ak;

	for (list = gkrellm_alert_list; list; list = list->next)
		{
		alert = (GkrellmAlert *) list->data;
		ad = &alert->ad;
		ak = &alert->ak;
		if (ak->krell)
			{
			ak->krell_position = (ak->krell_position + 2) % 100;
			gkrellm_update_krell(alert->panel, ak->krell, ak->krell_position);
			}
		if (ad->decal)
			{
			if (ad->frame <= 0)
				ad->dir = 1;
			else if (ad->frame >= ad->nframes - 1)
				ad->dir = 0;
			ad->frame += (ad->dir) ? 1 : -1;
			gkrellm_draw_decal_pixmap(alert->panel, ad->decal, ad->frame);
			}
		if (alert->do_panel_updates)
			gkrellm_draw_panel_layers(alert->panel);
		if (GK.second_tick)
			{
			if (alert->alarm_repeat > 0 && --alert->alarm_repeat == 0)
				{
				run_command(alert->alarm_command);
				alert->alarm_repeat = alert->alarm_repeat_set;
				}
			if (alert->warn_repeat > 0 && --alert->warn_repeat == 0)
				{
				run_command(alert->warn_command);
				alert->warn_repeat = alert->warn_repeat_set;
				}
			if (alert->voice_repeat > 0 && --alert->voice_repeat == 0)
				{
				send_gvoice(alert->voice_string);
				alert->voice_repeat = alert->voice_repeat_set;
				}
			}
		}
	}

/* ------------------------------------------------------------ */
void
gkrellm_alert_config_connect(GkrellmAlert *alert, void (*func)(),gpointer data)
	{
	if (!alert)
		return;
	alert->cb_config = func;
	alert->cb_config_data = data;
	}

static void
alert_delete(GtkWidget *widget, GkrellmAlert **ap)
	{
	GkrellmAlert	*alert;

	if (!ap)
		return;
	alert = *ap;
	alert->activated = FALSE;
	if (alert->cb_config)
		(*alert->cb_config)(alert, alert->cb_config_data);
	if (alert->config_window)
		gtk_widget_destroy(alert->config_window);
	destroy_alert(alert);
	*ap = NULL;
	}

static void
alert_close(GtkWidget *widget, GkrellmAlert **alert)
	{
	GkrellmAlert	*ap;

	if (!alert)
		return;
	ap = *alert;
	if (!ap->activated)
		alert_delete(NULL, alert);
	else if (ap->config_window)
		{
		gtk_widget_destroy(ap->config_window);
		ap->config_window = NULL;
		}
	}

void
gkrellm_alert_window_destroy(GkrellmAlert **alert)
	{
	alert_close(NULL, alert);
	}

static gint
alert_config_window_delete_event(GtkWidget *widget, GdkEvent *ev,
		GkrellmAlert **alert)
	{
	alert_close(widget, alert);
	return FALSE;
	}

static void
alert_apply(GtkWidget *widget, GkrellmAlert **ap)
	{
	GkrellmAlert	*alert;
	gchar			*s;
	GtkSpinButton	*spin;

	alert = *ap;
	if (alert->check_high)
		{
		spin = GTK_SPIN_BUTTON(alert->high.alarm_limit_spin_button);
		alert->high.alarm_limit = gtk_spin_button_get_value(spin);
		spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
		alert->high.warn_limit = gtk_spin_button_get_value(spin);
		}
	if (alert->check_low)
		{
		spin = GTK_SPIN_BUTTON(alert->low.alarm_limit_spin_button);
		alert->low.alarm_limit = gtk_spin_button_get_value(spin);
		spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
		alert->low.warn_limit = gtk_spin_button_get_value(spin);
		}
	spin = GTK_SPIN_BUTTON(alert->alarm_repeat_spin_button);
	alert->alarm_repeat_set = gtk_spin_button_get_value_as_int(spin);
	spin = GTK_SPIN_BUTTON(alert->warn_repeat_spin_button);
	alert->warn_repeat_set = gtk_spin_button_get_value_as_int(spin);
	if (alert->voice_repeat_spin_button)
		{
		spin = GTK_SPIN_BUTTON(alert->voice_repeat_spin_button);
		alert->voice_repeat_set = gtk_spin_button_get_value_as_int(spin);
		}
	s = gkrellm_gtk_entry_get_text(&alert->alarm_command_entry);
	gkrellm_dup_string(&alert->alarm_command, s);
	if (!*s)
		alert->alarm_repeat_set = 0;
	if (alert->high.alarm_on || alert->low.alarm_on)
		alert->alarm_repeat = alert->alarm_repeat_set;

	s = gkrellm_gtk_entry_get_text(&alert->warn_command_entry);
	gkrellm_dup_string(&alert->warn_command, s);
	if (!*s)
		alert->warn_repeat_set = 0;
	if (alert->high.warn_on || alert->low.warn_on)
		alert->warn_repeat = alert->warn_repeat_set;

	if (alert->voice_string_entry)
		{
		s = gkrellm_gtk_entry_get_text(&alert->voice_string_entry);
		gkrellm_dup_string(&alert->voice_string, s);
		if (!*s)
			alert->voice_repeat_set = 0;
		alert->voice_repeat = alert->voice_repeat_set;
		}
	alert->activated = TRUE;
	if (alert->cb_config)
		(*alert->cb_config)(alert, alert->cb_config_data);
	gkrellm_config_modified();
	}

static void
alert_ok(GtkWidget *widget, GkrellmAlert **alert)
	{
	alert_apply(NULL, alert);
	alert_close(NULL, alert);
	}

static void
cb_high_alarm_spin_changed(GtkAdjustment *adjustment, GkrellmAlert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn;

	spin = GTK_SPIN_BUTTON(alert->high.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value(spin);
	spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
	warn = gtk_spin_button_get_value(spin);
	if (alarm < warn)
		gtk_spin_button_set_value(spin, alarm);
	}

static void
cb_high_warn_spin_changed(GtkWidget *adjustment, GkrellmAlert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn, low_warn;

	spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
	warn = gtk_spin_button_get_value(spin);
	spin = GTK_SPIN_BUTTON(alert->high.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value(spin);
	if (alarm < warn)
		gtk_spin_button_set_value(spin, warn);
	if (alert->check_low)
		{
		spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
		low_warn = gtk_spin_button_get_value(spin);
		if (low_warn > warn)
			gtk_spin_button_set_value(spin, warn);
		}
	}

static void
cb_low_warn_spin_changed(GtkWidget *adjustment, GkrellmAlert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn, high_warn;

	spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
	warn = gtk_spin_button_get_value(spin);
	spin = GTK_SPIN_BUTTON(alert->low.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value(spin);
	if (alarm > warn)
		gtk_spin_button_set_value(spin, warn);
	if (alert->check_high)
		{
		spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
		high_warn = gtk_spin_button_get_value(spin);
		if (high_warn < warn)
			gtk_spin_button_set_value(spin, warn);
		}
	}

static void
cb_low_alarm_spin_changed(GtkWidget *adjustment, GkrellmAlert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn;

	spin = GTK_SPIN_BUTTON(alert->low.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value(spin);
	spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
	warn = gtk_spin_button_get_value(spin);
	if (alarm > warn)
		gtk_spin_button_set_value(spin, alarm);
	}

void
gkrellm_alert_config_window(GkrellmAlert **alert)
	{
	GtkWidget	*main_vbox, *vbox, *vbox1, *hbox;
	GtkWidget	*table;
	GtkWidget	*button;
	GtkWidget	*label;
	GtkWidget	*separator;
	GkrellmAlert *ap;
	gchar		*title;

	if (!alert || !*alert)
		return;
	ap = *alert;
	if (!ap->config_window)
		{
		ap->config_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		g_signal_connect(G_OBJECT(ap->config_window), "delete_event",
				G_CALLBACK(alert_config_window_delete_event), alert);
		gtk_window_set_title(GTK_WINDOW(ap->config_window),
				_("GKrellM Set Alerts"));
	
		gtk_container_set_border_width(GTK_CONTAINER(ap->config_window), 4);
		main_vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(ap->config_window), main_vbox);
		
		title = g_strdup_printf("%s - %s", ap->name, ap->unit_string);
		vbox = gkrellm_gtk_framed_vbox(main_vbox, title, 4, FALSE, 4, 3);
		g_free(title);

		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
		if (ap->check_high)
			{
			vbox1 = gkrellm_gtk_framed_vbox(hbox, _("High Limits"),
						2, FALSE, 2,2);
			gkrellm_gtk_spin_button(vbox1, &ap->high.alarm_limit_spin_button,
					ap->high.alarm_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_high_alarm_spin_changed, ap, FALSE,
					_("High alarm limit"));
		
			gkrellm_gtk_spin_button(vbox1, &ap->high.warn_limit_spin_button,
					ap->high.warn_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_high_warn_spin_changed, ap, FALSE,
					_("High warn limit"));
			}

		if (ap->check_low)
			{
			vbox1 = gkrellm_gtk_framed_vbox_end(hbox, _("Low Limits"),
					2, FALSE, 2, 2);
			gkrellm_gtk_spin_button(vbox1, &ap->low.warn_limit_spin_button,
					ap->low.warn_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_low_warn_spin_changed, ap, FALSE,
					_("Low warn limit"));
			gkrellm_gtk_spin_button(vbox1, &ap->low.alarm_limit_spin_button,
					ap->low.alarm_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_low_alarm_spin_changed, ap, FALSE,
					_("Low alarm limit"));
			}
		vbox1 = gkrellm_gtk_framed_vbox(vbox,
				_("Commands - with repeat intervals in seconds"),
				2, FALSE, 2, 2);

		table = gtk_table_new(3 /* across */, 3 /* down */, FALSE);
		gtk_table_set_col_spacings(GTK_TABLE(table), 4);
		gtk_box_pack_start(GTK_BOX(vbox1), table, FALSE, FALSE, 0);

		if (!ap->alarm_command)
			ap->alarm_command = g_strdup("");
		label = gtk_label_new(_("Alarm command:"));
		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
		ap->alarm_command_entry = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(ap->alarm_command_entry), 255);
		gtk_table_attach_defaults(GTK_TABLE(table), ap->alarm_command_entry,
				1, 2, 0, 1);
		gtk_entry_set_text(GTK_ENTRY(ap->alarm_command_entry),
				ap->alarm_command);
		gkrellm_gtk_spin_button(NULL, &ap->alarm_repeat_spin_button,
				ap->alarm_repeat_set, 0, 1000,
				1, 10, 0, 60, NULL, NULL, FALSE, NULL);
		gtk_table_attach_defaults(GTK_TABLE(table),
				ap->alarm_repeat_spin_button, 2, 3, 0, 1);

		if (!ap->warn_command)
			ap->warn_command = g_strdup("");
		label = gtk_label_new(_("Warn command:"));
		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
		ap->warn_command_entry = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(ap->warn_command_entry), 255);
		gtk_table_attach_defaults(GTK_TABLE(table), ap->warn_command_entry,
				1, 2, 1, 2);
		gtk_entry_set_text(GTK_ENTRY(ap->warn_command_entry),
				ap->warn_command);
		gtk_widget_set_size_request(ap->warn_command_entry, 300, -1);
		gkrellm_gtk_spin_button(NULL, &ap->warn_repeat_spin_button,
				ap->warn_repeat_set, 0, 1000,
				1, 10, 0, 60, NULL, NULL, FALSE, NULL);
		gtk_table_attach_defaults(GTK_TABLE(table),
				ap->warn_repeat_spin_button, 2, 3, 1, 2);

		if (!ap->voice_string)
			ap->voice_string = g_strdup("");
		if (gvoice_fifo_open())
			{
			label = gtk_label_new(_("Voice string:"));
			gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
			gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
			ap->voice_string_entry = gtk_entry_new();
			gtk_entry_set_max_length(GTK_ENTRY(ap->voice_string_entry), 255);
			gtk_table_attach_defaults(GTK_TABLE(table), ap->voice_string_entry,
					1, 2, 2, 3);
			gtk_entry_set_text(GTK_ENTRY(ap->voice_string_entry),
					ap->voice_string);
			gtk_widget_set_size_request(ap->voice_string_entry, 300, -1);
			gkrellm_gtk_spin_button(NULL, &ap->voice_repeat_spin_button,
					ap->voice_repeat_set, 0, 1000,
					1, 10, 0, 60, NULL, NULL, FALSE, NULL);
			gtk_table_attach_defaults(GTK_TABLE(table),
					ap->voice_repeat_spin_button, 2, 3, 2, 3);
			}
		else
			{
			ap->voice_string_entry = NULL;
			ap->voice_repeat_spin_button = NULL;
			}

		separator = gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox1), separator, FALSE, FALSE, 4);

		label = gtk_label_new(
		   _("A repeat of zero seconds executes the command once per alert."));
		gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
		gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE, 0);

		hbox = gtk_hbutton_box_new();
		gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
		gtk_box_set_spacing(GTK_BOX(hbox), 5);
		gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);

		button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(alert_delete), alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

		button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(alert_apply), alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

		button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(alert_close), alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

		button = gtk_button_new_from_stock(GTK_STOCK_OK);
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(alert_ok), alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		gtk_widget_grab_default(button);

		gtk_widget_show_all(ap->config_window);
		}
	else
		gtk_window_present(GTK_WINDOW(ap->config_window));
	}

void
gkrellm_save_alertconfig(FILE *f, GkrellmAlert *alert,
			gchar *mon_keyword, gchar *name)
	{
	gchar		*s;

	if (!f || !alert || !mon_keyword)
		return;
	if (name)
		s = g_strdup_printf("%s %s %s ", mon_keyword,
				GKRELLM_ALERTCONFIG_KEYWORD, name);
	else
		s = g_strdup_printf("%s %s ", mon_keyword,GKRELLM_ALERTCONFIG_KEYWORD);

	if (alert->alarm_command && *alert->alarm_command)
		fprintf(f, "%s alarm_command %s\n", s, alert->alarm_command);
	if (alert->warn_command && *alert->warn_command)
		fprintf(f, "%s warn_command %s\n", s, alert->warn_command);
	if (alert->voice_string && *alert->voice_string)
		fprintf(f, "%s voice_string %s\n", s, alert->voice_string);
	fprintf(f, "%s values %d %d %d %d\n", s, alert->do_panel_updates,
			alert->check_high, alert->check_low, alert->check_boolean);
	fprintf(f, "%s repeat %d %d %d\n", s, alert->alarm_repeat_set,
			alert->warn_repeat_set, alert->voice_repeat_set);
	fprintf(f, "%s limits %.2f %.2f %.2f %.2f\n", s,
			alert->high.alarm_limit, alert->high.warn_limit,
			alert->low.warn_limit, alert->low.alarm_limit);
	g_free(s);
	}

void
gkrellm_load_alertconfig(GkrellmAlert **ap, gchar *config_line)
	{
	GkrellmAlert *alert;
	gchar		config[32], item[CFG_BUFSIZE];
	gint		n;

	if (!ap || !config_line)
		return;
	if (!*ap)
		*ap = g_new0(GkrellmAlert, 1);
	alert = *ap;

	n = sscanf(config_line, "%31s %[^\n]", config, item);
	if (n != 2)
		return;

	if (!strcmp(config, "alarm_command"))
		gkrellm_dup_string(&alert->alarm_command, item);
	else if (!strcmp(config, "warn_command"))
		gkrellm_dup_string(&alert->warn_command, item);
	else if (!strcmp(config, "voice_string"))
		gkrellm_dup_string(&alert->voice_string, item);
	else if (!strcmp(config, "values"))
		sscanf(item, "%d %d %d %d", &alert->do_panel_updates,
				&alert->check_high, &alert->check_low, &alert->check_boolean);
	else if (!strcmp(config, "repeat"))
		sscanf(item, "%d %d %d", &alert->alarm_repeat_set,
				&alert->warn_repeat_set, &alert->voice_repeat_set);
	else if (!strcmp(config, "limits"))
		sscanf(item, "%f %f %f %f",
				&alert->high.alarm_limit, &alert->high.warn_limit,
				&alert->low.warn_limit, &alert->low.alarm_limit);

	alert->activated = TRUE;
	}


/* ------------------------------------------------------------------- */
/* gdk-pixbuf-csource --static alert_inline.png */

/* GdkPixbuf RGBA C-Source image dump 1-byte-run-length-encoded */

static const guint8 alert_inline[] = 
{ ""
  /* Pixbuf magic (0x47646b50) */
  "GdkP"
  /* length: header (24) + pixel_data (925) */
  "\0\0\3\265"
  /* pixdata_type (0x2010002) */
  "\2\1\0\2"
  /* rowstride (72) */
  "\0\0\0H"
  /* width (18) */
  "\0\0\0\22"
  /* height (18) */
  "\0\0\0\22"
  /* pixel_data: */
  "\227\0\0\0\0\10\204}{\377B89\377\20\24\20\377\10\2\10\377\0\2\0\377\30"
  "\24\20\3779<9\377\214\212\204\377\211\0\0\0\0\12e`T\37732*\377\202]@"
  "\377\251eF\377\267g?\377\265cF\377\243eG\377\202VG\37773$\377io`\377"
  "\207\0\0\0\0\14PVG\377a<*\377\257^?\377\310\211q\377\333\262\241\377"
  "\222\202t\377\311\254\234\377\333\262\233\377\310\211q\377\257^?\377"
  "Z=+\377\\bM\377\205\0\0\0\0\16cfS\377Z=+\377\257^?\377\320\234\203\377"
  "\333\270\241\377\252\217}\377\31\25\23\3772*%\377\342\274\251\377\333"
  "\270\241\377\320\234\203\377\257^?\377Z=+\377|~k\377\203\0\0\0\0\17\204"
  "\206{\37773$\377\251_?\377\301\202b\377\325\253\222\377\341\255\233\377"
  "aNE\377\222tb\377\0\0\0\377\310\232\211\377\333\257\233\377\325\253\222"
  "\377\302{h\377\251Z?\37773$\377\203\0\0\0\0\12JAB\377xQ1\377\257[?\377"
  "\320\234\203\377\325\234\203\377\333\236\205\3770#\35\377\333\236\205"
  "\377\30\21\17\377aF;\377\202\325\234\203\377\4\310\222u\377\257[?\377"
  "tJ1\377kik\377\202\0\0\0\0\5\20\24\20\377\235[G\377\257[?\377\317\212"
  "p\377\325\215u\377\202vOA\377\11\325\215u\377vM>\377\0\0\0\377\325\213"
  "q\377\317\212p\377\317\213v\377\260T8\377\235[G\377)$)\377\202\0\0\0"
  "\0\20\0\2\0\377\251Z?\377\267X?\377\310q\\\377\310{c\377.\32\25\377\270"
  "lW\377\316z\\\377sA6\377\0\0\0\377sA6\377\317vb\377\301mV\377\257Q?\377"
  "\245R8\377\10\2\10\377\202\0\0\0\0\6\0\2\0\377\251S?\377\260T8\377\277"
  "N9\377\226M9\377,\25\21\377\202\205@4\377\10\200B4\377,\25\21\377+\26"
  "\20\377\277N9\377\270K9\377\252M8\377\237N8\377\10\2\10\377\202\0\0\0"
  "\0\20\30\24\30\377\230Y@\377\252S8\377\267R?\377T)\34\377\200>.\377\300"
  "]F\377\307[F\377\300]F\377\200<.\377\0\0\0\377\221J<\377\260L8\377\245"
  "J8\377\230VG\377!(!\377\202\0\0\0\0\7""9<1\377xO@\377\245J8\377\227D"
  "1\377\0\0\0\377\266S8\377\302Q?\377\203\274S?\377\6\0\0\0\377N\37\31"
  "\377\252F8\377\236F2\377~N@\377cic\377\202\0\0\0\0\12\234\216\224\377"
  "73$\377\237G8\377\22\10\6\377\0\0\0\377N\40\26\377\242?1\377\257I2\377"
  "\266G8\377N\40\26\377\202\0\0\0\377\3F\37\26\377\231D9\37773$\377\204"
  "\0\0\0\0\5ii`\377T3#\377\231E2\377\237G8\377\252F8\377\203\251B2\377"
  "\6\252F8\377\236@2\377\231@2\377\231E2\377N6#\377{\204q\377\205\0\0\0"
  "\0\4baY\377T3#\377\223D2\377\231E2\377\202\236@2\377\6\237<6\377\231"
  "E2\377\231@2\377\223@2\377T3#\377ii`\377\207\0\0\0\0\4t}j\377-4#\377"
  "~N@\377\216K9\377\202\223@2\377\4\222K@\377xO@\37773$\377{\204q\377\212"
  "\0\0\0\0\6kik\377!(!\377\10\2\10\377\0\2\0\377)$)\377kik\377\230\0\0"
  "\0\0"};

static GdkPixbuf	*alert_pixbuf;

GdkPixbuf *
gkrellm_alert_pixbuf(void)
	{
	if (!alert_pixbuf)
		alert_pixbuf = gdk_pixbuf_new_from_inline(-1, alert_inline,
						FALSE, NULL);
	return alert_pixbuf;
	}
