/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"
#include "gkrellm-private.h"
#include "gkrellm-sysdeps.h"

#define DISK_ASSIGN_BY_DEVICE   0
#define DISK_ASSIGN_NTH         1
#define DISK_ASSIGN_BY_NAME     2

#define	DISK_CONFIG_KEYWORD	"disk"

#define	MIN_GRID_RES		10
#define	MAX_GRID_RES		10000000
#define	DEFAULT_GRID_RES	2000

typedef struct
	{
	gchar			*name;
	GtkWidget		*vbox;
	GtkWidget		*enable_button;
	GkrellmChart	*chart;
	GkrellmChartdata *rb_cd,
					 *wb_cd;
	GkrellmChartconfig *chart_config;
	gint			enabled;
	gint			major,
					minor,
					subdisk,
					order;
	gint			new_disk;
	gint			extra_info;
	GkrellmLauncher	launch;
	GtkWidget		*launch_entry,
					*tooltip_entry;
	GtkWidget		*launch_table;

	guint64			rb,
					wb;
	}
	DiskMon;


static GkrellmMonitor	*mon_disk;

static gint		n_disks;
static GList	*disk_mon_list;


static DiskMon	*composite_disk;
static gint		ascent;
static gint		style_id;
static gint		assign_method;
static gboolean	sys_handles_composite_reset;
static gboolean	units_are_blocks;

static void		(*read_disk_data)();
static gchar	*(*name_from_device)();
static gint		(*order_from_name)();



DiskMon *
lookup_disk_by_device(gint major, gint minor)
	{
	DiskMon	*disk;
	GList	*list;

	for (list = disk_mon_list->next; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (disk->major == major && disk->minor == minor)
			return disk;
		}
	return NULL;
	}

static DiskMon *
lookup_disk_by_name(gchar *name)
	{
	DiskMon	*disk;
	GList	*list;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (!strcmp(name, disk->name))
			return disk;
		}
	return NULL;
	}

static DiskMon *
new_disk(gchar *name)
	{
	DiskMon	*disk;

	disk = g_new0(DiskMon, 1);
	disk->name = g_strdup(name);
	disk->launch.command = g_strdup("");
	disk->launch.tooltip_comment = g_strdup("");
	return disk;
	}

static DiskMon *
add_disk(gchar *name, gint major, gint minor, gint order)
	{
	DiskMon	*disk;
	GList	*list;
	gint	i;

	if (lookup_disk_by_name(name))
		return NULL;
	disk = new_disk(name);

	disk->major = major;
	disk->minor = minor;
	disk->order = order;
	disk->subdisk = -1;
	if (order >= 0)
		{
		++disk->order;		/* Skip the composite disk */
		for (i = 1, list = disk_mon_list->next; list; list = list->next, ++i)
			if (disk->order < ((DiskMon *) list->data)->order)
				break;
		disk_mon_list = g_list_insert(disk_mon_list, disk, i);
		}
	else
		disk_mon_list = g_list_append(disk_mon_list, disk);
	++n_disks;
	return disk;
	}

static DiskMon *
add_subdisk(gchar *subdisk_name, gchar *disk_name, gint subdisk)
	{
	DiskMon	*disk, *sdisk;
	GList	*list = NULL;

	for (list = disk_mon_list->next; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (!strcmp(disk_name, disk->name))
			break;
		}
	if (!list)
		return NULL;
	sdisk = new_disk(subdisk_name);
	sdisk->order = disk->order;
	sdisk->subdisk = subdisk;
	for (list = list->next; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (disk->subdisk == -1 || disk->subdisk > subdisk)
			break;
		}
	disk_mon_list = g_list_insert_before(disk_mon_list, list, sdisk);
	++n_disks;
	return sdisk;
	}

static void
disk_assign_data(DiskMon *disk, guint64 rb, guint64 wb)
	{
	if (disk)
		{
		disk->rb = rb;
		disk->wb = wb;
		}
	if (disk->subdisk == -1)	/* not a subdisk (partition) */
		{
		composite_disk->rb += rb;
		composite_disk->wb += wb;
		}
	}

static gboolean
setup_disk_interface(void)
	{
	if (!read_disk_data && !_GK.client_mode && gkrellm_sys_disk_init())
		{
		read_disk_data = gkrellm_sys_disk_read_data;
		name_from_device = gkrellm_sys_disk_name_from_device;
		order_from_name = gkrellm_sys_disk_order_from_name;
		}
	/* Get a read in so I'll know the assign_method before config is loaded.
	*/
	if (read_disk_data)
		(*read_disk_data)();
	return read_disk_data ? TRUE : FALSE;
	}

/* ------------- Disk monitor to system dependent interface ------------- */
void
gkrellm_disk_client_divert(void (*read_func)(),
		gchar *(*name_from_device_func)(), gint (*order_from_name_func)())
	{
	read_disk_data = read_func;
	name_from_device = name_from_device_func;
	order_from_name = order_from_name_func;
	}

void
gkrellm_disk_reset_composite(void)
	{
	composite_disk->rb = 0;
	composite_disk->wb = 0;
	sys_handles_composite_reset = TRUE;
	}

void
gkrellm_disk_units_are_blocks(void)
	{
	units_are_blocks = TRUE;
	}
	
void
gkrellm_disk_assign_data_by_device(gint device_number, gint unit_number,
			guint64 rb, guint64 wb)
	{
	DiskMon	*disk;
	gchar	*name;
	gint	order = -1;

	assign_method = DISK_ASSIGN_BY_DEVICE;
	disk = lookup_disk_by_device(device_number, unit_number);
	if (!disk && name_from_device)
		{
		name = (*name_from_device)(device_number, unit_number, &order);
		if (name)
			disk = add_disk(name, device_number, unit_number, order);
		}
	disk_assign_data(disk, rb, wb);
	}

void
gkrellm_disk_assign_data_nth(gint n, guint64 rb, guint64 wb)
	{
	DiskMon	*disk;
	gchar	name[32];

	assign_method = DISK_ASSIGN_NTH;
	if (n < n_disks)
		disk = (DiskMon *) g_list_nth_data(disk_mon_list, n + 1);
	else
		{
		sprintf(name, "%s%c", _("Disk"), 'A' + n);
		disk = add_disk(name, 0, 0, n);
		}
	disk_assign_data(disk, rb, wb);
	}

void
gkrellm_disk_assign_data_by_name(gchar *name, guint64 rb, guint64 wb)
	{
	DiskMon	*disk;
	gint	order = -1;

	assign_method = DISK_ASSIGN_BY_NAME;
	if (!name)
		return;
	disk = lookup_disk_by_name(name);
	if (!disk)
		{
		if (order_from_name)
			order = (*order_from_name)(name);
		disk = add_disk(name, 0, 0, order);
		}
	disk_assign_data(disk, rb, wb);
	}

void
gkrellm_disk_subdisk_assign_data_by_name(gchar *subdisk_name, gchar *disk_name,
					guint64 rb, guint64 wb)
	{
	DiskMon	*disk;
	gchar	*s, *endptr;
	gint	subdisk;

	assign_method = DISK_ASSIGN_BY_NAME;
	if (!subdisk_name || !disk_name)
		return;
	disk = lookup_disk_by_name(subdisk_name);
	if (!disk)
		{
		/* A subdisk name is expected to be the disk_name with a number string
		|  appended.  Eg. "hda1" is a subdisk_name of disk_name "hda"
		*/
		s = subdisk_name + strlen(disk_name);
		subdisk = strtol(s, &endptr, 0);
		if (!*s || *endptr)
			return;
		disk = add_subdisk(subdisk_name, disk_name, subdisk);
		}
	disk_assign_data(disk, rb, wb);
	}

static GkrellmSizeAbbrev	disk_blocks_abbrev[]	=
	{
	{ KB_SIZE(1),		1,				"%.0f" },
	{ KB_SIZE(20),		KB_SIZE(1),		"%.1fK" },
	{ MB_SIZE(1),		KB_SIZE(1),		"%.0fK" },
	{ MB_SIZE(20),		MB_SIZE(1),		"%.1fM" }
	};


static gchar    *text_format,
				*text_format_locale;

static void
format_chart_text(DiskMon *disk, gchar *buf, gint size)
	{
	GkrellmChart	*cp;
	gchar			c, *s;
	size_t			tbl_size;
	gint			len, r_blocks, w_blocks, blocks;

	--size;
	*buf = '\0';
	cp = disk->chart;
	r_blocks = gkrellm_get_current_chartdata(disk->rb_cd);
	w_blocks = gkrellm_get_current_chartdata(disk->wb_cd);
	tbl_size = sizeof(disk_blocks_abbrev) / sizeof(GkrellmSizeAbbrev);
	for (s = text_format_locale; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			blocks = -1;
			if ((c = *(s + 1)) == 'T')
				blocks = r_blocks + w_blocks;
			else if (c == 'M')
				blocks = gkrellm_get_chart_scalemax(cp);
			else if (c == 'r')
				blocks = r_blocks;
			else if (c == 'w')
				blocks = w_blocks;
			else
				{
				*buf = *s;
				if (size > 1)
					{
					*(buf + 1) = *(s + 1);
					++len;
					}
				}
			if (blocks >= 0)
				len = format_size_abbrev(buf, size, (gfloat) blocks,
						&disk_blocks_abbrev[0], tbl_size);
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';	
	}

static void
draw_disk_extra(DiskMon *disk)
	{
	gchar	buf[128];

	if (!disk->extra_info)
		return;
	format_chart_text(disk, buf, sizeof(buf));
	gkrellm_draw_chart_text(disk->chart, style_id, buf);
	}

static void
draw_disk_chart(DiskMon *disk)
	{
	gkrellm_draw_chartdata(disk->chart);
	draw_disk_extra(disk);
	gkrellm_draw_chart_to_screen(disk->chart);
	}

static void
update_disk(void)
	{
	GList			*list;
	DiskMon			*disk;
	GkrellmChart	*cp;

	if (!sys_handles_composite_reset)
		{
		composite_disk->rb = 0;
		composite_disk->wb = 0;
		}
	(*read_disk_data)();
	if (n_disks == 0)
		return;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if ((cp = disk->chart) == NULL)		/* or disk->enabled FALSE */
			continue;
		if (GK.second_tick)
			{
			gkrellm_store_chartdata(cp, 0,
						(gulong) disk->wb, (gulong) disk->rb);
			draw_disk_chart(disk);
			}
		gkrellm_update_krell(cp->panel, KRELL(cp->panel),
					(gulong) (disk->wb + disk->rb));
		gkrellm_draw_panel_layers(cp->panel);
		}
	}


static gint
disk_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GList			*list;
	GkrellmChart	*cp;
	GdkPixmap		*pixmap	= NULL;

	for (list = disk_mon_list; list; list = list->next)
		{
		if ((cp = ((DiskMon *) list->data)->chart) == NULL)
			continue;
		if (widget == cp->drawing_area)
			pixmap = cp->pixmap;
		else if (widget == cp->panel->drawing_area)
			pixmap = cp->panel->pixmap;
		if (pixmap)
			{
			gdk_draw_drawable(widget->window, gkrellm_draw_GC(1), pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
			break;
			}
		}
	return FALSE;
	}

static gint
cb_disk_extra(GtkWidget *widget, GdkEventButton *ev)
	{
	GList	*list;
	DiskMon	*disk;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (!disk->enabled || widget != disk->chart->drawing_area)
			continue;
		if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
			{
			disk->extra_info = !disk->extra_info;
			draw_disk_chart(disk);
			gkrellm_config_modified();
			}
		else if (   ev->button == 3
				 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
				)
			gkrellm_chartconfig_window_create(disk->chart);
		break;
		}
	return FALSE;
	}

static void
setup_disk_scaling(GkrellmChartconfig *cf, GkrellmChart *cp)
	{
	gint	grids, res;

	grids = gkrellm_get_chartconfig_fixed_grids(cf);
	if (!grids)
		grids = FULL_SCALE_GRIDS;
	res = gkrellm_get_chartconfig_grid_resolution(cf);

	KRELL(cp->panel)->full_scale = res * grids / gkrellm_update_HZ();
	}

  /* Destroy everything in a DiskMon structure except for the vbox which
  |  is preserved so disk ordering will be maintained.  Compare this to
  |  destroying an InetMon where really everything is destroyed including
  |  the InetMon structure.  Here the DiskMon structure is not destroyed.
  */
static void
destroy_disk_monitor(DiskMon *disk)
	{
	if (disk->launch_table)
		gtk_widget_destroy(disk->launch_table);
	disk->launch_table = NULL;
	if (disk->launch.button)
		gkrellm_destroy_button(disk->launch.button);
	disk->launch.button = NULL;
	disk->launch.tooltip = NULL;
	gkrellm_dup_string(&disk->launch.command, "");
	gkrellm_dup_string(&disk->launch.tooltip_comment, "");
	gkrellm_chart_destroy(disk->chart);
	disk->chart = NULL;
	disk->enabled = FALSE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 3)
		gkrellm_open_config_window(mon_disk);
	return FALSE;
	}

static void
create_disk_monitor(DiskMon *disk, gint first_create)
	{
	GkrellmPanel	*p;
	GkrellmStyle	*style;
	GkrellmChart	*cp;

	if (first_create)
		{
		disk->chart = gkrellm_chart_new0();
		disk->chart->panel = gkrellm_panel_new0();
		}
	cp = disk->chart;
	p = cp->panel;

	style = gkrellm_panel_style(style_id);
	gkrellm_create_krell(p, gkrellm_krell_panel_piximage(style_id), style);

	gkrellm_chart_create(disk->vbox, mon_disk, cp, &disk->chart_config);
	disk->wb_cd = gkrellm_add_default_chartdata(cp, _("Write Blocks"));
	disk->rb_cd = gkrellm_add_default_chartdata(cp, _("Read Blocks"));
	gkrellm_set_draw_chart_function(cp, draw_disk_chart, disk);

	gkrellm_chartconfig_fixed_grids_connect(cp->config,
				setup_disk_scaling, cp);
	gkrellm_chartconfig_grid_resolution_connect(cp->config,
				setup_disk_scaling, cp);
	gkrellm_chartconfig_grid_resolution_adjustment(cp->config, TRUE,
				0, (gfloat) MIN_GRID_RES, (gfloat) MAX_GRID_RES, 0, 0, 0, 70);
	gkrellm_chartconfig_grid_resolution_label(cp->config,
				units_are_blocks ?
				_("Disk I/O blocks per sec") : _("Disk I/O bytes per sec"));
	if (gkrellm_get_chartconfig_grid_resolution(cp->config) < MIN_GRID_RES)
		gkrellm_set_chartconfig_grid_resolution(cp->config, DEFAULT_GRID_RES);
	gkrellm_alloc_chartdata(cp);

	setup_disk_scaling(cp->config, cp);

	gkrellm_panel_configure(p, disk->name, style);
	gkrellm_panel_create(disk->vbox, mon_disk, p);
	disk->enabled = TRUE;

	if (first_create)
		{
		g_signal_connect(G_OBJECT(cp->drawing_area), "expose_event",
				G_CALLBACK(disk_expose_event), NULL);
		g_signal_connect(G_OBJECT (p->drawing_area), "expose_event",
				G_CALLBACK(disk_expose_event), NULL);

		g_signal_connect(G_OBJECT(cp->drawing_area), "button_press_event",
				G_CALLBACK(cb_disk_extra), NULL);
		g_signal_connect(G_OBJECT(p->drawing_area), "button_press_event",
				G_CALLBACK(cb_panel_press), NULL);
		}
	else
		draw_disk_chart(disk);

	gkrellm_configure_tooltip(p, &disk->launch);
	if (*(disk->launch.command) != '\0')
		disk->launch.button = gkrellm_put_label_in_panel_button(p,
				gkrellm_launch_button_cb, &disk->launch, disk->launch.pad);
	}


static GtkWidget	*disk_vbox;

static void
create_disk(GtkWidget *vbox, gint first_create)
	{
	GList		*list;
	DiskMon		*disk;
	gboolean	any = FALSE;

	ascent = 0;
	disk_vbox = vbox;
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (first_create)
			{
			disk->vbox = gtk_vbox_new(FALSE, 0);
			gtk_box_pack_start(GTK_BOX(vbox), disk->vbox, FALSE, FALSE, 0);
			gtk_widget_show(disk->vbox);
			}
		gkrellm_setup_launcher(NULL, &disk->launch, CHART_PANEL_TYPE, 4);
		if (disk->enabled)
			{
			create_disk_monitor(disk, first_create);
			any = TRUE;
			}
		}
	if (any)
		gkrellm_spacers_show(mon_disk);
	else
		gkrellm_spacers_hide(mon_disk);
	}

  /* Kernel 2.4 will not show a disk until it has I/O, and some systems
  |  may dynamically add a drive.
  */
static void
check_for_new_disks(void)
	{
	GList	*list;
	DiskMon	*disk;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (disk->vbox == NULL)
			{
			disk->vbox = gtk_vbox_new(FALSE, 0);
			gtk_box_pack_start(GTK_BOX(disk_vbox), disk->vbox, FALSE, FALSE,0);
			gtk_widget_show(disk->vbox);
			gkrellm_setup_launcher(NULL, &disk->launch, CHART_PANEL_TYPE, 4);
			}
		}
	}

static void
save_disk_config(FILE *f)
	{
	GList	*list;
	DiskMon	*disk;

	if (n_disks == 0)
		return;
	fprintf(f, "%s assign_method %d\n", DISK_CONFIG_KEYWORD, assign_method);
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		fprintf(f, "%s device %s %d %d %d %d %d %d\n", DISK_CONFIG_KEYWORD,
					disk->name, disk->major, disk->minor, disk->order,
					disk->enabled, disk->extra_info, disk->subdisk);
		if (*(disk->launch.command) != '\0')
			fprintf(f, "%s launch %s %s\n", DISK_CONFIG_KEYWORD,
						disk->name, disk->launch.command);
		if (*(disk->launch.tooltip_comment) != '\0')
			fprintf(f, "%s tooltip_comment %s %s\n", DISK_CONFIG_KEYWORD,
						disk->name, disk->launch.tooltip_comment);
		gkrellm_save_chartconfig(f, disk->chart_config,
					DISK_CONFIG_KEYWORD, disk->name);
		}
	fprintf(f, "%s text_format %s\n", DISK_CONFIG_KEYWORD, text_format);
	}

static void
load_disk_config(gchar *arg)
	{
	DiskMon		*disk;
	gchar		config[32], item[CFG_BUFSIZE],
				name[32], command[CFG_BUFSIZE];
	gint		major, minor, n, extra, order, subdisk = -1;
	static gint	config_assign_method;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (!strcmp(config, "text_format"))
			gkrellm_locale_dup_string(&text_format, item, &text_format_locale);
		else if (!strcmp(config, "assign_method"))
			sscanf(item, "%d", &config_assign_method);
		else if (!strcmp(config, "device"))
			{
			/* Disk config can be invalid (different disk naming scheme)
			|  if user changes kernel version.
			*/
			if (   config_assign_method == assign_method
				&& sscanf(item, "%31s %d %d %d %d %d %d", name,
						&major, &minor, &order, &n, &extra, &subdisk) >= 6
			   )
				{
				/* The list of Linux disks reported can change as disks are
				|  mounted for the first time, so a disk may not be added when
				|  the config is read.  So add them from the config.  Except
				|  for subdisks.
				*/
				if (   (disk = lookup_disk_by_name(name)) == NULL
					&& subdisk == -1
				   )
					disk = add_disk(name, major, minor, order - 1);
				if (disk)
					{
					disk->enabled = n;
					disk->extra_info = extra;
					}
				}
			}
		else if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
			{
			sscanf(item, "%31s %[^\n]", name, command);
			if ((disk = lookup_disk_by_name(name)) != NULL)
				gkrellm_load_chartconfig(&disk->chart_config, command, 2);
			}
		else if (strcmp(config, "launch") == 0)
			{
			sscanf(item, "%31s %[^\n]", name, command);
			if ((disk = lookup_disk_by_name(name)) != NULL)
				disk->launch.command = g_strdup(command);
			}
		else if (strcmp(config, "tooltip_comment") == 0)
			{
			sscanf(item, "%31s %[^\n]", name, command);
			if ((disk = lookup_disk_by_name(name)) != NULL)
				disk->launch.tooltip_comment = g_strdup(command);
			}
		}
	}


/* --------------------------------------------------------------------	*/

enum
	{
	NAME_COLUMN,
	ENABLE_COLUMN,
	DISK_COLUMN,
	N_COLUMNS
	};

static GtkTreeView			*treeview;

static GtkWidget			*launch_vbox,
							*text_format_combo;


static GtkTreeModel *
create_model(void)
	{
	GtkTreeStore	*tree;
	GtkTreeIter		iter, citer;
	GList			*list, *clist;
	DiskMon			*disk;

	tree = gtk_tree_store_new(N_COLUMNS,
				G_TYPE_STRING,
                G_TYPE_BOOLEAN,
				G_TYPE_POINTER);
	for (list = disk_mon_list; list; )
		{
		disk = (DiskMon *) list->data;
		gtk_tree_store_append(tree, &iter, NULL);
		if (list == disk_mon_list)
			gtk_tree_store_set(tree, &iter,
				NAME_COLUMN, _("Composite chart combines data for all disks"),
				ENABLE_COLUMN, disk->enabled,
				DISK_COLUMN, disk,
				-1);
		else
			gtk_tree_store_set(tree, &iter,
					NAME_COLUMN, disk->name,
					ENABLE_COLUMN, disk->enabled,
					DISK_COLUMN, disk,
					-1);
		for (clist = list->next; clist; clist = clist->next)
			{
			disk = (DiskMon *) clist->data;
			if (disk->subdisk == -1)
				break;
			gtk_tree_store_append(tree, &citer, &iter);
			gtk_tree_store_set(tree, &citer,
					NAME_COLUMN, disk->name,
					ENABLE_COLUMN, disk->enabled,
					DISK_COLUMN, disk,
					-1);
			}
		list = clist;
		}

	return GTK_TREE_MODEL(tree);
	}

static void
cb_launch_entry(GtkWidget *widget, DiskMon *disk)
	{
	if (disk->enabled)
		gkrellm_apply_launcher(&disk->launch_entry, &disk->tooltip_entry,
				disk->chart->panel, &disk->launch, gkrellm_launch_button_cb);
	}

static void
add_launch_entry(GtkWidget *vbox, DiskMon *disk)
	{
	disk->launch_table = gkrellm_gtk_launcher_table_new(vbox, 1);
	gkrellm_gtk_config_launcher(disk->launch_table, 0,  &disk->launch_entry,
				&disk->tooltip_entry, disk->name,
				&disk->launch);
	g_signal_connect(G_OBJECT(disk->launch_entry), "changed",
				G_CALLBACK(cb_launch_entry), disk);
	g_signal_connect(G_OBJECT(disk->tooltip_entry), "changed",
				G_CALLBACK(cb_launch_entry), disk);
	gtk_widget_show_all(disk->launch_table);
    }

static void
cb_enable(GtkCellRendererText *cell, gchar *path_string, gpointer data)
	{
	GtkTreeModel	*model;
	GtkTreeIter		iter;
	GtkTreePath		*path;
	GList			*list;
	DiskMon			*disk;
	gboolean		enabled;

	model = GTK_TREE_MODEL(data);
	path = gtk_tree_path_new_from_string(path_string);
	gtk_tree_model_get_iter(model, &iter, path);
	gtk_tree_path_free(path);

	gtk_tree_model_get(model, &iter,
				ENABLE_COLUMN, &enabled,
				DISK_COLUMN, &disk,
				-1);
	gtk_tree_store_set(GTK_TREE_STORE(model), &iter,
				ENABLE_COLUMN, !enabled, 
				-1);
	if (enabled)
		destroy_disk_monitor(disk);
	else
		{
		create_disk_monitor(disk, TRUE);
		add_launch_entry(launch_vbox, disk);
		}
	disk->enabled = !enabled;
	enabled = FALSE;
	for (list = disk_mon_list; list; list = list->next)
		if (((DiskMon *) list->data)->enabled)
			enabled = TRUE;
	if (enabled)
		gkrellm_spacers_show(mon_disk);
	else
		gkrellm_spacers_hide(mon_disk);
	}

static void
cb_text_format(GtkWidget *widget, gpointer data)
	{
	GList	*list;
	gchar	*s;

	s = gkrellm_gtk_entry_get_text(&(GTK_COMBO(text_format_combo)->entry));
	gkrellm_locale_dup_string(&text_format, s, &text_format_locale);
	for (list = disk_mon_list; list; list = list->next)
		draw_disk_chart((DiskMon *) list->data);
	}


#define	DEFAULT_TEXT_FORMAT	"$T"

static gchar	*disk_info_text[] =
{
N_("<h>Chart Labels\n"),
N_("Substitution variables for the format string for chart labels:\n"),
N_("\t$M    maximum chart value\n"),
N_("\t$T    total read blocks + write blocks\n"),
N_("\t$r    read blocks\n"),
N_("\t$w    write blocks\n"),
};

static void
create_disk_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1, *hbox;
	GtkWidget		*text;
	GtkWidget		*scrolled;
	GtkTreeModel	*model;
	GtkCellRenderer	*renderer;
	GList			*list;
	DiskMon			*disk;
	gint			i;

	check_for_new_disks();
	if (n_disks == 0)
		return;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* -- Options tab */
	vbox = gkrellm_gtk_framed_notebook_page(tabs, _("Options"));
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

	model = create_model();

	treeview = GTK_TREE_VIEW(gtk_tree_view_new_with_model(model));
	gtk_tree_view_set_rules_hint(treeview, TRUE);

	renderer = gtk_cell_renderer_toggle_new();
	gtk_tree_view_insert_column_with_attributes(treeview, -1, _("Enable"),
				renderer,
				"active", ENABLE_COLUMN,
				NULL);
	g_signal_connect(G_OBJECT(renderer), "toggled",
				G_CALLBACK(cb_enable), model);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(treeview, -1, _("Disk"),
				renderer,
				"text", NAME_COLUMN,
				NULL);

	gtk_container_add(GTK_CONTAINER(scrolled), GTK_WIDGET(treeview));
	g_object_unref(G_OBJECT(model));


/* -- Setup tab */
	vbox = gkrellm_gtk_framed_notebook_page(tabs, _("Setup"));
	vbox1 = gkrellm_gtk_framed_vbox(vbox, _("Format String for Chart Labels"),
			 4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
	text_format_combo = gtk_combo_new();
	gtk_widget_set_size_request (GTK_WIDGET(text_format_combo), 350, -1);
	gtk_box_pack_start(GTK_BOX(hbox), text_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, text_format);
	list = g_list_append(list, DEFAULT_TEXT_FORMAT);
	list = g_list_append(list, "\\c\\f$M\\n$T");
	list = g_list_append(list, "\\c\\f$M\\b$T");
	list = g_list_append(list,
					_("\\f\\ww\\r\\f$M\\D2\\f\\ar\\. $r\\D1\\f\\aw\\. $w"));
	gtk_combo_set_popdown_strings(GTK_COMBO(text_format_combo), list);
	g_list_free(list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(text_format_combo)->entry),
			text_format);
	g_signal_connect(G_OBJECT(GTK_COMBO(text_format_combo)->entry), "changed",
			G_CALLBACK(cb_text_format), NULL);

	vbox = gkrellm_gtk_framed_vbox_end(vbox, _("Launch Commands"),
						4, TRUE, 0, 2);
	launch_vbox = gkrellm_gtk_scrolled_vbox(vbox, NULL,
						GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_show(launch_vbox);
	gtk_widget_realize(launch_vbox);
	for (i = 0, list = disk_mon_list; list; list = list->next, ++i)
		{
		disk = (DiskMon *) list->data;
		if (disk->enabled)
			add_launch_entry(launch_vbox, disk);
		}

/* --Info tab */
	vbox = gkrellm_gtk_framed_notebook_page(tabs, _("Info"));
	text = gkrellm_gtk_scrolled_text_view(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(disk_info_text)/sizeof(gchar *); ++i)
		gkrellm_gtk_text_view_append(text, _(disk_info_text[i]));
	}

static GkrellmMonitor	monitor_disk =
	{
	N_("Disk"),			/* Name, for config tab.	*/
	MON_DISK,			/* Id,  0 if a plugin		*/
	create_disk,		/* The create function		*/
	update_disk,		/* The update function		*/
	create_disk_tab,	/* The config tab create function	*/
	NULL,				/* Instant apply */

	save_disk_config,	/* Save user conifg			*/
	load_disk_config,	/* Load user config			*/
	DISK_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

GkrellmMonitor *
init_disk_monitor(void)
	{
	DiskMon	*disk;
	GList	*list;

	monitor_disk.name = _(monitor_disk.name);

	disk = new_disk(_("Disk"));
	disk_mon_list = g_list_append(disk_mon_list, disk);

	composite_disk = disk;
	gkrellm_locale_dup_string(&text_format, DEFAULT_TEXT_FORMAT,
				&text_format_locale);
	mon_disk = &monitor_disk;

	if (setup_disk_interface())
		{
		for (list = disk_mon_list->next; list; list = list->next)
			{
			disk = (DiskMon *) list->data;
			disk->enabled = FALSE;
			disk->extra_info = TRUE;
			}
		if (n_disks > 0)
			composite_disk->enabled = TRUE;
		style_id = gkrellm_add_chart_style(&monitor_disk, DISK_STYLE_NAME);
		return &monitor_disk;
		}
	return NULL;
	}
