/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"


/* ===============  string utility functions ================= */

gint
strcmp_basename(gchar *str1, gchar *str2)
	{
	gchar	*s1, *s2;

	if ((s1 = strrchr(str1, G_DIR_SEPARATOR)) == NULL)
		s1 = str1;
	else
		++s1;
	if ((s2 = strrchr(str2, G_DIR_SEPARATOR)) == NULL)
		s2 = str2;
	else
		++s2;
	return strcmp(s1, s2);
	}

  /* Obsolete */
gboolean
dup_string(gchar **dst, gchar *src)
	{
	return gkrellm_dup_string(dst, src);
	}

gboolean
gkrellm_dup_string(gchar **dst, gchar *src)
	{
	if (!dst || (!*dst && !src))
		return FALSE;
	if (*dst)
		{
		if (src && !strcmp(*dst, src))
			return FALSE;
		g_free(*dst);
		}
	*dst = g_strdup(src);
	return TRUE;
	}


  /* Cut out an optionally quoted string.  This is destructive to the src.
  */
gchar *
cut_quoted_string(gchar *src, gchar **endptr)
	{
	gchar	*s;

	while (*src == ' ' || *src == '\t')
		++src;
	if (*src == '"')
		{
		s = strchr(++src, '"');
		if (s == NULL)
			{
			if (endptr)
				*endptr = src;
			printf(_("Unterminated quote\n"));
			return NULL;
			}
		*s = '\0';
		if (endptr)
			*endptr = s + 1;
		}
	else
		{
		for (s = src; *s != '\0' && *s != ' ' && *s != '\t'; ++s)
			;
		if (endptr)
			*endptr = *s ? s + 1 : s;
		*s = '\0';
		}
	return src;
	}

  /* If there is a line in the gstring ('\n' delimited) copy it to the
  |  line buffer including the newline and erase it from the gstring.
  */
gboolean
getline_from_gstring(GString **gstring, gchar *line, gint size)
	{
	GString	*gstr	= *gstring;
	gchar   *s;
	gint    len, n;

	if (gstr && gstr->str && (s = strchr(gstr->str, '\n')) != NULL)
		{
		n = len = s - gstr->str + 1;
		if (n >= size)
			n = size - 1;				/* Truncate the line to fit */
		strncpy(line, gstr->str, n);
		line[n] = '\0';
		*gstring = g_string_erase(gstr, 0, len);
		return TRUE;
		}
	return FALSE;
	}

/* ===============  list utility functions ================= */

void
free_glist_and_data(GList **list_head)
	{
	GList	*list;

	if (*list_head == NULL)
		return;

	/* could use g_list_foreach(*list_head, (G_FUNC)g_free, NULL);
	*/
	for (list = *list_head; list; list = list->next)
		if (list->data)
			g_free(list->data);
	g_list_free(*list_head);
	*list_head = NULL;
	}

GList *
basename_in_list(GList *list, gchar *name)
	{
	gchar	*s, *slash;

	for ( ; list; list = list->next)
		{
		s = (gchar *) list->data;
		if (!s)
			continue;
		if (*s == '.')
			++s;
		if ((slash = strrchr(s, G_DIR_SEPARATOR)) != NULL)
			s = slash + 1;
		if (! strcmp(s, name))
			return list;
		}
	return NULL;
	}

GList *
string_in_list(GList *list, gchar *s)
	{
	if (!s)
		return NULL;
	for ( ; list; list = list->next)
		{
		if (!strcmp((gchar *) list->data, s))
			return list;
		}
	return NULL;
	}

gint
string_position_in_list(GList *list, gchar *s)
	{
	gint	i, n = -1;

	if (!s)
		return -1;
	for (i = 0 ; list; list = list->next, ++i)
		{
		if (!strcmp((gchar *) list->data, s))
			{
			n = i;
			break;
			}
		}
	return n;
	}



/* ===============  file utility functions ================= */
gchar *
gkrellm_homedir(void)
	{
	gchar	*homedir;

	homedir = (gchar *) g_get_home_dir();
	if (!homedir)
		homedir = ".";
	return homedir;
	}

gboolean
make_home_subdir(gchar *subdir, gchar **path)
	{
	gchar	*dir;
	gint	result	= FALSE;

	dir = g_build_path(G_DIR_SEPARATOR_S, gkrellm_homedir(), subdir, NULL);
	if (!g_file_test(dir, G_FILE_TEST_IS_DIR))
		{
		if (mkdir(dir, 0755) < 0)
			printf(_("Cannot create directory: %s\n"), dir);
		else
			result = TRUE;
		}
	if (path)
		*path = dir;
	else
		g_free(dir);
	return result;
	}

void
dup_home_pathname(gchar **path, gchar *subdir, gchar *fname)
	{
	if (*path)
		g_free(*path);
	if (subdir)
		*path = g_build_path(G_DIR_SEPARATOR_S, gkrellm_homedir(),
					subdir, fname, NULL);
	else
		*path = g_build_path(G_DIR_SEPARATOR_S, gkrellm_homedir(), fname, NULL);
	}


/* ===============  GtkWidget utility functions ================= */

gchar *
gkrellm_gtk_entry_get_text(GtkWidget **entry)
	{
	gchar	*s = "";

	if (*entry)
		s = (gchar *) gtk_entry_get_text(GTK_ENTRY(*entry));
	while (*s == ' ' || *s == '\t')
		++s;
	return s;
	}


/* ===============  Miscellaneous utility functions ================= */

  /* Print a size, abbreviating it to kilo, mega, or giga units depending
  |  on its magnitude.
  |  An aside:  Memory capacities are traditionally reported in binary
  |  units (Kib, Mib, etc) while just about everything else should be
  |  reported in decimal units (KB, MB, etc).  This includes transfer
  |  rates, and disk capacities, contrary to what many people think.
  |  Take a look at http://www.pcguide.com/intro/fun/bindec.htm
  */
gint
format_size_abbrev(gchar *buf, size_t buflen, gfloat size,
		GkrellmSizeAbbrev *tbl, size_t tbl_size)
	{
	gint	i;

	for (i = 0; i < tbl_size - 1; ++i)
		if (size < tbl[i].limit)
			break;
	return snprintf(buf, buflen, tbl[i].format, size / tbl[i].divisor);
	}

