/* GKrellM
|  Copyright (C) 1999-2003 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "gkrellm.h"
#include "gkrellm-private.h"
#include "gkrellm-sysdeps.h"

#if !defined(WIN32)
#include <sys/socket.h>
#include <utime.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#else
#define uint32_t u_long
#include <winsock2.h>
#endif

#include <errno.h>



#if defined(__linux__)
#if defined(__GLIBC__) && ((__GLIBC__>2)||(__GLIBC__==2 && __GLIBC_MINOR__>=1))
#define HAVE_GETADDRINFO	1
#endif
#endif

#if defined(__FreeBSD__)
#if __FreeBSD_version >= 400000
#define HAVE_GETADDRINFO	1
#endif
#endif

#if defined(__NetBSD__) || defined(__OpenBSD__)
#define HAVE_GETADDRINFO	1
#endif

#if defined(__solaris__)
# include <netconfig.h>
# if defined(NC_INET6)
#  define HAVE_GETADDRINFO	1
# endif
#endif


typedef struct
	{
	gchar	*key;
	void	(*func)();
	}
	KeyTable;

static gint		input_id;
static gchar	server_buf[4097];
static gint		buf_index;



/* ================================================================= */
/* CPU */

typedef struct
	{
	gint	instance;
	gulong	user,
			nice,
			sys,
			idle;
	}
	Cpu;

static GList	*cpu_list,
				*instance_list;
static gint		n_cpus = 1;
static gboolean	nice_time_supported = TRUE;

static void
client_cpu_line_from_server(gchar *line)
	{
	GList	*list;
	Cpu		*cpu = NULL;
	gint	n;
	gulong	user, nice, sys, idle;

	sscanf(line, "%d %lu %lu %lu %lu", &n, &user, &nice, &sys, &idle);
	for (list = cpu_list; list; list = list->next)
		{
		cpu = (Cpu *) list->data;
		if (cpu->instance == n)
			break;
		}
	if (list)
		{
		cpu->user = user;
		cpu->nice = nice;
		cpu->sys = sys;
		cpu->idle = idle;
		}
	}

static void
cpu_read_data(void)
	{
	Cpu		*cpu;
	GList 	*list;

	for (list = cpu_list; list; list = list->next)
		{
		cpu = (Cpu *) list->data;
		gkrellm_cpu_assign_data(cpu->instance, cpu->user, cpu->nice,
					cpu->sys, cpu->idle);
		}
	}

static void
client_sys_cpu_init(void)
	{
	GList	*list;
	Cpu		*cpu;
	gint	n;

	/* Do initialization based on info received in client_cpu_setup().  Here
	|  we need to get the same effective work done as would be done in the
	|  sysdep gkrellm_sys_cpu_init() when not running in client mode.
	*/
	for (n = 0; n < n_cpus; ++ n)
		{
		cpu = g_new0(Cpu, 1);
		if (instance_list && (list = g_list_nth(instance_list, n)) != NULL)
			{
			cpu->instance = GPOINTER_TO_INT(list->data);
			gkrellm_cpu_add_instance(cpu->instance);
			}
		else
			cpu->instance = n;
		cpu_list = g_list_append(cpu_list, cpu);
		}
	gkrellm_cpu_set_number_of_cpus(n_cpus);
	if (!nice_time_supported)
		gkrellm_cpu_nice_time_unsupported();

	/* Diverting the cpu_read_data function in cpu.c causes the cpu monitor
	|  to not call the gkrellm_sys_cpu_init() sysdep function and also to call
	|  our client cpu_read_data function instead of gkrellm_sys_cpu_read_data()
	*/
	gkrellm_cpu_client_divert(cpu_read_data);
	}

static void
client_cpu_setup(gchar *line)
	{
	gint	instance;

	if (!strncmp(line, "n_cpus", 6))
		sscanf(line, "%*s %d", &n_cpus);
	else if (!strncmp(line, "cpu_instance", 12))
		{
		sscanf(line, "%*s %d", &instance);
		instance_list = g_list_append(instance_list,
				GINT_TO_POINTER(instance));
		}
	else if (!strcmp(line, "nice_time_unsupported"))
		nice_time_supported = FALSE;
	}

/* ================================================================= */
static struct
	{
	gint	n_processes,
			n_running,
			n_users;
	gulong	n_forks;
	gfloat	load;
	}
	proc;

static void
client_proc_line_from_server(gchar *line)
	{
	sscanf(line, "%d %d %lu %f %d", &proc.n_processes, &proc.n_running,
			&proc.n_forks, &proc.load, &proc.n_users);
	}

static void
read_proc_data(void)
	{
	gkrellm_proc_assign_data(proc.n_processes, proc.n_running,
				proc.n_forks, proc.load);
	}

static void
read_user_data(void)
	{
	gkrellm_proc_assign_users(proc.n_users);
	}

static void
client_sys_proc_init(void)
	{
	gkrellm_proc_client_divert(read_proc_data, read_user_data);
	}

/* ================================================================= */
typedef struct
	{
	gchar	*name;
	gchar	*subdisk_parent;
	guint64	rblk,
			wblk;
	}
	DiskData;

static GList	*disk_list;

static gboolean	units_are_blocks;

static void
client_disk_line_from_server(gchar *line)
	{
	DiskData	*disk = NULL;
	GList		*list;
	gchar		name[16], s1[32], s2[32], s3[32];
	guint64		rblk, wblk;
	gint		n;

	n = sscanf(line, "%16s %32s %32s %32s", name, s1, s2, s3);
	if (n == 4)
		{
		rblk = strtoull(s2, NULL, 0);
		wblk = strtoull(s3, NULL, 0);
		}
	else if (n == 3)
		{
		rblk = strtoull(s1, NULL, 0);
		wblk = strtoull(s2, NULL, 0);
		s1[0] = '\0';
		}
	else
		return;
	for (list = disk_list; list; list = list->next)
		{
		disk = (DiskData *) list->data;
		if (!strcmp(disk->name, name))
			break;
		}
	if (!list)
		{
		disk = g_new0(DiskData, 1);
		disk->name = g_strdup(name);
		if (s1[0])		/* I expect server to send in order */
			disk->subdisk_parent = g_strdup(s1);
		disk_list = g_list_append(disk_list, disk);
		}
	if (disk)
		{
		disk->rblk = rblk;
		disk->wblk = wblk;
		}
	}

static void
read_disk_data(void)
	{
	GList		*list;
	DiskData	*disk;

	for (list = disk_list; list; list = list->next)
		{
		disk = (DiskData *) list->data;
		if (disk->subdisk_parent)
			gkrellm_disk_subdisk_assign_data_by_name(disk->name,
					disk->subdisk_parent, disk->rblk, disk->wblk);
		else
			gkrellm_disk_assign_data_by_name(disk->name,
					disk->rblk, disk->wblk);
		}
	}

static gint
order_from_name(gchar *name)
	{
	return -1;
	}

static void
client_sys_disk_init(void)
	{
	gkrellm_disk_client_divert(read_disk_data, NULL, order_from_name);
	if (units_are_blocks)
		gkrellm_disk_units_are_blocks();

	/* Disk monitor config needs to know will be using assign by name
	*/
	gkrellm_disk_assign_data_by_name(NULL, 0, 0);
	}

static void
client_disk_setup(gchar *line)
	{
	if (!strcmp(line, "units_are_blocks"))
		units_are_blocks = TRUE;
	}

/* ================================================================= */
typedef struct
	{
	gchar		*name;
	gboolean	up_event,
				down_event;
	gulong		rx,
				tx;
	}
	NetData;

GList	*net_list;

static gboolean	net_server_use_routed;

static void
client_net_line_from_server(gchar *line)
	{
	GList		*list;
	NetData		*net;
	gchar		name[32];
	gulong		rx, tx;

	sscanf(line, "%32s %lu %lu", name, &rx, &tx);
	for (list = net_list; list; list = list->next)
		{
		net = (NetData *) list->data;
		if (!strcmp(name, net->name))
			{
			net->rx = rx;
			net->tx = tx;
			break;
			}
		}
	if (!list)
		{
		net = g_new0(NetData, 1);
		net->name = g_strdup(name);
		net_list = g_list_append(net_list, net);
		net->rx = rx;
		net->tx = tx;
		}
	}

static void
client_net_routed_line_from_server(gchar *line)
	{
	GList		*list;
	NetData		*net = NULL;
	gchar		name[32];
	gboolean	routed;

	sscanf(line, "%32s %d", name, &routed);
	for (list = net_list; list; list = list->next)
		{
		net = (NetData *) list->data;
		if (!strcmp(name, net->name))
			break;
		}
	if (!list)
		{
		net = g_new0(NetData, 1);
		net->name = g_strdup(name);
		net_list = g_list_append(net_list, net);
		}
	if (net)
		{
		if (routed)
			net->up_event = TRUE;
		else
			net->down_event = TRUE;
		}
	}

static void
check_net_routes(void)
	{
	GList	*list;
	NetData	*net;

	for (list = net_list; list; list = list->next)
		{
		net = (NetData *) list->data;
		if (net->up_event || net->down_event)
			gkrellm_net_routed_event(net->name, net->up_event);
		net->up_event = net->down_event = FALSE;
		}
	}

static void
read_net_data(void)
	{
	GList	*list;
	NetData	*net;

	for (list = net_list; list; list = list->next)
		{
		net = (NetData *) list->data;
		gkrellm_net_assign_data(net->name, net->rx, net->tx);
		}
	}

  /* gkrellmd to gkrellm server/client interface always uses the net routed
  |  mode regardless if the server uses routed in its sysdep code.
  */
static void
client_sys_net_init(void)
	{
	gkrellm_net_client_divert(read_net_data, check_net_routes, NULL);
	gkrellm_net_use_routed(net_server_use_routed);
	}

static void
client_net_setup(gchar *line)
	{
	/* This is the server sysdep net_use_routed value.  The client <-> server
	|  link always uses routed mode.
	*/
	if (!strcmp(line, "net_use_routed"))
		net_server_use_routed = TRUE;
	}


/* ================================================================= */
#include "../src/inet.h"

static GList	*inet_list;

static gboolean	inet_unsupported;

static void
client_inet_line_from_server(gchar *line)
	{
	GList		*list;
	ActiveTCP	tcp, *t;
	gchar		*ap, *aap;
#if defined(INET6) && defined(HAVE_GETADDRINFO)
	struct addrinfo	hints, *res;
	gchar		buf[NI_MAXHOST];
#else
	gchar		buf[128];
#endif
	gint		n, slen;

	if (*(line + 1) == '0')
		{
		n = sscanf(line + 3, "%x %128[^:]:%x", &tcp.local_port,
			   buf, &tcp.remote_port);
		if (n != 3 || inet_aton(buf, &tcp.remote_addr) == 0)
			return;
		tcp.family = AF_INET;
		}
#if defined(INET6) && defined(HAVE_GETADDRINFO)
	else if (*(line + 1) == '6')
		{
#define STR(x)	#x
#define XSTR(x)	STR(x)
		n = sscanf(line + 3, "%x [%" XSTR(NI_MAXHOST) "[^]]]:%x",
			   &tcp.local_port, buf, &tcp.remote_port);
		if (n != 3)
			return;
		memset(&hints, 0, sizeof(hints));
		hints.ai_family = AF_INET6;
		hints.ai_socktype = SOCK_STREAM;
		hints.ai_flags = AI_PASSIVE | AI_NUMERICHOST;
		if (getaddrinfo(buf, NULL, &hints, &res) != 0)
			return;
		memcpy(&tcp.remote_addr6,
		       &((struct sockaddr_in6 *)res->ai_addr)->sin6_addr,
		       sizeof(struct in6_addr));
		freeaddrinfo(res);
		tcp.family = AF_INET6;
		}
#endif
	if (*line == '+')
		{
		t = g_new0(ActiveTCP, 1);
		*t = tcp;
		inet_list = g_list_append(inet_list, t);
		}
	else if (*line == '-')
		{
		for (list = inet_list; list; list = list->next)
			{
			t = (ActiveTCP *) list->data;
			if (t->family == AF_INET)
				{
				ap = (gchar *) &t->remote_addr;
				aap = (gchar *) &tcp.remote_addr;
				slen = sizeof(struct in_addr);
				}
#if defined(INET6)
			else if (t->family == AF_INET6)
				{
				ap = (gchar *) &t->remote_addr6;
				aap = (gchar *) &tcp.remote_addr6;
				slen = sizeof(struct in6_addr);
				}
#endif
			else
				return;
			if (   memcmp(aap, ap, slen) == 0
				&& tcp.remote_port == t->remote_port
				&& tcp.local_port == t->local_port
			   )
				{
				g_free(t);
				inet_list = g_list_remove_link(inet_list, list);
				break;
				}
			}
		}
	}

static void
read_tcp_data(void)
	{
	GList		*list;
	ActiveTCP	*tcp;

	for (list = inet_list; list; list = list->next)
		{
		tcp = (ActiveTCP *) list->data;
		gkrellm_inet_log_tcp_port_data(tcp);
		}
	}

static void
client_sys_inet_init(void)
	{
	if (inet_unsupported)
		return;
	gkrellm_inet_client_divert(read_tcp_data);
	}

static void
client_inet_setup(gchar *line)
	{
	if (!strcmp(line, "inet_unsupported"))
		inet_unsupported = TRUE;
	}

/* ================================================================= */
struct
	{
	guint64		total,
				used,
				free,
				shared,
				buffers,
				cached;
	guint64		swap_total,
				swap_used;
	gulong		swap_in,
				swap_out;
	}
	mem;

static void
client_mem_line_from_server(gchar *line)
	{
	gchar	*fmt;

/* Solaris 8 doesn't grok %L and I've seen a bug report about a FreeBSD
|  %ll bug: http://www.freebsd.org/cgi/query-pr.cgi?pr=34682
*/
#if defined(__solaris__)
	fmt = "%lld %lld %lld %lld %lld %lld";
#else
	fmt = "%Lu %Lu %Lu %Lu %Lu %Lu";
#endif
	sscanf(line, fmt,
			&mem.total, &mem.used, &mem.free,
			&mem.shared, &mem.buffers, &mem.cached);
	}

static void
client_swap_line_from_server(gchar *line)
	{
	gchar	*fmt;

#if defined(__solaris__)
	fmt = "%lld %lld %lu %lu";
#else
	fmt = "%Lu %Lu %lu %lu";
#endif
	sscanf(line, "%Lu %Lu %lu %lu",
			&mem.swap_total, &mem.swap_used,
			&mem.swap_in, &mem.swap_out);
	}

static void
read_mem_data(void)
	{
	gkrellm_mem_assign_data(mem.total, mem.used, mem.free, mem.shared,
				mem.buffers, mem.cached);
	}

static void
read_swap_data(void)
	{
	gkrellm_swap_assign_data(mem.swap_total, mem.swap_used,
				mem.swap_in, mem.swap_out);
	}

static void
client_sys_mem_init(void)
	{
	gkrellm_mem_client_divert(read_mem_data, read_swap_data);
	}

/* ================================================================= */
typedef struct
	{
	gchar	*directory,
			*device,
			*type;
	gulong	blocks,
			bavail,
			bfree,
			bsize;
	}
	Mount;

static GList	*mounts_list,
				*fstab_list;

static gboolean fstab_modified,
				mounting_unsupported;

static void
client_fstab_line_from_server(gchar *line)
	{
	GList	*list;
	Mount	*m;
	gchar	dir[128], dev[64], type[64];

	if (!strcmp(line, ".clear"))
		{
		for (list = fstab_list; list; list = list->next)
			{
			m = (Mount *) list->data;
			g_free(m->directory);
			g_free(m->device);
			g_free(m->type);
			g_free(m);
			}
		g_list_free(fstab_list);
		fstab_list = NULL;
		fstab_modified = TRUE;
		}
	else
		{
		m = g_new0(Mount, 1);
		sscanf(line, "%128s %64s %64s", dir, dev, type);
		m->directory = g_strdup(dir);
		m->device = g_strdup(dev);
		m->type = g_strdup(type);
		fstab_list = g_list_append(fstab_list, m);
		}
	}

static void
client_mounts_line_from_server(gchar *line)
	{
	GList	*list;
	Mount	*m;
	gchar	dir[128], dev[64], type[64];

	if (!strcmp(line, ".clear"))
		{
		for (list = mounts_list; list; list = list->next)
			{
			m = (Mount *) list->data;
			g_free(m->directory);
			g_free(m->device);
			g_free(m->type);
			g_free(m);
			}
		g_list_free(mounts_list);
		mounts_list = NULL;
		}
	else
		{
		m = g_new0(Mount, 1);
		sscanf(line, "%128s %64s %64s %lu %lu %lu %lu", dir, dev, type,
				&m->blocks, &m->bavail, &m->bfree, &m->bsize);
		m->directory = g_strdup(dir);
		m->device = g_strdup(dev);
		m->type = g_strdup(type);
		mounts_list = g_list_append(mounts_list, m);
		}
	}

static void
client_fs_line_from_server(gchar *line)
	{
	GList	*list;
	Mount	*m;
	gchar	dir[128], dev[64];
	gulong	blocks, bavail, bfree, bsize;

	sscanf(line, "%128s %64s %lu %lu %lu %lu", dir, dev,
				&blocks, &bavail, &bfree, &bsize);
	for (list = mounts_list; list; list = list->next)
		{
		m = (Mount *) list->data;
		if (!strcmp(m->directory, dir) && !strcmp(m->device, dev))
			{
			m->blocks = blocks;
			m->bavail = bavail;
			m->bfree = bfree;
			m->bsize = bsize;
			break;
			}
		}
	}

static void
get_fsusage(gpointer fs, gchar *dir)
	{
	GList	*list;
	Mount	*m;

	for (list = mounts_list; list; list = list->next)
		{
		m = (Mount *) list->data;
		if (!strcmp(m->directory, dir))
			{
			gkrellm_fs_assign_fsusage_data(fs, m->blocks, m->bavail,
						m->bfree, m->bsize);
			break;
			}
		}
	}

static void
get_mounts_list(void)
	{
	GList	*list;
	Mount	*m;

	for (list = mounts_list; list; list = list->next)
		{
		m = (Mount *) list->data;
		gkrellm_fs_add_to_mounts_list(m->directory, m->device, m->type);
		}
	}

static void
get_fstab_list(void)
	{
	GList	*list;
	Mount	*m;

	for (list = fstab_list; list; list = list->next)
		{
		m = (Mount *) list->data;
		gkrellm_fs_add_to_fstab_list(m->directory, m->device,
				m->type, "none" /* options NA since no mounting */ );
		}
	fstab_modified = FALSE;
	}

static gboolean
get_fstab_modified(void)
	{
	return fstab_modified;
	}

static void
client_sys_fs_init(void)
	{
	gkrellm_fs_client_divert(get_fsusage, get_mounts_list,
				get_fstab_list, get_fstab_modified);
	if (mounting_unsupported)
		gkrellm_fs_mounting_unsupported();
	}

static void
client_fs_setup(gchar *line)
	{
	if (!strcmp(line, "mounting_unsupported"))
		mounting_unsupported = TRUE;
	}

/* ================================================================= */
/* Need to handle more than one battery. */

struct
	{
    gboolean	battery_is_available,
				ac_is_on_line,
				battery_is_charging;
	gint		battery_percentage;
	gint		battery_time_left;
	}
	bat;


static void
client_battery_line_from_server(gchar *line)
	{
	sscanf(line, "%d %d %d %d %d", &bat.battery_is_available,
				&bat.ac_is_on_line, &bat.battery_is_charging,
				&bat.battery_percentage, &bat.battery_time_left);
	}

static void
read_battery_data(void)
	{
	gkrellm_battery_assign_data(0, bat.battery_is_available,
				bat.ac_is_on_line, bat.battery_is_charging,
				bat.battery_percentage, bat.battery_time_left);
	}

static void
client_sys_battery_init(void)
	{
	if (!bat.battery_is_available)
		return;
	gkrellm_battery_client_divert(read_battery_data);
	}

static void
client_battery_setup(gchar *line)
	{
	if (   !strcmp(line, "apm_available")
		|| !strcmp(line, "battery_available")
	   )
		bat.battery_is_available = TRUE;
	}

/* ================================================================= */
typedef struct
	{
	gint		type;

	gchar		*basename;
	gint		id;
	gint		iodev;
	gint		inter;

	gchar		*vref;
	gchar		*default_label;

	gfloat		factor;
	gfloat		offset;
	gfloat		raw_value;
	}
	Sensor;

static GList	*sensors_list;


static void
client_sensors_line_from_server(gchar *line)
	{
	GList	*list;
	Sensor	s, *sensor;
	gchar	basename[128];

	sscanf(line, "%d \"%128[^\"]\" %d %d %d %f",
			&s.type, basename, &s.id, &s.iodev, &s.inter, &s.raw_value);
	for (list = sensors_list; list; list = list->next)
		{
		sensor = (Sensor *) list->data;
		if (   sensor->type == s.type && !strcmp(sensor->basename, basename)
			&& sensor->id == s.id && sensor->iodev == s.iodev
			&& sensor->inter == s.inter
		   )
			{
			sensor->raw_value = s.raw_value;
			break;
			}
		}
	}

static gboolean
get_temperature(gchar *path, gint id, gint iodev, gint inter, gfloat *value)
	{
	GList	*list;
	Sensor	*s;

	for (list = sensors_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (   s->type == SENSOR_TEMPERATURE && !strcmp(s->basename, path)
			&& s->id == id && s->iodev == iodev && s->inter == inter
		   )
			{
			*value = s->raw_value;
			return TRUE;
			}
		}
	return FALSE;
	}

static gboolean
get_fan(gchar *path, gint id, gint iodev, gint inter, gfloat *value)
	{
	GList	*list;
	Sensor	*s;

	for (list = sensors_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (   s->type == SENSOR_FAN && !strcmp(s->basename, path)
			&& s->id == id && s->iodev == iodev && s->inter == inter
		   )
			{
			*value = s->raw_value;
			return TRUE;
			}
		}
	return FALSE;
	}

static gboolean
get_voltage(gchar *path, gint id, gint iodev, gint inter, gfloat *value)
	{
	GList	*list;
	Sensor	*s;

	for (list = sensors_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (   s->type == SENSOR_VOLTAGE && !strcmp(s->basename, path)
			&& s->id == id && s->iodev == iodev && s->inter == inter
		   )
			{
			*value = s->raw_value;
			return TRUE;
			}
		}
	return FALSE;
	}

static void
client_sys_sensors_init(void)
	{
	GList	*list;
	Sensor	*s;

	if (!sensors_list)
		return;

	gkrellm_sensors_client_divert(get_temperature, get_fan, get_voltage);
	for (list = sensors_list; list; list = list->next)
		{
		s = (Sensor *) list->data;

		/* The sysdep code in the server may be using the dir arg to get sensor
		|  values, but dir is no longer needed.
		*/
		gkrellm_sensors_add_sensor(s->type, NULL, s->basename,
					s->id, s->iodev, s->inter,
					s->factor, s->offset,
					s->vref, s->default_label);
		}
	}

static void
client_sensors_setup(gchar *line)
	{
	Sensor	*s;
	gchar	basename[128], vref[32], default_label[32];

	s = g_new0(Sensor, 1);
	if (sscanf(line,
			"%d \"%128[^\"]\" %d %d %d %f %f \"%32[^\"]\" \"%32[^\"]\"",
			&s->type, basename, &s->id, &s->iodev, &s->inter,
			&s->factor, &s->offset, vref, default_label) == 9)
		{
		s->basename = g_strdup(basename);
		if (strcmp(vref, "NONE"))
			s->vref = g_strdup(vref);
		if (strcmp(default_label, "NONE"))
			s->default_label = g_strdup(default_label);

		sensors_list = g_list_append(sensors_list, s);
		}
	else
		g_free(s);
	}

/* ================================================================= */
static time_t		server_uptime;

static void
client_uptime_line_from_server(gchar *s)
	{
	gulong		up_minutes;

	sscanf(s, "%lu", &up_minutes);
	server_uptime = ((time_t) up_minutes) * 60;
	}

static time_t
client_read_uptime(void)
	{
	return server_uptime;
	}

static void
client_sys_uptime_init(void)
	{
	gkrellm_uptime_client_divert(client_read_uptime);
	}

/* ================================================================= */
static struct tm	server_time;

  /* clock monitor doesn't have a sysdep interface, so it needs a hook
  |  to get server system time when in client mode.
  */
struct tm *
gkrellm_client_server_time(void)
	{
	return &server_time;
	}

static void
client_time_line_from_server(gchar *s)
	{
	struct tm	*t;

	t = &server_time;
	sscanf(s, "%d %d %d %d %d %d %d %d %d",
			&t->tm_sec, &t->tm_min, &t->tm_hour,
			&t->tm_mday, &t->tm_mon, &t->tm_year,
			&t->tm_wday, &t->tm_yday, &t->tm_isdst);
	}

/* ================================================================= */
static void
client_hostname_setup(gchar *s)
	{
	_GK.server_hostname = g_strdup(s);
	}

static void
client_sysname_setup(gchar *s)
	{
	_GK.server_sysname = g_strdup(s);
	}


KeyTable	monitor_table[] =
	{
	{"sensors",		client_sys_sensors_init },
	{"cpu",			client_sys_cpu_init },
	{"proc",		client_sys_proc_init },
	{"disk",		client_sys_disk_init },
	{"net",			client_sys_net_init },
	{"inet",		client_sys_inet_init },
	{"mem",			client_sys_mem_init },
	{"fs",			client_sys_fs_init },
	{"apm",			client_sys_battery_init },
	{"battery",		client_sys_battery_init },
	{"uptime",		client_sys_uptime_init },
	};

static gboolean	setup_done;		/* only one sys init */

static void
client_monitor_setup(gchar *line)
	{
	void			(*func)();
	gint			i;

	if (!*line || setup_done)
		return;
	for (i = 0; i < sizeof(monitor_table) / sizeof(KeyTable); ++i)
		{
		if (!strcmp(line, monitor_table[i].key))
			{
			func = monitor_table[i].func;
			(*func)();
			break;
			}
		}
	}

static void
client_server_error(gchar *line)
	{
	fprintf(stderr, "gkrellmd error: %s\n", line);
	exit(0);
	}

/* ================================================================= */
KeyTable	setup_table[] =
	{
	{"<sensors_setup>",	client_sensors_setup },
	{"<hostname>",		client_hostname_setup },
	{"<sysname>",		client_sysname_setup },
	{"<cpu_setup>",		client_cpu_setup },
	{"<disk_setup>",	client_disk_setup },
	{"<inet_setup>",	client_inet_setup },
	{"<net_setup>",		client_net_setup },
	{"<fs_setup>",		client_fs_setup },
	{"<apm_setup>",		client_battery_setup },
	{"<battery_setup>",	client_battery_setup },
	{"<time>",			client_time_line_from_server},
	{"<monitors>",		client_monitor_setup },
	{"<error>",			client_server_error },
	};

KeyTable	update_table[] =
	{
	{"<cpu>",			client_cpu_line_from_server},
	{"<proc>",			client_proc_line_from_server},
	{"<disk>",			client_disk_line_from_server},
	{"<net>",			client_net_line_from_server},
	{"<net_routed>",	client_net_routed_line_from_server},
	{"<mem>",			client_mem_line_from_server},
	{"<swap>",			client_swap_line_from_server},
	{"<fs>",			client_fs_line_from_server},
	{"<fs_fstab>",		client_fstab_line_from_server},
	{"<fs_mounts>",		client_mounts_line_from_server},
	{"<inet>",			client_inet_line_from_server},
	{"<apm>",			client_battery_line_from_server},
	{"<battery>",		client_battery_line_from_server},
	{"<sensors>",		client_sensors_line_from_server},
//	{"<>",			client__line_from_server},
	{"<time>",			client_time_line_from_server},
	{"<uptime>",		client_uptime_line_from_server},
	};


static gint
getline(gint fd, gchar *buf, gint len)
	{
	fd_set			read_fds;
	struct timeval	tv;
	gchar			*s;
	gint			result, n, nread = 0;

	FD_ZERO(&read_fds);
	FD_SET(fd, &read_fds);
	tv.tv_usec = 0;
	tv.tv_sec = 15;
	s = buf;
	*s = '\0';
	for (n = 0; n < len - 1; ++n)
		{
		nread = 0;
		result = select(fd + 1, &read_fds, NULL, NULL, &tv);
		if (result <= 0 || (nread = recv(fd, s, 1, 0)) != 1)
			break;
		if (*s == '\n')
			{
			*s = '\0';
			break;
			}
		*++s = '\0';
		}
	if (nread < 0 && errno != EINTR)
		{
		fprintf(stderr, "Broken server connection\n");
		exit(0);
		}
	if (_GK.debug_level & DEBUG_CLIENT)
		printf("%s\n", buf);
	return n;
	}

static void
process_server_line(KeyTable *table, gint table_size, gchar *line)
	{
	static void		(*func)(gchar *);
	gint			i;

	if (!*line || *line == '#')
		return;

	if (*line == '<')
		{
		func = NULL;
		if (line[1] == '.')
			{
			server_time.tm_sec = atoi(line + 2);
			return;
			}
		for (i = 0; i < table_size; ++i)
			{
			if (!strcmp(table[i].key, line))
				{
				func = table[i].func;
				break;
				}
			}
		}
	else if (func)
		(*func)(line);
	}


  /* Read setup info from gkrellmd server.  Stuff needed before the
  |  client_init calls must be read here.
  */
static void
read_server_setup(gint fd)
	{
	gchar	buf[256];
	gint	table_size;

	table_size = sizeof(setup_table) / sizeof(KeyTable);
	while (1)
		{
		getline(fd, buf, sizeof(buf));
		if (!strcmp(buf, "</gkrellmd_setup>"))
			break;
		process_server_line(&setup_table[0], table_size, buf);
		}

	/* Read the initial update data
	*/
	table_size = sizeof(update_table) / sizeof(KeyTable);
	while (1)
		{
		getline(fd, buf, sizeof(buf));
		if (!strcmp(buf, "</initial_update>"))
			break;
		process_server_line(&update_table[0], table_size, buf);
		}
	setup_done = TRUE;
	}

static void
read_server_input(gpointer data, gint fd, GdkInputCondition condition)
	{
	gchar	*line, *eol;
	gint	count, n, table_size;

	n = sizeof(server_buf) - buf_index - 1;
	count = recv(fd, server_buf + buf_index, n, 0);
	if (count <= 0)
		{
		close(fd);
		gdk_input_remove(input_id);
		input_id = 0;
		/* reconnect */
		return;
		}
	server_buf[buf_index + count] = '\0';
	line = server_buf;
	table_size = sizeof(update_table) / sizeof(KeyTable);
	while (*line && (eol = strchr(line, '\n')) != NULL)
		{
		*eol = '\0';
		if (_GK.debug_level & DEBUG_CLIENT)
			printf("%s\n", line);
		process_server_line(&update_table[0], table_size, line);
		line = eol + 1;
		}
	if (line != server_buf && *line)
		{
		buf_index = strlen(line);
		memmove(server_buf, line, buf_index);
		}
	else
		buf_index = 0;

	server_buf[buf_index] = '\0';
	}

static gint
server_shutdown(gint fd, gchar *message)
	{
	printf("%s\n", message);
	return -1;
	}

static gint
connect_to_server(void)
	{
	gint		fd	= -1;
#ifdef HAVE_GETADDRINFO
	gint 		rv	= 0;
	struct addrinfo	hints, *res, *res0;
	gchar		portnumber[6];
#else
	struct hostent	*addr;
	struct sockaddr_in s;
#endif

#ifdef HAVE_GETADDRINFO
	snprintf (portnumber, sizeof(portnumber), "%d", _GK.server_port);
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = 0;
	if ((rv = getaddrinfo(_GK.server, portnumber, &hints, &res0)) != 0)
		return server_shutdown(fd, _("Unable to connect."));

	for (res = res0; res; res = res->ai_next)
		{
		if ((fd = socket(res->ai_family, res->ai_socktype,
				res->ai_protocol)) < 0)
			continue;
		if (_GK.debug_level & DEBUG_CLIENT)
			printf("\t[connect_to_server: (%d,%d,%d) %s:%d]\n",
					res->ai_family, res->ai_socktype, res->ai_protocol,
					_GK.server, _GK.server_port);
		if (connect(fd, res->ai_addr, res->ai_addrlen) >= 0)
			break;
		close(fd);
		fd = -1;
		}
	freeaddrinfo(res0);
	if (fd == -1)
		return server_shutdown(fd, _("Unable to connect."));
#else
	if (_GK.debug_level & DEBUG_CLIENT)
		printf("\t[connect_to_server: %s:%d]\n", _GK.server, _GK.server_port);
	addr = gethostbyname(_GK.server);
	if (addr)
		{
		memcpy(&s.sin_addr, addr->h_addr, sizeof(struct in_addr));  
		fd = socket(AF_INET, SOCK_STREAM, 0);
		if (fd >= 0)
			{
			s.sin_family = AF_INET;
			s.sin_port = htons(_GK.server_port);
			if (connect(fd, (struct sockaddr *)&s, sizeof (s)) < 0)
				{
				close(fd);
				fd = -1;
				}
			}
		}
	if (fd < 0) 
		return server_shutdown(fd, _("Unable to connect."));
#endif

	return fd;
	}

gboolean
gkrellm_client_mode_connect(void)
	{
	gchar	buf[128];
	gint	fd;

	if (_GK.server_port == 0)
		_GK.server_port = GKRELLMD_SERVER_PORT;

	if ((fd = connect_to_server()) < 0)
		return FALSE;

	snprintf(buf, sizeof(buf), "gkrellm %d.%d.%d %s\n",
			GKRELLM_VERSION_MAJOR, GKRELLM_VERSION_MINOR,
			GKRELLM_VERSION_REV, GKRELLM_EXTRAVERSION);
	send(fd, buf, strlen(buf), 0);

	/* Initial setup lines from server are read in blocking mode.
	*/
	read_server_setup(fd);

	/* Extra stuff not handled in read_server_setup()
	*/
	gkrellm_mail_local_unsupported();

	/* Now switch to non blocking and set up a read handler.
	*/
	fcntl(fd, F_SETFL, O_NONBLOCK);
	input_id = gdk_input_add(fd, GDK_INPUT_READ,
					(GdkInputFunction) read_server_input, NULL);


	return TRUE;
	}


static gboolean	client_mode_thread_busy;

gint
gkrellm_client_server_connect_state(void)
	{
	if (client_mode_thread_busy)
		return 2;
	if (_GK.client_mode && input_id > 0)
		return 1;
	return 0;
	}

static gpointer
client_mode_connect_thread(void *data)
	{
	gkrellm_client_mode_connect();
	client_mode_thread_busy = FALSE;
	return NULL;
	}

void
gkrellm_client_mode_connect_thread(void)
	{
	if (client_mode_thread_busy || !_GK.client_mode)
		return;
	client_mode_thread_busy = TRUE;
	g_thread_create(client_mode_connect_thread, NULL, FALSE, NULL);
	}
