/*
 *  (gnome-alsamixer) An ALSA mixer for GNOME
 *
 *  Copyright (C) 2001-2002 Dennis J Houy <djhouy@paw.za.org>.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtklabel.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkmenuitem.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-app-helper.h>

#include "gam-app.h"
#include "gam-mixer.h"
#include "gam-prefs-dlg.h"

static void      gam_app_class_init            (GamAppClass *klass);
static void      gam_app_init                  (GamApp      *gam_app);
static void      gam_app_destroy               (GtkObject   *object);
static void      gam_app_load_prefs            (GamApp      *gam_app);
static void      gam_app_save_prefs            (GamApp      *gam_app);
static GamMixer *gam_app_get_current_mixer     (GamApp      *gam_app);

static void      gam_app_about_cb              (GtkWidget   *widget,
                                                gpointer     data);
static gint      gam_app_quit_cb               (GtkWidget   *widget,
                                                GamApp      *gam_app);

static void      preferences_cb                (GtkMenuItem *menuitem,
                                                GamApp      *gam_app);
static void      properties_cb                 (GtkMenuItem *menuitem,
                                                GamApp      *gam_app);

static void      mixer_display_name_changed_cb (GamMixer    *gam_mixer,
                                                GamApp      *gam_app);
static void      mixer_visibility_changed_cb   (GamMixer    *gam_mixer,
                                                GamApp      *gam_app);

static gpointer parent_class;

static GnomeUIInfo file_menu_uiinfo[] =
{
    GNOMEUIINFO_MENU_PROPERTIES_ITEM (properties_cb, NULL),
    GNOMEUIINFO_MENU_PREFERENCES_ITEM (preferences_cb, NULL),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_EXIT_ITEM (gam_app_quit_cb, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo help_menu_uiinfo[] =
{
    GNOMEUIINFO_HELP ("GNOME ALSA Mixer"),
    GNOMEUIINFO_MENU_ABOUT_ITEM (gam_app_about_cb, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo menubar_uiinfo[] =
{
    GNOMEUIINFO_MENU_FILE_TREE (file_menu_uiinfo),
    GNOMEUIINFO_MENU_HELP_TREE (help_menu_uiinfo),
    GNOMEUIINFO_END
};

GType
gam_app_get_type (void)
{
    static GType gam_app_type = 0;

    if (!gam_app_type) {
        static const GTypeInfo gam_app_info =
        {
            sizeof (GamAppClass),
            NULL,               /* base_init */
            NULL,               /* base_finalize */
            (GClassInitFunc) gam_app_class_init,
            NULL,               /* class_finalize */
            NULL,               /* class_data */
            sizeof (GamApp),
            0,                  /* n_preallocs */
            (GInstanceInitFunc) gam_app_init,
        };

        gam_app_type = g_type_register_static (GNOME_TYPE_APP, "GamApp",
                                               &gam_app_info, 0);
    }

    return gam_app_type;
}

static void
gam_app_class_init (GamAppClass *klass)
{
    GtkObjectClass *object_class;

    object_class = (GtkObjectClass*) klass;

    parent_class = g_type_class_peek_parent (klass);

    object_class->destroy = gam_app_destroy;
}

static void
gam_app_init (GamApp *gam_app)
{
    GtkWidget *mixer;
    GtkWidget *label;
    snd_ctl_t *ctl_handle;
    gint result, index = 0;
    gchar *card;

    g_return_if_fail (GAM_IS_APP (gam_app));

    g_signal_connect (G_OBJECT (gam_app), "delete_event",
                      G_CALLBACK (gam_app_quit_cb), NULL);

    gnome_window_icon_set_default_from_file (PIXMAP_ICONDIR "/gnome-alsamixer-icon.png");

    gnome_app_create_menus_with_data (GNOME_APP (gam_app), menubar_uiinfo, gam_app);

    gam_app->appbar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_NEVER);
    gtk_widget_show (gam_app->appbar);
    gnome_app_set_statusbar (GNOME_APP (gam_app), gam_app->appbar);

    gnome_app_install_menu_hints (GNOME_APP (gam_app), menubar_uiinfo);

    gam_app->notebook = gtk_notebook_new ();

    gtk_notebook_set_scrollable (GTK_NOTEBOOK (gam_app->notebook), TRUE);
    gtk_widget_show (gam_app->notebook);
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (gam_app->notebook), GTK_POS_TOP);

    do {
        card = g_strdup_printf ("hw:%d", index++);

        result = snd_ctl_open (&ctl_handle, card, 0);

        if (result == 0) {
            snd_ctl_close(ctl_handle);

            mixer = gam_mixer_new (card);

            if (gam_mixer_get_visible (GAM_MIXER (mixer)))
                gtk_widget_show (mixer);

            g_signal_connect (G_OBJECT (mixer), "display_name_changed",
                              G_CALLBACK (mixer_display_name_changed_cb), gam_app);

            g_signal_connect (G_OBJECT (mixer), "visibility_changed",
                              G_CALLBACK (mixer_visibility_changed_cb), gam_app);

            label = gtk_label_new (gam_mixer_get_display_name (GAM_MIXER (mixer)));
            gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
            gtk_widget_show (label);

            gtk_notebook_append_page (GTK_NOTEBOOK (gam_app->notebook), mixer, label);
        }

        g_free (card);
    } while (result == 0);

    gam_app->num_cards = index - 1;

    gnome_app_set_contents (GNOME_APP (gam_app), gam_app->notebook);
}

static void
gam_app_destroy (GtkObject *object)
{
    GamApp *gam_app = GAM_APP (object);

    gam_app->appbar = NULL;

    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gam_app_load_prefs (GamApp *gam_app)
{
    gint height, width;

    width = gnome_config_get_int ("/gnome-alsamixer/geometry/main_window_width=450");
    height = gnome_config_get_int ("/gnome-alsamixer/geometry/main_window_height=380");

    gtk_window_resize (GTK_WINDOW (gam_app), width, height);
}

static void
gam_app_save_prefs (GamApp *gam_app)
{
    gint height, width;

    gdk_window_get_geometry (GDK_WINDOW (GTK_WIDGET (gam_app)->window), NULL, NULL, &width, &height, NULL);

    gnome_config_set_int ("/gnome-alsamixer/geometry/main_window_width", width);
    gnome_config_set_int ("/gnome-alsamixer/geometry/main_window_height", height);

    gnome_config_sync ();
}

static GamMixer *
gam_app_get_current_mixer (GamApp *gam_app)
{
    GtkWidget *mixer;

    mixer = gtk_notebook_get_nth_page (GTK_NOTEBOOK (gam_app->notebook),
              gtk_notebook_get_current_page (GTK_NOTEBOOK (gam_app->notebook)));

    return (GAM_MIXER (mixer));
}

static void
gam_app_about_cb (GtkWidget *widget, gpointer data)
{
    GtkWidget *about;
    const gchar *authors[] = {
        "Dennis J Houy <djhouy@paw.co.za>",
        "David Fort <popo.enlighted@free.fr>",
        "Ben Liblit <liblit@acm.org>",
        NULL
    };

    about = gnome_about_new ("GNOME ALSA Mixer", VERSION,
                             _("\302\251 2002 PAW Digital Dynamics"),
                             _("An ALSA mixer for GNOME"),
                             authors,
                             NULL,
                             NULL,
                             NULL);
    gtk_widget_set_name (about, "about");
    gtk_window_set_wmclass (GTK_WINDOW (about), "GAMAbout", "GAMAbout");
    gtk_window_set_destroy_with_parent (GTK_WINDOW (about), TRUE);
    gtk_widget_show (about);
}

static gint
gam_app_quit_cb (GtkWidget *widget, GamApp *gam_app)
{
    gtk_main_quit ();

    return TRUE;
}

static void
preferences_cb (GtkMenuItem *menuitem, GamApp *gam_app)
{
    static GtkWidget *dialog = NULL;

    if (dialog != NULL) {
        gtk_window_present (GTK_WINDOW (dialog));
        gtk_window_set_transient_for (GTK_WINDOW (dialog),
                                      GTK_WINDOW (gam_app));

        return;
    }

    dialog = gam_prefs_dlg_new (GTK_WINDOW (gam_app));

    g_signal_connect (G_OBJECT (dialog), "destroy",
                      G_CALLBACK (gtk_widget_destroyed), &dialog);

    gtk_widget_show (dialog);
}

static void
properties_cb (GtkMenuItem *menuitem, GamApp *gam_app)
{
    gam_mixer_show_props_dialog (gam_app, gam_app_get_current_mixer (gam_app));
}

static void
mixer_display_name_changed_cb (GamMixer *gam_mixer, GamApp *gam_app)
{
    gtk_notebook_set_tab_label_text (GTK_NOTEBOOK (gam_app->notebook),
                                     GTK_WIDGET (gam_mixer),
                                     gam_mixer_get_display_name (gam_mixer));
}

static void
mixer_visibility_changed_cb (GamMixer *gam_mixer, GamApp *gam_app)
{
    if (gam_mixer_get_visible (gam_mixer))
        gtk_widget_show (GTK_WIDGET (gam_mixer));
    else
        gtk_widget_hide (GTK_WIDGET (gam_mixer));
}

GtkWidget *
gam_app_new (GnomeProgram *prog)
{
    GtkWidget *app;
    gint height = 380, width = 450;

    app = g_object_new (GAM_TYPE_APP,
                        "app_id", "gnome-alsamixer",
                        "title", "GNOME ALSA Mixer",
                        NULL);

    width = gnome_config_get_int ("/gnome-alsamixer/geometry/main_window_width=450");
    height = gnome_config_get_int ("/gnome-alsamixer/geometry/main_window_height=380");

    if ((height != 0) && (width != 0))
        gtk_window_set_default_size (GTK_WINDOW (app), width, height);

    return app;
}

void
gam_app_run (GamApp *gam_app)
{
    gtk_widget_show (GTK_WIDGET (gam_app));
    gtk_main ();
    gam_app_save_prefs (gam_app);
}

gint
gam_app_get_num_cards (GamApp *gam_app)
{
    return gam_app->num_cards;
}

GamMixer *gam_app_get_mixer (GamApp *gam_app, gint index)
{
    GtkWidget *mixer;

    g_return_val_if_fail (GAM_IS_APP (gam_app), NULL);
    g_return_val_if_fail ((index >= 0) && (index <= gam_app->num_cards), NULL);

    mixer = gtk_notebook_get_nth_page (GTK_NOTEBOOK (gam_app->notebook), index);

    return GAM_MIXER (mixer);
}
