/*
 *  (gnome-alsamixer) An ALSA mixer for GNOME
 *
 *  Copyright (C) 2001-2002 Dennis J Houy <djhouy@paw.za.org>.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <gnome.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkhscale.h>
#include <gtk/gtkvscale.h>
#include <gtk/gtkvseparator.h>
#include <gtk/gtktogglebutton.h>

#include "gam-slider.h"

#undef ABS
#define ABS(a)     (((a) < 0) ? -(a) : (a))
#undef MAX
#define MAX(a, b)  (((a) > (b)) ? (a) : (b))

enum {
    PROP_0,
    PROP_ELEM,
    PROP_MIXER
};

static void     gam_slider_class_init     (GamSliderClass        *klass);
static void     gam_slider_init           (GamSlider             *gam_slider);
static void     gam_slider_destroy        (GtkObject             *object);
static GObject *gam_slider_constructor    (GType                  type,
                                           guint                  n_construct_properties,
                                           GObjectConstructParam *construct_params);
static void     gam_slider_set_property   (GObject               *object,
                                           guint                  prop_id,
                                           const GValue          *value,
                                           GParamSpec            *pspec);
static void     gam_slider_get_property   (GObject               *object,
                                           guint                  prop_id,
                                           GValue                *value,
                                           GParamSpec            *pspec);
static gint     gam_slider_get_pan        (GamSlider             *gam_slider);
static gint     gam_slider_get_volume     (GamSlider             *gam_slider);
static void     gam_slider_update_volume   (GamSlider            *gam_slider);

static gint     pan_event_cb              (GtkWidget             *widget,
                                           GdkEvent              *event,
                                           GamSlider             *gam_slider);
static gint     pan_value_changed_cb      (GtkWidget             *widget,
                                           GamSlider             *gam_slider);
static gint     volume_value_changed_cb   (GtkWidget             *widget,
                                           GamSlider             *gam_slider);
static gint     mute_button_toggled_cb    (GtkWidget             *widget,
                                           GamSlider             *gam_slider);
static gint     capture_button_toggled_cb (GtkWidget             *widget,
                                           GamSlider             *gam_slider);

static int      gam_slider_refresh        (snd_mixer_elem_t      *elem,
                                           unsigned int           mask);

static gpointer parent_class;

GType
gam_slider_get_type (void)
{
    static GType gam_slider_type = 0;

    if (!gam_slider_type) {
        static const GTypeInfo gam_slider_info =
        {
            sizeof (GamSliderClass),
            NULL,               /* base_init */
            NULL,               /* base_finalize */
            (GClassInitFunc) gam_slider_class_init,
            NULL,               /* class_finalize */
            NULL,               /* class_data */
            sizeof (GamSlider),
            0,                  /* n_preallocs */
            (GInstanceInitFunc) gam_slider_init,
        };

        gam_slider_type = g_type_register_static (GTK_TYPE_HBOX, "GamSlider",
                                                  &gam_slider_info, 0);
    }

    return gam_slider_type;
}

static void
gam_slider_class_init (GamSliderClass *klass)
{
    GObjectClass *gobject_class;
    GtkObjectClass *object_class;

    gobject_class = G_OBJECT_CLASS (klass);

    object_class = (GtkObjectClass*) klass;

    parent_class = g_type_class_peek_parent (klass);

    object_class->destroy = gam_slider_destroy;

    gobject_class->constructor = gam_slider_constructor;
    gobject_class->set_property = gam_slider_set_property;
    gobject_class->get_property = gam_slider_get_property;

    g_object_class_install_property (gobject_class,
                                     PROP_ELEM,
                                     g_param_spec_pointer ("elem",
                                                           _("Element"),
                                                           _("ALSA mixer element"),
                                                           G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

    g_object_class_install_property (gobject_class,
                                     PROP_MIXER,
                                     g_param_spec_pointer ("mixer",
                                                           _("Mixer"),
                                                           _("Mixer"),
                                                           G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

static void
gam_slider_init (GamSlider *gam_slider)
{
    g_return_if_fail (GAM_IS_SLIDER (gam_slider));

    gam_slider->elem = NULL;
    gam_slider->mixer = NULL;
    gam_slider->name = NULL;
    gam_slider->pan_slider = NULL;
    gam_slider->vol_slider = NULL;
    gam_slider->pan_adjustment = NULL;
    gam_slider->vol_adjustment = NULL;
    gam_slider->mute_button = NULL;
    gam_slider->capture_button = NULL;
}


static void
gam_slider_destroy (GtkObject *object)
{
    GamSlider *gam_slider = GAM_SLIDER (object);

    g_free (gam_slider->name);

    gam_slider->pan_adjustment = NULL;
    gam_slider->vol_adjustment = NULL;

    gtk_widget_destroy (gam_slider->label);
    gam_slider->label = NULL;

    gtk_widget_destroy (gam_slider->pan_slider);
    gam_slider->pan_slider = NULL;

    gtk_widget_destroy (gam_slider->vol_slider);
    gam_slider->vol_slider = NULL;

    gtk_widget_destroy (gam_slider->mute_button);
    gam_slider->mute_button = NULL;

    gtk_widget_destroy (gam_slider->capture_button);
    gam_slider->capture_button = NULL;

    snd_mixer_elem_set_callback (gam_slider->elem, NULL);
    gam_slider->elem = NULL;

    gam_slider->mixer = NULL;

    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static GObject*
gam_slider_constructor (GType                  type,
                        guint                  n_construct_properties,
                        GObjectConstructParam *construct_params)
{
    gint value;
    GObject *object;
    GamSlider *gam_slider;
    GtkWidget *label, *vbox, *separator;

    object = (* G_OBJECT_CLASS (parent_class)->constructor) (type,
                                                             n_construct_properties,
                                                             construct_params);

    gam_slider = GAM_SLIDER (object);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);

    gtk_box_pack_start (GTK_BOX (gam_slider),
                        vbox, TRUE, TRUE, 0);

    separator = gtk_vseparator_new ();
    gtk_widget_show (separator);

    gtk_box_pack_start (GTK_BOX (gam_slider),
                        separator, FALSE, TRUE, 0);

    gam_slider->label = gtk_label_new (gam_slider_get_display_name (gam_slider));
    gtk_widget_show (gam_slider->label);

    gtk_box_pack_start (GTK_BOX (vbox),
                        gam_slider->label, FALSE, TRUE, 0);

    gam_slider->vol_adjustment = gtk_adjustment_new (gam_slider_get_volume (gam_slider), 0, 101, 1, 5, 1);

    g_signal_connect (G_OBJECT (gam_slider->vol_adjustment), "value-changed",
                      G_CALLBACK (volume_value_changed_cb), gam_slider);

    gam_slider->vol_slider = gtk_vscale_new (GTK_ADJUSTMENT (gam_slider->vol_adjustment));
    gtk_widget_show (gam_slider->vol_slider);
    gtk_scale_set_draw_value (GTK_SCALE (gam_slider->vol_slider), FALSE);

    gtk_box_pack_start (GTK_BOX (vbox),
                        gam_slider->vol_slider, TRUE, TRUE, 0);

    if (!snd_mixer_selem_is_playback_mono (gam_slider->elem)) {
        gam_slider->pan_adjustment = gtk_adjustment_new (gam_slider_get_pan (gam_slider), -100, 101, 1, 5, 1);

        g_signal_connect (G_OBJECT (gam_slider->pan_adjustment), "value-changed",
                          G_CALLBACK (pan_value_changed_cb), gam_slider);

        gam_slider->pan_slider = gtk_hscale_new (GTK_ADJUSTMENT (gam_slider->pan_adjustment));
        gtk_scale_set_draw_value (GTK_SCALE (gam_slider->pan_slider), FALSE);

        g_signal_connect (G_OBJECT (gam_slider->pan_slider), "event",
                          G_CALLBACK (pan_event_cb), gam_slider);
    } else {
        gam_slider->pan_slider = gtk_label_new (NULL);
    }

    gtk_widget_show (gam_slider->pan_slider);
    gtk_box_pack_start (GTK_BOX (vbox),
                        gam_slider->pan_slider, FALSE, FALSE, 0);

    if (snd_mixer_selem_has_playback_switch (gam_slider->elem)) {
        gam_slider->mute_button = gtk_toggle_button_new_with_label (_("Mute"));
        snd_mixer_selem_get_playback_switch (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &value);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gam_slider->mute_button), !value);

        g_signal_connect (G_OBJECT (gam_slider->mute_button), "toggled",
                          G_CALLBACK (mute_button_toggled_cb), gam_slider);
    } else {
        gam_slider->mute_button = gtk_label_new (NULL);
    }

    gtk_widget_show (gam_slider->mute_button);
    gtk_box_pack_start (GTK_BOX (vbox),
                        gam_slider->mute_button, FALSE, FALSE, 0);

    if (snd_mixer_selem_has_capture_switch (gam_slider->elem)) {
        gam_slider->capture_button = gtk_toggle_button_new_with_label (_("Rec."));
        snd_mixer_selem_get_capture_switch (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &value);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gam_slider->capture_button), value);

        g_signal_connect (G_OBJECT (gam_slider->capture_button), "toggled",
                          G_CALLBACK (capture_button_toggled_cb), gam_slider);
    } else {
        gam_slider->capture_button = gtk_label_new (NULL);
    }

    gtk_widget_show (gam_slider->capture_button);
    gtk_box_pack_start (GTK_BOX (vbox),
                        gam_slider->capture_button, FALSE, FALSE, 0);

    return object;
}

static void
gam_slider_set_property (GObject         *object,
                         guint            prop_id,
                         const GValue    *value,
                         GParamSpec      *pspec)
{
    GamSlider *gam_slider;

    gam_slider = GAM_SLIDER (object);

    switch (prop_id) {
        case PROP_ELEM:
            gam_slider_set_elem (gam_slider, g_value_get_pointer (value));
            break;
        case PROP_MIXER:
            gam_slider->mixer = g_value_get_pointer (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
gam_slider_get_property (GObject         *object,
                         guint            prop_id,
                         GValue          *value,
                         GParamSpec      *pspec)
{
    GamSlider *gam_slider;

    gam_slider = GAM_SLIDER (object);

    switch (prop_id) {
        case PROP_ELEM:
            g_value_set_pointer (value, gam_slider->elem);
            break;
        case PROP_MIXER:
            g_value_set_pointer (value, gam_slider->mixer);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static gint
gam_slider_get_pan (GamSlider *gam_slider)
{
    glong left_chn, right_chn;

    if (!snd_mixer_selem_is_playback_mono (gam_slider->elem)) {
        if (snd_mixer_selem_has_playback_volume (gam_slider->elem))
            snd_mixer_selem_get_playback_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &left_chn);
        else
            snd_mixer_selem_get_capture_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &left_chn);

        if (snd_mixer_selem_has_playback_volume (gam_slider->elem))
            snd_mixer_selem_get_playback_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_RIGHT, &right_chn);
        else
            snd_mixer_selem_get_capture_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_RIGHT, &right_chn);

        if ((gam_slider_get_volume (gam_slider) != 0) && (left_chn != right_chn))
            return rint (((gfloat)(right_chn - left_chn) / (gfloat)MAX(left_chn, right_chn)) * 100);
    }

    return 0;
}

static gint
gam_slider_get_volume (GamSlider *gam_slider)
{
    glong left_chn, right_chn, pmin, pmax;

    if (snd_mixer_selem_has_playback_volume (gam_slider->elem))
        snd_mixer_selem_get_playback_volume_range (gam_slider->elem, &pmin, &pmax);
    else
        snd_mixer_selem_get_capture_volume_range (gam_slider->elem, &pmin, &pmax);

    if (snd_mixer_selem_has_playback_volume (gam_slider->elem))
        snd_mixer_selem_get_playback_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &left_chn);
    else
        snd_mixer_selem_get_capture_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &left_chn);

    if (snd_mixer_selem_is_playback_mono (gam_slider->elem)) {
        return rint (100 - (left_chn * (100 / (gfloat)pmax)));
    } else {
        if (snd_mixer_selem_has_playback_volume (gam_slider->elem))
            snd_mixer_selem_get_playback_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_RIGHT, &right_chn);
        else
            snd_mixer_selem_get_capture_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_RIGHT, &right_chn);

        return rint (100 - (MAX(left_chn, right_chn) * (100 / (gfloat)pmax)));
    }
}

static void
gam_slider_update_volume (GamSlider *gam_slider)
{
    gint left_chn = 0, right_chn = 0;
    glong pmin, pmax;

    if (snd_mixer_selem_has_playback_volume (gam_slider->elem))
        snd_mixer_selem_get_playback_volume_range (gam_slider->elem, &pmin, &pmax);
    else
        snd_mixer_selem_get_capture_volume_range (gam_slider->elem, &pmin, &pmax);

    left_chn = right_chn = rint ((100 - GTK_ADJUSTMENT (gam_slider->vol_adjustment)->value) / (100 / (gfloat)pmax));

    if (!snd_mixer_selem_is_playback_mono (gam_slider->elem)) {
        if (GTK_ADJUSTMENT (gam_slider->pan_adjustment)->value < 0) {
            right_chn = rint (left_chn - ((gfloat)ABS(GTK_ADJUSTMENT (gam_slider->pan_adjustment)->value) / 100) * left_chn);
        } else if (GTK_ADJUSTMENT (gam_slider->pan_adjustment)->value > 0) {
            left_chn = rint (right_chn - ((gfloat)GTK_ADJUSTMENT (gam_slider->pan_adjustment)->value / 100) * right_chn);
        }
    }

    if (snd_mixer_selem_has_playback_volume (gam_slider->elem)) {
        snd_mixer_selem_set_playback_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, left_chn);
        if (!snd_mixer_selem_is_playback_mono (gam_slider->elem))
            snd_mixer_selem_set_playback_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_RIGHT, right_chn);
    } else {
        snd_mixer_selem_set_capture_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, left_chn);
        if (!snd_mixer_selem_is_playback_mono (gam_slider->elem))
            snd_mixer_selem_set_capture_volume (gam_slider->elem, SND_MIXER_SCHN_FRONT_RIGHT, right_chn);
    }
}

static gint
pan_event_cb (GtkWidget *widget, GdkEvent *event, GamSlider *gam_slider)
{
    if (event->type == GDK_2BUTTON_PRESS) {
        gtk_adjustment_set_value (GTK_ADJUSTMENT (gam_slider->pan_adjustment), 0.0);
        gtk_adjustment_value_changed (GTK_ADJUSTMENT (gam_slider->pan_adjustment));

        return TRUE;
    }

    return FALSE;
}

static gint
pan_value_changed_cb (GtkWidget *widget, GamSlider *gam_slider)
{
    gam_slider_update_volume (gam_slider);

    return TRUE;
}

static gint
volume_value_changed_cb (GtkWidget *widget, GamSlider *gam_slider)
{
    gam_slider_update_volume (gam_slider);

    return TRUE;
}

static gint
mute_button_toggled_cb (GtkWidget *widget, GamSlider *gam_slider)
{
    snd_mixer_selem_set_playback_switch_all (gam_slider->elem,
                !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)));

    return TRUE;
}

static gint
capture_button_toggled_cb (GtkWidget *widget, GamSlider *gam_slider)
{
    snd_mixer_selem_set_capture_switch_all (gam_slider->elem,
                gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)));

    return TRUE;
}

GtkWidget *
gam_slider_new (gpointer elem, GamMixer *gam_mixer)
{
    return g_object_new (GAM_TYPE_SLIDER,
                         "elem", elem,
                         "mixer", gam_mixer,
                         NULL);
}

snd_mixer_elem_t *
gam_slider_get_elem (GamSlider *gam_slider)
{
    g_return_val_if_fail (GAM_IS_SLIDER (gam_slider), NULL);

    return gam_slider->elem;
}

void
gam_slider_set_elem (GamSlider *gam_slider, snd_mixer_elem_t *elem)
{
    g_return_if_fail (GAM_IS_SLIDER (gam_slider));

    if (gam_slider->elem)
        snd_mixer_elem_set_callback (gam_slider->elem, NULL);

    if (elem) {
        snd_mixer_elem_set_callback_private (elem, gam_slider);
        snd_mixer_elem_set_callback (elem, gam_slider_refresh);
    }

    gam_slider->elem = elem;

    g_object_notify (G_OBJECT (gam_slider), "elem");
}

G_CONST_RETURN gchar *
gam_slider_get_name (GamSlider *gam_slider)
{
    g_return_val_if_fail (GAM_IS_SLIDER (gam_slider), NULL);

    return snd_mixer_selem_get_name (gam_slider->elem);
}

gchar *
gam_slider_get_display_name (GamSlider *gam_slider)
{
    gchar *key, *name, *disp_name;

    g_return_val_if_fail (GAM_IS_SLIDER (gam_slider), NULL);

    disp_name = g_strndup (gam_slider_get_name (gam_slider), 8);

    key = g_strdup_printf ("/gnome-alsamixer/slider_display_names/%s-%s=%s",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_slider->mixer)),
                           gam_slider_get_name (gam_slider),
                           disp_name);

    name = gnome_config_get_string (key);

    g_free (disp_name);
    g_free (key);

    return name;
}

void
gam_slider_set_display_name (GamSlider *gam_slider, const gchar *name)
{
    gchar *key;

    g_return_if_fail (GAM_IS_SLIDER (gam_slider));

    key = g_strdup_printf ("/gnome-alsamixer/slider_display_names/%s-%s",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_slider->mixer)),
                           gam_slider_get_name (gam_slider));

    gnome_config_set_string (key, name);

    gnome_config_sync ();

    gtk_label_set_text (GTK_LABEL (gam_slider->label), name);
}

gboolean
gam_slider_get_visible (GamSlider *gam_slider)
{
    gchar *key;
    gboolean visible;

    g_return_if_fail (GAM_IS_SLIDER (gam_slider));

    key = g_strdup_printf ("/gnome-alsamixer/display_sliders/%s-%s=true",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_slider->mixer)),
                           gam_slider_get_name (gam_slider));

    visible = gnome_config_get_bool (key);

    g_free (key);

    return visible;
}

void
gam_slider_set_visible (GamSlider *gam_slider, gboolean visible)
{
    gchar *key;

    g_return_if_fail (GAM_IS_SLIDER (gam_slider));

    key = g_strdup_printf ("/gnome-alsamixer/display_sliders/%s-%s",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_slider->mixer)),
                           gam_slider_get_name (gam_slider));

    gnome_config_set_bool (key, visible);

    gnome_config_sync ();

    if (visible)
        gtk_widget_show (GTK_WIDGET (gam_slider));
    else
        gtk_widget_hide (GTK_WIDGET (gam_slider));
}

void
gam_slider_set_size_groups (GamSlider *gam_slider,
                            GtkSizeGroup *pan_size_group,
                            GtkSizeGroup *mute_size_group,
                            GtkSizeGroup *capture_size_group)
{
    g_return_if_fail (GAM_IS_SLIDER (gam_slider));

    gtk_size_group_add_widget (pan_size_group, gam_slider->pan_slider);
    gtk_size_group_add_widget (mute_size_group, gam_slider->mute_button);
    gtk_size_group_add_widget (capture_size_group, gam_slider->capture_button);
}

static int
gam_slider_refresh (snd_mixer_elem_t *elem, unsigned int mask)
{
    GamSlider * const gam_slider = GAM_SLIDER (snd_mixer_elem_get_callback_private (elem));
    gint value;

    gtk_adjustment_set_value (GTK_ADJUSTMENT (gam_slider->vol_adjustment),
                              (gdouble) gam_slider_get_volume (gam_slider));
    gtk_adjustment_changed (GTK_ADJUSTMENT (gam_slider->vol_adjustment));

    if (!snd_mixer_selem_is_playback_mono (gam_slider->elem)) {
        gtk_adjustment_set_value (GTK_ADJUSTMENT (gam_slider->pan_adjustment),
                                  (gdouble) gam_slider_get_pan (gam_slider));
        gtk_adjustment_changed (GTK_ADJUSTMENT (gam_slider->pan_adjustment));
    }

    if (snd_mixer_selem_has_playback_switch (gam_slider->elem)) {
        snd_mixer_selem_get_playback_switch (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &value);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gam_slider->mute_button), !value);
    }

    if (snd_mixer_selem_has_capture_switch (gam_slider->elem)) {
        snd_mixer_selem_get_capture_switch (gam_slider->elem, SND_MIXER_SCHN_FRONT_LEFT, &value);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gam_slider->capture_button), value);
    }

    return 0;
}
