/*
 *  (gnome-alsamixer) An ALSA mixer for GNOME
 *
 *  Copyright (C) 2001-2002 Dennis J Houy <djhouy@paw.za.org>.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>

#include "gam-toggle.h"

enum {
    PROP_0,
    PROP_ELEM,
    PROP_MIXER
};

static void     gam_toggle_class_init   (GamToggleClass        *klass);
static void     gam_toggle_init         (GamToggle             *gam_toggle);
static void     gam_toggle_destroy      (GtkObject             *object);
static GObject *gam_toggle_constructor  (GType                  type,
                                         guint                  n_construct_properties,
                                         GObjectConstructParam *construct_params);
static void     gam_toggle_set_property (GObject               *object,
                                         guint                  prop_id,
                                         const GValue          *value,
                                         GParamSpec            *pspec);
static void     gam_toggle_get_property (GObject               *object,
                                         guint                  prop_id,
                                         GValue                *value,
                                         GParamSpec            *pspec);

static gint     toggled_cb              (GtkWidget             *widget,
                                         GamToggle             *gam_toggle);

static int      gam_toggle_refresh      (snd_mixer_elem_t      *elem,
                                         unsigned int           mask);

static gpointer parent_class;

GType
gam_toggle_get_type (void)
{
    static GType gam_toggle_type = 0;

    if (!gam_toggle_type) {
        static const GTypeInfo gam_toggle_info =
        {
            sizeof (GamToggleClass),
            NULL,               /* base_init */
            NULL,               /* base_finalize */
            (GClassInitFunc) gam_toggle_class_init,
            NULL,               /* class_finalize */
            NULL,               /* class_data */
            sizeof (GamToggle),
            0,                  /* n_preallocs */
            (GInstanceInitFunc) gam_toggle_init,
        };

        gam_toggle_type = g_type_register_static (GTK_TYPE_CHECK_BUTTON, "GamToggle",
                                                  &gam_toggle_info, 0);
    }

    return gam_toggle_type;
}

static void
gam_toggle_class_init (GamToggleClass *klass)
{
    GObjectClass *gobject_class;
    GtkObjectClass *object_class;

    gobject_class = G_OBJECT_CLASS (klass);

    object_class = (GtkObjectClass*) klass;

    parent_class = g_type_class_peek_parent (klass);

    object_class->destroy = gam_toggle_destroy;

    gobject_class->constructor = gam_toggle_constructor;
    gobject_class->set_property = gam_toggle_set_property;
    gobject_class->get_property = gam_toggle_get_property;

    g_object_class_install_property (gobject_class,
                                     PROP_ELEM,
                                     g_param_spec_pointer ("elem",
                                                           _("Element"),
                                                           _("ALSA mixer element"),
                                                           G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

    g_object_class_install_property (gobject_class,
                                     PROP_MIXER,
                                     g_param_spec_pointer ("mixer",
                                                           _("Mixer"),
                                                           _("Mixer"),
                                                           G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

static void
gam_toggle_init (GamToggle *gam_toggle)
{
    g_return_if_fail (GAM_IS_TOGGLE (gam_toggle));

    gam_toggle->elem = NULL;
    gam_toggle->mixer = NULL;
}

static void
gam_toggle_destroy (GtkObject *object)
{
    GamToggle *gam_toggle = GAM_TOGGLE (object);

    snd_mixer_elem_set_callback (gam_toggle->elem, NULL);
    gam_toggle->elem = NULL;
    gam_toggle->mixer = NULL;

    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static GObject*
gam_toggle_constructor (GType                  type,
                        guint                  n_construct_properties,
                        GObjectConstructParam *construct_params)
{
    GObject *object;
    GamToggle *gam_toggle;

    object = (* G_OBJECT_CLASS (parent_class)->constructor) (type,
                                                             n_construct_properties,
                                                             construct_params);

    gam_toggle = GAM_TOGGLE (object);

    gtk_button_set_label (GTK_BUTTON (gam_toggle), gam_toggle_get_display_name (gam_toggle));

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gam_toggle),
                                  gam_toggle_get_state (gam_toggle));

    g_signal_connect (G_OBJECT (gam_toggle), "toggled",
                      G_CALLBACK (toggled_cb), gam_toggle);

    return object;
}

static void
gam_toggle_set_property (GObject         *object,
                         guint            prop_id,
                         const GValue    *value,
                         GParamSpec      *pspec)
{
    GamToggle *gam_toggle;

    gam_toggle = GAM_TOGGLE (object);

    switch (prop_id) {
        case PROP_ELEM:
            gam_toggle_set_elem (gam_toggle, g_value_get_pointer (value));
            break;
        case PROP_MIXER:
            gam_toggle->mixer = g_value_get_pointer (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
gam_toggle_get_property (GObject         *object,
                         guint            prop_id,
                         GValue          *value,
                         GParamSpec      *pspec)
{
    GamToggle *gam_toggle;

    gam_toggle = GAM_TOGGLE (object);

    switch (prop_id) {
        case PROP_ELEM:
            g_value_set_pointer (value, gam_toggle->elem);
            break;
        case PROP_MIXER:
            g_value_set_pointer (value, gam_toggle->mixer);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static gint
toggled_cb (GtkWidget *widget, GamToggle *gam_toggle)
{
    gam_toggle_set_state (gam_toggle, 
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)));

    return TRUE;
}

GtkWidget *
gam_toggle_new (gpointer elem, GamMixer *gam_mixer)
{
    return g_object_new (GAM_TYPE_TOGGLE,
                         "elem", elem,
                         "mixer", gam_mixer,
                         NULL);
}

snd_mixer_elem_t *
gam_toggle_get_elem (GamToggle *gam_toggle)
{
    g_return_val_if_fail (GAM_IS_TOGGLE (gam_toggle), NULL);

    return gam_toggle->elem;
}

void
gam_toggle_set_elem (GamToggle *gam_toggle, snd_mixer_elem_t *elem)
{
    g_return_if_fail (GAM_IS_TOGGLE (gam_toggle));

    if (gam_toggle->elem)
        snd_mixer_elem_set_callback (gam_toggle->elem, NULL);

    if (elem) {
        snd_mixer_elem_set_callback_private (elem, gam_toggle);
        snd_mixer_elem_set_callback (elem, gam_toggle_refresh);
    }

    gam_toggle->elem = elem;

    g_object_notify (G_OBJECT (gam_toggle), "elem");
}

gboolean
gam_toggle_get_state (GamToggle *gam_toggle)
{
    gint value = 0;

    if (snd_mixer_selem_has_playback_switch (gam_toggle->elem)) {
        snd_mixer_selem_get_playback_switch (gam_toggle->elem, 0, &value);
    } else if (snd_mixer_selem_has_capture_switch (gam_toggle->elem)) {
        snd_mixer_selem_get_capture_switch (gam_toggle->elem, 0, &value);
    } else {
        g_warning ("%s (). No idea what to do for mixer element \"%s\"!",
                   __FUNCTION__, snd_mixer_selem_get_name (gam_toggle->elem));
    }

    return value;
}

void
gam_toggle_set_state (GamToggle *gam_toggle, gboolean state)
{
    int err;
    const gboolean internal_state = gam_toggle_get_state (gam_toggle);

    if (state == internal_state) return;

    if (snd_mixer_selem_has_playback_switch (gam_toggle->elem)) {
        err = snd_mixer_selem_set_playback_switch_all (gam_toggle->elem, state);
    } else if (snd_mixer_selem_has_capture_switch (gam_toggle->elem)) {
        err = snd_mixer_selem_set_capture_switch_all (gam_toggle->elem, state);
    } else {
        g_warning ("%s (). No idea what to do for mixer element \"%s\"!",
                   __FUNCTION__, snd_mixer_selem_get_name (gam_toggle->elem));
        err = 0;
    }

    if (err)
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gam_toggle),
                                      internal_state);
}

static int
gam_toggle_refresh (snd_mixer_elem_t *elem, unsigned int mask)
{
    GamToggle * const gam_toggle = GAM_TOGGLE (snd_mixer_elem_get_callback_private (elem));

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gam_toggle),
                                  gam_toggle_get_state (gam_toggle));

    return 0;
}

G_CONST_RETURN gchar *
gam_toggle_get_name (GamToggle *gam_toggle)
{
    g_return_val_if_fail (GAM_IS_TOGGLE (gam_toggle), NULL);

    return snd_mixer_selem_get_name (gam_toggle->elem);
}

gchar *
gam_toggle_get_display_name (GamToggle *gam_toggle)
{
    gchar *key, *name;

    g_return_val_if_fail (GAM_IS_TOGGLE (gam_toggle), NULL);

    key = g_strdup_printf ("/gnome-alsamixer/toggle_display_names/%s-%s=%s",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_toggle->mixer)),
                           gam_toggle_get_name (gam_toggle),
                           gam_toggle_get_name (gam_toggle));

    name = gnome_config_get_string (key);

    g_free (key);

    return name;
}

void
gam_toggle_set_display_name (GamToggle *gam_toggle, const gchar *name)
{
    gchar *key;

    g_return_if_fail (GAM_IS_TOGGLE (gam_toggle));

    key = g_strdup_printf ("/gnome-alsamixer/toggle_display_names/%s-%s",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_toggle->mixer)),
                           gam_toggle_get_name (gam_toggle));

    gnome_config_set_string (key, name);

    gnome_config_sync ();

    gtk_button_set_label (GTK_BUTTON (gam_toggle), name);
}

gboolean
gam_toggle_get_visible (GamToggle *gam_toggle)
{
    gchar *key;
    gboolean visible;

    g_return_if_fail (GAM_IS_TOGGLE (gam_toggle));

    key = g_strdup_printf ("/gnome-alsamixer/display_toggles/%s-%s=true",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_toggle->mixer)),
                           gam_toggle_get_name (gam_toggle));

    visible = gnome_config_get_bool (key);

    g_free (key);

    return visible;
}

void
gam_toggle_set_visible (GamToggle *gam_toggle, gboolean visible)
{
    gchar *key;

    g_return_if_fail (GAM_IS_TOGGLE (gam_toggle));

    key = g_strdup_printf ("/gnome-alsamixer/display_toggles/%s-%s",
                           gam_mixer_get_mixer_name (GAM_MIXER (gam_toggle->mixer)),
                           gam_toggle_get_name (gam_toggle));

    gnome_config_set_bool (key, visible);

    gnome_config_sync ();

    if (visible)
        gtk_widget_show (GTK_WIDGET (gam_toggle));
    else
        gtk_widget_hide (GTK_WIDGET (gam_toggle));
}
