/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001-2002 CodeFactory AB
 * Copyright (C) 2001-2002 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001-2002 Mikael Hallendal <micke@codefactory.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "mrp-private.h"
#include "mrp-intl.h"
#include "mrp-day.h"

struct _MrpDay {
        MrpProject *project;
        gint        id;
        gchar      *name;
        gchar      *description;
        gint        ref_count;
};

static MrpDay *work_day     = NULL;
static MrpDay *nonwork_day  = NULL;
static MrpDay *use_base_day = NULL;

static void  day_free           (MrpDay *day);

static void 
day_free (MrpDay *day)
{
        g_free (day->name);
        g_free (day->description);
        g_free (day);
}

void
imrp_day_setup_defaults (void)
{
	if (!work_day && !nonwork_day && !use_base_day) {
		work_day = mrp_day_add (NULL, 
					_("Working"), 
					_("A default working day"));
		nonwork_day = mrp_day_add (NULL,
					   _("Nonworking"),
					   _("A default non working day"));
		use_base_day = mrp_day_add (NULL,
					    _("Use base"),
					    _("Use day from base calendar"));
	}
}

MrpDay *     
mrp_day_add (MrpProject *project, const gchar *name, const gchar *description)
{     
        MrpDay *day;
        
        g_return_val_if_fail (name != NULL, NULL);

        day = g_new0 (MrpDay, 1);

        /* Weak ref? */
        day->project   = project;
        day->ref_count = 1;
        day->name      = g_strdup (name);
	day->id        = g_quark_from_string (name);
	
        if (description) {
                day->description = g_strdup (description);
        }

        if (project) {
		imrp_project_add_calendar_day (project, day);
	}

        return day;
}

GSList *
mrp_day_get_all (MrpProject *project)
{
	return imrp_project_get_calendar_days (project);
}

void
mrp_day_remove (MrpProject *project, MrpDay *day)
{
	imrp_project_remove_calendar_day (project, day);
}

gint
mrp_day_get_id (MrpDay *day)
{
        g_return_val_if_fail (day != NULL, -1);
        
        return day->id;
}

const gchar *
mrp_day_get_name (MrpDay *day)
{
        g_return_val_if_fail (day != NULL, NULL);
        
        return day->name;
}

void
mrp_day_set_name (MrpDay *day, const gchar *name)
{
        g_return_if_fail (day != NULL);
        
        g_free (day->name);
        day->name = g_strdup (name);

	if (day->project) {
		g_signal_emit_by_name (day->project,
				       "day_changed",
				       day);
	}
}

const gchar *
mrp_day_get_description (MrpDay *day)
{
        g_return_val_if_fail (day != NULL, NULL);
        
        return day->description;
}

void
mrp_day_set_description (MrpDay *day, const gchar *description)
{
        g_return_if_fail (day != NULL);
        
        g_free (day->description);
        day->description = g_strdup (description);

	if (day->project) {
		g_signal_emit_by_name (day->project,
				       "day_changed",
				       day);
	}
}

MrpDay *
mrp_day_ref (MrpDay *day)
{
        g_return_val_if_fail (day != NULL, NULL);

        day->ref_count++;

        return day;
}

void
mrp_day_unref (MrpDay *day)
{
        g_return_if_fail (day != NULL);

        day->ref_count--;
        if (day->ref_count <= 0) {
                day_free (day);
        }
}

MrpDay *
mrp_day_get_work (void)
{
        return work_day;
}

MrpDay *
mrp_day_get_nonwork (void)
{
        return nonwork_day;
}

MrpDay *
mrp_day_get_use_base (void)
{
        return use_base_day;
}
