/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001-2003 CodeFactory AB
 * Copyright (C) 2001-2003 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001-2002 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2001-2002 Alvaro del Castillo <acs@barrapunto.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>

#include <gsf/gsf-input-memory.h>
#include <gsf/gsf-input-stdio.h>

#include "mrp-error.h"
#include "mrp-intl.h"
#include "mrp-marshal.h"
#include "mrp-task-manager.h"
#include "mrp-storage-module-factory.h"
#include "mrp-storage-module.h"
#include "mrp-private.h"
#include "mrp-time.h"
#include "mrp-property.h"
#include "mrp-project.h"

struct _MrpProjectPriv {
	MrpApplication   *app;
	gchar            *uri;

	MrpTaskManager   *task_manager;

	GList            *resources;
	GList            *groups;
	
	MrpStorageModule *primary_storage;

	mrptime           project_start;

	gchar            *organization;
	gchar            *manager;
	gchar            *name;
	
	/*GList          *storage_modules;*/        /* <MrpStorageModule> */
	gboolean          needs_saving;
	gboolean          untouched;
	
	MrpGroup         *default_group;
	
	/* Property stuff */
	GParamSpecPool   *property_pool;

	/* Calendar stuff */
	MrpCalendar      *root_calendar;
	MrpCalendar      *calendar;
	GHashTable       *day_types;

	/* Project phases */
	GList            *phases;
	gchar            *phase;
};

/* Properties */
enum {
	PROP_0,
	PROP_PROJECT_START,
	PROP_NAME,
	PROP_ORGANIZATION,
	PROP_MANAGER,
	PROP_DEFAULT_GROUP,
	PROP_CALENDAR,
	PROP_PHASES,
	PROP_PHASE,
};

/* Signals */
enum {
	LOADED,
	RESOURCE_ADDED,
	RESOURCE_REMOVED,
	GROUP_ADDED,
	GROUP_REMOVED,
	DEFAULT_GROUP_CHANGED,
	TASK_INSERTED,
	TASK_REMOVED,
	TASK_MOVED,
	NEEDS_SAVING_CHANGED,
	PROPERTY_ADDED,
	PROPERTY_CHANGED,
	PROPERTY_REMOVED,
	CALENDAR_TREE_CHANGED,
	DAY_ADDED,
	DAY_REMOVED,
	DAY_CHANGED,
	LAST_SIGNAL
};


static void     project_class_init                (MrpProjectClass  *klass);
static void     project_init                      (MrpProject       *project);
static void     project_finalize                  (GObject          *object);
static void     project_set_property              (GObject          *object,
						   guint             prop_id,
						   const GValue     *value,
						   GParamSpec       *pspec);
static void     project_get_property              (GObject          *object,
						   guint             prop_id,
						   GValue           *value,
						   GParamSpec       *pspec);
static void     project_connect_object            (MrpObject        *object,
						   MrpProject       *project);
static void     project_setup_default_calendar    (MrpProject       *project);
static void     project_calendar_changed          (MrpCalendar      *calendar,
						   MrpProject       *project);
static void     project_set_calendar              (MrpProject       *project,
						   MrpCalendar      *calendar);



static GObjectClass *parent_class;
static guint signals[LAST_SIGNAL];

GType
mrp_project_get_type (void)
{
	static GType object_type = 0;

	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof (MrpProjectClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) project_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (MrpProject),
			0,              /* n_preallocs */
			(GInstanceInitFunc) project_init,
		};

		object_type = g_type_register_static (MRP_TYPE_OBJECT, 
						      "MrpProject", 
						      &object_info, 0);
	}

	return object_type;
}

static void
project_class_init (MrpProjectClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize     = project_finalize;
	object_class->set_property = project_set_property;
	object_class->get_property = project_get_property;

	signals[LOADED] = g_signal_new
		("loaded",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0, /*G_STRUCT_OFFSET (MrpProjectClass, method), */ 
		 NULL, NULL,
		 mrp_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);

	signals[RESOURCE_ADDED] = g_signal_new
		("resource_added",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0, /*G_STRUCT_OFFSET (MrpProjectClass, method), */
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_RESOURCE);

	signals[RESOURCE_REMOVED] = g_signal_new
		("resource_removed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0, /*G_STRUCT_OFFSET (MrpProjectClass, method), */
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_RESOURCE);
	
	signals[GROUP_ADDED] = g_signal_new
		("group_added",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0, /*G_STRUCT_OFFSET (MrpProjectClass, method), */
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_GROUP);

	signals[GROUP_REMOVED] = g_signal_new
		("group_removed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0, /*G_STRUCT_OFFSET (MrpProjectClass, method), */
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_GROUP);

	signals[DEFAULT_GROUP_CHANGED] = g_signal_new
		("default_group_changed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0, /*G_STRUCT_OFFSET (MrpProjectClass, method), */
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_GROUP);
	
	signals[TASK_INSERTED] = g_signal_new
		("task_inserted",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_TASK);

	signals[TASK_REMOVED] = g_signal_new
		("task_removed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_TASK);

	signals[TASK_MOVED] = g_signal_new
		("task_moved",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_TASK);

	signals[NEEDS_SAVING_CHANGED] = g_signal_new
		("needs_saving_changed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__BOOLEAN,
		 G_TYPE_NONE,
		 1, G_TYPE_BOOLEAN);

	signals[PROPERTY_ADDED] = g_signal_new
		("property_added",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__LONG_POINTER,
		 G_TYPE_NONE,
		 2, G_TYPE_LONG, G_TYPE_POINTER);
		
	signals[PROPERTY_CHANGED] = g_signal_new
		("property_changed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1, G_TYPE_POINTER);
		
	signals[PROPERTY_REMOVED] = g_signal_new
		("property_removed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1, G_TYPE_POINTER);

	signals[CALENDAR_TREE_CHANGED] = g_signal_new 
		("calendar_tree_changed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__OBJECT,
		 G_TYPE_NONE,
		 1, MRP_TYPE_CALENDAR);
	
	signals[DAY_ADDED] = g_signal_new
		("day_added",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1, G_TYPE_POINTER);
	
	signals[DAY_REMOVED] = g_signal_new
		("day_removed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1, G_TYPE_POINTER);
	
	signals[DAY_CHANGED] = g_signal_new
		("day_changed",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 0,
		 NULL, NULL,
		 mrp_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1, G_TYPE_POINTER);

	/* Properties. */
	g_object_class_install_property (object_class,
					 PROP_PROJECT_START,
					 mrp_param_spec_time ("project-start",
							      "Project start",
							      "The start date of the project",
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_NAME,
					 g_param_spec_string ("name",
							      "Project Name",
							      NULL,
							      "",
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_ORGANIZATION,
					 g_param_spec_string ("organization",
							      "Organization",
							      NULL,
							      "",
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_MANAGER,
					 g_param_spec_string ("manager",
							      "Project Manager",
							      NULL,
							      "",
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_DEFAULT_GROUP,
					 g_param_spec_object ("default-group",
							      "Default Group",
							      "Default group for new resources",
							      MRP_TYPE_GROUP,
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_CALENDAR,
					 g_param_spec_object ("calendar",
							      "Calendar",
							      "The calendar used in the project",
							      MRP_TYPE_CALENDAR,
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_PHASES,
					 g_param_spec_pointer ("phases",
							       "Project Phases",
							       "The various phases the project can be in",
							       G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class,
					 PROP_PHASE,
					 g_param_spec_string ("phase",
							      "Project Phase",
							      "The phase the project is in",
							      "",
							      G_PARAM_READWRITE));
}

static void
project_init (MrpProject *project)
{
	MrpProjectPriv *priv;
	MrpProperty    *property;
	MrpCalendar    *calendar;
	
	project->priv = g_new0 (MrpProjectPriv, 1);

	priv = project->priv;

	priv->needs_saving  = FALSE;
	priv->untouched     = TRUE;
	priv->project_start = mrp_time_align_day (mrp_time_current_time ());
	priv->resources     = NULL;
	priv->groups        = NULL;
	priv->organization  = g_strdup ("");;
	priv->manager       = g_strdup ("");;
	priv->name          = g_strdup ("");

	priv->property_pool = g_param_spec_pool_new (TRUE);
	priv->task_manager  = mrp_task_manager_new (project);

	priv->root_calendar = g_object_new (MRP_TYPE_CALENDAR,
					    "name", "-",
					    "project", project,
					    NULL);

	calendar = mrp_calendar_derive ("Default", priv->root_calendar);
	project_set_calendar (project, calendar);
	
	project_setup_default_calendar (project);
	priv->day_types = g_hash_table_new_full (
		NULL, NULL, NULL,
		(GDestroyNotify) mrp_day_unref);
	
        /* Move these into the cost plugin */
	property = mrp_property_new ("cost", 
				     MRP_PROPERTY_TYPE_COST,
				     "Cost",
				     "standard cost for a resource",
				     TRUE);
	mrp_project_add_property (project,
				  MRP_TYPE_RESOURCE,
				  property,
                                  TRUE);

	/* FIXME: Add back this when it's used. */
/*	property = mrp_property_new ("cost_overtime", 
	MRP_PROPERTY_TYPE_COST,
	"Overtime Cost",
	"overtime cost for a resource");
	mrp_project_add_property (project,
	MRP_TYPE_RESOURCE,
	property,
	FALSE);		
*/

	/* FIXME: Small temporary hack to get custom properties working on
	 * project.
	 */
	g_object_set (project, "project", project, NULL);

	priv->default_group = NULL;
}

static void
project_finalize (GObject *object)
{
	MrpProject *project = MRP_PROJECT (object);

	g_object_unref (project->priv->primary_storage);
	g_object_unref (project->priv->task_manager);
	
	g_free (project->priv->uri);
	g_free (project->priv);

	if (G_OBJECT_CLASS (parent_class)->finalize) {
		(* G_OBJECT_CLASS (parent_class)->finalize) (object);
	}
}

static void
project_set_property (GObject      *object,
		      guint         prop_id,
		      const GValue *value,
		      GParamSpec   *pspec)
{
	MrpProject     *project;
	MrpProjectPriv *priv;
	MrpGroup       *group;
	MrpCalendar    *calendar;
	
	project = MRP_PROJECT (object);
	priv    = project->priv;
	
	switch (prop_id) {
	case PROP_PROJECT_START:
		priv->project_start = g_value_get_long (value);
		imrp_project_set_needs_saving (project, TRUE);
		break;

	case PROP_ORGANIZATION:
		g_free (priv->organization);
		priv->organization = g_strdup (g_value_get_string (value));
		imrp_project_set_needs_saving (project, TRUE);
		break;

	case PROP_MANAGER:
		g_free (priv->manager);
		priv->manager = g_strdup (g_value_get_string (value));
		imrp_project_set_needs_saving (project, TRUE);
		break;

	case PROP_NAME:
		g_free (priv->name);
		priv->name = g_strdup (g_value_get_string (value));
		imrp_project_set_needs_saving (project, TRUE);
		break;

	case PROP_DEFAULT_GROUP:
		group = g_value_get_object (value);
		if (priv->default_group == group) {
			break;
		}
		
		if (priv->default_group) {
			g_object_unref (priv->default_group);
		}
		
		priv->default_group = group;

		if (priv->default_group) {
			g_object_ref (priv->default_group);
		}

		/* FIXME: isn't the notify enough, why do we have a signal for
		 * this one, but not the others?
		 */
		g_signal_emit (project,
			       signals[DEFAULT_GROUP_CHANGED], 
			       0,
			       priv->default_group);
		imrp_project_set_needs_saving (project, TRUE);
		break;

	case PROP_CALENDAR:
		calendar = g_value_get_object (value);
		project_set_calendar (project, calendar);
		imrp_project_set_needs_saving (project, TRUE);
		break;

	case PROP_PHASES: 
		mrp_string_list_free (priv->phases);
		priv->phases = mrp_string_list_copy (g_value_get_pointer (value));
		imrp_project_set_needs_saving (project, TRUE);
		break;

	case PROP_PHASE:
		g_free (priv->phase);
		priv->phase = g_value_dup_string (value);
		imrp_project_set_needs_saving (project, TRUE);
		break;

	default:
		break;
	}
}

static void
project_get_property (GObject    *object,
		      guint       prop_id,
		      GValue     *value,
		      GParamSpec *pspec)
{
	MrpProject     *project;
	MrpProjectPriv *priv;
	
	project = MRP_PROJECT (object);
	priv    = project->priv;
	
	switch (prop_id) {
	case PROP_PROJECT_START:
		g_value_set_long (value, priv->project_start);
		break;

	case PROP_ORGANIZATION:
		g_value_set_string (value, priv->organization);
		break;

	case PROP_NAME:
		g_value_set_string (value, priv->name);
		break;

	case PROP_MANAGER:
		g_value_set_string (value, priv->manager);
		break;

	case PROP_DEFAULT_GROUP:
		g_value_set_object (value, priv->default_group);
		break;

	case PROP_CALENDAR:
		g_value_set_object (value, priv->calendar);
		break;

	case PROP_PHASES:
		g_value_set_pointer (value, mrp_string_list_copy (priv->phases));
		break;

	case PROP_PHASE:
		g_value_set_string (value, priv->phase);
		break;

	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break; 
	}
}

static void
project_connect_object (MrpObject *object, MrpProject *project)
{
	g_object_set (object, "project", project, NULL);
}

MrpProject *
mrp_project_new (MrpApplication *app)
{
	MrpProject              *project;
	MrpProjectPriv          *priv;
	MrpStorageModuleFactory *factory;
	MrpStorageModule        *module;

	project = g_object_new (MRP_TYPE_PROJECT, NULL);
	priv = project->priv;
	
	priv->app = app;

	/* REMOVE THIS LATER */
	factory = mrp_storage_module_factory_get ("mrproject-1");
	if (factory == NULL) {
		g_error ("FIXME: Could not find storage module for method %s.",
			 "mrproject-1");
	}
	
	module = mrp_storage_module_factory_create_module (factory);
	if (module == NULL) {
		g_error ("FIXME: Could not create module for method %s.",
			 "mrproject-1");
	}

	g_type_module_unuse (G_TYPE_MODULE (factory));
	
	priv->primary_storage = module;
	
	imrp_storage_module_set_project (module, project);

	/* END OF REMOVAL */
	project->priv->app = app;

	/* FIXME: Is this right? The creating of the default calendar changes
	 * the project and sets the needs_saving flag. Not the cleanest solution
	 * but it works: unset it after creating the project. Bug #416.
	 */
	imrp_project_set_needs_saving (project, FALSE);
	priv->untouched = TRUE;
	
	return project;
}

MrpProject *
mrp_project_open (MrpApplication  *app,
		  const gchar     *uri,
		  GError         **error)
{
	MrpProject *project;
	
	project = mrp_project_new (app);
	
	if (!mrp_project_load (project, uri, error)) {
		g_object_unref (project);
		return NULL;
	}

	return project;
}

gboolean
mrp_project_load (MrpProject   *project,
		  const gchar  *uri,
		  GError      **error)
{
	MrpProjectPriv *priv;
	GsfInput       *input;
	GSList         *l;
	MrpCalendar    *old_default_calendar;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), FALSE);
	g_return_val_if_fail (uri != NULL, FALSE);

	priv = project->priv;

	/* Small hack for now. We will load the project, then remove the default
	 * calendar.
	 */
	old_default_calendar = priv->calendar;
	
 	input = GSF_INPUT (gsf_input_mmap_new (uri, NULL));
	
 	if (!input) {
		input = GSF_INPUT (gsf_input_stdio_new (uri, error));
	}

	if (!input) {
		/* FIXME: Set error. */
		g_warning ("Couldn't find file '%s'.", uri);
		return FALSE;
	}

	mrp_task_manager_set_block_scheduling (priv->task_manager, TRUE);
	
	l = mrp_application_get_all_file_readers (priv->app);
	for (; l; l = l->next) {
		MrpFileReader *reader = l->data;

		if (mrp_file_reader_read (reader, input, project, error)) {
			g_object_unref (input);
			
			g_signal_emit (project, signals[LOADED], 0, NULL);
			imrp_project_set_needs_saving (project, FALSE);

			/*g_free (priv->uri);*/
			priv->uri = g_strdup (uri);

			/* Remove old calendar. */
			mrp_calendar_remove (old_default_calendar);

			mrp_task_manager_set_block_scheduling (priv->task_manager, FALSE);

			/* FIXME: See bug #416. */
			imrp_project_set_needs_saving (project, FALSE);
			
			return TRUE;
		}
	}

	mrp_task_manager_set_block_scheduling (priv->task_manager, FALSE);

	g_object_unref (input);

	g_set_error (error, 
		     MRP_ERROR,
		     MRP_ERROR_NO_FILE_MODULE,
		     _("Couldn't find a suitable file module for loading '%s'"),
		     uri);

	return FALSE;
}

gboolean
mrp_project_save (MrpProject *project, GError **error)
{
	MrpProjectPriv *priv;
	gboolean        success;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), FALSE);
	
	priv = project->priv;

	if (!priv->needs_saving) {
		return TRUE;
	}
	
	/* Sanity check the URI. */
	if (priv->uri == NULL) {
		return FALSE;
	}

	/* FIXME: this should be in a primary-storage interface on the module,
	 * since only modules that support that interface should be able to
	 * save a project directly.
	 */
	success = mrp_storage_module_save (priv->primary_storage, 
					   priv->uri, TRUE, error);

	if (!success) {
		return FALSE;
	}

	imrp_project_set_needs_saving (project, FALSE);

	return TRUE;
}

gboolean
mrp_project_save_as (MrpProject   *project,
		     const gchar  *uri,
		     gboolean      force,
		     GError      **error)
{
	MrpProjectPriv *priv;
	gboolean        success;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), FALSE);
	g_return_val_if_fail (uri != NULL && uri[0] != 0, FALSE);
	
	priv = project->priv;

	/* FIXME: this should be in a primary-storage interface on the module,
	 * since only modules that support that interface should be able to
	 * save a project directly.
	 */
	success = mrp_storage_module_save (priv->primary_storage, 
					   uri, force, error);
	
	if (!success) {
		/* This means that the new file exists already, or that we
		 * couldn't write to it for some reason.
		 */
		return FALSE;
	}

	g_free (priv->uri);
	priv->uri = g_strdup (uri);

	imrp_project_set_needs_saving (project, FALSE);
	
	return TRUE;
}

void
mrp_project_close (MrpProject *project)
{
	/* FIXME: impl close. */
}

MrpTaskManager *
mrp_project_get_task_manager (MrpProject *project)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	return project->priv->task_manager;
}

GList *
mrp_project_get_resources (MrpProject *project)
{
	GList *resources;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	resources = g_list_copy (project->priv->resources);
	resources = g_list_reverse (resources);
	
	return resources;
}

/* Only to be used by storage modules. */
void
mrp_project_set_resources (MrpProject *project,
			   GList      *resources)
{
	g_return_if_fail (MRP_IS_PROJECT (project));
	
	project->priv->resources = resources;

	g_list_foreach (project->priv->resources, 
			(GFunc) project_connect_object,
			project);
}

void      
mrp_project_add_resource (MrpProject *project, MrpResource *resource)
{
	MrpProjectPriv  *priv;
	MrpGroup        *group;
	MrpResourceType  type;
	
	g_return_if_fail (MRP_IS_PROJECT (project));
	g_return_if_fail (MRP_IS_RESOURCE (resource));
	
	priv = project->priv;

	priv->resources = g_list_prepend (priv->resources, resource);

	g_object_get (resource, "group", &group, NULL);
	
	if (!group) {
		g_object_set (resource, "group", priv->default_group, NULL);
	}

	g_object_get (resource, "type", &type, NULL);
	
	if (type == MRP_RESOURCE_TYPE_NONE) {
		g_object_set (resource, "type", MRP_RESOURCE_TYPE_WORK, NULL);
	}

	project_connect_object (MRP_OBJECT (resource), project);
	
	g_signal_emit (project, signals[RESOURCE_ADDED], 0, resource);

	imrp_project_set_needs_saving (project, TRUE);
}

void
mrp_project_remove_resource (MrpProject *project, MrpResource *resource)
{
	MrpProjectPriv *priv;
	
	g_return_if_fail (MRP_IS_PROJECT (project));
	g_return_if_fail (MRP_IS_RESOURCE (resource));
	
	priv = project->priv;
	
	mrp_object_removed (MRP_OBJECT (resource));

	priv->resources = g_list_remove (priv->resources, resource);

	g_signal_emit (project, signals[RESOURCE_REMOVED], 0, resource);

	imrp_project_set_needs_saving (project, TRUE);
}

GList *
mrp_project_get_groups (MrpProject *project)
{
	GList *groups;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);
	
	groups = g_list_copy (project->priv->groups);
	groups = g_list_reverse (groups);

	return groups;
}

/* Only to be used by storage modules. */
void
mrp_project_set_groups (MrpProject *project,
			GList      *groups)
{
	g_return_if_fail (MRP_IS_PROJECT (project));
	
	project->priv->groups = groups;

	g_list_foreach (project->priv->groups, 
			(GFunc) project_connect_object,
			project);
	
}

void
mrp_project_add_group (MrpProject *project, MrpGroup *group)
{
	MrpProjectPriv *priv;
	
	g_return_if_fail (MRP_IS_PROJECT (project));
	g_return_if_fail (MRP_IS_GROUP (group));
	
	priv = project->priv;

	priv->groups = g_list_prepend (priv->groups, group);

	g_object_set (group, "project", project, NULL);

	project_connect_object (MRP_OBJECT (group), project);

	g_signal_emit (project, signals[GROUP_ADDED], 0, group);

	imrp_project_set_needs_saving (project, TRUE);
}

void
mrp_project_remove_group (MrpProject *project, MrpGroup *group)
{
	MrpProjectPriv *priv;
	
	g_return_if_fail (MRP_IS_PROJECT (project));
	g_return_if_fail (MRP_IS_GROUP (group));
	
	priv = project->priv;
	
	if (priv->default_group && priv->default_group == group) {
		priv->default_group = NULL;
	}
	
	priv->groups = g_list_remove (priv->groups, group);

	g_signal_emit (project, signals[GROUP_REMOVED], 0, group);

	mrp_object_removed (MRP_OBJECT (group));

	imrp_project_set_needs_saving (project, TRUE);
}

gboolean
mrp_project_is_untouched (MrpProject *project)
{
	return project->priv->untouched;
}

gboolean
mrp_project_needs_saving (MrpProject *project)
{
	return project->priv->needs_saving;
}

const gchar *    
mrp_project_get_uri (MrpProject *project)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	return project->priv->uri;
}

mrptime
mrp_project_get_project_start (MrpProject *project)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), MRP_TIME_INVALID);

	return project->priv->project_start;
}

void
mrp_project_set_project_start (MrpProject *project,
			       mrptime     start)
{
	g_return_if_fail (MRP_IS_PROJECT (project));

	project->priv->project_start = start;
}
 
gboolean
imrp_project_add_calendar_day (MrpProject *project, MrpDay *day)
{
	MrpProjectPriv *priv;
	MrpDay         *tmp_day;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), -1);
	g_return_val_if_fail (day != NULL, -1);

	priv = project->priv;
	
	tmp_day = (MrpDay *) g_hash_table_lookup (priv->day_types,
						  GINT_TO_POINTER (mrp_day_get_id (day)));
		
	if (tmp_day) {
		g_warning ("Trying to add already present day type: '%s'",
			   mrp_day_get_name (tmp_day));
		return FALSE;
	}

	g_hash_table_insert (priv->day_types,
			     GINT_TO_POINTER (mrp_day_get_id (day)), 
			     mrp_day_ref (day));
	
	g_signal_emit (project,
		       signals[DAY_ADDED],
		       0,
		       day);

	imrp_project_set_needs_saving (project, TRUE);

	return TRUE;
}

static void
foreach_day_add_to_list (gpointer key, MrpDay *day, GSList **list)
{
	*list = g_slist_prepend (*list, day);
}

GSList *
imrp_project_get_calendar_days (MrpProject *project)
{
	MrpProjectPriv *priv;
	GSList         *ret_val = NULL;

	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	priv = project->priv;

	g_hash_table_foreach (priv->day_types,
			      (GHFunc) foreach_day_add_to_list,
			      &ret_val);
	
	return ret_val;
}

static void
project_day_fallback_when_removed (MrpCalendar *calendar,
				   MrpDay      *day)
{
	GSList      *children, *l;
	MrpCalendar *child;
		
	children = mrp_calendar_get_children (calendar);
	for (l = children; l; l = l->next) {
		child = l->data;

		imrp_calendar_replace_day (child, day, mrp_day_get_work ());
		project_day_fallback_when_removed (child, day);
	}
}

void
imrp_project_remove_calendar_day (MrpProject *project, MrpDay *day)
{
	MrpProjectPriv *priv;

	g_return_if_fail (MRP_IS_PROJECT (project));
	g_return_if_fail (day != NULL);
	
	priv = project->priv;

	/* Check if the day type is used in a calendar, if so fallback to
	 * working day.
	 */
	project_day_fallback_when_removed (priv->root_calendar, day);
	
	g_signal_emit (project,
		       signals[DAY_REMOVED],
		       0,
		       day);
	
	g_hash_table_remove (priv->day_types, 
			     GINT_TO_POINTER (mrp_day_get_id (day)));

	imrp_project_set_needs_saving (project, TRUE);
}

void
imrp_project_signal_calendar_tree_changed (MrpProject *project)
{
	MrpProjectPriv *priv;
	
	if (!project) {
		return;
	}
	
	priv = project->priv;

	g_signal_emit (project, 
		       signals[CALENDAR_TREE_CHANGED],
		       0,
		       priv->root_calendar);
}

void
imrp_project_set_needs_saving (MrpProject *project, gboolean needs_saving) 
{
	MrpProjectPriv *priv;
	
	g_return_if_fail (MRP_IS_PROJECT (project));

	priv = project->priv;

	if (needs_saving == project->priv->needs_saving) {
		return;
	}

	if (needs_saving == TRUE) {
		priv->untouched = FALSE;
	}

	project->priv->needs_saving = needs_saving;

	g_signal_emit (project, 
		       signals[NEEDS_SAVING_CHANGED], 
		       0, 
		       needs_saving);
}

/* Task related functions.
 */

GList *
mrp_project_get_all_tasks (MrpProject *project)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	return mrp_task_manager_get_all_tasks (project->priv->task_manager);
}

void
mrp_project_insert_task (MrpProject *project,
			 MrpTask    *parent,
			 gint        position,
			 MrpTask    *task)
{
	g_return_if_fail (MRP_IS_PROJECT (project));

	mrp_task_manager_insert_task (project->priv->task_manager,
				      parent,
				      position,
				      task);

	g_object_set (MRP_OBJECT (task), "project", project, NULL);
}

void
mrp_project_remove_task (MrpProject *project,
			 MrpTask    *task)
{
	g_return_if_fail (MRP_IS_PROJECT (project));

	mrp_object_removed (MRP_OBJECT (task));

	mrp_task_manager_remove_task (project->priv->task_manager,
				      task);
	
	g_signal_emit (project, signals[TASK_REMOVED], 0, task);

	imrp_project_set_needs_saving (project, TRUE);
}

gboolean
mrp_project_move_task (MrpProject  *project,
		       MrpTask     *task,
		       MrpTask     *sibling,
		       MrpTask     *parent,
		       gboolean     before,
		       GError     **error)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), FALSE);
	g_return_val_if_fail (MRP_IS_TASK (task), FALSE);
	g_return_val_if_fail (sibling == NULL || MRP_IS_TASK (sibling), FALSE);
	g_return_val_if_fail (MRP_IS_TASK (parent), FALSE);

	return mrp_task_manager_move_task (project->priv->task_manager,
					   task,
					   sibling,
					   parent,
					   before,
					   error);
}

void
imrp_project_task_moved (MrpProject *project,
			 MrpTask    *task)
{
	g_signal_emit (project, signals[TASK_MOVED], 0, task);

	imrp_project_set_needs_saving (project, TRUE);
}

MrpTask *
mrp_project_get_root_task (MrpProject *project)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	return mrp_task_manager_get_root (project->priv->task_manager);
}

GNode *
mrp_project_get_task_tree (MrpProject *project)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	return mrp_task_manager_get_task_tree (project->priv->task_manager);
}

void
mrp_project_task_traverse (MrpProject          *project,
			   MrpTask             *root,
			   MrpTaskTraverseFunc  func,
			   gpointer             user_data)
{
	g_return_if_fail (MRP_IS_PROJECT (project));

	mrp_task_manager_traverse (project->priv->task_manager,
				   root,
				   func,
				   user_data);
}

void
mrp_project_recalc_tasks (MrpProject *project)
{
	g_return_if_fail (MRP_IS_PROJECT (project));

	mrp_task_manager_recalc (project->priv->task_manager, FALSE);
}

gint
mrp_project_calculate_task_work (MrpProject *project,
				 MrpTask    *task,
				 gint        duration)
{
	g_return_val_if_fail (MRP_IS_PROJECT (project), 0);

	return mrp_task_manager_calculate_task_work (project->priv->task_manager,
						     task,
						     duration);
}

void
imrp_project_task_inserted (MrpProject *project, MrpTask *task)
{
	g_return_if_fail (MRP_IS_PROJECT (project));

	g_signal_emit (project, signals[TASK_INSERTED], 0, task);

	imrp_project_set_needs_saving (project, TRUE);
}

/* Debug function. */
void
mrp_project_dump_task_tree (MrpProject *project)
{
	g_return_if_fail (MRP_IS_PROJECT (project));

	mrp_task_manager_dump_task_tree (project->priv->task_manager);
}

MrpProperty *
imrp_project_get_property (MrpProject  *project,
			   const gchar *name,
			   GType        object_type)
{
	MrpProjectPriv *priv;
	MrpProperty    *property;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	priv = project->priv;

	property = MRP_PROPERTY (g_param_spec_pool_lookup (priv->property_pool,
							   name,
							   object_type,
							   TRUE));
	
	if (!property) {
		g_warning ("%s: object of type `%s' has no property named `%s'",
			   G_STRLOC,
			   g_type_name (object_type),
			   name);
		return NULL;
	}

	return property;
}

void
imrp_project_property_changed (MrpProject *project, MrpProperty *property)
{
	g_return_if_fail (MRP_IS_PROJECT (project));
	g_return_if_fail (property != NULL);

	g_signal_emit (project, signals[PROPERTY_CHANGED], 0, property);
	imrp_project_set_needs_saving (project, TRUE);
}

GList *
mrp_project_get_properties (MrpProject *project, GType object_type)
{
	MrpProjectPriv *priv;
	GList          *properties;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);

	priv = project->priv;

	properties = g_param_spec_pool_list_owned (priv->property_pool,
						   object_type);

	return properties;
}

void
mrp_project_add_property (MrpProject  *project,
			  GType        object_type,
			  MrpProperty *property,
                          gboolean     user_defined)
{
	MrpProjectPriv *priv;

	g_return_if_fail (MRP_IS_PROJECT (project));
	
	priv = project->priv;

	/* FIXME: Check for name conflicts. */
	if (g_param_spec_pool_lookup (priv->property_pool,
				      G_PARAM_SPEC(property)->name,
				      object_type,
				      TRUE)) {
		g_warning ("%s: Param '%s' already exists for object '%s'.",
			   G_STRLOC,
			   G_PARAM_SPEC (property)->name,
			   g_type_name (object_type));
		return;
	}

	mrp_property_set_user_defined (property, user_defined);

	g_param_spec_pool_insert (priv->property_pool,
				  G_PARAM_SPEC (property),
				  object_type);

	imrp_property_set_project (property, project);

	g_signal_emit (project, signals[PROPERTY_ADDED], 0, 
		       object_type, 
		       property);

        if (user_defined) {
                imrp_project_set_needs_saving (project, TRUE);
        }
}

void
mrp_project_remove_property (MrpProject  *project,
			     GType        object_type,
			     const gchar *name)
{
	MrpProjectPriv *priv;
	MrpProperty    *property;
	
	g_return_if_fail (MRP_IS_PROJECT (project));
	
	priv = project->priv;

	property = imrp_project_get_property (project, name, object_type);

	if (!property) {
		g_warning ("%s: object of type '%s' has no property named '%s'",
			   G_STRLOC,
			   g_type_name (object_type),
			   name);
		return;
	}

	g_signal_emit (project, signals[PROPERTY_REMOVED], 0, property);
	
	g_param_spec_pool_remove (priv->property_pool,
				  G_PARAM_SPEC (property));

	imrp_project_set_needs_saving (project, TRUE);
}

gboolean
mrp_project_has_property (MrpProject  *project,
			  GType        owner_type,
			  const gchar *name)
{
	MrpProjectPriv *priv;

	g_return_val_if_fail (MRP_IS_PROJECT (project), FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	
	priv = project->priv;

	return NULL != g_param_spec_pool_lookup (priv->property_pool,
						 name,
						 owner_type,
						 TRUE);
}

MrpCalendar *
mrp_project_get_root_calendar (MrpProject *project)
{
	MrpProjectPriv *priv;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);
	
	priv = project->priv;
	
	return priv->root_calendar;
}

MrpCalendar *
mrp_project_get_calendar (MrpProject *project)
{
	MrpProjectPriv *priv;
	
	g_return_val_if_fail (MRP_IS_PROJECT (project), NULL);
	
	priv = project->priv;
	
	return priv->calendar;
}

static void
project_setup_default_calendar (MrpProject *project)
{
	MrpProjectPriv *priv;
	MrpInterval    *ival;
	GSList         *ivals = NULL;

	priv = project->priv;
	
	mrp_calendar_set_default_days (
		priv->calendar,
		MRP_CALENDAR_DAY_MON, mrp_day_get_work (),
		MRP_CALENDAR_DAY_TUE, mrp_day_get_work (),
		MRP_CALENDAR_DAY_WED, mrp_day_get_work (),
		MRP_CALENDAR_DAY_THU, mrp_day_get_work (),
		MRP_CALENDAR_DAY_FRI, mrp_day_get_work (),
		MRP_CALENDAR_DAY_SAT, mrp_day_get_nonwork (),
		MRP_CALENDAR_DAY_SUN, mrp_day_get_nonwork (),
		-1);
	
	ival = mrp_interval_new (8 * 60*60, 12 * 60*60);
	ivals = g_slist_append (ivals, ival);
	
	ival = mrp_interval_new (13 * 60*60, 17 * 60*60);
	ivals = g_slist_append (ivals, ival);
	
	mrp_calendar_day_set_intervals (priv->calendar, mrp_day_get_work (), ivals);

	g_slist_foreach (ivals, (GFunc) mrp_interval_unref, NULL);
	g_slist_free (ivals);
}

static void
project_calendar_changed (MrpCalendar *calendar,
			  MrpProject  *project)
{
	MrpProjectPriv *priv;

	priv = project->priv;

	mrp_task_manager_recalc (priv->task_manager, TRUE);
}

static void
project_set_calendar (MrpProject  *project,
		      MrpCalendar *calendar)
{
	MrpProjectPriv *priv;

	priv = project->priv;

	if (priv->calendar) {
		g_signal_handlers_disconnect_by_func (priv->calendar,
						      project_calendar_changed,
						      project);
		g_object_unref (priv->calendar);
	}

	if (calendar) {
		priv->calendar = g_object_ref (calendar);

		g_signal_connect_object (calendar,
					 "calendar_changed",
					 G_CALLBACK (project_calendar_changed),
					 project,
					 0);
	}

	mrp_task_manager_recalc (priv->task_manager, TRUE);
}

