/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2002 CodeFactory AB
 * Copyright (C) 2002 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2002 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2002 Alvaro del Castillo <acs@barrapunto.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "mrp-private.h"
#include "mrp-intl.h"
#include "mrp-project.h"
#include "mrp-property.h"

/* Quarks */
#define LABEL        "label"
#define PROJECT      "project"
#define DESCRIPTION  "description"
#define TYPE         "type"
#define USER_DEFINED "user_defined"

static void         property_set_type               (MrpProperty     *property,
						     MrpPropertyType  type);
static GParamSpec * property_param_spec_string_list (const gchar     *name,
						     const gchar     *nick,
						     const gchar     *blurb,
						     GParamFlags      flags);


static void
property_set_type (MrpProperty *property, MrpPropertyType type)
{
	g_param_spec_set_qdata (G_PARAM_SPEC (property),
				g_quark_from_static_string (TYPE),
				GINT_TO_POINTER (type));
}

void
imrp_property_set_project (MrpProperty *property, MrpProject *project)
{
	g_param_spec_set_qdata (G_PARAM_SPEC (property),
				g_quark_from_static_string (PROJECT),
				project);
}

MrpProperty *
mrp_property_new (const gchar     *name,
		  MrpPropertyType  type,
		  const gchar     *label,
		  const gchar     *description,
		  gboolean         user_defined)
{
	MrpProperty *property = NULL;

	switch (type) {
	case MRP_PROPERTY_TYPE_INT:
		property = MRP_PROPERTY (g_param_spec_int (name,
							   "",
							   "",
							   G_MININT,
							   G_MAXINT,
							   0,
							   G_PARAM_READWRITE));
		break;
	case MRP_PROPERTY_TYPE_FLOAT:
		property = MRP_PROPERTY (g_param_spec_float (name,
							     "",
							     "",
							     -G_MAXFLOAT,
							     G_MAXFLOAT,
							     0.0,
							     G_PARAM_READWRITE));
		break;
	case MRP_PROPERTY_TYPE_DURATION:
		property = MRP_PROPERTY (g_param_spec_int (name,
							     "",
							     "",
							     G_MININT,
							     G_MAXINT,
							     0,
							     G_PARAM_READWRITE));
		break;	
	case MRP_PROPERTY_TYPE_STRING:
		property = MRP_PROPERTY (g_param_spec_string (name,
							      "",
							      "",
							      NULL,
							      G_PARAM_READWRITE));
		break;
	case MRP_PROPERTY_TYPE_STRING_LIST:
		property = MRP_PROPERTY (property_param_spec_string_list (name,
									  "",
									  "",
									  G_PARAM_READWRITE));
		break;
	case MRP_PROPERTY_TYPE_DATE:
		property = MRP_PROPERTY (mrp_param_spec_time (name,
							      "",
							      "",
							      G_PARAM_READWRITE));
		break;
	case MRP_PROPERTY_TYPE_COST:
		property = MRP_PROPERTY (g_param_spec_float (name,
							     "",
							     "",
							     -G_MAXFLOAT,
							     G_MAXFLOAT,
							     0.0,
							     G_PARAM_READWRITE));
		break;
	default:
		break;
	};
	
	if (!property) {
		return NULL;
	}

	property_set_type (property, type);
	g_param_spec_set_qdata_full (G_PARAM_SPEC (property),
				     g_quark_from_static_string (LABEL),
				     g_strdup (label), 
				     g_free);
	g_param_spec_set_qdata_full (G_PARAM_SPEC (property),
				     g_quark_from_static_string (DESCRIPTION),
				     g_strdup (description), 
				     g_free);
	g_param_spec_set_qdata_full (G_PARAM_SPEC (property),
				     g_quark_from_static_string (USER_DEFINED),
				     GINT_TO_POINTER (user_defined),
				     NULL);
	return property;
}

const gchar *
mrp_property_get_name (MrpProperty *property)
{
	g_return_val_if_fail (property != NULL, NULL);
	
	return G_PARAM_SPEC (property)->name;
}

MrpPropertyType 
mrp_property_get_type (MrpProperty *property)
{
	g_return_val_if_fail (property != NULL, MRP_PROPERTY_TYPE_NONE);
	
	return GPOINTER_TO_INT (
		g_param_spec_get_qdata (G_PARAM_SPEC (property),
					g_quark_from_static_string (TYPE)));
}

void
mrp_property_set_label (MrpProperty *property, const gchar *label)
{
	gpointer project;
	
	g_param_spec_set_qdata_full (G_PARAM_SPEC (property),
				     g_quark_from_static_string (LABEL),
				     g_strdup (label), 
				     g_free);

	project = g_param_spec_get_qdata (G_PARAM_SPEC (property),
					  g_quark_from_static_string (PROJECT));
	if (project) {
		imrp_project_property_changed (MRP_PROJECT (project), 
					       property);
	}
}

const gchar *
mrp_property_get_label (MrpProperty *property)
{
	g_return_val_if_fail (property != NULL, NULL);

	return ((const gchar *) 
		g_param_spec_get_qdata (G_PARAM_SPEC (property),
					g_quark_from_static_string (LABEL)));
}

void
mrp_property_set_description (MrpProperty *property, const gchar *description)
{
	gpointer project;
	
	g_param_spec_set_qdata_full (G_PARAM_SPEC (property),
				     g_quark_from_static_string (DESCRIPTION), 
				     g_strdup (description), 
				     g_free);

	project = g_param_spec_get_qdata (G_PARAM_SPEC (property), 
					  g_quark_from_static_string (PROJECT));
	if (project) {
		imrp_project_property_changed (MRP_PROJECT (project), 
					       property);
	}
}

const gchar *
mrp_property_get_description (MrpProperty *property)
{
	g_return_val_if_fail (property != NULL, NULL);

	return ((const gchar *) 
		g_param_spec_get_qdata (G_PARAM_SPEC (property), 
					g_quark_from_static_string (DESCRIPTION)));
}

void
mrp_property_set_user_defined (MrpProperty *property, gboolean user_defined)
{
	g_param_spec_set_qdata_full (G_PARAM_SPEC (property),
				     g_quark_from_static_string (USER_DEFINED),
				     GINT_TO_POINTER (user_defined),
				     NULL);
}

gboolean
mrp_property_get_user_defined (MrpProperty *property)
{
	return ((gboolean)
		GPOINTER_TO_INT (g_param_spec_get_qdata (G_PARAM_SPEC (property),
							 g_quark_from_static_string (USER_DEFINED))));
}

MrpProperty *
mrp_property_ref (MrpProperty *property)
{
	g_return_val_if_fail (property != NULL, NULL);
	
	return MRP_PROPERTY (g_param_spec_ref (G_PARAM_SPEC (property)));
}

void
mrp_property_unref (MrpProperty *property)
{
	g_return_if_fail (property != NULL);
	
	g_param_spec_unref (G_PARAM_SPEC (property));
}

const gchar *
mrp_property_type_as_string (MrpPropertyType type)
{
	switch (type) {
	case MRP_PROPERTY_TYPE_STRING:
		return _("Text");
	case MRP_PROPERTY_TYPE_STRING_LIST:
		return _("String list");
	case MRP_PROPERTY_TYPE_INT:
		return _("Integer number");
	case MRP_PROPERTY_TYPE_FLOAT:
		return _("Floating-point number");
	case MRP_PROPERTY_TYPE_DATE:
		return _("Date");
	case MRP_PROPERTY_TYPE_DURATION:
		return _("Duration");
	case MRP_PROPERTY_TYPE_COST:
		return _("Cost");	
	case MRP_PROPERTY_TYPE_NONE:
		g_warning ("Requested name for type 'none'.");
		return _("None");
	default:
		g_assert_not_reached ();
	}

	return NULL;
}

static GParamSpec *
property_param_spec_string_list (const gchar *name,
				 const gchar *nick,
				 const gchar *blurb,
				 GParamFlags flags)
{
	GParamSpec *spec;
	
	spec = g_param_spec_string (name,
				    nick,
				    blurb,
				    NULL,
				    flags);

	return g_param_spec_value_array (name,
					 nick,
					 blurb,
					 spec,
					 flags);
}
