/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001-2002 CodeFactory AB
 * Copyright (C) 2001-2002 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001-2002 Mikael Hallendal <micke@codefactory.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include "mrp-marshal.h"
#include "mrp-intl.h"
#include "mrp-private.h"
#include "mrp-resource.h"
#include "mrp-error.h"

/* Properties */
enum {
	PROP_0,
	PROP_NAME,
	PROP_START,
	PROP_FINISH,
	PROP_LATEST_START,
	PROP_LATEST_FINISH,
	PROP_DURATION,
	PROP_WORK,
	PROP_CRITICAL,
	PROP_TYPE,
	PROP_SCHED,
	PROP_CONSTRAINT,
	PROP_NOTE,
	PROP_PERCENT_COMPLETE
};

/* Signals */
enum {
	RELATION_ADDED,
	RELATION_REMOVED,
	TASK_MOVED,
	ASSIGNMENT_ADDED,
	ASSIGNMENT_REMOVED,
	CHILD_ADDED,
	CHILD_REMOVED,
	LAST_SIGNAL
};

struct _MrpTaskPriv {
	guint             critical : 1;

	/* Used for topological order sorting. */
	guint             visited : 1;
	GNode            *sorted_node;

	/* FIXME: This might be a mistake... I can't think of any other types,
	 * besides milestone and normal. Should we have a boolean instead,
	 * is_milestone? That or a flags type variable.
	 */
	MrpTaskType       type;

	/* FIXME: See above, could use a flags type and keep this info there as
	 * well.
	 */
	MrpTaskSched      sched;
	
	/* Percent complete, 0-100. */
	gshort            percent_complete;
	
	gchar            *name;
	gchar            *note;

	/* The amount of work effort (duration*units) necessary to complete this
	 * task.
	 */
	gint              work;

	/* Calculated duration. */
	gint              duration;

	/* Slack. */
	gint              slack;
	
	/* Calculated start and finish values. */
	mrptime           start;
	mrptime           finish;

	/* The time of the first work performed on the task. Needed so clients
	 * know when to start drawing if they don't want to draw from the start
	 * if the start is during non-work time.
	 */
	mrptime           work_start;

	/* Represents the Work Breakdown Structure tree hierarchy. */
	GNode            *node;

	/* The acyclic dependency graph. */
	GList            *successors;
	GList            *predecessors;
	
	/* Calculated CPM values. */
	mrptime           latest_start;
	mrptime           latest_finish;

	MrpConstraint     constraint;

	/* List of assignments. */
	GHashTable       *assignments;
};


static void task_init                  (MrpTask            *module);
static void task_class_init            (MrpTaskClass       *class);
static void task_finalize              (GObject            *object);
static void task_set_property          (GObject            *object,
					guint               prop_id,
					const GValue       *value,
					GParamSpec         *pspec);
static void task_get_property          (GObject            *object,
					guint               prop_id,
					GValue             *value,
					GParamSpec         *pspec);
static void task_removed               (MrpObject          *object);
static void task_assignment_removed_cb (MrpAssignment      *assignment,
					MrpTask            *task);

static void task_remove_assignments    (MrpTask            *task);
static void task_remove_relations      (MrpTask            *task);


static MrpObjectClass *parent_class;
static guint signals[LAST_SIGNAL];


GType
mrp_task_get_type (void)
{
	static GType object_type = 0;

	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof (MrpTaskClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) task_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (MrpTask),
			0,              /* n_preallocs */
			(GInstanceInitFunc) task_init,
		};

		object_type = g_type_register_static (MRP_TYPE_OBJECT,
						      "MrpTask",
						      &object_info, 0);
	}

	return object_type;
}

static void
task_init (MrpTask *task)
{
	MrpTaskPriv *priv;
	
	priv = g_new0 (MrpTaskPriv, 1);
	task->priv = priv;

	priv->name = g_strdup ("");
	priv->node = g_node_new (task);
	priv->sorted_node = g_node_new (task);
	priv->assignments = g_hash_table_new (NULL, NULL);
	priv->constraint.type = MRP_CONSTRAINT_ASAP;
}

static void
task_class_init (MrpTaskClass *klass)
{
	GObjectClass   *object_class     = G_OBJECT_CLASS (klass);
	MrpObjectClass *mrp_object_class = MRP_OBJECT_CLASS (klass);
	
	parent_class = MRP_OBJECT_CLASS (g_type_class_peek_parent (klass));

	object_class->finalize = task_finalize;
	object_class->set_property = task_set_property;
	object_class->get_property = task_get_property;

	mrp_object_class->removed  = task_removed;
	
	signals[TASK_MOVED] =
		g_signal_new ("task_moved",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      mrp_marshal_VOID__OBJECT_INT,
			      G_TYPE_NONE, 2, MRP_TYPE_TASK, G_TYPE_INT);
	
	signals[RELATION_ADDED] =
		g_signal_new ("relation_added",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      mrp_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1, G_TYPE_POINTER);
	
	signals[RELATION_REMOVED] =
		g_signal_new ("relation_removed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      mrp_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1, G_TYPE_POINTER);

	signals[ASSIGNMENT_ADDED] = 
		g_signal_new ("assignment_added",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      mrp_marshal_VOID__OBJECT,
			      G_TYPE_NONE,
			      1, MRP_TYPE_ASSIGNMENT);

	signals[ASSIGNMENT_REMOVED] = 
		g_signal_new ("assignment_removed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      mrp_marshal_VOID__OBJECT,
			      G_TYPE_NONE,
			      1, MRP_TYPE_ASSIGNMENT);
	
	signals[CHILD_ADDED] = 
		g_signal_new ("child_added",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      mrp_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[CHILD_REMOVED] = 
		g_signal_new ("child_removed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      mrp_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	g_object_class_install_property (
		object_class,
		PROP_NAME,
		g_param_spec_string ("name",
				     "Task name",
				     NULL,
				     "",
				     G_PARAM_READWRITE));
	
	g_object_class_install_property (
		object_class,
		PROP_NOTE,
		g_param_spec_string ("note",
				     "Task note",
				     NULL,
				     "",
				     G_PARAM_READWRITE));

	g_object_class_install_property (
		object_class,
		PROP_START,
		mrp_param_spec_time ("start",
				     "Task start time",
				     NULL,
				     G_PARAM_READABLE));
	
	g_object_class_install_property (
		object_class,
		PROP_FINISH,
		mrp_param_spec_time ("finish",
				     "Task finish time",
				     NULL,
				     G_PARAM_READABLE));
	
	g_object_class_install_property (
		object_class,
		PROP_LATEST_START,
		mrp_param_spec_time ("latest_start",
				     "Latest task start time",
				     NULL,
				     G_PARAM_READABLE));
	
	g_object_class_install_property (
		object_class,
		PROP_LATEST_FINISH,
		mrp_param_spec_time ("latest_finish",
				     "Latest task finish time",
				     NULL,
				     G_PARAM_READABLE));
	
	g_object_class_install_property (
		object_class,
		PROP_DURATION,
		g_param_spec_int ("duration",
				  "Task duration",
				  NULL,
				  -1, G_MAXINT, 0,
				  G_PARAM_READWRITE));

	g_object_class_install_property (
		object_class,
		PROP_WORK,
		g_param_spec_int ("work",
				  "Task work",
				  NULL,
				  -1, G_MAXINT, 0,
				  G_PARAM_READWRITE));

	g_object_class_install_property (
		object_class,
		PROP_CRITICAL,
		g_param_spec_boolean ("critical",
				      "In critical path",
				      NULL,
				      FALSE,
				      G_PARAM_READWRITE));

	g_object_class_install_property (
		object_class,
		PROP_TYPE,
		g_param_spec_enum ("type",
				   "Task type",
				   NULL,
				   MRP_TYPE_TASK_TYPE,
				   MRP_TASK_TYPE_NORMAL,
				   G_PARAM_READWRITE));
	
	g_object_class_install_property (
		object_class,
		PROP_SCHED,
		g_param_spec_enum ("sched",
				   "Task scheduling type",
				   NULL,
				   MRP_TYPE_TASK_SCHED,
				   MRP_TASK_SCHED_NORMAL,
				   G_PARAM_READWRITE));

	g_object_class_install_property (
		object_class,
		PROP_CONSTRAINT,
		g_param_spec_boxed ("constraint",
				    "Time constraint",
				    NULL,
				    MRP_TYPE_CONSTRAINT,
				    G_PARAM_READWRITE));
	
	g_object_class_install_property (
		object_class,
		PROP_PERCENT_COMPLETE,
		g_param_spec_int ("percent_complete",
				  "Percent complete",
				  NULL,
				  0, 100, 0,
				  G_PARAM_READWRITE));
}

static void
task_finalize (GObject *object)
{
	MrpTask     *task;
	MrpTaskPriv *priv;

	task = MRP_TASK (object);
	priv = task->priv;
	
	g_free (priv->name);
	g_free (priv->note);

	/* Make sure we aren't left hanging in the tree. */
	g_assert (priv->node->parent == NULL);

	/* Make sure we don't have dangling relations. */
	g_assert (priv->predecessors == NULL);
	g_assert (priv->successors == NULL);
	
	g_node_destroy (priv->node);
	g_node_destroy (priv->sorted_node);

	g_free (priv);
	task->priv = NULL;

	if (G_OBJECT_CLASS (parent_class)->finalize) {
		(* G_OBJECT_CLASS (parent_class)->finalize) (object);
	}
}

static void
task_set_property (GObject      *object,
		   guint         prop_id,
		   const GValue *value,
		   GParamSpec   *pspec)
{
	MrpTask     *task;
	MrpTaskPriv *priv;
	const gchar *str;
	gint         i_val;
	MrpTaskType  type;
	gboolean     changed = FALSE;

	task = MRP_TASK (object);
	priv = task->priv;
	
	switch (prop_id) {
	case PROP_NAME:
		str = g_value_get_string (value);

		if (!priv->name || strcmp (priv->name, str)) {
			g_free (task->priv->name);
			priv->name = g_strdup (str);
			changed = TRUE;
		}
		
		break;

	case PROP_NOTE:
		str = g_value_get_string (value);

		if (!priv->note || strcmp (priv->note, str)) {
			g_free (task->priv->note);
			priv->note = g_strdup (str);
			changed = TRUE;
		}

		break;
		
 	case PROP_START:
		priv->start = g_value_get_long (value);
		break;

	case PROP_FINISH:
		priv->finish = g_value_get_long (value);
		break;

	case PROP_DURATION:
		if (mrp_task_get_n_children (task) > 0 ||
		    priv->type == MRP_TASK_TYPE_MILESTONE) {
			return;
		}

		if (priv->sched == MRP_TASK_SCHED_NORMAL) {
			return;
		}
		
		i_val = g_value_get_int (value);
		
		if (priv->duration != i_val) {
			priv->duration = i_val;

			g_object_notify (object, "work");
			changed = TRUE;
		}
		break;
		
	case PROP_WORK:
		if (mrp_task_get_n_children (task) > 0 ||
		    priv->type == MRP_TASK_TYPE_MILESTONE) {
			return;
		}

		i_val = g_value_get_int (value);

		if (priv->work != i_val) {
			priv->work = i_val;

			g_object_notify (object, "duration");
			changed = TRUE;
		}
		break;

	case PROP_CRITICAL:
		priv->critical = g_value_get_boolean (value);
		break;

	case PROP_TYPE:
		type = g_value_get_enum (value);
		if (type != priv->type) {
			priv->type = type;
			
			if (type == MRP_TASK_TYPE_MILESTONE) {
				priv->duration = 0;
				priv->work = 0;
			} else {
				/* FIXME: we need a way to specify a default
				 * work/duration for tasks.
				 */
				priv->duration = 60*60*8;
				priv->work = 60*60*8;
			}
			
			g_object_notify (G_OBJECT (task), "duration");
			g_object_notify (G_OBJECT (task), "work");
			
			changed = TRUE;
		}
		break;

	case PROP_SCHED:
		priv->sched = g_value_get_enum (value);
		changed = TRUE;
		break;

 	case PROP_CONSTRAINT:
		/* FIXME: compare */
		priv->constraint = *(MrpConstraint *) g_value_get_boxed (value);
		changed = TRUE;
		break;

	case PROP_PERCENT_COMPLETE:
		i_val = g_value_get_int (value);
		
		if (priv->percent_complete != i_val) {
			priv->percent_complete = i_val;
			changed = TRUE;
		}
		
		break;

	default:
		break;
	}

	if (changed) {
		mrp_object_set_needs_saving (MRP_OBJECT (object), TRUE);
	}
}

static void
task_get_property (GObject    *object,
		   guint       prop_id,
		   GValue     *value,
		   GParamSpec *pspec)
{
	MrpTask     *task;
	MrpTaskPriv *priv;

	task = MRP_TASK (object);
	priv = task->priv;
	
	switch (prop_id) {
	case PROP_NAME:
		g_value_set_string (value, priv->name);
		break;
	case PROP_NOTE:
		g_value_set_string (value, priv->note);
		break;
	case PROP_START:
		g_value_set_long (value, priv->start);
		break;
	case PROP_FINISH:
		g_value_set_long (value, priv->finish);
		break;
	case PROP_LATEST_START:
		g_value_set_long (value, priv->latest_start);
		break;
	case PROP_LATEST_FINISH:
		g_value_set_long (value, priv->latest_finish);
		break;
	case PROP_DURATION:
		g_value_set_int (value, priv->duration);
		break;
	case PROP_WORK:
		g_value_set_int (value, priv->work);
		break;
	case PROP_CRITICAL:
		g_value_set_boolean (value, priv->critical);
		break;
	case PROP_TYPE:
		g_value_set_enum (value, priv->type);
		break;
	case PROP_SCHED:
		g_value_set_enum (value, priv->sched);
		break;
	case PROP_CONSTRAINT:
		g_value_set_boxed (value, &priv->constraint);
		break;
	case PROP_PERCENT_COMPLETE:
		g_value_set_int (value, priv->percent_complete);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
task_removed (MrpObject *object)
{
	MrpTask     *task;
	MrpTaskPriv *priv;
	
	g_return_if_fail (MRP_IS_TASK (object));
	
	task = MRP_TASK (object);
	priv = task->priv;

	task_remove_assignments (task);
	task_remove_relations (task);
	
        if (MRP_OBJECT_CLASS (parent_class)->removed) {
                (* MRP_OBJECT_CLASS (parent_class)->removed) (object);
        }
}

static void
task_assignment_removed_cb (MrpAssignment *assignment, MrpTask *task)
{
	MrpTaskPriv *priv;
	MrpResource *resource;
	
	g_return_if_fail (MRP_IS_TASK (task));
	g_return_if_fail (MRP_IS_ASSIGNMENT (assignment));
	
	priv = task->priv;

	resource = mrp_assignment_get_resource (assignment);
	
	if (!resource) {
		g_warning ("Resource not found in tasks assignments list");
		return;
	}
	
	g_hash_table_remove (priv->assignments, resource);
	g_signal_emit (task, signals[ASSIGNMENT_REMOVED], 0, assignment);
	g_object_unref (assignment);

	mrp_object_set_needs_saving (MRP_OBJECT (task), TRUE);
}

static void
task_remove_relations (MrpTask *task)
{
	GList       *l, *next;
	MrpRelation *relation;
	MrpTask     *predecessor;
	MrpTask     *successor;

	g_return_if_fail (MRP_IS_TASK (task));
	
	/* Cut relations involving the task (make sure to be robust when
	 * removing during traversing).
	 */
	l = task->priv->predecessors;
	while (l) {
		next = l->next;
		relation = l->data;

		predecessor = mrp_relation_get_predecessor (relation);
		
		mrp_task_remove_predecessor (task, predecessor);

		l = next;
	}
	
	l = task->priv->successors;
	while (l) {
		next = l->next;
		relation = l->data;

		successor = mrp_relation_get_successor (relation);
		
		mrp_task_remove_predecessor (successor, task);

		l = next;
	}
}

static gboolean
foreach_remove_assignment (gpointer key, gpointer value, gpointer data)
{
	return TRUE;
}

static void
task_remove_assignments (MrpTask *task)
{
	MrpTaskPriv   *priv;
	GList         *list, *l;
	MrpAssignment *assignment;

	g_return_if_fail (MRP_IS_TASK (task));

	priv = task->priv;

	/* Remove all assignments, without corrupting the hashtable on the fly.
	 */
	list = mrp_task_get_assignments (task);

	g_hash_table_foreach_remove (priv->assignments, 
				     (GHRFunc) foreach_remove_assignment,
				     task);
	
	for (l = list; l; l = l->next) {
		assignment = l->data;
		
		g_signal_handlers_disconnect_by_func (assignment,
						      task_assignment_removed_cb,
						      task);
		mrp_object_removed (MRP_OBJECT (assignment));
	}
	
	g_list_free (list);
}

static gboolean
task_remove_subtree_cb (GNode *node, gpointer data)
{
	MrpTask     *task;
	MrpTaskPriv *priv;

	task = node->data;
	priv = task->priv;
	
	task_remove_relations (task);
	task_remove_assignments (task);

	g_node_unlink (priv->node);
	g_node_unlink (priv->sorted_node);

	g_object_unref (task);
	
	return FALSE;
}
				 
void
imrp_task_remove_subtree (MrpTask *task)
{
	MrpTask *parent;
	
	g_return_if_fail (MRP_IS_TASK (task));

	parent = NULL;
	if (task->priv->node->parent) {
		parent = task->priv->node->parent->data;
	}
	
	g_object_ref (task);
	
	/* Remove the tasks one by one using post order so we don't mess with
	 * the tree while traversing it.
	 */
	g_node_traverse (task->priv->node,
			 G_POST_ORDER,
			 G_TRAVERSE_ALL,
			 -1,
			 (GNodeTraverseFunc) task_remove_subtree_cb,
			 NULL);

	g_object_unref (task);

	if (parent) {
		g_signal_emit (parent, signals[CHILD_REMOVED], 0);
	}
}

void
imrp_task_detach (MrpTask *task)
{
	g_return_if_fail (MRP_IS_TASK (task));

	/* FIXME: Do some extra checking. */

	g_node_unlink (task->priv->node);

	/* Note: we don't unlink the sorted node, because when moving a task but
	 * keeping it under the same parent, we don't need to rebuild the
	 * tree. So when the caller wants that, it has to unlink the tree and
	 * rebuild it.
	 */
}

void
imrp_task_reattach (MrpTask  *task,
		    MrpTask  *sibling,
		    MrpTask  *parent,
		    gboolean  before)
{
	GNode *node;
	
	g_return_if_fail (MRP_IS_TASK (task));
	g_return_if_fail (sibling == NULL || MRP_IS_TASK (sibling));
	g_return_if_fail (MRP_IS_TASK (parent));

	/* FIXME: Do some extra checking. */

	if (sibling == NULL) {
		if (before) {
			node = g_node_first_child (parent->priv->node);
		} else {
			node = g_node_last_child (parent->priv->node);
		}			

		if (node) {
			sibling = node->data;
		}
	}

	if (before) {
		if (sibling != NULL) {
			g_node_insert_before (parent->priv->node,
					      sibling->priv->node,
					      task->priv->node);
		} else {
			g_node_prepend (parent->priv->node,
					task->priv->node);
		}
	} else {
		if (sibling != NULL) {
			g_node_insert_after (parent->priv->node,
					     sibling->priv->node,
					     task->priv->node);
		} else {
			g_node_append (parent->priv->node,
				       task->priv->node);
		}
	}
}

void
imrp_task_reattach_pos (MrpTask  *task,
			gint      pos,
			MrpTask  *parent)
{
	g_return_if_fail (MRP_IS_TASK (task));
	g_return_if_fail (pos >= -1);
	g_return_if_fail (MRP_IS_TASK (parent));

	/* FIXME: Do some extra checking. */

	g_node_insert (parent->priv->node,
		       pos,
		       task->priv->node);
}

MrpTask *
mrp_task_new (void)
{
	MrpTask *task;

	task = g_object_new (MRP_TYPE_TASK, NULL);

	return task;
}

void
imrp_task_insert_child (MrpTask *parent,
			gint     position,
			MrpTask *child)
{
	g_return_if_fail (MRP_IS_TASK (parent));
	g_return_if_fail (MRP_IS_TASK (child));

	if (child->priv->duration == -1) {
		child->priv->duration = parent->priv->duration;
	}
	
	g_node_insert (parent->priv->node,
		       position,
		       child->priv->node);

	g_node_prepend (parent->priv->sorted_node,
			child->priv->sorted_node);

	g_signal_emit (parent, signals[CHILD_ADDED], 0);
}

static MrpRelation *
task_get_predecessor_relation (MrpTask *task,
			       MrpTask *predecessor)
{
	GList *l;

	for (l = task->priv->predecessors; l; l = l->next) {
		MrpRelation *relation = l->data;

		if (mrp_relation_get_successor (relation) == task &&
		    mrp_relation_get_predecessor (relation) == predecessor) {
			return relation;
		}
	}

	return NULL;
}

static MrpRelation *
task_get_successor_relation (MrpTask *task,
			     MrpTask *successor)
{
	GList *l;

	for (l = task->priv->successors; l; l = l->next) {
		MrpRelation *relation = l->data;

		if (mrp_relation_get_predecessor (relation) == task &&
		    mrp_relation_get_successor (relation) == successor) {
			return relation;
		}
	}

	return NULL;
}

gboolean
mrp_task_has_relation_to (MrpTask *a, MrpTask *b)
{
	return (task_get_predecessor_relation (a, b) != NULL) ||
		(task_get_predecessor_relation (b, a) != NULL);
}

gboolean
mrp_task_has_relation (MrpTask *task)
{
	return (task->priv->predecessors != NULL ||
		task->priv->successors != NULL);
}

/**
 * mrp_task_add_predecessor:
 * @task: The task to add predecessor to
 * @predecessor: The predecessor
 * @type: Type of relation
 * @lag: Lag time. If negative, it means lead time.
 * 
 * Adds a predecessor task to a task. Depending on type, the predecessor
 * must be started or finished before task can be started or finished,
 * with an optional lag/lead.
 *
 * Return value: The relation that represents the predecessor/successor link.
 **/
MrpRelation *
mrp_task_add_predecessor (MrpTask          *task,
			  MrpTask          *predecessor,
			  MrpRelationType   type,
			  glong             lag,
			  GError          **error)
{
	MrpRelation    *relation;
	MrpProject     *project;
	MrpTaskManager *manager;
	
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	g_return_val_if_fail (MRP_IS_TASK (predecessor), NULL);

	/* Can't have more than one relation between the same two tasks. */
	if (mrp_task_has_relation_to (task, predecessor)) {
		g_set_error (error,
			     MRP_ERROR,
			     MRP_ERROR_TASK_RELATION_FAILED,
			     _("Could not add a predecessor relation, because the tasks are already related."));
		
		return NULL;
	}

	g_object_get (task, "project", &project, NULL);
	manager = mrp_project_get_task_manager (project);
	if (!mrp_task_manager_check_predecessor (manager, task, predecessor, error)) {
		return NULL;
	}
	
	relation = mrp_relation_new (task, predecessor, type, lag);

	task->priv->predecessors = g_list_prepend (task->priv->predecessors, relation);
	predecessor->priv->successors = g_list_prepend (predecessor->priv->successors, relation);

	g_signal_emit (task, signals[RELATION_ADDED], 0, relation);
	g_signal_emit (predecessor, signals[RELATION_ADDED], 0, relation);

	mrp_object_set_needs_saving (MRP_OBJECT (task), TRUE);
	mrp_object_set_needs_saving (MRP_OBJECT (predecessor), TRUE);

	return relation;
}

/**
 * mrp_task_remove_predecessor:
 * @task: The task to remove a predecessor from
 * @predecessor: The predecessor to remove
 * 
 * Removes a predecessor previously added to task.
 **/
void
mrp_task_remove_predecessor (MrpTask *task,
			     MrpTask *predecessor)
{
	MrpRelation *relation;
	
	g_return_if_fail (MRP_IS_TASK (task));
	g_return_if_fail (MRP_IS_TASK (predecessor));

	relation = task_get_predecessor_relation (task, predecessor);

	task->priv->predecessors = g_list_remove (task->priv->predecessors, relation);
	predecessor->priv->successors = g_list_remove (predecessor->priv->successors, relation);

	/* Notify. */
	mrp_object_removed (MRP_OBJECT (relation));

	g_signal_emit (task, signals[RELATION_REMOVED], 0, relation);
	g_signal_emit (predecessor, signals[RELATION_REMOVED], 0, relation);

	mrp_object_set_needs_saving (MRP_OBJECT (task), TRUE);
	mrp_object_set_needs_saving (MRP_OBJECT (predecessor), TRUE);

	/* FIXME: ? */
	g_object_unref (relation);
}

/**
 * mrp_task_get_relation:
 * @task_a: A task 
 * @task_b: A task
 * 
 * 
 * Return value: Any relation between two tasks.
 **/
MrpRelation *
mrp_task_get_relation (MrpTask *task_a,
		       MrpTask *task_b)
{
	MrpRelation *relation;
	
	g_return_val_if_fail (MRP_IS_TASK (task_a), NULL);
	g_return_val_if_fail (MRP_IS_TASK (task_b), NULL);
	
	relation = task_get_predecessor_relation (task_a, task_b);
	if (relation != NULL) {
		return relation;
	}

	return task_get_successor_relation (task_a, task_b);
}

/**
 * mrp_task_get_predecessor:
 * @task: 
 * @predecessor: 
 * 
 * 
 * 
 * Return value: 
 **/
MrpRelation *
mrp_task_get_predecessor (MrpTask *task,
			  MrpTask *predecessor)
{
	MrpRelation *relation;
	
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	g_return_val_if_fail (MRP_IS_TASK (predecessor), NULL);
	
	relation = task_get_predecessor_relation (task, predecessor);
	return relation;
}

/**
 * mrp_task_get_successor:
 * @task: 
 * @successor: 
 * 
 * 
 * 
 * Return value: 
 **/
MrpRelation *
mrp_task_get_successor (MrpTask *task,
			MrpTask *successor)
{
	MrpRelation *relation;
	
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	g_return_val_if_fail (MRP_IS_TASK (successor), NULL);
	
	relation = task_get_successor_relation (task, successor);
	return relation;
}

GList *
mrp_task_get_predecessors (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);

	return g_list_copy (task->priv->predecessors);
}

GList *
mrp_task_get_successors (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);

	return g_list_copy (task->priv->successors);
}

MrpTask *
mrp_task_get_parent (MrpTask *task)
{
	GNode *node;

	g_return_val_if_fail (MRP_IS_TASK (task), NULL);

	node = task->priv->node->parent;
	return node ? node->data : NULL;
}

MrpTask *
mrp_task_get_first_child (MrpTask *task)
{
	GNode *node;
	
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);

	node = g_node_first_child (task->priv->node);
	return node ? node->data : NULL;
}

MrpTask *
mrp_task_get_next_sibling (MrpTask *task)
{
	GNode *node;

	g_return_val_if_fail (MRP_IS_TASK (task), NULL);

	node = g_node_next_sibling (task->priv->node);
	return node ? node->data : NULL;
}

guint
mrp_task_get_n_children (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), 0);

	return g_node_n_children (task->priv->node);
}

MrpTask *
mrp_task_get_nth_child (MrpTask *task, guint n)
{
	GNode *node;

	g_return_val_if_fail (MRP_IS_TASK (task), 0);

	node = g_node_nth_child (task->priv->node, n);
	return node ? node->data : NULL;
}

gint
mrp_task_get_position (MrpTask *task)
{
	GNode *parent;
	
	g_return_val_if_fail (MRP_IS_TASK (task), 0);
	g_return_val_if_fail (task->priv->node->parent != NULL, 0);

	parent = task->priv->node->parent;
	
	return g_node_child_position (parent, task->priv->node);
}

void
mrp_task_add_assignment (MrpTask       *task,
			 MrpAssignment *assignment)
{
	MrpTaskPriv *priv;
	
	g_return_if_fail (MRP_IS_TASK (task));
	g_return_if_fail (MRP_IS_ASSIGNMENT (assignment));
	
	priv = task->priv;
	
	g_hash_table_insert (priv->assignments, 
			     mrp_assignment_get_resource (assignment),
			     g_object_ref (assignment));

	g_signal_connect (assignment,
			  "removed",
			  G_CALLBACK (task_assignment_removed_cb),
			  task);

	g_signal_emit (task, signals[ASSIGNMENT_ADDED], 0, assignment);

	mrp_object_set_needs_saving (MRP_OBJECT (task), TRUE);
}

mrptime
mrp_task_get_start (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), 0);

	return task->priv->start;
}

mrptime
mrp_task_get_finish (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), 0);

	return task->priv->finish;
}

mrptime
mrp_task_get_work_start (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), 0);
	
	return task->priv->work_start;
}

mrptime
mrp_task_get_latest_start (MrpTask *task)
{
	return task->priv->latest_start;
}

mrptime
mrp_task_get_latest_finish (MrpTask *task)
{
	return task->priv->latest_finish;
}

gint
mrp_task_get_duration (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), 0);

	return task->priv->duration;
}

gint
mrp_task_get_work (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), 0);
	
	return task->priv->work;
}

static void
foreach_assignment_add_to_list (gpointer key,
				gpointer value, 
				gpointer user_data)
{
	GList **list;
	
	list = (GList **) user_data;
	
	*list = g_list_prepend (*list, value);
}

GList *
mrp_task_get_assignments (MrpTask *task)
{
	GList *list = NULL;
	
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	
	g_hash_table_foreach (task->priv->assignments, 
			      foreach_assignment_add_to_list,
			      &list);
			      
	return list;
}

MrpAssignment *
mrp_task_get_assignment (MrpTask *task, MrpResource *resource)
{
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	g_return_val_if_fail (MRP_IS_RESOURCE (resource), NULL);

	return g_hash_table_lookup (task->priv->assignments, resource);
}

GList *
mrp_task_get_assigned_resources (MrpTask *task)
{
	GList         *list = NULL;
	GList         *assignments;
	GList         *node;
	MrpAssignment *assignment;
	MrpResource   *resource;
	
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);

	assignments = mrp_task_get_assignments (task);
	
	for (node = assignments; node; node = node->next) {
		assignment = MRP_ASSIGNMENT (node->data);
		resource   = MRP_RESOURCE (
			mrp_assignment_get_resource (assignment));
		
		list = g_list_prepend (list, resource);
	}

	list = g_list_sort (list, mrp_resource_compare);

	g_list_free (assignments);

	return list;
}

gint
mrp_task_compare (gconstpointer a, gconstpointer b)
{
	return strcmp (MRP_TASK (a)->priv->name, MRP_TASK (b)->priv->name);
}

void
mrp_task_reset_constraint (MrpTask *task)
{
	g_return_if_fail (MRP_IS_TASK (task));

	if (task->priv->constraint.type != MRP_CONSTRAINT_ASAP) {
		task->priv->constraint.type = MRP_CONSTRAINT_ASAP;
		g_object_notify (G_OBJECT (task), "constraint");
	}
}

/**
 * mrp_task_get_cost:
 * @task: A task object
 * 
 * 
 * 
 * Return value: the cost to complete the task
 **/
gfloat
mrp_task_get_cost (MrpTask *task)
{
	MrpTaskPriv *priv;
	MrpProject  *project;
	GList       *resources, *l;
	gfloat       total = 0;
	gfloat       cost;
	gint         work;
	
	g_return_val_if_fail (MRP_IS_TASK (task), 0);

	priv = task->priv;

	g_object_get (task, "project", &project, NULL);
	
	resources = mrp_task_get_assigned_resources (task);

	for (l = resources; l; l = l->next) {		
		mrp_object_get (l->data, "cost", &cost, NULL);

		/* FIXME: Hm, couldn't we use the work stored in the task? would be much faster */
		work = mrp_project_calculate_task_work (project,
							task,
							mrp_task_get_duration (task));
		
		/* Like this */
		/*work = priv->work;*/
		
		total += work / 3600 * cost;
	}

	g_list_free (resources);
	
	return total;
}

/* Boxed types. */

static MrpConstraint *
mrp_constraint_copy (const MrpConstraint *src)
{
	MrpConstraint *copy;

	copy = g_new (MrpConstraint, 1);
	memcpy (copy, src, sizeof (MrpConstraint));

	return copy;
}

GType
mrp_constraint_get_type (void)
{
	static GType our_type = 0;
  
	if (our_type == 0) {
		our_type = g_boxed_type_register_static ("MrpConstraint",
							 (GBoxedCopyFunc) mrp_constraint_copy,
							 (GBoxedFreeFunc) g_free);
	}
	
	return our_type;
}

/* Getters and setters. */

void
imrp_task_set_visited (MrpTask *task, gboolean visited)
{
	g_return_if_fail (MRP_IS_TASK (task));

	task->priv->visited = visited;
}

gboolean
imrp_task_get_visited (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), FALSE);

	return task->priv->visited;
}

void
imrp_task_set_start (MrpTask *task, mrptime start)
{
	g_return_if_fail (MRP_IS_TASK (task));

	task->priv->start = start;
}

void
imrp_task_set_work_start (MrpTask *task, mrptime work_start)
{
	g_return_if_fail (MRP_IS_TASK (task));

	task->priv->work_start = work_start;
}

void
imrp_task_set_finish (MrpTask *task, mrptime finish)
{
	g_return_if_fail (MRP_IS_TASK (task));

	task->priv->finish = finish;
}

void
imrp_task_set_duration (MrpTask *task, gint duration)
{
	g_return_if_fail (MRP_IS_TASK (task));

	task->priv->duration = duration;
}

void
imrp_task_set_work (MrpTask *task, gint work)
{
	g_return_if_fail (MRP_IS_TASK (task));

	task->priv->work = work;
}

void
imrp_task_set_latest_start (MrpTask *task,
			    mrptime  time)
{
	task->priv->latest_start = time;
}

void
imrp_task_set_latest_finish (MrpTask *task,
			     mrptime  time)
{
	task->priv->latest_finish = time;
}

MrpConstraint
impr_task_get_constraint (MrpTask *task)
{
	MrpConstraint c = { 0 };
	
	g_return_val_if_fail (MRP_IS_TASK (task), c);

	return task->priv->constraint;
}

void
impr_task_set_constraint (MrpTask *task, MrpConstraint constraint)
{
	g_return_if_fail (MRP_IS_TASK (task));

	task->priv->constraint = constraint;
}

gint
imrp_task_get_depth (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), 0);
	
	return g_node_depth (task->priv->node);
}

GNode *
imrp_task_get_node (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	
	return task->priv->node;
}

GNode *
imrp_task_get_sorted_node (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	
	return task->priv->sorted_node;
}

GList *
imrp_task_peek_predecessors (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	
	return task->priv->predecessors;
}

GList *
imrp_task_peek_successors (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), NULL);
	
	return task->priv->successors;
}

MrpTaskType
imrp_task_get_type (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), MRP_TASK_TYPE_NORMAL);

	return task->priv->type;
}

MrpTaskSched
imrp_task_get_sched (MrpTask *task)
{
	g_return_val_if_fail (MRP_IS_TASK (task), MRP_TASK_SCHED_NORMAL);

	return task->priv->sched;
}
