/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: NPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Netscape Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/NPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is 
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 1998
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 * Original Author: Daniel Glazman <glazman@netscape.com>
 *
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the NPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the NPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "ChangeCSSInlineStyleTxn.h"
#include "nsIDOMElement.h"
#include "nsEditor.h"
#include "nsIDOMCSSStyleDeclaration.h"
#include "nsIDOMElementCSSInlineStyle.h"
#include "nsIContent.h"
#include "nsIEditProperty.h"
#include "nsIParser.h"
#include "nsReadableUtils.h"
#include "nsUnicharUtils.h"
#include "nsCRT.h"
#include "nsIAtom.h"

void
ChangeCSSInlineStyleTxn::AppendDeclaration(nsAString & aOutputString,
                                           const nsAString & aProperty,
                                           const nsAString & aValues)
{
  aOutputString.Append(aProperty
                       + NS_LITERAL_STRING(": ")
                       + aValues
                       + NS_LITERAL_STRING("; "));
}

// answers true if aValue is in the string list of white-space separated values aValueList
// a case-sensitive search is performed if aCaseSensitive is true
PRBool
ChangeCSSInlineStyleTxn::ValueIncludes(const nsAString &aValueList, const nsAString &aValue, PRBool aCaseSensitive)
{
  nsAutoString  valueList(aValueList);
  PRBool result = PR_FALSE;

  valueList.Append(kNullCh);  // put an extra null at the end

  PRUnichar *value = ToNewUnicode(aValue);
  PRUnichar *start = (PRUnichar*)(const PRUnichar*)valueList.get();
  PRUnichar *end   = start;

  while (kNullCh != *start) {
    while ((kNullCh != *start) && nsCRT::IsAsciiSpace(*start)) {  // skip leading space
      start++;
    }
    end = start;

    while ((kNullCh != *end) && (PR_FALSE == nsCRT::IsAsciiSpace(*end))) { // look for space or end
      end++;
    }
    *end = kNullCh; // end string here

    if (start < end) {
      if (aCaseSensitive) {
        if (!nsCRT::strcmp(value, start)) {
          result = PR_TRUE;
          break;
        }
      }
      else {
        if (nsDependentString(value).Equals(nsDependentString(start),
                                            nsCaseInsensitiveStringComparator())) {
          result = PR_TRUE;
          break;
        }
      }
    }
    start = ++end;
  }
  nsCRT::free(value);
  return result;
}

// removes the value aRemoveValue from the string list of white-space separated values aValueList
void
ChangeCSSInlineStyleTxn::RemoveValueFromListOfValues(nsAString & aValues, const nsAString  & aRemoveValue)
{
  nsAutoString  classStr(aValues);  // copy to work buffer   nsAutoString  rv(aRemoveValue);
  nsAutoString  outString;
  classStr.Append(kNullCh);  // put an extra null at the end

  PRUnichar *start = (PRUnichar*)(const PRUnichar*)classStr.get();
  PRUnichar *end   = start;

  while (kNullCh != *start) {
    while ((kNullCh != *start) && nsCRT::IsAsciiSpace(*start)) {  // skip leading space
      start++;
    }
    end = start;

    while ((kNullCh != *end) && (PR_FALSE == nsCRT::IsAsciiSpace(*end))) { // look for space or end
      end++;
    }
    *end = kNullCh; // end string here

    if (start < end) {
      if (!aRemoveValue.Equals(start)) {
        outString.Append(start);
        outString.Append(PRUnichar(' '));
      }
    }

    start = ++end;
  }
  aValues.Assign(outString);
}

ChangeCSSInlineStyleTxn::ChangeCSSInlineStyleTxn()
  : EditTxn()
{
}

ChangeCSSInlineStyleTxn::~ChangeCSSInlineStyleTxn()
{
}

NS_IMETHODIMP ChangeCSSInlineStyleTxn::Init(nsIEditor      *aEditor,
                                            nsIDOMElement  *aElement,
                                            nsIAtom        *aProperty,
                                            const nsAString& aValue,
                                            PRBool aRemoveProperty)
{
  NS_ASSERTION(aEditor && aElement, "bad arg");
  if (!aEditor || !aElement) { return NS_ERROR_NULL_POINTER; }

  mEditor = aEditor;
  mElement = do_QueryInterface(aElement);
  mProperty = aProperty;
  NS_ADDREF(mProperty);
  mValue.Assign(aValue);
  mRemoveProperty = aRemoveProperty;
  mUndoAttributeWasSet = PR_FALSE;
  mRedoAttributeWasSet = PR_FALSE;
  mUndoValue.SetLength(0);
  mRedoValue.SetLength(0);
  return NS_OK;
}

NS_IMETHODIMP ChangeCSSInlineStyleTxn::DoTransaction(void)
{
  NS_ASSERTION(mEditor && mElement, "bad state");
  if (!mEditor || !mElement) { return NS_ERROR_NOT_INITIALIZED; }

  nsresult result = NS_OK;

  nsCOMPtr<nsIDOMCSSStyleDeclaration> cssDecl;
  nsCOMPtr<nsIDOMElementCSSInlineStyle> inlineStyles = do_QueryInterface(mElement);
  if (!inlineStyles) return NS_ERROR_NULL_POINTER;
  result = inlineStyles->GetStyle(getter_AddRefs(cssDecl));
  if (NS_FAILED(result)) return result;
  if (!cssDecl) return NS_ERROR_NULL_POINTER;

  nsAutoString propertyNameString;
  mProperty->ToString(propertyNameString);

  // does this property accept more than 1 value ?
  // we need to know that because of bug 62682
  PRBool multiple = AcceptsMoreThanOneValue(mProperty);
  
  nsAutoString styleAttr(NS_LITERAL_STRING("style"));

  result = mElement->HasAttribute(styleAttr, &mUndoAttributeWasSet);
  if (NS_FAILED(result)) return result;

  nsAutoString values;
  result = cssDecl->GetPropertyValue(propertyNameString, values);
  if (NS_FAILED(result)) return result;     
  mUndoValue.Assign(values);

  if (mRemoveProperty) {
    nsAutoString returnString;
    if (multiple) {
      // the property can have more than one value, let's remove only
      // the value we have to remove and not the others

      // the 2 lines below are a workaround because nsDOMCSSDeclaration::GetPropertyCSSValue
      // is not yet implemented (bug 62682)
      RemoveValueFromListOfValues(values, NS_LITERAL_STRING("none"));
      RemoveValueFromListOfValues(values, mValue);
      if (values.IsEmpty()) {
        result = cssDecl->RemoveProperty(propertyNameString, returnString);
        if (NS_FAILED(result)) return result;     
      }
      else {
        nsAutoString priority;
        result = cssDecl->GetPropertyPriority(propertyNameString, priority);
        if (NS_FAILED(result)) return result;     
        result = cssDecl->SetProperty(propertyNameString, values,
                                      priority);
        if (NS_FAILED(result)) return result;     
      }
    }
    else {
      result = cssDecl->RemoveProperty(propertyNameString, returnString);
      if (NS_FAILED(result)) return result;     
    }
  }
  else {
    nsAutoString priority;
    result = cssDecl->GetPropertyPriority(propertyNameString, priority);
    if (NS_FAILED(result)) return result;
    if (multiple) {
      // the property can have more than one value, let's add
      // the value we have to add to the others

      // the line below is a workaround because nsDOMCSSDeclaration::GetPropertyCSSValue
      // is not yet implemented (bug 62682)
      AddValueToMultivalueProperty(values, mValue);
    }
    else
      values.Assign(mValue);
    result = cssDecl->SetProperty(propertyNameString, values,
                                  priority);
    if (NS_FAILED(result)) return result;     
  }

  // let's be sure we don't keep an empty style attribute
  PRUint32 length;
  result = cssDecl->GetLength(&length);
  if (NS_FAILED(result)) return result;     
  if (!length) {
    result = mElement->RemoveAttribute(styleAttr);
    if (NS_FAILED(result)) return result;     
  }
  else
    mRedoAttributeWasSet = PR_TRUE;

  return cssDecl->GetPropertyValue(propertyNameString, mRedoValue);
}

nsresult ChangeCSSInlineStyleTxn::SetStyle(PRBool aAttributeWasSet,
                                           nsAString & aValue)
{
  NS_ASSERTION(mEditor && mElement, "bad state");
  if (!mEditor || !mElement) { return NS_ERROR_NOT_INITIALIZED; }

  nsresult result = NS_OK;
  if (aAttributeWasSet) {
    // the style attribute was set and not empty, let's recreate the declaration
    nsAutoString propertyNameString;
    mProperty->ToString(propertyNameString);

    nsCOMPtr<nsIDOMCSSStyleDeclaration> cssDecl;
    nsCOMPtr<nsIDOMElementCSSInlineStyle> inlineStyles = do_QueryInterface(mElement);
    if (!inlineStyles) return NS_ERROR_NULL_POINTER;
    result = inlineStyles->GetStyle(getter_AddRefs(cssDecl));
    if (NS_FAILED(result)) return result;
    if (!cssDecl) return NS_ERROR_NULL_POINTER;

    if (aValue.IsEmpty()) {
      // an empty value means we have to remove the property
      nsAutoString returnString;
      result = cssDecl->RemoveProperty(propertyNameString, returnString);
    }
    else {
      // let's recreate the declaration as it was
      nsAutoString priority;
      result = cssDecl->GetPropertyPriority(propertyNameString, priority);
      if (NS_FAILED(result)) return result;
      result = cssDecl->SetProperty(propertyNameString, aValue, priority);
    }
  }
  else
    result = mElement->RemoveAttribute(NS_LITERAL_STRING("style"));

  return result;
}

NS_IMETHODIMP ChangeCSSInlineStyleTxn::UndoTransaction(void)
{
  return SetStyle(mUndoAttributeWasSet, mUndoValue);
}

NS_IMETHODIMP ChangeCSSInlineStyleTxn::RedoTransaction(void)
{
  return SetStyle(mRedoAttributeWasSet, mRedoValue);
}

NS_IMETHODIMP ChangeCSSInlineStyleTxn::Merge(nsITransaction *aTransaction, PRBool *aDidMerge)
{
  if (nsnull!=aDidMerge)
    *aDidMerge=PR_FALSE;
  return NS_OK;
}

NS_IMETHODIMP ChangeCSSInlineStyleTxn::GetTxnDescription(nsAString& aString)
{
  aString.Assign(NS_LITERAL_STRING("ChangeCSSInlineStyleTxn: "));

  if (PR_FALSE==mRemoveProperty)
    aString += NS_LITERAL_STRING("[mRemoveProperty == false] ");
  else
    aString += NS_LITERAL_STRING("[mRemoveProperty == true] ");
  nsAutoString tempString;
  mProperty->ToString(tempString);
  aString += tempString;
  return NS_OK;
}

// answers true if the CSS property accepts more than one value
PRBool
ChangeCSSInlineStyleTxn::AcceptsMoreThanOneValue(nsIAtom *aCSSProperty)
{
  PRBool res = PR_FALSE;
  nsIAtom * textDecorationAtom = NS_NewAtom("text-decoration");
  if (textDecorationAtom == aCSSProperty) {
    res = PR_TRUE;
  }
  NS_IF_RELEASE(textDecorationAtom);
  return res;
}

// adds the value aNewValue to the list of white-space separated values aValues
NS_IMETHODIMP
ChangeCSSInlineStyleTxn::AddValueToMultivalueProperty(nsAString & aValues, const nsAString & aNewValue)
{
  if (aValues.IsEmpty()
      || aValues.Equals(NS_LITERAL_STRING("none"),
                        nsCaseInsensitiveStringComparator())) {
    // the list of values is empty of the value is 'none'
    aValues.Assign(aNewValue);
  }
  else if (!ValueIncludes(aValues, aNewValue, PR_FALSE)) {
    // we already have another value but not this one; add it
    aValues.Append(PRUnichar(' '));
    aValues.Append(aNewValue);
  }
  return NS_OK;
}

