/*
 * h323annexg.h
 *
 * Implementation of H.323 Annex G using H.501
 *
 * Open H323 Library
 *
 * Copyright (c) 2003 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 */

#ifndef __OPAL_H323ANNEXG_H
#define __OPAL_H323ANNEXG_H

#ifdef P_USE_PRAGMA
#pragma interface
#endif

#include "h323trans.h"
#include "h501pdu.h"

class PASN_Sequence;
class PASN_Choice;

class H323EndPoint;

///////////////////////////////////////////////////////////////////////////////

/**This class embodies the H.323 Annex G using the H.501 protocol 
  */
class H323_AnnexG : public H323Transactor
{
  PCLASSINFO(H323_AnnexG, H323Transactor);
  public:
  /**@name Construction */
  //@{
    enum {
      DefaultUdpPort = 2099,
      DefaultTcpPort = 2099
    };

    /**Create a new protocol handler.
     */
    H323_AnnexG(
      H323EndPoint & endpoint,  /// Endpoint gatekeeper is associated with.
      H323Transport * transport /// Transport over which gatekeepers communicates.
    );
    H323_AnnexG(
      H323EndPoint & endpoint,           /// Endpoint gatekeeper is associated with.
      const H323TransportAddress & addr /// Transport over which gatekeepers communicates.
    );

    /**Destroy protocol handler.
     */
    ~H323_AnnexG();
  //@}

  /**@name Overrides from PObject */
  //@{
    /**Print the name of the gatekeeper.
      */
    void PrintOn(
      ostream & strm    /// Stream to print to.
    ) const;
  //@}

  /**@name Operations */
  //@{
    /**Start the background thread.
      */
    BOOL StartChannel();

    virtual BOOL HandleH501PDU(const H501PDU & pdu);

    virtual BOOL OnReceiveServiceRequest              (const H501_MessageCommonInfo & common, const H501_ServiceRequest & pdu);
    virtual BOOL OnReceiveServiceConfirmation         (const H501_MessageCommonInfo & common, const H501_ServiceConfirmation & pdu);
    virtual BOOL OnReceiveServiceRejection            (const H501_MessageCommonInfo & common, const H501_ServiceRejection & pdu);
    virtual BOOL OnReceiveServiceRelease              (const H501_MessageCommonInfo & common, const H501_ServiceRelease & pdu);
    virtual BOOL OnReceiveDescriptorRequest           (const H501_MessageCommonInfo & common, const H501_DescriptorRequest & pdu);
    virtual BOOL OnReceiveDescriptorConfirmation      (const H501_MessageCommonInfo & common, const H501_DescriptorConfirmation & pdu);
    virtual BOOL OnReceiveDescriptorRejection         (const H501_MessageCommonInfo & common, const H501_DescriptorRejection & pdu);
    virtual BOOL OnReceiveDescriptorIDRequest         (const H501_MessageCommonInfo & common, const H501_DescriptorIDRequest & pdu);
    virtual BOOL OnReceiveDescriptorIDConfirmation    (const H501_MessageCommonInfo & common, const H501_DescriptorIDConfirmation & pdu);
    virtual BOOL OnReceiveDescriptorIDRejection       (const H501_MessageCommonInfo & common, const H501_DescriptorIDRejection & pdu);
    virtual BOOL OnReceiveDescriptorUpdate            (const H501_MessageCommonInfo & common, const H501_DescriptorUpdate & pdu);
    virtual BOOL OnReceiveDescriptorUpdateACK         (const H501_MessageCommonInfo & common, const H501_DescriptorUpdateAck & pdu);
    virtual BOOL OnReceiveAccessRequest               (const H501_MessageCommonInfo & common, const H501_AccessRequest & pdu);
    virtual BOOL OnReceiveAccessConfirmation          (const H501_MessageCommonInfo & common, const H501_AccessConfirmation & pdu);
    virtual BOOL OnReceiveAccessRejection             (const H501_MessageCommonInfo & common, const H501_AccessRejection & pdu);
    virtual BOOL OnReceiveRequestInProgress           (const H501_MessageCommonInfo & common, const H501_RequestInProgress & pdu);
    virtual BOOL OnReceiveNonStandardRequest          (const H501_MessageCommonInfo & common, const H501_NonStandardRequest & pdu);
    virtual BOOL OnReceiveNonStandardConfirmation     (const H501_MessageCommonInfo & common, const H501_NonStandardConfirmation & pdu);
    virtual BOOL OnReceiveNonStandardRejection        (const H501_MessageCommonInfo & common, const H501_NonStandardRejection & pdu);
    virtual BOOL OnReceiveUnknownMessageResponse      (const H501_MessageCommonInfo & common, const H501_UnknownMessageResponse & pdu);
    virtual BOOL OnReceiveUsageRequest                (const H501_MessageCommonInfo & common, const H501_UsageRequest & pdu);
    virtual BOOL OnReceiveUsageConfirmation           (const H501_MessageCommonInfo & common, const H501_UsageConfirmation & pdu);
    virtual BOOL OnReceiveUsageIndicationConfirmation (const H501_MessageCommonInfo & common, const H501_UsageIndicationConfirmation & pdu);
    virtual BOOL OnReceiveUsageIndicationRejection    (const H501_MessageCommonInfo & common, const H501_UsageIndicationRejection & pdu);
    virtual BOOL OnReceiveUsageRejection              (const H501_MessageCommonInfo & common, const H501_UsageRejection & pdu);
    virtual BOOL OnReceiveValidationRequest           (const H501_MessageCommonInfo & common, const H501_ValidationRequest & pdu);
    virtual BOOL OnReceiveValidationConfirmation      (const H501_MessageCommonInfo & common, const H501_ValidationConfirmation & pdu);
    virtual BOOL OnReceiveValidationRejection         (const H501_MessageCommonInfo & common, const H501_ValidationRejection & pdu);
    virtual BOOL OnReceiveAuthenticationRequest       (const H501_MessageCommonInfo & common, const H501_AuthenticationRequest & pdu);
    virtual BOOL OnReceiveAuthenticationConfirmation  (const H501_MessageCommonInfo & common, const H501_AuthenticationConfirmation & pdu);
    virtual BOOL OnReceiveAuthenticationRejection     (const H501_MessageCommonInfo & common, const H501_AuthenticationRejection & pdu);
    virtual BOOL OnReceiveUnknown(const H501PDU &);

    BOOL WriteTo(H501PDU & pdu, const H323TransportAddress & address);

    virtual unsigned GetRequestRetries() const      { return requestRetryCount; }

    const PTimeInterval & GetRequestTimeout() const { return requestTimeout; }

  protected:
    void Construct();

    PTimeInterval requestTimeout;
    unsigned      requestRetryCount;

    //Background thread handler.
    PDECLARE_NOTIFIER(PThread, H323_AnnexG, HandleH501Channel);

    BOOL SetUpCallSignalAddresses(
      H225_ArrayOf_TransportAddress & addresses
    );

    class Request : public PObject
    {
      PCLASSINFO(Request, PObject);
      public:
        Request(
          unsigned seqNum,
          H501PDU & pdu,
          const H323TransportAddress & address
        );

        BOOL Poll(H323_AnnexG &);
        void CheckResponse(unsigned, const PASN_Choice *);
        void OnReceiveRIP(const H501PDU & rip);

        // Inter-thread transfer variables
        unsigned rejectReason;
        void   * responseInfo;

        H323TransportAddress requestAddress;

        unsigned      sequenceNumber;
        H501PDU       & requestPDU;
        PTimeInterval whenResponseExpected;
        PSyncPoint    responseHandled;
        PMutex        responseMutex;

        enum {
          AwaitingResponse,
          ConfirmReceived,
          RejectReceived,
          BadCryptoTokens,
          RequestInProgress,
          NoResponseReceived
        } responseResult;
    };

    BOOL MakeRequest(Request & request);
    BOOL CheckForResponse(unsigned reqTag, unsigned seqNum, const PASN_Choice * reason = NULL);

    PDICTIONARY(RequestDict, POrdinalKey, Request);
    RequestDict requests;
    PMutex      requestsMutex;
    Request   * lastRequest;
    PMutex      transportMutex;
};


#endif // __OPAL_H323ANNEXG_H
