/*
 * h323trans.h
 *
 * H.323 Transactor handler
 *
 * Open H323 Library
 *
 * Copyright (c) 2003 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: h323trans.h,v $
 * Revision 1.4  2003/03/01 00:23:42  craigs
 * New PeerElement implementation
 *
 * Revision 1.3  2003/02/25 06:48:15  robertj
 * More work on PDU transaction abstraction.
 *
 * Revision 1.2  2003/02/25 03:14:58  robertj
 * Added missing virtual destructor.
 *
 * Revision 1.1  2003/02/21 05:28:39  craigs
 * Factored out code for user with peer elements
 *
 */

#ifndef __OPAL_H323TRANS_H
#define __OPAL_H323TRANS_H

#ifdef P_USE_PRAGMA
#pragma interface
#endif

#include "transports.h"
#include "h235auth.h"

#include <ptclib/asner.h>


class H323TransactionPDU {
  public:
    virtual const char * GetProtocolName() = 0;

    H323TransactionPDU();
    H323TransactionPDU(const H235Authenticators & auth);

    virtual ~H323TransactionPDU() { }

    virtual BOOL Read(H323Transport & transport);
    virtual BOOL Write(H323Transport & transport);

    virtual PASN_Object & GetPDU() = 0;
    virtual PASN_Choice & GetChoice() = 0;
    virtual const PASN_Object & GetPDU() const = 0;
    virtual const PASN_Choice & GetChoice() const = 0;
    virtual unsigned GetSequenceNumber() const = 0;

    const H235Authenticators & GetAuthenticators() const { return authenticators; }
    void SetAuthenticators(
      const H235Authenticators & auth
    ) { authenticators = auth; }

    H235Authenticator::ValidationResult Validate(
      const H225_ArrayOf_ClearToken & clearTokens,
      unsigned clearOptionalField,
      const H225_ArrayOf_CryptoH323Token & cryptoTokens,
      unsigned cryptoOptionalField
    ) const { return authenticators.ValidatePDU(*this, clearTokens, clearOptionalField, cryptoTokens, cryptoOptionalField, rawPDU); }

    void Prepare(
      H225_ArrayOf_ClearToken & clearTokens,
      unsigned clearOptionalField,
      H225_ArrayOf_CryptoH323Token & cryptoTokens,
      unsigned cryptoOptionalField
    ) { authenticators.PreparePDU(*this, clearTokens, clearOptionalField, cryptoTokens, cryptoOptionalField); }

  protected:
    H235Authenticators authenticators;
    PPER_Stream rawPDU;
};


///////////////////////////////////////////////////////////

class H323Transactor : public PObject
{
  PCLASSINFO(H323Transactor, PObject);
  public:
  /**@name Construction */
  //@{

    /**Create a new protocol handler.
     */
    H323Transactor(
      H323EndPoint & endpoint,   /// Endpoint gatekeeper is associated with.
      H323Transport * transport, /// Transport over which gatekeepers communicates.
      WORD defaultPort
    );
    H323Transactor(
      H323EndPoint & ep, 
      const H323TransportAddress & iface, 
      WORD defaultPort
    );

    /**Destroy protocol handler.
     */
    ~H323Transactor();
  //@}

  /**@name Overrides from PObject */
  //@{
    /**Print the name of the gatekeeper.
      */
    virtual void PrintOn(
      ostream & strm    /// Stream to print to.
    ) const = 0;
  //@}

    virtual BOOL StartChannel() = 0;

  /**@name Member variable access */
  //@{
    /**Get the gatekeepers associated endpoint.
      */
    H323EndPoint & GetEndPoint() const { return endpoint; }

    /**Get the gatekeepers transport channel.
      */
    H323Transport & GetTransport() const { return *transport; }

  //@}

  protected:
    unsigned GetNextSequenceNumber();
    BOOL SetUpCallSignalAddresses(
      H225_ArrayOf_TransportAddress & addresses
    );

    // Configuration variables
    H323EndPoint  & endpoint;
    H323Transport * transport;

    PMutex    pduWriteMutex;
    unsigned  nextSequenceNumber;
    PMutex    nextSequenceNumberMutex;
};


////////////////////////////////////////////////////////////////////////////////////

class H323TransactionServer : public PObject
{
  PCLASSINFO(H323TransactionServer, PObject);
  public:
  /**@name Construction */
  //@{
    /**Create a new gatekeeper.
     */
    H323TransactionServer(
      H323EndPoint & endpoint
    );

    /**Destroy gatekeeper.
     */
    ~H323TransactionServer();
  //@}

    virtual WORD GetDefaultUdpPort() = 0;

  /**@name Access functions */
  //@{
    /**Get the owner endpoint.
     */
    H323EndPoint & GetOwnerEndPoint() const { return ownerEndPoint; }

  /**@name Protocol Handler Operations */
  //@{
    /**Add listeners to the transaction server.
       If a listener already exists on the interface specified in the list
       then it is ignored. If a listener does not yet exist a new one is
       created and if a listener is running that is not in the list then it
       is stopped and removed.

       If the array is empty then the string "*" is assumed which will listen
       on the standard UDP port on INADDR_ANY.

       Returns TRUE if at least one interface was successfully started.
      */
    BOOL AddListeners(
      const H323TransportAddressArray & ifaces /// Interfaces to listen on.
    );

    /**Add a gatekeeper listener to this gatekeeper server given the
       transport address for the local interface.
      */
    BOOL AddListener(
      const H323TransportAddress & interfaceName
    );

    /**Add a gatekeeper listener to this gatekeeper server given the transport.
       Note that the transport is then owned by the listener and will be
       deleted automatically when the listener is destroyed. Note also the
       transport is deleted if this function returns FALSE and no listener was
       created.
      */
    BOOL AddListener(
      H323Transport * transport
    );

    /**Add a gatekeeper listener to this gatekeeper server.
       Note that the gatekeeper listener is then owned by the gatekeeper
       server and will be deleted automatically when the listener is removed.
       Note also the listener is deleted if this function returns FALSE and
       the listener was not used.
      */
    BOOL AddListener(
      H323Transactor * listener
    );

    /**Create a new H323GatkeeperListener.
       The user woiuld not usually use this function as it is used internally
       by the server when new listeners are added by H323TransportAddress.

       However, a user may override this function to create objects that are
       user defined descendants of H323GatekeeperListener so the user can
       maintain extra information on a interface by interface basis.
      */
    virtual H323Transactor * CreateListener(
      H323Transport * transport  // Transport for listener
    ) = 0;

    /**Remove a gatekeeper listener from this gatekeeper server.
       The gatekeeper listener is automatically deleted.
      */
    BOOL RemoveListener(
      H323Transactor * listener
    );

    BOOL SetUpCallSignalAddresses(H225_ArrayOf_TransportAddress & addresses);
  //@}

  protected:
    H323EndPoint & ownerEndPoint;

    PThread      * monitorThread;
    PSyncPoint     monitorExit;

    PMutex         mutex;
    PLIST(ListenerList, H323Transactor);
    ListenerList listeners;
};


#endif // __OPAL_H323TRANS_H


/////////////////////////////////////////////////////////////////////////////
