/*
 * peclient.h
 *
 * H.323 Annex G Peer Element client protocol handler
 *
 * Open H323 Library
 *
 * Copyright (c) 2003 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: peclient.h,v $
 * Revision 1.3  2003/03/01 00:23:51  craigs
 * New PeerElement implementation
 *
 * Revision 1.2  2003/02/21 07:23:18  robertj
 * Fixed up some comments
 *
 * Revision 1.1  2003/02/21 05:28:39  craigs
 * Factored out code for user with peer elements
 *
 */

#ifndef __OPAL_ANNEXGCLIENT_H
#define __OPAL_ANNEXGCLIENT_H

#include <ptlib\safecoll.h>

#include "h323annexg.h"
#include "h323trans.h"
#include "h323ep.h"

#ifdef P_USE_PRAGMA
#pragma interface
#endif

class PASN_Sequence;
class PASN_Choice;

////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////


/**This class embodies the an H.323 AnnexG server/client
  */
class H323PeerElement : public H323_AnnexG
{
	PCLASSINFO(H323PeerElement, H323_AnnexG);
  public:
  /**@name Construction */
  //@{
    /**Create a new client.
     */
    H323PeerElement(
      H323EndPoint & endpoint,  /// Endpoint gatekeeper is associated with.
      H323Transport * transport = NULL
    );
    H323PeerElement(
      H323EndPoint & endpoint,  /// Endpoint gatekeeper is associated with.
      const H323TransportAddress & addr
    );

    /**Destroy client.
     */
    ~H323PeerElement();
  //@}

  /**@name Overrides from PObject */
  //@{
    /**Print the name of the peer element.
      */
    void PrintOn(
      ostream & strm    /// Stream to print to.
    ) const;
  //@}


    BOOL SetClearingHouse(const H323TransportAddress & addr);

    /**Send a service request
     */
    BOOL ServiceRequest  (const H323TransportAddress & peer);

    /**Send a service release
     */
    BOOL ServiceRelease(const H323TransportAddress & peer, unsigned reason);

    /**Add a new descriptor to the local table
     */
    BOOL AddDescriptor(const PString & alias, const H323TransportAddress & addr);
    BOOL AddDescriptor(const PString & alias, const PIPSocket::Address & address, WORD port);

    BOOL MakeRequest(H323_AnnexG::Request & request);
    BOOL SetListener(const H323TransportAddress & iface);

    virtual void OnAddServiceRelationship(const H323TransportAddress &) { }
    virtual void OnRemoveServiceRelationship(const H323TransportAddress &) { }

    BOOL OnReceiveServiceConfirmation(const H501_MessageCommonInfo & common, const H501_ServiceConfirmation & pdu);

  protected:
    void Construct();

    PString elementIdentifier;

    PDECLARE_NOTIFIER(PThread, H323PeerElement, MonitorMain);
    PDECLARE_NOTIFIER(PTimer, H323PeerElement, TickleMonitor);

    PMutex clearingHouseMutex;
    BOOL hasClearingHouse;
    H323TransportAddress clearingHouse;

    PSemaphore requestMutex;
    PThread  * monitor;
    BOOL       monitorStop;
    PSyncPoint monitorTickle;
    PString    serverIdentifier;

    class ServiceRelationship : public PSafeObject {
      PCLASSINFO(ServiceRelationship, PSafeObject);
      public:
        ServiceRelationship(const H323TransportAddress & _peer)
          : peer(_peer) { }

        Comparison Compare(const PObject & obj) const
          { return peer.Compare(((ServiceRelationship &)obj).peer); }

        H323TransportAddress peer;
        OpalGloballyUniqueID serviceID;
        PTime expireTime;
    };

    PSafeSortedList<ServiceRelationship> serviceRelationships;
    PSafeSortedList<ServiceRelationship> ourServiceRelationships;

    class Descriptor : public PSafeObject {
      PCLASSINFO(Descriptor, PSafeObject);
      public:
        Descriptor(const PString & _alias)
          : alias(_alias)
         {}

        Comparison Compare(const PObject & obj) const
          { return alias.Compare(((Descriptor &)obj).alias); }

        void CopyTo(H501_Descriptor & h501Descriptor, const H323TransportAddress & contact);

        PString alias;
        PIPSocket::Address address;
        WORD port;
        OpalGloballyUniqueID descriptorID;
        PTime lastChanged;
    };

    PSafeSortedList<Descriptor> descriptors;

    BOOL UpdateAddDescriptor   (Descriptor * descriptor);
    BOOL UpdateChangeDescriptor(Descriptor * descriptor);
    BOOL UpdateDeleteDescriptor(Descriptor * descriptor);
};

#endif // __OPAL_ANNEXGCLIENT_H
