/*
 * h323annexg.cxx
 *
 * Implementation of H.323 Annex G using H.501
 *
 * Open H323 Library
 *
 * Copyright (c) 2001 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of this code were written with the assisance of funding from
 * iFace In, http://www.iface.com
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: h323annexg.cxx,v $
 * Revision 1.3  2003/03/01 00:22:10  craigs
 * New PeerElement implementation
 *
 * Revision 1.2  2003/02/25 06:48:19  robertj
 * More work on PDU transaction abstraction.
 *
 * Revision 1.1  2003/02/21 05:27:06  craigs
 * Initial version
 *
 */

#include <ptlib.h>

#ifdef __GNUC__
#pragma implementation "h323annexg.h"
#endif

#include <ptclib/random.h>

#include "h323annexg.h"
#include "h323ep.h"
#include "h323pdu.h"
#include "h501.h"

#define new PNEW


H323_AnnexG::H323_AnnexG(H323EndPoint & ep, H323Transport * trans)
  : H323Transactor(ep, trans, DefaultUdpPort),
    requestTimeout(0, 3),
    requestRetryCount(3)
{
  Construct();
}

H323_AnnexG::H323_AnnexG(H323EndPoint & ep, const H323TransportAddress & addr)
  : H323Transactor(ep, addr, DefaultUdpPort),
    requestTimeout(0, 3),
    requestRetryCount(3)
{
  Construct();
}

void H323_AnnexG::Construct()
{
  lastRequest = NULL;
  requests.DisallowDeleteObjects();
}

H323_AnnexG::~H323_AnnexG()
{
}

void H323_AnnexG::PrintOn(ostream & strm) const
{
  strm << "H501<";
  if (transport != NULL)
    strm << transport->GetLocalAddress();
  else
    strm << "no-transport";
  strm << '>';
}


BOOL H323_AnnexG::StartChannel()
{
  transport->AttachThread(PThread::Create(PCREATE_NOTIFIER(HandleH501Channel), 0,
                                          PThread::NoAutoDeleteThread,
                                          PThread::NormalPriority,
                                          "H501:%x"));
  return TRUE;
}

BOOL H323_AnnexG::SetUpCallSignalAddresses(H225_ArrayOf_TransportAddress & addresses)
{
  H225_TransportAddress h501Address;
  transport->SetUpTransportPDU(h501Address, TRUE);

  const H323ListenerList & listeners = endpoint.GetListeners();
  for (PINDEX i = 0; i < listeners.GetSize(); i++)
    listeners[i].SetUpTransportPDU(addresses, *transport);

  return addresses.GetSize() > 0;
}

void H323_AnnexG::HandleH501Channel(PThread &, INT)
{
  transport->SetReadTimeout(PMaxTimeInterval);
  H501PDU response;

  for (;;) {
    PTRACE(5, "H501\tReading PDU");
    lastRequest = NULL;
    if (response.Read(*transport)) {
      if (HandleH501PDU(response))
        lastRequest->responseHandled.Signal();
      if (lastRequest != NULL)
        lastRequest->responseMutex.Signal();
    }
    else {
      switch (transport->GetErrorNumber()) {
        case ECONNRESET:
        case ECONNREFUSED:
          PTRACE(2, "H501\tCannot access remote " << transport->GetRemoteAddress());
          break;

        default:
          PTRACE(1, "H501\tRead error: " << transport->GetErrorText(PChannel::LastReadError));
          return;
      }
    }
  }
}

BOOL H323_AnnexG::WriteTo(H501PDU & pdu, const H323TransportAddress & address)
{
  PWaitAndSignal m(transportMutex);
  transport->ConnectTo(address);
  return pdu.Write(*transport);
}


BOOL H323_AnnexG::HandleH501PDU(const H501PDU & pdu)
{
  switch (pdu.GetTag()) {
   case H501_MessageBody::e_serviceRequest :
      OnReceiveServiceRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_serviceConfirmation :
      return OnReceiveServiceConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_serviceRejection :
      return OnReceiveServiceRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_serviceRelease :
      OnReceiveServiceRelease(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_descriptorRequest :
      OnReceiveDescriptorRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_descriptorConfirmation :
      return OnReceiveDescriptorConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_descriptorRejection :
      return OnReceiveDescriptorRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_descriptorIDRequest :
      OnReceiveDescriptorIDRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_descriptorIDConfirmation :
      return OnReceiveDescriptorIDConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_descriptorIDRejection :
      return OnReceiveDescriptorIDRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_descriptorUpdate :
      OnReceiveDescriptorUpdate(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_descriptorUpdateAck :
      return OnReceiveDescriptorUpdateACK(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_accessRequest :
      OnReceiveAccessRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_accessConfirmation :
      return OnReceiveAccessConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_accessRejection :
      return OnReceiveAccessRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_requestInProgress :
      return OnReceiveRequestInProgress(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_nonStandardRequest :
      OnReceiveNonStandardRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_nonStandardConfirmation :
      return OnReceiveNonStandardConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_nonStandardRejection :
      return OnReceiveNonStandardRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_unknownMessageResponse :
      OnReceiveUnknownMessageResponse(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_usageRequest :
      OnReceiveUsageRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_usageConfirmation :
      return OnReceiveUsageConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_usageIndication :
      OnReceiveUnknownMessageResponse(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_usageIndicationConfirmation :
      return OnReceiveUsageIndicationConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_usageIndicationRejection :
      return OnReceiveUsageIndicationRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_usageRejection :
      return OnReceiveUsageRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_validationRequest :
      OnReceiveValidationRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_validationConfirmation :
      return OnReceiveValidationConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_validationRejection :
      return OnReceiveValidationRejection(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_authenticationRequest :
      OnReceiveAuthenticationRequest(pdu.m_common, pdu.m_body);
      break;

    case H501_MessageBody::e_authenticationConfirmation :
      return OnReceiveAuthenticationConfirmation(pdu.m_common, pdu.m_body);

    case H501_MessageBody::e_authenticationRejection :
      return OnReceiveAuthenticationRejection(pdu.m_common, pdu.m_body);

    default :
      OnReceiveUnknown(pdu);
  }

  return FALSE;
}

BOOL H323_AnnexG::OnReceiveUnknown(const H501PDU &)
{
  H501PDU response;
  response.BuildUnknownMessageResponse(0);
  return response.Write(*transport);
}

BOOL H323_AnnexG::OnReceiveServiceRequest(const H501_MessageCommonInfo & common, const H501_ServiceRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveServiceRequest - seq: " << common.m_sequenceNumber);
  H501PDU response;
  response.BuildServiceRejection(common.m_sequenceNumber);
  return response.Write(*transport);
}

BOOL H323_AnnexG::OnReceiveServiceConfirmation(const H501_MessageCommonInfo & common, const H501_ServiceConfirmation & /*pdu*/)
{
  return CheckForResponse(H501_MessageBody::e_serviceRequest, common.m_sequenceNumber);
}

BOOL H323_AnnexG::OnReceiveServiceRejection(const H501_MessageCommonInfo & common, const H501_ServiceRejection & pdu)
{
  return CheckForResponse(H501_MessageBody::e_serviceRequest, common.m_sequenceNumber, &pdu.m_reason);
}

BOOL H323_AnnexG::OnReceiveServiceRelease(const H501_MessageCommonInfo & /*common*/, const H501_ServiceRelease & /*pdu*/)
{
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorRequest(const H501_MessageCommonInfo & common, const H501_DescriptorRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorRequest - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorConfirmation(const H501_MessageCommonInfo & common, const H501_DescriptorConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorRejection(const H501_MessageCommonInfo & common, const H501_DescriptorRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorIDRequest(const H501_MessageCommonInfo & common, const H501_DescriptorIDRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorIDRequest - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorIDConfirmation(const H501_MessageCommonInfo & common, const H501_DescriptorIDConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorIDConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorIDRejection(const H501_MessageCommonInfo & common, const H501_DescriptorIDRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorIDRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorUpdate(const H501_MessageCommonInfo & common, const H501_DescriptorUpdate & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorUpdate - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveDescriptorUpdateACK(const H501_MessageCommonInfo & common, const H501_DescriptorUpdateAck & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveDescriptorUpdateACK - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveAccessRequest(const H501_MessageCommonInfo & common, const H501_AccessRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveAccessRequest - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveAccessConfirmation(const H501_MessageCommonInfo & common, const H501_AccessConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveAccessConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveAccessRejection(const H501_MessageCommonInfo & common, const H501_AccessRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveAccessRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveRequestInProgress(const H501_MessageCommonInfo & common, const H501_RequestInProgress & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveRequestInProgress - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveNonStandardRequest(const H501_MessageCommonInfo & common, const H501_NonStandardRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveNonStandardRequest - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveNonStandardConfirmation(const H501_MessageCommonInfo & common, const H501_NonStandardConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveNonStandardConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveNonStandardRejection(const H501_MessageCommonInfo & common, const H501_NonStandardRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveNonStandardRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveUnknownMessageResponse(const H501_MessageCommonInfo & common, const H501_UnknownMessageResponse & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveUnknownMessageResponse - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveUsageRequest(const H501_MessageCommonInfo & common, const H501_UsageRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveUsageRequest - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveUsageConfirmation(const H501_MessageCommonInfo & common, const H501_UsageConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveUsageConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveUsageIndicationConfirmation(const H501_MessageCommonInfo & common, const H501_UsageIndicationConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveUsageIndicationConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveUsageIndicationRejection(const H501_MessageCommonInfo & common, const H501_UsageIndicationRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveUsageIndicationRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveUsageRejection(const H501_MessageCommonInfo & common, const H501_UsageRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveUsageRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveValidationRequest(const H501_MessageCommonInfo & common, const H501_ValidationRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveValidationRequest - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveValidationConfirmation(const H501_MessageCommonInfo & common, const H501_ValidationConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveValidationConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveValidationRejection(const H501_MessageCommonInfo & common, const H501_ValidationRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveValidationRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveAuthenticationRequest(const H501_MessageCommonInfo & common, const H501_AuthenticationRequest & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveAuthenticationRequest - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveAuthenticationConfirmation(const H501_MessageCommonInfo & common, const H501_AuthenticationConfirmation & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveAuthenticationConfirmation - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::OnReceiveAuthenticationRejection(const H501_MessageCommonInfo & common, const H501_AuthenticationRejection & /*pdu*/)
{
  PTRACE(3, "AnnexG\tOnReceiveAuthenticationRejection - seq: " << common.m_sequenceNumber);
  return FALSE;
}

BOOL H323_AnnexG::MakeRequest(Request & request)
{
  PTRACE(3, "AnnexG\tMaking request: " << request.requestPDU.m_body.GetTagName());

  //OnSendPDU(*this, request.requestPDU);

  requestsMutex.Wait();
  requests.SetAt(request.sequenceNumber, &request);
  requestsMutex.Signal();

  BOOL ok = request.Poll(*this);

  requestsMutex.Wait();
  requests.SetAt(request.sequenceNumber, NULL);
  requestsMutex.Signal();

  return ok;
}

BOOL H323_AnnexG::CheckForResponse(unsigned reqTag, unsigned seqNum, const PASN_Choice * reason)
{
  requestsMutex.Wait();
  lastRequest = requests.GetAt(seqNum);
  requestsMutex.Signal();

  if (lastRequest == NULL) {
    PTRACE(3, "RAS\tTimed out or received sequence number (" << seqNum << ") for PDU we never requested");
    return FALSE;
  }

  lastRequest->responseMutex.Wait();
  lastRequest->CheckResponse(reqTag, reason);
  return TRUE;
}

///////////////////////////////////////////////////////////////////////

H323_AnnexG::Request::Request(unsigned seqNum, H501PDU & pdu, const H323TransportAddress & address)
  : requestAddress(address),
    requestPDU(pdu)
{
  sequenceNumber = seqNum;
}


BOOL H323_AnnexG::Request::Poll(H323_AnnexG & h501Channel)
{
  H323EndPoint & endpoint = h501Channel.GetEndPoint();

  responseResult = AwaitingResponse;

  for (unsigned retry = 1; retry <= h501Channel.GetRequestRetries(); retry++) {

    // To avoid race condition with RIP must set timeout before sending the packet
    whenResponseExpected = PTimer::Tick() + h501Channel.GetRequestTimeout();

    if (!h501Channel.WriteTo(requestPDU, requestAddress))
      break;

    PTRACE(3, "H501\tWaiting on response to seqnum=" << requestPDU.GetSequenceNumber()
           << " for " << setprecision(1) << h501Channel.GetRequestTimeout() << " seconds");

    do {
      // Wait for a response
      responseHandled.Wait(whenResponseExpected - PTimer::Tick());

      PWaitAndSignal mutex(responseMutex); // Wait till lastRequest goes out of scope

      switch (responseResult) {
        case AwaitingResponse :  // Was a timeout
          responseResult = NoResponseReceived;
          break;

        case ConfirmReceived :
          return TRUE;

        case RejectReceived :
          return FALSE;

        default : // RequestInProgress
          responseResult = AwaitingResponse; // Keep waiting
      }

      PTRACE(3, "H501\tWaiting again on response to seqnum=" << requestPDU.GetSequenceNumber()
             << " for " << setprecision(1) << (whenResponseExpected - PTimer::Tick()) << " seconds");
    } while (responseResult == AwaitingResponse);

    PTRACE(1, "H501\tTimeout on request seqnum=" << requestPDU.GetSequenceNumber()
           << ", try #" << retry << " of " << endpoint.GetRasRequestRetries());
  }

  return FALSE;
}

void H323_AnnexG::Request::CheckResponse(unsigned reqTag, const PASN_Choice * reason)
{
  if (requestPDU.GetTag() != reqTag) {
    PTRACE(3, "H501\tReceived reply for incorrect PDU tag.");
    responseResult = RejectReceived;
    rejectReason = UINT_MAX;
    return;
  }

  if (reason == NULL) {
    responseResult = ConfirmReceived;
    return;
  }

  PTRACE(1, "H501\t" << requestPDU.m_body.GetTagName()
         << " rejected: " << reason->GetTagName());

  responseResult = RejectReceived;
  rejectReason = reason->GetTag();
}


void H323_AnnexG::Request::OnReceiveRIP(const H501PDU & pdu)
{
  const H501_RequestInProgress & rip = (H501_RequestInProgress &)pdu;
  responseResult = RequestInProgress;
  whenResponseExpected = PTimer::Tick() + PTimeInterval(rip.m_delay);
}
