/*
 * peclient.cxx
 *
 * H.323 Annex G Peer Element client protocol handler
 *
 * Open H323 Library
 *
 * Copyright (c) 2003 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: peclient.cxx,v $
 * Revision 1.4  2003/03/01 00:22:26  craigs
 * New PeerElement implementation
 *
 * Revision 1.3  2003/02/25 06:48:19  robertj
 * More work on PDU transaction abstraction.
 *
 * Revision 1.2  2003/02/21 07:23:18  robertj
 * Fixed up some comments
 *
 * Revision 1.1  2003/02/21 05:27:06  craigs
 * Initial version
 *
 */

#include <ptlib.h>

#ifdef __GNUC__
#pragma implementation "peclient.h"
#endif

#include "peclient.h"

#include "h323ep.h"
#include "h323annexg.h"
#include "h323pdu.h"

#define new PNEW

////////////////////////////////////////////////////////////////

H323PeerElement::H323PeerElement(H323EndPoint & ep, H323Transport * trans)
  : H323_AnnexG(ep, trans),
    requestMutex(1, 1)
{
  Construct();
}

H323PeerElement::H323PeerElement(H323EndPoint & ep, const H323TransportAddress & addr)
  : H323_AnnexG(ep, addr),
    requestMutex(1, 1)
{
  Construct();
}

void H323PeerElement::Construct()
{
  hasClearingHouse  = FALSE;
  monitorStop       = FALSE;
  elementIdentifier = PProcess::Current().GetName();

  transport->SetPromiscuous(H323Transport::AcceptFromAny);
  StartChannel();

  monitor = PThread::Create(PCREATE_NOTIFIER(MonitorMain), 0,
                            PThread::NoAutoDeleteThread,
                            PThread::NormalPriority,
                            "AnnexGMonitor:%x");
}

H323PeerElement::~H323PeerElement()
{
  monitorStop = TRUE;
  monitorTickle.Signal();
  monitor->WaitForTermination();
  delete monitor;
}

BOOL H323PeerElement::SetClearingHouse(const H323TransportAddress & addr)
{
  PWaitAndSignal m(clearingHouseMutex);

  if (hasClearingHouse)
    ServiceRelease(clearingHouse, H501_ServiceReleaseReason::e_terminated); 

  clearingHouse = addr;
  hasClearingHouse = ServiceRequest(addr);

  return hasClearingHouse;
}


BOOL H323PeerElement::SetListener(const H323TransportAddress & iface)
{
  PWaitAndSignal m(transportMutex);

  if (transport->GetLocalAddress().IsEquivalent(iface)) {
    PTRACE(2, "AnnexG\tAlready have listener for " << iface);
    return TRUE;
  }

  PIPSocket::Address addr;
  WORD port = DefaultUdpPort;
  if (!iface.GetIpAndPort(addr, port)) {
    PTRACE(2, "AnnexG\tCannot create listener for " << iface);
    return FALSE;
  }

  transport->CleanUpOnTermination();
  delete transport;
  transport = new H323TransportUDP(endpoint, addr, port);
  transport->SetPromiscuous(H323Transport::AcceptFromAny);

  return TRUE;
}

/*
void H323PeerElement::Connect(const H323TransportAddress & address)
{
  transport->SetRemoteAddress(address);
  transport->Connect();
}
*/

void H323PeerElement::PrintOn(ostream & strm) const
{
  if (!elementIdentifier)
    strm << elementIdentifier << "@";

  H323TransportAddress addr = transport->GetLocalAddress();

  PIPSocket::Address ip;
  WORD port;
  if (addr.GetIpAndPort(ip, port)) {
    strm << PIPSocket::GetHostName(ip);
    if (port != DefaultUdpPort)
      strm << ':' << port;
  }
  else
    strm << addr;
}

void H323PeerElement::MonitorMain(PThread &, INT)
{
  PTRACE(3, "AnnexG\tBackground thread started");

  for (;;) {
    monitorTickle.Wait();
    if (monitorStop)
      break;
  }

  PTRACE(3, "AnnexG\tBackground thread ended");
}


void H323PeerElement::TickleMonitor(PTimer &, INT)
{
  monitorTickle.Signal();
}

BOOL H323PeerElement::AddDescriptor(const PString & alias, const H323TransportAddress & addr)
{
  PIPSocket::Address ip;
  WORD port;
  if (!addr.GetIpAndPort(ip, port))
    return FALSE;

  return AddDescriptor(alias, ip, port);
}


BOOL H323PeerElement::AddDescriptor(const PString & alias, const PIPSocket::Address & address, WORD port)
{
  // see if there is already a descriptor with this address
  PSafePtr<Descriptor> descriptor = descriptors.FindWithLock(Descriptor(alias), PSafeReadWrite);

  // if address already exists, then update information and send changed
  if (descriptor != NULL) {
    if (descriptor->address == address && descriptor->port == port)
      return FALSE;

    descriptor->address = address;
    descriptor->port    = port;
    UpdateChangeDescriptor(descriptor);
    return TRUE;
  }

  // else create new descriptor, and send add
  descriptor = new Descriptor(alias);
  descriptor->address = address;
  descriptor->port    = port;
  descriptors.Append(descriptor);

  UpdateAddDescriptor(descriptor);

  return TRUE;
}

BOOL H323PeerElement::UpdateAddDescriptor(Descriptor * descriptor)
{
  if (!hasClearingHouse)
    return FALSE;

  H501PDU pdu;
  H501_DescriptorUpdate & body = pdu.BuildDescriptorUpdate(GetNextSequenceNumber(), transport->GetLastReceivedAddress());

  // put our address into the sender field
  H323SetAliasAddress(transport->GetLastReceivedAddress(), body.m_sender, H225_AliasAddress::e_transportID);

  // add information
  body.m_updateInfo.SetSize(1);
  H501_UpdateInformation & info = body.m_updateInfo[0];
  info.m_descriptorInfo.SetTag(H501_UpdateInformation_descriptorInfo::e_descriptor);

  H501_Descriptor & h501Descriptor = info.m_descriptorInfo;
  descriptor->CopyTo(h501Descriptor, transport->GetLastReceivedAddress());

  // make the reauest
  Request request(pdu.GetSequenceNumber(), pdu, clearingHouse);
  return MakeRequest(request);
}

BOOL H323PeerElement::UpdateChangeDescriptor(Descriptor * /*descriptor*/)
{
  return FALSE;
}

BOOL H323PeerElement::UpdateDeleteDescriptor(Descriptor * /*descriptor*/)
{
  return FALSE;
}

BOOL H323PeerElement::ServiceRequest(const H323TransportAddress & peer)
{
  // build the service request
  H501PDU pdu;
  H501_ServiceRequest & body = pdu.BuildServiceRequest(GetNextSequenceNumber(), transport->GetLastReceivedAddress());

  ServiceRelationship * newSR = NULL;

  {
    // check to see if we have a service relationship with the peer already
    PSafePtr<ServiceRelationship> sr = serviceRelationships.FindWithLock(ServiceRelationship(peer), PSafeReadWrite);

    // if there is a previous service relationship, then get the serviceID
    if (sr != NULL) {
      pdu.m_common.IncludeOptionalField(H501_MessageCommonInfo::e_serviceID);
      pdu.m_common.m_serviceID = sr->serviceID;
    }

    else
      newSR = new ServiceRelationship(peer);
  }

  // include the element indentifier
  body.IncludeOptionalField(H501_ServiceRequest::e_elementIdentifier);
  body.m_elementIdentifier = elementIdentifier;

  // make the request
  Request request(pdu.GetSequenceNumber(), pdu, peer);
  H501_MessageCommonInfo info;
  request.responseInfo = &info;
  if (MakeRequest(request)) {
    newSR->serviceID = info.m_serviceID;
    if (newSR != NULL)
      serviceRelationships.Append(newSR);
    OnAddServiceRelationship(peer);
    return TRUE;
  }

  PTRACE(3, "AnnexG\tFailed service request to " << peer);

  if (newSR != NULL)
    delete newSR;

  return FALSE;
}

BOOL H323PeerElement::OnReceiveServiceConfirmation(const H501_MessageCommonInfo & common, const H501_ServiceConfirmation & pdu)
{
  if (!H323_AnnexG::OnReceiveServiceConfirmation(common, pdu))
    return FALSE;

  if (lastRequest->responseInfo != NULL)
    *(H501_MessageCommonInfo *)lastRequest->responseInfo = common;

  return TRUE;
}


BOOL H323PeerElement::ServiceRelease(const H323TransportAddress & peer, unsigned reason)
{
  // remove any previous check to see if we have a service relationship with the peer already
  PSafePtr<ServiceRelationship> sr = serviceRelationships.FindWithLock(ServiceRelationship(peer), PSafeReadWrite);
  if (sr == NULL)
    return FALSE;

  // send the request - no response
  H501PDU pdu;
  H501_ServiceRelease & body = pdu.BuildServiceRelease(GetNextSequenceNumber());
  pdu.m_common.m_serviceID = sr->serviceID;
  body.m_reason = reason;
  WriteTo(pdu, peer);

  OnRemoveServiceRelationship(peer);

  serviceRelationships.Remove(sr);

  return TRUE;
}


BOOL H323PeerElement::MakeRequest(Request & request)
{
  requestMutex.Wait();
  BOOL stat = H323_AnnexG::MakeRequest(request);
  requestMutex.Signal();
  return stat;
}

//////////////////////////////////////////////////////////////////////////////

void H323PeerElement::Descriptor::CopyTo(H501_Descriptor & descriptor, const H323TransportAddress & contact)
{
  descriptor.m_descriptorInfo.m_descriptorID = descriptorID;
  descriptor.m_descriptorInfo.m_lastChanged  = lastChanged.AsString("YYYYMMDDHHmmSS");

  descriptor.m_templates.SetSize(1);
  H501_AddressTemplate & addressTemplate = descriptor.m_templates[0];

  addressTemplate.m_pattern.SetSize(1);
  addressTemplate.m_pattern[0].SetTag(H501_Pattern::e_specific);
  H225_AliasAddress & address = addressTemplate.m_pattern[0];
  H323SetAliasAddress(alias, address, H225_AliasAddress::e_transportID);

  addressTemplate.m_routeInfo.SetSize(1);
  H501_RouteInformation & routeInfo = addressTemplate.m_routeInfo[0];
  routeInfo.m_messageType.SetTag(H501_RouteInformation_messageType::e_sendSetup);
  routeInfo.m_callSpecific = FALSE;

  routeInfo.m_contacts.SetSize(1);
  H501_ContactInformation & contactInfo = routeInfo.m_contacts[0];
  H323SetAliasAddress(contact, contactInfo.m_transportAddress, H225_AliasAddress::e_transportID);
  contactInfo.m_priority = 0;

  routeInfo.IncludeOptionalField(H501_RouteInformation::e_type);
	H225_EndpointType & epType = routeInfo.m_type;

  //info.IncludeOptionalField(H225_EndpointType::e_vendor);
  //SetVendorIdentifierInfo(info.m_vendor);

  //switch (terminalType) {
  //  case e_TerminalOnly :
  //  case e_TerminalAndMC :
      epType.IncludeOptionalField(H225_EndpointType::e_terminal);
  //    break;
  //  case e_GatewayOnly :
  //  case e_GatewayAndMC :
  //  case e_GatewayAndMCWithDataMP :
  //  case e_GatewayAndMCWithAudioMP :
  //  case e_GatewayAndMCWithAVMP :
  //    epType.IncludeOptionalField(H225_EndpointType::e_gateway);
  //    break;
  //  case e_GatekeeperOnly :
  //  case e_GatekeeperWithDataMP :
  //  case e_GatekeeperWithAudioMP :
  //  case e_GatekeeperWithAVMP :
  //    epType.IncludeOptionalField(H225_EndpointType::e_gatekeeper);
  //    break;
  //  case e_MCUOnly :
  //  case e_MCUWithDataMP :
  //  case e_MCUWithAudioMP :
  //  case e_MCUWithAVMP :
  //    epType.IncludeOptionalField(H225_EndpointType::e_mcu);
  //    epType.m_mc = TRUE;
  //}
}