/*
 * h263codec.cxx
 *
 * H.323 protocol handler
 *
 * Open H323 Library
 * 
 * Copyright (c) 1998-2000 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library. 
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of this code were written with the financial assistance of 
 *          AliceStreet (http://www.alicestreet.com) 
 *
 * Contributor(s): Guilhem Tardy (gtardy@marchnetworks.com)
 *
 * $Log: h263codec.cxx,v $
 * Revision 1.16  2003/06/06 06:32:08  rjongbloed
 * Fixed MSVC warning
 *
 * Revision 1.15  2003/06/06 05:18:54  dereksmithies
 * Fix startup delay bug. Remove all large packets from the network. Improve reliability.
 *
 * Revision 1.14  2003/06/03 05:01:23  rjongbloed
 * Fixed some trace logging (removed usage of cerr)
 *
 * Revision 1.13  2003/06/03 00:49:01  dereksmithies
 * Fix divide by zero error, which happens on win95 & win98 machines.
 *
 * Revision 1.12  2003/06/02 07:56:56  rjongbloed
 * Fixed media format passed to ancestor codec class
 *
 * Revision 1.11  2003/05/27 04:20:59  dereksmithies
 * Fix so that codec resizes correctly after capability exchange
 * No longer crashes if one endpoint is CIF, other is QCIF.
 *
 * Revision 1.10  2003/05/16 04:42:24  rjongbloed
 * Removed extraneous code, and other cosmetic changes.
 * Fixed compare function for capability.
 * Extra trace logging.
 *
 * Revision 1.9  2003/05/15 01:35:27  dereksmithies
 * Frame length fix
 *
 * Revision 1.8  2003/05/14 13:51:51  rjongbloed
 * Removed hack of using special payload type for H.263 for a method which
 *   would be less prone to failure in the future.
 * Removed static "initialisation" function as this should be done
 *   internally and not in the application.
 * Changed media format to be straight H.263 and not OpalH.263
 * Moved Win32 stderr output in ffmpeg AVCODEC interception from
 *   application to library.
 * Fixed some warnings.
 *
 * Revision 1.7  2003/05/14 03:07:17  rjongbloed
 * Made sure video buffer is large enough
 *
 * Revision 1.6  2003/05/05 11:59:25  robertj
 * Changed to use autoconf style selection of options and subsystems.
 *
 * Revision 1.5  2003/05/02 04:21:30  craigs
 * Added lots of extra H.263 support
 *
 * Revision 1.4  2003/04/21 21:50:22  dereks
 * Implement suggestion from Guilhem Tardy. Many thanks.
 *
 * Revision 1.3  2003/04/16 04:26:57  dereks
 * Initial release of h263 codec, which utilises the ffmpeg library.
 * Thanks to Guilhem Tardy, and to AliceStreet.
 *
 * Revision 1.2  2002/08/05 10:03:47  robertj
 * Cosmetic changes to normalise the usage of pragma interface/implementation.
 *
 * Revision 1.1  2002/05/19 22:31:12  dereks
 * Initial release of stub file for h263 codec. Thanks Guilhem Tardy.
 */

/*
 *  Initial release notes from Guilhem Tardy::
 *
 * Added support for video capabilities & codec, still needs the actual codec itself!
 * The code for varying bit rate is copied from h261codec.cxx,
 * until it is moved to a separate file common to both video codecs.
 *
 */
#include <ptlib.h>

#ifdef __GNUC__
#pragma implementation "h263codec.h"
#endif

#include "h263codec.h"

#if H323_AVCODEC

#include "h245.h"
#include "rtp.h"

extern "C" {
#include <avcodec.h>
};


#if defined(_MSC_VER)
#pragma comment(lib, H323_AVCODEC_LIBRARY)
#endif


#define new PNEW

H263PacketList H323_H263Codec::encodedPackets;

static OpalMediaFormat const H263_MediaFormat("H.263",
                                              OpalMediaFormat::DefaultVideoSessionID,
                                              RTP_DataFrame::H263,
                                              FALSE,  // No jitter for video
                                              180000, // bits/sec
                                              2000,   // Not sure of this value!
                                              0,      // No intrinsic time per frame
                                              OpalMediaFormat::VideoTimeUnits);



//////////////////////////////////////////////////////////////////////////////

H323_H263Capability::H323_H263Capability(unsigned _sqcifMPI,
					 unsigned _qcifMPI,
					 unsigned _cifMPI,
					 unsigned _cif4MPI,
					 unsigned _cif16MPI,
					 unsigned _maxBitRate,
					 unsigned _videoFrameRate,
					 BOOL _unrestrictedVector,
					 BOOL _arithmeticCoding,
					 BOOL _advancedPrediction,
					 BOOL _pbFrames,
					 BOOL _temporalSpatialTradeOff,
					 unsigned _hrd_B,
					 unsigned _bppMaxKb,
					 unsigned _slowSqcifMPI,
					 unsigned _slowQcifMPI,
					 unsigned _slowCifMPI,
					 unsigned _slowCif4MPI,
					 unsigned _slowCif16MPI,
					 BOOL _errorCompensation)
{
  sqcifMPI = (_sqcifMPI>0?_sqcifMPI:-(int)_slowSqcifMPI);
  qcifMPI = (_qcifMPI>0?_qcifMPI:-(int)_slowQcifMPI);
  cifMPI = (_cifMPI>0?_cifMPI:-(int)_slowCifMPI);
  cif4MPI = (_cif4MPI>0?_cif4MPI:-(int)_slowCif4MPI);
  cif16MPI = (_cif16MPI>0?_cif16MPI:-(int)_slowCif16MPI);

  maxBitRate = _maxBitRate;
  videoFrameRate = _videoFrameRate;

  temporalSpatialTradeOff = _temporalSpatialTradeOff;
  pbFrames = _pbFrames;
  advancedPrediction = _advancedPrediction;
  arithmeticCoding = _arithmeticCoding;
  unrestrictedVector = _unrestrictedVector;

  hrd_B = _hrd_B;
  bppMaxKb = _bppMaxKb;

  errorCompensation = _errorCompensation;
}


PObject * H323_H263Capability::Clone() const
{
  return new H323_H263Capability(*this);
}


PObject::Comparison H323_H263Capability::Compare(const PObject & obj) const
{
  Comparison result = H323Capability::Compare(obj);
  if (result != EqualTo)
    return result;

  PAssert(obj.IsDescendant(H323_H263Capability::Class()), PInvalidCast);
  const H323_H263Capability & other = (const H323_H263Capability &)obj;

  if ((sqcifMPI > other.sqcifMPI) ||
      (qcifMPI > other.qcifMPI) ||
      (cifMPI > other.cifMPI) ||
      (cif4MPI > other.cif4MPI) ||
      (cif16MPI > other.cif16MPI))
    return GreaterThan;

  if ((cif16MPI < other.cif16MPI) ||
      (cif4MPI < other.cif4MPI) ||
      (cifMPI < other.cifMPI) ||
      (qcifMPI < other.qcifMPI))
    return LessThan;

  return EqualTo;
}


PString H323_H263Capability::GetFormatName() const
{
  PString ret = H263_MediaFormat;

  if (sqcifMPI)
    ret += "-SQCIF";

  if (qcifMPI)
    ret += "-QCIF";

  if (cifMPI)
    ret += "-CIF";

  if (cif4MPI)
    ret += "-CIF4";

  if (cif16MPI)
    ret += "-CIF16";

  return ret;
}


unsigned H323_H263Capability::GetSubType() const
{
  return H245_VideoCapability::e_h263VideoCapability;
}


BOOL H323_H263Capability::OnSendingPDU(H245_VideoCapability & cap) const
{
  cap.SetTag(H245_VideoCapability::e_h263VideoCapability);

  H245_H263VideoCapability & h263 = cap;
  if (sqcifMPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_sqcifMPI);
    h263.m_sqcifMPI = sqcifMPI;
  }
  if (qcifMPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_qcifMPI);
    h263.m_qcifMPI = qcifMPI;
  }
  if (cifMPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_cifMPI);
    h263.m_cifMPI = cifMPI;
  }
  if (cif4MPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_cif4MPI);
    h263.m_cif4MPI = cif4MPI;
  }
  if (cif16MPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_cif16MPI);
    h263.m_cif16MPI = cif16MPI;
  }
  h263.m_temporalSpatialTradeOffCapability = temporalSpatialTradeOff;
  h263.m_maxBitRate = maxBitRate;
  if (sqcifMPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowSqcifMPI);
    h263.m_slowSqcifMPI = -sqcifMPI;
  }
  if (qcifMPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowQcifMPI);
    h263.m_slowQcifMPI = -qcifMPI;
  }
  if (cifMPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowCifMPI);
    h263.m_slowCifMPI = -cifMPI;
  }
  if (cif4MPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowCif4MPI);
    h263.m_slowCif4MPI = -cif4MPI;
  }
  if (cif16MPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowCif16MPI);
    h263.m_slowCif16MPI = -cif16MPI;
  }

  return TRUE;
}


BOOL H323_H263Capability::OnSendingPDU(H245_VideoMode & pdu) const
{
  pdu.SetTag(H245_VideoMode::e_h263VideoMode);
  H245_H263VideoMode & mode = pdu;
  mode.m_resolution.SetTag(cif16MPI ? H245_H263VideoMode_resolution::e_cif16
			  :(cif4MPI ? H245_H263VideoMode_resolution::e_cif4
			   :(cifMPI ? H245_H263VideoMode_resolution::e_cif
			    :(qcifMPI ? H245_H263VideoMode_resolution::e_qcif
			     : H245_H263VideoMode_resolution::e_sqcif))));
  mode.m_bitRate = maxBitRate;
  mode.m_unrestrictedVector = unrestrictedVector;
  mode.m_arithmeticCoding = arithmeticCoding;
  mode.m_advancedPrediction = advancedPrediction;
  mode.m_pbFrames = pbFrames;
  mode.m_errorCompensation = errorCompensation;

  return TRUE;
}


BOOL H323_H263Capability::OnReceivedPDU(const H245_VideoCapability & cap)
{
  if (cap.GetTag() != H245_VideoCapability::e_h263VideoCapability)
    return FALSE;

  const H245_H263VideoCapability & h263 = cap;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_sqcifMPI))
    sqcifMPI = h263.m_sqcifMPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowSqcifMPI))
    sqcifMPI = -(int)h263.m_slowSqcifMPI;
  else
    sqcifMPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_qcifMPI))
    qcifMPI = h263.m_qcifMPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowQcifMPI))
    qcifMPI = -(int)h263.m_slowQcifMPI;
  else
    qcifMPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_cifMPI))
    cifMPI = h263.m_cifMPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowCifMPI))
    cifMPI = -(int)h263.m_slowCifMPI;
  else
    cifMPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_cif4MPI))
    cif4MPI = h263.m_cif4MPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowCif4MPI))
    cif4MPI = -(int)h263.m_slowCif4MPI;
  else
    cif4MPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_cif16MPI))
    cif16MPI = h263.m_cif16MPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowCif16MPI))
    cif16MPI = -(int)h263.m_slowCif16MPI;
  else
    cif16MPI = 0;
  maxBitRate = h263.m_maxBitRate;
  unrestrictedVector = h263.m_unrestrictedVector;
  arithmeticCoding = h263.m_arithmeticCoding;
  advancedPrediction = h263.m_advancedPrediction;
  pbFrames = h263.m_pbFrames;
  temporalSpatialTradeOff = h263.m_temporalSpatialTradeOffCapability;
  hrd_B = h263.m_hrd_B;
  bppMaxKb = h263.m_bppMaxKb;
  errorCompensation = h263.m_errorCompensation;

  return TRUE;
}


H323Codec * H323_H263Capability::CreateCodec(H323Codec::Direction direction) const
{
  return new H323_H263Codec(direction, sqcifMPI, qcifMPI, cifMPI, cif4MPI, cif16MPI, maxBitRate, videoFrameRate);
}

//////////////////////////////////////////////////////////////////////////////

#if PTRACING 

extern "C" {
  extern void (*avcodec_fprintf_fn)(char *);
};

static void h263_printon(char * str)
{
  PTRACE(6, "AVCODEC\t" << str);
}

#endif


//////////////////////////////////////////////////////////////////////////////

H323_H263Codec::H323_H263Codec(Direction dir,
			       unsigned _sqcifMPI,
			       unsigned _qcifMPI,
			       unsigned _cifMPI,
			       unsigned _cif4MPI,
			       unsigned _cif16MPI,
			       unsigned _maxBitRate,
			       unsigned _videoFrameRate)
  : H323VideoCodec(H263_MediaFormat, dir)
{
  PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder created. Data rate=" << _maxBitRate 
	 << " Frame rate=" << _videoFrameRate);

  static BOOL uninitialised_avcodec = TRUE;
  bitsSent = 0;
  if (uninitialised_avcodec) {
    uninitialised_avcodec = FALSE;

    /* must be called before using avcodec lib */
    avcodec_init();

    register_avcodec(&h263_encoder);
    register_avcodec(&h263p_encoder);
    register_avcodec(&h263_decoder);

#if PTRACING
    avcodec_fprintf_fn = &h263_printon;
#endif
  }

  context = NULL;
  picture = NULL;
  
  PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder find"); 
  if (dir == Encoder){
    codec = avcodec_find_encoder(CODEC_ID_H263P);
  } else {
    codec = avcodec_find_decoder(CODEC_ID_H263);
  }

  if (codec == NULL) {   
    PTRACE(1, "H263\tCodec not found on " <<  CODEC_ID_H263P
           << " for H263+ " << (dir == Encoder ? "En" : "De") << "coder");
    return;
  }

  bitRateHighLimit = _maxBitRate;
  if (bitRateHighLimit == 0) {
    PTRACE(3, "H263\tData Rate is set to 1000 Kb/sec, as supplied value (0) is invalid");
    bitRateHighLimit = 1000 * 1024;
  }
  
  framesPerSec = _videoFrameRate;
  if (framesPerSec == 0) {
    PTRACE(3, "H263\tFrame Rate is set to 25 frames/sec, as supplied value (0) is invalid");
    framesPerSec = 25;
  }

  int shifts = -1;
  if (_sqcifMPI) { 
    shifts = 0;     PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder for _sqcifMPI ");
  }
  if (_qcifMPI) {
    shifts = 1;  PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder for _qcifMPI"); 
  }
  if (_cifMPI) { 
    shifts = 2; PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder for _cifMPI");
  }
  if (_cif4MPI) {
    shifts = 3; PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder for _cif4MPI");
  }
  if (_cif16MPI) {
    shifts = 4; PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder for _cif16MPI");
  }

  if (shifts < 0) {
    PTRACE(1, "H263\tERROR in definition of h263 size");
    return;
  }

  PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder created." 
	 << "for a size of " << (88 << shifts) << "x" << (72 << shifts));
  
  Resize(88 << shifts , 72 << shifts); //Fill picture structure, open codec.
  frameNum = 0;
}


H323_H263Codec::~H323_H263Codec()
{
  PWaitAndSignal mutex1(videoHandlerActive);

  CloseCodec();
}

void H323_H263Codec::InitialiseCodec()
{
  context = avcodec_alloc_context();
  picture = avcodec_alloc_frame();

  PTRACE(6, "H263\tCall avcodec_get_context_defaults");
  avcodec_get_context_defaults(context);

  if (direction == Encoder) {
    context->max_qdiff= 3; // max q difference between frames
    context->rc_qsquish = 0; // limit by clipping
    context->bit_rate = (bitRateHighLimit * 3) >> 2;
    context->bit_rate_tolerance = bitRateHighLimit  << 3;
    context->rc_min_rate = 0;
    context->rc_max_rate = bitRateHighLimit;    /* resolution must be a multiple of two */

    context->qmin = 2;
    PINDEX effRate = (frameWidth > 200) ? bitRateHighLimit / 2 : bitRateHighLimit;

    if (effRate > 0) {
      context->qmax = 31;    
    }

    if (effRate > 20000) 
      context->qmax = 20;    

    if (effRate > 40000) {
      context->qmax = 10;
    }

#ifdef FRAME_RATE_BASE  
    context->frame_rate = framesPerSec * FRAME_RATE_BASE;
#else
    context->frame_rate = framesPerSec * DEFAULT_FRAME_RATE_BASE;
#endif
    //Ensure that no frame is too big to be sent by the underlying openh323 code.
    //The ffmpeg code is faulty. Even with the max payload size set to 750, there are 
    //going to be packets that are larger than 1000.
    //Since the mtu is 1500, there is Plenty of room for expansion.
    context->rtp_payload_size = 750; 
    context->rtp_mode = 1;
    context->rtp_callback = &H323_H263Codec::RtpCallback;
  }

  int size = frameWidth * frameHeight; 
  picture->data[0] = rawFrameBuffer.GetPointer();
  picture->data[1] = picture->data[0] + size;
  picture->data[2] = picture->data[1] + (size / 4);
  picture->linesize[0] = frameWidth;
  picture->linesize[1] = frameWidth / 2;
  picture->linesize[2] = frameWidth / 2;
}

void H323_H263Codec::CloseCodec()
{
  PTRACE(6, "H263\tClose h263 video " <<(direction == Encoder ? "En" : "De") << "coder" );
  if (context == NULL) {
    return;
  }

  avcodec_close(context);

  while (direction == Encoder) {
    H263Packet *nextPacket = encodedPackets.GetPacketForThisThread();
    if (nextPacket != NULL) 
      delete nextPacket;
    else 
      break;
  } 
}



// This function is called from H323_RTPChannel::Transmit() in channels.cxx
// to grab, display, and compress a video frame into H263 packets.
//   1- get another frame if all packets of previous frame have been sent
//   2- get next packet on list and send that one
//   3- render the current frame if all of its packets have been sent
BOOL H323_H263Codec::Read(BYTE * /*buffer*/, // pointer to the RTP payload
			  unsigned & length, // returns size of the RTP payload
			  RTP_DataFrame & frame)
{
  PWaitAndSignal mutex1(videoHandlerActive);
  PTRACE(6, "H263\tAcquire next packet from H263 encoder.");

  if (frameNum == 0)
    startTime = PTime();

  if (rawDataChannel == NULL) {
    length = 0;
    PTRACE(3, "H263\tNo channel to connect to video grabber, close down video transmission thread.");
    return FALSE;
  }

  if (!rawDataChannel->IsOpen()) {
     length = 0;
     PTRACE(3, "H263\tVideo grabber is not initialised, close down video transmission thread.");
     return FALSE;
  }
  BOOL ok = TRUE;

  PINDEX out_size;
  if (encodedPackets.NoOutstandingPackets() && (partialPackets.GetFragmentsRemaining() == 0)) {
    // No data is waiting to be read. Go and get some with the read call.

    ((PVideoChannel *)rawDataChannel)->SetGrabberFrameSize(frameWidth, frameHeight);
    if (!rawDataChannel->Read(rawFrameBuffer.GetPointer(), rawFrameLen)) {
      PTRACE(3, "H263\tFailed to read data from video grabber, close down video transmission thread.");
      return FALSE;   //Read failed, return false.
    }
    frameNum++;  //Increment the number of frames grabbed.

    // If there is a Renderer attached, display the grabbed video.
    if (((PVideoChannel *)rawDataChannel)->IsRenderOpen())
      ok = RenderFrame((const BYTE *)rawFrameBuffer); // use data from grab process
    else
      PTRACE(6, "H263\t No renderer open");

    unsigned char *srcPointer = encFrameBuffer.GetPointer(encFrameLen);
#if PTRACING
    PTime encTime;
    out_size =
#endif
        avcodec_encode_video(context, srcPointer, encFrameLen, picture);
    PTRACE(6, "H263\tEncoded " << out_size << " bytes from " << frameWidth << "x" << frameHeight << "  " << 
	   PThread::Current()->GetThreadId() << " in  " << (PTime() - encTime) << " seconds");    
  }

  /* This is a sleazy hack to indicate that this RTP data frame contains
     the non-compliant H.263 RTP payload encoding OpenH323 uses */
  frame.SetExtensionType(263);

  H263Packet *packet;
  if (partialPackets.GetFragmentsRemaining() == 0) {
    packet = encodedPackets.GetPacketForThisThread();
    if (packet == NULL) {
      PTRACE(0, "H263\tEncoder internal error - there should outstanding packets at this point.");
      length = 0;
      return TRUE; //And hope the error condition will fix itself
    }
    partialPackets.AppendH263Packet(packet);
    delete packet;
  }

  packet = partialPackets.GetFragmentForThisThread();
  if (packet == NULL) {
    PTRACE(0, "H263\tEncoder internal error - there should outstanding PARTIAL packets at this point.");   
    length = 0;
    return TRUE; //And hope the error condition will fix itself
  }
  
  length = packet->GetSize();
  if (!frame.SetPayloadSize(length + 2)) {
    PTRACE(1, "Internal error in h263 codec, cause writing too big a packet (" << length << ")");
    length = 0;
    return TRUE;
  }

  memcpy(frame.GetPayloadPtr() + 2, packet->GetData(), length);
  delete packet;
  length += 2;

  *frame.GetPayloadPtr() = (BYTE)partialPackets.GetFragmentIndex();
  *(frame.GetPayloadPtr() + 1) = (BYTE)partialPackets.GetFragmentsTotal();

  PTimeInterval timeLastPacket(1000 * bitsSent / bitRateHighLimit);
  PTimeInterval deltaT = PTime() - startTime;
  if (timeLastPacket > deltaT) {
    PTRACE(5, "H263\tBit rate throttle, "
	   << bitsSent << " bits sent in " << deltaT
	   << " seconds, max=" << bitRateHighLimit
	   << " require time of " << timeLastPacket
	   << " waiting " << (timeLastPacket - deltaT).GetMilliSeconds() << " milli seconds");

   PThread::Current()->Sleep((timeLastPacket - deltaT));
  } 

  startTime = PTime();
  bitsSent = length << 3;

  if (encodedPackets.NoOutstandingPackets() && (partialPackets.GetFragmentsRemaining() == 0)) {
    PTRACE(6, "H263\tFinished sending out frame.");
    frame.SetMarker(TRUE);
  } else 
    frame.SetMarker(FALSE);

  return TRUE;
}


BOOL H323_H263Codec::Write(const BYTE * /*buffer*/,
                           unsigned length,
                           const RTP_DataFrame & frame,
                           unsigned & written)
{
  PWaitAndSignal mutex1(videoHandlerActive);

  if (rawDataChannel == NULL) 
    return FALSE;

  if (lastSequenceNumber == 1) {
    lastSequenceNumber = frame.GetSequenceNumber();
  } else if ((++lastSequenceNumber) != frame.GetSequenceNumber()) {
    PTRACE(3, "H263\tDetected loss of one video packet. Will recover.");
    lastSequenceNumber = frame.GetSequenceNumber();
  }

  // always indicate we have written the entire packet
  written = length;

  // get payload
  BYTE * payload    = frame.GetPayloadPtr();
  PINDEX payloadLen = frame.GetPayloadSize();

  /* This is a sleazy hack to indicate that this RTP data frame contains
     the non-compliant H.263 RTP payload encoding OpenH323 uses */
  if (frame.GetExtensionType() == 263) {
    PINDEX len = encFrameBuffer.GetSize();
    PINDEX fragIndex = *frame.GetPayloadPtr();
    PINDEX nFragments = *(frame.GetPayloadPtr() + 1);
    payloadLen -= 2;
    payload += 2;
    if (fragIndex == 1) {
      currentFragment = 0;
      packetFragments.SetSize(0);
    }

    if (nFragments != 1) {      
      if (fragIndex == (currentFragment + 1)) {
	PINDEX curSize = packetFragments.GetSize();
	memcpy(packetFragments.GetPointer(curSize + payloadLen) + curSize, payload, payloadLen);
	currentFragment++;
      } else {
	currentFragment = 0;
	packetFragments.SetSize(0);
	PTRACE(6, "H263\tError in fragment recovery, index " << fragIndex << " out of turn with " << currentFragment);
	return TRUE;     //*This is an error, but, just return and hope it recovers .
      }
      
      if ((fragIndex != nFragments) || (currentFragment != nFragments)) {
	return TRUE;  //*WE are still building the packet up from fragments.
      }

      payload = packetFragments.GetPointer();
      payloadLen = packetFragments.GetSize();
      memcpy(encFrameBuffer.GetPointer(len + payloadLen) + len, payload, payloadLen);
      payload = encFrameBuffer.GetPointer() + len;
    } else 
      memcpy(encFrameBuffer.GetPointer(len + payloadLen) + len, payload, payloadLen);
  }
  else if (frame.GetPayloadType() == RTP_DataFrame::H263) {

    // must be at least a mode A header
    if (payloadLen < 4)
      return TRUE;

    // get the bits indicating what type of H.263 packet this is as per rfc 2190
    // which tells us the header length

    // see how many bits we need to shift in and out
    PINDEX sbits = (payload[0] >> 3) & 0x7;
    PINDEX ebits = payload[0] & 0x7;

    PINDEX h263HdrLen = 0;

    // mode A - always a GOB start or picture start
    if ((payload[0] & 0x80) == 0) {
      h263HdrLen = 4;
#if PTRACING
      BOOL pbit  = (payload[0] & 0x40) != 0;
      BOOL ibit  = (payload[1] & 0x10) != 0;
      BOOL ubit  = (payload[1] & 0x08) != 0;
      BOOL sbit  = (payload[1] & 0x04) != 0;
      BOOL abit  = (payload[1] & 0x02) != 0;
      PINDEX dbq = (payload[2] >> 3) & 0x7;
      PINDEX tr  = payload[3];
      PINDEX trb = payload[2] & 0x7;
#endif
      PTRACE(6, "H263\tMode A:len=" << payloadLen - 4 << ",sbits=" << sbits << ",ebits=" << ebits << ",p=" << pbit << ",i=" << ibit << ",u=" << ubit << "s=" << sbit << ",a=" << abit << ",dbq=" << dbq << ",trb=" << trb << ",tr=" << tr);
    } 
  
    // mode B 
    else if ((payload[0] & 0x40) == 0) {
      // mode B
      h263HdrLen = 8;
      PTRACE(6, "H263\tMode B:sbits=" << sbits << ",ebits=" << ebits);

    }

    // mode C
    else {
      h263HdrLen = 12;
      PTRACE(6, "H263\tMode C:sbits=" << sbits << ",ebits=" << ebits);
    }

    payload    += h263HdrLen;
    payloadLen -= h263HdrLen;

    // make sure the sbits and ebits line up
    // and combine partial bytes if they do
    if (((sbits + lastebits) & 0x7) != 0) {
      PTRACE(1, "oops");
    }

    else if (sbits != 0) {
      PINDEX len = encFrameBuffer.GetSize();
      if (len == 0) {
        encFrameBuffer.SetSize(++len);
      }
      encFrameBuffer[len-1] |= *payload;
      payload++;
      payloadLen--;
    }

    // keep track of ebits for next time
    lastebits = ebits;

    // add in whole bytes
    if (payloadLen > 0) {
      PINDEX len = encFrameBuffer.GetSize();
      memcpy(encFrameBuffer.GetPointer(len + payloadLen) + len, payload, payloadLen);
    }
  }

  else if (frame.GetPayloadType() >= RTP_DataFrame::DynamicBase) {
    // not supported
    return TRUE;
  }

  // if full frame received, then process it
  if (frame.GetMarker()) {

    int got_picture;
    int len = avcodec_decode_video(context, picture, &got_picture, encFrameBuffer.GetPointer(), encFrameBuffer.GetSize());
    PTRACE(6, "H263\tDecoded " << encFrameBuffer.GetSize() << " byte packet. Image= " 
	   << frameWidth << "x" << frameHeight << "  id=" << 	   PThread::Current()->GetThreadId());
    encFrameBuffer.SetSize(0);
    lastebits = 0;
    
    if (len < 0) {
      PTRACE(1, "H263\tError while decoding frame");
      return TRUE;
    }

    if (got_picture) {
                /* the picture is allocated by the decoder. no need to
                   free it */

      RenderFrame(picture);
      frameNum++; // increment the number of frames written
    } 
  } 

  return TRUE;
}


void H323_H263Codec::RtpCallback(void *data, int size, int /*packetNumber*/)
{
  encodedPackets.AppendH263Packet(data, size);
}


BOOL H323_H263Codec::Resize(int _width, int _height)
{
  if ((frameWidth == _width) && (frameHeight == _height)) 
    return TRUE;

  PTRACE(6, "H263\t" << (direction == Encoder ? "En" : "De") << "coder resizing to "
	 << _width << "x" << _height << ".");
  
  frameWidth = _width;
  frameHeight = _height;
  
  rawFrameLen    = (_width * _height * 3) / 2;
  rawFrameBuffer.SetSize(rawFrameLen); // input video frame
  encFrameLen = rawFrameLen; // this could be set to some lower value
  //encFrameBuffer.SetSize(encFrameLen); // encoded video frame
  
  if (context != NULL) { 
    if (context->codec != NULL) {
      PTRACE(6, "H263\t" << (direction == Encoder ? "En" : "De") << "coder closing because of resize to "
	   << _width << "x" << _height << ".");      
      CloseCodec();
    } 
  }
  InitialiseCodec();

  context->width  = frameWidth;
  context->height = frameHeight;
  
  /* open it */
  if (avcodec_open(context, codec) < 0) {
    fprintf(stderr, "could not open codec for H263P\n");
    PTRACE(0, "H263\tVideo " <<(direction == Encoder ? "En" : "De") << "coder open FAILED");
    return FALSE;
  }

  encBytesRead = 0;
  lastebits = 0;
  return TRUE;
}


/* RenderFrame does two things:
   a) Set size of the display frame. This call happens with every frame.
	 A very small overhead.
   b) Display a frame.
*/

BOOL H323_H263Codec::RenderFrame(const void * buffer)
{
  if (rawDataChannel == NULL)
    return TRUE;

  //Now display local image.
  ((PVideoChannel *)rawDataChannel)->SetRenderFrameSize(frameWidth, frameHeight);

  return rawDataChannel->Write(buffer, 0 /*unused parameter*/);
}


BOOL H323_H263Codec::RenderFrame(AVFrame *pict)
{
  PINDEX offset = 0;
  for (PINDEX plane = 0; plane < 3; plane ++) {
    unsigned char *src = pict->data[plane];
    PINDEX        wrap = pict->linesize[plane];
    PINDEX        ysize = (plane == 0) ? context->height : context->height >> 1;
    PINDEX        xsize = (plane == 0) ? context->width  : context->width  >> 1;
    for (PINDEX line = 0; line < ysize; line++) {
      memcpy(rawFrameBuffer.GetPointer() + offset, src + (line * wrap), xsize);
      offset += xsize;
    }
  }
  return RenderFrame((const BYTE *)rawFrameBuffer);
}


void H323_H263Codec::SetTxQualityLevel(int /*qLevel*/)
{
#if 0
  qualityLevel = PMIN(14, PMAX(qLevel,3));

  lowLimit = PMIN(10, qLevel - 2);
  highLimit = qLevel + 12;

  // If there is no bandwidth limit, update the actual value as well
  if (videoBitRate == 0)
    actualQuality = qualityLevel;
#endif
}


void H323_H263Codec::SetBackgroundFill(int /*idle*/)
{
  //  fillLevel = PMIN(99, PMAX(idle,1));
}


void H323_H263Codec::OnLostPartialPicture()
{
  PTRACE(3, "H263\tLost partial picture message ignored, not implemented");
}


void H323_H263Codec::OnLostPicture()
{
  PTRACE(3, "H263\tLost picture message ignored, not implemented");
}


//////////////////////////////////////////////////////////////////////

H263Packet::H263Packet(void *newData, int newSize)
{
  size = newSize;
  data = newData;
  threadId = PThread::Current()->GetThreadId();
}

H263Packet::~H263Packet()
{
}

BOOL H263Packet::MatchingThreadIdentifer(PThreadIdentifer src)
{
  return src == threadId;
}

/////////////////////////////////////////////////////////////////////////////

void H263PacketList::AppendH263Packet(unsigned char *newData, int newSize)
{
  AppendH263Packet((void *)newData, newSize);
}

void H263PacketList::AppendH263Packet(void *newData, int newSize)
{
  PWaitAndSignal m(accessMutex);
  DisallowDeleteObjects();

  H263Packet *nextPacket = new H263Packet(newData, newSize);

  Append(nextPacket);

  PTRACE_IF(2, newSize > 1460, "H263\tEncoder has generated a large rtp packet of size " << newSize);
}


H263Packet *H263PacketList::GetPacketForThisThread()
{
  PWaitAndSignal m(accessMutex);

  PThreadIdentifer thisThread = PThread::Current()->GetThreadId();
  for (PINDEX i = 0; i < GetSize(); i++) {
    H263Packet *answer = (H263Packet *)GetAt(i);
    if (answer->MatchingThreadIdentifer(thisThread)) {
      RemoveAt(i);
      return answer;
    }
  }
  return NULL;
}

BOOL H263PacketList::NoOutstandingPackets()
{
  PWaitAndSignal m(accessMutex);

  PThreadIdentifer thisThread = PThread::Current()->GetThreadId();
  for (PINDEX i = 0; i < GetSize(); i++)
    if (((H263Packet *)GetAt(i))->MatchingThreadIdentifer(thisThread))
      return FALSE;

  return TRUE;
}

//////////////////////////////////////////////////////////////////////
void H263FragmentList::AppendH263Packet(H263Packet *packet)
{
  PINDEX length = packet->GetSize();
  nPackets = (length / 1400) + 1;
  
  PINDEX size = length / nPackets;
  unsigned char *data = (unsigned char *)packet->GetData();

  for (PINDEX i = 0; i < nPackets; i++) {
    PINDEX sendSize = (i == (nPackets - 1)) ? packet->GetSize() - (i * size) : size;
    H263PacketList::AppendH263Packet(data + (i * size), (PINDEX)sendSize);
  }

}

H263Packet *H263FragmentList::GetFragmentForThisThread()
{
  if (GetSize() > 0) {
    H263Packet *answer = (H263Packet *)GetAt(0);
    RemoveAt(0);
    return answer;
  }

  PTRACE(0, "H263\tPartial packet manipulation error - .");
  return NULL;
}

PINDEX  H263FragmentList::GetFragmentsRemaining()
{
  return GetSize();
}  

PINDEX H263FragmentList::GetFragmentIndex()
{
  return nPackets - GetSize();
}

PINDEX H263FragmentList::GetFragmentsTotal()
{
  return nPackets;
}

#endif // H323_AVCODEC


//////////////////////////////////////////////////////////////////////
