/*
 * h263codec.cxx
 *
 * H.323 protocol handler
 *
 * Open H323 Library
 * 
 * Copyright (c) 1998-2000 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): Guilhem Tardy (gtardy@marchnetworks.com)
 *
 * $Log: h263codec.cxx,v $
 * Revision 1.2  2002/08/05 10:03:47  robertj
 * Cosmetic changes to normalise the usage of pragma interface/implementation.
 *
 * Revision 1.1  2002/05/19 22:31:12  dereks
 * Initial release of stub file for h263 codec. Thanks Guilhem Tardy.
 */

/*
 *  Initial release notes from Guilhem Tardy::
 *
 * Added support for video capabilities & codec, still needs the actual codec itself!
 * The code for varying bit rate is copied from h261codec.cxx,
 * until it is moved to a separate file common to both video codecs.
 *
 */

#include <ptlib.h>

#ifdef __GNUC__
#pragma implementation "h263codec.h"
#endif

#include "h263codec.h"

#include "videoio.h"
#include "h245.h"
#include "rtp.h"

#include <netinet/in.h>

#include "marchnetworks/h263decoder.h"
#include "marchnetworks/h263encoder.h"
#include "marchnetworks/h263parser.h"


#define new			PNEW


#define int7_to_int32(a)	(((a) & 0x40) ? (a) | 0xFFFFFF80 : (a))
#define int32_to_int7(a)	((a) & 0x7F)
#define two_compl(a)		(-(a))


static OpalMediaFormat const H263_MediaFormat("H.263",
					      OpalMediaFormat::DefaultVideoSessionID,
					      RTP_DataFrame::H263,
					      FALSE,  // No jitter for video
					      180000, // bits/sec
					      2000,   // Not sure of this value!
					      0,      // No intrinsic time per frame
					      OpalMediaFormat::VideoTimeUnits);


H323_H263Capability::H323_H263Capability(unsigned _sqcifMPI,
					 unsigned _qcifMPI,
					 unsigned _cifMPI,
					 unsigned _cif4MPI,
					 unsigned _cif16MPI,
					 unsigned _maxBitRate,
					 BOOL _unrestrictedVector,
					 BOOL _arithmeticCoding,
					 BOOL _advancedPrediction,
					 BOOL _pbFrames,
					 BOOL _temporalSpatialTradeOff,
					 unsigned _hrd_B,
					 unsigned _bppMaxKb,
					 unsigned _slowSqcifMPI,
					 unsigned _slowQcifMPI,
					 unsigned _slowCifMPI,
					 unsigned _slowCif4MPI,
					 unsigned _slowCif16MPI,
					 BOOL _errorCompensation)
{
  sqcifMPI = (_sqcifMPI>0?_sqcifMPI:-_slowSqcifMPI);
  qcifMPI = (_qcifMPI>0?_qcifMPI:-_slowQcifMPI);
  cifMPI = (_cifMPI>0?_cifMPI:-_slowCifMPI);
  cif4MPI = (_cif4MPI>0?_cif4MPI:-_slowCif4MPI);
  cif16MPI = (_cif16MPI>0?_cif16MPI:-_slowCif16MPI);

  maxBitRate = _maxBitRate;

  temporalSpatialTradeOff = _temporalSpatialTradeOff;
  pbFrames = _pbFrames;
  advancedPrediction = _advancedPrediction;
  arithmeticCoding = _arithmeticCoding;
  unrestrictedVector = _unrestrictedVector;

  hrd_B = _hrd_B;
  bppMaxKb = _bppMaxKb;

  errorCompensation = _errorCompensation;
}


PObject * H323_H263Capability::Clone() const
{
  return new H323_H263Capability(*this);
}


PObject::Comparison H323_H263Capability::Compare(const PObject & obj) const
{
  Comparison result = H323Capability::Compare(obj);
  if (result != EqualTo)
    return result;

  PAssert(obj.IsDescendant(H323_H263Capability::Class()), PInvalidCast);
  const H323_H263Capability & other = (const H323_H263Capability &)obj;

  if ((sqcifMPI && other.sqcifMPI) ||
      (qcifMPI && other.qcifMPI) ||
      (cifMPI && other.cifMPI) ||
      (cif4MPI && other.cif4MPI) ||
      (cif16MPI && other.cif16MPI))
    return EqualTo;

  if ((!cif16MPI && other.cif16MPI) ||
      (!cif4MPI && other.cif4MPI) ||
      (!cifMPI && other.cifMPI) ||
      (!qcifMPI && other.qcifMPI))
    return LessThan;

  return GreaterThan;
}


PString H323_H263Capability::GetFormatName() const
{
  PString ret("H.263");

  if (sqcifMPI)
    ret = ret + "-SQCIF";

  if (qcifMPI)
    ret = ret + "-QCIF";

  if (cifMPI)
    ret = ret + "-CIF";

  if (cif4MPI)
    ret = ret + "-CIF4";

  if (cif16MPI)
    ret = ret + "-CIF16";

  return ret;
}


unsigned H323_H263Capability::GetSubType() const
{
  return H245_VideoCapability::e_h263VideoCapability;
}


BOOL H323_H263Capability::OnSendingPDU(H245_VideoCapability & cap) const
{
  cap.SetTag(H245_VideoCapability::e_h263VideoCapability);

  H245_H263VideoCapability & h263 = cap;
  if (sqcifMPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_sqcifMPI);
    h263.m_sqcifMPI = sqcifMPI;
  }
  if (qcifMPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_qcifMPI);
    h263.m_qcifMPI = qcifMPI;
  }
  if (cifMPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_cifMPI);
    h263.m_cifMPI = cifMPI;
  }
  if (cif4MPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_cif4MPI);
    h263.m_cif4MPI = cif4MPI;
  }
  if (cif16MPI > 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_cif16MPI);
    h263.m_cif16MPI = cif16MPI;
  }
  h263.m_temporalSpatialTradeOffCapability = temporalSpatialTradeOff;
  h263.m_maxBitRate = maxBitRate;
  if (sqcifMPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowSqcifMPI);
    h263.m_slowSqcifMPI = -sqcifMPI;
  }
  if (qcifMPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowQcifMPI);
    h263.m_slowQcifMPI = -qcifMPI;
  }
  if (cifMPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowCifMPI);
    h263.m_slowCifMPI = -cifMPI;
  }
  if (cif4MPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowCif4MPI);
    h263.m_slowCif4MPI = -cif4MPI;
  }
  if (cif16MPI < 0) {
    h263.IncludeOptionalField(H245_H263VideoCapability::e_slowCif16MPI);
    h263.m_slowCif16MPI = -cif16MPI;
  }

  return TRUE;
}


BOOL H323_H263Capability::OnSendingPDU(H245_VideoMode & pdu) const
{
  pdu.SetTag(H245_VideoMode::e_h263VideoMode);
  H245_H263VideoMode & mode = pdu;
  mode.m_resolution.SetTag(cif16MPI ? H245_H263VideoMode_resolution::e_cif16
			  :(cif4MPI ? H245_H263VideoMode_resolution::e_cif4
			   :(cifMPI ? H245_H263VideoMode_resolution::e_cif
			    :(qcifMPI ? H245_H263VideoMode_resolution::e_qcif
			     : H245_H263VideoMode_resolution::e_sqcif))));
  mode.m_bitRate = maxBitRate;
  mode.m_unrestrictedVector = unrestrictedVector;
  mode.m_arithmeticCoding = arithmeticCoding;
  mode.m_advancedPrediction = advancedPrediction;
  mode.m_pbFrames = pbFrames;
  mode.m_errorCompensation = errorCompensation;

  return TRUE;
}


BOOL H323_H263Capability::OnReceivedPDU(const H245_VideoCapability & cap)
{
  if (cap.GetTag() != H245_VideoCapability::e_h263VideoCapability)
    return FALSE;

  const H245_H263VideoCapability & h263 = cap;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_sqcifMPI))
    sqcifMPI = h263.m_sqcifMPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowSqcifMPI))
    sqcifMPI = -h263.m_slowSqcifMPI;
  else
    sqcifMPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_qcifMPI))
    qcifMPI = h263.m_qcifMPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowQcifMPI))
    qcifMPI = -h263.m_slowQcifMPI;
  else
    qcifMPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_cifMPI))
    cifMPI = h263.m_cifMPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowCifMPI))
    cifMPI = -h263.m_slowCifMPI;
  else
    cifMPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_cif4MPI))
    cif4MPI = h263.m_cif4MPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowCif4MPI))
    cif4MPI = -h263.m_slowCif4MPI;
  else
    cif4MPI = 0;
  if (h263.HasOptionalField(H245_H263VideoCapability::e_cif16MPI))
    cif16MPI = h263.m_cif16MPI;
  else if (h263.HasOptionalField(H245_H263VideoCapability::e_slowCif16MPI))
    cif16MPI = -h263.m_slowCif16MPI;
  else
    cif16MPI = 0;
  maxBitRate = h263.m_maxBitRate;
  unrestrictedVector = h263.m_unrestrictedVector;
  arithmeticCoding = h263.m_arithmeticCoding;
  advancedPrediction = h263.m_advancedPrediction;
  pbFrames = h263.m_pbFrames;
  temporalSpatialTradeOff = h263.m_temporalSpatialTradeOffCapability;
  hrd_B = h263.m_hrd_B;
  bppMaxKb = h263.m_bppMaxKb;
  errorCompensation = h263.m_errorCompensation;

  return TRUE;
}


H323Codec * H323_H263Capability::CreateCodec(H323Codec::Direction direction) const
{
  return new H323_H263Codec(direction);
}


//////////////////////////////////////////////////////////////////////////////

H323_H263Codec::H323_H263Codec(Direction dir) : H323VideoCodec("H.263", dir)
{
  PTRACE(3, "H263\t" << (dir == Encoder ? "En" : "De") << "coder created.");

  // no video encoder or parser until we receive a raw video frame
  videoEncoderBitStream = NULL;
  videoEncoder = NULL;
  videoParser = NULL;

  rawFrameBufLen = 0;
  rawFrameBuffer = NULL;
  encFrameBufLen = 0;
  encFrameBuffer = NULL;

  // no video decoder until we receive a packet or decompress a raw video frame
  videoDecoder = NULL;

  // initial size of the window is already 0

  // frame rate estimation accuracy
  // is also average bitrate accuracy
  estFps = 30.0;	// suppose 30 fps
  frameRateDivider = 1;	// varies between 1..6

  // automatic bitrate control
  videoBitRate = 128*1024;

  defBitsPerFrame = (int)((float)videoBitRate / estFps);
  reqBitsPerFrame = defBitsPerFrame * frameRateDivider;
  frameBits = reqBitsPerFrame;		// bit/frame counter
  numOf31 = 0;				// quality too low?
  numOfNot31 = 0;			// quality too high?
  actualQuality = qualityLevel = 10;	// quality around 10
  highLimit = 22;			// quality bounds
  lowLimit = 4;
  frameNum = 0;				// reset frame counter

  timestampDelta = 0;
}


H323_H263Codec::~H323_H263Codec()
{
  PWaitAndSignal mutex1(videoHandlerActive);

  if (videoDecoder) {
    delete videoDecoder;
  }

  if (rawFrameBuffer) {
    free(rawFrameBuffer);
  }

  if (encFrameBuffer) {
    free(encFrameBuffer);
  }

  if (videoEncoderBitStream) {
    delete videoEncoderBitStream;
  }

  if (videoEncoder) {
    delete videoEncoder;
  }

  if (videoParser) {
    delete videoParser;
  }
}


// This function is called from H323_RTPChannel::Transmit() in channels.cxx
// to grab, display, and compress a video frame into H263 packets.
//   1- get another frame if all packets of previous frame have been sent
//   2- get next packet on list and send that one
//   3- render the current frame if all of its packets have been sent
BOOL H323_H263Codec::Read(BYTE * buffer, // pointer to the RTP payload
				unsigned & length, // size of the RTP payload (undefined)
				RTP_DataFrame & frame)
{
  PWaitAndSignal mutex1(videoHandlerActive);
  PTRACE(6, "H263\tAcquire next packet from H263 encoder.");

  if (rawDataChannel == NULL) {
    length = 0;
    PTRACE(3, "H263\tNo channel to connect to video grabber, close down video transmission thread.");
    return FALSE;
  }

  if (!rawDataChannel->IsOpen()) {
     length = 0;
     PTRACE(3, "H263\tVideo grabber is not initialised, close down video transmission thread.");
     return FALSE;
  }

  if (Resize(((PVideoChannel *)rawDataChannel)->GetGrabWidth(), ((PVideoChannel *)rawDataChannel)->GetGrabHeight()) == FALSE) {
    length = 0;
    PTRACE(3, "H263\tNot enough memory to allocate frame buffers, close down video transmission thread.");
    return FALSE;
  }

  if (frameWidth == 0) {
    PTRACE(3, "H263\tVideo grab width is 0 x 0, close down video transmission thread.");
    length=0;
    return FALSE;
  }

  if (((PVideoChannel *)rawDataChannel)->GetVideoReader()->GetColourFormat() == "YUV420P") {

    if (videoEncoderBitStream == NULL) {
      videoEncoderBitStream = new MSBBitStream();
      PTRACE(6, "H263\tvideoEncoderBitStream created.");
    }

    if (videoEncoder == NULL) {
      videoEncoder = new H263Encoder();
      PTRACE(6, "H263\tvideoEncoder created.");
    }
  }
  else {
    PTRACE(3, "H263\tVideo grabber has non-supported colour format, close down video transmission thread.");
    length=0;
    return FALSE;
  }

  if (videoParser == NULL) {
    videoParser = new H263Parser();
    PTRACE(6, "H263\tvideoParser created.");
  }

  BOOL ok=TRUE;

  if (!videoParser->isPacketOutstanding()) {
    if (videoBitRate != 0 && (videoBitRateControlModes & DynamicVideoQuality)) {
      if ((frameNum % frameRateDivider) == 0) {
	int error = reqBitsPerFrame - frameBits; // error signal
	int aerror = PABS(error);
	int act; // action signal

	if (aerror < (reqBitsPerFrame>>4))
	  act = 0;
	else
	  if (aerror < (reqBitsPerFrame>>3))
	    act = error>0 ? 1 : -1;
	  else {
	    act = error * 10 / reqBitsPerFrame;
	    act = PMAX(PMIN(act, 10) , -10);
	  }

	actualQuality -= act;
	actualQuality = PMIN(PMAX(actualQuality, 1) , 31);

	if (actualQuality >= highLimit)
	  numOf31++;
	else
	  numOf31=0;

	if (actualQuality <= lowLimit)
	  numOfNot31++;
	else
	  numOfNot31=0;

	if (numOf31 == 10 /*quality too low?*/) {
	  frameRateDivider++;
	  frameRateDivider = PMIN(frameRateDivider, 6);
	  reqBitsPerFrame = defBitsPerFrame * frameRateDivider;
	  numOf31 = 0;
	}
	if (numOfNot31 == 10 /*quality too high?*/) {
	  frameRateDivider--;
	  frameRateDivider = PMAX(frameRateDivider, 1);
	  reqBitsPerFrame = defBitsPerFrame * frameRateDivider;
	  numOfNot31 = 0;
	}

	PTRACE(6, "H263\tReqBitsPerFrame=" << reqBitsPerFrame <<
		" bitrate=" << videoBitRate <<
		" DefBitsPerFrame=" << defBitsPerFrame <<
		" quality=" << actualQuality);
	PTRACE(6, "H263\tConstant bitrate control - FrameBits=" << frameBits
		<< " action=" << -act << " quality=" << actualQuality
		<< " estfps=" <<  estFps
		<< " framenum=" << frameNum
		<< " FramerateDivider=" << frameRateDivider);
      }
    }

    // No data is waiting to be read. Go and get some with the read call.
    if (rawDataChannel->Read(rawFrameBuffer, rawFrameBufLen)) {
	// If there is a Renderer attached, display the grabbed video.
	if (((PVideoChannel *)rawDataChannel)->IsRenderOpen()) {
	  ok = RenderFrame(rawFrameBuffer); // use data from grab process
	}

	/* Process one frame, encoding INTRA for the first 2 frames and every 32 times, INTER otherwise
	 * (each macroblock should be transmitted at least once every 132 times).
	 * The reason for the first *2* frames is that NetMeeting allegedly requires that many INTRA frames
	 * at the start of a session.
	 */
	PTRACE(6, "H263\tSet videoEncoderBitStream to the encFrameBuffer and compress.");
	videoEncoderBitStream->setBuffer(encFrameBuffer);
	(void)videoEncoder->encode((YUVFrame *)rawFrameBuffer, *videoEncoderBitStream, frameWidth, frameHeight,
					actualQuality, fillLevel, ((frameNum < 2) || !(frameNum & 0x1F) ? 1 : 0));
	PTRACE(6, "H263\tEncode: " << frameWidth << "x" << frameHeight << ", quality="
	       << actualQuality << ", inter=" << ((frameNum < 2) || !(frameNum & 0x1F) ? 1 : 0));

	videoParser->setBuffer(encFrameBuffer, encFrameBufLen);
      }

      if (videoBitRate != 0 && (videoBitRateControlModes & DynamicVideoQuality)) {
	if ((frameNum % frameRateDivider) == 0) {
	  frameBits = 0; // clear frame bits counter
	}

	// estimate fps every 5 frames
	if ((frameNum % 5) == 4) {
	  timeFor5Frames = PTimer::Tick() - timeFor5Frames;
	  estFps = (float)(5000.0 / timeFor5Frames.GetMilliSeconds());
	  defBitsPerFrame = (int) (videoBitRate / estFps);
	  reqBitsPerFrame = defBitsPerFrame * frameRateDivider;
	  timeFor5Frames = PTimer::Tick();
	}
      }

    frameNum++; // increment the number of frames read
    } else {
      PTRACE(3, "H263\tFailed to read data from video grabber, close down video transmission thread.");
      return FALSE;   //Read failed, return false.
    }

    /////////////////////////////////////////////////////////////////
    /// THIS VALUE MUST BE CALCULATED AND NOT JUST SET TO 29.97Hz!!!!
    /////////////////////////////////////////////////////////////////
    timestampDelta = 3003;

  } else { // if (!videoParser->isPacketOutstanding())
    PThread::Current()->Sleep(2);  // place a 2 ms interval between packets
    timestampDelta = 0;
  }

  // read packets of the same frame
  videoParser->readOnePacket(buffer, length); // get next packet and update 'length'
  frame.SetMarker(!videoParser->isPacketOutstanding());

  PTRACE(6, "H263\tSend one packet (" << length << " bytes), marker=" << frame.GetMarker());

  if (videoBitRate != 0 && (videoBitRateControlModes & DynamicVideoQuality)) {
    frameBits += length << 3;     // count current frame bits

#if PTRACING
    bytesSent += length;
    PTimeInterval pT = PTimer::Tick()-startTime;
    if (pT.GetSeconds()>0 && videoBitRate) {
      PTRACE(2, "H263\tTransmit video bit rate is "
	     << (bytesSent<<3)/ (1024 * pT.GetSeconds()) <<" k bps"
	     << " on quality of "<< actualQuality);
      startTime = PTimer::Tick();
    }
#endif
  }

  // This is a simple workaround for bandwidth control. We try to limit the
  // video bandwidth to certain value introducing a variable delay between
  // packets
  if (videoBitRate != 0 &&
      (videoBitRateControlModes & AdaptivePacketDelay))
  {
    PTimeInterval sinceLastReturnInterval = PTimer::Tick() - lastReadReturn;

    // ms = (bytes * 8) / (bps / 1024)
    PInt64 waitForNextPacket = (length*8)/(videoBitRate/1024)
				  - sinceLastReturnInterval.GetMilliSeconds();

    PTRACE(3, "H263\tFrame bits: " << length*8
	   << ", sinceLastReturnInterval: " << sinceLastReturnInterval.GetMilliSeconds()
	   << ", waitForNextPacket: " << waitForNextPacket);

    if (waitForNextPacket>0)
    {
      PThread::Current()->Sleep(waitForNextPacket);
    }

    lastReadReturn = PTimer::Tick();
  }

  return ok;
}


BOOL H323_H263Codec::Write(const BYTE * buffer,
				unsigned length,
				const RTP_DataFrame & frame,
				unsigned & written)
{
  PWaitAndSignal mutex1(videoHandlerActive);

  if (rawDataChannel == NULL) {
    //Some other task has killed our videohandler. Exit.
    return FALSE;
  }

  Buffer videoDecoderBuffer; // structure pointing to a H.263 frame before decoding

  if (videoDecoder == NULL) {
    videoDecoder = new H263Decoder();
    PTRACE(6, "H263\tvideoDecoder created.");
  }

  if ((++lastSequenceNumber) != frame.GetSequenceNumber()) {
    PTRACE(3, "H263\tDetected loss of one video packet. Will recover.");
    lastSequenceNumber = frame.GetSequenceNumber();
    SendMiscCommand(H245_MiscellaneousCommand_type::e_lostPartialPicture);
  }

  // always indicate we have written the entire packet
  written = length;

  // assume buffer is start of H.263 header
  u_int32_t * header = (u_int32_t *) buffer;

  // see if any contributing source
  PINDEX cnt = frame.GetContribSrcCount();
  if (cnt > 0) {
    header += cnt;
    length -= (cnt << 2);
  }

  const BYTE * payload = (const BYTE *) header;

  // decode values from the RTP H263 header
  if (frame.GetPayloadType() == RTP_DataFrame::H263) { // RFC 2190
    struct rfc2190hdr *h = (struct rfc2190hdr *) header;
    *header = ntohl(*header);

    if (!(h->modeA.f)) { // mode A
      // leave 4 bytes for H.263 header
      payload += 4;
      length -= 4;

      PTRACE(6, "H263\tpck rcv (" << length << " bytes): f=" << h->modeA.f << ", p=" << h->modeA.p
	     << ", sbit=" << h->modeA.sbit << ", ebit=" << h->modeA.ebit
	     << ", src=" << h->modeA.src << ", inter=" << h->modeA.i
	     << ", tr=" << h->modeA.tr);

      if (!h->modeA.p) { // I or P frame
        // Setup the buffer
	SetBuffer(&videoDecoderBuffer, payload);
	SetDataSize(&videoDecoderBuffer, (length << 3) - h->modeA.ebit); // get rid of end bit stuffing
	if (h->modeA.sbit) // get rid of start bit stuffing
	  flushbits(&videoDecoderBuffer, h->modeA.sbit);

	// Decode the H.263 frame
	(void)videoDecoder->decodeFromGOB(&videoDecoderBuffer, h->modeA.src, h->modeA.i);
      }
      else { // PB frame
	// not supported
      }
    }
    else { // mode B or C
      *(header + 1) = ntohl(*(header + 1));

      if (!h->modeBC.p) { // mode B
	unsigned hmv1, vmv1;

	// leave 8 bytes for H.263 header
	payload += 8;
	length -= 8;

	PTRACE(6, "H263\tpck rcv (" << length << " bytes): f=" << h->modeBC.f << ", p=" << h->modeBC.p
	       << ", sbit=" << h->modeBC.sbit << ", ebit=" << h->modeBC.ebit
	       << ", src=" << h->modeBC.src << ", q=" << h->modeBC.q
	       << ", gobn=" << h->modeBC.gobn << ", mba=" << h->modeBC.mba
	       << ", inter=" << h->modeBC.i << ", hmv1=" << h->modeBC.hmv1
	       << ", vmv1=" << h->modeBC.vmv1);

	// Setup the buffer
	SetBuffer(&videoDecoderBuffer, payload);
	SetDataSize(&videoDecoderBuffer, (length << 3) - h->modeBC.ebit); // get rid of end bit stuffing
	if (h->modeBC.sbit) // get rid of start bit stuffing
	  flushbits(&videoDecoderBuffer, h->modeBC.sbit);

	hmv1 = int7_to_int32(h->modeBC.hmv1);
	vmv1 = int7_to_int32(h->modeBC.vmv1);

	// Decode the H.263 frame
	(void)videoDecoder->decodeFromMB(&videoDecoderBuffer, h->modeBC.src, h->modeBC.i, h->modeBC.q, h->modeBC.gobn, h->modeBC.mba, two_compl(hmv1), two_compl(vmv1));
      }
      else { // mode C
	*(header + 2) = ntohl(*(header + 2));

	// leave 12 bytes for H.263 header
	payload += 12;
	length -= 12;

	// not supported
      }
    }
  }
  else { // RFC2429
    // not supported
  }

  // If the incoming video stream changes size, resize the rendering device.
  Resize(videoDecoder->width(), videoDecoder->height());

  BOOL ok = TRUE;

  if (frame.GetMarker()) {
    videoDecoder->sync();
    ok = RenderFrame(videoDecoder->getFramePtr());
    frameNum++; // increment the number of frames written
  }

  return ok;
}


/* Resize is relevant to the decoder only, as the encoder does not
   change size mid transmission, does it?
*/
BOOL H323_H263Codec::Resize(int _width, int _height)
{
  if ((frameWidth!=_width) || (frameHeight!=_height)) {

    PTRACE(6, "H263\t" << (direction == Encoder ? "En" : "De") << "coder resizing to "
	   << _width << "x" << _height << ".");

    frameWidth = _width;
    frameHeight = _height;

    if (direction == Encoder) {
      rawFrameBufLen = ((PVideoChannel *)rawDataChannel)->GetVideoReader()->GetMaxFrameBytes();
      rawFrameBuffer = (unsigned char *) realloc(rawFrameBuffer, rawFrameBufLen); // input video frame
      if (rawFrameBuffer == NULL)
	return FALSE;
      PTRACE(6, "H263\t" << rawFrameBufLen << " bytes of memory allocated for rawFrameBuffer.");

      encFrameBufLen = rawFrameBufLen; // this could be set to some lower value
      encFrameBuffer = (unsigned char *) realloc(encFrameBuffer, encFrameBufLen); // encoded video frame
      if (encFrameBuffer == NULL)
	return FALSE;
      PTRACE(6, "H263\t" << encFrameBufLen << " bytes of memory allocated for encFrameBuffer.");
    }
  }

  return TRUE;
}


/* RenderFrame does two things:
   a) Set size of the display frame. This call happens with every frame.
	 A very small overhead.
   b) Display a frame.
*/
BOOL H323_H263Codec::RenderFrame(const void * buffer)
{
  BOOL ok = TRUE;

  if (rawDataChannel != NULL) {
    //Now display local image.
    ((PVideoChannel *)rawDataChannel)->SetRenderFrameSize(frameWidth, frameHeight);
    PTRACE(6, "H263\tSize of video rendering frame set to " << frameWidth << "x" << frameHeight);

    ok = rawDataChannel->Write(buffer, 0 /*unused parameter*/);
  }

  return ok;
}


void H323_H263Codec::SetTxQualityLevel(int qLevel)
{
  qualityLevel = PMIN(14, PMAX(qLevel,3));

  lowLimit = PMIN(10, qLevel - 2);
  highLimit = qLevel + 12;

  // If there is no bandwidth limit, update the actual value as well
  if (videoBitRate == 0)
    actualQuality = qualityLevel;
}


void H323_H263Codec::SetBackgroundFill(int idle)
{
  fillLevel = PMIN(99, PMAX(idle,1));
}


void H323_H263Codec::OnLostPartialPicture()
{
  PTRACE(3, "H263\tLost partial picture message ignored, not implemented");
}


void H323_H263Codec::OnLostPicture()
{
  PTRACE(3, "H263\tLost picture message ignored, not implemented");
}


/////////////////////////////////////////////////////////////////////////////
