// -*- mode: c++ -*-
// Copyright (C) 2002 mediaWays GmbH Internet-Services, Verl, Germany
//
// PURPOSE OF THIS FILE: Implement the ldap_get/set_option functions
//
// - Automatic Version Information via RCS:
//   $Id: ldapoptions.cxx,v 1.2 2002/10/09 00:12:38 robertj Exp $
//   $Source: /home/cvsroot/openh323/src/ldapoptions.cxx,v $

/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations under
 * the License.
 *
 * The Original Code is the OpenLDAP Comatible LDAP-API for OpenH323
 * (libldapapi).
 *
 * The Initial Developer of the Original Code is
 * mediaWays GmbH Internet-Services, Verl, Germany
 * Portions created by the Initial Developer are Copyright (C) 2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Nils Bokermann <Nils.Bokermann@mediaWays.net> (*)
 *                 Martin Frhlich <Martin.Froehlich@mediaWays.net> (*)
 *
 * (*) Contributors marked this way are mediaWays employees, all their work
 *     is subject to mediaWays copyright by their contracts.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * For copies of the GPL or LGPL, see either http://www.fsf.org/licenses/
 * or write to the Free Software Foundation, Inc., 59 Temple Place - Suite
 * 330, Boston, MA 02111-1307, USA.
 *
 * ***** END LICENSE BLOCK ***** */

#include <ptlib.h>
#include <ldapapi.h>
#include "ldap-int.h"


#ifndef lint
// mark object with version info in such a way that it is retrievable by
// the std. version/revision control tools like RCS/CVS ident cmd.. At
// least the strings cmd will extract this info.
volatile static const char vcid[] = "@(#) $Id: ldapoptions.cxx,v 1.2 2002/10/09 00:12:38 robertj Exp $";
volatile static const char vcHid[] = LDAP_OPENH323_H;
#endif

extern struct ldapoptions ldap_int_global_options;


static const LDAPAPIFeatureInfo features[] = {
#ifdef LDAP_API_FEATURE_THREAD_SAFE
	{	/* Basic Thread Safe */
		LDAP_FEATURE_INFO_VERSION,
		"THREAD_SAFE",
		LDAP_API_FEATURE_THREAD_SAFE
	},
#endif
#ifdef LDAP_API_FEATURE_SESSION_THREAD_SAFE
	{	/* Session Thread Safe */
		LDAP_FEATURE_INFO_VERSION,
		"SESSION_THREAD_SAFE",
		LDAP_API_FEATURE_SESSION_THREAD_SAFE
	},
#endif
#ifdef LDAP_API_FEATURE_OPERATION_THREAD_SAFE
	{	/* Operation Thread Safe */
		LDAP_FEATURE_INFO_VERSION,
		"OPERATION_THREAD_SAFE",
		LDAP_API_FEATURE_OPERATION_THREAD_SAFE
	},
#endif
	{0, NULL, 0}
};


const char * ldap_listhosts(vector <LDAP_Hostentry*> hosts) {
  PString p;
  for(vector<LDAP_Hostentry*>::iterator iter=hosts.begin(); iter!=hosts.end(); iter++) {
    p+=(*iter)->hostname;
    p+=(*iter)->port;
  }
  return p;
}

void ldap_clear_hostlist(vector <LDAP_Hostentry*> hosts) {
  while(LDAP_Hostentry *h=hosts.front()) {
    hosts.pop_back();
    delete h;
  }
}

int ldap_get_option(LDAP *ld, int option, void *outvalue){
  struct ldapoptions *lo;
  if(NULL==ld)
    return LDAP_UNAVAILABLE;
  /* Get pointer to global option structure */
  lo = LDAP_INT_GLOBAL_OPT();
  if (NULL == lo)	{
    return LDAP_NO_MEMORY;
  }

  if( lo->ldo_valid != LDAP_INITIALIZED ) {
    ldap_int_initialize(lo);
  }

  if(ld != NULL) {
    lo = ld->ld_options;
  }
  if(outvalue == NULL) {
    /* no place to get to */
    return LDAP_OPT_ERROR;
  }

  switch(option) {
    case LDAP_OPT_API_INFO: {
      struct ldapapiinfo *info = (struct ldapapiinfo *) outvalue;

      if(info == NULL) {
	/* outvalue must point to an apiinfo structure */
	return LDAP_OPT_ERROR;
      }

      if(info->ldapai_info_version != LDAP_API_INFO_VERSION) {
	/* api info version mismatch */
	info->ldapai_info_version = LDAP_API_INFO_VERSION;
	return LDAP_OPT_ERROR;
      }

      info->ldapai_api_version = LDAP_API_VERSION;
      info->ldapai_protocol_version = LDAP_VERSION_MAX;

      if(features[0].ldapaif_name == NULL) {
	info->ldapai_extensions = NULL;
      } else {
	int i;
// 	info->ldapai_extensions = (char **)malloc(sizeof(char *) *
// 					      sizeof(features)/sizeof(LDAPAPIFeatureInfo));
	info->ldapai_extensions = new (char *)[sizeof(features)/sizeof(LDAPAPIFeatureInfo)];
	for(i=0; features[i].ldapaif_name != NULL; i++) {
	  info->ldapai_extensions[i] =
	    strndup(features[i].ldapaif_name,strlen(features[i].ldapaif_name));
	}

	info->ldapai_extensions[i] = NULL;
      }

      info->ldapai_vendor_name = strndup(LDAP_VENDOR_NAME,(size_t)strlen(LDAP_VENDOR_NAME));
      info->ldapai_vendor_version = LDAP_VENDOR_VERSION;

      return LDAP_OPT_SUCCESS;
    } break;

//     case LDAP_OPT_DESC:
//       if( ld == NULL || ld->ld_sb == NULL ) {
// 	/* bad param */
// 	break;
//       }

//       ber_sockbuf_ctrl( ld->ld_sb, LBER_SB_OPT_GET_FD, outvalue );
//       return LDAP_OPT_SUCCESS;
//       /* not in draft */
    case LDAP_OPT_TIMEOUT:
      /* the caller has to free outvalue ! */
      memcpy( outvalue, lo->ldo_tm_api,sizeof(struct timeval));
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_NETWORK_TIMEOUT:
      /* the caller has to free outvalue ! */
      memcpy( outvalue, lo->ldo_tm_net,sizeof(struct timeval));
      return LDAP_OPT_SUCCESS;
    case LDAP_OPT_DEREF:
      * (int *) outvalue = lo->ldo_deref;
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_SIZELIMIT:
      * (int *) outvalue = lo->ldo_sizelimit;
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_TIMELIMIT:
      * (int *) outvalue = lo->ldo_timelimit;
      return LDAP_OPT_SUCCESS;

//     case LDAP_OPT_REFERRALS:
//       * (int *) outvalue = (int) (lo->ldo_referrals==true);
//       return LDAP_OPT_SUCCESS;

//     case LDAP_OPT_RESTART:
//       * (int *) outvalue = (int) (lo->ldo_restart==true);;
//       return LDAP_OPT_SUCCESS;

    case LDAP_OPT_PROTOCOL_VERSION:
      * (int *) outvalue = lo->ldo_version;
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_SERVER_CONTROLS:
      * (LDAPControl ***) outvalue = NULL;
	//	ldap_controls_dup( lo->ldo_sctrls );

      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_CLIENT_CONTROLS:
      * (LDAPControl ***) outvalue = NULL;
	//	ldap_controls_dup( lo->ldo_cctrls );

      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_HOST_NAME:
      * (char **) outvalue = (char *) ldap_listhosts(lo->ldo_hostlist);
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_ERROR_NUMBER:
      if(ld == NULL) {
	/* bad param */
	break;
      }
      * (int *) outvalue = ld->ld_errno;
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_ERROR_STRING:
      if(ld == NULL) {
	/* bad param */
	break;
      }

      if(! ld->ld_error) {
	* (char **) outvalue = NULL;
      } else {
	* (char **) outvalue = strndup(ld->ld_error,strlen(ld->ld_error));
      }

      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_MATCHED_DN:
      if(ld == NULL) {
	/* bad param */
	break;
      }

      if( ld->ld_matched == NULL ) {
	* (char **) outvalue = NULL;
      } else {
	* (char **) outvalue = strndup(ld->ld_matched,strlen(ld->ld_error));
      }

      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_API_FEATURE_INFO: {
      LDAPAPIFeatureInfo *info = (LDAPAPIFeatureInfo *) outvalue;
      int i;

      if(info == NULL) return LDAP_OPT_ERROR;

      if(info->ldapaif_info_version != LDAP_FEATURE_INFO_VERSION) {
	/* api info version mismatch */
	info->ldapaif_info_version = LDAP_FEATURE_INFO_VERSION;
	return LDAP_OPT_ERROR;
      }

      if(info->ldapaif_name == NULL) return LDAP_OPT_ERROR;

      for(i=0; features[i].ldapaif_name != NULL; i++) {
	if(!strcmp(info->ldapaif_name, features[i].ldapaif_name)) {
	  info->ldapaif_version =
	    features[i].ldapaif_version;
	  return LDAP_OPT_SUCCESS;
	}
      }
    }
      break;
      /*
    case LDAP_OPT_DEBUG_LEVEL:
      * (int *) outvalue = lo->ldo_debug;
      return LDAP_OPT_SUCCESS;
      */
    default:
      /* bad param */
      break;
  }

  return LDAP_OPT_ERROR;
}


int ldap_set_option(LDAP *ld, int option, LDAP_CONST void *invalue) {
  struct ldapoptions *lo;
  //int *dbglvl = NULL;

  if(NULL==ld)
    return LDAP_UNAVAILABLE;
  /* Get pointer to global option structure */
  lo = LDAP_INT_GLOBAL_OPT();
  if (lo == NULL)	{
    return LDAP_NO_MEMORY;
  }

  /*
   * The architecture to turn on debugging has a chicken and egg
   * problem. Thus, we introduce a fix here.
   */

//   if (option == LDAP_OPT_DEBUG_LEVEL)
//     dbglvl = (int *) invalue;

  if( lo->ldo_valid != LDAP_INITIALIZED ) {
    ldap_int_initialize(lo);
  }

  if(ld != NULL) {
    lo = ld->ld_options;
  }

  switch(option) {
    // options which can withstand invalue == NULL
    case LDAP_OPT_SERVER_CONTROLS: {
      LDAPControl *const *controls =
	(LDAPControl *const *) invalue;

      //      ldap_controls_free( lo->ldo_sctrls );


      if( controls == NULL || *controls == NULL ) {
	lo->ldo_sctrls = NULL;
	return LDAP_OPT_SUCCESS;
      }

      //lo->ldo_sctrls = ldap_controls_dup( controls );

      if(lo->ldo_sctrls == NULL) {
	/* memory allocation error ? */
	break;
      }
    } return LDAP_OPT_SUCCESS;

    case LDAP_OPT_CLIENT_CONTROLS: {
      LDAPControl *const *controls =
	(LDAPControl *const *) invalue;

      //      ldap_controls_free( lo->ldo_cctrls );

      if( controls == NULL || *controls == NULL ) {
	lo->ldo_cctrls = NULL;
	return LDAP_OPT_SUCCESS;
      }

      //      lo->ldo_cctrls = ldap_controls_dup( controls );

      if(lo->ldo_cctrls == NULL) {
	/* memory allocation error ? */
	break;
      }
    } return LDAP_OPT_SUCCESS;

    case LDAP_OPT_TIMEOUT: {
      const struct timeval *tv =
	(const struct timeval *) invalue;

      if ( lo->ldo_tm_api != NULL ) {
	free(lo->ldo_tm_api);
	lo->ldo_tm_api = NULL;
      }

      memcpy(&lo->ldo_tm_api, tv,sizeof(struct timeval));
    } return LDAP_OPT_SUCCESS;

    case LDAP_OPT_NETWORK_TIMEOUT: {
      const struct timeval *tv =
	(const struct timeval *) invalue;

      if ( lo->ldo_tm_net != NULL ) {
	free( lo->ldo_tm_net );
	lo->ldo_tm_net = NULL;
      }

      memcpy( &lo->ldo_tm_net, tv, sizeof(struct timeval));
    } return LDAP_OPT_SUCCESS;
  }

  if(invalue == NULL) {
    /* no place to set from */
    return LDAP_OPT_ERROR;
  }

  /* options which cannot withstand invalue == NULL */

  switch(option) {
    case LDAP_OPT_API_INFO:
    case LDAP_OPT_DESC:
      /* READ ONLY */
      break;

    case LDAP_OPT_DEREF:
      lo->ldo_deref = * (const int *) invalue;
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_SIZELIMIT:
      lo->ldo_sizelimit = * (const int *) invalue;
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_TIMELIMIT:
      lo->ldo_timelimit = * (const int *) invalue;
      return LDAP_OPT_SUCCESS;

    case LDAP_OPT_PROTOCOL_VERSION: {
      int vers = * (const int *) invalue;
      if (vers < LDAP_VERSION_MIN || vers > LDAP_VERSION_MAX) {
	/* not supported */
	break;
      }
      lo->ldo_version = vers;
    } return LDAP_OPT_SUCCESS;


    case LDAP_OPT_HOST_NAME: {
      const char *hosts = (const char *) invalue;
      if(NULL==hosts)
	break;
      ldap_clear_hostlist(ld->hostlist); // Empty old vector.
      char *s;
      char *last=(char *)hosts;
      for(s=(char *)hosts; *s; s++) /* Copy the entries */
	if(' '==*s){
	  char *hostname=strndup(last,s-last);
	  char *first_colon;
	  int current_port=0;
	  if(NULL!=(first_colon=strchr(hostname,':'))){
	    char *tmphost=strndup(hostname,first_colon-hostname);
	    if(NULL==tmphost){
	      return LDAP_NO_MEMORY;
	    }
	    current_port=atoi(first_colon+1);
	    if((current_port<=0)||(current_port>65535))
	      current_port=LDAP_PORT;
	    ldap_memfree(hostname);
	    hostname=tmphost;
	  }
	  LDAP_Hostentry *host=new LDAP_Hostentry;
	  host->hostname=PString(hostname);
	  host->port=current_port;
	  ld->hostlist.push_back(host);
	  ldap_memfree(hostname);
	}
    }

    case LDAP_OPT_ERROR_NUMBER: {
      int err = * (const int *) invalue;

      if(ld == NULL) {
	/* need a struct ldap */
	break;
      }

      ld->ld_errno = err;
    } return LDAP_OPT_SUCCESS;

    case LDAP_OPT_ERROR_STRING: {
      const char *err = (const char *) invalue;

      if(ld == NULL) {
	/* need a struct ldap */
	break;
      }
      ld->ld_error = err;
    } return LDAP_OPT_SUCCESS;

    case LDAP_OPT_MATCHED_DN: {
      const char *err = (const char *) invalue;

      if(ld == NULL) {
	/* need a struct ldap */
	break;
      }
      ld->ld_matched = PString(err);
    } return LDAP_OPT_SUCCESS;

    case LDAP_OPT_API_FEATURE_INFO:
      /* read-only */
      break;

    default:
      /* bad param */
      break;
  }
  return LDAP_OPT_ERROR;
}
//
// End of options.cxx
//
