// -*- mode: c++ -*-
// Copyright (C) 2002 mediaWays GmbH Internet-Services, Verl, Germany
//
// PURPOSE OF THIS FILE: Implement the ldap_parse functions
//
// - Automatic Version Information via RCS:
//   $Id: ldapparse.cxx,v 1.1 2002/09/23 07:28:52 robertj Exp $
//   $Source: /home/cvsroot/openh323/src/ldapparse.cxx,v $

/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations under
 * the License.
 *
 * The Original Code is the OpenLDAP Comatible LDAP-API for OpenH323
 * (libldapapi).
 *
 * The Initial Developer of the Original Code is
 * mediaWays GmbH Internet-Services, Verl, Germany
 * Portions created by the Initial Developer are Copyright (C) 2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Nils Bokermann <Nils.Bokermann@mediaWays.net> (*)
 *                 Martin Frhlich <Martin.Froehlich@mediaWays.net> (*)
 *
 * (*) Contributors marked this way are mediaWays employees, all their work
 *     is subject to mediaWays copyright by their contracts.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * For copies of the GPL or LGPL, see either http://www.fsf.org/licenses/
 * or write to the Free Software Foundation, Inc., 59 Temple Place - Suite
 * 330, Boston, MA 02111-1307, USA.
 *
 * ***** END LICENSE BLOCK ***** */

#include <ptlib.h>
#include <ldapapi.h>
#include "ldap-int.h"


#ifndef lint
// mark object with version info in such a way that it is retrievable by
// the std. version/revision control tools like RCS/CVS ident cmd.. At
// least the strings cmd will extract this info.
volatile static const char vcid[] = "@(#) $Id: ldapparse.cxx,v 1.1 2002/09/23 07:28:52 robertj Exp $";
volatile static const char vcHid[] = LDAP_OPENH323_H;
#endif

struct ldaperror {
        int     e_code;
        char    *e_reason;
};

static const struct ldaperror ldap_errlist[] = {
	{LDAP_SUCCESS, 					"Success" },
	{LDAP_OPERATIONS_ERROR, 		"Operations error" },
	{LDAP_PROTOCOL_ERROR, 			"Protocol error" },
	{LDAP_TIMELIMIT_EXCEEDED,		"Time limit exceeded" },
	{LDAP_SIZELIMIT_EXCEEDED, 		"Size limit exceeded" },
	{LDAP_COMPARE_FALSE, 			"Compare false" },
	{LDAP_COMPARE_TRUE, 			"Compare true" },
	{LDAP_STRONG_AUTH_NOT_SUPPORTED, "Authentication method not supported" },
	{LDAP_STRONG_AUTH_REQUIRED, 	"Strong authentication required" },
	{LDAP_PARTIAL_RESULTS, 			"Partial results and referral received" },

	{LDAP_REFERRAL,					"Referral"},
	{LDAP_ADMINLIMIT_EXCEEDED,		"Administrative limit exceeded"},
	{LDAP_UNAVAILABLE_CRITICAL_EXTENSION,
									"Criticial extension is unavailable"},
	{LDAP_CONFIDENTIALITY_REQUIRED,	"Confidentiality required"},
	{LDAP_SASL_BIND_IN_PROGRESS,	"SASL bind in progress"},

	{LDAP_NO_SUCH_ATTRIBUTE, 		"No such attribute" },
	{LDAP_UNDEFINED_TYPE, 			"Undefined attribute type" },
	{LDAP_INAPPROPRIATE_MATCHING, 	"Inappropriate matching" },
	{LDAP_CONSTRAINT_VIOLATION, 	"Constraint violation" },
	{LDAP_TYPE_OR_VALUE_EXISTS, 	"Type or value exists" },
	{LDAP_INVALID_SYNTAX, 			"Invalid syntax" },

	{LDAP_NO_SUCH_OBJECT, 			"No such object" },
	{LDAP_ALIAS_PROBLEM, 			"Alias problem" },
	{LDAP_INVALID_DN_SYNTAX,		"Invalid DN syntax" },
	{LDAP_IS_LEAF, 					"Entry is a leaf" },
	{LDAP_ALIAS_DEREF_PROBLEM,	 	"Alias dereferencing problem" },

	{LDAP_INAPPROPRIATE_AUTH, 		"Inappropriate authentication" },
	{LDAP_INVALID_CREDENTIALS, 		"Invalid credentials" },
	{LDAP_INSUFFICIENT_ACCESS, 		"Insufficient access" },
	{LDAP_BUSY, 					"DSA is busy" },
	{LDAP_UNAVAILABLE, 				"DSA is unavailable" },
	{LDAP_UNWILLING_TO_PERFORM, 	"DSA is unwilling to perform" },
	{LDAP_LOOP_DETECT, 				"Loop detected" },

	{LDAP_NAMING_VIOLATION, 		"Naming violation" },
	{LDAP_OBJECT_CLASS_VIOLATION, 	"Object class violation" },
	{LDAP_NOT_ALLOWED_ON_NONLEAF, 	"Operation not allowed on nonleaf" },
	{LDAP_NOT_ALLOWED_ON_RDN,	 	"Operation not allowed on RDN" },
	{LDAP_ALREADY_EXISTS, 			"Already exists" },
	{LDAP_NO_OBJECT_CLASS_MODS, 	"Cannot modify object class" },
	{LDAP_RESULTS_TOO_LARGE,		"Results too large" },
	{LDAP_AFFECTS_MULTIPLE_DSAS,	"Operation affects multiple DSAs" },

	{LDAP_OTHER, 					"Unknown error" },

	/* API ResultCodes */
	{LDAP_SERVER_DOWN,				"Can't contact LDAP server" },
	{LDAP_LOCAL_ERROR,				"Local error" },
	{LDAP_ENCODING_ERROR,			"Encoding error" },
	{LDAP_DECODING_ERROR,			"Decoding error" },
	{LDAP_TIMEOUT,					"Timed out" },
	{LDAP_AUTH_UNKNOWN,				"Unknown authentication method" },
	{LDAP_FILTER_ERROR,				"Bad search filter" },
	{LDAP_USER_CANCELLED,			"User cancelled operation" },
	{LDAP_PARAM_ERROR,				"Bad parameter to an ldap routine" },
	{LDAP_NO_MEMORY,				"Out of memory" },

	{LDAP_CONNECT_ERROR,			"Connect error" },
	{LDAP_NOT_SUPPORTED,			"Not Supported" },
	{LDAP_CONTROL_NOT_FOUND,		"Control not found" },
	{LDAP_NO_RESULTS_RETURNED,		"No results returned" },
	{LDAP_MORE_RESULTS_TO_RETURN,	"More results to return" },
	{LDAP_CLIENT_LOOP,				"Client Loop" },
	{LDAP_REFERRAL_LIMIT_EXCEEDED,	"Referral Limit Exceeded" },

	{-1, NULL }
};

int ldap_parse_result (LDAP *ld, LDAPMessage *res, int *errcodep, char **matcheddnp,
		       char **errmsgp, char ***referralsp, LDAPControl ***serverctrls,
		       int freeit) {
  if(NULL==ld)
    return LDAP_UNAVAILABLE;
  LDAPMessage *result=res;
  while((NULL!=result) &&
	(&result->message->m_protocolOp.GetObject())->IsClass("LDAP_SearchResponse")){
    LDAP_SearchResponse & res=result->message->m_protocolOp;
#ifdef DEBUG
    cerr << "SearchResponse: " << endl;
    cerr << res << endl;
#endif
    if (res.GetTag()==LDAP_SearchResponse::e_resultCode)
      break;
    result=result->next;
  }

  if(NULL==result)
    return LDAP_UNDEFINED_TYPE;

  LDAP_LDAPResult ldapresult;
  int found=0;
  if((&result->message->m_protocolOp.GetObject())->IsClass("LDAP_SearchResponse")) {
    LDAP_SearchResponse &mod=result->message->m_protocolOp;
    if (mod.GetTag()==LDAP_SearchResponse::e_resultCode)
      ldapresult=static_cast<LDAP_LDAPResult&>(mod);
    else {
      return LDAP_UNDEFINED_TYPE;
    }
    found=1;
  } else {
    switch((&result->message->m_protocolOp)->GetTag()) {
      case LDAP_LDAPMessage_protocolOp::e_modifyResponse: {
	LDAP_ModifyResponse &mod=result->message->m_protocolOp;
	ldapresult=mod;
	found=1;
	break;
      }
      case LDAP_LDAPMessage_protocolOp::e_addResponse: {
	LDAP_AddResponse &mod=result->message->m_protocolOp;
	ldapresult=mod;
	found=1;
	break;
      }
      case LDAP_LDAPMessage_protocolOp::e_delResponse: {
	LDAP_DelResponse &mod=result->message->m_protocolOp;
	ldapresult=mod;
	found=1;
	break;
      }
      case LDAP_LDAPMessage_protocolOp::e_modifyRDNResponse: {
	LDAP_ModifyRDNResponse &mod=result->message->m_protocolOp;
	ldapresult=mod;
	found=1;
	break;
      }
      case LDAP_LDAPMessage_protocolOp::e_compareDNResponse: {
	LDAP_CompareResponse &mod=result->message->m_protocolOp;
	ldapresult=mod;
	found=1;
	break;
      }
      case LDAP_LDAPMessage_protocolOp::e_bindResponse: {
	LDAP_BindResponse &mod=result->message->m_protocolOp;
	ldapresult=mod;
	found=1;
      }
    }
  }
  if(!found) {
    return LDAP_UNDEFINED_TYPE;
  }

#ifdef DEBUG
  cerr << ldapresult << endl;
#endif

  ld->ld_errno=ldapresult.m_resultCode;
  ld->ld_matched=ldapresult.m_matchedDN.AsString();
  ld->ld_error=ldapresult.m_errorMessage.AsString();
  if(errcodep)
    *errcodep=ldapresult.m_resultCode;
  if(matcheddnp)
    *matcheddnp=strndup(ld->ld_matched,strlen(ld->ld_matched));
  if(errmsgp)
    *errmsgp=strndup(ld->ld_error,strlen(ld->ld_error));
  if(referralsp)
    *referralsp=NULL;
  if(serverctrls)
    *serverctrls=NULL;
  if(freeit)
    ldap_msgfree(res);
  return LDAP_SUCCESS;
}

int ldap_parse_sasl_bind_result (LDAP *ld, LDAPMessage *res, struct berval **servercredp,
				 int freeit) {
  if(NULL==ld)
    return LDAP_UNAVAILABLE;
  *servercredp=NULL;
  LDAPMessage *result=res;

  if(NULL==result)
    return LDAP_UNDEFINED_TYPE;
  if(freeit)
    ldap_msgfree(res);
  return LDAP_SUCCESS;
}

static const struct ldaperror * ldap_int_error(int err) {
  int i;

  for ( i = 0; ldap_errlist[i].e_code != -1; i++ ) {
    if ( err == ldap_errlist[i].e_code )
      return &ldap_errlist[i];
  }

  return NULL;
}

const char * ldap_err2string(int err) {
  const struct ldaperror *e;

  e = ldap_int_error( err );

  return ( e != NULL ) ? e->e_reason : "Unknown error";
}


/* deprecated */

int ldap_result2error (LDAP *ld, LDAPMessage *res, int freeit) {
  int error;
  int rv;
  if(LDAP_SUCCESS!=(rv=ldap_parse_result(ld, res, &error, NULL, NULL, NULL, NULL, freeit))){
    ld->ld_errno=rv;
    return rv;
  }
  return error;
}

/* deprecated */

void ldap_perror(LDAP *ld, LDAP_CONST char *str) {
  const char *s;
  const struct ldaperror *e;

  s = (str != NULL) ? str : "ldap_perror";

  if (ld == NULL) {
    perror(s);
    return;
  }

  e = ldap_int_error(ld->ld_errno);

  if (e != NULL) {
    fprintf(stderr, "%s: %s\n",
	     s, e->e_reason);
  } else {
    fprintf(stderr, "%s: unknown LDAP error number %d\n",
	     s, ld->ld_errno);
  }

  if (!ld->ld_matched.IsEmpty()) {
    fprintf( stderr, "\tmatched DN: \"%s\"\n",
	     (const char *)ld->ld_matched);
  }

  if (!ld->ld_error.IsEmpty()) {
    fprintf(stderr, "\tadditional info: %s\n",
	    (const char *)ld->ld_error);
  }

  fflush(stderr);
}

//
// End of parse.cxx
//
