// -*- mode: c++  -*-
// Copyright (C) 2002 mediaWays GmbH Internet-Services, Verl, Germany
//
// PURPOSE OF THIS FILE: Implement the ldap_add functions
//
// - Automatic Version Information via RCS:
//   $Id: ldapsearch.cxx,v 1.2 2002/10/11 10:28:07 rogerh Exp $
//   $Source: /home/cvsroot/openh323/src/ldapsearch.cxx,v $

/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations under
 * the License.
 *
 * The Original Code is the OpenLDAP Comatible LDAP-API for OpenH323
 * (libldapapi).
 *
 * The Initial Developer of the Original Code is
 * mediaWays GmbH Internet-Services, Verl, Germany
 * Portions created by the Initial Developer are Copyright (C) 2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Nils Bokermann <Nils.Bokermann@mediaWays.net> (*)
 *                 Martin Frhlich <Martin.Froehlich@mediaWays.net> (*)
 *
 * (*) Contributors marked this way are mediaWays employees, all their work
 *     is subject to mediaWays copyright by their contracts.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * For copies of the GPL or LGPL, see either http://www.fsf.org/licenses/
 * or write to the Free Software Foundation, Inc., 59 Temple Place - Suite
 * 330, Boston, MA 02111-1307, USA.
 *
 * ***** END LICENSE BLOCK ***** */

#include <ptlib.h>
#include <ldapapi.h>
#include "ldap-int.h"
#include "ldapfilter.h"

#include <iostream>

#ifndef lint
// mark object with version info in such a way that it is retrievable by
// the std. version/revision control tools like RCS/CVS ident cmd.. At
// least the strings cmd will extract this info.
volatile static const char vcid[] = "@(#) $Id: ldapsearch.cxx,v 1.2 2002/10/11 10:28:07 rogerh Exp $";
volatile static const char vcHid[] = LDAP_OPENH323_H;
#endif

int ldap_search_ext (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter,
		     char **attrs, int attrsonly, LDAPControl **serverctrls,
		     LDAPControl **clientctrls, struct timeval *timeout,
		     int sizelimit, int *msgidp){
  if(NULL==ld)
    return LDAP_UNAVAILABLE;

  struct timeval *time = timeout;
  if(NULL==time){
    time=new struct timeval;
    time->tv_sec=(0==ld->ld_options->ldo_timelimit) ? 0 : ld->ld_options->ldo_timelimit;
    time->tv_usec=0;
  }

  if(timeout &&(time->tv_sec==0)&&(time->tv_usec==0))
    return LDAP_PARAM_ERROR;
  if(timeout && (time->tv_sec==0))
    time->tv_sec=1;

  LDAP_LDAPMessage msg;

  *msgidp=++(ld->msg_id);
  msg.m_messageID=*msgidp;
  msg.m_protocolOp=LDAP_LDAPMessage_protocolOp(LDAP_LDAPMessage_protocolOp::e_searchRequest);

  LDAP_SearchRequest & search=msg.m_protocolOp;

  search.m_baseObject=base;
  LDAP_SearchRequest_scope & lscope=search.m_scope;
  switch (scope) {
    case LDAP_SCOPE_BASE:
      lscope=LDAP_SearchRequest_scope::e_baseObject;
      break;
    case LDAP_SCOPE_ONELEVEL:
      lscope=LDAP_SearchRequest_scope::e_singleLevel;
      break;
    case LDAP_SCOPE_SUBTREE:
    default:
      lscope=LDAP_SearchRequest_scope::e_wholeSubtree;
      break;
  }
  LDAP_SearchRequest_derefAliases & lderef=search.m_derefAliases;
  switch (ld->ld_options->ldo_deref) {
    case LDAP_DEREF_NEVER:
      lderef=LDAP_SearchRequest_derefAliases::e_neverDerefAliases;
      break;
    case LDAP_DEREF_SEARCHING:
      lderef=LDAP_SearchRequest_derefAliases::e_derefInSearching;
      break;
    case LDAP_DEREF_FINDING:
      lderef=LDAP_SearchRequest_derefAliases::e_derefFindingBaseObj;
      break;
    case LDAP_DEREF_ALWAYS:
    default:
      lderef=LDAP_SearchRequest_derefAliases::e_alwaysDerefAliases;
      break;
  }
  search.m_sizeLimit=PASN_Integer(((sizelimit<0) ? 0 : sizelimit));
  search.m_attrsOnly=PASN_Boolean(0!=attrsonly);
  search.m_timeLimit=PASN_Integer(time->tv_sec);

  if(filter==NULL) {
    search.m_filter=LDAP_SFilter("(objectclass=*)");
  }else{
    search.m_filter=LDAP_SFilter((char *)filter);
  }
  LDAP_ArrayOf_AttributeType &attr=search.m_attributes;
  if(attrs!=NULL) {
    int i=0;
    while(NULL!=attrs[i]) {
      attr.SetSize(i+1);
      attr[i]=attrs[i];
      i++;
    }
  }

  PBER_Stream encoding_stream;
  msg.Encode(encoding_stream);
#ifdef DEBUG
  cerr << search.m_filter << endl;
  cerr << msg << endl;
#endif
#ifdef BER_DEBUG
  cerr << encoding_stream << endl;
#endif

  if(encoding_stream.Write(*(ld->socket))) {
    return LDAP_SUCCESS;
  }
  ld->ld_errno=LDAP_UNAVAILABLE;
  return ld->ld_errno;
}

int ldap_search_ext_s (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter,
		       char **attrs, int attrsonly, LDAPControl **serverctrls,
		       LDAPControl **clientctrls, struct timeval *timeout,
		       int sizelimit, LDAPMessage **res) {
  int msgid;
  int rv;
  LDAPMessage *msg;
  rv=ldap_search_ext(ld, base, scope, filter, attrs, attrsonly, serverctrls, clientctrls,
		  timeout, sizelimit, &msgid);

  if(LDAP_SUCCESS!=rv)
    return rv;

  if(ldap_result(ld, msgid, 1, NULL, &msg ) == -1 )
    return (ld->ld_errno);

  res=&msg;
  return LDAP_SUCCESS;
}

int ldap_search (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter,
		 char **attrs, int attrsonly) {
  int msgid;
  int rv;

  rv=ldap_search_ext(ld, base, scope, filter, attrs, attrsonly, NULL, NULL, NULL,
		     LDAP_NO_LIMIT, &msgid);
  if(LDAP_SUCCESS!=rv){
    ld->ld_errno=rv;
    return (-1);
  }
  return msgid;
}

int ldap_search_s (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter,
		   char **attrs, int attrsonly, LDAPMessage **res){
  LDAPMessage *msg;
  int msgid;
  msgid=ldap_search(ld, base, scope, filter, attrs, attrsonly);
  if(-1==msgid)
    return ld->ld_errno;

  if(ldap_result(ld, msgid, 1, NULL, &msg ) == -1)
    return ld->ld_errno;
  *res=msg;
  return LDAP_SUCCESS;
}

int ldap_search_st (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter,
		    char **attrs, int attrsonly, struct timeval *timeout,
		    LDAPMessage **res){
  LDAPMessage *msg;
  int msgid;
  msgid=ldap_search(ld, base, scope, filter, attrs, attrsonly);
  if(-1==msgid)
    return ld->ld_errno;

  if(ldap_result(ld, msgid, 1, timeout, &msg ) == -1)
    return ld->ld_errno;
  *res=msg;
  return LDAP_SUCCESS;
}

//
// End of add.cxx
//
