/*
 * speex.cxx
 *
 * Speex codec handler
 *
 * Open H323 Library
 *
 * Copyright (c) 2002 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: speexcodec.cxx,v $
 * Revision 1.11  2002/10/09 10:55:21  rogerh
 * Update the bit rates to match what the codec now does
 *
 * Revision 1.10  2002/09/02 21:58:40  rogerh
 * Update for Speex 0.8.0
 *
 * Revision 1.9  2002/08/21 06:49:13  rogerh
 * Fix the RTP Payload size too small problem with Speex 0.7.0.
 *
 * Revision 1.8  2002/08/15 18:34:51  rogerh
 * Fix some more bugs
 *
 * Revision 1.7  2002/08/14 19:06:53  rogerh
 * Fix some bugs when using the speex library
 *
 * Revision 1.6  2002/08/14 04:35:33  craigs
 * CHanged Speex names to remove spaces
 *
 * Revision 1.5  2002/08/14 04:30:14  craigs
 * Added bit rates to Speex codecs
 *
 * Revision 1.4  2002/08/14 04:27:26  craigs
 * Fixed name of Speex codecs
 *
 * Revision 1.3  2002/08/14 04:24:43  craigs
 * Fixed ifdef problem
 *
 * Revision 1.2  2002/08/13 14:25:25  craigs
 * Added trailing newlines to avoid Linux warnings
 *
 * Revision 1.1  2002/08/13 14:14:59  craigs
 * Initial version
 *
 */

 /*
   This code requires the Speex codec code available from http://speex.sourceforge.net 
  */

#include <ptlib.h>

#ifdef SPEEX_CODEC

#ifdef __GNUC__
#pragma implementation "speexcodec.h"
#endif

#include "speexcodec.h"

#include "h323caps.h"
#include "h245.h"
#include "rtp.h"

#define new PNEW

#define SPEEX_COUNTRY_CODE       9  // Country code for Australia
#define SPEEX_T35EXTENSION       0
#define SPEEX_MANUFACTURER_CODE  61 // Allocated by Australian Communications Authority, Oct 2000

#define SAMPLES_PER_FRAME        160

#define SPEEX_BASE_NAME "Speex"

#define SPEEX_NARROW2_H323_NAME        SPEEX_BASE_NAME "Narrow-5.95k{sw}"
#define SPEEX_NARROW3_H323_NAME        SPEEX_BASE_NAME "Narrow-8k{sw}"
#define SPEEX_NARROW4_H323_NAME        SPEEX_BASE_NAME "Narrow-11k{sw}"
#define SPEEX_NARROW5_H323_NAME        SPEEX_BASE_NAME "Narrow-15k{sw}"
#define SPEEX_NARROW6_H323_NAME        SPEEX_BASE_NAME "Narrow-18.2k{sw}"

H323_REGISTER_CAPABILITY(SpeexNarrow2AudioCapability, SPEEX_NARROW2_H323_NAME);
H323_REGISTER_CAPABILITY(SpeexNarrow3AudioCapability, SPEEX_NARROW3_H323_NAME);
H323_REGISTER_CAPABILITY(SpeexNarrow4AudioCapability, SPEEX_NARROW4_H323_NAME);
H323_REGISTER_CAPABILITY(SpeexNarrow5AudioCapability, SPEEX_NARROW5_H323_NAME);
H323_REGISTER_CAPABILITY(SpeexNarrow6AudioCapability, SPEEX_NARROW6_H323_NAME);


static int Speex_Bits_Per_Second(int mode) {
    void *tmp_coder_state;
    int bitrate;
    tmp_coder_state = speex_encoder_init(&speex_nb_mode);
    speex_encoder_ctl(tmp_coder_state, SPEEX_SET_QUALITY, &mode);
    speex_encoder_ctl(tmp_coder_state, SPEEX_GET_BITRATE, &bitrate);
    speex_encoder_destroy(tmp_coder_state); 
    return bitrate;
}

static int Speex_Bytes_Per_Frame(int mode) {
    int bits_per_frame = Speex_Bits_Per_Second(mode) / 50; // (20ms frame size)
    return ((bits_per_frame+7)/8); // round up
}

static OpalMediaFormat const SpeexNarrow2_MediaFormat(SPEEX_NARROW2_H323_NAME,
                                                  OpalMediaFormat::DefaultAudioSessionID,
                                                  RTP_DataFrame::DynamicBase,
                                                  TRUE,  // Needs jitter
                                                  Speex_Bits_Per_Second(2),
                                                  Speex_Bytes_Per_Frame(2),
                                                  SAMPLES_PER_FRAME, // 20 milliseconds
                                                  OpalMediaFormat::AudioTimeUnits);

static OpalMediaFormat const SpeexNarrow3_MediaFormat(SPEEX_NARROW3_H323_NAME,
                                                  OpalMediaFormat::DefaultAudioSessionID,
                                                  RTP_DataFrame::DynamicBase,
                                                  TRUE,  // Needs jitter
                                                  Speex_Bits_Per_Second(3),
                                                  Speex_Bytes_Per_Frame(3),
                                                  SAMPLES_PER_FRAME, // 20 milliseconds
                                                  OpalMediaFormat::AudioTimeUnits);

static OpalMediaFormat const SpeexNarrow4_MediaFormat(SPEEX_NARROW4_H323_NAME,
                                                  OpalMediaFormat::DefaultAudioSessionID,
                                                  RTP_DataFrame::DynamicBase,
                                                  TRUE,  // Needs jitter
                                                  Speex_Bits_Per_Second(4),
                                                  Speex_Bytes_Per_Frame(4),
                                                  SAMPLES_PER_FRAME, // 20 milliseconds
                                                  OpalMediaFormat::AudioTimeUnits);

static OpalMediaFormat const SpeexNarrow5_MediaFormat(SPEEX_NARROW5_H323_NAME,
                                                  OpalMediaFormat::DefaultAudioSessionID,
                                                  RTP_DataFrame::DynamicBase,
                                                  TRUE,  // Needs jitter
                                                  Speex_Bits_Per_Second(5),
                                                  Speex_Bytes_Per_Frame(5),
                                                  SAMPLES_PER_FRAME, // 20 milliseconds
                                                  OpalMediaFormat::AudioTimeUnits);

static OpalMediaFormat const SpeexNarrow6_MediaFormat(SPEEX_NARROW6_H323_NAME,
                                                  OpalMediaFormat::DefaultAudioSessionID,
                                                  RTP_DataFrame::DynamicBase,
                                                  TRUE,  // Needs jitter
                                                  Speex_Bits_Per_Second(6),
                                                  Speex_Bytes_Per_Frame(6),
                                                  SAMPLES_PER_FRAME, // 20 milliseconds
                                                  OpalMediaFormat::AudioTimeUnits);

/////////////////////////////////////////////////////////////////////////

SpeexNonStandardAudioCapability::SpeexNonStandardAudioCapability(const char * name)
  : H323NonStandardAudioCapability(1, 1,
                                   SPEEX_COUNTRY_CODE,
                                   SPEEX_T35EXTENSION,
                                   SPEEX_MANUFACTURER_CODE,
                                   (const BYTE *)name, strlen(name), 0, strlen(name))
{
}

H323Codec * SpeexNonStandardAudioCapability::CreateCodec(int mode, H323Codec::Direction direction) const
{
  return new SpeexCodec(mode, direction);
}
                                                                         
PString SpeexNarrow2AudioCapability::GetFormatName() const
{
  return SPEEX_NARROW2_H323_NAME;
}

PString SpeexNarrow3AudioCapability::GetFormatName() const
{
  return SPEEX_NARROW3_H323_NAME;
}

PString SpeexNarrow4AudioCapability::GetFormatName() const
{
  return SPEEX_NARROW4_H323_NAME;
}

PString SpeexNarrow5AudioCapability::GetFormatName() const
{
  return SPEEX_NARROW5_H323_NAME;
}

PString SpeexNarrow6AudioCapability::GetFormatName() const
{
  return SPEEX_NARROW6_H323_NAME;
}

static char * Speex_Codec_Name(int mode) {
  switch (mode) {
    case 2 : return SPEEX_NARROW2_H323_NAME; break;
    case 3 : return SPEEX_NARROW3_H323_NAME; break;
    case 4 : return SPEEX_NARROW4_H323_NAME; break;
    case 5 : return SPEEX_NARROW5_H323_NAME; break;
    case 6 : return SPEEX_NARROW6_H323_NAME; break;
  }
  return NULL;
}
/////////////////////////////////////////////////////////////////////////////

const float MaxSampleValue   = 32767.0;
const float MinSampleValue   = -32767.0;

SpeexCodec::SpeexCodec(int mode, Direction dir)
  : H323FramedAudioCodec(Speex_Codec_Name(mode), dir)
{
  PTRACE(3, "Codec\tSpeex mode " << mode << " " << (dir == Encoder ? "en" : "de")
         << "coder created");

  speex_bits_init(&bits);

  if (direction == Encoder) {
    coder_state = speex_encoder_init(&speex_nb_mode);
    speex_encoder_ctl(coder_state, SPEEX_GET_FRAME_SIZE, &encoder_frame_size);
    speex_encoder_ctl(coder_state, SPEEX_SET_QUALITY,    &mode);
  } else {
    coder_state = speex_decoder_init(&speex_nb_mode);
  }
}

SpeexCodec::~SpeexCodec()
{
  speex_bits_destroy(&bits); 
  if (direction == Encoder)
    speex_encoder_destroy(coder_state); 
  else
    speex_decoder_destroy(coder_state); 
}


BOOL SpeexCodec::EncodeFrame(BYTE * buffer, unsigned & length)
{
  // convert PCM to float
  float floatData[SAMPLES_PER_FRAME];
  PINDEX i;
  for (i = 0; i < SAMPLES_PER_FRAME; i++)
    floatData[i] = sampleBuffer[i];

  // encode PCM data in sampleBuffer to buffer
  speex_bits_reset(&bits); 
  speex_encode(coder_state, floatData, &bits); 

  length = speex_bits_write(&bits, (char *)buffer, encoder_frame_size); 

  return TRUE;
}


BOOL SpeexCodec::DecodeFrame(const BYTE * buffer, unsigned length, unsigned &)
{
  float floatData[SAMPLES_PER_FRAME];

  // decode Speex data to floats
  speex_bits_read_from(&bits, (char *)buffer, length); 
  speex_decode(coder_state, &bits, floatData); 

  // convert float to PCM
  PINDEX i;
  for (i = 0; i < SAMPLES_PER_FRAME; i++) {
    float sample = floatData[i];
    if (sample < MinSampleValue)
      sample = MinSampleValue;
    else if (sample > MaxSampleValue)
      sample = MaxSampleValue;
    sampleBuffer[i] = (short)sample;
  }

  return TRUE;
}

/////////////////////////////////////////////////////////////////////////

#endif

