/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>

#include <string.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/gnksa.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>
#include <pan/base/status-item.h>

#include <pan/article-actions.h>
#include <pan/articlelist.h>
#include <pan/message-window.h>
#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/save-ui.h>
#include <pan/task.h>
#include <pan/task-bodies.h>
#include <pan/task-func-ptr.h>
#include <pan/task-save.h>
#include <pan/util.h>


/***
****  Finding a match in pan.sent
***/

typedef struct
{
	GMimeMessage * message;
	char * body;
	GMimeMessage * match;
}
FindMatchingStruct;

static void
find_matching_sent_foreach (GMimeMessage * compare, gpointer user_data)
{
	char * body;
	gboolean is_html;
	FindMatchingStruct * data = (FindMatchingStruct*) user_data;

	/* do we already have a match? */
	if (data->match != NULL)
		return;

	/* comparison #1: subject must match */
	if (pan_strcmp (g_mime_message_get_subject(compare), g_mime_message_get_subject(data->message)))
		return;

	/* comparison #2: author address must match */
	if (pan_strcmp (g_mime_message_get_sender(compare), g_mime_message_get_sender(data->message)))
		return;

	/* comparison #3: body must match, if we have one */
	body = g_mime_message_get_body (compare, TRUE, &is_html);
	if (body!=NULL && data->body!=NULL) {
		g_strstrip (body);
		if (!strcmp (body, data->body)) {
			g_object_ref (compare);
			data->match = compare;
			g_free (body);
		}
	}
}

/**
 * Find a matching article without comparing by message-id.
 * This is because many news servers alter the message-id of
 * an article being posted, so we can't rely on searching by
 * the message-id that Pan generated.
 */
static GMimeMessage*
find_matching_sent (GMimeMessage* message)
{
	gboolean is_html;
	FindMatchingStruct data;
	debug_enter ("find_matching_sent_article");

	/* sanity clause */
	g_return_val_if_fail (GMIME_IS_MESSAGE(message), NULL);

	/* find the match */
	data.message = message;
	data.body = g_strstrip (g_mime_message_get_body (data.message, TRUE, &is_html));
	data.match = NULL;
	acache_path_foreach (PAN_SENT, find_matching_sent_foreach, &data);
	g_free (data.body);
	return data.match;
}

/***
****
***/

static int
article_cancel_run (TaskFuncPtr* task, gpointer user_data)
{
	GMimeMessage * message = GMIME_MESSAGE (user_data);
	const char * message_id = g_mime_message_get_message_id (message);
        int status = nntp_cancel (STATUS_ITEM(task), message_id, TASK(task)->sock);

	if (status == TASK_SUCCESS)
	{
		const char * subject = g_mime_message_get_subject (message);
		const char * newsgroups = g_mime_message_get_header (message, HEADER_NEWSGROUPS);
		char ** groups = g_strsplit (newsgroups, ",", 0);
		guint i;

		for (i=0; groups[i]; ++i)
		{
			Group * group = server_get_named_group (TASK(task)->server, groups[i]);
			group_ref_articles (group, STATUS_ITEM(task));
			{
				Article * article = ARTICLE(group_get_article_by_message_id (group, message_id));
				if (article)
					group_remove_article (group, article);
			}
			group_unref_articles (group, STATUS_ITEM(task));
		}

		g_strfreev (groups);

		log_add_va (LOG_INFO|LOG_URGENT, _("Article %s (%s) canceled"),
			subject, message_id);
	}

	return 0;
}
static gchar*
article_cancel_describe (const StatusItem * status)
{
	return g_strdup (_("Canceling article"));
}
static void
article_cancel_dtor (gpointer message)
{
	debug_enter ("article_cancel_dtor");

	g_object_unref (message);

	debug_exit ("article_cancel_dtor");
}
void
article_cancel (Server * server, GMimeMessage * message)
{
	GMimeMessage * ours;
	
	ours = find_matching_sent (message);
	if (ours == NULL)
	{
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to cancel article: Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		queue_add (TASK(task_func_ptr_new (server,
		                article_cancel_describe,
		                article_cancel_run,
		                article_cancel_dtor, 
		                ours, FALSE)));
	}
}

/****
*****
****/

void
article_supersede (GMimeMessage * message)
{
	GMimeMessage * ours;
	debug_enter ("article_supersede");

	/* sanity clause */
	g_return_if_fail (GMIME_IS_MESSAGE (message));

	/* find the original copy so that the headers will match. */
	ours = find_matching_sent (message);
	if (ours == NULL)
	{
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to supersede article: Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		const char * old_message_id;
		const char * domain;
		char * new_message_id;

		/* get the new message-id */
		old_message_id = g_mime_message_get_message_id (ours);
		domain = strchr (old_message_id, '@');
		if (domain)
			++domain;
		new_message_id = gnksa_generate_message_id (domain);

		/* update the headers for superseding */
		g_mime_message_set_header (ours, HEADER_SUPERSEDES, old_message_id);
		g_mime_message_set_header (ours, HEADER_MESSAGE_ID, new_message_id);

		/* edit the message */
		message_edit_window (ours);

		/* cleanup */
		g_free (new_message_id);
	}

	debug_exit ("article_supersede");
}

/****
*****
****/

static void
article_copy_articles_to_folder_cb (gpointer obj, gpointer arg, gpointer data)
{
	int status;
	Task * task;
	Group * folder;
	debug_enter ("article_copy_articles_to_folder_cb");

	/* sanity clause */
	g_return_if_fail (obj!=NULL);
	g_return_if_fail (data!=NULL);
	folder = (Group*) data;
	g_return_if_fail (group_is_valid(folder));
	g_return_if_fail (group_is_folder(folder));

	/* pump argument list */
	status = GPOINTER_TO_INT(arg);
	task = TASK(obj);

	/* copy them over if the task succeeded ... */
	if (status == TASK_SUCCESS)
	{
		guint i;
		for (i=0; i<task->identifiers->len; ++i)
		{
			MessageIdentifier * mid = MESSAGE_IDENTIFIER (g_ptr_array_index (task->identifiers, i));
			GMimeMessage * message = acache_get_message (&mid, 1);
			folder_add_message (folder, message);
			g_object_unref (message);
		}

		/* clean up */
	}

	debug_exit ("articlelist_article_to_folder_cb");
}

gboolean
article_copy_articles_to_folder (Group * folder, const Article ** articles, int qty)
{
	Task * task;
	debug_enter ("article_copy_articles_to_folder");

	/* sanity clause */
	g_return_val_if_fail (folder!=NULL, FALSE);
	g_return_val_if_fail (group_is_folder(folder), FALSE);
	g_return_val_if_fail (articles!=NULL, FALSE);
	g_return_val_if_fail (qty>0, FALSE);
	g_return_val_if_fail (articles_are_valid(articles, qty), FALSE);

	/* make a task to download the bodies in case we don't already have them */
	task = TASK(task_bodies_new_from_articles (articles, qty));
	if (task != NULL) {
		pan_callback_add (task->task_ran_callback, article_copy_articles_to_folder_cb, folder);
		queue_add (task);
	}

	debug_exit ("article_copy_articles_to_folder");
	return TRUE;
}

/***
****
****
****   SAVING ARTICLES / OPENING ATTACHMENTS
****
****
***/

static void
save_attachments_impl (Article ** articles, int len)
{
	int i;
	GSList * tasks = NULL;
	debug_enter ("save_attachments_impl");

	g_return_if_fail (articles_are_valid ((const Article**)articles, len));

	for (i=0; i<len; ++i)
	{
		GSList * l;
		Article * a = articles[i];
		PanObject * task;
		GPtrArray * tmp = g_ptr_array_sized_new (64);

		/* build an array of this article and all its children. */
		g_ptr_array_add (tmp, a);
		for (l=a->threads; l!=NULL; l=l->next)
			g_ptr_array_add (tmp, l->data);

		/* queue for decode */
		task_save_weed_duplicates (tmp);
		task = task_save_new_from_articles (article_get_subject(a), (const Article**)tmp->pdata, tmp->len);
		task_save_set_attachments (TASK_SAVE(task), NULL, NULL);
		tasks = g_slist_prepend (tasks, task);

		/* cleanup */
		g_ptr_array_free (tmp, TRUE);
	}

	if (tasks != NULL)
	{
		tasks = g_slist_reverse (tasks);
		queue_insert_tasks (tasks, -1);
		tasks = NULL;
	}

	/* cleanup this loop */
	debug_exit ("save_attachments_impl");
}

void
article_action_selected_save_attachments (void)
{
	GPtrArray * sel;
	debug_enter ("article_action_selected_save_attachments");

	/* decode each selected article */
	sel = articlelist_get_selected_articles_nolock ();
	save_attachments_impl ((Article**)sel->pdata, sel->len);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("article_action_selected_save_attachments");
}

void
article_action_selected_save_as (void)
{
	guint i;
	GPtrArray * sel;
	Group * group = articlelist_get_group ();
	GSList * gslists_of_articles = NULL;
	debug_enter ("articlelist_selected_decode_as");

	g_return_if_fail (group != NULL);

	/* loop through all the selected articles */
	sel = articlelist_get_selected_articles_nolock ();
	for (i=0; i!=sel->len; ++i)
	{
		GSList * l = NULL;
		Article * article = ARTICLE(g_ptr_array_index(sel,i));

		/* build an array of this node and all its children. */
		if (article->parts > 1)
			l = g_slist_copy (article->threads);
		l = g_slist_prepend (l, article);

		/* add this to the gslist of articles */
		gslists_of_articles = g_slist_prepend (gslists_of_articles, l);
	}

	/* save_attach_as owns gslists_of_articles now */
	gslists_of_articles = g_slist_reverse (gslists_of_articles);
	save_attachment_as (group, gslists_of_articles);

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_decode_as");
}

/***
****
****
****   DELETING ARTICLES
****
****
***/

static void
remove_duplicate_articles (GPtrArray * articles)
{
	int i;
	GHashTable * hash;

	/* sanity claus */
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len>0);
	if (articles->len==1) return;

	/* remove duplicates */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	for (i=0; i<articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		g_hash_table_insert (hash, (gpointer)article_get_message_id(a), a);
	}
	pan_hash_to_ptr_array  (hash, articles);

	/* cleanup */
	g_hash_table_destroy (hash);
}


/**
 * Pairs of Group* and GArrays of gulong article numbers to remove from
 * the group.  We pass these arrays on to group_remove_crossposts(),
 * then free the array because we're done with it.
 * @see articlelist_delete_articles
 */
static void
articlelist_remove_xrefs_ghfunc (gpointer key,
                                 gpointer value,
                                 gpointer user_data)
{
	Group * g = GROUP(key);
	GArray * numbers = (GArray*) value;
	group_remove_crossposts (g, (gulong*)numbers->data, numbers->len);
	g_array_free (numbers, TRUE);
}

/**
 * data is a GHashTable of (key=group,val=GArray) where we fill those
 * GArrays with the article numbers of this crosspost.
 */
static void
articlelist_remove_xrefs_xreffunc (Server   * server,
                                   Group    * group,
                                   gulong     number,
                                   gpointer   data)
{
	GHashTable * hash = (GHashTable*) data;

	if (group_is_subscribed (group))
	{
		GArray * a = g_hash_table_lookup (hash, group);
		if (a == NULL)
			g_hash_table_insert (hash,
			                     group,
			                     a = g_array_new (FALSE, FALSE, sizeof(gulong)));
		g_array_append_val (a, number);
	}
}

/**
 * @see remove_duplicate_articles
 * @see articlelist_remove_xrefs_xreffunc
 * @see articlelist_remove_xrefs_ghfunc
 */
static void
delete_articles (GPtrArray * articles)
{
	debug_enter ("article_action_delete_articles");

	g_return_if_fail (articles!=NULL);

	if (articles->len)
	{
		Group * group = ARTICLE(g_ptr_array_index(articles,0))->group;

		/* uniqueness check */
		remove_duplicate_articles (articles);

		/* remove these articles from the cache */
		if (1) {
			guint i;
			GPtrArray * ids = g_ptr_array_new ();
			for (i=0; i<articles->len; ++i) {
				Article * a = ARTICLE(g_ptr_array_index(articles,i));
				g_ptr_array_add (ids, (gpointer)article_get_message_id(a));
			}
			acache_expire_messages ((const gchar **)ids->pdata, ids->len);
			g_ptr_array_free (ids, TRUE);
		}

		/* if these articles are posted elsewhere, delete those crossposts too */
		if (1) {
			guint i;
			GHashTable * hash = g_hash_table_new (g_direct_hash, g_direct_equal);
			for (i=0; i<articles->len; ++i) {
				Article * a = ARTICLE(g_ptr_array_index(articles,i));
				article_xref_foreach (a, articlelist_remove_xrefs_xreffunc, hash, SERVER_GROUPS_SUBSCRIBED, TRUE);
			}
			g_hash_table_foreach (hash, articlelist_remove_xrefs_ghfunc, NULL);
			g_hash_table_destroy (hash);
		}

		/* delete the articles in this group */
		group_remove_articles (group, articles);

		/* log */
		log_add_va (LOG_INFO, _("Deleted %u articles from \"%s\""),
		            articles->len,
		            group_get_name(group));
	}

	debug_exit ("article_action_delete_articles");
}

void
article_action_delete_selected_articles (void)
{
	GPtrArray * articles;
	int i, len;
	debug_enter ("article_action_delete_selected_articles");

	/* if it's a multipart, delete the children too */
       	articles = articlelist_get_selected_articles_nolock ();
	for (i=0, len=articles->len; i!=len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		if (a->part==1 && a->parts>1) {
			GSList * l;
			for (l=a->threads; l!=NULL; l=l->next)
				if (ARTICLE(l->data)->part > 1)
					g_ptr_array_add (articles, l->data);
		}
	}

	/* delete the articles */
	delete_articles (articles);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("article_action_delete_selected_articles");
}

void
article_action_edit_selected (void)
{
        Article * a = NULL;
	MessageIdentifier * mid = NULL;
	GMimeMessage * message = NULL;
	debug_enter ("article_action_edit_selected");

	a = articlelist_get_selected_article_nolock ();
	if (a != NULL)
		mid = message_identifier_new_from_article (a);
	if (mid != NULL)
		message = acache_get_message (&mid, 1);
	if (message != NULL)
		message_edit_window (message);

	g_object_unref (mid);
	debug_exit ("article_action_edit_selected");
}
