/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <pan/base/acache.h>
#include <pan/base/argset.h>
#include <pan/base/article-thread.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/file-headers.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/message-identifier.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>
#include <pan/base/status-item.h>

#include <pan/article-actions.h>
#include <pan/article-filter-ui.h>
#include <pan/articlelist.h>
#include <pan/article-toolbar.h>
#include <pan/flagset.h>
#include <pan/grouplist.h>
#include <pan/group-action.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-bodies.h>
#include <pan/task-save.h>
#include <pan/task-headers.h>
#include <pan/thread-watch.h>
#include <pan/text.h>
#include <pan/util.h>

#include <pan/xpm/pan-pixbufs.h>

#include <pan/filters/filter-top.h>

/***
****
****  Abandon hope, all ye who enter here
****
***/

/**********************************************************
 * Private Variables
 **********************************************************/

typedef enum
{
	REFRESH_SORT            = (1<<0),
	REFRESH_FILTER          = (1<<2),
	REFRESH_RELOAD          = (1<<3)
}
RefreshActions;

#define COLUMNS 6

/* state */
static GPtrArray * my_articles                   = NULL;
static Group * my_group                          = NULL;
static GHashTable * messageid_to_node            = NULL;
static int sort_type                             = ARTICLE_SORT_SUBJECT;
static GStaticMutex article_ctree_lock           = G_STATIC_MUTEX_INIT;
static GStaticMutex hash_mutex                   = G_STATIC_MUTEX_INIT;
static GStaticMutex switch_mutex                 = G_STATIC_MUTEX_INIT;

typedef struct
{
	const guint8 * pixbuf_txt;
	GdkPixbuf * pixbuf;
	GdkPixmap * pixmap;
	GdkBitmap * mask;
}
Icon;

enum
{
	ICON_READ,
	ICON_UNREAD,
	ICON_NEW,
	ICON_COMPLETE,
	ICON_INCOMPLETE,
	ICON_CACHED,
	ICON_BINARY_DECODED,
	ICON_QUEUED,
	ICON_ERROR,
	ICON_FLAGGED_FOR_DL,
	ICON_NEWSGROUP,
	ICON_QTY
};

Icon _icons[ICON_QTY] = {
	{ icon_article_read, NULL, NULL, NULL },
	{ icon_article_unread, NULL, NULL, NULL },
	{ icon_article_new, NULL, NULL, NULL },
	{ icon_binary_complete, NULL, NULL, NULL },
	{ icon_binary_incomplete, NULL, NULL, NULL },
	{ icon_disk, NULL, NULL, NULL },
	{ icon_binary, NULL, NULL, NULL },
	{ icon_bluecheck, NULL, NULL, NULL },
	{ icon_x, NULL, NULL, NULL },
	{ icon_blue_flag, NULL, NULL, NULL },
	{ icon_newsgroup, NULL, NULL, NULL }
};


PanCallback * articlelist_thread_changed         = NULL;
PanCallback * articlelist_selection_changed      = NULL;

static GString * cur_message_id                  = NULL;
static GString * prev_message_id                 = NULL;

static GtkStyle * normal_style[2]                = { NULL, NULL };
static GtkStyle * read_style[2]                  = { NULL, NULL };
static GtkStyle * watched_style[2]               = { NULL, NULL };
static GtkStyle * killfile_style[2]              = { NULL, NULL };

GdkColor killfile_color;
GdkColor watched_color;

static GtkWidget * threads_popup_menu = NULL;
static GtkItemFactory * threads_popup_factory = NULL;
static GtkItemFactoryEntry threads_popup_entries[];
static int threads_popup_entries_qty;

static gboolean _articlelist_repopulating = FALSE;

/**********************************************************
 * Private Functions
 **********************************************************/

static void
refresh_impl (RefreshActions actions);

static void
apply_filter_tests (Filter       * filter,
                    FilterShow     show,
                    GPtrArray    * headers);

static int
fire_group_changed (gpointer group);

static void
articlelist_menu_popup_nolock (GdkEventButton* bevent);
static void
articlelist_unread_inc_nolock (GtkCTreeNode * child, int inc);

static void
articlelist_repopulate_nolock (const Group        * group,
                               const Article     ** article_buf,
                       	       int                  article_qty,
                               GPtrArray          * sel_articles);

static GtkCTreeNode* articlelist_get_node_from_message_id (
	const char* message_id);

static void articlelist_update_node_impl (const Article* article);

static void articlelist_click_column (GtkCList*, int col_clicked);

static void article_ctree_destroy_cb (void);

static void articlelist_update_node_fast_nolock (GtkCTreeNode*, const Article*);

/***
****
****
****  SELECTIONS
****
****
***/


void
articlelist_select_all_nolock (void)
{
	gtk_ctree_select_recursive (GTK_CTREE(Pan.article_ctree), NULL);
}

void
articlelist_deselect_all_nolock (void)
{
	gtk_ctree_unselect_recursive (GTK_CTREE(Pan.article_ctree), NULL);
}

static Article*
articlelist_get_article_from_node (GtkCTreeNode* node)
{
	Article* article = NULL;

	if (node != NULL)
		article = ARTICLE(
			  gtk_ctree_node_get_row_data (
				  GTK_CTREE(Pan.article_ctree), node));

	return article;
}

static GPtrArray*
articlelist_get_selected_nodes_nolock (void)
{
	GPtrArray * a;
	const GList * l;
	debug_enter ("articlelist_get_selected_nodes_nolock");

       	a = g_ptr_array_sized_new (64);
	for (l=GTK_CLIST(Pan.article_ctree)->selection; l!=NULL; l=l->next)
		g_ptr_array_add (a, GTK_CTREE_NODE(l->data));

	debug_exit ("articlelist_get_selected_nodes_nolock");
	return a;
}

GtkCTreeNode*
articlelist_get_selected_node (void)
{
	GtkCTreeNode* node = NULL;

	const GList* list =  GTK_CLIST(Pan.article_ctree)->selection;
	if (list != NULL)
		node = GTK_CTREE_NODE(list->data);

	return node;
}


GPtrArray*
articlelist_get_selected_articles_nolock (void)
{
	GPtrArray * retval;
	GList * list;
	debug_enter ("articlelist_get_selected_articles_nolock");

	retval = g_ptr_array_sized_new (64);
	list = GTK_CLIST(Pan.article_ctree)->selection;
	for (; list!=NULL; list=list->next)
	{
		GtkCTreeNode * n = GTK_CTREE_NODE(list->data);
		Article * a = articlelist_get_article_from_node(n);
		if (a != NULL)
			g_ptr_array_add (retval, a);
	}

	debug_exit ("articlelist_get_selected_articles_nolock");
	return retval;
}

guint
articlelist_get_selected_count_nolock (void)
{
	guint retval;
	debug_enter ("articlelist_get_selected_count_nolock");

	retval = g_list_length (GTK_CLIST(Pan.article_ctree)->selection);

	debug_exit ("articlelist_get_selected_count_nolock");
	return retval;

}


Article*
articlelist_get_selected_article_nolock (void)
{
	return articlelist_get_article_from_node (
		articlelist_get_selected_node());
}

void
articlelist_set_selected_nodes_nolock (GtkCTreeNode **nodes, int node_qty)
{
	int i;
	gfloat hadj;
	GtkCTree * tree;
	GtkCList * list;
	GtkCTreeNode * first = NULL;
	debug_enter ("articlelist_set_selected_nodes_nolock");

	if (nodes==NULL || node_qty<=0)
		return;

	tree = GTK_CTREE(Pan.article_ctree);
	list = GTK_CLIST(Pan.article_ctree);
	gtk_clist_freeze (list);
	gtk_ctree_unselect_recursive (tree, gtk_ctree_node_nth(tree, 0));

	for (i=0; i<node_qty; ++i)
	{
		GtkCTreeNode * n = NULL;
		GtkCTreeNode * node = nodes[i];

		if (node != NULL)
		{
			/* remember the first good node */
			if (first == NULL)
				first = node;

			/* select the node */
			gtk_ctree_select (tree, node);

			/* show all the make sure it's visible */
			for (n=GTK_CTREE_ROW(node)->parent; n; n=GTK_CTREE_ROW(n)->parent)
				if (!GTK_CTREE_ROW(n)->expanded)
					gtk_ctree_expand(tree, n);
		}
	}

	/* hack to sync the focus row on new selection -
	 * thanks to Julien Plissonneau Duquene. */
	if (list->selection != NULL)
		list->focus_row = g_list_position (list->row_list,
		                                   (GList*)(list->selection->data));

	if (first!=NULL) {
		hadj = gtk_clist_get_hadjustment(list)->value;
		gtk_ctree_node_moveto (tree, nodes[0], 0, (gfloat)0.5, (gfloat)0.0);
		gtk_adjustment_set_value (gtk_clist_get_hadjustment(list), hadj);
	}

	gtk_clist_thaw (list);
	debug_exit ("articlelist_set_selected_nodes_nolock");
}

static void
add_to_selection_nolock (ThreadGet get)
{
	int i;
       	GPtrArray * sel = articlelist_get_selected_articles_nolock ();
       	GPtrArray * all = article_get_unique_threads (sel, get);
	GPtrArray * nodes = g_ptr_array_new ();

	for (i=0; i<all->len; ++i) {
		const Article * a = ARTICLE(g_ptr_array_index(all,i));
		const char * message_id = article_get_message_id (a);
		GtkCTreeNode * node = articlelist_get_node_from_message_id (message_id);
		if (node != NULL)
			g_ptr_array_add (nodes, node);
	}

	articlelist_set_selected_nodes_nolock ((GtkCTreeNode**)nodes->pdata, nodes->len);

	g_ptr_array_free (nodes, TRUE);
	g_ptr_array_free (all, TRUE);
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_add_replies_to_selection_nolock (void)
{
	add_to_selection_nolock (GET_SUBTHREAD);
}
void
articlelist_add_thread_to_selection_nolock (void)
{
	add_to_selection_nolock (GET_WHOLE_THREAD);
}

static void
select_node_nolock (GtkCTreeNode * node)
{
	GtkCTreeNode * sel = NULL;

	if (node != NULL)
		sel = articlelist_get_selected_node ();
	if (node!=sel)
		articlelist_set_selected_nodes_nolock (&node, 1);
}


/***
****
****
****  NAVIGATION
****
****
***/

GtkCTreeNode*
articlelist_node_next (GtkCTreeNode * node)
{
	GtkCTreeNode* n;

	if (!node)
		return NULL;

	if (GTK_CTREE_ROW(node)->children)
		return GTK_CTREE_ROW(node)->children;

	for (n=node; n!=NULL; n=GTK_CTREE_ROW(n)->parent)
		if (GTK_CTREE_ROW(n)->sibling)
			return GTK_CTREE_ROW(n)->sibling;

	return NULL;
}

GtkCTreeNode*
articlelist_node_prev (GtkCTreeNode * node, gboolean sibling_only)
{
	GtkCTreeNode * parent;
	GtkCTreeNode * sibling;
	
	if (!node)
		return NULL;

	/* get parent */	
	parent = GTK_CTREE_ROW(node)->parent;
	if (!parent)
		return NULL;

	/* parent's first child */
	sibling=GTK_CTREE_ROW(parent)->children;
	if (sibling==node) /* this is the first child */
		return parent;

	/* previous sibling of node */
	while (GTK_CTREE_ROW(sibling)->sibling != node) {
		sibling = GTK_CTREE_ROW(sibling)->sibling;
		pan_warn_if_fail (sibling != NULL);
	}

	if (sibling_only)
		return sibling;

	/* find the absolutely last child of the older sibling */
	for (;;) {
		GtkCTreeNode* tmp = GTK_CTREE_ROW(sibling)->children;
		if (!tmp)
			return sibling;

		/* last child of sibling */
		while (GTK_CTREE_ROW(tmp)->sibling != NULL)
			tmp = GTK_CTREE_ROW(tmp)->sibling;

		sibling = tmp;
	}

	pan_warn_if_reached ();
}

static GtkCTreeNode*
find_prev_thread (GtkCTreeNode * node)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	gboolean wrap = FALSE;

	/* nothing to do */
	if (row_qty < 2)
		return NULL;

	/* get starting point */	
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* find previous */
	if (node != NULL)
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;

	/* either no node selected or we're at the first thread; wrap */
	if (!node || articlelist_node_prev(node,TRUE)==top) {
		wrap = TRUE;
		node = gtk_ctree_node_nth (tree, row_qty-1);
		while (node!=top && GTK_CTREE_ROW(node)->parent!=top)
			node = GTK_CTREE_ROW(node)->parent;
	}

	/* if we didn't wrap around, we need to back up one. */
	if (!wrap)
		node = articlelist_node_prev (node, TRUE);

	return node;
}

static GtkCTreeNode*
find_next_thread (GtkCTreeNode * node)
{
	GtkCTree* tree;
	GtkCTreeNode * top;
	GtkCTreeNode * in = node;
	const int row_qty = GTK_CLIST(Pan.article_ctree)->rows;

	tree = GTK_CTREE(Pan.article_ctree);
	top = gtk_ctree_node_nth (tree, 0);

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows<2)
		return NULL;

	/* find the start node */
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* walk backwards */
	if (node != NULL)
	{
		while (node!=top && GTK_CTREE_ROW(node)->parent!=top)
			node = GTK_CTREE_ROW(node)->parent;
		if (node==top)
			return NULL;
		node = GTK_CTREE_ROW(node)->sibling;
		if (node== NULL && row_qty>=1) /* end of list -- loop around */
			node = gtk_ctree_node_nth (tree, 1);
		if (node == in) /* detect loop */
			node = NULL;

	}
	else if (row_qty > 1)
	{
		node = gtk_ctree_node_nth (tree, 1);
	}

	return node;
}

static GtkCTreeNode*
find_prev_article (GtkCTreeNode * node)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	const int row_qty = GTK_CLIST(tree)->rows;
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	gboolean wrap = FALSE;
	debug_enter ("find_prev_article");

	/* if nothing to read... */
	if (row_qty<2)
		return NULL;

	/* get a starting node... */
	if (node==NULL) {
		node = articlelist_get_selected_node (); 
		if ( !node || articlelist_node_prev(node,TRUE)==top ) {
			wrap = TRUE;
			node = gtk_ctree_node_nth (tree, row_qty-1 );
		}
	}

	/* if we didn't wrap, we need to move backwards by hand */
	if (!wrap)
		node = articlelist_node_prev (node, FALSE);

	debug_exit ("find_prev_article");
	return node;
}

static GtkCTreeNode*
find_next_article (GtkCTreeNode * node)
{
	GtkCTreeNode * retval = NULL;
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows <2)
		return NULL;

	/* get starting point */
	if (!node) {
		node = articlelist_get_selected_node ();
		if (!node)
			node = gtk_ctree_node_nth (tree, 0);
	}

	/* move to next node */
	retval = articlelist_node_next (node);
	if (!retval)
		retval = gtk_ctree_node_nth (tree, 1);

	return retval;
}

static void
articlelist_read_article (Article * article)
{
	g_string_assign (prev_message_id, cur_message_id->str);

	if (article == NULL)
	{
		g_string_assign (cur_message_id, "");
	}
	else
	{
		g_string_assign (cur_message_id, article_get_message_id (article));

		if (article->parts < 2)
		{
			MessageIdentifier * mid = message_identifier_new_from_article (article);
			text_set_from_identifiers (article->group->server, group_get_default_charset(article->group), &mid, 1);
			g_object_unref (mid);
		}
		else
		{
			int i;
			GSList * l;
			GPtrArray * articles;
			MessageIdentifier ** mids;

			/* make an array of weeded Ariticles */
			articles = g_ptr_array_sized_new (64);
			g_ptr_array_add (articles, article);
			for (l=article->threads; l!=NULL; l=l->next)
				g_ptr_array_add (articles, l->data);
			task_save_weed_duplicates (articles);

			/* make an array of MessageIdentifiers */
			mids = g_alloca (sizeof(MessageIdentifier*) * articles->len);
			for (i=0; i<articles->len; ++i)
				mids[i] = message_identifier_new_from_article (ARTICLE(g_ptr_array_index(articles,i)));
			text_set_from_identifiers (article->group->server, group_get_default_charset(article->group), mids, articles->len);
		
			/* cleanup */
			g_ptr_array_free (articles, TRUE);
			for (i=0; i<articles->len; ++i)
				g_object_unref (mids[i]);
		}
	}
}

void
articlelist_read_selected (void)
{
	GtkCTreeNode * node = articlelist_get_selected_node ();
	Article * article = node==NULL ? NULL : articlelist_get_article_from_node (node);
	articlelist_read_article (article);
}

void
articlelist_read_next (void)
{
	GtkCTreeNode * node;

	/* find the first "next" article */
	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else
		node = find_next_article (node);

	/* select/read the article */
	pan_lock ();
	select_node_nolock (node);
	pan_unlock ();
}

void
articlelist_read_next_unread (void)
{
	const Article * a = NULL;
	GtkCTreeNode * node;
	GtkCTreeNode * in;

	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else
		node = find_next_article (node);
	in = node;

	do {
		if ((a = articlelist_get_article_from_node (node)) == NULL)
			break;
		if (!article_is_read(a) && !article_flag_on(a,STATE_ERROR))
			break;
		if ((node = find_next_article (node)) == in) {
			node = NULL;
			break;
		}
	} while (a!=NULL);

	pan_lock ();
	select_node_nolock (node);
	pan_unlock ();
}

void
articlelist_read_next_thread (void)
{
	GtkCTreeNode * node;
	
	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else for (;;) {
		const Article * a;
		if ((node = find_next_thread (node)) == NULL)
			break;
		a = articlelist_get_article_from_node (node);
		if (a!=NULL && acache_has_message (article_get_message_id (a)))
			break;
	}

	select_node_nolock (node);
}

void
articlelist_read_next_unread_thread (void)
{
	const Article * a = NULL;
	GtkCTreeNode * in;
	GtkCTreeNode * node;
	
	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else
		node = find_next_thread (node);
	in = node;

	do {
		a = articlelist_get_article_from_node (node);
		if ((a = articlelist_get_article_from_node (node)) == NULL)
			break;
		if (!article_is_read (a))
			break;
		if ((node = find_next_article (node)) == in) {
			node = NULL;
			break;
		}
	} while (a!=NULL);

	select_node_nolock (node);
}

gboolean
articlelist_has_prev_read (void)
{
	return prev_message_id!=NULL && prev_message_id->len!=0;
}

void
articlelist_read_prev_read (void)
{
	if (articlelist_has_prev_read ())
		articlelist_read_message_id (prev_message_id->str);
}

void
articlelist_read_prev (void)
{
	select_node_nolock (find_prev_article(NULL));
}

void
articlelist_read_prev_thread (void)
{
	GtkCTreeNode * node = NULL;

	for (;;)
	{
		Article * a;
		if ((node = find_prev_thread (node)) == NULL)
			break;
		a = articlelist_get_article_from_node (node);
		if (a!=NULL && acache_has_message (article_get_message_id (a)))
			break;
	}

	select_node_nolock (node);
}

void
articlelist_read_top_of_thread (void)
{
	GtkCTreeNode * n;
	GtkCTreeNode * node = articlelist_get_selected_node ();

	if ((node = articlelist_get_selected_node ()) == NULL)
		return;
	for (n=GTK_CTREE_ROW(node)->parent; n; n=GTK_CTREE_ROW(n)->parent) {
		if (articlelist_get_article_from_node (n) != NULL)
			node = n;
	}

	select_node_nolock (node);
}

void
articlelist_read_parent (void)
{
	GtkCTreeNode * node = articlelist_get_selected_node ();

	if ((node = articlelist_get_selected_node ()) == NULL)
		return;

	node = GTK_CTREE_ROW(node)->parent;
	if (articlelist_get_article_from_node (node) == NULL)
		node = NULL;

	select_node_nolock (node);
}

void
articlelist_read_message_id (const char * message_id)
{
	GtkCTreeNode * node = articlelist_get_node_from_message_id (message_id);
	if (node != NULL)
		select_node_nolock (node);
}



/***
****
****
****  CLEARING OUT THE ARTICLELIST
****
****
***/

static void
clear_hash_table (void)
{
	g_static_mutex_lock (&hash_mutex);
	if (messageid_to_node != NULL)
		g_hash_table_destroy (messageid_to_node);
	messageid_to_node = g_hash_table_new (g_str_hash, g_str_equal);
	g_static_mutex_unlock (&hash_mutex);
}


/***
****
****
****   SORTING
****
****
***/

static const char*
column_to_title (int col)
{
	switch (col) {
		case 0: return " ";
		case 1: return " ";
		case 2: return _("Subject");
		case 3: return _("Lines");
		case 4: return _("Author");
		case 5: return _("Date");
		default: break;
	}
	
	pan_warn_if_reached();
	return "BUG!!";
}

static int
sort_type_to_column (int type)
{
	int col = 0;
	switch (abs(type)) {
		case ARTICLE_SORT_READ_STATE:   col=0; break;
		case ARTICLE_SORT_ACTION_STATE: col=1; break;
		case ARTICLE_SORT_SUBJECT:      col=2; break;
		case ARTICLE_SORT_LINES:        col=3; break;
		case ARTICLE_SORT_AUTHOR:       col=4; break;
		case ARTICLE_SORT_DATE:         col=5; break;
		default: pan_warn_if_reached(); break;
	}

	return col;
}

static void
articlelist_set_sort_bits_nolock (int type)
{
	if (type != sort_type)
	{
		char buf[64];
		int col = 0;

		/* update old column */
		if (abs(sort_type) != abs(type)) {
			col = sort_type_to_column (sort_type);
			gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, column_to_title(col));
		}

		sort_type = type;

		/* update new column */
		col = sort_type_to_column (sort_type);
		g_snprintf (buf, sizeof(buf), "%c%s", (type>0?'+':'-'), column_to_title(col));
		gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, buf);
	}
}

static void
articlelist_set_sort_type (int sort)
{
	articlelist_set_sort_bits_nolock (sort);

	if (my_group != NULL)
		group_set_sort_style (my_group, sort);

	refresh_impl (REFRESH_SORT);
}

static void
articlelist_click_column (GtkCList* clist, int n)
{
	int type;

	switch (n) {
		case 0: type = ARTICLE_SORT_READ_STATE; break;
		case 1: type = ARTICLE_SORT_ACTION_STATE; break;
		case 2: type = ARTICLE_SORT_SUBJECT; break;
		case 3: type = ARTICLE_SORT_LINES; break;
		case 4: type = ARTICLE_SORT_AUTHOR; break;
		case 5: type = ARTICLE_SORT_DATE; break;
		default: pan_warn_if_reached(); type = -ARTICLE_SORT_DATE; break;
	}

	if (type == sort_type)
		type = -type;

	articlelist_set_sort_type (type);
}

/***
****
****
****
***/

Group*
articlelist_get_group (void)
{
	return my_group;
}


/***
****
****  ARTICLES ADDED
****
***/

static void
group_articles_added_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	Group * group = GROUP(call_obj);
	GPtrArray * added = (GPtrArray*) call_arg;
	debug_enter ("group_articles_added_cb");

	/* sanity checks */
	g_return_if_fail (group_is_valid(group));
	g_return_if_fail (added!=NULL);
	g_return_if_fail (added->len>0u);

	if (group == my_group)
	{
		pan_g_ptr_array_append (my_articles, added->pdata, added->len);
		refresh_impl (~0);
	}

	debug_exit ("group_articles_added_cb");
}

/***
****
****  ARTICLES REMOVED
****
***/

typedef struct
{
	Article * a;
	int depth;
}
SortDepthStruct;

static int
compare_pSDS_to_pSDS_by_depth (const void * a, const void * b)
{
	return ((SortDepthStruct*)a)->depth -
	       ((SortDepthStruct*)b)->depth;
}

static void
sort_articles_by_depth (GPtrArray * articles, gboolean parents_first)
{
	guint i;
	SortDepthStruct * buf;
	debug_enter ("sort_articles_by_depth");

	/* make an array of article,depth pairs */
	buf = g_new0 (SortDepthStruct, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		SortDepthStruct * item = buf + i;
		const Article * tmp;
		item->a = ARTICLE(g_ptr_array_index(articles,i));
		item->depth = 0;
		for (tmp=item->a; tmp->parent!=NULL; tmp=tmp->parent)
			++item->depth;
	}

	/* sort the array */
	qsort (buf,
	       articles->len,
	       sizeof(SortDepthStruct),
	       compare_pSDS_to_pSDS_by_depth);


	/* put back into the original GPtrArray */
	for (i=0; i!=articles->len; ++i)
		g_ptr_array_index(articles,i)= buf[parents_first ? i : articles->len-1-i].a;

	/* cleanup */
	g_free (buf);
	debug_exit ("sort_articles_by_depth");
}

/**
 * This is for selecting the next non-deleted article.
 * Given the list of articles (about to be deleted), we return the next
 * article in the articlelist which is not in the articles array.
 */
static const Article*
articlelist_get_next_article_not_in_list (GPtrArray * articles)
{
	guint i = 0;
	const Article * retval = NULL;
	const Article * first = NULL;
	GHashTable * h = NULL;
	GtkCTreeNode * node = NULL;

	/* sanity clause */
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (articles->len!=0, NULL);

	/* build a hashtable of doomed articles for quick lookup */
	h = g_hash_table_new (g_direct_hash, g_direct_equal);
	for (i=0; i<articles->len; ++i) {
		gpointer g = g_ptr_array_index (articles, i);
		if (g != NULL)
			g_hash_table_insert (h, g, g);
	}

	/* now walk through articles, looking for a doomed one */
	node = articlelist_get_selected_node ();
	first = articlelist_get_article_from_node (node);
	while (retval==NULL)
	{
		const Article * a = NULL;
		node = find_next_article (node);
		a = articlelist_get_article_from_node (node);
		if (a==NULL || a==first) /* end of list */
			break;
		if (g_hash_table_lookup (h, a) == NULL)
			retval = a;
	}

	/* cleanup */
	g_hash_table_destroy (h);
	return retval;
}

static void
remove_article_nolock (const Article * article)
{
	const char * message_id;
	GtkCTreeNode * node = NULL;
	GtkCTree * ctree = GTK_CTREE(Pan.article_ctree);

	/* sanity checks */
	g_return_if_fail (article_is_valid(article));

	/* if this article currently has a node, remove the node */
	message_id = article_get_message_id (article);
	node = articlelist_get_node_from_message_id (message_id);
	if (node!=NULL)
	{
		GSList *l=NULL, *l2;
		GtkCTreeNode * child;

		/* make a list of children */
		child = GTK_CTREE_ROW(node)->children;
		while (child != NULL) {
			l = g_slist_prepend (l, child);
			child = GTK_CTREE_ROW(child)->sibling;
		}
		l = g_slist_reverse (l);

		/* for each child, reparent to the granparent */
		for (l2=l; l2!=NULL; l2=l2->next) {
			GtkCTreeNode * child = (GtkCTreeNode*) l2->data;
			gtk_ctree_move (ctree, child,
			                GTK_CTREE_ROW(node)->parent,
			                GTK_CTREE_ROW(node)->sibling);
		}

		/* remove from hashtable */
		g_static_mutex_lock (&hash_mutex);
		g_hash_table_remove (messageid_to_node, message_id);
		g_static_mutex_unlock (&hash_mutex);

		/* cleanup */
		g_slist_free (l);
	}

	/* if the article was unread, update the parents' unread children count */
	if (!article_is_read (article))
	{
		GtkCTreeNode *n = node;
		if (n == NULL)
		{
			const Article* a;
			for (a=article->parent; a!=NULL && !n; a=a->parent)
				n = articlelist_get_node_from_message_id (article_get_message_id(article));
		}
		if (n != NULL) 
		{
			articlelist_unread_inc_nolock (n, -1);
		}
	}

	/* remove this node from the tree */
	if (node != NULL)
		gtk_ctree_remove_node (ctree, node);
}

static gboolean _sel_from_delete = FALSE;

static int
articles_removed_mainthread (gpointer p)
{
	ArgSet * argset;
	Group * group;
	GPtrArray * articles;
	const gboolean has_selection = GTK_CLIST(Pan.article_ctree)->selection != NULL;
	debug_enter ("articles_removed_mainthread");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);
	articles = (GPtrArray*) argset_get (argset, 1);

	if (group == my_group)
	{
		int i;
		const Article * next = NULL;

		/* sort the articles so that we remove the children first --
		   this reduces the number of cnodes we have to reparent */
		sort_articles_by_depth (articles, FALSE);

		if (has_selection)
			next = articlelist_get_next_article_not_in_list (articles);

		/* lock tree */
		g_static_mutex_lock (&article_ctree_lock);
		pan_lock ();
		gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));

		/* remove the articles from the ctree */
		pan_g_ptr_array_foreach (articles, (GFunc)remove_article_nolock, NULL);

		/* try to select the next article that wasn't deleted */
		if (next != NULL) {
			GtkCTreeNode * n = articlelist_get_node_from_message_id (article_get_message_id(next));
			if (n != NULL) {
				_sel_from_delete = TRUE;
				select_node_nolock (n);
			}
		}

		/* unlock tree */
		gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
		pan_unlock ();
		g_static_mutex_unlock (&article_ctree_lock);

		/* remove them from our array */
		for (i=0; i<articles->len; ++i) {
			const Article * needle = ARTICLE(g_ptr_array_index(articles,i));
			g_ptr_array_remove (my_articles, (gpointer)needle);
		}
	}

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	group_unref_articles (group, NULL);
	argset_free (argset);

	debug_exit ("articles_removed_mainthread");
	return 0;
}

static void
group_articles_removed_cb (gpointer call_obj, gpointer call_arg, gpointer client_data)
{
	ArgSet * argset;
	Group * group = GROUP(call_obj);
	GPtrArray * removed = (GPtrArray*) call_arg;

	/* ref the articles and callback info for safekeeping */
	group_ref_articles (group, NULL);
       	argset = argset_new2 (group, pan_g_ptr_array_dup(removed));

	/* push the rest of the work to the main thread */
	gui_queue_add (articles_removed_mainthread, argset);
}


/****
*****
*****  MARKING ARTICLES READ/UNREAD
*****
****/

static void
articlelist_mark_articles_nolock (const GPtrArray * articles, gboolean read)
{
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len!=0);

	articles_set_read ((Article**)articles->pdata, articles->len, read);
}

void
articlelist_selected_mark_unread_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_mark_unread_nolock");

	sel = articlelist_get_selected_articles_nolock ();
	if (sel->len != 0)
		articlelist_mark_articles_nolock (sel, FALSE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_mark_unread_nolock");
}
void
articlelist_selected_mark_read_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_mark_read_nolock");

       	sel = articlelist_get_selected_articles_nolock ();
	if (sel->len != 0)
		articlelist_mark_articles_nolock (sel, TRUE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_mark_read_nolock");
}


/**
 * http://bugzilla.gnome.org/show_bug.cgi?id=90388
 * ``I think that selecting the thread is more common than selecting the top
 *   article. That's why I think that any operation should operate just like in
 *   Agent i.e. on the entire thread if it is closed and on the top article
 *   otherwise.''
 */
static GPtrArray*
get_flag_articles (void)
{
	guint i;
	GPtrArray * nodes;
	GPtrArray * retval;
	GPtrArray * parents;
	debug_enter ("get_flag_articles");

	parents = g_ptr_array_sized_new (128);
	retval = g_ptr_array_sized_new (128);

	/* add collapsed && !leaves to "parents", all others to "retval" */
	nodes = articlelist_get_selected_nodes_nolock ();
	for (i=0; i<nodes->len; ++i)
	{
		GtkCTreeNode * n = GTK_CTREE_NODE (g_ptr_array_index (nodes, i));
		Article * a = articlelist_get_article_from_node(n);
		GtkCTreeRow * row;

		if (a == NULL)
			continue;

		row = GTK_CTREE_ROW (n);
		if (row->is_leaf || row->expanded)
			g_ptr_array_add (retval, a);
		else
			g_ptr_array_add (parents, a);
	}

	/* if any parents, get the entire subthreads and add to retval */
	if (parents->len) {
		GPtrArray * tmp = article_get_unique_threads (parents, GET_SUBTHREAD);
		pan_g_ptr_array_append (retval, tmp->pdata, tmp->len);
		g_ptr_array_free (tmp, TRUE);
	}

	/* cleanup */
	g_ptr_array_free (parents, TRUE);
	g_ptr_array_free (nodes, TRUE);

	debug_exit ("get_flag_articles");
	return retval;
}

void
articlelist_selected_unflag_for_dl_nolock (void)
{
	GPtrArray * s;
	debug_enter ("articlelist_selected_unflag_for_dl_nolock");

       	s = get_flag_articles ();
	if (s->len)
		flagset_remove_articles ((const Article **)s->pdata, s->len);
	g_ptr_array_free (s, TRUE);

	debug_exit ("articlelist_selected_unflag_for_dl_nolock");
}

void
articlelist_selected_flag_for_dl_nolock (void)
{
	GPtrArray * s;
	debug_enter ("articlelist_selected_unflag_for_dl_nolock");

       	s = get_flag_articles ();
	if (s->len)
		flagset_add_articles ((const Article **)s->pdata, s->len);
	g_ptr_array_free (s, TRUE);

	debug_exit ("articlelist_selected_unflag_for_dl_nolock");
}


/****
*****
*****  CANCEL / SUPERCEDE
*****
****/

void
articlelist_selected_cancel (void)
{
	Article * a;
	debug_enter ("articlelist_selected_cancel");

       	a = articlelist_get_selected_article_nolock ();
	if (a != NULL)
	{
		MessageIdentifier * mid = message_identifier_new_from_article (a);
		GMimeMessage * message = acache_get_message (&mid, 1);

		if (message != NULL)
		{
			article_cancel (a->group->server, message);
			g_object_unref (message);
		}
		else
		{
			log_add (LOG_ERROR|LOG_URGENT, _("You must download the article before you can cancel it."));
		}

		g_object_unref (mid);
	}

	debug_exit ("articlelist_selected_cancel");
}

void
articlelist_selected_supersede (void)
{
	Article * a;
	debug_enter ("articlelist_selected_supersede");

       	a = articlelist_get_selected_article_nolock ();
	if (a != NULL)
	{
		MessageIdentifier * mid = message_identifier_new_from_article (a);
		GMimeMessage * message = acache_get_message (&mid, 1);

		if (message != NULL)
		{
			article_supersede (message);
			g_object_unref (message);
		}
		else
		{
			log_add (LOG_ERROR|LOG_URGENT, _("You must download the article before you can supersede it."));
		}

		g_object_unref (mid);
	}

	debug_exit ("articlelist_selected_supersede");
}

/****
*****
*****  DOWNLOADING ARTICLE BODIES
*****
****/


void
articlelist_selected_download_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_download_nolock");

	g_return_if_fail (my_group != NULL);

	/* add the task */
       	sel = articlelist_get_selected_articles_nolock ();
	if (sel->len)
		queue_add (TASK(task_bodies_new_from_articles ((const Article**)sel->pdata, sel->len)));

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_download_nolock");
}


/***
****
****
****   IGNORING / WATCHING THREADS
****
****
***/

static void
rekill_thread_nolock (Article * article)
{
	guint i;
	GPtrArray * articles;
	int state;
	debug_enter ("rekill_thread_nolock");

	g_return_if_fail (article != NULL);

	/* get the articles and their state */
	state = pan_thread_get_state (article);
       	articles = g_ptr_array_new ();
	article_get_entire_thread (article, articles);

	/* update the tree */
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	for (i=0; i!=articles->len; ++i) {
		const Article * a = ARTICLE(g_ptr_array_index(articles,i));
		articlelist_update_node_impl (a);
	}
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("rekill_thread_nolock");
}

void
articlelist_spamize_nolock (void)
{
	guint i;
	GPtrArray * articles;
	debug_enter ("articlelist_spamize_nolock");

	articles = articlelist_get_selected_articles_nolock ();
	for (i=0; i<articles->len; ++i)
	{
		GtkWidget * w = article_filter_dialog_new (Pan.window, ARTICLE(g_ptr_array_index(articles,i)));
	        gtk_widget_show_all (w);
	}

	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_spamize_nolock");
}

static void
articlelist_selected_thread_watch_nolock_ghfunc (gpointer key, gpointer val, gpointer data)
{
	Article * a = ARTICLE(val);
	int o = pan_thread_get_state (a);
	int n = o==THREAD_WATCHED ? THREAD_NOTHING : THREAD_WATCHED;

	pan_thread_set_state (a, n);
	rekill_thread_nolock (a);

	g_free (key);
}
void
articlelist_selected_thread_watch_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_thread_watch_nolock");

	sel = articlelist_get_selected_articles_nolock ();
	if (sel->len > 0u)
	{
		guint i;
		GHashTable * hash = g_hash_table_new (g_str_hash, g_str_equal);

		/* build a hash table of {root message_id,root Article*} pairs */
		for (i=0; i<sel->len; ++i) {
			const Article * a = ARTICLE(g_ptr_array_index (sel, i));
			char * thread_message_id = article_get_thread_message_id (a);
			if (g_hash_table_lookup (hash, thread_message_id) == NULL)
				g_hash_table_insert (hash, thread_message_id, (gpointer)a);
			else
				g_free (thread_message_id);
		}

		if (g_hash_table_size(hash) > 0u) {
			gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
			g_hash_table_foreach (hash, articlelist_selected_thread_watch_nolock_ghfunc, NULL);
			g_hash_table_destroy (hash);
			gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
		}
	}

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_thread_watch_nolock");
}

void
articlelist_selected_thread_ignore_nolock (void)
{
	guint i;
	GPtrArray * sel;
	debug_enter ("articlelist_selected_thread_ignore_nolock");

	sel = articlelist_get_selected_articles_nolock ();
	for (i=0; i<sel->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(sel,i));
		int o = pan_thread_get_state (a);
		int n = o==THREAD_IGNORED ? THREAD_NOTHING : THREAD_IGNORED;
		pan_thread_set_state (a, n);
		rekill_thread_nolock (a);
	}

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_thread_ignore_nolock");
}

/***
****
****
****  POPULATING THE TREE:  UTILITY FUNCTIONS
****
****
***/

static const Icon*
get_column_1_icon (const Article * a)
{
	const Icon * retval = NULL;
	const gboolean is_new = article_is_new (a);
	const gboolean is_read = article_is_read (a);

	g_return_val_if_fail (a!=NULL, NULL);

	if (article_flag_on (a, STATE_ERROR))
		retval = _icons + ICON_ERROR;
	else if (article_flag_on (a, STATE_MULTIPART_ALL))
		retval = _icons + ICON_COMPLETE;
	else if (article_flag_on (a, STATE_MULTIPART_SOME))
		retval = _icons + ICON_INCOMPLETE;
	else if (is_new)
		retval = _icons + ICON_NEW;
	else if (!is_new && !is_read)
		retval = _icons + ICON_UNREAD;
	else
		retval = _icons + ICON_READ;

	return retval;
}

static const Icon*
get_column_2_icon (const Article * a)
{
	const Icon * retval = NULL;

	g_return_val_if_fail (article_is_valid(a), NULL);

	if (flagset_has_message (a->message_id))
		retval = _icons + ICON_FLAGGED_FOR_DL;
	else if (article_flag_on (a, STATE_DECODED))
		retval = _icons + ICON_BINARY_DECODED;
	else if (queue_get_message_id_status (a->message_id))
		retval = _icons + ICON_QUEUED;
	else if (acache_has_message (article_get_message_id (a)))
		retval = _icons + ICON_CACHED;;

	return retval;
}

static PangoFontDescription * normal_pfd = NULL;
static PangoFontDescription * new_replies_pfd = NULL;

static gboolean rebuild_style = TRUE;

#define NORMAL 0
#define NEW_REPLIES 1

static void
articlelist_set_node_style_nolock (GtkCTreeNode   * node,
			           const Article  * article)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);

	/* initialize the styles/fonts the first time through */
	if (rebuild_style)
	{
		int i;
		double lo;
		double hi;
		GdkColor bg;
		GdkColor fg;
		GdkColor unread_color;
		GdkColor read_color;
		GtkStyle * s = gtk_widget_get_style (GTK_WIDGET(Pan.window));

		rebuild_style = FALSE;

		/* read color should be mostly the background color with a little foreground */
		bg = s->bg[0];
		fg = unread_color = s->fg[0];
		hi = MAX (fg.red, bg.red);
		lo = MIN (fg.red, bg.red);
		read_color.red = lo + ((lo==bg.red ? 0.2 : 0.8) * (hi-lo));
		hi = MAX (fg.green, bg.green);
		lo = MIN (fg.green, bg.green);
		read_color.green = lo + ((lo==bg.green ? 0.2 : 0.8) * (hi-lo));
		hi = MAX (fg.blue, bg.blue);
		lo = MIN (fg.blue, bg.blue);
		read_color.blue = lo + ((lo==bg.blue ? 0.2 : 0.8) * (hi-lo));
		gdk_color_alloc (cmap, &read_color);

		/* build the styles */
		for (i=0; i<2; ++i)
		{
			PangoFontDescription * pfd = i ? new_replies_pfd : normal_pfd;

			normal_style[i] = gtk_style_copy (s);
			normal_style[i]->fg[0] = unread_color;
			if (pfd != NULL) {
				PangoFontDescription * tmp = normal_style[i]->font_desc;
				normal_style[i]->font_desc = pango_font_description_copy_static (pfd);
				pango_font_description_free (tmp);
			}

			read_style[i] = gtk_style_copy (s);
			read_style[i]->fg[0] = read_color;
			if (pfd != NULL) {
				PangoFontDescription * tmp = read_style[i]->font_desc;
				read_style[i]->font_desc = pango_font_description_copy_static (pfd);
				pango_font_description_free (tmp);
			}

			killfile_style[i] = gtk_style_copy (s);
			killfile_style[i]->fg[0] = killfile_color;
			if (pfd != NULL) {
				PangoFontDescription * tmp = killfile_style[i]->font_desc;
				killfile_style[i]->font_desc = pango_font_description_copy_static (pfd);
				pango_font_description_free (tmp);
			}

			watched_style[i] = gtk_style_copy (s);
			watched_style[i]->fg[0] = watched_color;
			if (pfd != NULL) {
				PangoFontDescription * tmp = watched_style[i]->font_desc;
				watched_style[i]->font_desc = pango_font_description_copy_static (pfd);
				pango_font_description_free (tmp);
			}
		}
	}

	if (1)
	{
		GtkStyle * style = NULL;
		const gboolean is_unread = article ? !article_is_read (article) : TRUE;
		const gboolean is_new = article ? article_is_new (article) : FALSE;
		const gboolean new_children = article ? article->new_children!=0 : FALSE;
		const gboolean unread_children = article ? article->unread_children!=0 : FALSE;
		const int thread_state = article ? pan_thread_get_state(article) : THREAD_NOTHING;
		const gboolean collapsed_subthread = node ?
			!GTK_CTREE_ROW(node)->is_leaf && !GTK_CTREE_ROW(node)->expanded : FALSE;
		const gboolean do_mark_new = is_new || (collapsed_subthread && new_children);
		const gboolean do_mark_unread = is_unread || (collapsed_subthread && unread_children);

		if (thread_state == THREAD_IGNORED)
			style = killfile_style [do_mark_new ? NEW_REPLIES : NORMAL];
		else if (thread_state == THREAD_WATCHED)
			style = watched_style [do_mark_new ? NEW_REPLIES : NORMAL];
		else if (do_mark_new)
			style = normal_style[NEW_REPLIES];
		else if (do_mark_unread)
			style = normal_style[NORMAL];
		else
			style = read_style[NORMAL];

		gtk_ctree_node_set_row_style (tree, node, style);
	}
}

static void
apply_filter_tests (Filter * filter, FilterShow show, GPtrArray * headers)
{
	register int i;
	register Article * a;
	GPtrArray * tmp;
	GPtrArray * family;

	g_return_if_fail (headers!=NULL);

	/* make a working copy of the headers */
	tmp = pan_g_ptr_array_dup (headers);

	/* clear the filter state */
	for (i=0; i!=tmp->len; ++i) {
		a = ARTICLE(g_ptr_array_index(tmp,i));
		a->self_passes_filter = FALSE;
		a->tree_passes_filter = FALSE;
	}

	/* remove the articles that don't pass */
	if (filter != NULL)
		filter_remove_failures (filter, tmp);

	/* mark up the successes */
	family = g_ptr_array_new ();
	for (i=0; i<tmp->len; ++i)
	{
		int j;
		a = ARTICLE(g_ptr_array_index(tmp,i));

		/* this article passes the filter */
		a->self_passes_filter = TRUE;
		a->tree_passes_filter = TRUE;

		/* process the family */
		g_ptr_array_set_size (family, 0);
		switch (show) {
			case FILTER_SHOW_MATCHES:
				break;
			case FILTER_SHOW_MATCHES_AND_REPLIES:
				article_get_subthread (a, family);
				break;
			case FILTER_SHOW_MATCHES_AND_REFERENCES:
				article_get_references (a, family);
				break;
			case FILTER_SHOW_THREAD:
				article_get_entire_thread (a, family);
				break;
		}
		for (j=0; j<family->len; ++j) {
			Article * relative = ARTICLE(g_ptr_array_index(family,j));
			relative->tree_passes_filter = TRUE;
		}
	}

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);
	g_ptr_array_free (family, TRUE);
}

/***
****
****
****  POPULATING THE TREE:  THE DIRTY WORK
****
****
***/

static char*
articlelist_get_subject (char            * buf,
                         gulong            buflen,
                         const Article   * article,
                         gboolean          expanded)
{
	g_return_val_if_fail (buf!=NULL, NULL);
	g_return_val_if_fail (buflen!=0ul, NULL);
	g_return_val_if_fail (article_is_valid(article), NULL);

	pan_warn_if_fail (article->unread_children >= article->new_children);

	if (expanded || !article->threads || !article->new_children)
		g_strlcpy (buf, article_get_subject(article), buflen);
	else
		g_snprintf (buf, buflen, "%s (%u)", article_get_subject(article), article->new_children);

	return buf;
}

static GtkCTreeNode*
add_article_to_ctree_nolock (GtkCTree      * tree,
                             GtkCTreeNode  * parent,
                             GtkCTreeNode  * sibling,
                             const Article * article,
                             gboolean        expanded)
{
	const Icon * col_1 = NULL;
	const Icon * col_2 = NULL;
	GtkCTreeNode * node;
	gulong lcountsum;
	char * text[COLUMNS] = { NULL, NULL, NULL, NULL, NULL, NULL };
	char author[512];
	char subject[512];
	char date[512];
	char linebuf[32];
	int i;
	char * date_utf8, * freeme;

	article_get_short_author_str (article, author, sizeof(author));

	articlelist_get_subject (subject, sizeof(subject), article, expanded);

	get_date_display_string (article->date, thread_date_format, date, sizeof(date));
	date_utf8 = (char *) pan_utf8ize (date, -1, &freeme);

	lcountsum = article->linecount;
	if (article_flag_on (article, STATE_MULTIPART_ALL)) {
		int part = 2;
		GSList * l;
		for (l=article->threads; l!=NULL; l=l->next) {
			const Article * child = ARTICLE(l->data);
			if (child->part == part) {
				lcountsum += child->linecount;
				++part;
			}
		}
	}
	g_snprintf (linebuf, sizeof(linebuf), "%lu", lcountsum);

	/* populate the line */
	i = 2; /* skip columns 1 & 2; icons go there */
	text[i++] = subject;
	text[i++] = linebuf;
	text[i++] = author;
	text[i++] = date_utf8;
	g_assert (i==COLUMNS);

	/* insert the node */
	node = gtk_ctree_insert_node (tree, parent, sibling,
	                              (char**)text, GUI_PAD_SMALL,
	                              NULL, NULL, NULL, NULL,
	                              FALSE, expanded);
	if ((col_1 = get_column_1_icon (article)) != NULL)
		gtk_ctree_node_set_pixmap (tree, node, 0, col_1->pixmap, col_1->mask);
	if ((col_2 = get_column_2_icon (article)) != NULL)
		gtk_ctree_node_set_pixmap (tree, node, 1, col_2->pixmap, col_2->mask);
	gtk_ctree_node_set_row_data (tree, node, (gpointer)article);
	articlelist_set_node_style_nolock (node, article);

	/* update hash table */
	g_static_mutex_lock (&hash_mutex);
	g_hash_table_insert (messageid_to_node, (gpointer)article_get_message_id(article), node);
	g_static_mutex_unlock (&hash_mutex);

	/* cleanup */
	g_free (freeme);

	return node;
}

static void
build_article_ctree_recursive (GtkCTree      * tree,
                               GtkCTreeNode  * parent,
                               GSList        * list,
                               gboolean        expanded)
{
	GSList * tmp;
	GSList * pos;
	GtkCTreeNode * sibling = NULL;

	/* sanity check */
	g_return_if_fail (parent!=NULL);

	/* this item gets inserted before the 'sibling' node,
	 * so we must add in reverse order */
	tmp = g_slist_reverse(g_slist_copy(list));
        for (pos=tmp; pos!=NULL; pos=pos->next)
        {
		const Article* article = ARTICLE(pos->data);
		GtkCTreeNode* node = NULL;

		/* possibly don't show subtrees that don't measure up to filter */
		if (header_pane_is_threaded && !article->tree_passes_filter)
			continue;

		/* possibly don't show 2...n of multipart nodes */
		if (hide_mpart_child_nodes
			&& (article->parent!=NULL)
			&& (article->parent->state&STATE_MULTIPART_ALL)
			&& (article->part>1))
			continue;

		node = add_article_to_ctree_nolock (tree, parent, sibling, article, expanded);
		if (article->threads!=NULL)
			build_article_ctree_recursive (tree, node, article->threads, TRUE);

		sibling = node;
        }

	/* cleanup */
	g_slist_free (tmp);
}

static void
articlelist_repopulate_nolock (const Group    * group,
                               const Article ** article_buf,
                               int             article_qty,
                               GPtrArray      * sel_articles)
{
	int i;
	GtkCTree * tree = GTK_CTREE (Pan.article_ctree);
	GtkCTreeNode * sibling = NULL;
	GtkCTreeNode * root = NULL;
	char * text[COLUMNS] = { NULL, NULL, NULL, NULL, NULL, NULL };
	debug_enter ("articlelist_repopulate_nolock");

	/**
	*** Make sure we don't trigger any * 'tree-select-row' signals
	*** FIXME: better way of doing this ?
	**/

	_articlelist_repopulating = TRUE;

	/* clear out old */
	gtk_clist_freeze (GTK_CLIST(tree));
	gtk_clist_clear (GTK_CLIST(tree));
	clear_hash_table ();
	
	/* add root node */
	text[2] = (char*) group_get_name (group);
	root = gtk_ctree_insert_node (tree, NULL, NULL,
	                              text, GUI_PAD_SMALL,
				      _icons[ICON_NEWSGROUP].pixmap,
				      _icons[ICON_NEWSGROUP].mask,
				      _icons[ICON_NEWSGROUP].pixmap,
				      _icons[ICON_NEWSGROUP].mask,
	                              FALSE, FALSE);
	gtk_ctree_node_set_row_data (tree, root, NULL);
	articlelist_set_node_style_nolock (root, NULL);

	/* add the articles */
	for (i=article_qty-1; i>=0; --i) /* slightly faster to add to the top of ctree */
	{
		const Article * article = article_buf[i];
		GtkCTreeNode * node = NULL;
		gboolean has_parent;

		/* make sure we pass the state filter */
		if (!article)
			continue;
		if (header_pane_is_threaded && !article->tree_passes_filter)
			continue;
		if (!header_pane_is_threaded && !article->self_passes_filter)
			continue;

		has_parent = article->parent != NULL;

		if ((!header_pane_is_threaded && article->part<2) ||
			(header_pane_is_threaded && (!has_parent || !article->parent->tree_passes_filter)))
			sibling = node = add_article_to_ctree_nolock (tree, root, sibling, article, FALSE);

		/* if threads are turned on, and there are children of this thread... */
		if (header_pane_is_threaded && node!=NULL && article->threads!=NULL)
			build_article_ctree_recursive (tree, node, article->threads, TRUE);
	}

	/**
	***  Update Selection
	**/

	if (sel_articles!=NULL && sel_articles->len)
	{
		int i;
		GPtrArray * tmp;

		tmp = g_ptr_array_sized_new (sel_articles->len);
		for (i=0; i<sel_articles->len; ++i) {
			const Article * a = ARTICLE(g_ptr_array_index(sel_articles,i));
			GtkCTreeNode * node = articlelist_get_node_from_message_id (article_get_message_id(a));
			if (node != NULL)
				g_ptr_array_add (tmp, node);
		}

		if (tmp->len)
			articlelist_set_selected_nodes_nolock ((GtkCTreeNode**)tmp->pdata, tmp->len);

		/* cleanup */
		g_ptr_array_free (tmp, TRUE);
	}

	/**
	***  Finished; display
	**/

	if (expand_all_threads_by_default)
		gtk_ctree_expand_recursive (tree, root);
	else
		gtk_ctree_expand (tree, root);
	gtk_clist_thaw (GTK_CLIST(tree));
	gtk_widget_queue_resize (Pan.article_ctree);

	/**
	*** OK to follow up on tree-select-row now
	**/
	_articlelist_repopulating = FALSE;

	debug_exit ("articlelist_repopulate_nolock");
}

/***
****
****
****  UPDATING EXISTING NODES: THE DIRTY WORK
****
****
***/

static void 
articlelist_update_node_fast_nolock (GtkCTreeNode   * node,
                                     const Article  * article)
{
	GtkCTree * ctree = GTK_CTREE(Pan.article_ctree);
	gboolean collapsed  = !GTK_CTREE_ROW(node)->is_leaf && !GTK_CTREE_ROW(node)->expanded;
	const Icon * col_1 = get_column_1_icon (article);
	const Icon * col_2 = get_column_2_icon (article);
	char subject[512];

	articlelist_get_subject (subject, sizeof(subject), article, !collapsed);

	gtk_ctree_set_node_info (ctree, node, subject, GUI_PAD_SMALL,
	                         NULL, NULL, NULL, NULL,
	                         GTK_CTREE_ROW(node)->is_leaf,
	                         GTK_CTREE_ROW(node)->expanded);

	if (col_1 != NULL)
		gtk_ctree_node_set_pixmap (ctree, node, 0, col_1->pixmap, col_1->mask);
	if (col_2 == NULL)
		gtk_ctree_node_set_text (ctree, node, 1, "");
	else
		gtk_ctree_node_set_pixmap (ctree, node, 1, col_2->pixmap, col_2->mask);

	articlelist_set_node_style_nolock (node, article);
}

	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
/* these lines left blank for 0.13.0's string freeze */	

static GtkCTreeNode*
articlelist_get_node_from_message_id (const char* message_id)
{
	GtkCTreeNode * retval = NULL;

	if (is_nonempty_string(message_id))
	{
		g_static_mutex_lock (&hash_mutex);
		retval = g_hash_table_lookup (messageid_to_node, message_id);
		g_static_mutex_unlock (&hash_mutex);
	}

	return retval;
}
static void
articlelist_update_node_impl (const Article* article)
{
	GtkCTreeNode* node =
		articlelist_get_node_from_message_id (
			article_get_message_id (article));

        if (node != NULL)
                articlelist_update_node_fast_nolock (node, article);
}

static int
articlelist_update_articles_idle (gpointer data)
{
	int i;
	GPtrArray * a = (GPtrArray*) data;

	/* sanity clause */
	g_return_val_if_fail (a!=NULL, 0);
	g_return_val_if_fail (a->len>0, 0);
	g_return_val_if_fail (g_ptr_array_index(a,0)!=NULL, 0);

	/* update the nodes */
	pan_lock ();
	for (i=0; i<a->len; ++i) {
		const Article * article = ARTICLE(g_ptr_array_index(a,i));
		if (article->group == my_group)
			articlelist_update_node_impl (article);
	}
	pan_unlock ();

	/* cleanup */
	group_unref_articles (ARTICLE(g_ptr_array_index(a,0))->group, NULL);
	g_ptr_array_free (a, TRUE);
	return 0;
}

static void
articlelist_update_articles (const Article** articles, int qty)
{
	GPtrArray * a;

	/* sanity clause */
	g_return_if_fail (qty>0);
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles[0]!=NULL);

	/* batch it up for the idle func */
	group_ref_articles (articles[0]->group, NULL);
	a = g_ptr_array_sized_new (qty);
	pan_g_ptr_array_assign (a, (gpointer*)articles, qty);
	gui_queue_add (articlelist_update_articles_idle, a);
}

/***
****
****
****  MANAGING UNREAD CHILDREN COUNTS
****
****
***/

static void
articlelist_unread_inc_nolock (GtkCTreeNode* node, int inc)
{
	debug_enter ("articlelist_unread_inc_nolock");

	/* update the node & parents */
	while (node != NULL) {
		const Article * a = articlelist_get_article_from_node (node);
		if (a == NULL)
			break;
		articlelist_update_node_fast_nolock (node, a);
		node = GTK_CTREE_ROW(node)->parent;
	}

	debug_exit ("articlelist_unread_inc_nolock");
}

static int
articlelist_articles_read_changed_idle (gpointer data)
{
	int i;
	GPtrArray * articles;
	GtkCList * clist;
	debug_enter ("articlelist_articles_read_changed_idle");

 	articles = (GPtrArray *) data;
	clist = GTK_CLIST(Pan.article_ctree);

	pan_lock ();
	gtk_clist_freeze (clist);
	for (i=0; i!=articles->len; ++i)
	{
		const Article * article = ARTICLE(g_ptr_array_index(articles,i));
		const Article * a = NULL;
		GtkCTreeNode* node = NULL;

		/* find the matching node & article */
		node = articlelist_get_node_from_message_id (article_get_message_id(article));
		if (node != NULL)
			a = articlelist_get_article_from_node (node);

		/* if we've got a match, update the node */
		if (a != NULL) {
			int inc = article_is_read (article) ? -1 : 1;
			articlelist_unread_inc_nolock (node, inc);
		}
	}
	gtk_clist_thaw (clist);
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_articles_read_changed_idle");
	return 0;
}

static void
articlelist_articles_read_changed (const Article ** articles, int len)
{
	debug_enter ("articlelist_articles_read_changed");

	g_return_if_fail (articles!=NULL);
	g_return_if_fail (len>=1);

	if (my_group == articles[0]->group)
	{
		GPtrArray * a = g_ptr_array_new ();
		pan_g_ptr_array_assign  (a, (gpointer*)articles, len);
		gui_queue_add (articlelist_articles_read_changed_idle, a);
	}

	debug_exit ("articlelist_articles_read_changed");
}



/***
****
****
****  MANAGING NEW CHILD COUNTS
****
****
***/

static void
articlelist_new_inc_nolock (GtkCTreeNode* node, int inc)
{
	debug_enter ("articlelist_new_inc_nolock");

	/* update the node & parents */
	while (node != NULL) {
		const Article * a = articlelist_get_article_from_node (node);
		if (a != NULL)
			break;
		articlelist_update_node_fast_nolock (node, a);
		node = GTK_CTREE_ROW(node)->parent;
	}

	debug_exit ("articlelist_new_inc_nolock");
}

static int
articlelist_articles_new_changed_idle (gpointer data)
{
	int i;
	GPtrArray * articles;
	GtkCList * clist;
	debug_enter ("articlelist_articles_new_changed_idle");

 	articles = (GPtrArray *) data;
	clist = GTK_CLIST(Pan.article_ctree);

	pan_lock ();
	gtk_clist_freeze (clist);
	for (i=0; i!=articles->len; ++i)
	{
		const Article * article = ARTICLE(g_ptr_array_index(articles,i));
		const Article * a = NULL;
		GtkCTreeNode* node = NULL;

		/* find the matching node & article */
		node = articlelist_get_node_from_message_id (article_get_message_id(article));
		if (node != NULL)
			a = articlelist_get_article_from_node (node);

		/* if we've got a match, update the node */
		if (a != NULL) {
			int inc = article_is_new (article) ? 1 : -1;
			articlelist_new_inc_nolock (node, inc);
		}
	}
	gtk_clist_thaw (clist);
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_articles_new_changed_idle");
	return 0;
}

static void
articlelist_articles_new_changed (const Article ** articles, int len)
{
	debug_enter ("articlelist_articles_new_changed");

	g_return_if_fail (articles!=NULL);
	g_return_if_fail (len>=1);

	if (my_group == articles[0]->group)
	{
		GPtrArray * a = g_ptr_array_new ();
		pan_g_ptr_array_assign  (a, (gpointer*)articles, len);
		gui_queue_add (articlelist_articles_new_changed_idle, a);
	}

	debug_exit ("articlelist_articles_new_changed");
}

/***
****
****  LIVING WITH ACACHE
****
***/

static int
articlelist_acache_changed_idle (gpointer data)
{
	char * message_id = NULL;
	GtkCTreeNode * node = NULL;
	const Article * a = NULL;
	debug_enter ("articlelist_acache_changed_idle");

	/* message_id -> node -> article */
 	message_id = (char*) data;
	if (is_nonempty_string (message_id))
		node = articlelist_get_node_from_message_id (message_id);
	if (node != NULL)
		a = articlelist_get_article_from_node (node);
	if (a != NULL) {
		pan_lock ();
		articlelist_update_node_fast_nolock (node, a);
		pan_unlock ();
	}

	/* cleanup */
	g_free (message_id);
	debug_exit ("articlelist_acache_changed_idle");
	return 0;
}

static void
articlelist_acache_changed_cb (gpointer call_object, gpointer call_arg, gpointer user_data)
{
	const char * message_id;
	debug_enter ("articlelist_acache_changed_cb");

 	message_id = (const char*) call_object;
	if (articlelist_get_node_from_message_id (message_id) != NULL)
		gui_queue_add (articlelist_acache_changed_idle, g_strdup(message_id));

	debug_exit ("articlelist_acache_changed_cb");
}

/***
****
****  LIVING WITH QUEUE
****
***/

static int
queue_message_id_status_changed_idle (gpointer data)
{
	int i;
	GPtrArray * message_ids = (GPtrArray*) data;

	/* update the articlelist */
	pan_lock ();
	for (i=0; i<message_ids->len; ++i)
	{
		const char * message_id = NULL;
		GtkCTreeNode * node = NULL;
		const Article * a = NULL;

		if (1)
	       		message_id = (const char *) g_ptr_array_index (message_ids, i);
		if (is_nonempty_string (message_id))
			node = articlelist_get_node_from_message_id (message_id);
		if (node != NULL)
			a = articlelist_get_article_from_node (node);
		if (a != NULL)
			articlelist_update_node_fast_nolock (node, a);
	}
	pan_unlock ();

	/* cleanup */
	pan_g_ptr_array_foreach (message_ids, (GFunc)g_free, NULL);
	g_ptr_array_free (message_ids, TRUE);
	return 0;
}

static void
refresh_articles_from_message_id_array_cb (gpointer call_obj,
                                           gpointer call_arg,
                                           gpointer user_data)
{
	int i;
	const int qty = GPOINTER_TO_INT (call_arg);
	const char ** message_ids = (const char **) call_obj;
	GPtrArray * a;

	/* make a temporary copy of the message ids */
	a = g_ptr_array_sized_new (qty);
	for (i=0; i<qty; ++i)
		g_ptr_array_add (a, g_strdup(message_ids[i]));

	gui_queue_add (queue_message_id_status_changed_idle, a);
}

/***
****
****
****  SELECTION / KEYPRESS / BUTTONPRESS HANDLING
****
****
***/

/**
 * The following buttonpress, keypress, and selection callbacks are all here
 * to interact with each other to do one thing: force this damn GtkCTree to
 * have the UI behavior that we want, which is described in 
 * articlelist_selection_changed()
 */

static int _button_click_count = -1;
static int _mb = -1;
static gboolean _modifiers = FALSE;
static gboolean _keyboard_nav = FALSE;

static void
articlelist_selection_changed_cb (gpointer call_object,
                                  gpointer call_arg,
                                  gpointer user_data)
{
	GPtrArray * articles;
	Article * article = NULL;
	debug_enter ("articlelist_selection_changed_cb");

	/* find a single article */
       	articles = (GPtrArray*) call_arg;
	if (articles!=NULL && articles->len==1)
		article = ARTICLE(g_ptr_array_index(articles,0));

	/* decide whether or not the user wants to load it */
        if (article==NULL || !article_is_valid(article)) {
		/* selection_changed: do nothing (no single selection) */
	}
	else if (_sel_from_delete) {
		/* selection_changed: do nothing (previous article deleted) */
	}
	else if (_modifiers || _mb==2) {
		/* selection_changed: do nothing (modifiers) */
	}
	else if (_keyboard_nav) {
		/* selection_changed: do nothing (keyboard navigation) */
	}
	else if (_button_click_count==1 && single_click_selects) {
		/* selection_changed: do nothing (single-click activation turned off) */
	}
	else if (article_flag_on(article,STATE_MULTIPART_ALL) && article->parts>3 && article->part==1 && _mb!=-1) {
		/* selection_changed: save as... (clicked on multipart) */
		article_action_selected_save_as ();
	}
	else if (article_flag_on(article,STATE_MULTIPART_ALL) && article->parts>3 && article->part==1) {
		/* selection_changed: do nothing (big multipart) */
	}
	else if (acache_has_message (article_get_message_id (article))) {
		/* selection_changed: display cached */
		articlelist_read_article (article);
	}
	else if (!queue_is_online()) {
		/* selection_changed: do nothing (offline) */
	}
	else {
		/* selection_changed: fetch & display */
		articlelist_read_article (article);
	}

	/* reset everything for the next time around */
	_sel_from_delete = FALSE;
	_keyboard_nav = FALSE;
	_button_click_count = -1;
	_modifiers = FALSE;
	_mb = -1;

	debug_exit ("articlelist_selection_changed_cb");
}

/**
 * FIME: workaround for #82901; remove when fixed & gtk version bumped
 */
static int
resync (gpointer p) {
	GtkCList * clist = GTK_CLIST (Pan.article_ctree);
	GTK_CLIST_GET_CLASS (clist)->resync_selection (clist, NULL);
	return 0;
}
static gboolean
articlelist_key_release (GtkWidget * w, GdkEventKey * e, gpointer data) {
	gui_queue_add (resync, NULL);
	return FALSE;
}

static gboolean
articlelist_key_press (GtkWidget      * widget,
                       GdkEventKey    * event,
                       gpointer         data)
{
	gboolean retval = FALSE;

	switch (event->keyval)
	{
		case GDK_Up:
		case GDK_Down:
		case GDK_Page_Up:
		case GDK_Page_Down:
		case GDK_Home:
		case GDK_End:
			_keyboard_nav = TRUE;
			break;

		case GDK_leftarrow:
		case GDK_Left:
			_keyboard_nav = TRUE;
			articlelist_collapse_selected_threads ();
			retval = TRUE;
			break;

		case GDK_rightarrow:
		case GDK_Right:
			_keyboard_nav = TRUE;
			articlelist_expand_selected_threads ();
			retval = TRUE;
			break;

		default:
			_keyboard_nav = FALSE;
	}

	return retval;
}

static gboolean
articlelist_button_press (GtkWidget* widget, GdkEventButton* bevent)
{
	gboolean retval = FALSE;

	GdkModifierType modifiers = 0;
	gdk_event_get_state ((GdkEvent*)bevent, &modifiers);
	_modifiers = modifiers & (GDK_SHIFT_MASK || GDK_CONTROL_MASK || GDK_MOD1_MASK || GDK_MOD2_MASK || GDK_MOD3_MASK || GDK_MOD4_MASK || GDK_MOD5_MASK);

	switch (bevent->button)
	{
		case 1:
		case 2:
			_mb = bevent->button;
			if (bevent->type == GDK_2BUTTON_PRESS)
				_button_click_count = 2;
			else
				_button_click_count = 1;
			retval = FALSE;
			break;
		case 3:
			articlelist_menu_popup_nolock (bevent);
			retval = TRUE;
			break;
	}

	return retval;
}

static gboolean select_callback_pending = FALSE;
 
static int
tree_select_row_idle (gpointer data)
{
	Group * group;
	GPtrArray * articles;
	debug_enter ("tree_select_row_idle");

	group = GROUP (data);

	/* let everyone know that the selection has changed */
	pan_lock ();
       	articles = articlelist_get_selected_articles_nolock ();
	pan_unlock ();
	pan_callback_call (articlelist_selection_changed,
			   Pan.article_ctree,
			   articles);

	/* matching unref is in tree_select_row_cb */
	if (group != NULL)
		group_unref_articles (group, NULL);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	select_callback_pending = FALSE;
	debug_exit ("tree_select_row_idle");
	return 0;
}
static void
tree_select_row_cb (GtkCTree     *tree,
		    GtkCTreeNode *node,
		    int           column,
		    gpointer      data)
{
	if (!select_callback_pending && !_articlelist_repopulating)
	{
		select_callback_pending = TRUE;

		/* matching unref is in tree_select_row_idle */
		if (my_group != NULL)
			group_ref_articles (my_group, NULL);

		gui_queue_add (tree_select_row_idle, my_group);
	}
}

/***
****
****   EXPAND / COLLAPSE THREADS
****
***/

void
articlelist_expand_selected_threads (void)
{
	guint i;
	GPtrArray * a;
	GtkCTree * tree;
	debug_enter ("articlelist_expand_selected_threads");

	/* expand */
	pan_lock ();
	a = articlelist_get_selected_nodes_nolock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	tree = GTK_CTREE(Pan.article_ctree);
	for (i=0; i<a->len; ++i)
		gtk_ctree_expand (tree, (GtkCTreeNode*)g_ptr_array_index(a,i));
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("articlelist_expand_selected_threads");
}

void
articlelist_collapse_selected_threads (void)
{
	guint i;
	GPtrArray * a;
	GtkCTree * tree;
	debug_enter ("articlelist_collapse_selected_threads");

	/* collapse */
	pan_lock ();
	a = articlelist_get_selected_nodes_nolock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	tree = GTK_CTREE(Pan.article_ctree);
	for (i=0; i<a->len; ++i)
		gtk_ctree_collapse (tree, (GtkCTreeNode*)g_ptr_array_index(a,i));
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("articlelist_collapse_selected_threads");
}

static int
tree_expand_idle (gpointer message_id)
{
	GtkCTreeNode * node = NULL;
	const Article * article = NULL;
	debug_enter ("tree_expand_idle");

	pan_lock ();
	if (is_nonempty_string ((char*)message_id))
		node = articlelist_get_node_from_message_id ((char*)message_id);
	if (node != NULL)
		article = articlelist_get_article_from_node (node);
	if (article != NULL) {
		articlelist_set_node_style_nolock (node, article);
		articlelist_update_node_fast_nolock (node, article);
	}
	pan_unlock ();

	g_free (message_id);
	debug_exit ("tree_expand_idle");
	return 0;
}
static void
tree_expand_cb (GtkCTree * ctree, GList * list, gpointer user_data)
{
	GtkCTreeNode * node = NULL;
	const Article * article = NULL;
	char * message_id = NULL;
	debug_enter ("tree_expand_cb");

	if (list != NULL)
		node = GTK_CTREE_NODE(list);
	if (node != NULL)
		article = articlelist_get_article_from_node (node);
	if (article != NULL)
		message_id = g_strdup (article_get_message_id (article));
	if (message_id != NULL)
		gui_queue_add (tree_expand_idle, message_id);

	debug_exit ("tree_expand_cb");
}

static void
tree_collapse_cb (GtkCTree * tree, GList * list, gpointer user_data)
{
	tree_expand_cb (tree, list, user_data);
}

/***
****  Callbacks
***/

static void
articlelist_groups_removed_cb (gpointer server_data,
                               gpointer groups_array,
                               gpointer unused)
{
	guint i;
	gboolean my_group_removed;
	const GPtrArray * groups;

	/* setup */	
	groups = (const GPtrArray*) groups_array;

	/* was our group one of the ones removed? */
	my_group_removed = FALSE;
	for (i=0; !my_group_removed && i<groups->len; ++i)
		if (GROUP(g_ptr_array_index(groups,i)) == my_group)
			my_group_removed = TRUE;

	/* if so, clear */
	if (my_group_removed)
		gui_queue_add ((GSourceFunc)articlelist_set_group, NULL);
}

static void
article_toolbar_user_changed_filter_cb (gpointer event, gpointer foo, gpointer bar)
{
	refresh_impl (REFRESH_FILTER);
}

static void
articlelist_articles_changed_cb (gpointer event, gpointer foo, gpointer bar)
{
	const ArticleChangeEvent * e = (ArticleChangeEvent*) event;
	switch (e->type)
	{
		case ARTICLE_CHANGED_READ:
			articlelist_articles_read_changed ((const Article**)e->articles, e->article_qty);
			break;
		case ARTICLE_CHANGED_NEW:
			articlelist_articles_new_changed ((const Article**)e->articles, e->article_qty);
			break;
		case ARTICLE_CHANGED_DIRTY:
			articlelist_update_articles ((const Article**)e->articles, e->article_qty);
			break;
		default:
			pan_warn_if_reached ();
			break;
	}
}

/***
****
***/

static void
article_ctree_destroy_cb (void)
{
	int i;

	/* stop listening to events */
	pan_callback_remove (flagset_get_added_cb(),
	                     refresh_articles_from_message_id_array_cb, NULL);
	pan_callback_remove (flagset_get_removed_cb(),
	                     refresh_articles_from_message_id_array_cb, NULL);
	pan_callback_remove (queue_get_message_id_status_changed(),
	                     refresh_articles_from_message_id_array_cb, NULL);
	pan_callback_remove (article_toolbar_get_user_changed_filter_callback(),
	                     article_toolbar_user_changed_filter_cb, NULL);
	pan_callback_remove (article_get_articles_changed_callback(),
	                     articlelist_articles_changed_cb, NULL);
	pan_callback_remove (server_get_groups_removed_callback(),
	                     articlelist_groups_removed_cb, NULL);
	pan_callback_remove (acache_bodies_added_callback,
	                     articlelist_acache_changed_cb, NULL);
	pan_callback_remove (acache_bodies_removed_callback,
	                     articlelist_acache_changed_cb, NULL);
	pan_callback_remove (articlelist_selection_changed,
	                     articlelist_selection_changed_cb, NULL);

	/* delete the icons */
	for (i=0; i<ICON_QTY; ++i) {
		g_object_unref (_icons[i].pixmap); _icons[i].pixmap = NULL;
		g_object_unref (_icons[i].mask); _icons[i].mask = NULL;
		g_object_unref (_icons[i].pixbuf); _icons[i].pixbuf = NULL;
	}

	pan_callback_free (articlelist_thread_changed);
	pan_callback_free (articlelist_selection_changed);
}

static void
tree_toggle_focus_row_cb (GtkCList   * clist,
                          gpointer     user_data)
{
	/* This is just a hack to allow space-reading when the article ctree
	   has focus.  The space key is used in clists to toggle the row
	   focus, so if we don't stop the signal here ctree calls
	   resync_selection, which reselects the focused node, and makes
	   space reading loop back on the same article each time. */
	gtk_signal_emit_stop_by_name (GTK_OBJECT(clist), "toggle_focus_row");
}


/*--------------------------------------------------------------------
 * generate the listing of articles, for the "Articles" tab
 *--------------------------------------------------------------------*/
GtkWidget*
create_articlelist_ctree (void)
{
	int i;
	GtkWidget * w;
	GtkCList * list;
	GtkCTree * tree;
	GtkWidget * s_window;
	GtkWidget * vbox;
	const char * titles[COLUMNS];

	cur_message_id = g_string_new (NULL);
	prev_message_id = g_string_new (NULL);
	my_articles = g_ptr_array_new ();

	/* callbacks */
	articlelist_thread_changed = pan_callback_new ();
	articlelist_selection_changed = pan_callback_new ();

	pan_callback_add (flagset_get_added_cb(),
	                  refresh_articles_from_message_id_array_cb, NULL);
	pan_callback_add (flagset_get_removed_cb(),
	                  refresh_articles_from_message_id_array_cb, NULL);
	pan_callback_add (queue_get_message_id_status_changed(),
	                  refresh_articles_from_message_id_array_cb, NULL);
	pan_callback_add (article_toolbar_get_user_changed_filter_callback(),
	                  article_toolbar_user_changed_filter_cb, NULL);
	pan_callback_add (article_get_articles_changed_callback(),
	                  articlelist_articles_changed_cb, NULL);
	pan_callback_add (server_get_groups_removed_callback(),
	                  articlelist_groups_removed_cb, NULL);
	pan_callback_add (acache_bodies_added_callback,
	                  articlelist_acache_changed_cb, NULL);
	pan_callback_add (acache_bodies_removed_callback,
	                  articlelist_acache_changed_cb, NULL);
	pan_callback_add (articlelist_selection_changed,
	                  articlelist_selection_changed_cb, NULL);

	vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_SMALL);

	/* filter */
	gtk_box_pack_start (GTK_BOX(vbox), article_toolbar_new(), FALSE, FALSE, 0);

	/* get the titles */
	for (i=0; i<COLUMNS; ++i)
		titles[i] = column_to_title (i);

	for (i=0; i<ICON_QTY; ++i) {
	        _icons[i].pixbuf = gdk_pixbuf_new_from_inline (-1, _icons[i].pixbuf_txt, FALSE, NULL);
		gdk_pixbuf_render_pixmap_and_mask_for_colormap (_icons[i].pixbuf, cmap, 
		                                                &_icons[i].pixmap,
		                                                &_icons[i].mask,
		                                                128);
	}

	clear_hash_table ();

	/* create the widget */
	Pan.article_ctree = w = gtk_ctree_new_with_titles (COLUMNS, 2, (char**)titles);
	tree = GTK_CTREE(w);
	list = GTK_CLIST(w);
	list->button_actions[1] = list->button_actions[0];
	g_signal_connect (GTK_OBJECT(w), "tree_expand", G_CALLBACK(tree_expand_cb), NULL);
	g_signal_connect (GTK_OBJECT(w), "tree_collapse", G_CALLBACK(tree_collapse_cb), NULL);
	g_signal_connect (GTK_OBJECT(w), "tree-select-row", G_CALLBACK(tree_select_row_cb), NULL);
	g_signal_connect (GTK_OBJECT(w), "tree-unselect-row", G_CALLBACK(tree_select_row_cb), NULL);
	g_signal_connect (GTK_OBJECT(w), "toggle_focus_row", G_CALLBACK(tree_toggle_focus_row_cb), NULL);

	/* fonts */
	if (use_custom_fonts) {
		normal_pfd = pango_font_description_from_string (header_pane_font);
		new_replies_pfd = pango_font_description_copy (normal_pfd);
	} else {
		GtkStyle * style = gtk_widget_get_default_style ();
		normal_pfd = pango_font_description_copy (style->font_desc);
		new_replies_pfd = pango_font_description_copy (style->font_desc);
	}
	pango_font_description_set_weight (new_replies_pfd, PANGO_WEIGHT_BOLD);
	gtk_widget_modify_font (w, new_replies_pfd);

	/* wrap it in a scrolled window */
	s_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW(s_window),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(s_window), Pan.article_ctree);

	/* ui settings */
	gtk_ctree_set_line_style (tree, GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_selection_mode (list, GTK_SELECTION_MULTIPLE);
	gtk_clist_set_column_justification (list, 3, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_width (list, 0, 16);
	gtk_clist_set_column_width (list, 1, 16);
	gtk_clist_set_column_width (list, 2, 220);
	gtk_clist_set_column_width (list, 3, 30);
	gtk_clist_set_column_width (list, 4, 120);
	gtk_clist_set_column_width (list, 5, 40);

	/* create the right click popup menu */
	threads_popup_menu = menu_create_items (threads_popup_entries,
	                                        threads_popup_entries_qty,
	                                        "<ThreadView>",
	                                        &threads_popup_factory,
	                                        NULL);

	/* connect signals */
	g_signal_connect (tree, "button_press_event",
	                  G_CALLBACK(articlelist_button_press), NULL);
	g_signal_connect (tree, "key_press_event",
	                  G_CALLBACK(articlelist_key_press), NULL);
	g_signal_connect (tree, "key_release_event",
	                  G_CALLBACK(articlelist_key_release), NULL);
	g_signal_connect (tree, "click_column",
	                  G_CALLBACK(articlelist_click_column), NULL);
	g_signal_connect (tree, "destroy",
	                  G_CALLBACK(article_ctree_destroy_cb), NULL);

	pan_callback_add (group_get_articles_removed_callback(),
			  group_articles_removed_cb,
			  NULL);

	pan_callback_add (group_get_articles_added_callback(),
			  group_articles_added_cb,
			  NULL);

	gtk_box_pack_start (GTK_BOX(vbox), s_window, TRUE, TRUE, 0);

	return vbox;
}

void
articlelist_reset_style_nolock (void)
{
	pan_lock ();
	if (use_custom_fonts) {
		normal_pfd = pango_font_description_from_string (header_pane_font);
		new_replies_pfd = pango_font_description_copy (normal_pfd);
	} else {
		GtkStyle * style = gtk_widget_get_default_style ();
		normal_pfd = pango_font_description_copy (style->font_desc);
		new_replies_pfd = pango_font_description_copy (style->font_desc);
	}
	pango_font_description_set_weight (new_replies_pfd, PANGO_WEIGHT_BOLD);
	gtk_widget_modify_font (Pan.article_ctree, new_replies_pfd);
	pan_unlock ();

	rebuild_style = TRUE;
	articlelist_refresh ();
}

/**
***
***  ARTICLE THREADING
***
**/


void
articlelist_set_threaded (gboolean threaded_on)
{
	if (threaded_on != header_pane_is_threaded)
	{
		header_pane_is_threaded = threaded_on;
		refresh_impl (0);
	}
}

/****
*****
*****  COPY TO FOLDER
*****
****/

static void
copy_to_folder_cb (gpointer user_data, int index, GtkWidget * w)
{
	Server * server;
	GPtrArray * folders;
	GPtrArray * articles;
	Group * folder;

	/* get the target folder */
	server = serverlist_get_named_server (INTERNAL_SERVER_NAME);
	folders = server_get_groups (server, SERVER_GROUPS_ALL);
	folder = GROUP (g_ptr_array_index (folders, index));

	/* get selected articles */
       	articles = articlelist_get_selected_articles_nolock ();

	/* copy them */
	article_copy_articles_to_folder (folder,
	                                 (const Article**)articles->pdata,
	                                 articles->len);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	g_ptr_array_free (folders, TRUE);
}

static void
refresh_folder_popup_menu (void)
{
	int i;
	Server * server;
	GPtrArray * folders;
	static GtkItemFactoryEntry * entries = NULL;
	static int qty = 0;
	debug_enter ("refresh_folder_popup_menu");

	/* remove any old server buttons */
	if (entries != NULL) {
		gtk_item_factory_delete_entries (threads_popup_factory, qty, entries);
		for (i=0; i<qty; ++i)
			g_free (entries[i].path);
		g_free (entries);
		entries = NULL;
		qty = 0;
	}

	/* get a list of folders */
	server = serverlist_get_named_server (INTERNAL_SERVER_NAME);
	folders = server_get_groups (server, SERVER_GROUPS_ALL);

	/* build the ItemFactoryEntries */
	entries = g_new0 (GtkItemFactoryEntry, folders->len);
	for (i=qty=0; i<folders->len; ++i)
	{
		const Group * folder = GROUP(g_ptr_array_index(folders,i));
		const char * name = folder->name;

		/* build an entry: the callback arg is the index into the folders array */
		entries[qty].path = g_strdup_printf ("/_Copy to Folder/%s", name);
		entries[qty].item_type = NULL;
		entries[qty].callback = copy_to_folder_cb;
		entries[qty].callback_action = i;

		++qty;
	}

	/* add the servers */
	gtk_item_factory_create_items (threads_popup_factory, qty, entries, NULL);

	/* cleanup */
	g_ptr_array_free (folders, TRUE);

	debug_exit ("refresh_folder_popup_menu");
}

/****
*****
*****   POPUP MENU
*****
****/

static void
articlelist_menu_popup_nolock (GdkEventButton* bevent)
{
	GtkItemFactory * gif = threads_popup_factory;
	/*static gboolean _folders_dirty = TRUE;*/
	const Article * article = articlelist_get_selected_article_nolock ();
	const gboolean have_article = article!=NULL;

	/*if (_folders_dirty) {
		_folders_dirty = FALSE;*/
		refresh_folder_popup_menu ();
	/*}*/

	menu_set_sensitive (gif, "/Read Article", have_article);
	menu_set_sensitive (gif, "/Save Article As...", have_article);
	menu_set_sensitive (gif, "/Save Article Attachments", have_article);
	menu_set_sensitive (gif, "/Download Flagged", have_article);
	menu_set_sensitive (gif, "/Flag", have_article);
	menu_set_sensitive (gif, "/Unflag", have_article);
	menu_set_sensitive (gif, "/Watch Thread", have_article);
	menu_set_sensitive (gif, "/Ignore Thread", have_article);
	menu_set_sensitive (gif, "/Add to Bozo or Spam Filter...", have_article);
	menu_set_sensitive (gif, "/Copy to Folder", have_article);
	menu_set_sensitive (gif, "/Delete", have_article);

	gtk_menu_popup (GTK_MENU (threads_popup_menu),
			NULL, NULL, NULL, NULL,
			bevent->button, bevent->time);
}

static int threads_popup_entries_qty = 16;
static GtkItemFactoryEntry threads_popup_entries[] =
{
	/* r */ {N_("/_Read Article"), NULL, articlelist_read_selected, 0, NULL},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* s */ {N_("/_Save Article As..."), NULL, article_action_selected_save_as, 0, NULL},
	/* t */ {N_("/Save Article A_ttachments"), NULL, article_action_selected_save_attachments, 0, NULL},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* o */ {N_("/D_ownload Flagged"), NULL, flagset_flush, 0, NULL},
	/* f */ {N_("/_Flag"), NULL, articlelist_selected_flag_for_dl_nolock, 0, NULL},
	/* u */ {N_("/_Unflag"), NULL, articlelist_selected_unflag_for_dl_nolock, 0, NULL},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* w */ {N_("/_Watch Thread"), NULL, articlelist_selected_thread_watch_nolock, 0, NULL},
	/* i */ {N_("/_Ignore Thread"), NULL, articlelist_selected_thread_ignore_nolock, 0, NULL},
	/* z */ {N_("/Add to Bo_zo or Spam Filter..."), NULL, articlelist_spamize_nolock, 0, NULL},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* c */ {N_("/_Copy to Folder"), NULL, NULL, 0, "<Branch>"},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* d */ {N_("/_Delete"), NULL, article_action_delete_selected_articles, 0, NULL}
};

/***
****  Events
***/

PanCallback*
articlelist_get_group_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static int
fire_group_changed (gpointer group_gp)
{
	Group * group = GROUP(group_gp);

	pan_callback_call (articlelist_get_group_changed_callback(),
	                   Pan.article_ctree,
	                   group);
	return 0;
}

/***
****
****  REFRESH
****
***/

static int
refresh_mainthread_end (gpointer p)
{
	ArgSet * argset;
	Group * group;
	RefreshActions actions;
	StatusItem * status;
	GPtrArray * articles;
	GPtrArray * tmp;
	debug_enter ("refresh_mainthread_end");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	actions = GPOINTER_TO_INT (argset_get (argset, 0));
	group = (Group*) argset_get (argset, 1);
	articles = (GPtrArray*) argset_get (argset, 2);
	status = (StatusItem*) argset_get (argset, 3);

	/* update the UI */
	pan_lock ();
	tmp = articlelist_get_selected_articles_nolock ();
	articlelist_repopulate_nolock (group, (const Article**)articles->pdata, articles->len, tmp);
	g_ptr_array_free (tmp, TRUE);
	pan_g_ptr_array_assign (my_articles, articles->pdata, articles->len);
	pan_unlock ();

	/* cleanup - unref the safe-keeping refs */
	group_unref_articles (group, NULL);

	/* cleanup - done with status-item */
	status_item_set_active (status, FALSE);
	pan_object_unref (PAN_OBJECT(status));
	
	/* cleanup - free the tmp stuff */
	g_ptr_array_free (articles, TRUE);
	argset_free (argset);

	debug_exit ("refresh_mainthread_end");
	return 0;
}

static void*
refresh_worker (void * p)
{
	ArgSet         * argset;
	Group          * group;
	RefreshActions   actions;
	StatusItem     * status;
	GPtrArray      * articles;
	int              old_sort_style;
	int              new_sort_style;
	debug_enter ("refresh_worker");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	actions = GPOINTER_TO_INT (argset_get (argset, 0));
	group = (Group*) argset_get (argset, 1);
	articles = (GPtrArray*) argset_get (argset, 2);
	status = (StatusItem*) argset_get (argset, 3);
	old_sort_style = group->old_sort_style;
	new_sort_style = group->new_sort_style;
	if (!new_sort_style || abs(new_sort_style)>ARTICLE_SORT_TYPES_QTY)
		new_sort_style = sort_type;

	/* refresh the articles */
	if (actions & REFRESH_RELOAD) {
		GPtrArray * tmp;
		char buf[512];
		g_snprintf (buf, sizeof(buf), _("Loading group \"%s\""), group_get_name(group));
		status_item_emit_status (status, buf);
		tmp = group_get_article_array (group);
		pan_g_ptr_array_assign (articles, tmp->pdata, tmp->len);
		g_ptr_array_free (tmp, TRUE);
	}
	if (actions & REFRESH_SORT) {
		status_item_emit_status (status, _("Sorting Articles"));
		if (old_sort_style && abs(new_sort_style) != abs(old_sort_style)) 
		{
			sort_articles ((Article**)articles->pdata, 
				articles->len, abs(old_sort_style), 
				old_sort_style>0);
		}
		sort_articles ((Article**)articles->pdata, articles->len, 
			abs(new_sort_style), new_sort_style>0);
	}
	if (actions & REFRESH_FILTER) {
		Filter * filter = NULL;
		FilterShow show = 0;
		article_toolbar_get_filter (&filter, &show);
		status_item_emit_status (status, _("Filtering Articles"));
		apply_filter_tests (filter, show, articles);
		pan_object_unref (PAN_OBJECT(filter));
		article_toolbar_set_group_filter (group);
	}
	status_item_emit_status (status, _("Updating Header Pane..."));
	g_usleep (75000);

	/* hard work done; back to the main thread */
	gui_queue_add (refresh_mainthread_end, argset);

	debug_exit ("refresh_worker");
	return NULL;
}

static int
refresh_mainthread_begin (gpointer p)
{
	Group * group;
	ArgSet * argset;
	debug_enter ("refresh_mainthread_begin");

	/* pump out the arguments */
	argset = (ArgSet*) p;

	group = my_group;
	if (group != NULL)
	{
		char buf[512];
		GPtrArray * articles;
		StatusItem * status;

		argset_add (argset, group);

		/* ref the articles for safe-keeping */
		group_ref_articles (group, NULL);
		articles = pan_g_ptr_array_dup (my_articles);
		argset_add (argset, articles);

		/* create the status-item */
		g_snprintf (buf, sizeof(buf), _("Refreshing Group \"%s\""), group_get_name(group));
		status = status_item_new_with_description (buf);
		status_item_set_active (status, TRUE);
		argset_add (argset, status);

		/* pass the hard work to another thread */
		run_in_worker_thread (refresh_worker, argset);
	}
	else
	{
		argset_free (argset);
	}

	debug_exit ("refresh_mainthread_begin");
	return 0;
}

static void
refresh_impl (RefreshActions actions)
{
	gui_queue_add (refresh_mainthread_begin, argset_new1(GINT_TO_POINTER(actions)));
}

void
articlelist_refresh (void)
{
	refresh_impl (~0);
}


/***
****
****  SET THE GROUP
****
***/

static int
clear_group (gpointer unused)
{
	/* clear the UI */
	gtk_clist_clear (GTK_CLIST(Pan.article_ctree));
	clear_hash_table ();

	/* clear the model */
	if (my_group != NULL)
	{
		file_headers_save (my_group, NULL);
		server_save_grouplist_if_dirty (my_group->server, NULL);
		group_unref_articles (my_group, NULL);
		g_ptr_array_set_size (my_articles, 0);
		my_group = NULL;
		g_string_truncate (prev_message_id, 0);
		g_string_truncate (cur_message_id, 0);
		gui_queue_add (fire_group_changed, my_group);
	}

	return 0;
}

static int
set_group_mainthread_end (gpointer p)
{
	ArgSet * argset;
	Group * group;
	Group * old_group;
	StatusItem * status;
	GPtrArray * articles;
	GPtrArray * tmp;
	int sort;
	debug_enter ("set_group_mainthread_end");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);
	status = (StatusItem*) argset_get (argset, 1);
	articles = (GPtrArray*) argset_get (argset, 2);
	sort = GPOINTER_TO_INT (argset_get (argset, 3));

	/* update the UI */
	pan_lock ();
	articlelist_set_sort_bits_nolock (sort);
	tmp = articlelist_get_selected_articles_nolock ();
	articlelist_repopulate_nolock (group, (const Article**)articles->pdata, articles->len, tmp);
	g_ptr_array_free (tmp, TRUE);
	pan_unlock ();

	/* update the model */
	old_group = my_group;
	my_group = group;
	g_string_truncate (prev_message_id, 0);
	g_string_truncate (cur_message_id, 0);
	pan_g_ptr_array_assign (my_articles, articles->pdata, articles->len);
	if (old_group != NULL) {
		file_headers_save (old_group, NULL);
		server_save_grouplist_if_dirty (old_group->server, NULL);
		group_unref_articles (old_group, NULL);
	}
	my_group->loaded_since_last_fetch = TRUE;
	fire_group_changed (my_group);

	/* no longer using the status */
	status_item_set_active (status, FALSE);
	pan_object_unref (PAN_OBJECT(status));

	/* cleanup memory */
	g_ptr_array_free (articles, TRUE);
	argset_free (argset);

	/* Now that the articlelist is fully loaded,
	   get new headers, if required. */
	if (fetch_new_on_group_load && !group_is_folder(group))
		queue_insert_tasks (g_slist_append(NULL,task_headers_new(group,HEADERS_NEW)), 0);

	debug_exit ("set_group_mainthread_end");
	return 0;
}


static void*
set_group_worker (void * p)
{
	ArgSet * argset;
	Group * group;
	StatusItem * status;
	GPtrArray * articles;
	int old_sort_style, new_sort_style;
	debug_enter ("set_group_worker");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);
	status = (StatusItem*) argset_get (argset, 1);

	/* we're switching. all other switch requests: please hold */
	g_static_mutex_lock (&switch_mutex);

	/* get the articles */
	group_ref_articles (group, status);
	articles = group_get_article_array (group);
	argset_add (argset, articles);

	/* get sorting information */
	old_sort_style = group->old_sort_style;
	new_sort_style = group->new_sort_style;
	if (!new_sort_style || abs(new_sort_style)>ARTICLE_SORT_TYPES_QTY)
		new_sort_style = -ARTICLE_SORT_DATE;
	argset_add (argset, GINT_TO_POINTER(new_sort_style));

	/* massage them into shape */
	status_item_emit_status (status, _("Sorting Articles"));
	if (old_sort_style && abs(old_sort_style) != abs(new_sort_style))
	{
		sort_articles ((Article**)articles->pdata, articles->len, 
			abs(old_sort_style), old_sort_style>0);
	}
	sort_articles ((Article**)articles->pdata,
	               articles->len,
	               abs(new_sort_style),
	               new_sort_style>0);
	if (1) {
		Filter * filter = NULL;
		FilterShow show = 0;
		status_item_emit_status (status, _("Filtering Articles"));
		article_toolbar_set_group (group);
		article_toolbar_get_filter (&filter, &show);
		apply_filter_tests (filter, show, articles);
		pan_object_unref (PAN_OBJECT(filter));
	}
	status_item_emit_status (status, _("Updating Header Pane..."));
	g_usleep (75000);

	/* hard work over, pass back to main thread */
	gui_queue_add (set_group_mainthread_end, argset);

	/* safe to do another switch */
	g_static_mutex_unlock (&switch_mutex);

	debug_exit ("set_group_worker");
	return NULL;
}

static int
set_group_mainthread_begin (gpointer p)
{
	char buf[512];
	ArgSet * argset;
	Group * group;
	StatusItem * status;
	debug_enter ("set_group_mainthread_begin");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);

	/* create the status-item */
	g_snprintf (buf, sizeof(buf), _("Loading group \"%s\""), group_get_name(group));
	status = status_item_new_with_description (buf);
	status_item_set_active (status, TRUE);
	argset_add (argset, status);

	/* pass the hard work to another thread */
	run_in_worker_thread (set_group_worker, argset);
	debug_exit ("set_group_mainthread_begin");
	return 0;
}

static int
open_download_dialog (gpointer data)
{
	/* This is a bit ugly */
	group_action_selected_download_dialog ();
	return 0;
}

void
articlelist_set_group (Group * group)
{
	if (group == NULL)
	{
		gui_queue_add (clear_group, NULL);
	}
	else if (!group_is_folder(group)
	         && !group->article_qty
	         && !group->article_high)
	{
		gui_queue_add (open_download_dialog, NULL);
	}
	else
	{
		gui_queue_add (set_group_mainthread_begin, argset_new1(group));
	}
}
