/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include <gmime/gmime.h>
#include <gmime/memchunk.h>

#include <pan/base/acache.h>
#include <pan/base/argset.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/message-identifier.h>
#include <pan/base/serverlist.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>
#include <pan/base/util-mime.h>
#include <pan/base/log.h>

PanCallback* acache_bodies_added_callback = NULL;
PanCallback* acache_bodies_removed_callback = NULL;

static GStaticMutex entries_mutex = G_STATIC_MUTEX_INIT;
static MemChunk * acache_entry_chunk = NULL;
static GHashTable * messageid_to_entry = NULL;
static GHashTable * key_to_path = NULL;

/**
 * This represents one directory (folder) in the cache
 */
typedef struct
{
	char * key;
	char * path;
	gssize size;
}
AcachePath;

/**
 * This represents one file (message) in the cache
 */
typedef struct
{
	AcachePath * path;
	int refcount;
	char * message_id;
	size_t size;
	time_t date;
}
AcacheEntry;

/**
 * This is the name of the default directory
 */
const char * ACACHE_DEFAULT_KEY = "cache";

#define ACACHE_ENTRY_CHUNK_SIZE 256

/***
****
***/

static void
acache_path_free (AcachePath * path)
{
	g_free (path->key);
	g_free (path->path);
	g_free (path);
}

static gboolean
get_message_id_from_file (char * msgid, int msgid_len, const char * filename)
{
	char buf [2048];
	char * retval;
	FILE * fp;
	debug_enter ("get_message_id_from_file");

	g_return_val_if_fail (is_nonempty_string(filename), FALSE);
	g_return_val_if_fail (msgid!=NULL, FALSE);
	g_return_val_if_fail (msgid_len!=0, FALSE);

	/* try to find the line */
	*buf = '\0';
	fp = fopen (filename, "r");
	if (fp != NULL) {
		const char * key = "Message-I";
		const size_t key_len = 9;
		gboolean found;
		for (found=FALSE; !found && fgets(buf,sizeof(buf),fp); ) {
			if (buf[0]=='\0' || buf[1]=='\0') /* watch for header/body delimiter */
				break;
			if (!memcmp (key, buf, key_len))
				found = TRUE;
		}
		if (!found)
			*buf = '\0';
		fclose (fp);
	}

	/* parse the line */
	retval = NULL;
	if (*buf != '\0') {
		char * pch = strchr (buf, ':');
		if (pch != NULL) {
			++pch;
			while (*pch && isspace((guchar)*pch))
				++pch;
			retval = pch;
			while (*pch && !isspace((guchar)*pch))
				++pch;
			*pch = '\0';
		}
	}

	if (retval != NULL) 
		g_snprintf (msgid, msgid_len, "%s", retval);

	debug_exit ("get_message_id_from_file");
	return retval != NULL;
}

/***
****
****  PRIVATE UTIL
****
***/

static char*
acache_get_message_base (void)
{
	return g_build_filename (get_data_dir(), "messages", NULL);
}
static char*
acache_get_folder_base (void)
{
	char * base = acache_get_message_base ();
	char * retval = g_build_filename (base, "folders", NULL);
	g_free (base);
	return retval;
}

static char*
acache_get_filename (char * buf, int len, const char * path, const char * message_id)
{
	register char * pch;
	const register char * cpch;
	const int path_len = strlen (path);
	debug_enter ("acache_get_filename");

	/* sanity clause */
	g_return_val_if_fail (buf!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);
	g_return_val_if_fail (len >= (path_len + 1 + strlen(message_id) + 4), NULL);

	/* build acache string.  path/msg-id.msg */
	pch = g_stpcpy (buf, path);
	*pch++ = G_DIR_SEPARATOR;
	for (cpch=message_id; *cpch!='\0'; ++cpch)
		if (*cpch!='%' && *cpch!='$' && *cpch!='<' && *cpch!='>' && *cpch!='/' && *cpch!='\\')
			*pch++ = *cpch;
	strcpy (pch, ".msg");
	pan_file_normalize_inplace (buf);

	debug_exit ("acache_get_filename");
	return is_nonempty_string(buf) ? buf : NULL;
}

static gpointer
add_folder_directories (gpointer data)
{
	Server * server = (Server *) data;
	char * message_base;
	char * folder_base;
	char * path;
	GDir * dir;
	GError * err;
	debug_enter ("add_folder_directories");

	/* ensure the default directories exist */
	message_base = acache_get_message_base ();
	path = g_build_filename (message_base, ACACHE_DEFAULT_KEY, NULL);
	acache_add_path (ACACHE_DEFAULT_KEY, path);
	replace_gstr (&path, NULL);

	/* ensure pan.sent exists */
	folder_base = acache_get_folder_base ();
	path = g_build_filename (folder_base, PAN_SENT, NULL);
	pan_file_ensure_path_exists (path);
	replace_gstr (&path, NULL);

	/* ensure pan.sendlater exists */
	path = g_build_filename (folder_base, PAN_SENDLATER, NULL);
	pan_file_ensure_path_exists (path);
	replace_gstr (&path, NULL);

	/* add all the folder directories */
	err = NULL;
	dir = g_dir_open (folder_base, 0, &err);
	if (err != NULL)
	{
		log_add_va (LOG_ERROR, _("Error opening directory \"%s\": %s)"), folder_base, err->message);
		g_error_free (err);
	}
	else
	{
		const char * key;
		GPtrArray * folders = g_ptr_array_new ();

		/* add the folders */
		while ((key = g_dir_read_name (dir)))
		{
			/* add the folder */
			Group * folder = group_new (server, key);
			group_set_is_folder (folder, TRUE);
			g_ptr_array_add (folders, folder);
			acache_add_folder (key);
		}
		g_dir_close (dir);

		if (folders->len)
			server_add_groups (server, (Group**)folders->pdata, folders->len, NULL, NULL);

		g_ptr_array_free (folders, TRUE);
	}

	/* cleanup */
	g_free (folder_base);
	g_free (message_base);
	debug_exit ("add_folder_directories");
	return NULL;
}

void
acache_init (void)
{
	Server *server;

	/* sanity clause -- don't init twice */
	if (acache_entry_chunk != NULL)
		return;

	/* init the callbacks */
	acache_bodies_added_callback = pan_callback_new ();
	acache_bodies_removed_callback = pan_callback_new ();

	/* init the message id hash  */
	acache_entry_chunk = memchunk_new (sizeof(AcacheEntry), ACACHE_ENTRY_CHUNK_SIZE, FALSE);
	messageid_to_entry = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
	key_to_path = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, (GDestroyNotify)acache_path_free);

	/* add the folder server */
	server = server_new ();
	server->address = g_strdup ("Mock Server for Folders");
	server->name = g_strdup (INTERNAL_SERVER_NAME);
	serverlist_add_server (server);

	/* add all the folder directories */
	run_in_worker_thread (add_folder_directories, server);
}

void
acache_shutdown (void)
{
	if (messageid_to_entry) {
		if (acache_flush_on_exit)
			acache_expire_all ();
		g_hash_table_destroy (messageid_to_entry);
		messageid_to_entry = NULL;
	}

	if (key_to_path) {
		g_hash_table_destroy (key_to_path);
		key_to_path = NULL;
	}

	if (acache_entry_chunk) {
		memchunk_destroy (acache_entry_chunk);
		acache_entry_chunk = NULL;
	}
}

/***
****
****  EXPIRE
****
***/

/**
 * This must be called from inside an entries_mutex lock.
 */
static void
acache_remove_entry_from_path (AcacheEntry * entry)
{
	char filename[PATH_MAX] = { '\0' };

	g_return_if_fail (entry!=NULL);
	g_return_if_fail (entry->path!=NULL);

	/* update the acache size */
	entry->path->size -= entry->size;

	/* remove the file */
	acache_get_filename (filename, sizeof(filename), entry->path->path, entry->message_id);
	unlink (filename);
}

/**
 * This must be called from inside an entries_mutex lock.
 */
static void
acache_remove_entry (AcacheEntry * entry)
{
	/* sanity clause */
	g_return_if_fail (entry!=NULL);

	/* let everyone know */
	debug2 (DEBUG_ACACHE, "Removing %s from cache with refcount %d", entry->message_id, entry->refcount);
	pan_callback_call (acache_bodies_removed_callback, (gpointer)entry->message_id, NULL);

	/* if this entry had a file associated with it ... */
	if (entry->path)
		acache_remove_entry_from_path (entry);

	/* update acache tables */
	g_hash_table_remove (messageid_to_entry, entry->message_id);
	memchunk_free (acache_entry_chunk, entry);
}

static int
compare_ppEntry_ppEntry_by_youth (gconstpointer a, gconstpointer b, gpointer unused)
{
	const time_t atime = (**(AcacheEntry**)a).date;
	const time_t btime = (**(AcacheEntry**)b).date;
	return (int) -difftime (atime, btime);
}

void
acache_expire_messages (const char   ** message_ids,
	                int             message_id_qty)
{
	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty>0);

	g_static_mutex_lock (&entries_mutex);
	{
		int i;

		for (i=0; i<message_id_qty; ++i)
		{
			AcacheEntry * entry;
			const char * message_id;

			/* find the entry */
			message_id = message_ids[i];
			entry = (AcacheEntry*) g_hash_table_lookup (messageid_to_entry, message_id);

			/* if we have such an entry, remove it */
			if (entry != NULL)
				acache_remove_entry (entry);
		}

		memchunk_clean (acache_entry_chunk);
	}
	g_static_mutex_unlock (&entries_mutex);
}


static int
acache_expire_to_size (double max_megs)
{
	int files_removed = 0;
	const size_t cache_max = (size_t) max_megs * (1024 * 1024);
	AcachePath * apath;
	debug_enter ("acache_expire_to_size");

	/* get the scratch directory path */
	apath = (AcachePath*) g_hash_table_lookup (key_to_path, ACACHE_DEFAULT_KEY);
	g_return_val_if_fail (apath!=NULL, 0);

	g_static_mutex_lock (&entries_mutex);
	{
		debug2 (DEBUG_ACACHE, "expiring to %lu; current size is %lu", cache_max, apath->size);

		if (cache_max < apath->size)
		{
			int i;
			GPtrArray * entries;

			/* get an array of files sorted from oldest to youngest*/
			entries = g_ptr_array_new ();
			pan_hash_to_ptr_array (messageid_to_entry, entries);
			g_ptr_array_sort_with_data (entries, compare_ppEntry_ppEntry_by_youth, NULL);

			/* Start blowing away files */
			for (i=entries->len; i>0 && cache_max<apath->size; --i) {
				AcacheEntry * entry = (AcacheEntry*) g_ptr_array_index (entries, i-1);
				if (entry->path==apath && entry->refcount<=0) {
					acache_remove_entry (entry);
					++files_removed;
				}
			}

			/*  cleanup */
			g_ptr_array_free (entries, TRUE);
			memchunk_clean (acache_entry_chunk);
		}
	}
	g_static_mutex_unlock (&entries_mutex);

	/* log 
	if (files_removed != 0)
		log_add_va (LOG_INFO, _("Removed %d articles from \"%s\", which now has %.1f MB"),
		                      files_removed,
		                      apath->path,
		                      (double)(apath->size/(1024.0*1024.0)));*/

	/* done */
	debug_exit ("acache_expire_to_size");
	return files_removed;
}

int 
acache_expire (void)
{
	return acache_expire_to_size (acache_max_megs * 0.8);
}

int
acache_expire_all (void)
{
	int retval;
	printf ("Pan is flushing article cache... ");
	retval = acache_expire_to_size(0);
	printf ("%d files erased.\n", retval);
	fflush (NULL);
	return retval;
}

/***
****
****  CHECKIN / CHECKOUT
****
***/

/* this must be called inside an entries_mutex lock */
static void
acache_update_refcount_nolock (MessageIdentifier ** identifiers,
                               int                  identifier_qty,
                               int                  inc)
{
	int i;
	g_return_if_fail (identifiers != NULL);
	g_return_if_fail (identifier_qty >= 1);
	for (i=0; i<identifier_qty; ++i)
		g_return_if_fail (message_identifier_is_valid(identifiers[i]));

	for (i=0; i<identifier_qty; ++i)
	{
		const char * message_id = identifiers[i]->message_id;
		AcacheEntry * entry;

		entry = g_hash_table_lookup (messageid_to_entry, message_id);
		if (entry == NULL)
		{
			entry = (AcacheEntry*) memchunk_alloc (acache_entry_chunk);
			entry->path = NULL;
			entry->refcount = 0;
			entry->message_id = g_strdup (message_id);
			entry->size = 0;
			entry->date = time (NULL);
			g_hash_table_replace (messageid_to_entry, entry->message_id, entry);
			debug1 (DEBUG_ACACHE, "Added new entry %d", entry->message_id);
		}

		/* If we're checking it out, then move it to the safe end of the
		   least-recently-used kill heuristic */
		if (inc > 0)
			entry->date = time (NULL);

		entry->refcount += inc;
		debug3 (DEBUG_ACACHE, "%s refcount - inc by %d to %d", entry->message_id, inc, entry->refcount);
	}
}

void
acache_checkout (MessageIdentifier  ** identifiers,
                 int                   identifier_qty)
{
	g_return_if_fail (identifiers!=NULL);
	g_return_if_fail (identifier_qty >= 1);

	g_static_mutex_lock (&entries_mutex);
	acache_update_refcount_nolock (identifiers, identifier_qty, 1);
	g_static_mutex_unlock (&entries_mutex);
}
void
acache_checkin (MessageIdentifier  ** identifiers,
                int                   identifier_qty)
{
	g_return_if_fail (identifiers!=NULL);
	g_return_if_fail (identifier_qty >= 1);

	g_static_mutex_lock (&entries_mutex);
	acache_update_refcount_nolock (identifiers, identifier_qty, -1);
	g_static_mutex_unlock (&entries_mutex);

	acache_expire ();
}

/***
****
****  ADD FILES
****
***/

void
acache_set_message (const char    * key,
                    const char    * message_id,
                    const char    * message,
                    guint           message_len)
{
       	FILE * fp = NULL;
	char filename[PATH_MAX];
	AcachePath * apath;

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(key));
	g_return_if_fail (is_nonempty_string(message));
	g_return_if_fail (is_nonempty_string(message_id));
	g_return_if_fail (message_len > 0);

	/* get the path */
	apath = (AcachePath*) g_hash_table_lookup (key_to_path, key);
	g_return_if_fail (apath!=NULL);

	/* find out where to write the message */
	if (acache_get_filename (filename, sizeof(filename), apath->path, message_id))
		fp = fopen (filename, "w+");

	if (fp != NULL)
	{
		const size_t bytes_written = fwrite (message, sizeof(char), message_len, fp);
		fclose (fp);

		if (bytes_written < message_len)
		{
			/* couldn't save the whole message */
			char * path = g_path_get_dirname (filename);
			log_add_va (LOG_ERROR, _("Error saving article \"%s\" (is %s full?)"), message_id, path);
			g_free (path);
		}
		else
		{
			AcacheEntry * entry;

			g_static_mutex_lock (&entries_mutex);
			{
				/* if this mesasge existed before and is moving ... */
				entry = g_hash_table_lookup (messageid_to_entry, message_id);
				if (entry!=NULL && entry->path!=NULL && entry->path!=apath)
					acache_remove_entry_from_path (entry);

				/* if this message didn't exist before ... */
				if (entry == NULL) {
					MessageIdentifier * mid = mid = message_identifier_new (message_id);
					acache_update_refcount_nolock (&mid, 1, 0);
					g_object_unref (mid);
					entry = g_hash_table_lookup (messageid_to_entry, message_id);
				}

				g_assert (entry!=NULL);
				entry->size = message_len;
				entry->date = time (NULL);
				entry->path = apath;
			}
			g_static_mutex_unlock (&entries_mutex);

			/* notify everyone that this message has been added */
			pan_callback_call (acache_bodies_added_callback, (gpointer)message_id, NULL);

			/* if the acache is too big, purge the old stuff */
			apath->size += message_len;
			acache_expire ();
		}
	}
}


/***
****
****  GETTING MESSAGES
****
***/

static GMimeStream*
acache_get_message_file_stream (const AcacheEntry * entry)
{
	GMimeStream * retval = NULL;
	char filename[PATH_MAX];

	/* sanity clause */
	g_return_val_if_fail (entry!=NULL, NULL);
	if (entry->path==NULL) return NULL; /* message has entry, but no body */

	/* open the file */
	if (acache_get_filename (filename, sizeof(filename), entry->path->path, entry->message_id))
	{
		FILE * fp;
		errno = 0;
		fp = fopen (filename, "r");
		if (!fp)
			log_add_va (LOG_ERROR, _("Error opening file \"%s\": %s"), filename, g_strerror(errno));
		else
		{
			GMimeStream * file_stream = g_mime_stream_file_new (fp);
			retval = g_mime_stream_buffer_new (file_stream, GMIME_STREAM_BUFFER_BLOCK_READ);
			g_object_unref (file_stream);
		}
	}

	return retval;
}

static GMimeStream*
acache_get_message_mem_stream (const AcacheEntry * entry)
{
	GMimeStream * retval = NULL;
	char filename[PATH_MAX];

	/* sanity clause */
	g_return_val_if_fail (entry!=NULL, NULL);
	if (entry->path==NULL) return NULL; /* message has entry, but no body */

	/* open the file */
	if (acache_get_filename (filename, sizeof(filename), entry->path->path, entry->message_id))
	{
		gsize len = 0;
		char * buf = NULL;
		GError * err = NULL;

		if (g_file_get_contents (filename, &buf, &len, &err)) {
			retval = g_mime_stream_mem_new_with_buffer (buf, len);
			g_free (buf);
		} else {
			log_add_va (LOG_ERROR, _("Error reading file \"%s\": %s"), filename, err->message);
			g_error_free (err);
		}
	}

	return retval;
}

GMimeMessage*
acache_get_message  (MessageIdentifier  ** mids,
                     int                   mid_qty)
{
	int i;
	int stream_qty;
	GMimeStream ** streams;
	GMimeMessage * retval = NULL;

	/* sanity clause */
	g_return_val_if_fail (mid_qty>0, NULL);
	g_return_val_if_fail (mids!=NULL, NULL);
	for (i=0; i<mid_qty; ++i)
		g_return_val_if_fail (message_identifier_is_valid(mids[i]), NULL);

	/* get streams */
	streams = g_alloca (sizeof(GMimeStream*) * mid_qty);
	for (i=stream_qty=0; i<mid_qty; ++i)
	{
		GMimeStream * stream;
		const AcacheEntry * entry = (const AcacheEntry*)
			g_hash_table_lookup (messageid_to_entry, mids[i]->message_id);
		if (entry == NULL)
			break;

		stream = mid_qty>2
			? acache_get_message_file_stream (entry)
			: acache_get_message_mem_stream (entry);
		if (!stream)
			break;

		streams[stream_qty++] = stream;
	}

	/* build the message */
	if (stream_qty == mid_qty)
		retval = pan_g_mime_parser_construct_message (streams, stream_qty);

	/* cleanup */
	for (i=0; i<stream_qty; ++i)
		g_object_unref (streams[i]);

	return retval;
}

gboolean
acache_has_message (const char * message_id)
{
	gboolean retval = FALSE;
	AcacheEntry * entry = NULL;

	g_return_val_if_fail (is_nonempty_string(message_id), FALSE);

	entry = g_hash_table_lookup (messageid_to_entry, message_id);
	if (entry != NULL)
		retval = entry->size != 0;

	return retval;
}

/***
****
****  PATHS
****
***/

typedef struct
{
	AcachePath * path;
	AcachePathForeachFunc func;
	gpointer user_data;
}
ForeachStruct;

static void
acache_path_hash_foreach (gpointer message_id,
                          gpointer entry_gpointer,
                          gpointer user_data)
{
	ForeachStruct * fs = (ForeachStruct*) user_data;
	AcacheEntry * entry = (AcacheEntry*) entry_gpointer;

	if (fs->path == entry->path)
	{
		GMimeStream * stream = acache_get_message_mem_stream (entry);
		GMimeMessage * message = pan_g_mime_parser_construct_message (&stream, 1);

		/* call the user's function */
		(*fs->func)(message, fs->user_data);

		/* cleanup */
		g_object_unref (message);
		g_object_unref (stream);
	}
}

void
acache_path_foreach (const char             * key,
                     AcachePathForeachFunc    func,
                     gpointer                 user_data)
{
	AcachePath * apath;
	ForeachStruct tmp;

	/* get the scratch directory path */
	debug1 (DEBUG_ACACHE, "Calling acache_path_foreach for key \"%s\"", key);
	apath = (AcachePath*) g_hash_table_lookup (key_to_path, key);
	g_return_if_fail (apath!=NULL);

	tmp.path = apath;
	tmp.func = func;
	tmp.user_data = user_data;
	g_hash_table_foreach (messageid_to_entry, acache_path_hash_foreach, &tmp);
}

void
acache_add_folder (const char * key)
{
	char * folder_base = acache_get_folder_base ();
	char * path = g_build_filename (folder_base, key, NULL);

	acache_add_path (key, path);

	g_free (path);
	g_free (folder_base);
}

void
acache_add_path (const char * key,
                 const char * path)
{
	int qty;
	GDir * dir;
	GError * err;
	AcachePath * apath;
	debug_enter ("acache_add_path");

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(key));
	g_return_if_fail (is_nonempty_string(path));
	g_return_if_fail (pan_file_ensure_path_exists(path));

	/* make the path entry */
	debug2 (DEBUG_ACACHE, "Adding path: key \"%s\", path \"%s\"", key, path);
	apath = g_new (AcachePath, 1);
	apath->key = g_strdup (key);
	apath->path = g_strdup (path);
	apath->size = 0;
	g_hash_table_replace (key_to_path, apath->key, apath);

	/* walk through the directory */
	err = NULL;
	dir = g_dir_open (path, 0, &err);
	if (err != NULL)
	{
		log_add_va (LOG_ERROR, _("Error opening directory \"%s\": %s)"), path, err->message);
		g_error_free (err);
	}
	else
	{
		const char * fname;
		char message_id[2048];
		char filename[PATH_MAX];

		qty = 0;
		while ((fname = g_dir_read_name (dir)))
		{
			struct stat stat_p;

			if (!string_ends_with (fname, ".msg"))
				continue;

			g_snprintf (filename, sizeof(filename), "%s%c%s", path, G_DIR_SEPARATOR, fname);
			if (stat (filename, &stat_p) == 0)
			{
				if (get_message_id_from_file (message_id, sizeof(message_id), filename))
				{
					AcacheEntry * entry;

					/* FIXME: if Pan is shut down while 
					 * this thread is still indexing the 
					 * cache, acache_shutdown() will clear
					 * acache_entry_chunk and calling 
					 * memchunk_alloc() will segfault. 
		 			 * This check can be removed once a 
					 * faster cache index can be done in 
					 * the main thread.
					 */
					if (!acache_entry_chunk)
						break;

					entry = (AcacheEntry*) memchunk_alloc (acache_entry_chunk);
					entry->path = apath;
					entry->message_id = g_strdup (message_id);
					entry->size = stat_p.st_size;
					entry->date = stat_p.st_mtime;
					entry->refcount = 0;
					g_hash_table_replace (messageid_to_entry, entry->message_id, entry);

					apath->size += entry->size;

					++qty;
				}
			}
		}

		g_dir_close (dir);
	}

	log_add_va (LOG_INFO, _("Directory \"%s\" contains %.1f MB in %d files"),
		apath->path, (double)apath->size/(1024*1024), qty);

	debug_exit ("acache_add_path");
}
