/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/gnksa.h>
#include <pan/base/group.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
/*#include <pan/base/run.h>*/
#include <pan/base/server.h>
#include <pan/base/group.h>
#include <pan/base/util-mime.h>

const char * default_incoming_name_addr;
const char * default_incoming_name_real;

static void fire_articles_changed (Group*, Article**, int article_qty, ArticleChangeType);

/***
****
****  LIFE CYCLE
****
***/

void
article_destructor (Article * a)
{
        g_return_if_fail (a != NULL);

	/* clean up article parts */
	if (a->threads != NULL) {
		g_slist_free (a->threads);
		a->threads = NULL;
	}
	a->date = 0;
	a->number = 0;
	a->linecount = 0;
	a->message_id = NULL;
	a->xref = NULL;
	a->references = NULL;
	a->subject = NULL;
	a->author_addr = NULL;
	a->author_real = NULL;
	a->group = NULL;
}

static void
article_constructor (Article * article, Group * group)
{
	g_return_if_fail (article!=NULL);
	g_return_if_fail (group!=NULL);

	/* init this class */
	article->group = group;
	article->is_new = FALSE;
	article->self_passes_filter = FALSE;
	article->tree_passes_filter = FALSE;
	article->part = (gint16)0;
	article->parts = (gint16)0;
	article->score = (gint16)0;
	article->linecount = (guint16)0;
	article->state = (guint16)0;
	article->unread_children = (guint16)0;
	article->new_children = (guint16)0;
	article->date = (time_t)0;
	article->number = (gulong)0;
	article->subject = NULL;
	article->author_real = NULL;
	article->author_addr = NULL;
	article->message_id = NULL;
	article->xref = NULL;
	article->references = NULL;
	article->parent = NULL;
	article->threads = NULL;
}

Article*
article_new (Group * group)
{
	Article * article;

	g_return_val_if_fail (group!=NULL, NULL);

	article = group_alloc_new_article (group);
	article_constructor (article, group);
	return article;
}


Article*
article_dup (Group * group, const Article * article)
{
	Article * retval;

	g_return_val_if_fail (article_is_valid(article), NULL);
	g_return_val_if_fail (group!=NULL, NULL);

#define chunk_if_exists(A, share) \
	is_nonempty_string (A) ? group_chunk_string (group, A, share) : NULL

	retval                      = article_new (group);
	retval->self_passes_filter  = FALSE;
	retval->tree_passes_filter  = FALSE;
	retval->unread_children     = 0;
	retval->new_children        = 0;
	retval->group               = group;
	retval->parts               = article->parts;
	retval->part                = article->part;
	retval->linecount           = article->linecount;
	retval->state               = article->state;
	retval->date                = article->date;
	retval->number              = article->number;
	retval->xref                = chunk_if_exists (article->xref, FALSE);
	retval->subject             = chunk_if_exists (article->subject, TRUE);
	retval->author_real         = chunk_if_exists (article->author_real, TRUE);
	retval->author_addr         = chunk_if_exists (article->author_addr, TRUE);
	retval->message_id          = chunk_if_exists (article->message_id, FALSE);
	retval->references          = chunk_if_exists (article->references, TRUE);
	retval->parent              = NULL;
	retval->threads             = NULL;

	return retval;
}

/***
****
***/

static void
articles_set_dirty (Article ** articles, int qty)
{
	Group * group;
	debug_enter ("articles_set_dirty");

	/* sanity check */
	g_return_if_fail (qty >= 1);
	g_return_if_fail (articles_are_valid ((const Article **)articles, qty));

	/* mark the articles' group as read */
	group = articles[0]->group;
	group_set_articles_dirty (group);
	fire_articles_changed (group, articles, qty, ARTICLE_CHANGED_DIRTY);

	debug_exit ("articles_set_dirty");
}


gboolean
article_is_new (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), FALSE);

	return a->is_new && !article_is_read(a);
}

/***
****  NEW / OLD
***/

static void
articles_set_new_impl (Article ** articles, int article_qty, gboolean is_new, gboolean fire)
{
	int i;
	GPtrArray * changed;
	debug_enter ("articles_set_new_impl");

	/* sanity check */
	g_return_if_fail (article_qty > 0);
	g_return_if_fail (articles_are_valid ((const Article **)articles, article_qty));

	/* mark each article's newness field */
	changed = g_ptr_array_new ();
	for (i=0; i<article_qty; ++i) {
		Article * a = articles[i];
		if (!!is_new != !!a->is_new) {
			a->is_new = is_new;
			g_ptr_array_add (changed, a);
		}
	}

	/* update parent counts of changed articles */
	for (i=0; i<changed->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(changed, i));
		for (a=a->parent; a!=NULL; a=a->parent)
			a->new_children += is_new ? 1 : -1;
	}

	/* fire notification of changed articles */
	if (fire && changed->len!=0)
		fire_articles_changed (articles[0]->group, (Article**)changed->pdata, changed->len, ARTICLE_CHANGED_NEW);

	/* cleanup */
	g_ptr_array_free (changed, TRUE);
	debug_exit ("articles_set_new_impl");
}

void
articles_set_new (Article ** articles, int article_qty, gboolean is_new)
{
	articles_set_new_impl (articles, article_qty, is_new, TRUE);
}

/***
****
****  READ / UNREAD
****
***/

gboolean
article_is_read (const Article * a)
{
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->group!=NULL, FALSE);

	return newsrc_is_article_read (group_get_newsrc(a->group), a->number);
}

static void
articles_set_read_numbers_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	int i;
	int changed_qty;
	Group * g = GROUP(key);
	GArray * numbers = (GArray*) value;
	gboolean read = user_data!=NULL;
	Newsrc * newsrc = group_get_newsrc (g);

	/**
	***  Mark each number as read/unread
	**/

	for (i=changed_qty=0; i<numbers->len; ++i)
	{
		const gulong number = g_array_index (numbers, gulong, i);

		if (read != newsrc_mark_article (newsrc, number, read))
			++changed_qty;
	}

	/**
	***  If any changed, update the group's count
	**/

	if (changed_qty != 0)
	{
		group_inc_article_read_qty (g, read ? changed_qty : -changed_qty);
	}

	g_array_free (numbers, TRUE);
}

void
articles_set_read_simple (Article    ** articles,
                          int           article_qty,
                          gboolean      read)
{
	guint i;
	Group * g;
	Newsrc * newsrc;
	GPtrArray * changed;

	/* sanity clause */
	g_return_if_fail (article_qty>0);
	g_return_if_fail (articles_are_valid ((const Article **)articles, article_qty));

	/* prep local variables */
	g = articles[0]->group;
	newsrc = group_get_newsrc (g);
       	changed = g_ptr_array_sized_new (article_qty);

	/* find the articles that will be changed by this action */
	for (i=0; i<article_qty; ++i)
		if (!!read != !!newsrc_mark_article (newsrc, articles[i]->number, read))
			g_ptr_array_add (changed, articles[i]);

	/**
	***  If anything is to be changed:
	***  (1) update parent counts
	***  (2) update group's counts
	***  (3) fire an event
	**/

	if (changed->len != 0)
	{
		/* update parent counts */
		for (i=0; i!=changed->len; ++i)
		{
			Article * a = ARTICLE(g_ptr_array_index(changed,i));
			for (a=a->parent; a!=NULL; a=a->parent)
				a->unread_children += read ? -1 : 1;
		}
		if (read)
			articles_set_new_impl ((Article**)changed->pdata, changed->len, FALSE, FALSE);

		/* update the group's count */
		group_inc_article_read_qty (g, read ? changed->len : -changed->len);

		/* fire an event */
		fire_articles_changed (g, (Article**)changed->pdata, changed->len, ARTICLE_CHANGED_READ);
	}
}

static void
articles_set_read_articles_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	GArray * articles = (GArray*) value;
	const gboolean read = user_data != NULL;
	articles_set_read_simple ((Article**)articles->data, articles->len, read);
	g_array_free (articles, TRUE);
}

/**
 * The hashtable maps from a Group* to a GArray*, which is returned.
 * If no matching GArray* exists, it's created and added to the hash before returning.
 */
static GArray*
get_group_array_from_hash (GHashTable * hash, Group * group, int size)
{
	GArray * arr = (GArray*) g_hash_table_lookup (hash, group);
	if (arr == NULL)
		g_hash_table_insert (hash, group, arr = g_array_new (FALSE, FALSE, size));
	return arr;
}

/**
 * Builds a GHashTable of Group* -> GArray*, where the GArray is filled with message numbers.
 */
static void
mark_read_xreffunc (Server * server, Group * g, gulong number, gpointer user_data)
{
	GHashTable ** group_to_numbers = (GHashTable**) user_data;

	if (group_is_subscribed(g))
	{
		GArray * group_numbers;

		if (*group_to_numbers == NULL)
			*group_to_numbers = g_hash_table_new (g_direct_hash, g_direct_equal);

		group_numbers = get_group_array_from_hash (*group_to_numbers, g, sizeof(gulong));
		g_array_append_val (group_numbers, number);
	}
}

void
articles_set_read (Article ** articles, int article_qty, gboolean read)
{
	int i;
	GHashTable * group_to_articles;
	GHashTable * group_to_numbers = NULL;
	debug_enter ("articles_set_read");

	/* sanity check */
	g_return_if_fail (article_qty > 0);
	g_return_if_fail (articles_are_valid ((const Article **)articles, article_qty));

	/* get a big hash, keyed by Group*,  of all the articles that we're
	   marking read.  We have two hashes -- one for the articles passed
	   in, and one for article numbers of the xposts.  */
	group_to_articles = g_hash_table_new (g_direct_hash, g_direct_equal);
	for (i=0; i<article_qty; ++i)
	{
		GArray * arr;
		Article * a = articles[i];

		/* first add this article to the group's array */
	       	arr = get_group_array_from_hash (group_to_articles, a->group, sizeof(Article*));
		g_array_append_val (arr, a);

		/* secondly, add the xrefs to their group's arrays */
		article_xref_foreach (a, mark_read_xreffunc, &group_to_numbers,
		                      SERVER_GROUPS_SUBSCRIBED, TRUE);

		/* lastly, if it's a multipart, process the child parts too */
		if (a->part==1 && a->parts>1 && a->threads!=NULL)
		{
			GSList * l;
			for (l=a->threads; l!=NULL; l=l->next)
			{
				Article * child = ARTICLE(l->data);
				if (child->part>a->part && child->parts==a->parts)
				{
					/* add this article to the group's array */
					g_array_append_val (arr, child);

					/* add the xrefs to their group's arrays */
					article_xref_foreach (child,
						mark_read_xreffunc,
						&group_to_numbers,
						SERVER_GROUPS_SUBSCRIBED, TRUE);
				}
			}
		}
	}

	/* mark the passed-in articles as read/unread */
	g_hash_table_foreach (group_to_articles,
	                      articles_set_read_articles_ghfunc,
	                      GINT_TO_POINTER(read));
	g_hash_table_destroy (group_to_articles);


	/* mark their crosspost counterparts, if any, as read/unread */
	if (group_to_numbers != NULL) {
		g_hash_table_foreach (group_to_numbers,
		                      articles_set_read_numbers_ghfunc,
		                      GINT_TO_POINTER(read));
		g_hash_table_destroy (group_to_numbers);
	}

	debug_exit ("articles_set_read");
}


/**
*** PUBLIC MUTATORS
**/

void
articles_add_flag (Article  ** articles,
                   int         qty,
                   gushort     flag)
{
	int i;
	int changed_qty;
	Article ** changed;

	/* sanity clause */
	g_return_if_fail (qty >= 1);
	g_return_if_fail (articles_are_valid ((const Article **)articles, qty));

	/* mark 'em */
	changed = g_malloc (sizeof(Article*) * qty);
	for (changed_qty=i=0; i<qty; ++i) {
		if (!(articles[i]->state & flag)) {
			articles[i]->state |= flag;
			changed[changed_qty++] = articles[i];
		}
	}

	if (changed_qty > 0)
		articles_set_dirty (changed, changed_qty);

	g_free (changed);
}


void
articles_remove_flag (Article         ** articles,
                      int                qty,
                      gushort            flag)
{
	int i;
	int changed_qty;
	Article ** changed;

	/* sanity clause */
	g_return_if_fail (qty >= 1);
	g_return_if_fail (articles_are_valid ((const Article **)articles, qty));

	/* mark 'em */
       	changed = g_malloc (sizeof(Article*) * qty);
	for (changed_qty=i=0; i<qty; ++i) {
		if (articles[i]->state & flag) {
			articles[i]->state &= ~flag;
			changed[changed_qty++] = articles[i];
		}
	}

	if (changed_qty > 0)
		articles_set_dirty (changed, changed_qty);

	g_free (changed);
}



/***
****
****  THREADS
****
***/

char*
article_get_thread_message_id (const Article* article)
{
	char * pch;
	const char * refs;

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(article), NULL);

	/* go up as high as we can from our own threading, because sometimes
	 * the References: header is broken by other newreaders. */
	while (article->parent != NULL)
		article = article->parent;

	refs = article->references;
	if (is_nonempty_string(refs))
	{
		pch = get_next_token_str (refs, ' ', NULL);
	}
	else /* top of the thread */
	{
		const char * msg_id = article_get_message_id (article);
		g_return_val_if_fail (is_nonempty_string(msg_id), NULL);
		pch = g_strdup (msg_id);
	}

	return pch;
}

static void
article_get_entire_thread_impl (Article* top, GPtrArray* setme)
{
	GSList *l;
	g_ptr_array_add (setme, top);
	for (l=top->threads; l!=NULL; l=l->next)
		article_get_entire_thread_impl (ARTICLE(l->data), setme);
}

static void
article_get_thread_impl (Article * article,
                         GPtrArray * setme,
                         gboolean children_only)
{
	Article* top = article;

	/* sanity clause */
	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (setme!=NULL);

	if (top != NULL)
	{
		if (!children_only)
			while (top->parent != NULL)
				top = top->parent;

		pan_g_ptr_array_reserve (setme, 128);
		article_get_entire_thread_impl (top, setme);
	}
}

void
article_get_subthread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, TRUE);
}

void
article_get_entire_thread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, FALSE);
}

void
article_get_references (Article* article, GPtrArray* setme)
{
	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (setme!=NULL);

	pan_g_ptr_array_reserve (setme, 128);
	while (article != NULL) {
		g_ptr_array_add (setme, article);
		article = article->parent;
	}
}

GPtrArray*
article_get_unique_threads (const GPtrArray    * articles,
                            ThreadGet            thread_get)
{
	int i;
	GPtrArray * thread;
	GPtrArray * retval;
	GHashTable * all;
	debug_enter ("articlelist_get_unique_threads");

	/* sanity clause */
	retval = g_ptr_array_sized_new (128);
	g_return_val_if_fail (articles_are_valid ((const Article **)articles->pdata, articles->len), retval);

	/* get the thread for each article passed in */
	thread = g_ptr_array_sized_new (128);
	all = g_hash_table_new (g_str_hash, g_str_equal);
	for (i=0; i<articles->len; ++i)
	{
		int j;
		Article * a = ARTICLE(g_ptr_array_index(articles,i));

		/* if we already have the article, then we have its thread, so skip */
		if (g_hash_table_lookup (all, article_get_message_id(a)) != NULL)
			continue;

		/* get the thread associated with sel... */
		g_ptr_array_set_size (thread, 0);
		if (thread_get == GET_SUBTHREAD)
			article_get_subthread (a, thread);
		else
			article_get_entire_thread (a, thread);

		/* add the articles into "all" hash */
		for (j=0; j<thread->len; ++j)
		{
			Article * b = ARTICLE(g_ptr_array_index(thread,j));
			const char * message_id = article_get_message_id (b);

			if (g_hash_table_lookup (all, message_id) == NULL)
			{
				g_hash_table_insert (all, (gpointer)article_get_message_id(b), (gpointer)b);
				g_ptr_array_add (retval, b);
			}
		}
	}

	/* cleanup */
	g_hash_table_destroy (all);
	g_ptr_array_free (thread, TRUE);
	debug_exit ("article_get_unique_threads");
	return retval;
}


const char **
article_get_message_ids  (const Article ** articles,
                          int              article_qty)
{
	int i;
	const char ** message_ids;

	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (article_qty>=0, NULL);
	g_return_val_if_fail (articles_are_valid (articles, article_qty), NULL);

	message_ids = g_new (const char *, article_qty);
	for (i=0; i<article_qty; ++i)
		message_ids[i] = article_get_message_id (ARTICLE(articles[i]));

	return message_ids;
}


/***
****
****  OTHER HEADERS
****
***/

int
article_get_crosspost_qty (const Article  * article)
{
	int retval;

	g_return_val_if_fail (article_is_valid (article), 0);

	if (!is_nonempty_string (article->xref))
		retval = 1;
	else {
		const char * pch;
		retval = 0;
		for (pch=article->xref+6; *pch; ++pch)
			if (*pch==':')
				++retval;
	}

	return retval;
}

void
article_remove_header (Article * a, const char * key)
{
	g_return_if_fail (a!=NULL);
	g_return_if_fail (is_nonempty_string(key));
		 
	if (!strcmp(key, HEADER_MESSAGE_ID))
		a->message_id = NULL;
	else if (!strcmp(key, HEADER_XREF))
		a->xref = NULL;
	else if (!strcmp(key, HEADER_SUBJECT))
		a->subject = NULL;
	else if (!strcmp(key, HEADER_REFERENCES))
		a->references = NULL;
	else if (!strcmp(key, HEADER_FROM))
	{
		a->author_addr = NULL;
		a->author_real = NULL;
	}
	else g_error (_("Don't know what to do with header \"%s\""), key);
}

gboolean
article_header_is_internal (const char * key)
{
	return is_nonempty_string(key) && !g_strncasecmp(key,"X-Pan-Internal-",15);
}

gboolean
article_header_is_extra (const char * key)
{
	/* sanity check */
	if (!is_nonempty_string(key)) return FALSE;

	/* pan internals aren't user-specified headers */
	if (article_header_is_internal(key)) return FALSE;

	/* other headers that are handled explicitly elsewhere */
	if (!strcmp(key,HEADER_FOLLOWUP_TO)) return FALSE;
	if (!strcmp(key,HEADER_NEWSGROUPS)) return FALSE;
	if (!strcmp(key,HEADER_ORGANIZATION)) return FALSE;
	if (!strcmp(key,HEADER_REPLY_TO)) return FALSE;
	if (!strcmp(key,HEADER_DATE)) return FALSE;
	if (!strcmp(key,HEADER_XREF)) return FALSE;
	if (!strcmp(key,HEADER_FROM)) return FALSE;

	return TRUE;
}

void
xref_foreach (const char         * xref_string,
              Server             * server,
              ServerGroupsType     set,
              Group              * skip,
              ArticleXRefFunc      func,
              gpointer             user_data)
{
	/* sanity clause */
	g_return_if_fail (server_is_valid(server));

	/* get the xref header */
	if (is_nonempty_string(xref_string))
	{
		const char * march, * run_str = NULL;
		int run_len;

		/* skip the servername; we've got the server already */
		skip_next_token (xref_string, ' ', &march);

		/* walk through the xrefs, of format "group1:number group2:number" */
		while (get_next_token_run (march, ' ', &march, &run_str, &run_len))
		{
			const char * delimit = g_strstr_len (run_str, run_len, ":");
			if (delimit != NULL)
			{
				const gulong number = strtoul (delimit+1, NULL, 10);
				Group * group = NULL;
				char * group_name;

				pan_strndup_alloca (group_name, run_str, delimit-run_str);
				group = server_get_named_group_in_type (server, group_name, set);
				if (group!=NULL && group!=skip)
					(*func)(server, group, number, user_data);
			}
		}
	}
}

void
article_xref_foreach (const Article       * a,
                      ArticleXRefFunc       func,
                      gpointer              user_data,
		      ServerGroupsType      set,
                      gboolean              skip_group_a)
{
	g_return_if_fail (article_is_valid(a));

	xref_foreach (a->xref, a->group->server, set, skip_group_a?a->group:NULL, func, user_data);
}

static void
article_set_header_impl (Article        * a,
                         const char     * key,
                         const char     * val,
                         HeaderAction     action,
                         gboolean         init)
{
	g_return_if_fail (a!=NULL);
	g_return_if_fail (is_nonempty_string(key));

	/* chunk if necessary */
	if (action==DO_CHUNK || action==DO_CHUNK_SHARE)
		val = group_chunk_string (a->group, val, action==DO_CHUNK_SHARE);

	/* if we're possibly erasing a header, remove the old value */
	if (!init && !is_nonempty_string (val))
		article_remove_header (a, key);

	/* add the new header */
	if (is_nonempty_string (val))
	{
		if      (!strcmp (key, HEADER_MESSAGE_ID)) a->message_id = val;
		else if (!strcmp (key, HEADER_SUBJECT)) a->subject = val;
		else if (!strcmp (key, HEADER_REFERENCES)) a->references = val;
		else if (!strcmp (key, HEADER_XREF)) a->xref = val;
		else if (!strcmp (key, HEADER_FROM))
		{
			if (init)
				article_init_author_from_header (a, val);
			else
				article_set_author_from_header (a, val);
		}
		else pan_warn_if_reached ();
	}
}

void
article_init_header (Article         * a,
                     const char      * key,
                     const char      * val,
                     HeaderAction      action)
{
	article_set_header_impl (a, key, val, action, TRUE);
}

void
article_set_header (Article         * a,
                    const char      * key,
                    const char      * val,
                    HeaderAction      action)
{
	article_set_header_impl (a, key, val, action, FALSE);
}

/***
****
****  BODY
****
***/

char*
article_format_author_str (const char * author_addr,
                           const char * author_real,
                           char       * buf,
                           int          bufsize)
{
	const gboolean have_addr = is_nonempty_string (author_addr);
	const gboolean have_name = is_nonempty_string (author_real);

	*buf = '\0';

	if (have_addr && have_name)
		g_snprintf (buf, bufsize, "\"%s\" <%s>", author_real, author_addr);
	else if (have_addr)
		g_strlcpy (buf, author_addr, bufsize);
	else if (have_name)
		g_strlcpy (buf, author_real, bufsize);

	g_strstrip (buf);
	return buf;
}

char*
article_get_author_str (const Article * a, char * buf, int bufsize)
{
	*buf = '\0';

	/* sanity clause */
	g_return_val_if_fail (buf!=NULL, buf);
	g_return_val_if_fail (a!=NULL, buf);
	g_return_val_if_fail (bufsize>0, buf);

	return article_format_author_str (a->author_addr, a->author_real, buf, bufsize);
}

int
article_format_short_author_str (const char * author_addr,
                                 const char * author_real,
                                 char       * buf,
                                 int          buflen)
{
	int retval;

	/* sanity clause */
	g_return_val_if_fail (buf!=NULL, 0);
	g_return_val_if_fail (buflen>0, 0);

	if (is_nonempty_string (author_real) && strcmp(author_real, default_incoming_name_real))
	{
		retval = g_strlcpy (buf, author_real, buflen);
	}
	else if (is_nonempty_string (author_addr) && strcmp(author_addr,default_incoming_name_addr))
	{
		const char * author = author_addr;
		const char * p = strchr (author, '@');
		if (p == NULL)
			p = author + strlen(author);
		retval = g_strlcpy (buf, author, MIN(buflen,p-author));
	}
	else
	{
		retval = g_strlcpy (buf, default_incoming_name_real, buflen);
	}

	return retval;
}

int
article_get_short_author_str (const Article * a, char * buf, int len)
{
	return article_format_short_author_str (a->author_addr,
	                                        a->author_real,
	                                        buf, len);
}

static void
article_set_author_from_header_impl (Article       * a,
                                     const char    * header_from,
                                     gboolean        init)
{
	char addr_buf[512];
	char real_buf[512];
	char * addr = addr_buf;
	char * real = real_buf;
	char * pch;
	const char * charset;

	/* sanity clause */
	g_return_if_fail (a!=NULL);
	g_return_if_fail (a->group!=NULL);
	g_return_if_fail (is_nonempty_string(header_from));

	/* get the author address & real name */
	charset = group_get_default_charset (a->group);
	pch = pan_header_to_utf8 (header_from, -1, charset);
	gnksa_do_check_from (pch,
	                     addr_buf, sizeof(addr_buf),
	                     real_buf, sizeof(real_buf),
	                     FALSE); /* not strict */
	g_free (pch);

	/* use the real mail address, or fill in a default */
	pch = addr;
	if (!is_nonempty_string (pch))
		pch = (char*)default_incoming_name_addr;
	a->author_addr = group_chunk_string (a->group, pch, TRUE);

	/* use the real name, or fill in a default. */
	if (is_nonempty_string(real)) {
		gnksa_strip_realname (real);
		a->author_real = group_chunk_string (a->group, real, TRUE);
	}
	else if (addr!=NULL && ((pch=strchr(addr,'@'))!=NULL)) {
		char * tmp;
		pan_strndup_alloca (tmp, addr, pch-addr);
		a->author_real = group_chunk_string (a->group, tmp, TRUE);
	}
	else {
		const char * cpch = (const char*)default_incoming_name_real;
		a->author_real = group_chunk_string (a->group, cpch, TRUE);
	}

	if (!init)
		articles_set_dirty (&a, 1);
}

void
article_init_author_from_header (Article         * a,
                                 const char      * header_from)
{
	article_set_author_from_header_impl (a, header_from, TRUE);
}
void
article_set_author_from_header (Article         * a,
                                const char      * header_from)
{
	article_set_author_from_header_impl (a, header_from, FALSE);
}

void
article_set_from_g_mime_message   (Article         * a,
                                   GMimeMessage    * msg)
{
	const char * cpch;
	debug_enter ("article_set_from_raw_message");

	/* get the message */
	g_return_if_fail (a != NULL);
	g_return_if_fail (GMIME_IS_MESSAGE(msg));

	g_mime_message_get_date (msg, &a->date, NULL);

	cpch = g_mime_message_get_sender (msg);
	if (cpch != NULL)
		article_init_header (a, HEADER_FROM, cpch, DO_CHUNK_SHARE);

	cpch = g_mime_message_get_reply_to (msg);
	if (cpch != NULL)
		article_init_header (a, HEADER_REPLY_TO, cpch, DO_CHUNK_SHARE);

	cpch = g_mime_message_get_subject (msg);
	if (cpch != NULL)
		article_init_header (a, HEADER_SUBJECT, cpch, DO_CHUNK_SHARE);

	cpch = g_mime_message_get_message_id (msg);
	if (cpch != NULL)
		article_init_header (a, HEADER_MESSAGE_ID, cpch, DO_CHUNK);

	cpch = g_mime_message_get_header (msg, HEADER_REFERENCES);
	if (cpch != NULL)
		article_init_header (a, HEADER_REFERENCES, cpch, DO_CHUNK);
	
	/* fallback if no message-id */
	if (!is_nonempty_string(a->message_id)) {
		char * id;
		g_warning (_("Couldn't parse a Message-ID from the raw message!"));
		id = gnksa_generate_message_id_from_email_addr (a->author_addr);
		article_init_header (a, HEADER_MESSAGE_ID, id, DO_CHUNK);
		g_free (id);
		article_init_header (a, HEADER_SUBJECT, _("Unparseable Subject"), DO_CHUNK_SHARE);
	}


	/* cleanup */
	debug_exit ("article_set_from_raw_message");
}

/****
*****  SANITY CHECKS
****/

gboolean
article_is_valid (const Article * a)
{
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->group!=NULL, FALSE);
	g_return_val_if_fail (group_is_valid(a->group), FALSE);
	g_return_val_if_fail (a->group->_articles_refcount>=0, FALSE);
	g_return_val_if_fail (group_is_folder(a->group) || a->group->_articles_refcount>0, FALSE);
	g_return_val_if_fail (a->message_id!=NULL, FALSE);
	g_return_val_if_fail (*a->message_id=='<', FALSE);
	g_return_val_if_fail (is_nonempty_string(a->subject), FALSE);
	g_return_val_if_fail (a->number!=0, FALSE);
	g_return_val_if_fail (!a->references || *a->references=='<', FALSE);

	return TRUE;
}

gboolean
articles_are_valid (const Article ** a, int qty)
{
	int i;

	/* sanity clause */
	g_return_val_if_fail (qty>=0, FALSE);
	g_return_val_if_fail (qty==0 || a!=NULL, FALSE);

	/* check each article */
	for (i=0; i<qty; ++i)
		g_return_val_if_fail (article_is_valid (a[i]), FALSE);

	return TRUE;
}

const char*
article_get_message_id (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), "");
	return a->message_id;
}
 
const char*
article_get_subject (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), "");
	return is_nonempty_string(a->subject) ? a->subject : "";
}


gulong
article_get_combined_linecount (const Article * a)
{
	gulong retval = 0;

	g_return_val_if_fail (article_is_valid(a), FALSE);

	retval = a->linecount;
	if (a->part!=0 && a->parts!=0 && a->threads!=NULL) {
		GSList * l;
		int part = a->part + 1;
		for (l=a->threads; l; l=l->next) {
			Article * child = ARTICLE(l->data);
			if (child->part == part) {
				retval += child->linecount;
				++part;
			}
		}
	}

	return retval;
}


/***
****  EVENTS
***/

PanCallback*
article_get_articles_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_articles_changed (Group              * group,
                       Article           ** articles,
                       int                  article_qty,
                       ArticleChangeType    type)
{
	ArticleChangeEvent e;
	debug_enter ("fire_articles_changed");

	g_return_if_fail (group!=NULL);
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (article_qty>0);
	g_return_if_fail (type==ARTICLE_CHANGED_READ || type==ARTICLE_CHANGED_DIRTY || type==ARTICLE_CHANGED_NEW);

	e.group = group;
	e.articles = articles;
	e.article_qty = article_qty;
	e.type = type;
	pan_callback_call (article_get_articles_changed_callback(), &e, NULL);

	debug_exit ("fire_articles_changed");
}
