/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <gmime/memchunk.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/file-headers.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/status-item.h>
#include <pan/base/util-file.h>

/***
****
****
****
***/

typedef struct
{
	Group * folder;
	GPtrArray * articles;
}
FolderForeachStruct;

static void
file_headers_load_folder_foreach (GMimeMessage * message, gpointer user_data)
{
	Article * article;
	FolderForeachStruct * ffs = (FolderForeachStruct*) user_data;

	article = article_new (ffs->folder);
	article_set_from_g_mime_message (article, message);
	article->number = ffs->articles->len + 1;
	g_ptr_array_add (ffs->articles, article);
}

static gboolean
file_headers_load_folder (Group * folder, StatusItem * status)
{
	FolderForeachStruct ffs;
	debug_enter ("file_headers_load_folder");

	/* sanity clause */
	g_return_val_if_fail (group_is_valid(folder), FALSE);
	g_return_val_if_fail (group_is_folder(folder), FALSE);

	/* prep for foreach */
	ffs.folder = folder;
	ffs.articles = g_ptr_array_new ();
	acache_path_foreach (folder->name, file_headers_load_folder_foreach, &ffs);

	/* if we've got articles then add them; otherwise; clean up */
	if (ffs.articles->len != 0)
		group_init_articles (folder, ffs.articles, status);
	else if (folder->_articles) {
		g_hash_table_destroy (folder->_articles);
		folder->_articles = NULL;
	}
	folder->articles_dirty = FALSE;

	g_ptr_array_free (ffs.articles, TRUE);
	debug_exit ("file_headers_load_folder");
	return folder->_articles!=NULL && g_hash_table_size(folder->_articles)!=0;
}

static gboolean
file_headers_load_group (Group * group, StatusItem * status)
{
	gboolean success = FALSE;
	char * dat = NULL;
	char * idx = NULL;
	size_t dat_len = 0;
	size_t idx_len = 0;
	char path[PATH_MAX];
	debug_enter ("file_headers_load_group");

	g_return_val_if_fail (group_is_valid (group), FALSE);

	if (status!=NULL)
		status_item_emit_status_va (status, _("Loading group \"%s\""), group_get_name(group));

	/* open the index file */
	g_snprintf (path, sizeof(path), "%s%c%s%c%s.idx",
	            get_data_dir(), G_DIR_SEPARATOR,
	            server_get_name (group->server), G_DIR_SEPARATOR,
	            group->name);
	g_file_get_contents (path, &idx, &idx_len, NULL);

	/* open the data file */
	g_snprintf (path, sizeof(path), "%s%c%s%c%s.dat",
	            get_data_dir(), G_DIR_SEPARATOR,
	            server_get_name (group->server), G_DIR_SEPARATOR,
	            group->name);
	g_file_get_contents (path, &dat, &dat_len, NULL);

	/* allocate the articles array */
	pan_warn_if_fail (group->_articles==NULL);
	group_get_articles (group);

	if (dat!=NULL && idx!=NULL)
	{
		const char * march = idx;
		const glong version = get_next_token_int (march, '\n', &march);

		if (version==1 || version==2 || version==3 || version==4 || version==5 || version==6 || version==7 || version==8)
		{
			int i;
			int purged_article_count = 0;
			long l;
			const long qty = get_next_token_long (march, '\n', &march);
			GPtrArray * addme;

			/* set the "one big chunk" */
			pan_warn_if_fail (group->_one_big_chunk == NULL);
			group->_one_big_chunk = dat;

			/* load the articles */
		       	addme = g_ptr_array_sized_new (qty);
			for (i=0; i!=qty; ++i)
			{
				Article * a = article_new (group);

				/* message id */
				l = get_next_token_long (march, '\n', &march);
				if (0<=l && l<dat_len)
					a->message_id = dat + l;

				/* author */
				if (version<2) /* version 2 split author into 2 fields */
				{
					l = get_next_token_long (march, '\n', &march);
					if (0<=l && l<dat_len)
						article_init_author_from_header (a, dat+l);
				}
				else
				{
					l = get_next_token_long (march, '\n', &march);
					if (0<=l && l<dat_len)
						a->author_addr = dat + l;

					l = get_next_token_long (march, '\n', &march);
					if (0<=l && l<dat_len)
						a->author_real = dat + l;
				}

				/* subject */
				l = get_next_token_long (march, '\n', &march);
				if (0<=l && l<dat_len)
					a->subject = dat + l;

				/* date string - removed in version 3 */
				if (version<3)
					skip_next_token (march, '\n', &march);

				/* references */
				l = get_next_token_long (march, '\n', &march);
				if (0<=l && l<dat_len)
					a->references = dat + l;

				/* xrefs added in version 4 */
				if (version>=4) {
					l = get_next_token_long (march, '\n', &march);
					if (0<=l && l<dat_len)
						a->xref = dat + l;
				}

				/* numeric fields */
				a->part           = (gint16) get_next_token_int (march, '\n', &march);
				a->parts          = (gint16) get_next_token_int (march, '\n', &march);
				a->linecount      = (guint16) get_next_token_int (march, '\n', &march);

				/* crosspost_qty - removed in version 6 */
				if (version<6)
					skip_next_token (march, '\n', &march);

				/* state - removed in version 6, back in 7 */
				if (version!=6)
					a->state  = (guint16) get_next_token_int (march, '\n', &march);

				a->date           = (time_t) get_next_token_ulong (march, '\n', &march);
				a->number         = (gulong) get_next_token_ulong (march, '\n', &march);

				if (1) {
					/* an article is new if it's flagged as new AND it's unread */
					gboolean b = FALSE;
					if (version >= 5)
						b = get_next_token_int (march, '\n', &march) != 0;
					if (b)
						b = !newsrc_is_article_read (group_get_newsrc(group), a->number);
					a->is_new = b;
				}

				/* extra headers removed in version 8 */
				if (version<8)
					skip_next_token (march, '\n', &march);

				/* let the user know what we're doing */
				if (status != NULL) {
					status_item_emit_next_step (status);
					if (!(addme->len % 256))
						status_item_emit_status_va (status,
							_("Loaded %d of %d articles"), i, qty);
				}

				/* add the article to the group if it looks sane */
				if (article_is_valid (a))
					g_ptr_array_add (addme, a);
				else {
					++purged_article_count;
				}
			}

			if (purged_article_count != 0)
			{
				log_add_va (LOG_ERROR,
					_("Pan skipped %d corrupt headers from the local cache for group \"%s\"."),
					purged_article_count,
					group_get_name(group));
				log_add (LOG_ERROR,
					_("You may want to empty this group and download fresh headers."));
			}

			group_init_articles (group, addme, status);
			group_set_article_qty (group, addme->len);
			group->articles_dirty = purged_article_count!=0;
			success = TRUE;
			g_ptr_array_free (addme, TRUE);
		}
		else
		{
			log_add_va (LOG_ERROR|LOG_URGENT,
				_("Unsupported data version for %s headers: %d.\nAre you running an old version of Pan by accident?"), group->name, version);
		}
	}

	/* cleanup */
	g_free (idx);
	if (!success)
		g_free (dat);

	debug_exit ("file_headers_load_group");
	return success;
}


void
file_headers_load (Group * group, StatusItem * status)
{
	gboolean success = FALSE;
	int size;
	double diff;
	GTimeVal start;
	GTimeVal finish;
	debug_enter ("file_headers_load");

	/* start the stopwatch */
	g_get_current_time (&start);

	success = group_is_folder(group)
		? file_headers_load_folder (group, status)
		: file_headers_load_group (group, status);

	/* expire the old articles, if any */
	if (success)
	{
		gulong low = 0;
		gulong high = 0;

		group_get_article_range (group, &low, &high);
		group_expire_articles_not_in_range (group, low, high);
	}

	/* timing stats */
	g_get_current_time (&finish);
	diff = finish.tv_sec - start.tv_sec;
	diff += (finish.tv_usec - start.tv_usec)/(double)G_USEC_PER_SEC;
	size = g_hash_table_size(group_get_articles(group));
	if (size != 0)
		log_add_va (LOG_INFO, _("Loaded %d articles for group \"%s\" in %.1f seconds (%.0f art/sec)"),
			size,
			group_get_name(group),
			diff,
			size/(fabs(diff)<0.001?0.001:diff));

	debug_exit ("file_headers_load");
}

/**
 * This is used to make sure we don't write the same string to the output
 * file more than once.  This way we can get some space savings from authors
 * and subjects that are repeated many times.
 */
long
get_string_offset (GHashTable   * hash,
                   MemChunk     * long_chunk,
                   const char   * str,
                   FILE         * fp,
                   long         * pos)
{
	gpointer p;
	long retval;

	g_assert (fp != NULL);
	g_assert (pos != NULL);

	if (!is_nonempty_string(str)) /* nothing to write */
	{
		retval = -1;
	}
	else if (hash == NULL) /* don't bother weeding duplicates */
	{
		retval = *pos;
		*pos += fwrite (str, 1, strlen(str)+1, fp);
	}
	else if ((p = g_hash_table_lookup(hash, str)) != NULL) /* a match! */
	{
		retval = *(glong*)p;
	}
	else /* first time; remember where we kept it */
	{
		glong * plong = (glong*) memchunk_alloc (long_chunk);
		retval = *plong = *pos;
		g_hash_table_insert (hash, (gpointer)str, plong);
		*pos += fwrite (str, 1, strlen(str)+1, fp);
	}

	return retval;
}


static gboolean
file_headers_save_group (Group * group, StatusItem * status)
{
	guint i;
	FILE * idx_fp;
	FILE * dat_fp;
	char * pch;
	char * idx_path;
	char * dat_path;
	long pos = 0;
	GHashTable * hash = NULL;
	MemChunk * long_chunk = NULL;
	GPtrArray * articles;
	gboolean ok = TRUE;
	debug_enter ("file_headers_save_group");

	g_return_val_if_fail (group_is_valid (group), FALSE);

	/* if nothing to save */
	articles = group_get_article_array (group);
	if (articles->len == 0) {
		g_ptr_array_free (articles, TRUE);
		file_headers_destroy (group);
		return FALSE;
	}

	/**
	***  Save the Headers
	**/

	if (1) {
		char * pch = g_strdup_printf ("%s%c%s", get_data_dir(), G_DIR_SEPARATOR, server_get_name (group->server));
		if (!pan_file_ensure_path_exists (pch))
			log_add_va (LOG_ERROR, _("Can't create directory \"%s\""), pch);
		g_free (pch);
	}


	/* open index file */
	idx_path = g_strdup_printf ("%s%c%s%c%s.idx.tmp",
		get_data_dir(), G_DIR_SEPARATOR,
		server_get_name (group->server), G_DIR_SEPARATOR,
		group->name);
	idx_fp = fopen (idx_path, "w+");
	if (idx_fp == NULL) {
		log_add_va (LOG_ERROR, _("The group will not be saved -- can't create file \"%s\""), idx_path);
		g_free (idx_path);
		return FALSE;
	}

	/* open data file */
	dat_path = g_strdup_printf ("%s%c%s%c%s.dat.tmp",
		get_data_dir(), G_DIR_SEPARATOR,
		server_get_name (group->server), G_DIR_SEPARATOR,
		group->name);
	dat_fp = fopen (dat_path, "w+");
	if (dat_fp == NULL) {
		log_add_va (LOG_ERROR, _("The group will not be saved -- can't create file \"%s\""), dat_path);
		fclose (idx_fp);
		remove (idx_path);
		g_free (idx_path);
		g_free (dat_path);
		return FALSE;
	}

	/* Write DATBASE_VERSION */
	fprintf (idx_fp, "8\n%ld\n", (long)articles->len);

	/* Write the article information... */
	pos = 0;
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	long_chunk = memchunk_new (sizeof(glong), 4096, FALSE);
	for (i=0; i!=articles->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index (articles, i));
		const char * subject = article_get_subject (a);
		const char * message_id = article_get_message_id (a);
		const long id_idx          = get_string_offset (hash, long_chunk, message_id,     dat_fp, &pos);
		const long author_addr_idx = get_string_offset (hash, long_chunk, a->author_addr, dat_fp, &pos);
		const long author_real_idx = get_string_offset (hash, long_chunk, a->author_real, dat_fp, &pos);
		const long subj_idx        = get_string_offset (hash, long_chunk, subject,        dat_fp, &pos);
		const long refs_idx        = get_string_offset (hash, long_chunk, a->references,  dat_fp, &pos);
		const long xref_idx        = get_string_offset (hash, long_chunk, a->xref,        dat_fp, &pos);

		pan_warn_if_fail (a->number != 0);

		/* write the non-string fields. */
		fprintf (idx_fp,
			"%ld\n" "%ld\n" "%ld\n" "%ld\n" "%ld\n" "%ld\n"
			"%d\n" "%d\n"
			"%u\n"
			"%u\n"
			"%lu\n"
			"%lu\n"
			"%d\n",
			id_idx, author_addr_idx, author_real_idx, subj_idx, refs_idx, xref_idx,
			(int)a->part, (int)a->parts,
			(unsigned int)a->linecount,
			(int)a->state,
			(unsigned long)a->date,
			(unsigned long)a->number,
			(int)(a->is_new != 0));
	}
	g_hash_table_destroy (hash);
	hash = NULL;
	memchunk_destroy (long_chunk);

	/* did the files write out okay? */
	ok = !ferror(idx_fp) && !ferror(dat_fp);

	/* close the temp files */
	fclose (idx_fp);
	fclose (dat_fp);

	if (!ok)
	{
		log_add_va (LOG_ERROR, _("Unable to save headers for group \"%s\" - is the disk full?"), group->name);
		unlink (idx_path);
		unlink (dat_path);
	}

	if (ok)
	{
		pch = g_strdup_printf ("%s%c%s%c%s.idx",
			get_data_dir(), G_DIR_SEPARATOR,
			server_get_name (group->server), G_DIR_SEPARATOR,
			group->name);
		ok = pan_file_rename (idx_path, pch);
		g_free (idx_path);
		g_free (pch);
	}

	if (ok)
	{
		pch = g_strdup_printf ("%s%c%s%c%s.dat",
			get_data_dir(), G_DIR_SEPARATOR,
			server_get_name (group->server), G_DIR_SEPARATOR,
			group->name);
		ok = pan_file_rename (dat_path, pch);
		g_free (dat_path);
		g_free (pch);
	}

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("file_headers_save_group");
	return ok;
}

void
file_headers_save_noref (Group * group, StatusItem * status)
{
	static GStaticMutex _mutex = G_STATIC_MUTEX_INIT;

	debug_enter ("file_headers_save_noref");
	g_static_mutex_lock (&_mutex);

	if (group->articles_dirty)
	{
		GPtrArray * articles;

		/* since we're saving, mark dirty to FALSE */
		group->articles_dirty = FALSE;

		articles = group_get_article_array (group);
		if (articles->len == 0) /* destroy the group */
		{
			file_headers_destroy (group);
		}
		else if (!group_is_folder (group))/* save the group */
		{
			GTimeVal start;
			GTimeVal finish;
			double diff;

			/* start the stopwatch */
			g_get_current_time (&start);

			/* save the group */
			if (!file_headers_save_group (group, status))
				group->articles_dirty = TRUE;
			else {
				/* end the stopwatch */
				g_get_current_time (&finish);
				diff = finish.tv_sec - start.tv_sec;
				diff += (finish.tv_usec - start.tv_usec)/(double)G_USEC_PER_SEC;
				log_add_va (LOG_INFO, _("Saved %d articles in \"%s\" in %.1f seconds (%.0f art/sec)"),
					articles->len,
					group_get_name (group),
					diff,
					articles->len/(fabs(diff)<0.001?0.001:diff));
			}
		}

		/* cleanup */
		g_ptr_array_free (articles, TRUE);
	}

	g_static_mutex_unlock (&_mutex);

	debug_exit ("file_headers_save_noref");
}
void
file_headers_save (Group * group, StatusItem * status)
{
	debug_enter ("file_headers_save");
	g_return_if_fail (group!=NULL);

	group_ref_articles (group, status);
	file_headers_save_noref (group, status);
	group_unref_articles (group, status);

	debug_exit ("file_headers_save");
}

void
file_headers_destroy (const Group * group)
{
	debug_enter ("file_headers_destroy");

	g_return_if_fail (group_is_valid (group));

	if (!group_is_folder(group))
	{
		char * path;

		path = g_strdup_printf ("%s%c%s%c%s.idx",
		                        get_data_dir(), G_DIR_SEPARATOR,
		                        server_get_name (group->server), 
					G_DIR_SEPARATOR, group->name);
		remove (path);
		g_free (path);

		path = g_strdup_printf ("%s%c%s%c%s.dat",
		                        get_data_dir(), G_DIR_SEPARATOR,
		                        server_get_name (group->server), 
					G_DIR_SEPARATOR, group->name);
		remove (path);
		g_free (path);
	}

	debug_exit ("file_headers_destroy");
}
