/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <pan/base/debug.h>
#include <pan/base/message-identifier.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/base/serverlist.h>
#include <pan/base/group.h>
#include <pan/base/article.h>

/**
***  GObject stuff
**/

static GObjectClass * parent_class = NULL;

static void
pan_message_identifier_init (MessageIdentifier * mid, MessageIdentifierClass * klass)
{
	mid->line_qty = 0ul;
	mid->message_id = NULL;
	mid->message_sources = g_ptr_array_new ();
}

static void
pan_message_identifier_finalize (GObject * object)
{
	int i;
	MessageIdentifier * mid = (MessageIdentifier*) object;

	for (i=0; i<mid->message_sources->len; ++i)
	{
		MessageSource * source = (MessageSource*) g_ptr_array_index (mid->message_sources, i);
		g_free (source->server_name);
		g_free (source->group_name);
		g_free (source);
	}
	g_ptr_array_free (mid->message_sources, TRUE);
	g_free (mid->message_id);

	mid->message_sources = NULL;
	mid->message_id = NULL;

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
pan_message_identifier_class_init (MessageIdentifierClass * klass)
{
	GObjectClass * object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_ref (G_TYPE_OBJECT);

	object_class->finalize = pan_message_identifier_finalize;
}

GType
pan_message_identifier_get_type (void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof (MessageIdentifierClass),
			NULL, /* base_class_init */
			NULL, /* base_class_finalize */
			(GClassInitFunc) pan_message_identifier_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof (MessageIdentifier),
			16,   /* n_preallocs */
			(GInstanceInitFunc) pan_message_identifier_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "MessageIdentifier", &info, 0);
	}

	return type;
}

/**
***  MessageIdentifier Stuff
**/

MessageIdentifier*
message_identifier_new (const char * message_id)
{
	MessageIdentifier * mid;

	mid = g_object_new (PAN_TYPE_MESSAGE_IDENTIFIER, NULL, NULL);
	mid->message_id = g_strdup (message_id);

	return mid;
}

MessageIdentifier*
message_identifier_new_from_article (const Article * article)
{
	MessageIdentifier * mid;

	g_return_val_if_fail (article_is_valid(article), NULL);

	mid = message_identifier_new (article_get_message_id(article));
	message_identifier_add_source_from_article (mid, article);

	return mid;
}

void
message_identifier_add_source (MessageIdentifier  * mid,
	                       const char         * server,
	                       const char         * group,
	                       gulong               number)
{
	MessageSource * source;

	g_return_if_fail (PAN_IS_MESSAGE_IDENTIFIER(mid));
       
	source = g_new (MessageSource, 1);
	source->server_name = g_strdup (server);
	source->group_name = g_strdup (group);
	source->number = number;
	g_ptr_array_add (mid->message_sources, source);
}

static void
add_source_from_article_xreffunc (Server * server, Group * group, gulong number, gpointer user_data)
{
	MessageIdentifier * mid = MESSAGE_IDENTIFIER(user_data);

	message_identifier_add_source (mid, server->name, group->name, number);
}

void
message_identifier_add_source_from_article  (MessageIdentifier  * mid,
                                             const Article      * article)
{
	g_return_if_fail (PAN_IS_MESSAGE_IDENTIFIER(mid));
	g_return_if_fail (article_is_valid(article));

	mid->line_qty = article->linecount;
	message_identifier_add_source (mid, article->group->server->name, article->group->name, article->number);
	article_xref_foreach (article, add_source_from_article_xreffunc, mid, SERVER_GROUPS_ALL, FALSE);
}

gboolean
message_identifier_is_valid (const MessageIdentifier * mid)
{
	g_return_val_if_fail (PAN_IS_MESSAGE_IDENTIFIER(mid), FALSE);
	g_return_val_if_fail (is_nonempty_string(mid->message_id), FALSE);

	return TRUE;
}

MessageSource*
message_identifier_get_source_for_server   (MessageIdentifier  * mid,
                                            const char         * server)
{
	int i;
	MessageSource * retval = NULL;

	/* sanity clause */
	g_return_val_if_fail (PAN_IS_MESSAGE_IDENTIFIER(mid), NULL);
	g_return_val_if_fail (is_nonempty_string(server), NULL);

	/* return the first for the matching that server */
	for (i=0; !retval && i<mid->message_sources->len; ++i) {
		MessageSource * source = (MessageSource*) g_ptr_array_index (mid->message_sources, i);
		if (!pan_strcmp (server, source->server_name))
			retval = source;
	}

	return retval;
}

const char*
message_identifier_get_primary_group (MessageIdentifier * mid)
{
	const char * retval = NULL;

	/* sanity clause */
	g_return_val_if_fail (PAN_IS_MESSAGE_IDENTIFIER(mid), NULL);

	/* find the first group name */
	if (mid->message_sources->len) {
		MessageSource * source = (MessageSource*) g_ptr_array_index (mid->message_sources, 0);
		retval = source->group_name;
	}

	return retval;
}

/***
****
***/

static void
group2art_mark_read_ghfunc (gpointer key, gpointer val, gpointer user_data)
{
	Group * group = GROUP (key);
	GHashTable * article_hash = (GHashTable*) val;
	GPtrArray * article_array;

       	article_array = g_ptr_array_new ();
	pan_hash_to_ptr_array (article_hash, article_array);
	articles_set_read_simple ((Article**)article_array->pdata, article_array->len, TRUE);
	g_ptr_array_free (article_array, TRUE);

	g_hash_table_destroy (article_hash);
	group_unref_articles (group, NULL);
}

void
message_identifiers_mark_read (MessageIdentifier ** mids,
                               int                  mid_qty,
                               gboolean             read,
			       ServerGroupsType     type)
{
	int i;
	GHashTable * group2art;
	debug_enter ("message_identifiers_mark_read");

	/* sanity clause */
	g_return_if_fail (mids!=NULL);
	g_return_if_fail (mid_qty>0);
	g_return_if_fail (PAN_IS_MESSAGE_IDENTIFIER(mids[0]));

	group2art = g_hash_table_new (g_direct_hash, g_direct_equal);

	/* walk through all the mids */
	for (i=0; i<mid_qty; ++i)
	{
		int j;
		MessageIdentifier * mid = mids[i];
		for (j=0; j<mid->message_sources->len; ++j)
		{
			MessageSource * source = (MessageSource*) g_ptr_array_index (mid->message_sources, j);
			Server * server;
			Group * group;

			/* find the source's server */
			server = serverlist_get_named_server (source->server_name);
			if (server == NULL)
				continue;

			/* find the source's group */
			group = server_get_named_group_in_type (server, source->group_name, type);
			if (group == NULL)
				continue;

			/* if the group doesn't have the articles loaded, then nobody
			 * is listening to the article changing state, so we can poke
			 * the newsrc directly and not need to fire an event.
			 * Otherwise, the articles are loaded, so we need to
			 * use the Article objects. */
			if (!group_ref_articles_if_loaded (group)) {
				newsrc_mark_article (group_get_newsrc(group), source->number, TRUE);
			} else {
				Article * a = group_get_article_by_message_id (group, mid->message_id);
				if (a != NULL) {
					/* The Article container we use is a Hash, to weed out duplicates. */
					GHashTable * art_hash = (GHashTable*) g_hash_table_lookup (group2art, group);
					if (art_hash == NULL) {
						art_hash = g_hash_table_new (g_direct_hash, g_direct_equal);
						g_hash_table_insert (group2art, group, art_hash);
						group_ref_articles (group, NULL);
					}

					g_hash_table_insert (art_hash, a, a);
				}
				group_unref_articles (group, NULL);
			}
		}
	}

	g_hash_table_foreach (group2art, group2art_mark_read_ghfunc, NULL);
	g_hash_table_destroy (group2art);

	debug_exit ("group_mark_numbers_read");
}
