/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>
#include <pan/flagset.h>
#include <pan/queue.h>
#include <pan/task-bodies.h>

static GHashTable * _id_to_mid = NULL;
static void fire_messages_added (MessageIdentifier** mids, int qty);
static void fire_messages_removed (MessageIdentifier** mids, int qty);

void
flagset_init (void)
{
	_id_to_mid = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, g_object_unref);
}

gboolean
flagset_has_message (const char * message_id)
{
	g_return_val_if_fail (is_nonempty_string(message_id), FALSE);

	return g_hash_table_lookup (_id_to_mid, message_id) != NULL;
}

void
flagset_add_messages (MessageIdentifier  ** mids,
                      int                   qty)
{
	int i;

	/* sanity clause */
	g_return_if_fail (qty>=1);
	for (i=0; i<qty; ++i)
		g_return_if_fail (message_identifier_is_valid (mids[i]));


	/* add these message-ids to the set */
	for (i=0; i<qty; ++i) {
		g_object_ref (mids[i]);
		g_hash_table_replace (_id_to_mid, g_strdup(mids[i]->message_id), mids[i]);
	}

	fire_messages_added (mids, qty);
}

void
flagset_add_articles (const Article   ** articles,
                      int                qty)
{
	int i;
	MessageIdentifier ** mids;

	/* sanity clause */
	g_return_if_fail (qty>=1);
	g_return_if_fail (articles_are_valid(articles, qty));

	/* convert to message-identifiers */
	mids = g_alloca (sizeof(MessageIdentifier*) * qty);
	for (i=0; i<qty; ++i)
		mids[i] = message_identifier_new_from_article (articles[i]);

	/* add */
	flagset_add_messages (mids, qty);

	/* cleanup */
	for (i=0; i<qty; ++i)
		g_object_unref (mids[i]);
}

void
flagset_remove_messages (MessageIdentifier  ** mids,
                         int                   qty)
{
	int i;

	/* sanity clause */
	g_return_if_fail (qty>=1);
	for (i=0; i<qty; ++i)
		g_return_if_fail (message_identifier_is_valid (mids[i]));

	/* remove these message-ids from the set */
	for (i=0; i<qty; ++i)
		g_hash_table_remove (_id_to_mid, mids[i]->message_id);

	fire_messages_removed (mids, qty);
}

void
flagset_remove_articles (const Article   ** articles,
                         int                qty)
{
	int i;
	MessageIdentifier ** mids;

	/* sanity clause */
	g_return_if_fail (qty>=1);
	g_return_if_fail (articles_are_valid(articles, qty));

	/* convert to message-identifiers */
	mids = g_alloca (sizeof(MessageIdentifier*) * qty);
	for (i=0; i<qty; ++i)
		mids[i] = message_identifier_new_from_article (articles[i]);

	/* add */
	flagset_remove_messages (mids, qty);

	/* cleanup */
	for (i=0; i<qty; ++i)
		g_object_unref (mids[i]);
}


void
flagset_clear (void)
{
	GPtrArray * mids;

	/* get an array of the identifiers and ref them so they
	 * can be used after the hash is destroyed */
       	mids = g_ptr_array_new ();
	pan_hash_to_ptr_array (_id_to_mid, mids);
	pan_g_ptr_array_foreach (mids, (GFunc)g_object_ref, NULL);

	/* reset the hash table */
	g_hash_table_destroy (_id_to_mid);
	_id_to_mid = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, g_object_unref);

	/* fire clear event */
	fire_messages_removed ((MessageIdentifier**)mids->pdata, mids->len);

	/* cleanup */
	pan_g_ptr_array_foreach (mids, (GFunc)g_object_unref, NULL);
	g_ptr_array_free (mids, TRUE);
}

void
flagset_flush (void)
{
	Server * server;

	/* find the server */
	server = serverlist_get_active_server ();
	if (server != NULL)
	{
		GPtrArray * a;

		/* queue the download */
		a = g_ptr_array_new ();
		pan_hash_to_ptr_array (_id_to_mid, a);
		if (a->len)
			queue_add (TASK (task_bodies_new (server, (MessageIdentifier**)a->pdata, a->len)));

		/* cleanup */
		g_ptr_array_free (a, TRUE);
		flagset_clear ();
	}

}

PanCallback*
flagset_get_added_cb (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_messages_added (MessageIdentifier ** mids, int qty)
{
	int i;
	const char ** message_ids;

	g_return_if_fail (mids!=NULL);
	g_return_if_fail (qty>0);

	message_ids = g_alloca (sizeof(char*) * qty);
	for (i=0; i<qty; ++i)
		message_ids[i] = mids[i]->message_id;
	pan_callback_call (flagset_get_added_cb(),
	                   message_ids,
	                   GINT_TO_POINTER(qty));
}

PanCallback*
flagset_get_removed_cb (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_messages_removed (MessageIdentifier ** mids, int qty)
{
	int i;
	const char ** message_ids;

	g_return_if_fail (mids!=NULL);
	g_return_if_fail (qty>0);

	message_ids = g_alloca (sizeof(char*) * qty);
	for (i=0; i<qty; ++i)
		message_ids[i] = mids[i]->message_id;
	pan_callback_call (flagset_get_removed_cb(),
	                   message_ids,
	                   GINT_TO_POINTER(qty));
}
