/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <errno.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/file-grouplist.h>
#include <pan/base/fnmatch.h>
#include <pan/base/pan-config.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>

#include <pan/articlelist.h>
#include <pan/dialogs/dialogs.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/group-action.h>
#include <pan/group-ui.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-grouplist.h>
#include <pan/util.h>

#include <pan/xpm/pan-pixbufs.h>

static GtkItemFactoryEntry group_popup_entries[] =
{
	/* r */ {N_("/Mark Group _Read"), NULL, group_action_selected_mark_read, 0, NULL},
	/* d */ {N_("/_Delete Group's Articles"), NULL, group_action_selected_empty, 0, "<StockItem>", GTK_STOCK_DELETE},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* h */ {N_("/Get New _Headers"), NULL, group_action_selected_download_new, 0, NULL},
	/* b */ {N_("/Get New Headers and _Bodies"), NULL, group_action_selected_download_new_and_bodies, 0, NULL},
	/* o */ {N_("/More Download _Options..."), NULL, group_action_selected_download_dialog, 0, NULL},
	/* c */ {N_("/Refresh Article _Counts"), NULL, group_action_selected_update_count_info, 0, "<StockItem>", GTK_STOCK_REFRESH},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* s */ {N_("/_Subscribe"), NULL, group_action_selected_subscribe, 0, "<ImageItem>", icon_newsgroup},
	/* u */ {N_("/_Unsubscribe"), NULL, group_action_selected_unsubscribe, 0, NULL},
	/* p */ {N_("/Group _Properties..."), NULL, group_action_selected_properties, 0, "<StockItem>", GTK_STOCK_PROPERTIES},
	/*   */ {N_("/---"), NULL, NULL, 0, "<Separator>"},
	/* t */ {N_("/Dele_te Group"), NULL, group_action_selected_destroy, 0, "<StockItem>", GTK_STOCK_DELETE}
};

static gint grouplist_update_ui_mainthread (gpointer);
static void group_menu_popup (GdkEventButton * event);
static gboolean grouplist_filter_changed_cb_nolock (void);
static void grouplist_refresh_filter_nolock (void);
static void grouplist_update_groups (const Group ** groups, int group_qty);
static void grouplist_set_server_nolock (Server *);

extern GtkWidget   * ttips;
static GtkItemFactory * popup_factory        = NULL;
static GtkWidget * popup_menu                = NULL;

static int           group_mode              = GROUP_ALL;
static Server      * my_server               = NULL;
static GtkWidget   * grouplist_mode_menu     = NULL;
static GtkWidget   * groupname_filter_entry  = NULL;
static gchar       * groupname_filter_string = NULL;

/**
 * This lock must be gotten _before_ the gui lock, and released
 * _after_ the gui lock is released.  Otherwise deadlock may occur.
 */
static GStaticMutex  grouplist_ui_mutex      = G_STATIC_MUTEX_INIT;

PanCallback * grouplist_server_set_callback = NULL;
PanCallback * grouplist_group_selection_changed = NULL;

static void
grouplist_get_group_name (const Group * group, char * buf, int buflen)
{
	const char * name = group_get_name (group);
	int bufleft;

	/**
	***  fill the name
	**/

	if (!collapse_group_names)
		g_strlcpy (buf, name, buflen);
	else
		group_get_collapsed_name (group, buf, buflen);
	bufleft = buflen - strlen(buf);

	/**
	***  add the parenthetical status messages
	**/

	if (group_is_moderated (group)) {
		strncat (buf, _(" (Moderated)"), bufleft);
		buf[buflen-1] = '\0';
		bufleft = buflen - strlen(buf);
	}

	if (group_is_read_only (group)) {
		strncat (buf, _(" (Read-Only)"), bufleft);
		buf[buflen-1] = '\0';
		bufleft = buflen - strlen(buf);
	}
}

static Server *
grouplist_get_visible_server (void)
{
	return group_mode==GROUP_FOLDERS
		? serverlist_get_named_server(INTERNAL_SERVER_NAME)
		: my_server;
}

static void
grouplist_dialog_response_cb (GtkDialog * dialog, int response, gpointer user_data)
{
	if (response == GTK_RESPONSE_YES)
		queue_add (TASK(task_grouplist_new (SERVER(user_data), GROUPLIST_ALL)));
	gtk_widget_destroy (GTK_WIDGET(dialog));
}


/*****
******
*****/

/*
 * This thing is frustrating... basically we're trying to reconcile
 * the desired behavior with GtkCList's quirks.  The desired behavior:
 *
 * (1) If the user selects a range, don't activate the group
 *     because it's more than likely that the user is selecting
 *     the range to perform a menu popup operation.
 *
 * (2) If a group is single-clicked and single-click prefs are on,
 *     or if a group was double-clicked, activate it.
 *
 * The problem with (1) is that the selection callback fires once per
 * selected item, so there's no way to tell if the first callback is
 * part of a range being selected or not.  So the workaround here is
 * to push the work off to an idle function that gets invoked after
 * all the selection callbacks are done, then query the grouplist for
 * selection info and fire the grouplist selection pan_callback.
 * (2) isn't as hard to do -- the * variable `button_click_count' is
 * used to keep information from the * keypress callback so that we can
 * use it in the selection callback.
 */
static gint button_click_count = -1;
static gint mb = -1;

static void
grouplist_selection_changed_cb (gpointer call_object,
                                gpointer call_arg,
                                gpointer user_data)
{
	GPtrArray * groups = (GPtrArray*)call_arg;
	Group * single_group = NULL;
	debug_enter ("grouplist_selection_changed_cb");

	/* load the specified group */
	if (groups!=NULL && groups->len==1)
		single_group = GROUP(g_ptr_array_index(groups,0));

	if ((single_group!=NULL)
		&& ((button_click_count==2) /* double click */
			|| (button_click_count==1 && mb==1 && !single_click_selects)))
	{
		gui_page_set (HEADERS_PANE);
		articlelist_set_group (single_group);
	}

	/* reset the click count for next time... */
	button_click_count = -1;
	mb = -1;

	debug_exit ("grouplist_selection_changed_cb");
}

static gboolean
grouplist_button_press (GtkWidget *widget, GdkEventButton * event)
{
	debug_enter ("grouplist_button_press");

	switch (event->button)
	{
		case 1:
		case 2:
			mb = event->button;
			if (event->type == GDK_2BUTTON_PRESS)
				button_click_count = 2;
			else
				button_click_count = 1;
			break;
		case 3:
			group_menu_popup (event);
			break;
	}

	debug_exit ("grouplist_button_press");
	return FALSE;
}

static gboolean
grouplist_key_press (GtkWidget     * widget,
                     GdkEventKey   * event,
                     gpointer        user_data)
{
	debug_enter ("grouplist_key_press");
	switch (event->keyval)
	{
		case GDK_Up: case GDK_Down:
		case GDK_Left: case GDK_Right:
		case GDK_Page_Up: case GDK_Page_Down:
			button_click_count = 1;
			break;
		default:
			button_click_count = -1;
	} 
	debug_exit ("grouplist_key_press");
	return FALSE;
}

static gboolean select_callback_pending = FALSE;

static gint
select_row_cb_mainthread (gpointer data)
{
	GPtrArray * groups;
	debug_enter ("select_row_cb_mainthread");

       	groups = grouplist_get_selected_groups ();
	pan_callback_call (grouplist_group_selection_changed,
	                   Pan.group_tree,
			   groups);
	select_callback_pending = FALSE;
	g_ptr_array_free (groups, TRUE);

	debug_exit ("select_row_cb_mainthread");
	return FALSE;
}
static void
select_row_cb (GtkWidget  * widget,
               int          row,
               int          col,
               GdkEvent   * event)
{
	debug_enter ("select_row_cb");

	if (!select_callback_pending)
	{
		select_callback_pending = TRUE;
		gui_queue_add (select_row_cb_mainthread, NULL);
	}

	debug_exit ("select_row_cb");
}

/*****
******
*****/

static void
grouplist_set_selected_groups_nolock (const GPtrArray * a)
{
	GtkCList * clist;
	debug_enter ("grouplist_set_selected_groups_nolock");

	if (a != NULL)
	{
		guint i;

		clist = GTK_CLIST(Pan.group_tree);
		gtk_clist_freeze (clist);
		gtk_clist_unselect_all (clist);
		for (i=0; i!=a->len; ++i)
		{
			const Group * g = GROUP(g_ptr_array_index(a,i));
			const int row = gtk_clist_find_row_from_data (clist, (gpointer)g);
			gtk_clist_select_row (clist, row, 0);
			if (!gtk_clist_row_is_visible (clist, row))
			{
				gfloat valign = 0.5;
				gfloat halign = 0.0;
				gtk_clist_moveto (clist, row, 0, (double)valign, (double)halign);
			}
		}
		gtk_clist_thaw (clist);
	}

	debug_exit ("grouplist_set_selected_groups_nolock");
}

Group*
grouplist_get_selected_group (void)
{
	GList * l;
	GtkCList * clist;
	Group * retval = NULL;
	debug_enter ("grouplist_get_selected_group");

	pan_lock ();
	clist = GTK_CLIST(Pan.group_tree);
	l = clist->selection;
	if (l != NULL)
		retval = GROUP(gtk_clist_get_row_data(clist, GPOINTER_TO_INT(l->data)));
	pan_unlock ();

	debug_exit ("grouplist_get_selected_group");
	return retval;
}

GPtrArray*
grouplist_get_selected_groups (void)
{
	GtkCList * clist = GTK_CLIST(Pan.group_tree);
	GPtrArray * retval = g_ptr_array_new ();
	GList * l = NULL;
	debug_enter ("grouplist_get_selected_groups");

	/* get the group selected items; otherwise, 
	   use the articlelist's group */
	pan_lock ();
	for (l=clist->selection; l!=NULL; l=l->next)
		g_ptr_array_add (retval, gtk_clist_get_row_data (clist, GPOINTER_TO_INT (l->data)));
	pan_unlock ();

	if (!retval->len) {
		Group * group = articlelist_get_group ();
		if (group != NULL)
			g_ptr_array_add (retval, group);
	}

	debug_exit ("grouplist_get_selected_groups");
	return retval;
}

/*****
******
*****/

static void
grouplist_remove_row (const Group *group)
{
	debug_enter ("grouplist_remove_row");

	if (group->server == grouplist_get_visible_server())
	{
		GtkCList *list = GTK_CLIST(Pan.group_tree);
		int row;

		g_static_mutex_lock (&grouplist_ui_mutex);
		pan_lock();
		row = gtk_clist_find_row_from_data (list, (gpointer)group);
		if (row != -1)
			gtk_clist_remove (list, row);
		pan_unlock();
		g_static_mutex_unlock (&grouplist_ui_mutex);
	}

	debug_exit ("grouplist_remove_row");
}

/*****
******
*****/

void
grouplist_get_all (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		queue_add (TASK(task_grouplist_new (my_server, GROUPLIST_ALL)));
}

void
grouplist_get_new (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		queue_add (TASK(task_grouplist_new (my_server, GROUPLIST_NEW)));
}

/*****
******
*****/

void
grouplist_add_subscribed_to_selection_nolock (void)
{
	GtkCList * clist = GTK_CLIST (Pan.group_tree);
	const int row_qty = clist->rows;
	if (row_qty != 0)
	{
		int row;
		gtk_clist_freeze (clist);
		for (row=0; row<row_qty; ++row)
			if (group_is_subscribed (GROUP(gtk_clist_get_row_data(clist,row))))
				gtk_clist_select_row (clist, row, -1);
		gtk_clist_thaw (clist);
	}
}

/*****
******
*****/

static gint
grouplist_select_all_idle (gpointer unused) {
	pan_lock ();
	gtk_clist_select_all (GTK_CLIST (Pan.group_tree));
	pan_unlock ();
	return 0;
}
void
grouplist_select_all (void) {
	gui_queue_add (grouplist_select_all_idle, NULL);
}

/*****
******
*****/

static gint
grouplist_deselect_all_idle (gpointer unused)
{
	pan_lock ();
	gtk_clist_unselect_all (GTK_CLIST(Pan.group_tree));
	pan_unlock ();
	return 0;
}
void
grouplist_deselect_all (void)
{
	gui_queue_add (grouplist_deselect_all_idle, NULL);
}

/*****
******
*****/

static void
grouplist_set_view_mode_nolock_impl (gint view_mode)
{
	GtkWidget *menu = NULL;
	debug_enter ("grouplist_set_view_mode_nolock_impl");

        menu = gtk_option_menu_get_menu (GTK_OPTION_MENU(grouplist_mode_menu));
        g_object_ref (G_OBJECT(menu));
        gtk_option_menu_remove_menu (GTK_OPTION_MENU (grouplist_mode_menu));
	switch (group_mode = view_mode) {
		case GROUP_ALL:
			gtk_menu_set_active (GTK_MENU (menu), 0);
			break;
		case GROUP_SUBSCRIBED:
			gtk_menu_set_active (GTK_MENU (menu), 1);
			break;
		case GROUP_NEW:
			gtk_menu_set_active (GTK_MENU (menu), 2);
			break;
		case GROUP_FOLDERS:
			gtk_menu_set_active (GTK_MENU (menu), 3);
			break;
		default: break;
	}
        gtk_option_menu_set_menu (GTK_OPTION_MENU(grouplist_mode_menu), menu);
        g_object_unref (G_OBJECT(menu));

	debug_exit ("grouplist_set_view_mode_nolock_impl");
}

static void
grouplist_set_view_mode_nolock (GtkWidget *widget, gpointer data)
{
	grouplist_set_view_mode_nolock_impl (GPOINTER_TO_INT(data));
	grouplist_refresh_nolock ();
}

typedef struct
{
	gchar * name;
	int mode;
}
GrouplistModeMenuStruct;

static GtkWidget*
grouplist_mode_menu_create (void)
{
	GtkWidget *option_menu = gtk_option_menu_new();
	GtkWidget *menu = gtk_menu_new ();
	int index = 0;
	int i;
	GrouplistModeMenuStruct foo[] = {
		{NULL, GROUP_ALL},
		{NULL, GROUP_SUBSCRIBED},
		{NULL, GROUP_NEW},
		{NULL, GROUP_FOLDERS}
	};
	const int row_qty = sizeof(foo) / sizeof(foo[0]);
	foo[0].name = _("All Groups");
	foo[1].name = _("Subscribed");
	foo[2].name = _("New Groups");
	foo[3].name = _("Folders");

	for (i=0; i<row_qty; ++i) {
		GtkWidget *item = gtk_menu_item_new_with_label (foo[i].name);
		gtk_widget_show (item);
		gtk_menu_append (GTK_MENU (menu), item);
		g_signal_connect (item, "activate",
				  G_CALLBACK(grouplist_set_view_mode_nolock), GINT_TO_POINTER(foo[i].mode));
		if (group_mode == foo[i].mode)
			index = i;
	}

	gtk_menu_set_active (GTK_MENU(menu), index);
        gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
	gtk_widget_show_all (GTK_WIDGET(option_menu));

	return option_menu;
}

/*****
******  CALLBACKS
*****/

static void
grouplist_server_groups_added_cb (gpointer server, gpointer groups, gpointer foo)
{
	Server * visible;
	debug_enter ("grouplist_server_groups_added_cb");

	visible = grouplist_get_visible_server ();
	if (visible!=NULL && visible==SERVER(server))
		grouplist_refresh_nolock ();

	debug_exit ("grouplist_server_groups_added_cb");
}

static void
grouplist_server_groups_removed_cb (gpointer server, gpointer groups, gpointer foo)
{
	Server * visible;
	debug_enter ("grouplist_server_groups_removed_cb");

	visible = grouplist_get_visible_server ();
	if (visible!=NULL && visible==SERVER(server)) {
		GPtrArray * a = (GPtrArray*) groups;
		pan_g_ptr_array_foreach (a, (GFunc)grouplist_remove_row, NULL);
	}

	debug_exit ("grouplist_server_groups_removed_cb");
}

static void
grouplist_groups_changed_cb (gpointer gp_groups, gpointer gp_qty, gpointer unused2)
{
	gint qty = GPOINTER_TO_INT(gp_qty);
	const Group ** groups = (const Group**) gp_groups;

	g_return_if_fail (qty>0);
	g_return_if_fail (groups!=NULL);

	if (groups[0]->server == grouplist_get_visible_server())
		grouplist_update_groups (groups, qty);
}

static void
grouplist_server_activated_cb (gpointer server_data, gpointer unused1, gpointer unused2)
{
	pan_lock ();
	grouplist_set_server_nolock (SERVER(server_data));
	pan_unlock ();
}

/*****
******
*****/

void
grouplist_shutdown_module (void)
{
	/* stop listening... */
	pan_callback_remove (serverlist_get_server_activated_callback(),
	                     grouplist_server_activated_cb, NULL);
	pan_callback_remove (group_get_groups_changed_callback(),
	                     grouplist_groups_changed_cb, NULL);
	pan_callback_remove (grouplist_group_selection_changed,
	                     grouplist_selection_changed_cb, NULL);
	pan_callback_remove (server_get_groups_added_callback(),
	                     grouplist_server_groups_added_cb, NULL);
	pan_callback_remove (server_get_groups_removed_callback(),
	                     grouplist_server_groups_removed_cb, NULL);

	/* remember server so that we can default to it next time */
	if (my_server!=NULL && is_nonempty_string(my_server->name))
		pan_config_set_string ("/Pan/State/Server", my_server->name);

	pan_config_sync ();
}

extern GtkAccelGroup * _main_accel_group;

static gboolean
entry_focus_in_cb (GtkWidget * w, GdkEventKey * event, gpointer user_data) {
	g_object_ref (_main_accel_group);
	gtk_window_remove_accel_group (GTK_WINDOW(Pan.window), _main_accel_group);
	return FALSE;
}
static gboolean
entry_focus_out_cb (GtkWidget * w, GdkEventKey * event, gpointer user_data) {
	gtk_window_add_accel_group (GTK_WINDOW(Pan.window), _main_accel_group);
	g_object_unref (_main_accel_group);
	return FALSE;
}

gpointer
grouplist_create (void)
{
	GtkWidget * w;
        GtkWidget * vbox;
        GtkWidget * hbox;
	GtkWidget * toolbar;
	GtkCList * clist;
	char * titles[5];
	debug_enter ("grouplist_create");

	titles[0] = "";
	titles[1] = _("Groups");
	titles[2] = _("Unread");
	titles[3] = _("Total");
	titles[4] = _("Description");

	/* create callbacks */
	grouplist_group_selection_changed = pan_callback_new ();
	grouplist_server_set_callback = pan_callback_new ();

	/* register callbacks */
	pan_callback_add (serverlist_get_server_activated_callback(),
	                  grouplist_server_activated_cb, NULL);
	pan_callback_add (group_get_groups_changed_callback(),
	                  grouplist_groups_changed_cb, NULL);
	pan_callback_add (grouplist_group_selection_changed,
	                  grouplist_selection_changed_cb, NULL);
	pan_callback_add (server_get_groups_added_callback(),
	                  grouplist_server_groups_added_cb, NULL);
	pan_callback_add (server_get_groups_removed_callback(),
	                  grouplist_server_groups_removed_cb, NULL);

	toolbar = gtk_toolbar_new ();
	gtk_toolbar_set_orientation (GTK_TOOLBAR(toolbar), GTK_ORIENTATION_HORIZONTAL);
	gtk_toolbar_set_style (GTK_TOOLBAR(toolbar), GTK_TOOLBAR_TEXT);

	vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_SMALL); 
	hbox = gtk_hbox_new (FALSE, GUI_PAD);

	Pan.group_tree = gtk_clist_new_with_titles (5, titles);
	clist = GTK_CLIST(Pan.group_tree);
	clist->button_actions[1] = clist->button_actions[0];
	gtk_clist_set_column_justification (clist, 2, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (clist, 3, GTK_JUSTIFY_RIGHT);
	grouplist_update_font ();

	w = gtk_label_new (_("Groups"));
	gtk_toolbar_append_widget(GTK_TOOLBAR(toolbar), w, NULL, NULL);
	gtk_toolbar_append_space (GTK_TOOLBAR(toolbar));

	grouplist_mode_menu = grouplist_mode_menu_create();
	gtk_box_pack_start (GTK_BOX(hbox), grouplist_mode_menu, FALSE, FALSE, 0);

	w = gtk_label_new (_("Find:"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	groupname_filter_entry = w = gtk_entry_new ();
	g_signal_connect (w, "focus_in_event", G_CALLBACK(entry_focus_in_cb), w);
	g_signal_connect (w, "focus_out_event", G_CALLBACK(entry_focus_out_cb), w);
	g_signal_connect (w, "focus_out_event", G_CALLBACK(grouplist_filter_changed_cb_nolock), w);
	g_signal_connect (w, "activate", G_CALLBACK(grouplist_filter_changed_cb_nolock), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Type in a group search string and press ENTER.  "
		  "Wildcards are allowed."), NULL);
	gtk_widget_show_all (toolbar);
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	w = gtk_scrolled_window_new (NULL, NULL);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (w),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(w), Pan.group_tree);
        gtk_box_pack_start (GTK_BOX (vbox), w, TRUE, TRUE, 0);

	g_signal_connect (Pan.group_tree, "select_row",
	                  G_CALLBACK(select_row_cb), NULL);
	g_signal_connect (Pan.group_tree, "unselect_row",
	                  G_CALLBACK(select_row_cb), NULL);
	g_signal_connect (Pan.group_tree, "button_press_event",
	                  G_CALLBACK(grouplist_button_press), NULL);
	g_signal_connect (Pan.group_tree, "key_press_event",
	                  G_CALLBACK(grouplist_key_press), NULL);


	gtk_clist_set_column_width     (clist, 0, 20);
	gtk_clist_set_column_min_width (clist, 0, 20);
	gtk_clist_set_column_max_width (clist, 0, 20);

	gtk_clist_set_column_width (clist, 1, 200);
	gtk_clist_set_column_width (clist, 2, 50);
	gtk_clist_set_column_width (clist, 3, 50);
	gtk_clist_set_column_width (clist, 4, 400);

	gtk_clist_set_selection_mode (clist, GTK_SELECTION_EXTENDED);

	/* Create the Popup Menu while we are here. */
	popup_menu = menu_create_items (group_popup_entries,
	                                G_N_ELEMENTS (group_popup_entries),
	                                "<GroupView>",
	                                &popup_factory,
	                                NULL);


	/* bootstrap */
	grouplist_set_server_nolock (serverlist_get_active_server());

	gtk_widget_show_all (vbox);
	debug_exit ("grouplist_create");
	return vbox;
}

static gchar * groupname_last_filter_string = NULL;

static gboolean
grouplist_filter_changed_cb_nolock (void)
{
	gchar * pch = gtk_editable_get_chars (GTK_EDITABLE(groupname_filter_entry), 0, -1);

	if (!pan_strcmp (pch, groupname_last_filter_string))
		g_free (pch);
	else {
		replace_gstr (&groupname_last_filter_string, pch);
		grouplist_refresh_filter_nolock ();
	}

	return FALSE;
}

/*****
******
*****/

static GdkPixmap * sub_pixmap = NULL;
static GdkBitmap * sub_bitmap = NULL;
static GdkPixmap * folder_pixmap = NULL;
static GdkBitmap * folder_bitmap = NULL;

static void
grouplist_add_group_nolock (const Group * group)
{
	gint row;
	char total_buf[16] = { '\0' };
	char unread_buf[16] = { '\0' };
	char name_buf[256] = { '\0' };
	char * freeme1 = NULL;
	char * freeme2 = NULL;
	const int total = group->article_qty;
	const int unread = MAX(0, total - group->article_read_qty);
	char * text[5];
	static gboolean inited = FALSE;

	/* inistialize the picture */
	if (!inited)
	{
		GdkPixbuf * pixbuf;

		pixbuf = gdk_pixbuf_new_from_inline (-1, icon_newsgroup, FALSE, NULL);
		gdk_pixbuf_render_pixmap_and_mask_for_colormap (pixbuf,
		                                                gtk_widget_get_colormap(Pan.group_tree),
		                                                &sub_pixmap,
		                                                &sub_bitmap, 128);
		g_object_unref (G_OBJECT(pixbuf));

		/* initialize the folder icon */
		pixbuf = gdk_pixbuf_new_from_inline (-1, icon_folder, FALSE, NULL);
		gdk_pixbuf_render_pixmap_and_mask_for_colormap (pixbuf,
		                                                gtk_widget_get_colormap(Pan.group_tree),
		                                                &folder_pixmap,
		                                                &folder_bitmap, 128);
		g_object_unref (G_OBJECT(pixbuf));

		inited = TRUE;
	}

	/* convert numbers to strings */
	g_snprintf (total_buf, sizeof(total_buf), "%d", total);
	g_snprintf (unread_buf, sizeof(unread_buf), "%d", unread);

	/* look for special qualities of the group */
	grouplist_get_group_name (group, name_buf, sizeof(name_buf));
     
	text[0] = "";
	text[1] = (char*) pan_utf8ize (name_buf, -1, &freeme1);
	text[2] = unread_buf;
	text[3] = total_buf;
	text[4] = (char*) pan_utf8ize (group->description, -1, &freeme2);

	/* add the row */
	row = gtk_clist_prepend (GTK_CLIST(Pan.group_tree), (char**)text);
	gtk_clist_set_row_data (GTK_CLIST(Pan.group_tree), row, (gpointer)group);

	/* add the pixmap, if any */
	if (group_is_folder (group))
		gtk_clist_set_pixmap (GTK_CLIST(Pan.group_tree), row, 0, folder_pixmap, folder_bitmap);
	else if (group_is_subscribed (group))
		gtk_clist_set_pixmap (GTK_CLIST(Pan.group_tree), row, 0, sub_pixmap, sub_bitmap);

	/* cleanup */
	g_free (freeme1);
	g_free (freeme2);
}

static void
grouplist_rebuild_nolock (GPtrArray * groups, GPtrArray * sel)
{
	gint i;
	GtkCList * clist = GTK_CLIST (Pan.group_tree);

	gtk_clist_freeze (clist);
	gtk_clist_clear (clist);

	for (i=groups->len-1; i>=0; --i)
	{
		Group * group = GROUP(g_ptr_array_index (groups, i));

		if (is_nonempty_string(groupname_filter_string) && fnmatch(groupname_filter_string,group->name,PAN_CASEFOLD))
			continue;

		grouplist_add_group_nolock (group);

	}

	if (sel!=NULL && sel->len!=0)
		grouplist_set_selected_groups_nolock (sel);

	gtk_clist_thaw (clist);
}

static gint
grouplist_update_ui_mainthread (gpointer unused)
{
	gchar * title;
	GtkCList * clist = GTK_CLIST(Pan.group_tree);
	GPtrArray * groups;
	GPtrArray * sel;
	debug_enter ("grouplist_update_ui_mainthread");

	/**
	***  Get the list of groups that we're going to add.
	***
	***  Note that we temporarily turn off the groups_added
	***  callback -- the groups may be loaded as a result of
	***  our call, and we don't want to update the clist twice.
	***  This is a temporary workaround that should dissapear
	***  in 0.10.x
	**/
	pan_callback_remove (server_get_groups_added_callback(),
	                     grouplist_server_groups_added_cb, NULL);
	if (group_mode == GROUP_SUBSCRIBED)
	{
		groups = my_server==NULL
			? g_ptr_array_new ()
			: server_get_groups (my_server, SERVER_GROUPS_SUBSCRIBED);
	}
	else if (group_mode == GROUP_ALL)
	{
		groups = my_server==NULL
			? g_ptr_array_new ()
			: server_get_groups (my_server, SERVER_GROUPS_ALL);
	}
	else if (group_mode == GROUP_FOLDERS)
	{
		Server * s = serverlist_get_named_server(INTERNAL_SERVER_NAME);
		groups = s==NULL
			? g_ptr_array_new()
			: server_get_groups (s, SERVER_GROUPS_ALL);
	}
	else if (group_mode == GROUP_NEW)
	{
		groups = g_ptr_array_new ();
		if (my_server != NULL) {
			guint i;
			GPtrArray * a = server_get_groups (my_server, SERVER_GROUPS_ALL);
			for (i=0; i<a->len; ++i) {
				Group * g = GROUP(g_ptr_array_index(a,i));
				if (group_is_new (g))
					g_ptr_array_add (groups, g);
			}
			g_ptr_array_free (a, TRUE);
		}
	}
	else
	{
		pan_warn_if_reached ();
		groups = g_ptr_array_new ();
	}
	pan_callback_add (server_get_groups_added_callback(),
	                  grouplist_server_groups_added_cb, NULL);

	/* get the list of selected groups */
       	sel = grouplist_get_selected_groups ();

	/* set the groups column title */
	switch (group_mode) {
		case GROUP_SUBSCRIBED: title = _("Subscribed");         break;
		case GROUP_ALL:        title = _("All Groups");         break;
		case GROUP_NEW:        title = _("New Groups");         break;
		case GROUP_FOLDERS:    title = _("Folders");            break;
		default:               title = "Bug!";                  break;
	}

	/* update UI */
	pan_lock ();
	g_static_mutex_lock (&grouplist_ui_mutex);
	gtk_clist_set_column_title (clist, 1, title);
	grouplist_rebuild_nolock (groups, sel);
	g_static_mutex_unlock (&grouplist_ui_mutex);
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (groups, TRUE);
	debug_exit ("grouplist_update_ui_mainthread");
	return 0;
}

void
grouplist_update_font (void)
{
	pan_lock ();
	if (!use_custom_fonts)
		gtk_widget_modify_font (Pan.group_tree, gtk_widget_get_default_style()->font_desc);
	else {
		PangoFontDescription * pfd = pango_font_description_from_string (group_pane_font);
		gtk_widget_modify_font (Pan.group_tree, pfd);
		pango_font_description_free (pfd);
	}
	pan_unlock ();
}

void
grouplist_refresh_nolock (void)
{
	Server * server = grouplist_get_visible_server ();

	if (server!=NULL && server==grouplist_get_visible_server())
		gui_queue_add (grouplist_update_ui_mainthread, NULL);
}

/* firing this callback is delegated to here because you should never fire
 * a callback inside a pan_lock. */
static void*
fire_server_changed_callback (void * unused)
{
	pan_callback_call (grouplist_server_set_callback, my_server, NULL);
	return NULL;
}

static void
grouplist_set_server_nolock (Server * server)
{
	Server * old_server;
	debug_enter ("grouplist_set_server_nolock");

	/* find out what server we have, and what kind of groups it has */
	old_server = my_server;
	my_server = server;
	run_in_worker_thread (fire_server_changed_callback, NULL);

	if (my_server==NULL)
	{
		if (grouplist_get_visible_server()==old_server)
			gtk_clist_clear (GTK_CLIST(Pan.group_tree));
	}
	else
	{
		gint view_mode;
		GPtrArray * groups = NULL;
		const ServerGroupsType have_type = file_grouplist_exists (my_server);

		/* if we have subbed, always load those, otherwise load all */
		if (have_type & SERVER_GROUPS_SUBSCRIBED)
		{
			view_mode = GROUP_SUBSCRIBED;

			groups = my_server==NULL
				? g_ptr_array_new ()
				: server_get_groups (my_server, SERVER_GROUPS_SUBSCRIBED);
		}
		else if (have_type)
		{
			view_mode = GROUP_ALL;

			groups = my_server==NULL
				? g_ptr_array_new ()
				: server_get_groups (my_server, SERVER_GROUPS_ALL);
		}
		else
		{
			GtkWidget * w;

                        view_mode = GROUP_ALL;
			groups = NULL;

			if (grouplist_get_visible_server()==old_server)
				grouplist_rebuild_nolock (NULL, NULL);

			w = gtk_message_dialog_new (GTK_WINDOW(Pan.window),
			                            GTK_DIALOG_DESTROY_WITH_PARENT,
			                            GTK_MESSAGE_QUESTION,
			                            GTK_BUTTONS_YES_NO,
			                            _("We don't have a list of groups for \"%s\".\nShall we get download it?"),
			                            server->name);
			gtk_dialog_set_default_response (GTK_DIALOG(w), GTK_RESPONSE_YES);
			g_signal_connect (w, "response", G_CALLBACK(grouplist_dialog_response_cb), server);
			gtk_widget_show_all (w);
		}

		/* update the UI */
		grouplist_set_view_mode_nolock_impl (view_mode);
		if (groups != NULL)
			grouplist_rebuild_nolock (groups, NULL);

		/* cleanup */
		if (groups != NULL)
			g_ptr_array_free (groups, TRUE);
	}

	debug_exit ("grouplist_set_server_nolock");
}


/***
****
****
***/

/**
 * @group: Group who's row is to be updated.
 *
 * Update the CList row for a group to show it's message counts or
 * any other relative data (subscribed/on disk) for that group.
 **/
static void
grouplist_update_group_nolock (const Group * group)
{
	GtkCList * list;
	gint row;

	/* sanity check */
	g_return_if_fail (group != NULL);

	list = GTK_CLIST (Pan.group_tree);
       	row = gtk_clist_find_row_from_data (list,(gpointer)group);
	if (row != -1)
	{
		char buf[256] = { '\0' };
		const char * utf8_name = NULL;
		char * freeme;
		const int total = group->article_qty;
		const int unread = MAX(0, total-group->article_read_qty);

		/* icon column */
		if (group_is_subscribed (group))
			gtk_clist_set_pixmap (GTK_CLIST(Pan.group_tree), row, 0, sub_pixmap, sub_bitmap);
		else if (!group_is_folder(group))
			gtk_clist_set_text (GTK_CLIST(Pan.group_tree), row, 0, "");

		/* name column */
		grouplist_get_group_name (group, buf, sizeof(buf));
		utf8_name = pan_utf8ize (buf, -1, &freeme);
		gtk_clist_set_text (list, row, 1, utf8_name);
		g_free (freeme);

		/* total column */
		g_snprintf (buf, sizeof(buf), "%d", total);
		gtk_clist_set_text (list, row, 3, buf);

		/* unread column */
		g_snprintf (buf, sizeof(buf), "%d", unread);
		gtk_clist_set_text (list, row, 2, buf);
	}
}

static gint
grouplist_update_groups_mainthread (gpointer data)
{
	guint i;
	GPtrArray * a = (GPtrArray*)data;
	GtkCList * list = GTK_CLIST (Pan.group_tree);
	debug_enter ("grouplist_update_groups_mainthread");

	g_static_mutex_lock (&grouplist_ui_mutex);
	pan_lock ();
	gtk_clist_freeze (list);
	for (i=0; i!=a->len; ++i)
		grouplist_update_group_nolock (GROUP(g_ptr_array_index(a,i)));
	gtk_clist_thaw (list);
	pan_unlock ();
	g_static_mutex_unlock (&grouplist_ui_mutex);

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("grouplist_update_groups_mainthread");
	return 0;
}

static void
grouplist_update_groups (const Group ** groups, int group_qty)
{
	gint i;
	GPtrArray * a;
	Server * server;
	debug_enter ("grouplist_update_groups");

	g_return_if_fail (groups != NULL);
	g_return_if_fail (group_qty >= 1);

	/* build an array of groups that we might need to update */
	a = g_ptr_array_new ();
	server = grouplist_get_visible_server ();
	for (i=0; i<group_qty; ++i)
		if (groups[i]!=NULL && groups[i]->server==server)
			g_ptr_array_add (a, (gpointer)groups[i]);

	/* if we have any that we should update, queue it */
	if (a->len != 0)
		gui_queue_add (grouplist_update_groups_mainthread, a);
	else
		g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_update_groups");
}

/***
****
****
***/

static void
grouplist_internal_select_row (gint      row,
                               gboolean  activate,
                               gboolean  lock)
{
        GtkCList* list = GTK_CLIST(Pan.group_tree);

	if (activate)
		button_click_count = 2;

	if (lock) pan_lock ();
	gtk_clist_freeze (list);
	gtk_clist_unselect_all (list);
	gtk_clist_select_row (list,row,-1);
	gtk_clist_thaw (list);
	if (lock) pan_unlock ();
}

static gboolean
grouplist_select_row_if_unread (int row,
                                gboolean activate,
                                gboolean lock)
{
        GtkCList * list = GTK_CLIST(Pan.group_tree);
	Group * g = gtk_clist_get_row_data(list,row);
	const gulong total = g->article_qty;
	const gulong read = g->article_read_qty;
	gboolean retval = FALSE;

	if (read < total) {
		grouplist_internal_select_row (row, activate, lock);
		retval = TRUE;
	}

	return retval;
}
static void
grouplist_next_unread_group (gboolean activate)
{
        GtkCList* list;
	int row_qty, i, row=0;

	pan_lock ();
	list = GTK_CLIST(Pan.group_tree);
        row_qty = list->rows;

	if (row_qty != 0)
	{
		/* get the first row to check */
		row = 0;
		if (list->selection) {
			int sel = GPOINTER_TO_INT(list->selection->data);
			row = (sel + 1) % row_qty;
		}

		/* walk through */
		for ( i=0; i<row_qty; ++i, row=(row+1)%row_qty)
			if (grouplist_select_row_if_unread (row, activate, 0))
				break;
	}

	pan_unlock ();
}
void
grouplist_select_next_unread_group (void)
{
	grouplist_next_unread_group (FALSE);
}
void
grouplist_activate_next_unread_group (void)
{
	grouplist_next_unread_group (TRUE);
}
static void
grouplist_next_group (gboolean activate)
{
        GtkCList* list;
	int row_qty;
	int row = 0;

	pan_lock ();
	list = GTK_CLIST(Pan.group_tree);
        row_qty = list->rows;

	if (row_qty != 0)
	{
		/* get the first row to check */
		row = 0;
		if (list->selection) {
			int sel = GPOINTER_TO_INT(list->selection->data);
			row = (sel + 1) % row_qty;
		}

		grouplist_internal_select_row (row, activate, 0);
	}

	pan_unlock ();
}
void
grouplist_select_next_group (void)
{
	grouplist_next_group (FALSE);
}
void
grouplist_activate_next_group (void)
{
	grouplist_next_group (TRUE);
}

/***
****
***/

static void
group_menu_popup (GdkEventButton *event)
{
	GtkItemFactory * gif = popup_factory;
	const Group * group = grouplist_get_selected_group ();
	const gboolean have_group = group!=NULL;
	debug_enter ("group_menu_popup");

	menu_set_sensitive (gif, "/Mark Group Read", have_group);
	menu_set_sensitive (gif, "/Delete Group's Articles", have_group);
	menu_set_sensitive (gif, "/Get New Headers", have_group);
	menu_set_sensitive (gif, "/Get New Headers and Bodies", have_group);
	menu_set_sensitive (gif, "/More Download Options...", have_group);
	menu_set_sensitive (gif, "/Refresh Article Counts", have_group);
	menu_set_sensitive (gif, "/Subscribe", have_group);
	menu_set_sensitive (gif, "/Unsubscribe", have_group);
	menu_set_sensitive (gif, "/Group Properties...", have_group);
	menu_set_sensitive (gif, "/Delete Group", have_group);

	gtk_menu_popup (GTK_MENU(popup_menu),
	                NULL, NULL, NULL, NULL,
	                event->button, event->time);

	debug_exit ("group_menu_popup");
}

static void
grouplist_refresh_filter_nolock (void)
{
	gchar * new_filter;
        gchar * pch;

	pch  = gtk_editable_get_chars (GTK_EDITABLE(groupname_filter_entry), 0, -1);

	/* build the new filter string */
	new_filter = NULL;
	if (is_nonempty_string (pch)) {
		if (strchr(pch,'*') != NULL)
			new_filter = g_strdup (pch);
		else 
			new_filter = g_strdup_printf("*%s*",pch);
	}

	/* if it differs from the old filter string, refresh */
	if (pan_strcmp (new_filter, groupname_filter_string)) {
		replace_gstr (&groupname_filter_string, g_strdup(new_filter));
		grouplist_refresh_nolock ();
	}

	/* cleanup */
	g_free (pch);
	g_free (new_filter);
}
