/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#ifdef HAVE_LIBGTKSPELL
#include <gtkspell/gtkspell.h>
#endif

#include <locale.h>

#include <pan/base/acache.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/newsrc-port.h>
#include <pan/base/pan-config.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>
#include <pan/filters/filter-manager.h>
#include <pan/rules/rule-manager.h>
#include <pan/identities/identity-manager.h>
#include <pan/dialogs/dialogs.h>

#include <gmime/gmime.h>

#include <pan/flagset.h>
#include <pan/group-action.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/pan.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-xml.h>
#include <pan/thread-watch.h>
#include <pan/util.h>

extern char *
bind_textdomain_codeset (const char * domainname,
                         const char * codeset);

static int
restore_tasks_dialog (gpointer filename_gpointer);


#ifdef GNOME
/***
****   Session management
***/

/*
** save_session
** saves the current session for later revival
*/
static gint
save_session (GnomeClient         * client,
              gint                  phase,
              GnomeSaveStyle        save_style,
              gint                  is_shutdown,
              GnomeInteractStyle    interact_style,
              gint                  is_fast,
              gpointer              client_data)
{
	gchar **argv;
	guint argc;

	argv = g_new0 (gchar*, 4);
	argc = 1;

	argv[0] = client_data;
	/* add any addtional state info here. */
	gnome_client_set_clone_command (client, argc, argv);
	gnome_client_set_restart_command (client, argc, argv);
	return TRUE;
}

/**
*** session_die: Gracefullly end the session
**/
static void
session_die (GnomeClient *    client,
             gpointer         client_data)
{
	pan_shutdown ();
}
#endif

static void
show_help (void)
{
	const char * help = _(
		"Pan " VERSION "\n"
		"\n"
		"  --version        Prints Pan's version number\n"
		"  --help           Prints this message and exits\n"
		"  --debug          Turns on the most commonly-useful options for a bug report\n"
		"\n"
		"  --mute           Debugging tool to send new posts to console, not the server\n"
		"  --debug-cache    Print debugging messages for the article cache\n"
		"  --debug-sockets  Print debugging messages when talking to the news server\n"
		"  --debug-queue    Print debugging messages related to managing tasks\n"
		"  --debug-decode   Print debugging messages related to decoding attachments\n"
		"  --debug-newsrc   Print debugging messages related to .newsrc files\n"
		"  --debug-gnksa    Print debugging messages related to gnksa correctness\n"
		"  --debug-trace    Print debugging messages when entering/leaving functions\n"
		"  --debug-lock     Print debugging messages related to threaded gui locks\n"
		"  --debug-object   Print debugging messages related to refcounted objects\n"
		"\n");
	fprintf (stderr, "%s", help);
}

const gchar* sockread_err_msg = NULL;
const gchar* sockwrite_err_msg = NULL;

PanApp Pan;

/*---[ main ]---------------------------------------------------------
 * _the_ main function!
 *--------------------------------------------------------------------*/
int
main (int argc, char *argv[])
{
	int i;
	gulong debug_flags = 0;
	gboolean mute = FALSE;

	/* init the text domain & codeset,
	   then force all of the locale segments to update from the
	   environment.  Unless we do this they will all default to C. */
	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
	setlocale (LC_ALL, "");

	/* set some globals */
	sockread_err_msg = _("Error reading from socket.");
	sockwrite_err_msg = _("Error writing to socket.");
	PAN_SENT = _("pan.sent");
	PAN_SENDLATER = _("pan.sendlater");

	/* init gtk */
	g_set_prgname ("Pan");
	gtk_set_locale ();
	g_thread_init (NULL);
	gdk_threads_init ();
	gtk_init (&argc, &argv);
	g_mime_init (GMIME_INIT_FLAG_UTF8);

	/* parse arguments  */
	mute = FALSE;
	debug_flags = 0;
	for (i=0; i<argc; ++i)
	{
		const char * arg = argv[i];

		if (!strcmp(arg,"--mute"))
			mute = TRUE;
		else if (!strcmp (arg, "--debug-cache"))
			debug_flags |= DEBUG_ACACHE;
		else if (!strcmp (arg, "--debug-socket-input"))
			debug_flags |= DEBUG_SOCKET_INPUT;
		else if (!strcmp (arg, "--debug-socket-output"))
			debug_flags |= DEBUG_SOCKET_OUTPUT;
		else if (!strcmp (arg, "--debug-socket"))
			debug_flags |= DEBUG_SOCKET_INPUT|DEBUG_SOCKET_OUTPUT;
		else if (!strcmp (arg, "--debug-sockets"))
			debug_flags |= DEBUG_SOCKET_INPUT|DEBUG_SOCKET_OUTPUT;
		else if (!strcmp (arg, "--debug-queue"))
			debug_flags |= DEBUG_QUEUE;
		else if (!strcmp (arg, "--debug-decode"))
			debug_flags |= DEBUG_DECODE;
		else if (!strcmp (arg, "--debug-newsrc"))
			debug_flags |= DEBUG_NEWSRC;
		else if (!strcmp (arg, "--debug-gnksa"))
			debug_flags |= DEBUG_GNKSA;
		else if (!strcmp (arg, "--debug-trace"))
			debug_flags |= DEBUG_TRACE;
		else if (!strcmp (arg, "--debug-lock"))
			debug_flags |= DEBUG_LOCK;
		else if (!strcmp (arg, "--debug-object"))
			debug_flags |= DEBUG_PAN_OBJECT;
		else if (!strcmp (arg, "--debug"))
			debug_flags |= DEBUG_TRACE|DEBUG_DECODE|DEBUG_ACACHE|DEBUG_SOCKET_INPUT|DEBUG_SOCKET_OUTPUT;
		else if (!strcmp (arg, "--version")) {
			printf ("Pan %s\n", VERSION);
			return 0;
		}
		else if (!strcmp (arg, "--help") || !strcmp(arg, "-help")) {
			show_help ();
			return 0;
		}
	}
	set_debug_level (debug_flags);
	log_add_va (LOG_INFO, _("Pan %s Started"), VERSION);

	cmap = gdk_colormap_get_system ();
	flagset_init ();
	prefs_init ();
	serverlist_init ();
	filter_manager_init ();
#ifdef HAVE_LIBGTKSPELL
	if (do_spellcheck) {
		const int status = gtkspell_init ();
		do_spellcheck = status >= 0;
		if (!do_spellcheck)
			log_add_va (LOG_ERROR,
			            _("Spellchecking disabled: gtkspell_init() failed with status %d"),
			            status);
	}
#endif

	/* Connect up to the session manager */
#ifdef GNOME
	if (1) {
		GnomeClient * client = gnome_master_client ();
		g_signal_connect (client, "save_yourself",
		                  G_CALLBACK(save_session), argv[0]);
		g_signal_connect (client, "die",
		                  G_CALLBACK(session_die), NULL);
	}
#endif

	/* initialize variables in the global app structure */
	Pan.main_t = g_thread_self ();
	pan_mute = mute;
	if (pan_mute)
		g_message ("Running in mute mode: messages will be sent to "
		           "stdout instead of smtp/nntp server");

	queue_init (pan_config_get_int("/Pan/General/Max_Connections_Total=4"),
	            pan_config_get_int("/Pan/General/Max_Task_Tries_Total=4"));

	/* if it's a new user, pop up the newuser dialog.
	   Otherwise show the main window. */

	if (identity_manager_count () == 0)
	{
		dialog_newuser ();
	}
	else
	{
		gui_construct (NULL);
		gtk_widget_show_all (Pan.window);
	}

	if (fetch_new_and_bodies_on_startup)
		group_action_subscribed_download_new_and_bodies ();
	else if (fetch_new_on_startup)
		group_action_subscribed_download_new ();

	if (1) {
		char * tasks_filename = queue_get_tasks_filename ();
		if (g_file_test (tasks_filename, G_FILE_TEST_EXISTS))
			gui_queue_add (restore_tasks_dialog, tasks_filename);
		else
			g_free (tasks_filename);
	}

	gtk_main ();

	return 0;
}

/**
 * pan_shutdown:
 *
 * Shutdown Pan, closing all TCP connections and saving all preferences
 **/
void
pan_shutdown (void)
{

	queue_shutdown ();

	g_free (message_body_font);

	identity_manager_shutdown_module ();
	serverlist_shutdown ();
	acache_shutdown ();
	thread_shutdown_module ();
	rule_manager_shutdown_module ();
	log_shutdown_module ();

	pan_object_bookkeeping ();

	gtk_main_quit ();
}

/***
****
***/

static int
restore_tasks_dialog (gpointer filename_gpointer)
{
	int response;
	GPtrArray * tasks;
	GtkWidget * dialog;
	char * filename = (char*) filename_gpointer;

	pan_lock ();
	dialog = gtk_message_dialog_new (GTK_WINDOW(Pan.window),
	                                 GTK_DIALOG_DESTROY_WITH_PARENT,
	                                 GTK_MESSAGE_QUESTION,
	                                 GTK_BUTTONS_NONE,
					 _("Pan appears to have crashed or been killed the last time it was run.  Do you want to revert to the last session's task list?"));
	gtk_dialog_add_buttons (GTK_DIALOG(dialog),
	                        GTK_STOCK_REVERT_TO_SAVED,
	                        GTK_RESPONSE_ACCEPT,
	                        GTK_STOCK_NO,
	                        GTK_RESPONSE_REJECT,
	                        NULL);
	gtk_widget_show_all (dialog);
	response = gtk_dialog_run (GTK_DIALOG(dialog));
	gtk_widget_destroy (dialog);
	pan_unlock ();

	/* if the user wants to restore, load the old task file */
	tasks = g_ptr_array_new ();
	if (response == GTK_RESPONSE_ACCEPT)
		task_xml_read (filename, tasks);

	/* delete the old task file */
	unlink (filename);

	/* if tasks were read, add them to the queue */
	if (tasks->len) {
		guint i;
		GSList * l = NULL;
		for (i=0; i<tasks->len; ++i)
			l = g_slist_prepend (l, g_ptr_array_index(tasks,i));
		l = g_slist_reverse (l);
		queue_insert_tasks (l, -1);
		log_add_va (LOG_INFO, _("Restored %d tasks from \"%s\""), (int)tasks->len, filename);
	}

	/* cleanup */
	g_ptr_array_free (tasks, TRUE);
	g_free (filename);
	return 0;
}
