/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>

#include <pan/base/acache.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/pan-config.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>

#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/gui-paned.h>
#include <pan/nntp.h>
#include <pan/pan-color-picker.h>
#include <pan/pan-font-picker.h>
#include <pan/pan-file-entry.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/server-menu-ui.h>
#include <pan/task-headers.h>
#include <pan/text.h>
#include <pan/util.h>

#include <pan/xpm/pan-pixbufs.h>

#define NORMAL_PANGO_DESCRIPTION "sans 11"
#define MONOSPACE_PANGO_DESCRIPTION  "courier 11"


typedef struct
{
	GtkWidget * dialog;
	GtkWidget * notebook;

	/* smtp server */
	GtkWidget * smtp_address;
	GtkWidget * smtp_port;

	/* cache */
	GtkWidget * cache_megs_sb;
	GtkWidget * flush_cache_on_exit_check;

	/* general */
	GtkWidget * single_click_selects_cbutton;
	GtkWidget * remove_failed_tasks_cbutton;
	GtkWidget * fetch_new_on_group_load_cbutton;
	GtkWidget * fetch_new_on_startup_cbutton;
	GtkWidget * fetch_new_and_bodies_on_startup_cbutton;
	GtkWidget * break_thread_when_subject_changes_cbutton;
	GtkWidget * expand_all_threads_by_default_cbutton;
	GtkWidget * hide_mpart_child_nodes_cbutton;
	GtkWidget * external_editor_combo;
	GtkWidget * external_browser_combo;

	/* directories */
	GtkWidget * dir_download;
	GtkWidget * dir_temp;
	GtkWidget * dir_data;

	/* display */
	GtkWidget * use_custom_fonts_check;
	GtkWidget * group_pane_font_lb;
	GtkWidget * group_pane_font_gfp;
	GtkWidget * header_pane_font_lb;
	GtkWidget * header_pane_font_gfp;
	GtkWidget * message_lb;
	GtkWidget * message_gfp;
	GtkWidget * message_fixed_gfp;
	GtkWidget * ignored_pcp;
	GtkWidget * watched_pcp;
	GtkWidget * text_fg_pcp;
	GtkWidget * text_bg_pcp;
	GtkWidget * text_url_pcp;
	GtkWidget * text_quoted_pcp[3];
	GtkWidget * text_quoted_chars_entry;
	GtkWidget * thread_date_entry;
	GtkWidget * smooth_scrolling_check;
	GtkWidget * smooth_scrolling_speed_sb;

	/* layout */
	GtkWidget * layout_page;
	GtkWidget * display_page;

	/* connections */
	GtkWidget  * max_connections;
	GtkWidget  * max_tries;
}
PrefsWindow;


static char* layout_get_new_string (GtkWidget * layout_page);
static gulong get_header_flags (void);

static PrefsWindow * win;

extern GtkTooltips * ttips;

static char * text_quote_chars                = NULL;
gboolean collapse_group_names                 = FALSE;
gboolean use_system_bg                        = TRUE;
gboolean use_system_fg                        = TRUE;
gboolean text_window_smooth_scrolling         = TRUE;
gboolean do_spellcheck                        = TRUE;
int      text_window_smooth_scrolling_speed   = 10;

char * thread_date_format                     = NULL;
char * layout_str                             = NULL;
int wrap_column                               = 74;
int mail_server_port                          = 0;
gboolean use_custom_fonts                     = FALSE;
gboolean text_use_fixed_font                  = FALSE;
gboolean expand_all_threads_by_default        = FALSE;
gboolean hide_mpart_child_nodes               = TRUE;
gboolean pan_mute                             = FALSE;
gboolean fetch_new_on_group_load              = TRUE;
gboolean fetch_new_on_startup                 = FALSE;
gboolean fetch_new_and_bodies_on_startup      = FALSE;
gboolean header_pane_is_threaded              = TRUE;
gboolean remove_failed_tasks                  = TRUE;
gboolean single_click_selects                 = FALSE;
gboolean show_group_pane                      = TRUE;
gboolean show_header_pane                     = TRUE;
gboolean show_body_pane                       = TRUE;

char * group_pane_font = NULL;
char * header_pane_font = NULL;
char * message_body_font = NULL;
char * message_body_font_fixed = NULL;
char * attribution_line = NULL;
char * external_editor = NULL;
char * external_web_browser = NULL;
char * mail_server_address = NULL;


static void
set_color (GdkColor      * color,
           const char    * key,
           guint           r,
           guint           g,
           guint           b)
{
	char buf[1024];

	g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_r=%u", key, r);
	color->red = pan_config_get_int (buf);

	g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_g=%u", key, g);
	color->green = pan_config_get_int (buf);

	g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_b=%u", key, b);
	color->blue = pan_config_get_int (buf);

	if (!gdk_color_alloc (cmap, color))
		g_error ("couldn't allocate \"%s\" color", key);
}

static void
date_help_clicked_cb (void)
{
	const char * str = _("%a - abbreviated weekday name\n"
	                     "%A - full weekday name\n"
	                     "%b - abbreviated month name\n"
	                     "%B - full month name\n"
	                     "%c - local date & time\n"
	                     "%d - day of the month\n"
	                     "%H - hour (24-hour clock)\n"
	                     "%I - hour (12-hour clock)\n"
	                     "%j - day of the year (001-366)\n"
	                     "%m - month (01-12)\n"
	                     "%M - minute (00-59)\n"
	                     "%p - local equivalent of AM/PM\n"
	                     "%S - second (00-61)\n"
	                     "%x - local date\n"
	                     "%X - local time\n"
	                     "%y - two-digit year\n"
	                     "%Y - four-digit year\n"
	                     "%% - %");
	GtkWidget * w = gtk_message_dialog_new (GTK_WINDOW(win->dialog),
	                                        GTK_DIALOG_DESTROY_WITH_PARENT,
	                                        GTK_MESSAGE_INFO,
	                                        GTK_BUTTONS_CLOSE, "%s", str);
	g_signal_connect_swapped (GTK_OBJECT(w), "response",
	                          G_CALLBACK(gtk_widget_destroy), GTK_OBJECT (w));
	gtk_widget_show_all (w);
}

static void
pan_prefs_changed (GtkDialog * dialog)
{
        gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_OK, TRUE);
        gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_APPLY, TRUE);
}

#define connect_signal_to_prefs_changed(object,signal_name) \
	g_signal_connect_swapped (object, signal_name, G_CALLBACK(pan_prefs_changed), win->dialog)

/**
***  UPDATE UTILS
**/

static gboolean
update_entry_and_bool_from_toggle_button (gboolean * setme,
                                          const char * key,
                                          GtkWidget * toggle)
{
	gboolean changed = FALSE;
	const gboolean b = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON(toggle));

	if (b != *setme) {
		*setme = b;
		pan_config_set_bool (key, b);
		changed = TRUE;
	}

	return changed;
}

static void
set_config_from_editable (const char   * key,
                          GtkWidget    * entry)
{
	char * text = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	g_strstrip (text);
	if (is_nonempty_string(text))
		pan_config_set_string (key, text);
	else
		pan_config_clean_key (key);
	g_free (text);
}

static void
replace_string_and_config_from_editable (char        ** setme,
                                         const char   * key,
                                         GtkWidget    * e)
{
	char * s = gtk_editable_get_chars (GTK_EDITABLE(e), 0, -1);
	g_strstrip (s);
	if (!pan_strcmp(*setme, s))
		g_free (s);
	else {
		pan_config_set_string (key, s);
		replace_gstr (setme, s);
	}
}

static gboolean
handle_font_picker (GtkWidget * pfp, char ** p_old_font, const char * config_key)
{
	gboolean changed = FALSE;

	/* get new... */
	char * new_font = pan_font_picker_get_font (pfp);

	/* compare old with new... */
	if (pan_strcmp (*p_old_font, new_font)) {
		replace_gstr (p_old_font, g_strdup(new_font));
		pan_config_set_string (config_key, new_font);
		changed = TRUE;
	}

	g_free (new_font);
	return changed;
}

static gboolean
handle_color_picker (const char    * key,
                     GdkColor      * color,
                     GtkWidget     * w)
{
	GdkColor picker_color;
	gboolean color_changed = FALSE;

	pan_color_picker_get_color (w, &picker_color);

	color_changed = (color->red != picker_color.red)
		     || (color->green != picker_color.green)
		     || (color->blue != picker_color.blue);

	if (color_changed)
	{
		char buf[1024];

		color_changed = TRUE;

		/* update the color */
		gdk_colormap_free_colors (cmap, color, 1);
		color->red = picker_color.red;
		color->green = picker_color.green;
		color->blue = picker_color.blue;
		if (!gdk_color_alloc (cmap, color))
			g_error ("couldn't allocate \"%s\" color", key);

		/* update the config */
		g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_r", key);
		pan_config_set_int (buf, picker_color.red);
		g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_g", key);
		pan_config_set_int (buf, picker_color.green);
		g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_b", key);
		pan_config_set_int (buf, picker_color.blue);
	}

	return color_changed;
}

/**
***  UPDATE EVERYTHING
**/



/* Ok or Apply pressed in the Preferences dialog, save all information */
static void
prefs_apply (void)
{
	gboolean articlelist_changed = FALSE;
	gboolean server_connections_changed = FALSE;
	gboolean color_changed;
	gboolean b;

	/* Commit all of the data to the config file */

	/* No entry for these yet
	 * pan_config_set_int("/Pan/Articles/Expiration", 30);
	 */

	/* thread date format */
	if (1) {
		char * tmp = g_strdup (thread_date_format);
		replace_string_and_config_from_editable (
			&thread_date_format,
			"/Pan/Display/Thread_Date_Format",
			win->thread_date_entry);
		if (pan_strcmp (tmp, thread_date_format))
			articlelist_changed = TRUE;
		g_free (tmp);
	}

	set_config_from_editable (
		"/Pan/Paths/download_dir",
		pan_file_entry_gtk_entry(win->dir_download));

	/**
	***  Mail Server
	**/

	replace_string_and_config_from_editable (&mail_server_address,
	                                         "/Pan/Mail/smtp_address",
	                                         win->smtp_address);

	if (1) {
		int i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(win->smtp_port));
		if (i != mail_server_port) {
			pan_config_set_int ("/Pan/Mail/smtp_port", i);
			mail_server_port = i;
		}
	}

	/**
	***  Windows
	**/

	/* update the articlelist colors */
	articlelist_changed |= handle_color_picker ("killfile_color", &killfile_color, win->ignored_pcp);
	articlelist_changed |= handle_color_picker ("watched_color", &watched_color, win->watched_pcp);

	/* text window colors */
	b = 0;
	b |= handle_color_picker ("text_url_color", &text_url_color, win->text_url_pcp);
	b |= handle_color_picker ("text_quoted_color_1", &text_quoted_color[0], win->text_quoted_pcp[0]);
	b |= handle_color_picker ("text_quoted_color_2", &text_quoted_color[1], win->text_quoted_pcp[1]);
	b |= handle_color_picker ("text_quoted_color_3", &text_quoted_color[2], win->text_quoted_pcp[2]);
	color_changed = b;

	/* quote prefix characters */
	if (1) {
		char * tmp = g_strdup (text_quote_chars);
		replace_string_and_config_from_editable (
			&text_quote_chars,
			"/Pan/Display/text_quote_chars",
			win->text_quoted_chars_entry);
		if (pan_strcmp (tmp, text_quote_chars))
			color_changed = TRUE;
		g_free (tmp);
	}

	if (color_changed)
	{
		GtkStyle * style;

		style = gtk_rc_get_style (Pan.text);

		style = style != NULL ? gtk_style_copy (style) 
				      : gtk_style_new ();

		gtk_widget_set_style (Pan.text, style);

		text_set_font ();
	}

	update_entry_and_bool_from_toggle_button (
		&remove_failed_tasks,
		"/Pan/General/remove_failed_tasks",
		win->remove_failed_tasks_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_on_group_load,
		"/Pan/General/fetch_new_on_group_load",
		win->fetch_new_on_group_load_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_on_startup,
		"/Pan/General/fetch_new_from_subscribed_on_startup",
		win->fetch_new_on_startup_cbutton);

	update_entry_and_bool_from_toggle_button (
		&single_click_selects,
		"/Pan/General/single_click_selects",
		win->single_click_selects_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_and_bodies_on_startup,
		"/Pan/General/fetch_new_and_bodies_from_subscribed_on_startup",
		win->fetch_new_and_bodies_on_startup_cbutton);

	/**
	***  Font Pickers
	**/

	if (update_entry_and_bool_from_toggle_button (&use_custom_fonts,
	                                              "/Pan/Display/use_custom_fonts",
	                                              win->use_custom_fonts_check))
	{
		articlelist_changed = TRUE;
		grouplist_update_font ();
		text_set_font ();
	}

	if (handle_font_picker (win->header_pane_font_gfp,
	                        &header_pane_font,
	                        "/Pan/Display/header_pane_font"))
		articlelist_changed = TRUE;

	if (handle_font_picker (win->group_pane_font_gfp,
	                        &group_pane_font,
	                        "/Pan/Display/group_pane_font"))
		grouplist_update_font ();

	b = handle_font_picker (win->message_gfp,
	                        &message_body_font,
	                        "/Pan/Display/message_body_font_norm");

	b |= handle_font_picker (win->message_fixed_gfp,
	                        &message_body_font_fixed,
	                        "/Pan/Display/message_body_font_fixed");

	if (b)
		text_set_font ();


	/* cache */
	if (1) {
		acache_max_megs = gtk_spin_button_get_value_as_int (
			GTK_SPIN_BUTTON(win->cache_megs_sb));
		pan_config_set_int ("/Pan/Cache/MaxMegs", acache_max_megs);
		acache_expire ();
	}
	pan_config_set_bool ("/Pan/Cache/FlushOnExit",
			gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON( win->flush_cache_on_exit_check)));

	/* article headers */
	if (1) {
		gulong flags = get_header_flags ( );
		if (header_flags != flags) {
			pan_config_set_int ("/Pan/State/Headers", flags);
			header_flags = flags;
			text_refresh ();
		}
	}
	
	/**
	***  Filling
	**/

	if (update_entry_and_bool_from_toggle_button (
		&break_thread_when_subject_changes,
		"/Pan/General/break_thread_when_subject_changes",
		win->break_thread_when_subject_changes_cbutton))
			articlelist_changed = TRUE;

	if (update_entry_and_bool_from_toggle_button (
		&expand_all_threads_by_default,
		"/Pan/General/expand_all_threads_by_default",
		win->expand_all_threads_by_default_cbutton))
			articlelist_changed = TRUE;

	if (update_entry_and_bool_from_toggle_button (
		&hide_mpart_child_nodes,
		"/Pan/General/hide_mpart_child_nodes",
		win->hide_mpart_child_nodes_cbutton))
			articlelist_changed = TRUE;


	update_entry_and_bool_from_toggle_button (&text_window_smooth_scrolling,
	                                          "/Pan/Display/smooth_scrolling",
	                                          win->smooth_scrolling_check);

	if (1) {
		GtkSpinButton* w = GTK_SPIN_BUTTON(
			win->smooth_scrolling_speed_sb);
		const gint i = gtk_spin_button_get_value_as_int (w);
		if (i != text_window_smooth_scrolling_speed) {
			text_window_smooth_scrolling_speed = i;
			pan_config_set_int (
				"/Pan/Display/smooth_scrolling_speed_2", i);

		}
	}

	replace_string_and_config_from_editable (
		&external_editor,
		"/Pan/User/external_editor_2",
		GTK_COMBO(win->external_editor_combo)->entry);

	replace_string_and_config_from_editable (
		&external_web_browser,
		"/Pan/User/external_web_browser",
		GTK_COMBO(win->external_browser_combo)->entry);

	/* maximum number of connections */
	if (1) {
		GtkSpinButton * sb = GTK_SPIN_BUTTON (win->max_connections);
		const int i = gtk_spin_button_get_value_as_int (sb);
		if (i != queue_get_max_sockets()) {
			queue_set_max_sockets (i);
			server_connections_changed = TRUE;
			pan_config_set_int ("/Pan/General/Max_Connections_Total", i);
		}
	}

	/* maximum number of tries */
	if (1) {
		GtkSpinButton * sb = GTK_SPIN_BUTTON (win->max_tries);
		const int i = gtk_spin_button_get_value_as_int (sb);
		if (i != queue_get_max_tries()) {
			queue_set_max_tries (i);
			server_connections_changed = TRUE;
			pan_config_set_int ("/Pan/General/Max_Task_Tries_Total", i);
		}
	}

	/* layout */
	if (1) {
		char * str = layout_get_new_string (win->layout_page);
		if (pan_strcmp (str, layout_str)) {
			replace_gstr (&layout_str, g_strdup(str));
			pan_config_set_string ("/Pan/Display/Layout", layout_str);
			gui_layout_refresh ();
		}
		g_free (str);
	}

	/**
	***
	***
	**/

	if (articlelist_changed)
		articlelist_reset_style_nolock ();

	if (server_connections_changed)
		queue_wakeup ();

	pan_config_sync ();

	prefs_init ();
	gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_CANCEL, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_APPLY, FALSE);
}


static void
ensure_trailing_slash (char ** pch)
{
	char * m = strrchr (*pch, G_DIR_SEPARATOR);
	if (!m || m[1]!='\0')
		replace_gstr (pch, g_strdup_printf ("%s%c", *pch, G_DIR_SEPARATOR));
}

/**
 * This is where all the user preferences get loaded in.
 */
void
prefs_init (void)
{
	debug_enter ("prefs_init");

	/* get the layout string */
	replace_gstr (&layout_str, pan_config_get_string ("/Pan/Display/Layout"));
	if (!is_nonempty_string (layout_str))
		replace_gstr (&layout_str, g_strdup ("4gta"));

	/* get download directory */
	replace_gstr (&download_dir, pan_config_get_string ("/Pan/Paths/download_dir"));
	if (is_nonempty_string (download_dir))
		ensure_trailing_slash (&download_dir);
	else
		download_dir = g_strdup_printf ("%s/News/Pan/", g_get_home_dir());


	acache_max_megs = pan_config_get_int ("/Pan/Cache/MaxMegs=10");

	/* general preferences */

	/* display preferences */
	pan_config_push_prefix ("/Pan/Display/");
	header_pane_is_threaded = pan_config_get_bool("header_pane_is_threaded=TRUE");
	show_group_pane = pan_config_get_bool("show_group_pane=TRUE");
	show_header_pane = pan_config_get_bool("show_header_pane=TRUE");
	show_body_pane = pan_config_get_bool("show_body_pane=TRUE");
	text_set_wrap (pan_config_get_bool("do_wrap=FALSE"));
	collapse_group_names = pan_config_get_bool("collapse_group_names=FALSE");
	text_window_smooth_scrolling = pan_config_get_bool("smooth_scrolling=TRUE");
	text_window_smooth_scrolling_speed = pan_config_get_int("smooth_scrolling_speed_2=10");
	text_use_fixed_font = pan_config_get_bool("use_fixed_font=FALSE");
	use_custom_fonts = pan_config_get_bool ("use_custom_fonts=FALSE");
	replace_gstr (&text_quote_chars, pan_config_get_string ("text_quote_chars"));
	if (!is_nonempty_string (text_quote_chars))
		replace_gstr (&text_quote_chars, g_strdup (">:}|"));
	pan_config_pop_prefix ();

	/* general preferences */
	pan_config_push_prefix ("/Pan/General/");
	single_click_selects = pan_config_get_bool ("single_click_selects=FALSE");
	remove_failed_tasks = pan_config_get_bool ("remove_failed_tasks=TRUE");
	wrap_column = pan_config_get_int ("wrap_column=74");
	fetch_new_on_group_load = pan_config_get_bool ("fetch_new_on_group_load=TRUE");
	fetch_new_on_startup = pan_config_get_bool ("fetch_new_from_subscribed_on_startup=FALSE");
	fetch_new_and_bodies_on_startup = pan_config_get_bool ("fetch_new_and_bodies_from_subscribed_on_startup=FALSE");
	hide_mpart_child_nodes = pan_config_get_bool ("hide_mpart_child_nodes=TRUE");
	break_thread_when_subject_changes = pan_config_get_bool (
			"break_thread_when_subject_changes=FALSE"); 
	expand_all_threads_by_default = pan_config_get_bool (
			"expand_all_threads_by_default=FALSE");
	pan_config_pop_prefix ();


	header_flags = (gulong) pan_config_get_int ("/Pan/State/Headers=243");

	/**
	***  Unsorted
	**/

	do_spellcheck = pan_config_get_bool ("/Pan/Compose/do_spellcheck=true");

	replace_gstr (&thread_date_format, pan_config_get_string ("/Pan/Display/Thread_Date_Format"));
	if (!is_nonempty_string (thread_date_format))
		replace_gstr (&thread_date_format, g_strdup ("%x %X"));

	replace_gstr (&external_editor, pan_config_get_string ("/Pan/User/external_editor_2"));
	if (!external_editor)
		external_editor = g_strdup ("xterm -e vi %t");

	replace_gstr (&external_web_browser, pan_config_get_string ("/Pan/User/external_web_browser"));
	if (!external_web_browser) {
		char * browser = getenv ("BROWSER");
		if (!is_nonempty_string(browser))
			browser = "netscape %s";
		external_web_browser = g_strdup (browser);
	}

	/* mail server preferences */
	replace_gstr (&mail_server_address, pan_config_get_string ("/Pan/Mail/smtp_address"));
	mail_server_port = pan_config_get_int ("/Pan/Mail/smtp_port=25");

	/**
	***  Fonts
	**/

	replace_gstr (&group_pane_font, pan_config_get_string ("/Pan/Display/group_pane_font"));
	if (!is_nonempty_string (group_pane_font))
		replace_gstr (&group_pane_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&header_pane_font, pan_config_get_string ("/Pan/Display/header_pane_font"));
	if (!is_nonempty_string (header_pane_font))
		replace_gstr (&header_pane_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&message_body_font, pan_config_get_string ("/Pan/Display/message_body_font_norm"));
	if (!is_nonempty_string(message_body_font))
		replace_gstr (&message_body_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&message_body_font_fixed, pan_config_get_string ("/Pan/Display/message_body_font_fixed"));
	if (!is_nonempty_string(message_body_font_fixed))
		replace_gstr (&message_body_font_fixed, g_strdup(MONOSPACE_PANGO_DESCRIPTION));

	/**
	***  Colors
	**/

	use_system_bg = pan_config_get_bool ("/Pan/Display/use_system_bg=true");
	use_system_fg = pan_config_get_bool ("/Pan/Display/use_system_fg=true");
	set_color (&killfile_color,        "killfile_color",        35723, 17733,  4883);
	set_color (&watched_color,         "watched_color",             0, 52428,     0);
	set_color (&text_url_color,        "text_url_color",            0,     0, 40000);
	set_color (&text_quoted_color[0],  "text_quoted_color_1",   40000, 10000, 40000);
	set_color (&text_quoted_color[1],  "text_quoted_color_2",   50000,     0,  2000);
	set_color (&text_quoted_color[2],  "text_quoted_color_3",   45000, 28000,     0);

	base_prefs_init (download_dir,
	                 text_quote_chars,
	                 acache_max_megs,
			 pan_config_get_bool("/Pan/Cache/FlushOnExit=false"),
	                 break_thread_when_subject_changes);

	debug_exit ("prefs_init");
}


void
show_group_substitution_help_dialog (gpointer window)
{
	const char * str = _("%g - group as one directory (alt.binaries.pictures.trains)\n"
	                     "%G - group as nested directory (/alt/binaries/pictures/trains)\n"
	                     " \n"
	                     "\"/home/user/News/Pan/%g\" becomes\n"
	                     "\"/home/user/News/Pan/alt.binaries.pictures.trains\", and\n"
	                     "\"/home/user/News/Pan/%G\" becomes\n"
	                     "\"/home/user/News/Pan/alt/binaries/pictures/trains\",");
	GtkWidget * w = gtk_message_dialog_new (GTK_WINDOW(window),
	                                        GTK_DIALOG_DESTROY_WITH_PARENT,
	                                        GTK_MESSAGE_INFO,
	                                        GTK_BUTTONS_CLOSE, "%s", str);
	g_signal_connect_swapped (GTK_OBJECT(w), "response",
	                          G_CALLBACK(gtk_widget_destroy), GTK_OBJECT (w));
	gtk_widget_show_all (w);
}


static GtkWidget*
tasks_page (void)
{
	int row;
	int items_in_section;
	GtkWidget * t;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * l;
	GtkAdjustment * adj;
	char buf[512];
	char * p;

	row = 0;
	t = gtk_table_new (4, 15, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Tasks"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 4;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* max connections total */
		l = gtk_label_new_with_mnemonic (_("Ma_ximum connections:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		adj = (GtkAdjustment *) gtk_adjustment_new (128.0, 1.0, 128.0, 1.0, 1.0, 0.0);
		win->max_connections = w = gtk_spin_button_new (adj, 0, 0);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (win->max_connections), queue_get_max_sockets());
		connect_signal_to_prefs_changed (w, "changed");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* task retries */
		l = gtk_label_new_with_mnemonic (_("Maximum task retr_ies:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		adj = (GtkAdjustment *) gtk_adjustment_new (4.0, 1.0, 999.0, 1.0, 1.0, 0.0);
		win->max_tries = w = gtk_spin_button_new (adj, 0, 0);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (win->max_tries), queue_get_max_tries());
		connect_signal_to_prefs_changed (w, "changed");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* download directory */
		l = gtk_label_new_with_mnemonic (_("Do_wnload directory:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		w = win->dir_download = pan_file_entry_new (_("Download Directory"));
		if ((p = pan_config_get_string ("/Pan/Paths/download_dir")) == NULL)
			p = g_strdup_printf ("%s/News/Pan/", g_get_home_dir());
		pan_file_entry_set (win->dir_download, p);
		g_free (p);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), pan_file_entry_gtk_entry(w));
		connect_signal_to_prefs_changed (pan_file_entry_gtk_entry(w), "changed");
		gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);

		w = gtk_button_new_with_mnemonic (_("_Help"));
		g_signal_connect_swapped (w, "clicked", G_CALLBACK (show_group_substitution_help_dialog), win->dialog);
		gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);
		++row;

		/* remove failed tasks */
		w = gtk_check_button_new_with_mnemonic (_("Automatically remove _failed tasks from the task manager"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), remove_failed_tasks);
		connect_signal_to_prefs_changed (w, "toggled");
		win->remove_failed_tasks_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Headers"));
	w = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

		items_in_section = 4;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* download new headers */
		w = gtk_check_button_new_with_mnemonic (
			_("Download _new headers from subscribed groups when starting Pan"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_on_startup);
		connect_signal_to_prefs_changed (w, "toggled");
		win->fetch_new_on_startup_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* download new headers and bodies */
		w = gtk_check_button_new_with_mnemonic (
			_("Download new headers and bo_dies from subscribed groups when starting Pan"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_and_bodies_on_startup);
		connect_signal_to_prefs_changed (w, "toggled");
		win->fetch_new_and_bodies_on_startup_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* download new headers */
		w = gtk_check_button_new_with_mnemonic (
			_("Download new headers when loading a _group"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_on_group_load);
		connect_signal_to_prefs_changed (w, "toggled");
		win->fetch_new_on_group_load_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* mouse behavior */
		w = gtk_check_button_new_with_mnemonic (
			_("Mouse single clic_k selects, rather than activates, groups and articles"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), single_click_selects);
		connect_signal_to_prefs_changed (w, "toggled");
		win->single_click_selects_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	return t;
}

/**
***  prefs_text_page
**/

static void
smooth_scrolling_check_toggled (GtkToggleButton * tb, gpointer user_data)
{
	gtk_widget_set_sensitive (GTK_WIDGET(win->smooth_scrolling_speed_sb), gtk_toggle_button_get_active(tb));
}

static gulong
get_header_flags (void)
{
	guint i;
	gulong flags = 0;
	GPtrArray * header_toggles = (GPtrArray*) g_object_get_data (G_OBJECT(win->display_page), "header_toggles");

	for (i=0; i<header_toggles->len; ++i) {
		GtkWidget * w = GTK_WIDGET (g_ptr_array_index (header_toggles, i));
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w)))
			flags |= GPOINTER_TO_UINT (g_object_get_data (G_OBJECT(w), "bit"));
	}

	return flags;
}

typedef struct
{
	char* text;
	gulong flag;
}
MessageListThrowawayStruct;

static GtkWidget*
display_page (void)
{
	GtkAdjustment * adj;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * t;
	GtkWidget * l;
	GPtrArray * header_toggles = g_ptr_array_new ();
	int row;
	int items_in_section;
	char buf[512];
	const gulong flags = (gulong) pan_config_get_int ("/Pan/State/Headers=243");

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Header Pane"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 4;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* break threads when subject changes? */
		w = gtk_check_button_new_with_mnemonic (_("When a _Followup subject header changes, show as new thread"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), break_thread_when_subject_changes);
		connect_signal_to_prefs_changed (w, "toggled");
		win->break_thread_when_subject_changes_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* expand threads by default? */
		w = gtk_check_button_new_with_mnemonic (_("E_xpand all threads by default"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), expand_all_threads_by_default);
		connect_signal_to_prefs_changed (w, "toggled");
		win->expand_all_threads_by_default_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* multipart: one node or all */
		w = gtk_check_button_new_with_mnemonic (_("Show complete m_ultipart posts as a single article"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), hide_mpart_child_nodes);
		connect_signal_to_prefs_changed (w, "toggled");
		win->hide_mpart_child_nodes_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* date format label */
		l = gtk_label_new_with_mnemonic (_("_Date Format:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* date format control */
		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		w = gtk_entry_new_with_max_length (128);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		pan_gtk_entry_set_text (w, thread_date_format);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
		win->thread_date_entry = w;
		w = gtk_button_new_with_mnemonic (_("_Help"));
		g_signal_connect (w, "clicked",
				  G_CALLBACK (date_help_clicked_cb), NULL);
		gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);
		++row;

	/**
	***  Header Pane
	**/

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Body Pane"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 11;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* quoted text label */
		l = gtk_label_new_with_mnemonic (_("Characters denoting _quoted text:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* quoted text control */
		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		w = gtk_entry_new_with_max_length (20);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_entry_set_text (GTK_ENTRY(w), text_quote_chars);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
		win->text_quoted_chars_entry = w;
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);
		++row;

		/* smooth scrolling label */
		w = gtk_check_button_new_with_mnemonic (_("Smooth _scrolling at speed:"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), text_window_smooth_scrolling);
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(smooth_scrolling_check_toggled), 0);
		win->smooth_scrolling_check = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);

		/* smooth scrolling control */
		adj = (GtkAdjustment *) gtk_adjustment_new (1.0, 1.0, 10.0, 1.0, 1.0, 0.0);
		w = gtk_spin_button_new (adj, 0, 0);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), text_window_smooth_scrolling_speed);
		connect_signal_to_prefs_changed (w, "changed");
		win->smooth_scrolling_speed_sb = w;
		gtk_widget_set_sensitive (w, text_window_smooth_scrolling);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

#define ADD_HEADER_TOGGLE(name,bit) \
		g_snprintf (buf, sizeof(buf), _("Show \"%s\" header"), name); \
		w = gtk_check_button_new_with_mnemonic (buf); \
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), (flags & bit?1:0)); \
		g_object_set_data (G_OBJECT(w), "bit", GUINT_TO_POINTER(bit)); \
		connect_signal_to_prefs_changed (w, "toggled"); \
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1); \
		g_ptr_array_add (header_toggles, w); \
		++row;
		ADD_HEADER_TOGGLE(_("From"), UI_HEADER_AUTHOR)
		ADD_HEADER_TOGGLE(_("Subject"), UI_HEADER_SUBJECT)
		ADD_HEADER_TOGGLE(_("Date"), UI_HEADER_DATE)
		ADD_HEADER_TOGGLE(_("Reply-To"), UI_HEADER_REPLY_TO)
		ADD_HEADER_TOGGLE(_("Followup-To"), UI_HEADER_FOLLOWUP_TO)
		ADD_HEADER_TOGGLE(_("Newsgroups"), UI_HEADER_NEWSGROUPS)
		ADD_HEADER_TOGGLE(_("Message-ID"), UI_HEADER_MESSAGE_ID)
		ADD_HEADER_TOGGLE(_("Newsreader"), UI_HEADER_NEWSREADER)
#undef ADD_HEADER_TOGGLE


	g_object_set_data (G_OBJECT(t), "header_toggles", header_toggles);
	return t;
}

static void
use_custom_fonts_toggled (GtkToggleButton * tb, gpointer user_data)
{
	const gboolean b = gtk_toggle_button_get_active (tb);

	gtk_widget_set_sensitive (win->group_pane_font_lb, b);
	gtk_widget_set_sensitive (win->group_pane_font_gfp, b);
	gtk_widget_set_sensitive (win->header_pane_font_lb, b);
	gtk_widget_set_sensitive (win->header_pane_font_gfp, b);
	gtk_widget_set_sensitive (win->message_lb, b);
	gtk_widget_set_sensitive (win->message_gfp, b);
}

static GtkWidget*
appearance_page (void)
{
	GtkWidget * w;
	GtkWidget * t;
	GtkWidget * l;
	int row;
	int items_in_section;
	char buf[512];

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Fonts"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 5;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* use custom fonts */
		w = gtk_check_button_new_with_mnemonic (_("Use custom _fonts"));
		win->use_custom_fonts_check = w;
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_custom_fonts);
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(use_custom_fonts_toggled), 0);
		++row;

		/* group pane font */
		l = win->group_pane_font_lb = gtk_label_new_with_mnemonic (_("_Group Pane:"));
		gtk_widget_set_sensitive (l, use_custom_fonts);
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->group_pane_font_gfp = pan_font_picker_new ();
		pan_font_picker_set_font (w, group_pane_font);
		gtk_widget_set_sensitive (w, use_custom_fonts);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* header pane font */
		l = win->header_pane_font_lb = gtk_label_new_with_mnemonic (_("_Header Pane:"));
		gtk_widget_set_sensitive (l, use_custom_fonts);
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->header_pane_font_gfp = pan_font_picker_new ();
		pan_font_picker_set_font (w, header_pane_font);
		gtk_widget_set_sensitive (w, use_custom_fonts);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* message body font */
		l = win->message_lb = gtk_label_new_with_mnemonic (_("Message _Body:"));
		gtk_widget_set_sensitive (l, use_custom_fonts);
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->message_gfp = pan_font_picker_new ();
		pan_font_picker_set_font (w, message_body_font);
		gtk_widget_set_sensitive (w, use_custom_fonts);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* message body font */
		l = gtk_label_new_with_mnemonic (_("Message Body (_Monospace):"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->message_fixed_gfp = pan_font_picker_new ();
		pan_font_picker_set_font (w, message_body_font_fixed);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Colors"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 6;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* quoted text 1 */
		l = gtk_label_new_with_mnemonic (_("Quoted Text _1:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_quoted_pcp[0] = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_quoted_color[0]);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* quoted text 2 */
		l = gtk_label_new_with_mnemonic (_("Quoted Text _2:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_quoted_pcp[1] = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_quoted_color[1]);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* quoted text 3 */
		l = gtk_label_new_with_mnemonic (_("Quoted Text _3:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_quoted_pcp[2] = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_quoted_color[2]);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* URLs */
		l = gtk_label_new_with_mnemonic (_("_URLs:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_url_pcp = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_url_color);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* ignored threads */
		l = gtk_label_new_with_mnemonic (_("_Ignored Threads:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->ignored_pcp = pan_color_picker_new ();
		pan_color_picker_set_color (w, &killfile_color);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* watched threads */
		l = gtk_label_new_with_mnemonic (_("_Watched Threads:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->watched_pcp = pan_color_picker_new ();
		pan_color_picker_set_color (w, &watched_color);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	return t;
}


static GtkWidget*
apps_page (void)
{
	GtkWidget * w;
	GtkWidget * t;
	GtkWidget * l;
	GList * s;
	int row;
	int items_in_section;
	char buf[512];

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Applications"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 2;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* group pane font */
		l = gtk_label_new_with_mnemonic (_("E_ditor (%t is filename):"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->external_editor_combo = gtk_combo_new ();
		gtk_widget_set_usize (w, 300u, 0u);
		s = g_list_append (NULL, "gedit %t");
		s = g_list_append (s, "mgedit --no-fork %t");
		s = g_list_append (s, "emacs %t");
		s = g_list_append (s, "xemacs %t");
		s = g_list_append (s, "xterm -e jed %t");
		s = g_list_append (s, "xterm -e vi %t");
		s = g_list_append (s, "konsole -e jed %t");
		s = g_list_append (s, "konsole -e vi %t");
		s = g_list_append (s, "gnome-terminal -e jed %t");
		s = g_list_append (s, "gnome-terminal -e vi %t");
		gtk_combo_set_popdown_strings (GTK_COMBO(w), s);
		g_list_free (s);
		pan_gtk_entry_set_text (GTK_COMBO(w)->entry, external_editor);
		connect_signal_to_prefs_changed (GTK_COMBO(w)->entry, "changed");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), GTK_COMBO(w)->entry);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* group pane font */
		l = gtk_label_new_with_mnemonic (_("_Web Browser (%s is URL):"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);
		w = win->external_browser_combo = gtk_combo_new ();
		gtk_widget_set_usize (w, 300u, 0u);
		s = g_list_append (NULL, "gnome-moz-remote --raise --newwin '%s'");
		s = g_list_append (s, "galeon --existing '%s'");
		s = g_list_append (s, "galeon --new-tab '%s'");
		s = g_list_append (s, "netscape '%s'");
		s = g_list_append (s, "netscape -remote 'openURL(%s,raise)'");
		s = g_list_append (s, "mozilla -remote 'openURL(%s,raise)'");
		s = g_list_append (s, "konqueror '%s'");
		s = g_list_append (s, "xterm -e w3m \"%s\"");
		s = g_list_append (s, "xterm -e lynx \"%s\"");
		s = g_list_append (s, "xterm -e links \"%s\"");
		s = g_list_append (s, "konsole -e w3m \"%s\"");
		s = g_list_append (s, "konsole -e lynx \"%s\"");
		s = g_list_append (s, "konsole -e links \"%s\"");
		s = g_list_append (s, "gnome-terminal -e w3m \"%s\"");
		s = g_list_append (s, "gnome-terminal -e lynx \"%s\"");
		s = g_list_append (s, "gnome-terminal -e links \"%s\"");
		gtk_combo_set_popdown_strings (GTK_COMBO(w), s);
		g_list_free (s);
		pan_gtk_entry_set_text (GTK_COMBO(w)->entry, external_web_browser ? external_web_browser : NULL);
		connect_signal_to_prefs_changed (GTK_COMBO(w)->entry, "changed");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), GTK_COMBO(w)->entry);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	return t;
}

/**
***
***  LAYOUT
***
**/

static char*
layout_get_new_string (GtkWidget * page)
{
	int i;
	int layout = -1;
	const char * order;
	GtkWidget ** layout_buttons = (GtkWidget**) g_object_get_data (G_OBJECT(page), "layout_buttons");
	GSList * l = (GSList*) g_object_get_data (G_OBJECT(page), "layout_radio_group");

	/* get the layout number */
	for (i=0; i<5; ++i) {
		GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
		if (gtk_toggle_button_get_active(tb)) {
			layout = i + 1;
			break;
		}
	}

	/* get the ordering string */
	for (; l!=NULL; l=l->next) {
		GtkRadioButton * rb = GTK_RADIO_BUTTON(l->data);
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(rb))) {
			order = (const char*) g_object_get_data (G_OBJECT(rb), "layout");
			g_assert (order != NULL);
			g_assert (strlen(order) == 3);
			break;
		}
	}

	/* build & return the string */
	return g_strdup_printf ("%d%s", layout, order);
}

static void
layout_changed_cb (GtkToggleButton * togglebutton, gpointer user_data)
{
	GtkWidget ** layout_buttons = (GtkWidget**) user_data;
	static gboolean dampen_feedback_loop = FALSE;

	if (!dampen_feedback_loop) {
		int i;
		dampen_feedback_loop = TRUE;
		for (i=0; i<5; ++i) {
			GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
			gboolean active = togglebutton==tb;
			if (gtk_toggle_button_get_active(tb) != active)
				gtk_toggle_button_set_active (tb, active);
		}
		dampen_feedback_loop = FALSE;
	}
}

static GtkWidget*
layout_page (void)
{
	int i;
	GtkWidget * h;
	GtkWidget * w;
	GtkWidget * radio;
	GtkWidget * l;
	GtkWidget * t;
	const char * pch_group;
	const char * pch_header;
	const char * pch_body;
	const char lch = *layout_str;
	int row;
	char buf [256];
	const guint8 * inline_txt [] = { icon_layout_1, icon_layout_2, icon_layout_3, icon_layout_4, icon_layout_5};
	GtkWidget ** layout_buttons = g_malloc (sizeof(GtkWidget*) * 5);
	int items_in_section;

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Pane Layout"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 5;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* pane layout */
		h = gtk_hbox_new (TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (h), 0);
		for (i=0; i<5; ++i)
		{
			char buf[64];
			GdkPixbuf * pixbuf;
			GdkPixmap * pixmap;
			GdkBitmap * bitmap;

			g_snprintf (buf, sizeof(buf), "%d", i+1);
			w = gtk_toggle_button_new ();
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), *buf==lch);

			pixbuf = gdk_pixbuf_new_from_inline (-1, inline_txt[i], FALSE, NULL);
			gdk_pixbuf_render_pixmap_and_mask_for_colormap (pixbuf, gtk_widget_get_colormap(w), &pixmap, &bitmap, 128);
			gtk_container_add (GTK_CONTAINER(w), gtk_image_new_from_pixmap (pixmap, bitmap));

			g_object_unref (pixbuf);
			g_object_unref (pixmap);
			g_object_unref (bitmap);

			g_signal_connect (w, "toggled", G_CALLBACK(layout_changed_cb), layout_buttons);
			connect_signal_to_prefs_changed (w, "toggled");
			layout_buttons[i] = w;
			gtk_box_pack_start (GTK_BOX(h), w, TRUE, FALSE, 0);

		}
		gtk_table_attach_defaults (GTK_TABLE(t), h, 1, 4, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Pane Order"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 6;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* pane order */
		pch_group = _("Group");
		pch_header = _("Headers");
		pch_body = _("Body");
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_group, pch_header, pch_body);
		w = radio = gtk_radio_button_new_with_mnemonic_from_widget (NULL, buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"gta"));
		g_object_set_data (G_OBJECT(w), "layout", "gta");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_group, pch_body, pch_header);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"gat"));
		g_object_set_data (G_OBJECT(w), "layout", "gat");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_header, pch_group, pch_body);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"tga"));
		g_object_set_data (G_OBJECT(w), "layout", "tga");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_header, pch_body, pch_group);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"tag"));
		g_object_set_data (G_OBJECT(w), "layout", "tag");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_body, pch_group, pch_header);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"agt"));
		g_object_set_data (G_OBJECT(w), "layout", "agt");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_body, pch_header, pch_body);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"atg"));
		g_object_set_data (G_OBJECT(w), "layout", "atg");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	g_object_set_data_full (G_OBJECT(t), "layout_buttons", layout_buttons, g_free);
	g_object_set_data (G_OBJECT(t), "layout_radio_group", gtk_radio_button_get_group(GTK_RADIO_BUTTON(radio)));
	return t;
}

/**
***
**/

/* Build and return the 'Cache' page */
static GtkWidget*
cache_page (void)
{
	const gboolean flush_on_exit = pan_config_get_bool("/Pan/Cache/FlushOnExit=false");
	GtkObject * adj;
	GtkWidget * w;
	GtkWidget * t;
	GtkWidget * l;
	int row;
	int items_in_section;
	char buf[512];

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Article Cache"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 2;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* max cache size */
		l = gtk_label_new_with_mnemonic (_("Ma_ximum Cache Size (Megs):"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		adj = gtk_adjustment_new  (acache_max_megs, 1.0, 20000, 1.0, 1.0, 1.0);
		w = win->cache_megs_sb = gtk_spin_button_new (GTK_ADJUSTMENT(adj), 1.0, 0);
		connect_signal_to_prefs_changed (w, "changed");
		//gtk_widget_set_usize (GTK_WIDGET (w), 100u, 0u);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* flush on exit */
		w = gtk_check_button_new_with_mnemonic (_("_Flush cache when exiting Pan"));
		win->flush_cache_on_exit_check = w;
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), flush_on_exit);
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	return t;
}

static GtkWidget *
mail_page (void)
{
	char buf[128];
	GtkWidget * l;
	GtkWidget * t;
	GtkWidget * w;
	int row;
	int items_in_section;

	row = 0;
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Sending Mail"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 2;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 1, row, row+items_in_section);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 2, 3, row, row+items_in_section);

		/* address label */
		l = gtk_label_new_with_mnemonic (_("Mail Server A_ddress:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* address control */
		w = win->smtp_address = gtk_entry_new ();
		if (is_nonempty_string(mail_server_address))
			pan_gtk_entry_set_text (w, mail_server_address);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* address label */
		l = gtk_label_new_with_mnemonic (_("Ma_il Server Port:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* address control */
		w = win->smtp_port = gtk_spin_button_new (
			GTK_ADJUSTMENT(gtk_adjustment_new (mail_server_port, 0, 65536, 1, 1, 1)), 1, 0);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	return t;
}

/**
***  
**/

static void
prefs_destroy_cb (GtkObject * object, gpointer user_data)
{
	/* cleanup main window */
	g_free (win);
	win = NULL;
}

static void
prefs_response_cb (GtkDialog * dialog, int response, gpointer user_data)
{
	if (response == GTK_RESPONSE_OK || response == GTK_RESPONSE_APPLY)
		prefs_apply ();

	if (response == GTK_RESPONSE_OK || response == GTK_RESPONSE_CANCEL)
		gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
prefs_spawn (void)
{
	GtkWidget * w;
	GtkWidget * v;
	GtkWidget * c;

	/* There can be only one! (properties window) */
	if (win) {
		gdk_window_raise (GTK_WIDGET(win->dialog)->window);
		return;
	}

	win = g_new0 (PrefsWindow, 1);

	win->dialog = w = gtk_dialog_new_with_buttons (_("Pan: Preferences"),
	                                               GTK_WINDOW(Pan.window),
	                                               GTK_DIALOG_DESTROY_WITH_PARENT,
	                                               GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                               GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
	                                               GTK_STOCK_OK, GTK_RESPONSE_OK,
	                                               NULL);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(w), GTK_RESPONSE_OK, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(w), GTK_RESPONSE_APPLY, FALSE);

	/* put a notebook in the dialog's workarea */
	w = win->notebook = gtk_notebook_new ();
	gtk_container_set_border_width (GTK_CONTAINER (w), 18);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(win->dialog)->vbox), w, TRUE, TRUE, 0);

	/* add the tabs */
#define add_tab(tab,func) \
	c = func(); \
	gtk_container_set_border_width (GTK_CONTAINER(c), 18); \
	v = gtk_vbox_new (FALSE, 0); \
	gtk_box_pack_start (GTK_BOX(v), c, FALSE, FALSE, 0); \
	gtk_notebook_append_page (GTK_NOTEBOOK(w), v, gtk_label_new_with_mnemonic (tab));
	add_tab (_("Appea_rance"), appearance_page)
	add_tab (_("_Layout"), win->layout_page=layout_page)
	add_tab (_("Displa_y"), win->display_page=display_page)
	add_tab (_("_Tasks"), tasks_page)
	add_tab (_("Cach_e"), cache_page)
	add_tab (_("_Mail"), mail_page)
	add_tab (_("A_pplications"), apps_page)
#undef add_tab

	/* callbacks */
	g_signal_connect (win->dialog, "response", G_CALLBACK(prefs_response_cb), NULL);
	g_signal_connect (win->dialog, "destroy", G_CALLBACK(prefs_destroy_cb), NULL);

	gtk_widget_show_all (win->dialog);
}
