/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/task-bodies.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static int task_bodies_run (Task* item);

static char* task_bodies_describe (const StatusItem* item);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
task_bodies_destructor (PanObject * obj)
{
	debug_enter ("task_bodies_destructor");

	/* sanity clause */
	g_return_if_fail (obj!=NULL);

	/* task-bodies dtor */
	acache_checkin ((MessageIdentifier **)TASK(obj)->identifiers->pdata, TASK(obj)->identifiers->len);
	TASK_BODIES(obj)->working_index = 0;

	/* destroy parent class */
	task_destructor (obj);

	debug_exit ("task_bodies_destructor");
}

/************
*************  PUBLIC ROUTINES
************/

PanObject*
task_bodies_new (Server               * server,
                 MessageIdentifier   ** identifiers,
                 int                    identifier_qty)
{
	int i;
	TaskBodies* bodies = NULL;
	gboolean needs_socket;
	debug_enter ("task_bodies_new");

	/* sanity clause */
	g_return_val_if_fail (identifiers!=NULL, NULL);
	g_return_val_if_fail (identifier_qty!=0, NULL);

	/* check out all the articles */
	acache_checkout (identifiers, identifier_qty);

	/* do we need a connection? */
	needs_socket = FALSE;
	for (i=0; i<identifier_qty && !needs_socket; ++i)
		if (!acache_has_message (identifiers[i]->message_id))
			needs_socket = TRUE;

	/* create the object */
       	bodies = g_new0 (TaskBodies, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_bodies_new: %p", identifiers);

	/* initialize the parent class */
	task_constructor (TASK(bodies), task_bodies_destructor,
	                                task_bodies_describe,
	                                task_bodies_run,
	                                server, TRUE,
	                                needs_socket);
	TASK(bodies)->gets_bodies = TRUE;
	TASK(bodies)->type = TASK_TYPE_BODIES;

	/* initialize the task-bodies */
	bodies->working_index = 0;
	task_add_identifiers (TASK(bodies), identifiers, identifier_qty);

	debug_exit ("task_bodies_new");
	return PAN_OBJECT(bodies);
}

PanObject*
task_bodies_new_from_articles (const Article       ** articles,
                               int                    article_qty)
{
	int i;
	MessageIdentifier ** mids;
	PanObject * retval;

	/* sanity clause */
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (article_qty>0, NULL);
	g_return_val_if_fail (articles_are_valid (articles, article_qty), NULL);

	/* build the task */
	mids = g_alloca (sizeof(MessageIdentifier*) * article_qty);
	for (i=0; i<article_qty; ++i)
		mids[i] = message_identifier_new_from_article (articles[i]);
	retval = task_bodies_new (articles[0]->group->server, mids, article_qty);
	for (i=0; i<article_qty; ++i)
		g_object_unref (mids[i]);

	/* cleanup */
	return retval;
}

/*****
******
*****/

static char*
task_bodies_describe (const StatusItem* si)
{
	return g_strdup_printf (_("Downloading %d articles"), TASK(si)->identifiers->len);
}

/*****
******
*****/

static int
task_bodies_run (Task* task)
{
	int retval;
	int steps = (int) task->identifiers->len;
	debug_enter ("task_bodies_run");

	if (!task->needs_socket)
	{
		retval = TASK_SUCCESS;
	}
	else
	{
		status_item_emit_init_steps (STATUS_ITEM(task), steps);

		retval = nntp_download_bodies (STATUS_ITEM(task),
					       task->sock,
					       &task->hint_abort,
					       (MessageIdentifier **)task->identifiers->pdata,
					       task->identifiers->len,
					       &TASK_BODIES(task)->working_index,
					       FALSE);

		if (retval == TASK_SUCCESS)
			log_add_va (LOG_INFO, ("Fetched %u bodies"), steps);
	}

	debug_exit ("task_bodies_run");
	return retval;
}
