
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/newsrc.h>
#include <pan/base/newsrc-port.h>
#include <pan/base/util-file.h>


static gboolean
files_differ (const gchar * filename1, const gchar * filename2)
{
	char *f1=NULL, *f2=NULL;
	size_t f1_len=0, f2_len=0;
	gboolean diff;

	g_file_get_contents (filename1, &f1, &f1_len, NULL);
	g_file_get_contents (filename2, &f2, &f2_len, NULL);
	diff = f1==NULL || f2==NULL || f1_len!=f2_len || memcmp(f1,f2,f1_len);

	g_message ("files [%s] and [%s] are %s", filename1, filename2, diff?"different":"equal");
	g_free (f1);
	g_free (f2);
	return diff;
}

static int
import_test (void)
{
	gint test = 0;
	gchar *base, *noisy, *less_noisy, *in, *in_sub_only, *out, *out_bak;
	Server * server;

	base = getenv ("srcdir");
	base = g_strdup_printf ("%s/", base!=NULL ? base : getenv("PWD"));
	noisy       = g_strdup_printf ("%s/data/newsrc_noise", base);
	less_noisy  = g_strdup_printf ("%s/data/newsrc_noise_cleaned_up", base);
	in          = g_strdup_printf ("%s/data/newsrc_1", base);
	in_sub_only = g_strdup_printf ("%s/data/newsrc_1_subscribed_only", base);
	out         = g_strdup_printf ("%s/newsrc_2", g_get_tmp_dir());
	out_bak     = g_strdup_printf ("%s/newsrc_2.bak", g_get_tmp_dir());

	/* create server */
	server = server_new ();
	server->name = g_strdup ("Temp");

	/* simple import/export */
	++test;
	newsrc_import (server, in, FALSE);
	newsrc_export (server, out, FALSE);
	if (files_differ (in, out)) return test;
	g_message ("PASS simple newsrc import/export");

	/* export + backup */
	++test;
	newsrc_export (server, out, FALSE);
	if (files_differ (in, out)) return test;
	if (files_differ (out_bak, out)) return test;
	unlink (out);
	unlink (out_bak);
	g_message ("PASS newsrc export + backup");

	/* export subscribed only */
	++test;
	if (!pan_file_exists (in_sub_only)) return test;
	newsrc_export (server, out, TRUE);
	if (files_differ (in_sub_only, out)) return test;
	if (pan_file_exists (out_bak)) return test;
	unlink (out);
	g_message ("PASS newsrc export subscribed only");

	/* import noisy without crashing */
	++test;
	server = server_new ();
	server->name = g_strdup ("Temp");
	newsrc_import (server, noisy, FALSE);
	newsrc_export (server, out, FALSE);
	if (files_differ (less_noisy, out)) return test;
	g_message ("PASS survive importing corrupt newsrc file");
	unlink (out);

	/* try to import nonexistant file w/o crashing */
	++test;
	newsrc_import (server, "file_that_doesnt_exist", FALSE);
	g_message ("PASS survive importing nonexistant newsrc file");

	return 0;
}


static void
init_pan (void)
{
	gchar * base;
	gchar * download_dir;

	base = getenv ("srcdir");
	base = g_strdup_printf ("%s/", base!=NULL ? base : getenv("PWD"));
	download_dir = g_strdup ("/tmp/");
	base_prefs_init (download_dir, ">", 100, FALSE, FALSE);

	g_free (download_dir);
	g_free (base);
}

extern int server_saving_enabled;

int
main (void)
{
	gint num;
	gboolean b;
	Newsrc * n;
	gint i;

	g_thread_init (NULL);
	init_pan ();
	server_saving_enabled = FALSE;

	/* init newsrc */
       	n = newsrc_new (NULL, 0, 1000);
	newsrc_init (n, "1-20,100,102,200-300", 0, 1000);

	/* simple test */
	num = 1;
	if (!newsrc_is_article_read(n,1)) return num;
	g_message ("PASS newsrc newsrc_is_article_read");

	/* testing boundaries */
	num = 2;
	if (!newsrc_is_article_read(n,19)) return num;
	if (!newsrc_is_article_read(n,20)) return num;
	if ( newsrc_is_article_read(n,21)) return num;
	g_message ("PASS newsrc newsrc_is_article_read boundaries");

	/* more boundary testing */
	num = 3;
	if (!newsrc_is_article_read(n,100)) return num;
	if ( newsrc_is_article_read(n,101)) return num;
	if (!newsrc_is_article_read(n,102)) return num;
	g_message ("PASS newsrc newsrc_is_article_read boundaries 2");

	/* make sure marking read doesn't corrupt boundaries */
	num = 4;
	b = newsrc_mark_article (n, 101, TRUE);
	if (b!=FALSE) return num;
	if ( newsrc_is_article_read(n, 99)) return num;
	if (!newsrc_is_article_read(n,100)) return num;
	if (!newsrc_is_article_read(n,101)) return num;
	if (!newsrc_is_article_read(n,102)) return num;
	if ( newsrc_is_article_read(n,103)) return num;
	g_message ("PASS newsrc don't corrupt boundaries when marking read");

	/* make sure marking unread doesn't corrupt boundaries */
	num = 5;
	b = newsrc_mark_article (n, 101, FALSE);
	if (b!=TRUE) return num;
	if ( newsrc_is_article_read(n, 99)) return num;
	if (!newsrc_is_article_read(n,100)) return num;
	if ( newsrc_is_article_read(n,101)) return num;
	if (!newsrc_is_article_read(n,102)) return num;
	if ( newsrc_is_article_read(n,103)) return num;
	g_message ("PASS newsrc don't corrupt boundaries when marking unread");

	/* mark all read */
	num = 6;
	newsrc_mark_all (n, TRUE);
	if (!newsrc_is_article_read(n, 99)) return num;
	if (!newsrc_is_article_read(n,100)) return num;
	if (!newsrc_is_article_read(n,101)) return num;
	if (!newsrc_is_article_read(n,102)) return num;
	if (!newsrc_is_article_read(n,103)) return num;
	g_message ("PASS newsrc mark all read");

	/* mark all unread */
	num = 7;
	newsrc_mark_all (n, FALSE);
	if ( newsrc_is_article_read(n, 99)) return num;
	if ( newsrc_is_article_read(n,100)) return num;
	if ( newsrc_is_article_read(n,101)) return num;
	if ( newsrc_is_article_read(n,102)) return num;
	if ( newsrc_is_article_read(n,103)) return num;
	g_message ("PASS newsrc mark all unread");

	/* mark it read again & test the read string */
	num = 8;
	newsrc_mark_all (n, TRUE);
	if (!newsrc_is_article_read(n, 99)) return num;
	if (!newsrc_is_article_read(n,100)) return num;
	if (!newsrc_is_article_read(n,101)) return num;
	if (!newsrc_is_article_read(n,102)) return num;
	if (!newsrc_is_article_read(n,103)) return num;
	g_message ("PASS newsrc re-mark all read");

	/* newsrc_get_read_str */
	num = 9;
	newsrc_init (n, "1-20,100,102,200-300", 0, 1000);
	if (strcmp(newsrc_get_read_str(n),"1-20,100,102,200-300")) return num;
	b = newsrc_mark_article (n, 15, FALSE);
	if (b!=TRUE) return num;
	if (strcmp(newsrc_get_read_str(n),"1-14,16-20,100,102,200-300")) return num;
	b = newsrc_mark_article (n, 101, TRUE);
	if (b!=FALSE) return num;
	if (strcmp(newsrc_get_read_str(n),"1-14,16-20,100-102,200-300")) return num;
	b = newsrc_mark_article (n, 103, TRUE);
	if (b!=FALSE) return num;
	if (strcmp(newsrc_get_read_str(n),"1-14,16-20,100-103,200-300")) return num;
	b = newsrc_mark_article (n, 105, TRUE);
	if (b!=FALSE) return num;
	if (strcmp(newsrc_get_read_str(n),"1-14,16-20,100-103,105,200-300")) return num;
	g_message ("PASS newsrc_get_read_str");

	/* newsrc_mark_range */
	num = 10;
	newsrc_init (n, "1-1000", 0, 1000);
	i = newsrc_mark_range (n, 100, 199, FALSE);
	if (i!=100) return num;
	if (strcmp(newsrc_get_read_str(n),"1-99,200-1000")) return num;
	i = newsrc_mark_range (n, 100, 199, TRUE);
	if (i!=100) return num;
	if (strcmp(newsrc_get_read_str(n),"1-1000")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 11;
	newsrc_init (n, "1-500", 0, 1000);
	i = newsrc_mark_range (n, 400, 800, TRUE);
	if (strcmp(newsrc_get_read_str(n),"1-800")) return num;
	if (i!=300) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 12;
	newsrc_init (n, "250-500", 0, 1000);
	i = newsrc_mark_range (n, 100, 400, TRUE);
	if (strcmp(newsrc_get_read_str(n),"100-500")) return num;
	if (i!=150) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 13;
	newsrc_init (n, "250-500", 0, 1000);
	i = newsrc_mark_range (n, 100, 600, TRUE);
	if (strcmp(newsrc_get_read_str(n),"100-600")) return num;
	if (i!=250) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 14;
	newsrc_init (n, "250-500", 0, 1000);
	i = newsrc_mark_range (n, 100, 600, TRUE);
	if (strcmp(newsrc_get_read_str(n),"100-600")) return num;
	if (i!=250) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 15;
	newsrc_init (n, "100-500", 0, 1000);
	i = newsrc_mark_range (n, 50, 600, FALSE);
	if (strcmp(newsrc_get_read_str(n),"")) return num;
	if (i!=401) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 16;
	newsrc_init (n, "100-500", 0, 1000);
	i = newsrc_mark_range (n, 50, 200, FALSE);
	if (strcmp(newsrc_get_read_str(n),"201-500")) return num;
	if (i!=101) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 17;
	newsrc_init (n, "100-500", 0, 1000);
	i = newsrc_mark_range (n, 400, 600, FALSE);
	if (strcmp(newsrc_get_read_str(n),"100-399")) return num;
	if (i!=101) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 18;
	newsrc_init (n, "1-50,100-150,200-250,300-350", 0, 1000);
	i = newsrc_mark_range (n, 75, 275, FALSE);
	if (strcmp(newsrc_get_read_str(n),"1-50,300-350")) return num;
	if (i!=102) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 19;
	newsrc_init (n, "1-50,100-150,200-250,300-350", 0, 1000);
	i = newsrc_mark_range (n, 75, 300, FALSE);
	if (strcmp(newsrc_get_read_str(n),"1-50,301-350")) return num;
	if (i!=103) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 20;
	newsrc_init (n, "250-500", 0, 1000);
	i = newsrc_mark_range (n, 600, 700, FALSE);
	if (i!=0) return num;
	if (strcmp(newsrc_get_read_str(n),"250-500")) return num;
	i = newsrc_mark_range (n, 50, 200, FALSE);
	if (i!=0) return num;
	if (strcmp(newsrc_get_read_str(n),"250-500")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 21;
	newsrc_init (n, "1-498,500-599,601-1000", 0, 1000);
	i = newsrc_mark_range (n, 500, 599, FALSE);
	if (i!=100) return num;
	if (strcmp(newsrc_get_read_str(n),"1-498,601-1000")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 22;
	newsrc_init (n, "1-498,500-599,601-1000", 0, 1000);
	i = newsrc_mark_range (n, 498, 601, FALSE);
	if (i!=102) return num;
	if (strcmp(newsrc_get_read_str(n),"1-497,602-1000")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	num = 23;
	newsrc_init (n, "1-498,500-599,601-1000", 0, 1000);
	i = newsrc_mark_range (n, 499, 601, FALSE);
	if (i!=101) return num;
	if (strcmp(newsrc_get_read_str(n),"1-498,602-1000")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	/* Janus Sandsgaard reported this one against 0.9.7pre5 */
	num = 24;
	newsrc_init (n, "17577-17578,17581-17582,17589-17590", 0, 999999);
	i = newsrc_mark_range (n, 17578, 17578, TRUE);
	if (i!=0) return num;
	if (strcmp(newsrc_get_read_str(n),"17577-17578,17581-17582,17589-17590")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	/* Jiri Kubicek reported this one against 0.11.2.90 */
	num = 25;
	newsrc_init (n, "", 1, 10);
	newsrc_mark_range (n, 1, 4, TRUE);
	newsrc_mark_range (n, 7, 9, TRUE);
	newsrc_mark_range (n, 3, 4, FALSE);
	if (strcmp(newsrc_get_read_str(n),"1-2,7-9")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	/* found this during a code review for 0.12.90 */
	num = 26;
	newsrc_init (n, "1-75, 100-120, 150-200", 1, 200);
	i = newsrc_mark_range (n, 50, 175, TRUE);
	if (i!=53) return num;
	if (strcmp(newsrc_get_read_str(n), "1-200")) return num;
	g_message ("PASS newsrc_mark_range %d", num);

	/* newsrc_set_group_range
	 * note that after a mark all, the lower bound is 0.
	 * this follows newsrc conventions. */
	num = 27;
	newsrc_init (n, "1-20,100,102,200-300", 0, 1000);
	newsrc_set_group_range (n, 15, 220);
	if (strcmp(newsrc_get_read_str(n),"15-20,100,102,200-220")) return num;
	newsrc_set_group_range (n, 50, 219);
	if (strcmp(newsrc_get_read_str(n),"100,102,200-219")) return num;
	newsrc_set_group_range (n, 10, 1000);
	if (strcmp(newsrc_get_read_str(n),"100,102,200-219")) return num;
	newsrc_mark_all (n, TRUE);
	if (strcmp(newsrc_get_read_str(n),"0-1000")) return num;
	g_message ("PASS newsrc_set_group_range");

	/* http://bugzilla.gnome.org/show_bug.cgi?id=77878 */
	newsrc_init (n, "", 1, 10);
	newsrc_mark_range (n, 1, 4, TRUE);
	newsrc_mark_range (n, 7, 9, TRUE);
	newsrc_mark_range (n, 3, 4, FALSE);
	if (strcmp(newsrc_get_read_str(n),"1-2,7-9")) return num;
	g_message ("PASS Jiri Kubicek");

	/* import/export test */
	if ((num = import_test ())) return num;

	/* success */
	g_message ("PASSED all newsrc tests");
	return 0;
}
