/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/newsrc-port.h>
#include <pan/base/pan-config.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/server.h>
#include <pan/base/serverlist.h>
#include <pan/base/util-file.h>

static GSList * serverlist = NULL;

static Server * _active = NULL;

const gchar * PAN_SENT;
const gchar * PAN_SENDLATER;

/**
***  Private Utilities
**/

static gint
server_name_compare (gconstpointer a, gconstpointer b)
{
	const Server* sa = (const Server*) a;
	const Server* sb = (const Server*) b;
	return g_strcasecmp (server_get_name (sa), server_get_name (sb));
}

/**
***  Server Module
**/

static Server*
server_fetch (const char *name)
{
	Server * server = NULL;
	gboolean has_section;
	char * buf;
	debug_enter ("server_fetch");

	/* find out if we've got a pan_config section for this server */
	buf = g_strdup_printf ("/Pan/%s/", name);
	pan_config_push_prefix (buf);
	has_section = pan_config_has_section (buf);
	replace_gstr (&buf, NULL);

	if (has_section)
	{
		server = server_new ();
		server->address = pan_config_get_string ("Address");
		if (server->address == NULL)
		{
			pan_object_unref (PAN_OBJECT(server));
			server = NULL;
		}
		else
		{
			char * path;
			char * pch;
			
			/* make sure the server's config directory exists */
		       	path = g_strdup_printf ("%s%c%s", get_data_dir(), G_DIR_SEPARATOR, name);
			pan_file_ensure_path_exists (path);

			/* load the server from pan_config */
			server->port = pan_config_get_int ("Port=119");
			server->name = g_strdup(name);
			server->use_newsrc = pan_config_get_bool ("Use_Newsrc=0");
			server->newsrc_filename = pan_config_get_string ("Newsrc_Filename");
			server->need_auth = pan_config_get_int ("Auth_Required=0");
			server->username = pan_config_get_string ("Username");
			server->password = pan_config_get_string ("Password");
			server->max_connections = pan_config_get_int ("Max_Connections=2");
			server->idle_secs_before_timeout = pan_config_get_int ("Idle_Timeout_Secs=180");
			server->reserve_connection_for_bodies = pan_config_get_bool ("Reserve_Reader_Connection=TRUE");
			/* no good way to get a ulong from pan_config? */
			server->last_newgroup_list_time = 1;
			pch = pan_config_get_string ("Last_Grouplist_Update");
			if (is_nonempty_string(pch))
				server->last_newgroup_list_time = strtoul (pch, NULL, 10);

			/* cleanup */
			g_free (path);
			g_free (pch);
		}
	}
	pan_config_pop_prefix ();

	debug_exit ("server_fetch");
	return server;
}

/**
 * Load a list of servers from the user's ~/.gnome/Pan file
 * @return a GSList of Server's for each server found
 */
static GSList*
server_list_load (void)
{
	GSList * l = NULL;
	gchar * p;
	debug_enter ("server_list_load");

	/* build all the servers in the list */
       	p = pan_config_get_string ("/Pan/Servers/Names"); /*commas*/
	if (p != NULL)
	{
		const char * walk = p;
		GString * str = g_string_new (NULL);

		/* create a server object for each server in the list */
		while (get_next_token_g_str (walk, ',', &walk, str))
		{
			Server * server = server_fetch (str->str);
			if (server != NULL)
				l = g_slist_insert_sorted (l,
				                           server,
				                           server_name_compare);
		}

		/* cleanup */
		g_string_free (str, TRUE);
		g_free (p);
	}

	debug_exit ("server_list_load");
	return l;
}

static void
server_init_servers (void)
{
	static gboolean inited = FALSE;
	debug_enter ("server_init_servers");

	if (!inited)
	{
		inited = TRUE;
		serverlist = server_list_load ();
	}

	debug_exit ("server_init_servers");
}

void
serverlist_get_servers (GPtrArray * fillme)
{
	GSList * l;

	g_return_if_fail (fillme!=NULL);

	pan_g_ptr_array_reserve (fillme, g_slist_length(serverlist));
	for (l=serverlist; l!=NULL; l=l->next)
		g_ptr_array_add (fillme, l->data);
}

void
serverlist_shutdown (void)
{
	GSList * l;

	/* save the updated server info */
	serverlist_save ();

	/* close down the servers */
	l = serverlist;
	serverlist = NULL;
	g_slist_foreach (l, (GFunc)pan_object_unref, NULL);
	g_slist_free (l);
}

Server*
serverlist_get_active_server (void)
{
	return _active;
}

void
serverlist_set_active_server (Server * server)
{
	g_return_if_fail (server != NULL);

	/* import the first time the server is activated */
	if (server->use_newsrc && !server->newsrc_imported) {
		newsrc_import (server, server->newsrc_filename, FALSE);
		server->newsrc_imported = TRUE;
	}

	/* activate it */
	_active = server;
	if (_active!=NULL && is_nonempty_string(_active->name))
		pan_config_set_string ("/Pan/State/Server", _active->name);
	else
		pan_config_clean_key ("/Pan/State/Server");
	pan_callback_call (serverlist_get_server_activated_callback(), _active, NULL);
}

Server*
serverlist_get_named_server (const char* servername)
{
	GSList * l;
	Server * retval = NULL;
	debug_enter ("server_get_by_name");

	g_return_val_if_fail (is_nonempty_string(servername), NULL);

	server_init_servers ();

	for (l=serverlist; retval==NULL && l!=NULL; l=l->next) {
		Server * s = SERVER(l->data);
		if (!pan_strcmp (server_get_name (s), servername))
			retval = s;
	}

	debug_exit ("server_get_by_name");
	return retval;
}

Group*
serverlist_get_named_folder (const gchar * foldername)
{
	Server * server;
		 
	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string(foldername), NULL);
			 
	/* get the folder server... */
	server = serverlist_get_named_server (INTERNAL_SERVER_NAME);
	g_return_val_if_fail (server_is_valid (server), NULL);
					 
	/* get the folder... */
	return server_get_named_group (server, foldername);
}

static Server*
get_default_server (void)
{
	GSList * l;
	gchar * name;
	Server * retval = NULL;

	g_return_val_if_fail (serverlist!=NULL, NULL);

	/* look in the gnome config for a default */
	name = pan_config_get_string ("/Pan/State/Server");
	if (is_nonempty_string (name))
		retval = serverlist_get_named_server (name);
	g_free (name);

	/* otherwise just use the first server */
	for (l=serverlist; retval==NULL && l!=NULL; l=l->next) {
		Server * s = SERVER(l->data);
		if (strcmp (server_get_name (s), INTERNAL_SERVER_NAME))
			retval = s;
	}

	/* return what we've got */
	return retval;
}



/****
*****
*****  SERVER LOADING ON STARTUP
*****
****/

void
serverlist_init (void)
{
	Server * s;

	server_init_servers ();

	s = get_default_server ();
	if (s != NULL)
		serverlist_set_active_server (s);
}

void
serverlist_add_server (Server * server)
{
	g_return_if_fail (server_is_valid (server));

	server_init_servers ();

	serverlist = g_slist_insert_sorted (serverlist,
	                                    server,
	                                    server_name_compare);

	if (pan_strcmp (INTERNAL_SERVER_NAME, server->name))
		serverlist_save ();
}

void
serverlist_remove_server (Server * server)
{
	g_return_if_fail (server_is_valid (server));

	serverlist = g_slist_remove (serverlist, server);
	/* might be best to just let this leak.
	pan_object_unref (PAN_OBJECT(server)); */
}


void
serverlist_save (void)
{
	GString * servers = g_string_new (NULL);
	GSList *l = NULL;
	gchar * p;
	char buf[PATH_MAX];

	/* Out with the old */
	p = pan_config_get_string ("/Pan/Servers/Names");
	if (p != NULL) {
		int str_len = 0;
		const char * str_begin = NULL;
		const char * walk = p;
		while (get_next_token_run (walk, ',', &walk, &str_begin, &str_len)) {
			char * line = g_alloca (str_len+1);
			memcpy (line, str_begin, str_len);
			line[str_len] = '\0';

			g_snprintf (buf, sizeof(buf), "/Pan/%s", line);
			pan_config_clean_section (buf);
		}
		g_free (p);
	}
	pan_config_sync ();

	/* In with the new */
	for (l=serverlist; l!=NULL; l=l->next)
	{
		Server *s = SERVER(l->data);
		const char * sname = server_get_name (s);

		/* don't need to load/save this one */
		if (!pan_strcmp (INTERNAL_SERVER_NAME, sname))
			continue;

		g_snprintf (buf, sizeof(buf), "/Pan/%s", sname);
		pan_config_push_prefix (buf);
		g_snprintf (buf, sizeof(buf), "%lu", s->last_newgroup_list_time);
		pan_config_set_string ("Last_Grouplist_Update", buf);
		pan_config_set_string ("Address", s->address);
		pan_config_set_int ("Port", s->port);
		pan_config_set_bool ("Use_Newsrc", s->use_newsrc);
		pan_config_set_string ("Newsrc_Filename", s->newsrc_filename);
		pan_config_set_int ("Auth_Required", s->need_auth);
		pan_config_set_string ("Username", s->username);
		pan_config_set_string ("Password", s->password);
		pan_config_set_int ("Max_Connections", s->max_connections);
		pan_config_set_int ("Idle_Timeout_Secs", s->idle_secs_before_timeout);
		pan_config_set_bool ("Reserve_Reader_Connection", s->reserve_connection_for_bodies);
		pan_config_pop_prefix ();

		if (servers->len)
			g_string_append_c (servers, ',');
		g_string_append (servers, sname);
	}

	pan_config_set_string ("/Pan/Servers/Names", servers->str);
	g_string_free (servers, TRUE);
	pan_config_sync ();
}

/**
***  Server Activated
**/

PanCallback*
serverlist_get_server_activated_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}
