/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkradiomenuitem.h>
#include <gtk/gtktogglebutton.h>
#include <pan/base/group.h>

#include <pan/base/debug.h>

#include <pan/filters/filter.h>
#include <pan/filters/filter-manager.h>
#include <pan/filters/filter-aggregate.h>
#include <pan/filters/filter-binary.h>
#include <pan/filters/filter-cached.h>
#include <pan/filters/filter-manager.h>
#include <pan/filters/filter-mine.h>
#include <pan/filters/filter-new.h>
#include <pan/filters/filter-phrase.h>
#include <pan/filters/filter-saved.h>
#include <pan/filters/filter-thread.h>
#include <pan/filters/filter-top.h>
#include <pan/filters/filter-manager.h>

#include <pan/filter-mediator.h>
#include <pan/gui.h>
#include <pan/util.h>


static FilterCurrentMediatorCtor _buttons;
static gulong _filter_bits = ~STATE_FILTER_IGNORED;
static FilterShow _filter_show = FILTER_SHOW_MATCHES_AND_REFERENCES;
static void fire_user_changed_filter (void);

/**
***  REFRESH
**/

/**
 * Updates the Filter menu buttons to reflect the current state of the filter.
 */
static void
refresh (void)
{
	GtkCheckMenuItem * ckm;
	GtkToggleButton * tb;
	gboolean model;
	gboolean view;

	pan_lock ();

	/**
	***  New / Unread / Read
	**/

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_new_ckm);
	model = _filter_bits & STATE_FILTER_NEW ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_unread_ckm);
	model = _filter_bits & STATE_FILTER_UNREAD ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_read_ckm);
	model = _filter_bits & STATE_FILTER_READ ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	tb = GTK_TOGGLE_BUTTON(_buttons._match_only_new_tb);
	model = (_filter_bits & STATE_FILTER_NEW)
		&& !(_filter_bits & STATE_FILTER_UNREAD)
		&& !(_filter_bits & STATE_FILTER_READ);
	view = gtk_toggle_button_get_active (tb);
	if (model != view)
		gtk_toggle_button_set_active (tb, model);

	/**
	***  Binaries
	**/

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_complete_ckm);
	model = _filter_bits & STATE_FILTER_COMPLETE_BINARIES ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_incomplete_ckm);
	model = _filter_bits & STATE_FILTER_INCOMPLETE_BINARIES ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_nonbinaries_ckm);
	model = _filter_bits & STATE_FILTER_NONBINARIES ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	tb = GTK_TOGGLE_BUTTON(_buttons._match_only_complete_tb);
	model = (_filter_bits & STATE_FILTER_COMPLETE_BINARIES)
		&& !(_filter_bits & STATE_FILTER_INCOMPLETE_BINARIES)
		&& !(_filter_bits & STATE_FILTER_NONBINARIES);
	view = gtk_toggle_button_get_active (tb);
	if (model != view)
		gtk_toggle_button_set_active (tb, model);

	/**
	***  Watched
	**/

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_watched_ckm);
	model = _filter_bits & STATE_FILTER_WATCHED ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_ignored_ckm);
	model = _filter_bits & STATE_FILTER_IGNORED ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_normal_ckm);
	model = _filter_bits & STATE_FILTER_NORMAL_RANK ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	tb = GTK_TOGGLE_BUTTON(_buttons._match_only_watched_tb);
	model = (_filter_bits & STATE_FILTER_WATCHED)
		&& !(_filter_bits & STATE_FILTER_IGNORED)
		&& !(_filter_bits & STATE_FILTER_NORMAL_RANK);
	view = gtk_toggle_button_get_active (tb);
	if (model != view)
		gtk_toggle_button_set_active (tb, model);

	/**
	***  Cached
	**/

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_cached_ckm);
	model = _filter_bits & STATE_FILTER_CACHED ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_noncached_ckm);
	model = _filter_bits & STATE_FILTER_NOT_CACHED ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	tb = GTK_TOGGLE_BUTTON(_buttons._match_only_cached_tb);
	model = (_filter_bits & STATE_FILTER_CACHED)
		&& !(_filter_bits & STATE_FILTER_NOT_CACHED);
	view = gtk_toggle_button_get_active (tb);
	if (model != view)
		gtk_toggle_button_set_active (tb, model);

	/**
	***  Author
	**/

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_mine_ckm);
	model = _filter_bits & STATE_FILTER_MINE ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	ckm = GTK_CHECK_MENU_ITEM(_buttons._match_others_ckm);
	model = _filter_bits & STATE_FILTER_NOT_MINE ? 1 : 0;
	view = ckm->active ? 1 : 0;
	if (model != view)
		gtk_check_menu_item_set_active (ckm, model);

	tb = GTK_TOGGLE_BUTTON(_buttons._match_only_mine_tb);
	model = (_filter_bits & STATE_FILTER_MINE)
		&& !(_filter_bits & STATE_FILTER_NOT_MINE);
	view = gtk_toggle_button_get_active (tb);
	if (model != view)
		gtk_toggle_button_set_active (tb, model);

	/**
	***  Show
	**/

	switch (_filter_show) {
		case FILTER_SHOW_MATCHES:
			ckm = GTK_CHECK_MENU_ITEM(_buttons._show_articles_rmi);
			break;
		case FILTER_SHOW_THREAD:
			ckm = GTK_CHECK_MENU_ITEM(_buttons._show_threads_rmi);
			break;
		case FILTER_SHOW_MATCHES_AND_REPLIES:
			ckm = GTK_CHECK_MENU_ITEM(_buttons._show_articles_and_replies_rmi);
			break;
		case FILTER_SHOW_MATCHES_AND_REFERENCES:
		default:
			ckm = GTK_CHECK_MENU_ITEM(_buttons._show_articles_and_refs_rmi);
			break;
	}
	if (!ckm->active)
		gtk_check_menu_item_set_active (ckm, TRUE);

	pan_unlock ();
}

/**
 * Calls refresh in an idle thread.
 */
static int
refresh_idle (gpointer unused)
{
	refresh ();
	return 0;
}

/**
 * Sets the current filter state and updates the menu to reflect the new state.
 */
static void
set (gulong value, FilterShow show)
{
	static gboolean _mute_feedback = FALSE;

	if (!_mute_feedback)
	{
		_mute_feedback = TRUE;

		if (_filter_bits!=value || _filter_show!=show)
		{
			_filter_show = show;
			_filter_bits = value;

			gui_queue_add (refresh_idle, NULL);
			fire_user_changed_filter ();
		}

		_mute_feedback = FALSE;
	}
}

/**
 * Set the current filter's bit value (ie, what it matches)
 */
static void
set_value (gulong value)
{
	set (value, _filter_show);
}

/**
 * Set the current filter's show value (ie, the FilterShow apply to the matched articles)
 */
static void
set_show (FilterShow show)
{
	set (_filter_bits, show);
}


/**
***  MENU BUTTON CALLBACKS
**/

static void
menu_button_toggled (GtkCheckMenuItem * item, gpointer user_data)
{
	gulong value = _filter_bits;
	gsize flag = GPOINTER_TO_SIZE (user_data);

	if (item->active)
		value |= flag;
	else
		value &= ~flag;

	set_value (value);
}

static void
show_only_new_cb (GtkToggleButton * tb, gpointer user_data)
{
	gulong value = _filter_bits;
	const gboolean active = gtk_toggle_button_get_active (tb);

	if (!active)
		value |= STATE_FILTER_NEW | STATE_FILTER_UNREAD | STATE_FILTER_READ;
	else {
		value |= STATE_FILTER_NEW;
		value &= ~STATE_FILTER_UNREAD;
		value &= ~STATE_FILTER_READ;
	}

	set_value (value);
}

static void
show_only_complete_cb (GtkToggleButton * tb, gpointer user_data)
{
	gulong value = _filter_bits;
	const gboolean active = gtk_toggle_button_get_active (tb);

	if (!active)
		value |= STATE_FILTER_COMPLETE_BINARIES | STATE_FILTER_INCOMPLETE_BINARIES | STATE_FILTER_NONBINARIES;
	else {
		value |= STATE_FILTER_COMPLETE_BINARIES;
		value &= ~STATE_FILTER_INCOMPLETE_BINARIES;
		value &= ~STATE_FILTER_NONBINARIES;
	}

	set_value (value);
}

static void
show_only_watched_cb (GtkToggleButton * tb, gpointer user_data)
{
	gulong value = _filter_bits;
	const gboolean active = gtk_toggle_button_get_active (tb);

	if (!active)
		value |= STATE_FILTER_WATCHED | STATE_FILTER_NORMAL_RANK;
	else {
		value |= STATE_FILTER_WATCHED;
		value &= ~STATE_FILTER_IGNORED;
		value &= ~STATE_FILTER_NORMAL_RANK;
	}

	set_value (value);
}

static void
show_only_cached_cb (GtkToggleButton * tb, gpointer user_data)
{
	gulong value = _filter_bits;
	const gboolean active = gtk_toggle_button_get_active (tb);

	if (!active)
		value |= STATE_FILTER_CACHED | STATE_FILTER_NOT_CACHED;
	else {
		value |= STATE_FILTER_WATCHED;
		value &= ~STATE_FILTER_NOT_CACHED;
	}

	set_value (value);
}

static void
show_only_mine_cb (GtkToggleButton * tb, gpointer user_data)
{
	gulong value = _filter_bits;
	const gboolean active = gtk_toggle_button_get_active (tb);

	if (!active)
		value |= STATE_FILTER_MINE | STATE_FILTER_NOT_MINE;
	else {
		value |= STATE_FILTER_MINE;
		value &= ~STATE_FILTER_NOT_MINE;
	}

	set_value (value);
}

static void
show_toggled_cb (GtkCheckMenuItem * check, gpointer user_data)
{
	g_return_if_fail (GTK_IS_CHECK_MENU_ITEM(check));

	if (check->active)
		set_show (GPOINTER_TO_SIZE(user_data));
}

/***
****
****  Custom Filter Menu
****
***/

static GStaticMutex _custom_filter_entries_mutex = G_STATIC_MUTEX_INIT;
static GtkItemFactoryEntry * _custom_filter_entries = NULL;
static int _custom_filter_entry_qty = 0;

static void
custom_filter_selected_cb (gpointer user_data, int unused, GtkWidget * w)
{
	fire_user_changed_filter ();
}

extern GtkItemFactory * _main_menu_factory;

static void
rebuild_custom_filter_menu_with_filters (GPtrArray * filters)
{
	int i;

	pan_lock ();
	g_static_mutex_lock (&_custom_filter_entries_mutex);

	/* remove any old filter buttons */
	if (_custom_filter_entries != NULL) {
		gtk_item_factory_delete_entries (_main_menu_factory, _custom_filter_entry_qty, _custom_filter_entries);
		for (i=0; i<_custom_filter_entry_qty; ++i)
			g_free (_custom_filter_entries[i].path);
		g_free (_custom_filter_entries);
		_custom_filter_entries = NULL;
		_custom_filter_entry_qty = 0;
	}

	/* build the new filter buttons */
	if (filters->len)
	{
		_custom_filter_entries = g_new0 (GtkItemFactoryEntry, filters->len+1);

		/* add the separator */
		_custom_filter_entries[_custom_filter_entry_qty=0].path = g_strdup_printf ("/Filter/---");
		_custom_filter_entries[_custom_filter_entry_qty].item_type = "<Separator>";
		_custom_filter_entries[_custom_filter_entry_qty++].callback = NULL;

		/* add the custom filters */
		for (i=0; i<filters->len; ++i)
		{
			const FilterTop * top = FILTER_TOP(g_ptr_array_index(filters,i));
			char * underscore_escaped_path = pan_substitute (top->name, "_", "__");

			/* build an entry */
			_custom_filter_entries[_custom_filter_entry_qty].path = g_strdup_printf ("/Filter/%s", underscore_escaped_path);
			_custom_filter_entries[_custom_filter_entry_qty].item_type = "<ToggleItem>";
			_custom_filter_entries[_custom_filter_entry_qty++].callback = custom_filter_selected_cb;

			g_free (underscore_escaped_path);
		}

		/* add the filters to the menu */
		gtk_item_factory_create_items (_main_menu_factory, _custom_filter_entry_qty, _custom_filter_entries, NULL);

		/* add the filter name as data item in the menu buttons.
		 * This is used by custom_filter_selected_Cb() callback to
		 * get a FilterTop* from the GtkMenuToggleButton */
		for (i=0; i<filters->len; ++i)
		{
			char * unescaped_path;
			const char * top_name;
			GtkWidget * w;

		       	top_name = FILTER_TOP(g_ptr_array_index(filters,i))->name;
			unescaped_path = g_strdup_printf ("/Filter/%s", top_name);

			w = gtk_item_factory_get_widget (_main_menu_factory, unescaped_path);
			g_object_set_data_full (G_OBJECT(w), "filter_name", g_strdup(top_name), g_free);
			_custom_filter_entries[i+1].extra_data = w;

			g_free (unescaped_path);
		}
	}

	g_static_mutex_unlock (&_custom_filter_entries_mutex);
	pan_unlock ();
}

static void
rebuild_custom_filter_menu (void)
{
	GPtrArray * a = g_ptr_array_sized_new (64);
	filter_manager_get_filters (a);
	rebuild_custom_filter_menu_with_filters (a);
	pan_g_ptr_array_foreach (a, (GFunc)pan_object_unref, NULL);
}

static int
filter_manager_filters_changed_idle (gpointer unused)
{
	rebuild_custom_filter_menu ();
	fire_user_changed_filter ();
	return 0;
}

static void
filter_manager_filters_changed_callback (gpointer call_obj, gpointer call_arg, gpointer user_data)
{
	gui_queue_add (filter_manager_filters_changed_idle, NULL);
}



/***
****
****  Public
****
***/

void
filter_mediator_init (FilterCurrentMediatorCtor * ctor)
{
	_buttons = *ctor;

	/* New / Read / Unread */

	g_signal_connect (G_OBJECT(_buttons._match_new_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_NEW));
	g_signal_connect (G_OBJECT(_buttons._match_unread_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_UNREAD));
	g_signal_connect (G_OBJECT(_buttons._match_read_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_READ));
	g_signal_connect (G_OBJECT(_buttons._match_only_new_tb), "toggled",
	                  G_CALLBACK(show_only_new_cb), NULL);

	/* Text / Complete / Incomplete */

	g_signal_connect (G_OBJECT(_buttons._match_complete_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_COMPLETE_BINARIES));
	g_signal_connect (G_OBJECT(_buttons._match_incomplete_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_INCOMPLETE_BINARIES));
	g_signal_connect (G_OBJECT(_buttons._match_nonbinaries_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_NONBINARIES));
	g_signal_connect (G_OBJECT(_buttons._match_only_complete_tb), "toggled",
	                  G_CALLBACK(show_only_complete_cb), NULL);

	/* Watched */

	g_signal_connect (G_OBJECT(_buttons._match_watched_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_WATCHED));
	g_signal_connect (G_OBJECT(_buttons._match_ignored_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_IGNORED));
	g_signal_connect (G_OBJECT(_buttons._match_normal_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_NORMAL_RANK));
	g_signal_connect (G_OBJECT(_buttons._match_only_watched_tb), "toggled",
	                  G_CALLBACK(show_only_watched_cb), NULL);

	/* Cached */

	g_signal_connect (G_OBJECT(_buttons._match_cached_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_CACHED));
	g_signal_connect (G_OBJECT(_buttons._match_noncached_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_NOT_CACHED));
	g_signal_connect (G_OBJECT(_buttons._match_only_cached_tb), "toggled",
	                  G_CALLBACK(show_only_cached_cb), NULL);

	/* Author */

	g_signal_connect (G_OBJECT(_buttons._match_mine_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_MINE));
	g_signal_connect (G_OBJECT(_buttons._match_others_ckm), "toggled",
	                  G_CALLBACK(menu_button_toggled), GSIZE_TO_POINTER(STATE_FILTER_NOT_MINE));
	g_signal_connect (G_OBJECT(_buttons._match_only_mine_tb), "toggled",
	                  G_CALLBACK(show_only_mine_cb), NULL);

	/* show */

	g_signal_connect (G_OBJECT(_buttons._show_articles_rmi), "toggled",
	                  G_CALLBACK(show_toggled_cb), GSIZE_TO_POINTER(FILTER_SHOW_MATCHES));
	g_signal_connect (G_OBJECT(_buttons._show_articles_and_replies_rmi), "toggled",
	                  G_CALLBACK(show_toggled_cb), GSIZE_TO_POINTER(FILTER_SHOW_MATCHES_AND_REPLIES));
	g_signal_connect (G_OBJECT(_buttons._show_articles_and_refs_rmi), "toggled",
	                  G_CALLBACK(show_toggled_cb), GSIZE_TO_POINTER(FILTER_SHOW_MATCHES_AND_REFERENCES));
	g_signal_connect (G_OBJECT(_buttons._show_threads_rmi), "toggled",
	                  G_CALLBACK(show_toggled_cb), GSIZE_TO_POINTER(FILTER_SHOW_THREAD));

	pan_callback_add (filter_manager_get_filters_changed_callback(), filter_manager_filters_changed_callback, NULL);
	rebuild_custom_filter_menu ();

	refresh ();
}

PanCallback*
filter_mediator_get_change_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_user_changed_filter (void)
{
	pan_callback_call (filter_mediator_get_change_callback(), GSIZE_TO_POINTER(_filter_bits), NULL);
}

void
filter_mediator_get_bits (gulong      * setme_bits,
                          FilterShow  * setme_show)
{
	if (setme_bits != NULL)
		*setme_bits = _filter_bits;
	if (setme_show != NULL)
		*setme_show = _filter_show;
}

void
filter_mediator_set_bits (gulong        bits,
                          FilterShow    show)
{
	set (bits, show);
}

Filter*
filter_mediator_get_filter  (gulong        bits,
                             gboolean      negate)
{
	int i;
	int n;
	Filter * retval;
	FilterAggregate * top;
	const gulong l = bits;
	const gboolean show_new = (l & STATE_FILTER_NEW) != 0;
	const gboolean show_unread = (l & STATE_FILTER_UNREAD) != 0;
	const gboolean show_read = (l & STATE_FILTER_READ) != 0;
	const gboolean show_bin = (l & STATE_FILTER_COMPLETE_BINARIES) != 0;
	const gboolean show_inc_bin = (l & STATE_FILTER_INCOMPLETE_BINARIES) != 0;
	const gboolean show_nonbinary = (l & STATE_FILTER_NONBINARIES) != 0;
	const gboolean show_watched = (l & STATE_FILTER_WATCHED) != 0;
	const gboolean show_ignored = (l & STATE_FILTER_IGNORED) != 0;
	const gboolean show_normal = (l & STATE_FILTER_NORMAL_RANK) != 0;
	const gboolean show_saved = (l & STATE_FILTER_SAVED) != 0;
	const gboolean show_idle = (l & STATE_FILTER_IDLE) != 0;
	const gboolean show_queued = (l & STATE_FILTER_QUEUED) != 0;
	const gboolean show_cached = (l & STATE_FILTER_CACHED) != 0;
	const gboolean show_non_cached = (l & STATE_FILTER_NOT_CACHED) != 0;
	const gboolean show_by_me = (l & STATE_FILTER_MINE) != 0;
	const gboolean show_by_others = (l & STATE_FILTER_NOT_MINE) != 0;

	/* top */
	top = FILTER_AGGREGATE(filter_aggregate_new ());
	filter_aggregate_set_type (top, AGGREGATE_TYPE_AND);

	/* new/unread/read */
	n = 0;
	if (show_new) ++n;
	if (show_unread) ++n;
	if (show_read) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_new_article_new (FILTER_NEW_NEW);
		if (n==1) {
			if (show_new)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_NEW);
			else if (show_unread)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_UNREAD);
			else if (show_read)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_READ);
		} else {
			if (!show_new)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_NEW);
			else if (!show_unread)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_UNREAD);
			else if (!show_read)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_READ);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* attachments */
	n = 0;
	if (show_bin) ++n;
	if (show_inc_bin) ++n;
	if (show_nonbinary) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_binary_new ();
		if (n==1) {
			if (show_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_COMPLETE);
			else if (show_inc_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_INCOMPLETE);
			else if (show_nonbinary)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_NONBINARY);
		} else {
			if (!show_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_COMPLETE);
			else if (!show_inc_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_INCOMPLETE);
			else if (!show_nonbinary)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_NONBINARY);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* thread state */
	n = 0;
	if (show_watched) ++n;
	if (show_ignored) ++n;
	if (show_normal) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_thread_new ();
		if (n==1) {
			if (show_watched)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_WATCHED);
			else if (show_ignored)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_IGNORED);
			else if (show_normal)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_NOTHING);
		} else {
			if (!show_watched)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_WATCHED);
			else if (!show_ignored)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_IGNORED);
			else if (!show_normal)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_NOTHING);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* saved/queued/idle */
	n = 0;
	if (show_saved) ++n;
	if (show_queued) ++n;
	if (show_idle) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_saved_new ();
		if (n==1) {
			if (show_saved)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_SAVED);
			else if (show_queued)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_QUEUED);
			else if (show_idle)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_NOT_SAVED);
		} else {
			if (!show_saved)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_SAVED);
			else if (!show_queued)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_QUEUED);
			else if (!show_idle)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_NOT_SAVED);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* cached */
	n = 0;
	if (show_cached) ++n;
	if (show_non_cached) ++n;
	if (n == 1) {
		Filter * filter = filter_cached_new ();
		if (show_non_cached)
			filter_negate (filter);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* by me / others */
	n = 0;
	if (show_by_me) ++n;
	if (show_by_others) ++n;

	if (n == 1) {
		Filter * filter = filter_mine_new ();
		if (show_by_others)
			filter_negate (filter);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* custom filters */
	for (i=1; i<_custom_filter_entry_qty; ++i) {
		GtkWidget * w = GTK_WIDGET (_custom_filter_entries[i].extra_data);
		const char * name = (const char*) g_object_get_data (G_OBJECT(w), "filter_name");
		const gboolean active = GTK_CHECK_MENU_ITEM(w)->active;
		if (active) {
			Filter * filter = filter_manager_get_named_filter (name);
			if (filter != NULL)
				filter_aggregate_add (top, &filter, 1);
		}
	}

	/* retval */
	if (filter_aggregate_child_size(top) != 1)
		retval = FILTER(top);
	else {
		retval = filter_aggregate_get_child_at (top, 0);
		pan_object_ref (PAN_OBJECT(retval));
		pan_object_unref (PAN_OBJECT(top)); /* remove previous filter */
	}

	/* negate */
	if (negate)
		filter_negate (retval);

	return retval;
}
