/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * A note on implementation
 * The queue defers all public tasks (like insert, remove, etc.)
 * to the queue thread.  This ensures that all tasks are handled
 * in sequence, rather than concurrently.  This ensures that callbacks
 * will be invoked in sync with the queue, so the queue's state won't
 * change during the callback.
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <errno.h>

#include <glib.h>

#include <unistd.h>

#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/server.h>

#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/task-xml.h>

/*********************
**********************  Defines / Enumerated types
*********************/

typedef enum
{
	TODO_INSERT,
	TODO_REMOVE,
	TODO_MOVE,
	TODO_REQUEUE,
	TODO_TASK_ABORT,
	TODO_SET_MAX_SOCKET_QTY,
	TODO_SET_MAX_TRIES,
	TODO_SHUTDOWN,
	TODO_ONLINE
}
QueueTodoAction;

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GSList * tasks;
	int index_1;
	QueueTodoAction action;
}
QueueTodo;

typedef struct
{
	PanSocket *socket;
	Server *server;
	Task *task;	/* NULL if idle */
}
SocketInfo;

/*********************
**********************  Private Function Prototypes
*********************/

static void fire_queue_size_changed (int running_qty, int total_qty);
static void fire_connecting_state_changed (gboolean connecting);
static void fire_connection_size_changed (int increment);
static void fire_online_status_changed (gboolean online);
static void fire_message_id_status_changed (const char ** message_ids, int message_id_qty);

static void queue_run_thread (gpointer data, gpointer user_data);

static void* queue_mainloop (void*);

static void queue_run_what_we_can (void);

static void queue_set_task_status (Task* task,
                                   QueueTaskStatus status);

static void socket_cleanup (Task* task);

static void queue_new_todo (QueueTodoAction     action,
                            GSList            * tasks,
                            int                 i);

static guint queue_get_length (void);

static void queue_do_todo (void);

/*********************
**********************  Constants
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

PanCallback * queue_tasks_added           = NULL;
PanCallback * queue_tasks_removed         = NULL;
PanCallback * queue_tasks_moved           = NULL;
PanCallback * queue_task_status_changed   = NULL;
PanCallback * queue_max_tries_changed     = NULL;

/***********
************  Private
***********/


/* FIXME glib 2.0: for efficiency of adding to end, use a GQueue instead of GSList */
static GQueue * todo_queue = NULL;
static GSList* task_list = NULL;

static GThreadPool * _tpool = NULL;

static GMutex * sock_lock = NULL;
static GMutex * task_lock = NULL;
static GMutex * cond_mutex = NULL;
static GMutex * todo_mutex = NULL;

static GCond * qcond = NULL;
static gboolean work_to_do = FALSE;

static GHashTable * task_to_status = NULL;
static GHashTable * message_id_hash = NULL;

static GSList * sockets = NULL;

static int max_socket_qty = 4;

static int max_tries = 5;

static guint running_tasks = 0;

static gboolean _online = TRUE;
static gboolean _remove_failed_tasks = TRUE;
static gboolean _dirty = FALSE;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

void
queue_wakeup (void)
{
	/*gboolean do_wakeup;*/
	debug_enter ("queue_wakeup");

	/*g_mutex_lock (todo_mutex);
	do_wakeup = !g_queue_is_empty (todo_queue);
	g_mutex_unlock (todo_mutex);*/

	if (TRUE /*do_wakeup*/)
	{
		g_mutex_lock (cond_mutex);
		work_to_do = TRUE;
		g_cond_signal (qcond);
		g_mutex_unlock (cond_mutex);
	}

	debug_exit ("queue_wakeup");
}

void
queue_init (gboolean online, int socket_qty, int tries, gboolean remove_failed_tasks)
{
	debug_enter ("queue_init");

	task_to_status = g_hash_table_new (g_direct_hash, g_direct_equal);
	message_id_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

	todo_queue = g_queue_new ();

	queue_tasks_added = pan_callback_new ();
	queue_tasks_removed = pan_callback_new ();
	queue_tasks_moved = pan_callback_new ();
	queue_task_status_changed = pan_callback_new ();
	queue_max_tries_changed = pan_callback_new ();

	sock_lock = g_mutex_new ();
	task_lock = g_mutex_new ();
	cond_mutex = g_mutex_new ();
	todo_mutex = g_mutex_new ();

	qcond = g_cond_new ();

	_online = online;

	_remove_failed_tasks = remove_failed_tasks;
	max_tries = tries;
	max_socket_qty = socket_qty;
	_tpool = g_thread_pool_new (queue_run_thread, NULL, max_socket_qty, TRUE, NULL);

	/* fire up the queue thread */
	g_thread_create (queue_mainloop, NULL, FALSE, NULL);

	debug_exit ("queue_init");
}

/***
****
***/

int
queue_get_message_id_status (const char * message_id)
{
	return GPOINTER_TO_INT (g_hash_table_lookup (message_id_hash, message_id));
}
static void
queue_add_message_id_state (const char   ** message_ids,
                            int             message_id_qty,
                            int             state)
{
	int i;

	/* sanity clause */
	g_return_if_fail (state);
	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty>0);
	g_return_if_fail (is_nonempty_string(message_ids[0]));

	/* add the state */
	for (i=0; i<message_id_qty; ++i)
	{
		const char * id = message_ids[i];
		const int new_state = state | queue_get_message_id_status(id);
		g_hash_table_insert (message_id_hash, g_strdup(id), GINT_TO_POINTER(new_state));
	}

	fire_message_id_status_changed (message_ids, message_id_qty);
}
static void
queue_remove_message_id_state (const char **   message_ids,
                               int             message_id_qty,
                               int             state)
{
	int i;

	/* sanity clause */
	g_return_if_fail (state);
	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty>0);
	g_return_if_fail (is_nonempty_string(message_ids[0]));

	/* remove the state */
	for (i=0; i<message_id_qty; ++i)
	{
		const char * id = message_ids[i];
		const int new_state = queue_get_message_id_status(id) & ~state;
		if (!new_state)
			g_hash_table_remove (message_id_hash, id);
		else
			g_hash_table_insert (message_id_hash, g_strdup(id), GINT_TO_POINTER(new_state));
	}

	fire_message_id_status_changed (message_ids, message_id_qty);
}

static int
task_get_state (const Task * task)
{
	int state = 0;

	g_return_val_if_fail (task!=NULL, 0);

	switch (task->type)
	{
		case TASK_TYPE_BODY:
		case TASK_TYPE_BODIES:
			state |= QUEUE_MESSAGE_ID_DOWNLOAD;
			break;
		case TASK_TYPE_SAVE:
			state |= QUEUE_MESSAGE_ID_SAVE;
			break;
		case TASK_TYPE_HEADERS:
			break;
		default:
			state = 0;
			break;
	}

	return state;
}

static void
queue_add_message_ids (Task * task)
{
	const int state = task_get_state (task);

	if (state)
	{
		int i;
		GPtrArray * message_ids = g_ptr_array_new ();
		for (i=0; i<task->identifiers->len; ++i) {
			MessageIdentifier * mid = MESSAGE_IDENTIFIER (g_ptr_array_index (task->identifiers, i));
			g_ptr_array_add (message_ids, mid->message_id);
		}

		queue_add_message_id_state ((const char **)message_ids->pdata, message_ids->len, state);

		g_ptr_array_free (message_ids, TRUE);
	}
}

static void
queue_remove_message_ids (Task * task)
{
	const int state = task_get_state (task);

	if (state)
	{
		int i;
		GPtrArray * message_ids = g_ptr_array_new ();
		for (i=0; i<task->identifiers->len; ++i) {
			MessageIdentifier * mid = MESSAGE_IDENTIFIER (g_ptr_array_index (task->identifiers, i));
			g_ptr_array_add (message_ids, mid->message_id);
		}

		queue_remove_message_id_state ((const char **)message_ids->pdata, message_ids->len, state);

		g_ptr_array_free (message_ids, TRUE);
	}
}

/***
****
***/

gboolean
queue_is_online (void)
{
	return _online;
}
void
queue_set_online (gboolean online)
{
	queue_new_todo (TODO_ONLINE, NULL, online);
}
static void
real_queue_set_online (gint p, gboolean * do_wakeup)
{
	if (_online == p)
	{
		if (_online)
			*do_wakeup = TRUE;
	}
	else
	{
		_online = p;

		fire_online_status_changed (_online);

		if (_online)
			*do_wakeup = TRUE;
		else
		{
			GSList * l;

			g_mutex_lock (sock_lock);
			for (l=sockets; l!=NULL; l=l->next)
			{
				SocketInfo * i = (SocketInfo*)l->data;
				if (i->task != NULL)
					task_hint_abort (i->task);
				queue_set_task_status (i->task, QUEUE_TASK_STATUS_ABORTING);
			}
			g_mutex_unlock (sock_lock);
		}
	}
}


void
queue_shutdown (void)
{
	queue_new_todo (TODO_SHUTDOWN, NULL, 0);
}
static void
real_queue_shutdown (void)
{
	gint len;
	GSList * l;
	GSList * sockets_tmp;

	/* clear the status hashtable */
	g_hash_table_destroy (task_to_status);
	task_to_status = NULL;



	/* optionally save the tasks */

	/* unref the tasks */

	/* get the socket array */
	g_mutex_lock (sock_lock);
	sockets_tmp = sockets;
	sockets = NULL;
	g_mutex_unlock (sock_lock);

	/* close the sockets */
	len = g_slist_length (sockets_tmp);
	for (l=sockets_tmp; l!=NULL; l=l->next)
		pan_object_unref (PAN_OBJECT(((SocketInfo*)l->data)->socket));
	fire_connection_size_changed (-len);
	g_slist_foreach (sockets_tmp, (GFunc)g_free, NULL);
	g_slist_free (sockets_tmp);
}


void
queue_add (Task * task)
{
	g_return_if_fail (task!=NULL);
	queue_insert_tasks (g_slist_append (NULL, task), task->high_priority ? 0 : -1);
}

GPtrArray*
queue_get_tasks (void)
{
	GSList * l;
	GPtrArray * a = g_ptr_array_new ();

        g_mutex_lock (task_lock);
	for (l=task_list; l!=NULL; l=l->next) {
		pan_object_ref (PAN_OBJECT(l->data));
		g_ptr_array_add (a, l->data);
	}
        g_mutex_unlock (task_lock);

	return a;
}

gboolean
queue_is_empty (void)
{
	gboolean retval;

	g_mutex_lock (task_lock);
	retval = task_list == NULL;
	g_mutex_unlock (task_lock);

	return retval;
}

/**
***
**/

char*
queue_get_tasks_filename (void)
{
	return g_strdup_printf ("%s%ctasks.xml", get_data_dir(), G_DIR_SEPARATOR);
}

static void
queue_save_tasks (void)
{
	GPtrArray * tasks;
	char * filename;

	/* save the tasks */
       	filename = queue_get_tasks_filename ();
	tasks = queue_get_tasks ();
	task_xml_write (filename, (const Task**)tasks->pdata, tasks->len);

	/* cleanup */
	pan_g_ptr_array_foreach (tasks, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (tasks, TRUE);
	g_free (filename);
}


/**
***
**/

int
queue_get_max_sockets (void)
{
	return max_socket_qty;
}
void
queue_set_max_sockets (int i)
{
	queue_new_todo (TODO_SET_MAX_SOCKET_QTY, NULL, i);
	queue_wakeup ();
}
static void
real_queue_set_max_sockets (int i, gboolean * do_wakeup)
{
	if (max_socket_qty != i)
	{
		if (i > max_socket_qty)
			*do_wakeup = TRUE;

		max_socket_qty = i;
		g_thread_pool_set_max_threads (_tpool, max_socket_qty, NULL);
	}
}

guint
queue_get_running_task_count (void)
{
	gint running_count = 0;
	GSList *l;

	g_mutex_lock (task_lock);
	for (l=task_list; l!=NULL; l=l->next)
	{
		QueueTaskStatus status = queue_get_task_status (l->data);

		if (status == QUEUE_TASK_STATUS_RUNNING ||
		    status == QUEUE_TASK_STATUS_QUEUED ||
		    status == QUEUE_TASK_STATUS_CONNECTING)
			++running_count;
	}
	g_mutex_unlock (task_lock);

	return running_count;
}

/**
***
**/

void
queue_abort_tasks (GSList * tasks)
{
	queue_new_todo (TODO_TASK_ABORT, tasks, -1);
}
static void
real_queue_abort_tasks (GSList * tasks)
{
	GSList * l;
	for (l=tasks; l!=NULL; l=l->next)
	{
		Task * task = TASK(l->data);
		QueueTaskStatus status = queue_get_task_status (task);

		if (status==QUEUE_TASK_STATUS_RUNNING)
		{
			/* ask a running task to abort */
			task_hint_abort (TASK(task));
			queue_set_task_status (task, QUEUE_TASK_STATUS_ABORTING);
		}
		else if (task->hint_abort && !queue_is_online())
		{
			/* the task's been stopped because the queue is offline */
			task->tries = 0;
			task->hint_abort = FALSE;
			queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
		}
		else
		{
			/* the task has failed */
			task->tries = 0;
			task->hint_abort = FALSE;
			queue_set_task_status (task, QUEUE_TASK_STATUS_FAILED);
		}
	}
}

/**
***
**/

void
queue_insert_tasks (GSList * tasks, int index)
{
	if (tasks != NULL)
		queue_new_todo (TODO_INSERT, tasks, index);
}

static void
real_queue_insert_tasks (GSList * new_tasks, int index, gboolean * do_wakeup)
{
	GSList * tmp;
	GSList * insertme;
	guint task_len;
	debug_enter ("real_queue_insert_tasks");

	/* sanity clause */
	g_return_if_fail (new_tasks!=NULL);
	g_return_if_fail (index==-1 || (guint)index<=queue_get_length());

	/* make our own GSList nodes for inserting */
	insertme = g_slist_copy (new_tasks);

	/* inside a task_lock... */
	g_mutex_lock (task_lock);
	{
		/**
		***  Add to the list
		**/
		if (task_list == NULL) /* no previous list */
			task_list = insertme;
		else { /* insert into the list */
			GSList * nth = g_slist_nth (task_list, index);
			if (nth == NULL) { /* index out of range, append to end */
				index = g_slist_length (task_list);
				nth = g_slist_last (task_list);
			}
			g_slist_last (insertme)->next = nth->next;
			nth->next = insertme;
		}

		/**
		***  Mark the new tasks as queued
		**/
		for (tmp=new_tasks; tmp!=NULL; tmp=tmp->next) {
			Task * task = TASK(tmp->data);
			queue_add_message_ids (task);
			g_hash_table_insert (task_to_status, task, GINT_TO_POINTER(QUEUE_TASK_STATUS_QUEUED));
		}

		/* we need this for fire_queue_size_changed */
		task_len = g_slist_length (task_list);
	}
	_dirty = TRUE;
	g_mutex_unlock (task_lock);

	/* cleanup */
	fire_queue_size_changed (running_tasks, task_len);
	pan_callback_call (queue_tasks_added, new_tasks, GINT_TO_POINTER(index));
	*do_wakeup = TRUE;
	debug_exit ("real_queue_insert_tasks");
}

/**
***
**/

void
queue_remove_tasks (GSList * tasks)
{
	queue_new_todo (TODO_REMOVE, tasks, -1);
}
static void
real_queue_remove_tasks (GSList * tasks, gboolean * do_wakeup)
{
	GSList * l;
	GSList * removed = NULL;

	for (l=tasks; l!=NULL; l=l->next)
	{
		Task * task = TASK(l->data);

		if (task->is_running)
		{
			/* abort the task; after it's done running
			 * queue_run_thread() will send the task back here
			 * for part two  */
			task_hint_abort (task);
			queue_set_task_status (task, QUEUE_TASK_STATUS_REMOVING);
		}
		else /* not running */
		{
			/* not running */
			socket_cleanup (task);

			/* remove the task */
			g_mutex_lock (task_lock);
			task_list = g_slist_remove (task_list, task);
			removed = g_slist_prepend (removed, task);
			_dirty = TRUE;
			g_mutex_unlock (task_lock);
		}
	}

	if (removed != NULL)
	{
		removed = g_slist_reverse (removed);

		fire_queue_size_changed (running_tasks, g_slist_length(task_list));
		*do_wakeup = TRUE;
		pan_callback_call (queue_tasks_removed, removed, NULL);

		for (l=removed; l!=NULL; l=l->next) {
			Task * task = TASK(l->data);
			g_hash_table_remove (task_to_status, task);
			queue_remove_message_ids (task);
			pan_object_unref (PAN_OBJECT(task));
		}

		g_slist_free (removed);
	}
}

/**
***
**/

void
queue_move_tasks (GSList * tasks, int index)
{
	g_return_if_fail (tasks!=NULL);

	queue_new_todo (TODO_MOVE, tasks, index);
}

static void
real_queue_move_tasks (GSList * tasks, int moveto_index)
{
	GSList * l;
	GSList * ref;

	/* sanity clause */
	g_return_if_fail (tasks!=NULL);
	g_return_if_fail (task_list!=NULL);

	g_mutex_lock (task_lock);

	/* find the point of reference */
	if (moveto_index == 0)
		ref = NULL;
	else {
		ref = g_slist_nth (task_list, moveto_index-1);
		while (ref != NULL) {
			if (g_slist_find (tasks, ref->data) == NULL)
				break;
			ref = ref->next;
		}
		if (ref == NULL)
			ref = g_slist_last (task_list);
	}

	/* remove the tasks */
	for (l=tasks; l!=NULL; l=l->next)
		task_list = g_slist_remove (task_list, l->data);

	/* add them back in */
	if (ref == NULL) {
		l = g_slist_copy (tasks);
		g_slist_last(l)->next = task_list;
		task_list = l;
		moveto_index = 0;
	} else {
		l = g_slist_copy (tasks);
		if (task_list != NULL) {
			g_slist_last(l)->next = ref->next;
			ref->next = l;
		} else
			task_list = l;	
		moveto_index = g_slist_index (task_list, tasks->data);
	}

	/* let everyone know */
	_dirty = TRUE;
	g_mutex_unlock (task_lock);
	pan_callback_call (queue_tasks_moved, tasks, GINT_TO_POINTER(moveto_index));
	debug_exit ("real_queue_move_tasks");
}

/**
***
**/

void
queue_requeue_failed_tasks (GSList * tasks)
{
	GSList * l;

	/* sanity clause */
	g_return_if_fail (tasks != NULL);
	for (l=tasks; l!=NULL; l=l->next) {
		Task * task = (Task*) l->data;
		g_return_if_fail (queue_get_task_status(task) == QUEUE_TASK_STATUS_FAILED);
	}

	queue_new_todo (TODO_REQUEUE, tasks, -1);
}
static void
real_requeue_failed_tasks (GSList * tasks, gboolean * do_wakeup)
{
	if (tasks != NULL)
	{
		GSList * l;

		for (l=tasks; l!=NULL; l=l->next) {
			Task * task = TASK(l->data);
			task->tries = 0;
			task->hint_abort = FALSE;
			queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
		}

		*do_wakeup = TRUE;
	}
}

/**
***
**/

gboolean
queue_get_remove_failed_tasks (void)
{
	return _remove_failed_tasks;
}

void
queue_set_remove_failed_tasks (gboolean b)
{
	_remove_failed_tasks = b;
}

/**
***
**/

int
queue_get_max_tries (void)
{
	return max_tries;
}

void
queue_set_max_tries (int i)
{
	queue_new_todo (TODO_SET_MAX_TRIES, NULL, i);
}
static void
real_queue_set_max_tries (int i)
{
	max_tries = i;
	pan_callback_call (queue_max_tries_changed, NULL, GINT_TO_POINTER(i));
}

/**
***
**/

QueueTaskStatus
queue_get_task_status (const Task* task)
{
	QueueTaskStatus status = QUEUE_TASK_STATUS_NOT_QUEUED;

	if (task_to_status != NULL) {
		gpointer p = g_hash_table_lookup (task_to_status, task);
		if (p != NULL)
			status = GPOINTER_TO_INT (p);
	}

	return status;
}

static void
queue_set_task_status (Task* task, QueueTaskStatus status)
{
	g_return_if_fail (task!=NULL);
	g_return_if_fail (g_slist_index(task_list, task) != -1);

	g_hash_table_insert (task_to_status, task,
	                     GINT_TO_POINTER(status));
	pan_callback_call (queue_task_status_changed, task,
	                   GINT_TO_POINTER(status)); 
}

/************
*************  PRIVATE ROUTINES
************/

static void
queue_new_todo (QueueTodoAction action, GSList * tasks, int i)
{
	QueueTodo *todo = g_new (QueueTodo, 1);
	todo->tasks = tasks;
	todo->index_1 = i;
	todo->action = action;

	g_mutex_lock (todo_mutex);
	g_queue_push_tail (todo_queue, todo);
	debug3 (DEBUG_QUEUE, "todo queue now has %u tasks (new task type: %d, int 1: %d)",
		todo_queue->length, action, i);
	g_mutex_unlock (todo_mutex);

	queue_wakeup ();
}

static guint
queue_get_length (void)
{
	guint size;
	g_mutex_lock (task_lock);
	size = g_slist_length (task_list);
	g_mutex_unlock (task_lock);
	return size;
}

static void
queue_run_thread (gpointer data, gpointer user_data)
{
	int status;
	int result;
	gboolean aborting;
	Task* task = TASK(data);
	gchar* desc = status_item_describe (STATUS_ITEM(task));
	debug2 (DEBUG_QUEUE, "queue_run_thread: task %p [%s]", task, desc);

	/* run */
	status_item_set_active (STATUS_ITEM(task), TRUE);
	if (task->sock)
		pan_socket_reset_statistics (task->sock);
	fire_queue_size_changed (++running_tasks, queue_get_length());
	result = task_run (task);
	fire_queue_size_changed (--running_tasks, queue_get_length());
	debug3 (DEBUG_QUEUE, "queue task[%p] (%s) returned: %d", task, desc, result);
	g_free(desc);

	/* clean up gui */
	status_item_set_active (STATUS_ITEM(task), FALSE);

	/* clean up socket.
	   We throw away any socket of a failed or aborting task,
	   just in case the read buffer has junk in there. */
	status = queue_get_task_status (task);
	aborting = status==QUEUE_TASK_STATUS_ABORTING || status==QUEUE_TASK_STATUS_REMOVING;
	if (task->sock!=NULL && (aborting || result!=TASK_SUCCESS))
		task->sock->error = TRUE;
	socket_cleanup (task);

	/* clean up the task */
	task->is_running = FALSE;

	if (result==TASK_SUCCESS)
	{
		/* success, remove the task */
		queue_remove_tasks (g_slist_append(NULL,task));
	}
	else if (result==TASK_FAIL_HOPELESS && _remove_failed_tasks)
	{
		/* failed, and requeueing won't help, so remove it now */
		queue_remove_tasks (g_slist_append(NULL,task));
	}
	else if (status == QUEUE_TASK_STATUS_REMOVING)
	{
		/* someone is waiting until run() finishes to remove the task */
		queue_remove_tasks (g_slist_append(NULL,task));
	}
	else if (status == QUEUE_TASK_STATUS_ABORTING)
	{
		/* someone is waiting until run() finishes to abort the task */
		task->tries = 0;
		queue_abort_tasks (g_slist_append(NULL,task));
	}
	else if (task->tries<max_tries && result != TASK_FAIL_HOPELESS)
	{
		/* failed, but still has tries left to requeue */
		++task->tries;
		queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
		queue_wakeup ();
	}
	else
	{
		/* failed, let the user requeue later */
		task->tries = 0;
		task->hint_abort = FALSE;
		queue_set_task_status (task, QUEUE_TASK_STATUS_FAILED);
		queue_wakeup ();
	}

	debug1 (DEBUG_QUEUE, "queue_run_thread: task %p done", data);
}


static void
queue_run (Task* task)
{
	g_return_if_fail (task!=NULL);
	debug2 (DEBUG_QUEUE, "starting a thread to run task %p, sock %p", task, task->sock);

	queue_set_task_status (task, QUEUE_TASK_STATUS_RUNNING);

	task->is_running = TRUE;
	g_thread_pool_push (_tpool, task, NULL);
}


/**
 * Allocates a socket for a task.  First it looks for an idle socket in the
 * socket pool; otherwise, if we're not maxed out, it tries to create a new
 * socket.
 *
 * Return -1 if we can't connect because all lines are busy
 * Return -2 if we can't connect because of a connection failure
 */
static int
queue_get_socket_for_task (Task *task)
{
	GSList* sl = NULL;
	int retval = -1;
	const gboolean gets_bodies = TASK(task)->gets_bodies;
	gboolean skip_first_idle = task->server->reserve_connection_for_bodies && !gets_bodies;
	gboolean reader_socket_exists = FALSE;
	Server * server = task->server;
	int socket_qty;
	debug_enter ("queue_get_socket_for_task");

	g_mutex_lock (sock_lock);

	/* if an idle socket for that server, use it. */
	socket_qty = 0;
	for (sl=sockets; sl!=NULL && retval!=0; sl=sl->next)
	{
		SocketInfo * si = (SocketInfo*) sl->data;

		if (si->server != server) /* ignore other servers */
			continue;

		++socket_qty;

		if (!queue_is_online()) /* offline */
			continue;
		if (si->task != NULL) /* socket already in use */
			continue;

		if (skip_first_idle) { /* save one article reader socket */
			reader_socket_exists = TRUE;
			skip_first_idle = FALSE;
			continue;
		}

		si->task = task;
		task->sock = si->socket;

		debug3 (DEBUG_QUEUE,
			"queue_get_socket_for_task %p "
			"reusing socket #%d (%p)",
			task, g_slist_index(sockets, si), si->socket);

		retval = 0;
	}

	/* if no idle sockets for that server... */
	if (retval && queue_is_online() && socket_qty<max_socket_qty)
	{
		const int current_connections = socket_qty;
		int slots_open = server->max_connections - current_connections;
		debug1 (DEBUG_QUEUE, "queue_get_socket_for_task %p no idle sockets", task);

		/* leave an article reader socket free */
		if (!reader_socket_exists
			&& server->reserve_connection_for_bodies
			&& !gets_bodies
			&& max_socket_qty>1
			&& server->max_connections>1)
				--slots_open;

		if (slots_open <= 0)
		{
			debug2 (DEBUG_QUEUE,
				"we've already got %d connections "
				"and %d reader connections",
				current_connections,
				(reader_socket_exists?1:0) );

			retval = -1;
		}
		else	
		{
			char * errmsg = NULL;

			/* open a new socket.. */
			PanSocket *sock = NULL;
			debug1 (DEBUG_QUEUE, "making a new socket to %s", server->address);

			fire_connecting_state_changed (TRUE);
			sock = pan_socket_new (server->name, server->address, server->port);
			pan_socket_set_nntp_auth (sock, server->need_auth, server->username, server->password); 

			debug1 (DEBUG_QUEUE, "new socket %p", sock);
			if (sock == NULL || sock->sockfd < 0)
			{
				if (is_nonempty_string (server->address))
					errmsg = g_strdup_printf (_("Unable to connect to server %s:%d"), server->address, server->port);
				else
					errmsg = g_strdup (_("No server address specified in the Preferences"));
			}
			else /* handshake */
			{
				StatusItem * status = STATUS_ITEM(task);
				gboolean posting_ok = FALSE;
				gint val = nntp_handshake (status, sock, &posting_ok);
				if (val != TASK_SUCCESS)
					errmsg = g_strdup(g_slist_last(status->errors)->data);
			}
			if (errmsg!=NULL)
			{
				log_add (LOG_ERROR, errmsg);
				g_free (errmsg);
				if (sock)
					pan_object_unref (PAN_OBJECT(sock));
				sock = NULL;

				debug1 (DEBUG_QUEUE, "%p can't log in!", sock);

				retval = -2;
			}
			else
			{
				SocketInfo * si = NULL;

				debug1 (DEBUG_QUEUE, "%p logged in", sock);

				/* looks like we got a new socket okay... */
				si = g_new (SocketInfo, 1);
				si->socket = sock;
				si->server = server;
				si->task = task;
				task->sock = sock;
				sockets = g_slist_prepend (sockets, si);
				retval = 0;
				fire_connection_size_changed (1);
			}

			fire_connecting_state_changed (FALSE);
		}
	}

	g_mutex_unlock (sock_lock);

	debug_exit ("queue_get_socket_for_item");
	return retval;
}

/**
 * Run any tasks in the task_list which can be run (ie, can get a socket,
 * if necessary) right now.
 */
static void
queue_run_what_we_can (void)
{
	GSList * l = NULL;
	debug_enter ("queue_run_what_we_can");

	/* walk through the queue and see if any tasks are waiting to be run. */
	l = task_list;
	while (l!=NULL)
	{
		Task *task = TASK(l->data);

		/* if it's not waiting to be run, leave it alone */
		if (queue_get_task_status(task) != QUEUE_TASK_STATUS_QUEUED)
		{
			debug0 (DEBUG_QUEUE, "task isn't waiting to be run");

		}

		/* if it doesn't need a socket, no problemo! */
		else if (!task->needs_socket)
		{
			debug1 (DEBUG_QUEUE,
				"task %d needs no socket; running it now",
				task);

			queue_run (task);
		}

		/* try to get a socket... */
		else if (queue_is_online() && max_socket_qty>0)
		{
			int i;
			queue_set_task_status (task, QUEUE_TASK_STATUS_CONNECTING);
			i = queue_get_socket_for_task (task);
			if (i==0)
			{
				debug1 (DEBUG_QUEUE,
					"task %d got a socket!",
					task);

				queue_run (task);
			}
			else
			{
				if (i==-2) /* connection failure */
				{
					status_item_emit_error (STATUS_ITEM(task), _("Connect Failure"));

					++task->tries;
					if (task->tries > max_tries)
						queue_abort_tasks (g_slist_append(NULL,task));
				}

				queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
			}
		}

		if (!g_queue_is_empty (todo_queue))
		{
				queue_do_todo ();
				l = task_list;
		}
		else
		{
				l = l->next;
		}
	}

	debug_exit ("queue_run_what_we_can done");
}

/**
 * Destroy or recycle a socket when a task is done with it.
 * @param task the finished task.
 */
static void
socket_cleanup (Task* task)
{
	debug_enter ("socket_cleanup");

	if (task && task->sock)
	{
		PanSocket* socket = task->sock;
		GSList* l = NULL;
		SocketInfo * info = NULL;
		int server_socket_qty;
		int socket_qty;

		g_mutex_lock (sock_lock);

		/* count the number of sockets this server has */
		socket_qty = 0;
		server_socket_qty = 0;
		for (l=sockets; l!=NULL; l=l->next) {
			SocketInfo * sockinfo = (SocketInfo*) l->data;
			if (sockinfo->socket == socket)
				info = sockinfo;
			if (sockinfo->server == task->server)
				++server_socket_qty;
			++socket_qty;
		}

		/* find this SocketInfo in our tables */
		g_assert (info!=NULL);

		/**
                 *  Recycle the socket if all are true
		 *  + there was no error
		 *  + the server is still online
		 *  + we don't have too many sockets for that server
		 *  + we don't have too many sockets overall
		 */
		if (!socket->error
		    && queue_is_online()
		    && socket_qty<=max_socket_qty
		    && server_socket_qty<=task->server->max_connections)
		{
		       	/* if socket's okay, reuse it */
			info->task = NULL;

			debug1 (DEBUG_QUEUE, "socket_cleanup: recycling socket %p", socket);
		}
		else
		{
			/* otherwise throw it away */
			nntp_disconnect (NULL, socket);
			pan_object_unref (PAN_OBJECT(socket));
			fire_connection_size_changed (-1);
			sockets = g_slist_remove (sockets, info);
			g_free (info);

			debug1 (DEBUG_QUEUE,
				"socket_cleanup: throwing away socket %p",
				socket);
		}
		task->sock = NULL;
		g_mutex_unlock (sock_lock);
	}
	debug_exit ("socket_cleanup");
}

static gchar*
noop_describe (const StatusItem * item)
{
	return g_strdup (_("Sending 'stay connected' request"));
}

static gchar*
disconnect_describe (const StatusItem * item)
{
	return g_strdup (_("Disconnecting"));
}

/**
 * socket upkeep:  Cull out idle sockets that have been connected too long,
 * and send keepalive commands the rest of the idle sockets so that the
 * server doesn't drop us.
 */
static void
sockets_upkeep (void)
{
	GSList* l = NULL;
	GSList* cull = NULL;
	int socket_qty;
	debug_enter ("socket_upkeep");

	g_mutex_lock (sock_lock);

	socket_qty = g_slist_length (sockets);

	for (l=sockets; l!=NULL; l=l->next)
	{
		gint age;
		gboolean destroy, idle, old, too_many, offline;
		SocketInfo * info;
		PanSocket * sock;
		const time_t current_time = time(0);

		info = (SocketInfo*) l->data;
		g_assert (info != NULL);
		sock = info->socket;
		g_assert (sock != NULL);

		/* calculate some information about the socket
		   to decide whether or not we want to keep it */
		age = current_time - sock->last_action_time;
		idle = info->task == NULL;
		old = age > info->server->idle_secs_before_timeout;
		too_many = socket_qty > max_socket_qty;
		offline = !queue_is_online ();
		destroy = idle && (old||too_many||offline);

		/* if it's idle and we're keeping it, send a keepalive msg.
		   Note that we don't let this affect the socket's
		   age, otherwise they would never grow old and close... */
		if (info->task==NULL && !destroy)
		{
			const time_t last_action_time = sock->last_action_time;
			StatusItem * status = status_item_new (noop_describe);
			status_item_set_active (status, TRUE);

			destroy = nntp_noop (STATUS_ITEM(status), sock) != TASK_SUCCESS;
			sock->last_action_time = last_action_time;

			status_item_set_active (status, FALSE);
			pan_object_unref (PAN_OBJECT(status));
		}

		/* either keep the socket or cull it from the herd */
		if (destroy) {
			--socket_qty;
			cull = g_slist_prepend (cull, info);
		}
	}

	debug1 (DEBUG_QUEUE, "sockets_upkeep: culling %u sockets",
		g_slist_length(cull));

	for (l=cull; l!=NULL; l=l->next)
	{
		const time_t current_time = time(0);
		SocketInfo * info = (SocketInfo*) l->data;
		PanSocket * sock = info->socket;
		StatusItem * status = STATUS_ITEM(status_item_new (disconnect_describe));
		const int age = current_time - sock->last_action_time;

		/* politely disconnect */
		status_item_set_active (status, TRUE);
		if (!sock->error)
			nntp_disconnect (status, sock);
		status_item_set_active (status, FALSE);
		log_add_va (LOG_INFO, _("Closing connection %p after %d seconds idle"), sock, age);

		/* destroy the socket */
		pan_object_unref (PAN_OBJECT(sock));
		fire_connection_size_changed (-1);
		sockets = g_slist_remove (sockets, info);

		/* cleanup local */
		pan_object_unref (PAN_OBJECT(status));
		g_free (info);
	}

	pan_warn_if_fail (g_slist_length(sockets) == socket_qty);

	g_mutex_unlock (sock_lock);

	g_slist_free (cull);

	debug_exit ("socket_upkeep");
}

/**
 * Process the tasks that are waiting in the todo list
 */
static void
queue_do_todo (void)
{
	gboolean do_wakeup = FALSE;
	gboolean do_shutdown = FALSE;
	GQueue * queue;
	debug_enter ("queue_do_todo");

	g_mutex_lock (todo_mutex);
	queue = todo_queue;
	todo_queue = g_queue_new ();
	g_mutex_unlock (todo_mutex);

	while (!g_queue_is_empty(queue))
	{
		QueueTodo* a = (QueueTodo*) g_queue_pop_head (queue);

		switch (a->action)
		{
			case TODO_INSERT:
				real_queue_insert_tasks (a->tasks, a->index_1, &do_wakeup);
				break;
			case TODO_REMOVE:
				real_queue_remove_tasks (a->tasks, &do_wakeup);
				break;
			case TODO_MOVE:
				real_queue_move_tasks (a->tasks, a->index_1);
				break;
			case TODO_REQUEUE:
				real_requeue_failed_tasks (a->tasks, &do_wakeup);
				break;
			case TODO_TASK_ABORT:
				real_queue_abort_tasks (a->tasks);
				break;
			case TODO_SET_MAX_SOCKET_QTY:
				real_queue_set_max_sockets (a->index_1, &do_wakeup);
				break;
			case TODO_SET_MAX_TRIES:
				real_queue_set_max_tries (a->index_1);
				break;
			case TODO_SHUTDOWN:
				do_shutdown = TRUE;
				break;
			case TODO_ONLINE:
				real_queue_set_online (a->index_1, &do_wakeup);
				break;
			default:
				pan_warn_if_reached();
				break;
		}

		g_slist_free (a->tasks);
		g_free (a);
	}

	if (do_wakeup)
		queue_wakeup ();
	if (do_shutdown)
		real_queue_shutdown ();

	/* throw the todo list away */
	g_queue_free (queue);
	debug_exit ("queue_do_todo");
}

static void*
queue_mainloop (void* unused)
{
	const int TIMEOUT_SECS = 60;

	for (;;)
	{
		gboolean was_timeout;
		GTimeVal sleep_until;

		g_mutex_lock (cond_mutex);

		/* sleep for TIMEOUT_SECS unless someone wakes us up */
		g_get_current_time (&sleep_until);
		g_time_val_add (&sleep_until, TIMEOUT_SECS*G_USEC_PER_SEC);
		was_timeout = FALSE;
		while (!work_to_do && !was_timeout)
			was_timeout = !g_cond_timed_wait (qcond, cond_mutex, &sleep_until);
		work_to_do = FALSE;
                g_mutex_unlock (cond_mutex);

		/* do work */
		if (was_timeout)
			sockets_upkeep ();
		else {
			queue_do_todo ();
                	queue_run_what_we_can ();
			if (_dirty) {
				_dirty = FALSE;
				queue_save_tasks ();
			}
		}
	}
}


PanCallback*
queue_get_online_status_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_online_status_changed (gboolean online)
{
	pan_callback_call (queue_get_online_status_changed_callback(),
	                   GINT_TO_POINTER(online), NULL);
}

PanCallback*
queue_get_message_id_status_changed (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_message_id_status_changed (const char ** message_ids, int message_id_qty)
{
	pan_callback_call (queue_get_message_id_status_changed(),
	                   message_ids,
	                   GINT_TO_POINTER(message_id_qty));
}

PanCallback*
queue_get_connection_size_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_connection_size_changed (int increment)
{
	pan_callback_call (queue_get_connection_size_changed_callback(),
	                   GINT_TO_POINTER(increment),
	                   NULL);
}

PanCallback*
queue_get_connecting_state_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_connecting_state_changed (gboolean connecting)
{
	pan_callback_call (queue_get_connecting_state_changed_callback(),
	                   GINT_TO_POINTER(connecting),
	                   NULL);
}

PanCallback*
queue_get_size_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_queue_size_changed (int running_qty, int total_qty)
{
	pan_callback_call (queue_get_size_changed_callback(),
	                   GINT_TO_POINTER(running_qty),
	                   GINT_TO_POINTER(total_qty));
}
