/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <glib.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/task.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

void
task_constructor (Task *item,
                  PanObjectDestructor destructor,
                  StatusItemDescribeFunc describe,
                  TaskRunFunc run,
                  Server *server,
                  gboolean high_priority,
                  gboolean needs_socket)
{
	g_return_if_fail (item != NULL);

	/* construct parent's bits */
	status_item_constructor (STATUS_ITEM(item), destructor, describe);

	/* construct task's bits */
	pan_warn_if_fail (server != NULL);
	item->server = server;
	item->hint_abort = FALSE;
	item->high_priority = high_priority;
	item->is_running = FALSE;
	item->needs_socket = needs_socket;
	item->sock = NULL;
	item->gets_bodies = FALSE;
	item->is_article_reader = FALSE;
	item->tries = 0;
	item->run = run;
	item->task_ran_callback = pan_callback_new ();
	item->identifiers = g_ptr_array_new ();

        debug1 (DEBUG_PAN_OBJECT, "task constructor: %p", item);
}

void
task_destructor (PanObject* obj)
{
	Task * task;

	/* sanity clause */
	g_return_if_fail (obj != NULL);

	/* clean up task bits */
        debug1 (DEBUG_PAN_OBJECT, "task destructor: %p", obj);
	task = TASK(obj);
	pan_callback_free (task->task_ran_callback);
	pan_g_ptr_array_foreach (task->identifiers, (GFunc)g_object_unref, NULL);
	g_ptr_array_free (task->identifiers, TRUE);

	/* clean up parent's bits */
	status_item_destructor (obj);
}

/*****
******
*****/

void
task_add_identifiers (Task                * task,
                      MessageIdentifier  ** identifiers,
                      int                   identifier_qty)
{
	guint i;

	/* sanity clause */
	g_return_if_fail (task!=NULL);
	g_return_if_fail (identifiers!=NULL);
	g_return_if_fail (identifier_qty>0);
	for (i=0; i<identifier_qty; ++i)
		g_return_if_fail (PAN_IS_MESSAGE_IDENTIFIER(identifiers[i]));

	/* add the message-ids */	
	for (i=0; i<identifier_qty; ++i) {
		g_object_ref (identifiers[i]);
		g_ptr_array_add (task->identifiers, identifiers[i]);
	}
}

GPtrArray*
task_get_identifiers (Task * task)
{
	GPtrArray * retval = g_ptr_array_new ();

	/* sanity clause */
	g_return_val_if_fail (task!=NULL, retval);

	/* get the message-ids */
	pan_g_ptr_array_assign (retval, task->identifiers->pdata, task->identifiers->len);
	return retval;
}


/*****
******
*****/

int
task_run (Task * task)
{
	int retval = TASK_FAIL_HOPELESS;

	if (task != NULL)
	{
        	retval = (*task->run)(task);
		status_item_emit_done (STATUS_ITEM(task), retval);
	}

	pan_warn_if_fail (retval==TASK_SUCCESS ||
	                  retval==TASK_FAIL ||
	                  retval==TASK_FAIL_HOPELESS);

	pan_callback_call (task->task_ran_callback,
	                   task,
			   GINT_TO_POINTER(retval));

	return retval;
}

/*****
******
*****/

void
task_hint_abort (Task * task)
{
	g_return_if_fail (task!=NULL);

	task->hint_abort = TRUE;
}
