/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdio.h>
#include <errno.h>

#include <glib.h>

#include <pan/base/base-prefs.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/thread-watch.h>

/*********************
**********************  Defines / Enumerated types
*********************/

#define WATCHED_THREAD_SPAN_DAYS 31

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static GHashTable * hash = NULL;

static gboolean dirty = FALSE;

static guint hash_size = 0;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

/*****
******
*****/

static char*
get_filename (void)
{
	return g_strdup_printf ("%s%cthreads.txt", get_data_dir(), G_DIR_SEPARATOR);
}

/*****
******
*****/

typedef struct
{
	size_t start_time;
	int state;
}
RecordStruct;

static void
thread_init_module (void)
{
	if (hash == NULL)
	{
		const time_t current_time = time(0);
		RecordStruct * data;
		char * filename;
		char * file;
		char ** v;
		int i;

		/* create the hashtable */
		hash = g_hash_table_new (g_str_hash, g_str_equal);
		hash_size = 0;
		dirty = FALSE;

		/* load the file */
		filename = get_filename ();
		g_file_get_contents (filename, &file, NULL, NULL);
		v = file==NULL ? NULL : g_strsplit (file, "\n", -1);
		g_hash_table_freeze (hash);
		for (i=0; v!=NULL && v[i]!=NULL; ++i)
		{
			int state;
			time_t start_time;
			char * id;
			const char * march = v[i];
			time_t age_secs;

			/* skip comments */
			if (*march=='#')
				continue;

			/* get info */
			state      = get_next_token_int   (march, ',', &march);
			start_time = get_next_token_ulong (march, ',', &march);
			id         = get_next_token_str   (march, ',', &march);

			/* age out threads > one month old */
			age_secs = current_time - start_time;
			if ((age_secs/86400) > WATCHED_THREAD_SPAN_DAYS) {
				dirty = TRUE;
				g_free (id);
				continue;
			}

			/* insert to the hash table */
			data = g_new (RecordStruct, 1);
			data->start_time = start_time;
			data->state = state;
			g_hash_table_insert (hash, id, data);
			++hash_size;
		}
		g_hash_table_thaw (hash);

		/* cleanup */
		g_free (filename);
		g_free (file);
		g_strfreev (v);
	}
}

/*****
******
*****/

/************
*************  PUBLIC
************/

int
pan_thread_get_state (const Article * a)
{
	int retval = THREAD_NOTHING;

	/* sanity clause */
	g_return_val_if_fail(a, retval);

	/* make sure module's ready */
	if (hash == NULL)
		thread_init_module ();

	/* get the thread state */
	if (hash_size != 0)
	{
		if (a != NULL)
		{
			char * id = article_get_thread_message_id (a);

			if (id != NULL)
			{
				gpointer data = g_hash_table_lookup (hash, id);

				if (data != NULL)
				{
					retval = ((RecordStruct*)data)->state;
				}

				g_free (id);
			}
		}
	}


	return retval;
}

void
pan_thread_set_state (const Article * a, int state)
{
	char * id;

	/* sanity checks */
	g_return_if_fail (article_is_valid(a));

	/* make sure module's ready */
	if (hash == NULL)
		thread_init_module ();

	/* check that thread's message id */
       	id = article_get_thread_message_id (a);
	if (is_nonempty_string(id))
	{
		gpointer val = g_hash_table_lookup (hash, id);

		/* update/insert the thread's state */
		if (val != NULL)
		{
			RecordStruct * data = (RecordStruct*) val;
			data->start_time = time(0);
			data->state = state;
			g_free (id);
		}
		else
		{
			RecordStruct * data = g_new (RecordStruct, 1);
			data->start_time = time(0);
			data->state = state;
			g_hash_table_insert (hash, id, data);
			++hash_size;
		}
		dirty = TRUE;
	}

	if (a->group != NULL)
		a->group->articles_dirty = TRUE;
}


/**
***
***  Shutdown
***
**/

static void
hash_freekeyval (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	g_free (value);
}
static void
free_hash (void)
{
	if (hash != NULL)
	{
		g_hash_table_foreach (hash, hash_freekeyval, NULL);
		g_hash_table_destroy (hash);
		hash = NULL;
	}

	hash_size = 0;
}

static void
thread_write_line (gpointer key, gpointer value, gpointer user_data)
{
	const RecordStruct * data = (const RecordStruct*) value;
	if (data->state != THREAD_NOTHING)
	{
		const char * id = (const char*) key;
		FILE * fp = (FILE*)user_data;

		fprintf (fp, "%d,%lu,%s\n",
			data->state,
			(gulong)data->start_time,
			id);
	}
}

void
thread_shutdown_module (void)
{
	if (dirty)
	{
		/* write the thread states to disk */
		char * filename = get_filename ();
		FILE * fp = fopen (filename, "w");

		if (fp == NULL)
			g_warning (_("Error opening file \"%s\": %s"), filename, g_strerror(errno));
		else
		{
			fprintf (fp, "# one record per line, comma-delimited fields\n");
			fprintf (fp, "# each record has three fields:\n");
			fprintf (fp, "#    state (0=nothing,1=watched,2=ignored)\n");
			fprintf (fp, "#    when this state was set, in secs since Jan 1, 1970.  These states expire after 1 month.\n");
			fprintf (fp, "#    message-id of parent thread\n");
			g_hash_table_foreach (hash, thread_write_line, fp);
			fclose (fp);
		}

		g_free (filename);
		dirty = FALSE;
	}

	free_hash ();
}
